/***********************************************************/
/* File Id.                  Mouse.C                       */
/* Author.                   Stan Milam.                   */
/* Date Written.             11/27/88.                     */
/* Date Last Modified.                                     */
/*                                                         */
/*           (c) Copyright 1989-90 by Stan Milam           */
/*                                                         */
/* Comments:  This file represents a group of mouse func-  */
/* tions for use with a Microsoft Mouse or compatible.     */
/***********************************************************/

#include <dos.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include "pcwproto.h"

static void get_x_coord(void);

static union REGS regs;
static int   m_hide_flag = 1;               /* Hidden set to true */
static int   x_coord;

int  mbuttons;
int  mpresent = 0;

/***********************************************************/
/*                         Init_Mouse                      */
/*                                                         */
/* This function will initialize the Mouse if one is pre-  */
/* sent.  It will return 1 if a mouse is present and NULL  */
/* if one is not.  If a Mouse is installed the number of   */
/* mouse buttons will be stored in a GLOBAL, mbuttons.     */
/***********************************************************/

int init_mouse(void) {

    memset(&regs, '\0', sizeof(union REGS));
    regs.x.ax = 0;
    int86(0x33,&regs,&regs);
    if (regs.x.ax == 0) {
       mbuttons = 0;
       return(regs.x.ax);
    }
    else {
       mbuttons = regs.x.bx;
       if (mpresent) mpresent = 0;     /* Turn off the Mouse */
       else mpresent = 1;              /* Tell the world we have a mouse */
       get_mpressed(0);                /* Clear LEFT Mouse Button */
       get_mpressed(1);                /* Clear RITE Mouse Button */
       return(1);
   }
}

/***********************************************************/
/*                         Show_Mouse                      */
/*                                                         */
/* This function will show the mouse.  It will first check */
/* a STATIC variable to make sure it is already hidden     */
/* before proceeding.                                      */
/***********************************************************/

void show_mouse(void) {

     if (m_hide_flag) {
        m_hide_flag = 0;
        memset(&regs, '\0', sizeof(union REGS));
        regs.x.ax = 1;
        int86(0x33, &regs, &regs);
     }
}

/***********************************************************/
/*                         Hide_Mouse                      */
/*                                                         */
/* This function will hide the Mouse cursor if it is visi- */
/* ble and reset the hide flag.                            */
/***********************************************************/

void hide_mouse(void) {

     if (m_hide_flag == 0) {
        m_hide_flag = 1;
        memset(&regs, '\0', sizeof(union REGS));
        regs.x.ax = 2;
        int86(0x33, &regs, &regs);
     }
}

/***********************************************************/
/*                          Get_Mpos                       */
/*                                                         */
/* This function will take three pointers to integers as   */
/* arguments.  They will return the row, column, and the   */
/* button status of the Mouse.  This information is return-*/
/* ed REAL TIME.  This means this info may not be what you */
/* want.  For example, the mouse may be in motion when you */
/* make this call, but you will not know the destination of*/
/* the Mouse. Good for use in a tight loop.                */
/* Button Status:  Bit 0 set if left button pressed.       */
/*                 Bit 1 set if rite button pressed.       */
/***********************************************************/

void get_mpos(int *row, int *col, int *bstatus)  {

     memset(&regs, '\0', sizeof(union REGS));
     get_x_coord();
     regs.x.ax = 3;
     int86(0x33, &regs, &regs);
     *row      = (regs.x.dx / 8) + 1;
     *col      = (regs.x.cx / x_coord) + 1;
     *bstatus  = regs.x.bx;
}

/***********************************************************/
/*                          Set_Mpos                       */
/*                                                         */
/* This function will set the mouse position.              */
/***********************************************************/

void set_mpos(int row, int col)  {

     memset(&regs, '\0', sizeof(union REGS));
     get_x_coord();
     regs.x.ax = 4;
     regs.x.dx = (row * 8) - 1;
     regs.x.cx = (col * x_coord) - 1;
     int86(0x33, &regs, &regs);
}

/***********************************************************/
/*                        Get_Mpressed                     */
/*                                                         */
/* This function will return button pressed info.          */
/* Specifically, it will return the number of times a but- */
/* ton has been pressed since the last call for that button*/
/* Call with: 0 = left button.                             */
/*            1 = rite button.                             */
/*            2 = mid  button.  (Logitech Only)            */
/***********************************************************/

int get_mpressed(int button) {

    memset(&regs, '\0', sizeof(union REGS));
    regs.x.ax = 5;
    regs.x.bx = button;
    int86(0x33, &regs, &regs);
    return(regs.x.bx);
}

/***********************************************************/
/*                        Get_Mreleased                    */
/*                                                         */
/* This function works as above except that it returns the */
/* number of times a button has been released.             */
/***********************************************************/

int get_mreleased(int button) {

    memset(&regs, '\0', sizeof(union REGS));
    regs.x.ax = 6;
    regs.x.bx = button;
    int86(0x33, &regs, &regs);
    return (regs.x.bx);
}

/***********************************************************/
/*                           Mframe                        */
/*                                                         */
/* This function incorporates two mouse driver functions:  */
/* set horizontal & vertical boundries.  It will confine   */
/* the Mouse cursor to rectangular block on the screen.    */
/***********************************************************/

void mframe(int urow, int ucol, int lrow, int lcol) {

/* Set vertical boundries of Mouse */

     memset(&regs, '\0', sizeof(union REGS));
     regs.x.ax = 8;
     regs.x.dx = (urow * 8) - 1;
     regs.x.cx = (lrow * 8) - 1;
     int86(0x33, &regs, &regs);

/* Set horizontal boundries of Mouse */

     get_x_coord();
     regs.x.ax = 7;
     regs.x.dx = (ucol * x_coord) - 1;
     regs.x.cx = (lcol * x_coord) - 1;
     int86(0x33, &regs, &regs);
}

/***********************************************************/
/*                         Set_Mtype                       */
/*                                                         */
/* This function will set the mouse cursor Type. You have  */
/* two choices: The default Mouse cursor or taking over the*/
/* hardware cursor.  If you take the default, there will be*/
/* two cursors on the screen, the Mouse & the Hardware.    */
/* If you choose the default you can specify which char-   */
/* acter it will be and its color.                         */
/*                                                         */
/* Default Cursor:                                         */
/*      ctype = 0;                                         */
/*      arg1  = 0;                                         */
/*      arg2  = (attr << 8) | char                         */
/* Hardware Cursor:                                        */
/*      ctype = 1;                                         */
/*      arg1  = 0x77ff;                                    */
/*      arg2  = 0x7700;                                    */
/***********************************************************/

void set_mtype(int ctype, int arg1, int arg2) {

     memset(&regs, '\0', sizeof(union REGS));
     regs.x.ax = 10;
     regs.x.bx = ctype;
     regs.x.cx = arg1;
     regs.x.dx = arg2;
     int86(0x33, &regs, &regs);
}

/**********************************************************/
/*                   Save_Mouse_State                     */
/*                                                        */
/* This function can be used to save the state of the     */
/* Mouse before executing another program that uses the   */
/* Mouse.  On return from the child process a call to Re- */
/* store_Mouse_State will, of course, restore your Mouse. */
/**********************************************************/

char *save_mouse_state(void) {

   void far *buffer;
   char *temp;
   struct SREGS sregs;

   memset(&regs, '\0', sizeof(union REGS));
   regs.x.ax = 21;
   int86(0x33,&regs,&regs);
   temp = malloc(regs.x.bx);
   buffer = (void far *) temp;
   if (buffer == (void far *) 0) return(0);
   regs.x.ax = 22;
   regs.x.dx = FP_OFF(buffer);
   sregs.es  = FP_SEG(buffer);
   int86x(0x33,&regs,&regs,&sregs);
   return(temp);
}

/**********************************************************/
/*                   Restore_Mouse_State                  */
/*                                                        */
/* Restores the Mouse state that was previously saved.    */
/* Accepts a pointer to the saved buffer.                 */
/**********************************************************/

void restore_mouse_state(char *buffer) {

   struct SREGS sregs;
   void far *temp;

   memset (&regs, 0, sizeof(union REGS));
   temp = (void far *) buffer;
   regs.x.ax = 23;
   regs.x.dx = FP_OFF(temp);
   sregs.es  = FP_SEG(temp);
   int86x(0x33,&regs,&regs,&sregs);
   free(buffer);
}


/**********************************************************/
/*                       Get_X_Coord                      */
/*                                                        */
/* This function sets the variable x_coord to the proper  */
/* value to determine the true column of the mouse. Is    */
/* determined by the video mode.                          */
/**********************************************************/

static void get_x_coord(void) {

   int cols, mode, apage;

   vgetmode(&cols, &mode, &apage);
   switch(mode) {
      case  0 :
      case  1 : x_coord = 16; break;
      case  2 :
      case  3 :
      case  7 : x_coord =  8; break;
      case  4 :
      case  5 : x_coord =  2; break;
      case 13 : x_coord = 16; break;
      default : x_coord =  1; break;
   }
}
