
/*============================================================================

	FILE4DRV.C
	Determines PSCDxxxx driver paramters for a given file.
  This file is part of the

  PseudoCD package (version 02)
  Copyright (C) C.Kulms, 1997

  This is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This software is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this software; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

============================================================================*/

#include <stdio.h>
#include <stdlib.h>
#include <mem.h>
#include <ctype.h>
#include <string.h>
#include <dos.h>
#include <dir.h>

#include "biosdriv.h"		/* some low-level stuff for BIOS-supported drives */
#include "drvdriv.h"		/* some low-level stuff for device-drivers */

/* lower function; see file 'FINDFIRS.ASM' */
extern int DOS_FindFirstByFCB( struct fcb far *fpsFCB );

typedef struct s_EDIRENTRY	/* extended DOS directory entry */
{
	char					cDrive;				/* drive number (A=1,B=2,...) */
	char					acName[8];		/* file name */
	char					acExt[3];			/* file extension */
	char					cAttrib;			/* file attributes */
	char					acRsvd[10];
	unsigned			uTime;				/* time last write */
	unsigned			uDate;				/* data last write */
	unsigned			uCluster;			/* first cluster */
	unsigned long	ulSize;				/* size in bytes */
}	t_EDIRENTRY;


t_HDDATA	_tHDData[8];			/* parameters of BIOS-supported HDs */
int				_iNumberOfHDs;		/* number of BIOS-supported HDs */
t_DISK		_tDosDisk[24];		/* parameters of DOS's disks on -"- */
int				_iNumberOfDisks;	/* number of -"- */

/*
	BiosRWSectors
	Reads/writes absolute sectors with BIOS int 13h.
	return: BIOS's errorcode
*/
int BiosRWSectors( int iCmd, void *pBuf, int iDrive, unsigned long ulSector, int iCount )
{
int		iError,	/* holds error-code */
			iTrack,	/* track of sector */
			iHead,	/* head of sector */
			iSec;		/* sector on track under head of sector */
unsigned	uTmp;
	
	/* convert sector number into track-head-sector */
	iTrack = (int)(ulSector / (unsigned long)_tHDData[iDrive].uHSPT);
	uTmp = (unsigned)(ulSector % (unsigned long)_tHDData[iDrive].uHSPT);
	iHead = (int)(uTmp / (unsigned)_tHDData[iDrive].ucSectors);
	iSec = (int)(uTmp % (unsigned)_tHDData[iDrive].ucSectors) + 1;

	/* call int 13h */
	iError = BIOS_RWSectors( iCmd, (void far *)pBuf,
													_tHDData[iDrive].ucBiosDriveNo,
													iTrack, iHead, iSec, iCount );

	return( iError );
}

/*
	GetHDDrives
	Uses BIOS int 13h functions to determine the number of HD-drives supported
	by the BIOS. Also stores the parameters of the drives found.
	return: 1
*/
int GetHDDrives( void )
{
t_MPR	tMPR;			/* holds a Master Partition Record (resp. Master Boot Record) */
int		iDrive,		/* drive counter */
			iBiosNo,	/* BIOS's drive number (>=0x80) */
			iType;		/* type of drive (0x03xx = HD) */

	/* detect HD-drives supported by BIOS */
	iDrive = 0;
	for( iBiosNo = 0x0080; iBiosNo < 0x0088; iBiosNo++ )
	{
		if( !(iType = BIOS_GetDriveType( iBiosNo )) )
			break;	/* abort at first 'no drive present' */

		if( (iType >> 8) == 0x03 )	/* a HD ? */
			_tHDData[iDrive++].ucBiosDriveNo = (unsigned char)iBiosNo;
	}
	_iNumberOfHDs = iDrive;

	/* get parameters of detected HD-drives */
	for( iDrive--; iDrive >= 0; iDrive-- )
	{
		/* get paramters of HD-drive */
		iType = BIOS_GetDriveData( _tHDData[iDrive].ucBiosDriveNo,
											(unsigned far *)&_tHDData[iDrive].uTracks,
											(unsigned char far *)&_tHDData[iDrive].ucHeads,
											(unsigned char *)&_tHDData[iDrive].ucSectors );
		if( iType )
		{
			fprintf( stderr, "int 13h ('get drive parameters'), drive 0x%02X"
								" failed (errorcode %04X)\n",
								_tHDData[iDrive].ucBiosDriveNo, (unsigned)iType );
			_tHDData[iDrive].ucBiosDriveNo = 0;
		}
		else
			_tHDData[iDrive].uHSPT = (unsigned)_tHDData[iDrive].ucHeads
																* (unsigned)_tHDData[iDrive].ucSectors;
	}

	/* read MPRs and extract partition data */
	for( iDrive = 0; iDrive < _iNumberOfHDs; iDrive++ )
	{
		if( !_tHDData[iDrive].ucBiosDriveNo )
			continue;	/* skip the erraneous ones */

		/* read sector 0 */
		iType = BIOS_RWSectors( BIOS_READ, (void far *)&tMPR,
											_tHDData[iDrive].ucBiosDriveNo,
											0,	/* track */
											0,	/* head */
											1,	/* sector */
											1	/* number of sectors */ );
		if( iType )
		{
			fprintf( stderr, "int 13h ('read sector'), drive 0x%02X"
								" failed (errorcode %04X)\n",
								_tHDData[iDrive].ucBiosDriveNo, (unsigned)iType );
			_tHDData[iDrive].ucBiosDriveNo = 0;
			continue;
		}
		if( tMPR.uBootId != BOOTID )	/* valid boot-record? */
		{
			fprintf( stderr, "drive 0x%02X: invalid MPR\n",
								_tHDData[iDrive].ucBiosDriveNo );
			_tHDData[iDrive].ucBiosDriveNo = 0;
		}
		else	/* store partition table */
			memcpy( _tHDData[iDrive].tPartition, tMPR.tPartition,
							4*sizeof(t_PARTITION) );
	}

	return( 1 );
}

/*
	AssignDisks
	Assigns DOS's disk-letters to the partitions on the BIOS-supported HDs.
	return: 1
*/
int AssignDisks( void )
{
t_MPR				tEPR;			/* holds an Extended Partition Record */
t_PARTITION	*ptPart;	/* for walk-through partition table */
int		iIndex,		/* general counter/index */
			iDisk,		/* disk counter */
			iDrive,		/* drive counter */
			iError,		/* holds error-code */
			iEIndex;	/* extended-partition number */
unsigned long	ulSector;	/* start-sector of a partition */

	iDisk = 0;	/* disk 'C' */

	/* count the primary partitions first, the active one gets letter 'C' */
	for( iDrive = 0; iDrive < _iNumberOfHDs; iDrive++ )
	{
		if( _tHDData[iDrive].ucBiosDriveNo == 0 )
			continue;	/* skip the erraneous ones */

		/* walk through MPR's partition table */
		ptPart = _tHDData[iDrive].tPartition;
		for( iIndex = 0; iIndex < 4; iIndex++ )
		{
			/* check primary partitions only */
			if( ptPart->ucOsId == 0x01 || ptPart->ucOsId == 0x04
					|| ptPart->ucOsId == 0x06 )
			{
				/* active partition on first drive is 'C' */
				if( iDrive == 0 && ptPart->ucBootFlag == 0x80 )
				{
					/* store partition data */
					_tDosDisk[0].iDrive = 0;
					_tDosDisk[0].uPartitionNo = (unsigned)iIndex;
					_tDosDisk[0].ulFirstSector = ptPart->ulFirstSector;
					_tDosDisk[0].ulNumSectors = ptPart->ulNumSectors;
				}
				else
				{
					/* increment disk counter */
					iDisk++;
					/* store partition data */
					_tDosDisk[iDisk].iDrive = iDrive;
					_tDosDisk[iDisk].uPartitionNo = (unsigned)iIndex;
					_tDosDisk[iDisk].ulFirstSector = ptPart->ulFirstSector;
					_tDosDisk[iDisk].ulNumSectors = ptPart->ulNumSectors;
				}
			}
			ptPart++;	/* next partition entry */
		}
	} /* for( iDrive = 0; iDrive < _iNumberOfHDs; iDrive++ ) */

	/* now count the extended partitions */
	for( iDrive = 0; iDrive < _iNumberOfHDs; iDrive++ )
	{
		if( _tHDData[iDrive].ucBiosDriveNo == 0 )
			continue;	/* skip the erraneous ones */

		/* walk through MPR's partition table */
		ptPart = _tHDData[iDrive].tPartition;
		for( iIndex = 0; iIndex < 4; iIndex++ )
		{
			iEIndex = 0x00E0;
			if( ptPart->ucOsId == 0x05 )	/* extended partition */
			{
				/* store start sector of extended partition (first EPR in chain) */
				ulSector = ptPart->ulFirstSector;
				/* walk through the EPR-chain */
				do
				{
					/* read the EPR */
					iError = BiosRWSectors( BIOS_READ, &tEPR, iDrive, ulSector, 1 );
					if( iError )
					{
						fprintf( stderr, "drive %d (0x%02X): read error %04X at sector %lu.\n",
											iDrive, _tHDData[iDrive].ucBiosDriveNo, iError, ulSector );
						exit( -1 );
					}

					if( tEPR.tPartition[0].ucOsId == 0x01
							|| tEPR.tPartition[0].ucOsId == 0x04
							|| tEPR.tPartition[0].ucOsId == 0x06 )
					{
						iDisk++;
						/* store partition bounds */
						_tDosDisk[iDisk].iDrive = iDrive;
						_tDosDisk[iDisk].uPartitionNo = ((unsigned)iIndex << 8) | iEIndex;
						_tDosDisk[iDisk].ulFirstSector = ulSector + tEPR.tPartition[0].ulFirstSector;
						_tDosDisk[iDisk].ulNumSectors = tEPR.tPartition[0].ulNumSectors;
					}

					/* calculate sector-number of next EPR */
					if( tEPR.tPartition[1].ucOsId == 0x05 )
						ulSector += tEPR.tPartition[1].ulFirstSector;
					iEIndex++;
				}	/* EPR-chain ends if OsId of second partition-entry != 0x05 */
				while( tEPR.tPartition[1].ucOsId == 0x05 );
			}
			ptPart++;	/* next partition entry */
		}
	} /* for( iDrive = 0; iDrive < _iNumberOfHDs; iDrive++ ) */

	_iNumberOfDisks = iDisk + 1;

	return( 1 );
}

char	_acFullPath[MAXPATH+MAXFILE+MAXEXT];	/* full path of target file */
/*
	GetEDirEntry
	Get extended directory entry for given file.
	return: pointer to dir-entry (malloc()ed buffer) or 0 if error
*/
t_EDIRENTRY *GetEDirEntry( const char *pcFileName )
{
char far	*fpcDTA;						/* current DTA */
char			*pcBuf;							/* used DTA */
char			acDrive[MAXDRIVE],	/* "X:" */
					acDir[MAXDIR],			/* directory path */
					acName[MAXFILE],		/* name of the file ("nnnnnnnn") */
					acExt[MAXEXT],			/* extension of the file ("eee") */
					acString[MAXPATH];	/* scratch string */
char			acCwd[MAXPATH];			/* current directory on target drive */
struct fcb	sFcb;							/* FCB used for int 21h, function 11h */
int		iCurDrive,							/* current drive */
			iIndex;

	/* create a FCB for given file */
	fnsplit( pcFileName, acDrive, acDir, acName, acExt );
	iCurDrive = getdisk();
	if( !*acDrive )
	{
		acDrive[0] = iCurDrive + 'A';
		acDrive[1] = ':';
		acDrive[2] = 0;
	}
	setdisk( toupper( *acDrive ) - 'A' );		/* switch to target disk */
	if( *acDir )	/* change the directory set for target drive */
	{
		getcwd( acCwd, MAXPATH );		/* save current directory on target drive */
		strcpy( acString, acDir );		/* get target directory */
		iIndex = strlen( acString );
		if( iIndex > 1 )
			acString[iIndex - 1] = 0;
		if( chdir( acString ) )			/* change directory */
			perror( acString ), exit( -1 );
	}
	getcwd( acString, MAXPATH );		/* save full path of target */
	fnmerge( _acFullPath, "", acString, acName, acExt );

	fnmerge( acString, acDrive, "", acName, acExt );
	if( !parsfnm( acString, &sFcb, 3 ) )
	{
		fputs( "**** ERROR: not a legal filename ****\n", stderr );
		exit( -1 );
	}

	/* get the directory entry */
	fpcDTA = getdta();							/* save current DTA */
	if( !(pcBuf = malloc( 128 )) )	/* use my own DTA */
		return( 0 );
	setdta( (char far *)pcBuf );
	if( !DOS_FindFirstByFCB( (struct fcb far *)&sFcb ) )
	{
		free( pcBuf );
		pcBuf = 0;
	}
	setdta( fpcDTA );	/* restore DTA */

	if( *acDir )	/* if directroy changed restore old */
	{
		if( chdir( acCwd ) )
			perror( acCwd ), exit( -1 );
	}
	setdisk( iCurDrive );

	return( (t_EDIRENTRY *)pcBuf );
}

/*
	main
*/
int main( int iArgc, char *apcArgv[] )
{
unsigned			uSector;			/* sector counter */
int						iIndex;				/* general index */
unsigned long	ulFileSector,	/* first sector of file */
							ulDriveParm;	/* BIOS drive-parameters */
t_EDIRENTRY		*ptEDE;
t_DPB					tDPB;					/* DOS' Disk Paramter Block */

	fputs( "FILE4DRV v02  Copyright (C) 1997 C.Kulms\n"
				"FILE4DRV comes with ABSOLUTELY NO WARRANTY; this is free software, and you are\n"
				"welcome to redistribute it under the conditions of the GNU General Public\n"
				"License; see file 'gnu.gpl' for details.\n\n", stderr );

	if( iArgc < 2 )
	{
		fputs( "Determines PseudoCD driver paramters of an image file.\n"
						"use: FILE4DRV [drive:][path]filename\n", stderr );
		exit( -1 );
	}

	/* get (extended) directory entry */
	ptEDE = GetEDirEntry( apcArgv[1] );
	if( !ptEDE )
	{
		fprintf( stderr, "**** ERROR: file '%s' not found ****", apcArgv[1] );
		exit( -1 );
	}

	/* get DPB */
	if( !GetDPB( ptEDE->cDrive + '@', (t_DPB far *)&tDPB ) )
	{
		fputs( "unable to get DPB\n", stderr );
		exit( -1 );
	}

	/* calculate sector of cluster-number */
	ulFileSector = (unsigned long)(ptEDE->uCluster - 2);
	ulFileSector <<= tDPB.cSpCshift;
	ulFileSector += (unsigned long)tDPB.uFirstSec;

	/*
		If the disk isn't a BIOS-supported one we're done here.
		Else the sector-offset the disk (partition) has to be added.
		So get the drive-parameters now.
	*/
	GetHDDrives();	/* get partition data */
	AssignDisks();	/* assign DOS's disk-letters to partitions */

	/*
		check if disk is on a BIOS-supported drive
		Since the BIOS-supported disks are handled by DOS's own drivers they
		get the first letters, so any other disk must have a higher letter.
	*/
	iIndex = ptEDE->cDrive - 3;
	if( iIndex < 0 )
	{
		fputs( "**** ERROR: floppy drives are not allowed ****\n", stderr );
		exit( -1 );
	}
	if( iIndex >= _iNumberOfDisks )
	{
		/* display PSCDDDRV paramter (PSCDBIOS could not be used) */
		fprintf( stdout, "use: PSCDDDRV.SYS /devicename /%c /%lX\n",
							ptEDE->cDrive + '@', ulFileSector );
	}
	else
	{
		/* convert sector number to an absolute sector number */
		ulFileSector += _tDosDisk[iIndex].ulFirstSector;
		/* build 'drive-paramter' for PSCDBIOS command-line */
		iIndex = _tDosDisk[iIndex].iDrive;
		ulDriveParm = ((unsigned long)_tHDData[iIndex].ucBiosDriveNo << 16)
									| ((unsigned long)_tHDData[iIndex].ucHeads << 8)
									| (unsigned long)_tHDData[iIndex].ucSectors;
		/* display PSCDBIOS paramter (PSCDDRV should not be used) */
		fprintf( stdout, "use: PSCDBIOS.SYS /devicename /%lX /%lX\n", ulDriveParm, ulFileSector );
	}

	fprintf( stdout, "\nYou may also use: PSCDDRIV.SYS /devicename /%s\n",
						_acFullPath );

	return( 0 );
}

/* end of file 'file4drv.c' */

