/************************************************************************
 * getcrc.c - data CRC routines
 * Copyright (C) 1987, Tymnet MDNSC
 * All Rights Reserved
 *
 * SUMMARY:
 *    This module contains routines concerned with generation of data
 *    transmission CRC's.
 *
 * REVISION HISTORY:
 *
 *   Date    Version  By    Purpose of Revision
 * --------  ------- -----  ---------------------------------------------
 * 03/04/87   4.00    KJB   Initial Draft
 *
 ************************************************************************/
#include "stddef.h"
#include "xpc.h"

/* declare the byte mirror table
 */
LOCAL UBYTE mtab[] = {
    0x00, 0x80, 0x40, 0xc0, 0x20, 0xa0, 0x60, 0xe0,	/* 0x00 - 0x07 */
    0x10, 0x90, 0x50, 0xd0, 0x30, 0xb0, 0x70, 0xf0,	/* 0x08 - 0x0f */
    0x08, 0x88, 0x48, 0xc8, 0x28, 0xa8, 0x68, 0xe8,	/* 0x10 - 0x17 */
    0x18, 0x98, 0x58, 0xd8, 0x38, 0xb8, 0x78, 0xf8,	/* 0x18 - 0x1f */
    0x04, 0x84, 0x44, 0xc4, 0x24, 0xa4, 0x64, 0xe4,	/* 0x20 - 0x27 */
    0x14, 0x94, 0x54, 0xd4, 0x34, 0xb4, 0x74, 0xf4,	/* 0x28 - 0x2f */
    0x0c, 0x8c, 0x4c, 0xcc, 0x2c, 0xac, 0x6c, 0xec,	/* 0x30 - 0x37 */
    0x1c, 0x9c, 0x5c, 0xdc, 0x3c, 0xbc, 0x7c, 0xfc,	/* 0x38 - 0x3f */
    0x02, 0x82, 0x42, 0xc2, 0x22, 0xa2, 0x62, 0xe2,	/* 0x40 - 0x47 */
    0x12, 0x92, 0x52, 0xd2, 0x32, 0xb2, 0x72, 0xf2,	/* 0x48 - 0x4f */
    0x0a, 0x8a, 0x4a, 0xca, 0x2a, 0xaa, 0x6a, 0xea,	/* 0x50 - 0x57 */
    0x1a, 0x9a, 0x5a, 0xda, 0x3a, 0xba, 0x7a, 0xfa,	/* 0x58 - 0x5f */
    0x06, 0x86, 0x46, 0xc6, 0x26, 0xa6, 0x66, 0xe6,	/* 0x60 - 0x67 */
    0x16, 0x96, 0x56, 0xd6, 0x36, 0xb6, 0x76, 0xf6,	/* 0x68 - 0x6f */
    0x0e, 0x8e, 0x4e, 0xce, 0x2e, 0xae, 0x6e, 0xee,	/* 0x70 - 0x77 */
    0x1e, 0x9e, 0x5e, 0xde, 0x3e, 0xbe, 0x7e, 0xfe,	/* 0x78 - 0x7f */
    0x01, 0x81, 0x41, 0xc1, 0x21, 0xa1, 0x61, 0xe1,	/* 0x80 - 0x87 */
    0x11, 0x91, 0x51, 0xd1, 0x31, 0xb1, 0x71, 0xf1,	/* 0x88 - 0x8f */
    0x09, 0x89, 0x49, 0xc9, 0x29, 0xa9, 0x69, 0xe9,	/* 0x90 - 0x97 */
    0x19, 0x99, 0x59, 0xd9, 0x39, 0xb9, 0x79, 0xf9,	/* 0x98 - 0x9f */
    0x05, 0x85, 0x45, 0xc5, 0x25, 0xa5, 0x65, 0xe5,	/* 0xa0 - 0xa7 */
    0x15, 0x95, 0x55, 0xd5, 0x35, 0xb5, 0x75, 0xf5,	/* 0xa8 - 0xaf */
    0x0d, 0x8d, 0x4d, 0xcd, 0x2d, 0xad, 0x6d, 0xed,	/* 0xb0 - 0xb7 */
    0x1d, 0x9d, 0x5d, 0xdd, 0x3d, 0xbd, 0x7d, 0xfd,	/* 0xb8 - 0xbf */
    0x03, 0x83, 0x43, 0xc3, 0x23, 0xa3, 0x63, 0xe3,	/* 0xc0 - 0xc7 */
    0x13, 0x93, 0x53, 0xd3, 0x33, 0xb3, 0x73, 0xf3,	/* 0xc8 - 0xcf */
    0x0b, 0x8b, 0x4b, 0xcb, 0x2b, 0xab, 0x6b, 0xeb,	/* 0xd0 - 0xd7 */
    0x1b, 0x9b, 0x5b, 0xdb, 0x3b, 0xbb, 0x7b, 0xfb,	/* 0xd8 - 0xdf */
    0x07, 0x87, 0x47, 0xc7, 0x27, 0xa7, 0x67, 0xe7,	/* 0xe0 - 0xe7 */
    0x17, 0x97, 0x57, 0xd7, 0x37, 0xb7, 0x77, 0xf7,	/* 0xe8 - 0xef */
    0x0f, 0x8f, 0x4f, 0xcf, 0x2f, 0xaf, 0x6f, 0xef,	/* 0xf0 - 0xf7 */
    0x1f, 0x9f, 0x5f, 0xdf, 0x3f, 0xbf, 0x7f, 0xff};	/* 0xf8 - 0xff */

/* declare the crc vector table
 */
LOCAL CRC ctab[] = {
    0x0000, 0x1021, 0x2042, 0x3063,			/* 0x00 - 0x03 */
    0x4084, 0x50a5, 0x60c6, 0x70e7,			/* 0x04 - 0x07 */
    0x8108, 0x9129, 0xa14a, 0xb16b,			/* 0x08 - 0x0b */
    0xc18c, 0xd1ad, 0xe1ce, 0xf1ef,			/* 0x0c - 0x0f */
    0x1231, 0x0210, 0x3273, 0x2252,			/* 0x10 - 0x13 */
    0x52b5, 0x4294, 0x72f7, 0x62d6,			/* 0x14 - 0x17 */
    0x9339, 0x8318, 0xb37b, 0xa35a,			/* 0x18 - 0x1b */
    0xd3bd, 0xc39c, 0xf3ff, 0xe3de,			/* 0x1c - 0x1f */
    0x2462, 0x3443, 0x0420, 0x1401,			/* 0x20 - 0x23 */
    0x64e6, 0x74c7, 0x44a4, 0x5485,			/* 0x24 - 0x27 */
    0xa56a, 0xb54b, 0x8528, 0x9509,			/* 0x28 - 0x2b */
    0xe5ee, 0xf5cf, 0xc5ac, 0xd58d,			/* 0x2c - 0x2f */
    0x3653, 0x2672, 0x1611, 0x0630,			/* 0x30 - 0x33 */
    0x76d7, 0x66f6, 0x5695, 0x46b4,			/* 0x34 - 0x37 */
    0xb75b, 0xa77a, 0x9719, 0x8738,			/* 0x38 - 0x3b */
    0xf7df, 0xe7fe, 0xd79d, 0xc7bc,			/* 0x3c - 0x3f */
    0x48c4, 0x58e5, 0x6886, 0x78a7,			/* 0x40 - 0x43 */
    0x0840, 0x1861, 0x2802, 0x3823,			/* 0x44 - 0x47 */
    0xc9cc, 0xd9ed, 0xe98e, 0xf9af,			/* 0x48 - 0x4b */
    0x8948, 0x9969, 0xa90a, 0xb92b,			/* 0x4c - 0x4f */
    0x5af5, 0x4ad4, 0x7ab7, 0x6a96,			/* 0x50 - 0x53 */
    0x1a71, 0x0a50, 0x3a33, 0x2a12,			/* 0x54 - 0x57 */
    0xdbfd, 0xcbdc, 0xfbbf, 0xeb9e,			/* 0x58 - 0x5b */
    0x9b79, 0x8b58, 0xbb3b, 0xab1a,			/* 0x5c - 0x5f */
    0x6ca6, 0x7c87, 0x4ce4, 0x5cc5,			/* 0x60 - 0x63 */
    0x2c22, 0x3c03, 0x0c60, 0x1c41,			/* 0x64 - 0x67 */
    0xedae, 0xfd8f, 0xcdec, 0xddcd,			/* 0x68 - 0x6b */
    0xad2a, 0xbd0b, 0x8d68, 0x9d49,			/* 0x6c - 0x6f */
    0x7e97, 0x6eb6, 0x5ed5, 0x4ef4,			/* 0x70 - 0x73 */
    0x3e13, 0x2e32, 0x1e51, 0x0e70,			/* 0x74 - 0x77 */
    0xff9f, 0xefbe, 0xdfdd, 0xcffc,			/* 0x78 - 0x7b */
    0xbf1b, 0xaf3a, 0x9f59, 0x8f78,			/* 0x7c - 0x7f */
    0x9188, 0x81a9, 0xb1ca, 0xa1eb,			/* 0x80 - 0x83 */
    0xd10c, 0xc12d, 0xf14e, 0xe16f,			/* 0x84 - 0x87 */
    0x1080, 0x00a1, 0x30c2, 0x20e3,			/* 0x88 - 0x8b */
    0x5004, 0x4025, 0x7046, 0x6067,			/* 0x8c - 0x8f */
    0x83b9, 0x9398, 0xa3fb, 0xb3da,			/* 0x90 - 0x93 */
    0xc33d, 0xd31c, 0xe37f, 0xf35e,			/* 0x94 - 0x97 */
    0x02b1, 0x1290, 0x22f3, 0x32d2,			/* 0x98 - 0x9b */
    0x4235, 0x5214, 0x6277, 0x7256,			/* 0x9c - 0x9f */
    0xb5ea, 0xa5cb, 0x95a8, 0x8589,			/* 0xa0 - 0xa3 */
    0xf56e, 0xe54f, 0xd52c, 0xc50d,			/* 0xa4 - 0xa7 */
    0x34e2, 0x24c3, 0x14a0, 0x0481,			/* 0xa8 - 0xab */
    0x7466, 0x6447, 0x5424, 0x4405,			/* 0xac - 0xaf */
    0xa7db, 0xb7fa, 0x8799, 0x97b8,			/* 0xb0 - 0xb3 */
    0xe75f, 0xf77e, 0xc71d, 0xd73c,			/* 0xb4 - 0xb7 */
    0x26d3, 0x36f2, 0x0691, 0x16b0,			/* 0xb8 - 0xbb */
    0x6657, 0x7676, 0x4615, 0x5634,			/* 0xbc - 0xbf */
    0xd94c, 0xc96d, 0xf90e, 0xe92f,			/* 0xc0 - 0xc3 */
    0x99c8, 0x89e9, 0xb98a, 0xa9ab,			/* 0xc4 - 0xc7 */
    0x5844, 0x4865, 0x7806, 0x6827,			/* 0xc8 - 0xcb */
    0x18c0, 0x08e1, 0x3882, 0x28a3,			/* 0xcc - 0xcf */
    0xcb7d, 0xdb5c, 0xeb3f, 0xfb1e,			/* 0xd0 - 0xd3 */
    0x8bf9, 0x9bd8, 0xabbb, 0xbb9a,			/* 0xd4 - 0xd7 */
    0x4a75, 0x5a54, 0x6a37, 0x7a16,			/* 0xd8 - 0xdb */
    0x0af1, 0x1ad0, 0x2ab3, 0x3a92,			/* 0xdc - 0xdf */
    0xfd2e, 0xed0f, 0xdd6c, 0xcd4d,			/* 0xe0 - 0xe3 */
    0xbdaa, 0xad8b, 0x9de8, 0x8dc9,			/* 0xe4 - 0xe7 */
    0x7c26, 0x6c07, 0x5c64, 0x4c45,			/* 0xe8 - 0xeb */
    0x3ca2, 0x2c83, 0x1ce0, 0x0cc1,			/* 0xec - 0xef */
    0xef1f, 0xff3e, 0xcf5d, 0xdf7c,			/* 0xf0 - 0xf3 */
    0xaf9b, 0xbfba, 0x8fd9, 0x9ff8,			/* 0xf4 - 0xf7 */
    0x6e17, 0x7e36, 0x4e55, 0x5e74,			/* 0xf8 - 0xfb */
    0x2e93, 0x3eb2, 0x0ed1, 0x1ef0};			/* 0xfc - 0xff */

/************************************************************************
 * CRC get_crc(pc, off, len)
 *     BUFLET *pc;		pointer to buflet chain containing data
 *     BYTES off;		(first) buflet data offset
 *     BYTES len;		total data length
 *
 *     get_crc generates a CRC value from the len bytes of data contained
 *     in the buflet chain at pc, starting at offset off. The CRC will be
 *     generated using the CCITT specified polynomial:
 *
 *                       x**16 + x**12 + x**5 + 1
 *
 *     get_crc uses a compile-time vector table to speed calculation. The
 *     table is based on the following theory...
 *
 *     ...given an existing CRC, with bits:
 *
 *        C15 C14 C13 C12 C11 C10 C9  C8  C7  C6  C5  C4  C3  C2  C1  C0
 *
 *     ...run it through the CRC algorithm with byte:
 *
 *                      D7  D6  D5  D4  D3  D2  D1  D0
 *
 *     ...and the following result will be produced:
 *
 *        C7  C6  C5  C4  C3  C2  C1  C0  D7  D6  D5  D4  D3  D2  D1  D0
 *        C11 C10 C9  C15 C14 C13 C12 C11 C15 C14 C13 C12 C11 C10 C9  C8
 *        C15 C14 C13 C8              C15 C10 C9  C8      C15 C14 C13 C12
 *                    C12                 C14 C13 C12
 *
 *     ...where all bits in each column are exculsive ORed. Notice all the
 *     exlusive ORs of bits are from the upper byte of the original CRC.
 *     Thus, from that byte, we can create a table of 256 words which, when
 *     exclusive ORed with the low byte of the original CRC shifted up to
 *     the top and added to the input byte, will produce a resulting CRC.
 *
 * Notes: It is assumed the buflet chain at pc is a null terminated
 *     list of 1 or more buflets, obtained via buf_alloc. No checks
 *     are made for illegal values of pc.
 *
 *     The offset argument is not validated; this value must be less
 *     than the data buffer size of a buflet (DATA_BUF_SIZ)
 *
 *     The buflet chain at pc is assumed to contain at least off + len
 *     data bytes.
 *
 * Returns: get_crc returns a 16 bit CRC. Since X.25 insists a CRC is to
 *     be transmitted hob first, the returned CRC will be reversed.
 *
 ************************************************************************/
CRC get_crc(pc, off, len)
    BUFLET *pc;				/* pointer to buflet chain */
    BYTES off;				/* (first) buflet data offset */
    BYTES len;				/* data length */
    {
    UBYTE *pd;				/* running buflet data pointer */
    FAST BYTES count;			/* running buflet data counter */
    FAST CRC crc;			/* calculated crc (returned) */

    crc = (CRC)~0;			/* initialize crc to all 1 bits */
    while (len)				/* for each data byte */
	{

	/* set pd to the off'th data byte in the current buflet
	 */
	pd = &pc->bufdata[off];

	/* calculate the number of bytes to be processed in the current
	 * buflet and decrement the total length by that number (note min
	 * is a macro (defined in stddef.h); redundant calculations may
	 * be performed here)
	 */
	count = min(len, DATA_BUF_SIZ - off);
	len -= count;

	/* calculate the crc for the next count bytes
	 */
	do  {
	    crc = (crc << 8) ^ ctab[(crc >> 8) ^ mtab[*pd++]];
	    } while (--count);
	pc = pc->bufnext;		/* set pc to the next buflet */
	off = 0;			/* offset no longer applies */
	}

    /* since the crc is transmitted hob first, reverse
     */
    crc = (mtab[crc >> 8] << 8)| mtab[crc & 0xff];
    return (~crc);			/* 1's complement of crc is returned */
    }
