#ifndef __PAINT_H_
#define __PAINT_H_

/*  Platform and drawing library (in theory) independent drawing tools.
    Basic drawing primitives, rotation, mirror reflection, complex rotation,
    zooming and scrolling.
*/

#include <math.h>
#include "geom.h"
#include "bgi_font.h"
#include "trigon.h"

#define R_STEP 20
/*   Rotation could be simply processed if it is single operation. To use
     nested rotations we use stack of rotations info structures. Example of
     complex rotation:
     void f1(int x, int y, int alpha) { ... perform rotation ... }
     void f2(int x, int y, int alpha) { rotate(10,10,90); f1(x, y, alpha); }
     Function f2() call rotation procedure and then call f1(), which call
     another rotation. We could a) cancel first rotation and b) add first
     and second rotations to complex transformation.
*/

struct RInfo
    {
    loc center;
    int angle;
    };
///////////////
struct Stack
    {
    int used, total;
    RInfo* list;

    Stack();
    ~Stack();
    void push(RInfo* r);
    void pop();
    void flash();
    };

///////////////
/*      Some additional facilities for any graphics library. Class
    functions may be overloaded for adaptation to concrete graphic
    interface (BGI, GDI, and so on).
	Functions could set error code in kh_error_code variable.
	The zoom and addzoom are deformation coeficients. You could
    use zoom in any part of program. Addzoom is used only once to
    set the additional deformation of the whole picture (for example
    if context is changed from screen to printer), for preview and so on.
*/

class Paint : public BGI_Font, public Trigonometry
    {
    protected:
	loc zoom;                     // Image deformation
	loc add_zoom;                 // Additional deformation
	loc lt;                       // Scroll of part of picture
	loc add_scroll;               // Left - top clip of the whole picture
	int fill;                     // Fill flag
	int mirror;
    public:
	loc center;                   // Rotation center
	int alpha;                    // Rotation angle
	bool R_STACK;                 // Use or not stack of rotations
	Stack* r_stack;               // Stack of rotations

	Paint();
	~Paint() { delete r_stack; }

	loc get_add_zoom() { return add_zoom; }
	loc get_zoom() { return zoom; }

	void set_mirror(int m) { mirror = m; } // X coord. if no rotation
	void set_stack(bool r) { R_STACK = r; r_stack->flash();
	    rotate(loc(0, 0), 0); }
	void set_stack_soft(bool r) { R_STACK = r; }
	void set_zoom(double x, double y) { zoom.X = x * add_zoom.X;
	    zoom.Y = y * add_zoom.Y; }    // No out-of-range control !!!
	void set_add_zoom(double x, double y) { add_zoom.X = 100 * x;
	    add_zoom.Y = 100 * y; }
	void set_scroll(int x, int y) { lt.X = x; lt.Y = y; }
	void set_add_scroll(int x, int y) { add_scroll.X = x;
	    add_scroll.Y = y; }
	void rotate(loc c, int a);
	void rotate(int x, int y, int alpha) { rotate(loc(x, y), alpha); }
	void set_fill(bool f) { fill = f; }

// Using alpha and center returns rotated coordinates
	loc rot(int x, int y);
// Return completely transformed point: rotated, zoomed and scrolled
	loc transform(int x, int y);
    };


#endif __PAINT_H_

