#include <dir.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <io.h>
#include <fcntl.h>
#include <sys\stat.h>
#include "sprlib.h"
#include "fileutil.h"

_pascal_ zSprHeader::zSprHeader( char *name, WORD size )
    :Size(size)
{
    if ( name )
        strncpy( Name, name, MAXFILE+MAXEXT );
}

_pascal_ zSprLib::zSprLib( char *name )
    :head(0), cur(0), CurIdx(0), tail(0), SpriteCount(0),
     opened(-1)
{
    char *t = "";
    if ( name ) t = name;
    strncpy( LibName, t, MAXPATH );
}

_pascal_ zSprLib::~zSprLib()
{
    if ( opened ) close( opened );
}

int _pascal_ zSprLib::Init( void )
{
    if ( opened >= 0 ) close(opened);

    opened = open( LibName, O_RDWR|O_BINARY|O_TRUNC|O_CREAT,
                   S_IREAD|S_IWRITE );
    if ( opened == -1 ) {
        printf("opened %s failed.\n", LibName);
    }
    return opened;
}

int _pascal_ zSprLib::AddSpr( char _FAR_ *buf, WORD size, char *sname )
{
    int i = SpriteCount;
    zSprData *temp = new zSprData;
    zSprHeader &h = temp->SprHeader;
    if ( !temp ) {
        printf( "new failed in AddSpr.\n" );
        return -1;
    } else {
        if ( !head ) {
            head = cur = tail = temp;
            CurIdx = 0;
        } else {
            tail->next = temp;
            tail = temp;
        }
        h.Size = size;
        strncpy( h.Name, sname, MAXFILE+MAXEXT );
        temp->Buf = buf;
        SpriteCount ++;
    }
    return i;
}

int _pascal_ zSprLib::InsertSpr( char _FAR_ *buf, WORD size, char *sname,
                                 char *insName )
{
    zSprData *trail, *temp;
    trail = temp = head;
    int i = 0;
    while( temp ) {
        if( strcmp( temp->SprHeader.Name, insName ) == 0 ) break;
        trail = temp;
        temp = temp->next;
        i ++;
    }
    zSprData *NewTmp = 0;
    if ( temp == head ) {
        NewTmp = head = new zSprData;
        head->next = temp->next;
    } else if ( temp ) {
        NewTmp = new zSprData;
        NewTmp->next = temp;
        trail->next = NewTmp;
    }
    if ( NewTmp ) {
        NewTmp->SprHeader.Size = size;
        NewTmp->Buf = new char _FAR_ [size];
        memcpy( NewTmp->Buf, buf, size );
        strncpy( NewTmp->SprHeader.Name, sname, MAXFILE+MAXEXT );
        return i;
    }
    return -1;
}

int _pascal_ zSprLib::DelSpr( char *sname )
{
    zSprData *trail, *temp;
    trail = temp = head;
    int i = 0;
    while( temp ) {
        if ( strcmpi( temp->SprHeader.Name, sname ) == 0 ) break;
        trail = temp;
        temp = temp->next;
        i ++;
    }
    if ( temp == head ) {
        head = temp->next;
        if ( !head ) {
             head = tail = cur = 0;
             CurIdx = 0;
        }
    } else if ( temp == tail ) {
        trail->next = 0;
        tail = trail;
    } else if ( temp ) {
        trail->next = temp->next;
    }
    if ( cur == temp ) {
        cur = head;
        CurIdx = 0;
    }
    if ( temp ) {
        delete temp;
        SpriteCount --;
        return i;
    } else
        return -1;
}

int _pascal_ zSprLib::WriteLib( void )
{
    zSprData * temp = head;
    if ( Init() > -1 ) {
        int i = 0, size;
        WORD sig = SPRLIB_SIG;
        _write( opened, &sig, sizeof( sig ));
        _write( opened, &SpriteCount, sizeof(SpriteCount) );
        sig = SPRHEADER_SIG;
        while( temp ) {
            if( temp->Buf ) {
                _write( opened, &sig, sizeof( sig ));
                _write( opened, &temp->SprHeader, sizeof( zSprHeader ));
                size = _write( opened, temp->Buf, temp->SprHeader.Size );
                if ( size != temp->SprHeader.Size ) {
                    printf("WriteLib: size %d and temp->Size %d mismatched.\n",
                                      size, temp->SprHeader.Size );
                }
            }
            temp = temp->next;
            i ++;
        }
        if ( i != SpriteCount ) {
            printf("Count i %d and SprCount %d mismatched.\n", i, SpriteCount );
        }
        close(opened);
        opened = -1;
        return i;
    }
    return -1;
}

int _pascal_ zSprLib::ReadLib( int Type, int BeforeWhich )
{
    char _FAR_ *buf;
    zSprHeader temp;

    if ( opened > -1 ) close ( opened );
    opened = open( LibName, O_RDONLY|O_BINARY, S_IREAD );
    if ( opened > -1 ) {
        int i = 0, size = 0;
        int SprCount;
        WORD sig;
        _read( opened, &sig, sizeof( sig ));
        if ( sig != SPRLIB_SIG ) {
            printf("Invalid LIB header sig.\n");
            return -1;
        }
        _read( opened, &SprCount, sizeof(SprCount));
        for( i = 0; i < SprCount; i ++ ) {
            size = _read( opened, &sig, sizeof( sig ));
            if ( sig != SPRHEADER_SIG ) {
                printf("Invalid header sig.\n");
                return -1;
            }
            if ( size == -1 ) {
                printf( "Read error.\n");
                return -1;
            }
            if ( size == 0 )
                break;
            size = _read( opened, &temp, sizeof( zSprHeader ));
            if ( size == -1 ) {
                printf( "Read error.\n");
                return -1;
            }
            if ( size == 0 )
                break;
            buf = new char _FAR_ [temp.Size];
            size = _read( opened, buf, temp.Size );
            if ( size == -1 ) {
                printf( "Read error.\n");
                return -1;
            }
            AddSpr( buf, temp.Size, temp.Name );
            if ( size == 0 )
                break;
        }
        if ( i != SprCount ) {
            printf( "%d %d unmatched counts.\n", i, SprCount );
        }
        close ( opened);
        opened = -1;
        return i;
    }
    return -1;
}

int _pascal_ zSprLib::ReadSpr( char *sname, int Type, int BWhich )
{
    char _FAR_ *buf;
    int size;
    buf = readfile( sname, size );
    if ( buf ) {
        return AddSpr( buf, size, sname );
    }
    return -1;
}

int _pascal_ zSprLib::WriteSpr( char *sname, char *fname )
{
    int i;
    zSprData * temp = FindSpr( sname, i );
    if ( temp ) {
        if ( temp->Buf ) {
            unsigned r = writefile( fname, temp->Buf, temp->SprHeader.Size );
            if ( r < 1 )
               printf("Write failed.\n");
            else
                return i;
        }
    }
    return -1;
}

char * _pascal_ zSprLib::GetFirst( void )
{
    if ( head ) {
        cur = head->next;
        CurIdx = 1;
        return head->SprHeader.Name;
    }
    return 0;
}

char * _pascal_ zSprLib::GetNext( void )
{
    if ( cur ) {
        char *n = cur->SprHeader.Name;
        cur = cur->next;
        CurIdx ++;
        return n;
    }
    return 0;
}

zSprData * _pascal_ zSprLib::FindSpr( char *sname, int &i )
{
    zSprData * temp = head;
    i = 0;
    if ( cur && strcmpi( cur->SprHeader.Name, sname) == 0 ) {
        i = CurIdx;
        return cur;
    }
    while( temp ) {
        if( strcmpi( temp->SprHeader.Name, sname ) == 0 ) break;
        temp = temp->next;
        i ++;
    }
    return temp;
}

int _pascal_ zSprLib::SprExist( char *sname )
{
    int i;
    zSprData *t = FindSpr( sname, i );
    if ( t ) {
        cur = t;
        CurIdx = i;
        return i;
    }
    return -1;
}


int _pascal_ zSprLib::SwapSpr( char *sname1, char *sname2 )
{
    zSprData *temp1, *temp2;
    int i1, i2;
    temp1 = FindSpr( sname1, i1 );
    temp2 = FindSpr( sname2, i2 );
    if( temp1 && temp2 ) {
        zSprData swp;
        memcpy( &swp, temp1, sizeof( zSprData ));
        memcpy( temp1, temp2, sizeof( zSprData ));
        memcpy( temp2, &swp, sizeof( zSprData ));
        return 0;
    } else {
        return -1;
    }
}


int _pascal_ zSprLib::PutCurSpr( int Idx, char _FAR_ *buf, WORD size )
{
    if ( Idx != CurIdx ) return -1;
    if ( size != cur->SprHeader.Size ) {
        cur->SprHeader.Size = size;
        delete cur->Buf;
        cur->Buf = new char _FAR_[size];
        if ( !cur->Buf ) return -1;
    }
    memcpy( cur->Buf, buf, size );
    return Idx;
}

int _pascal_ zSprLib::PutSpr( char _FAR_ *buf, WORD size, char *sname )
{
    int i;
    zSprData *temp = FindSpr( sname, i );
    if ( temp ) {
        if ( size != temp->SprHeader.Size ) {
            temp->SprHeader.Size = size;
            delete temp->Buf;
            temp->Buf = new char _FAR_ [size];
            if ( !temp->Buf) return -1;
        }
        memcpy( temp->Buf, buf, size );
        return i;
    } else {
        return AddSpr( buf, size, sname );
    }
}

const char _FAR_ * _pascal_ zSprLib::GetSpr( char *sname, WORD &size )
{
    int i;
    zSprData *temp = FindSpr( sname, i );
    if ( temp ) {
        size = i;
        return temp->Buf;
    }
    size = 0;
    return 0;
}

char * _pascal_ zSprLib::GetNth( int i )
{
    zSprData *temp = head;
    int j = 0;
    while( j != i && temp ) {
        temp = temp->next;
        j ++;
    }
    if ( temp ) {
        cur = temp;
        CurIdx = i;
        return cur->SprHeader.Name;
    }
    return 0;
}

#define TEST
#ifdef TEST

#include <conio.h>

main( int argc, char **argv )
{
    if ( argc < 2 ) {
        printf( "\nUsage: slibdir filename [savename] [!]\n\n");
        printf( "    If savename is given and a sprite with the same\n");
        printf( "    name is found in the library, then the sprite will\n");
        printf( "    be saved as a PBM file.  If '!' is given, then\n");
        printf( "    'savename' is treated as an index of the sprite\n");
        printf( "    you want to save.\n");
        return 1;
    }
    zSprLib myLib( argv[1] );
    myLib.ReadLib();
    int i = 0, idx;
    if( argc == 4)
        idx = atoi(argv[2]);
    char *n = myLib.GetFirst();
    while( n ) {
        printf("%5d: %s\n", i, n);
        if ( argc == 3 && strcmpi( n, argv[2] ) == 0 ) {
            myLib.WriteSpr( n, n );
        } else if ( argc == 4 && i == idx )
            myLib.WriteSpr( n, n );
        i ++;
        n = myLib.GetNext();
    }
    return 0;
}

#endif

