*=================================================================
* STANDARD.PRG
* Copyright (c) 1996-1997   Daniele Giacomini
*
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU General Public License as
* published by the Free Software Foundation; either version 2 of
* the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public
* License along with this program; if not, write to the Free
* Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139,
* USA.
*
*=================================================================
* STANDARD.PRG
*
* This file contains some standard functions.
*
*
* This source uses a mouse library:
*
*      ClipMous.LIB
*      Simple library for mouse support under Clipper
*      (c) Martin Brousseau
*      FREEWARE
*      original package: CLIPMOUS.ZIP
*
*
*=================================================================
* Functions:
*
*
* accept()              ACCEPT() with window.
*
* aChoice()             ACHOICE() with mouse support.
*
* aChoiceWindow()       aChoice() with header and borders.
*
* alertBox()            ALERT() with mouse support.
*
* alertButtons()        Similar to ALERTBOX() without message and
*                       a specific screen position.
*
* atb()                 Array TBrowse.
*
* bCompile()            Code block compile.
*
* button()              Add button for the new read().
*
* cm()                  Macro "compiler".
*
* colorArray()          Returns an array of colors.
*
* coordinate()          Recalculate window coordinate
*                       to align windows before display.
*
* copyFile()            COPY FILE
*
* dbApp()               APPEND FROM
*
* dbClose()             DBCLOSEALL() for macros.
*
* dbContinue()          CONTINUE
*
* dbCopy()              COPY TO
*
* dbCopyStruct          COPY STRUCTURE TO
*
* dbCopyXStruct         COPY STRUCTURE EXTENDED
*
* dbDelim()             COPY TO | APPEND FROM  DELIMITED
*
* dbJoin()              JOIN WITH
*
* dbLabelForm()         LABEL FORM
*
* dbList()              LIST
*
* dbLocate()            LOCATE
*
* dbOldCreate()         CREATE ... FROM
*
* dbPack()              PACK
*
&&&&&&&* dbReportForm()        REPORT FORM
*
* dbSdf()               COPY TO | APPEND FROM  SDF
*
* dbSort()              SORT TO
*
* dbTotal()             TOTAL ON
*
* dbUpdate()            UPDATE FORM
*
* dbZap()               ZAP
*
* dbiAllStatus()        Status information on all Aliases.
*
* dbiStatus()           Active Alias status information.
*
* dbiStructure()        Active Alias structure information.
*
* default()             DEFAULT command substitute.
*
* dispBoxColor()        Used with DispBoxShadow()
*
* dispBoxShadow()       DISPBOX() with two colors.
*
* dir()                 Directory multipurpose function.
*
* dotLine()             Pop-up calculator/dot line.
*
* dteMonth()            Translate the month into a different
*                       language.
*
* dteWeek()             Translate the month into a dirrentet
*                       language.
*
* errorHandler()        General error handler.
*
* errorMacro()          Error handler for Macros.
*
* ex()                  Macro executor (run).
*
* get()                 @... GET with mouse support
*
* gvAdd()               Get validation: Add data to the field.
*
* gvDefault()           Get validation: Default data for empty
*                       fields.
*
* gvFileDir()           Get validation: Search for a file.
*
* gvFileExist()         Get validation: Check if a file exists.
*
* gvFileExtention()     Get validation: Add a default extention.
*
* gvSubst()             Get validation: Field content
*                       substitution.
*
* htf()                 Help text file reader.
*
* htfGenerate()         Help text file generator.
*
* htf2Html()            Translate a HTF text file into HTML.
*
* htmlTranslate()       Translate text into HTML.
*
* isFile()              Check if the file name is valid and if
*                       exists.
*
* isWild()              Check if a string contains "*" or "?".
*
* isMemvar()            Check if a Memvar exists.
*
* isConsoleOn()         Check if qOut() and qqOut() will print
*                       on the console.
*
* isPrinterOn()         Check if qOut() and qqOut() will print
*                       on the printer.
*
* keyboard()            KEYBOARD
*
* memoWindow()          A modifyed Memoedit().
*
* memPublic()           PUBLIC
*
* memRelease()          RELEASE
*
* memRestore()          RESTORE FROM
*
* memSave()             SAVE TO
*
* menuPrompt()          @... PROMPT with mouse support.
*
* menuTo()              MENU TO with mouse support.
*
* messageLine()         A message line appearing on the screen.
*
* mouse()               A special function for mouse handling.
*
* mouseScrSave()        Save screen with a mouse hide before.
*
* mouseScrRestore()     Restore screen with a mouse hide before.
*
* picChrMax()           A picture for char files.
*
* quit()                QUIT
*
* read()                New readmodal with mouse support.
*
* readStop()            READKILL() for READ()
*
* rf()                  Report Form in a different way.
*
* rpt()                 Print a text with print command.
*
* rptMany()             Print a text using the active Alias.
*
* rptTranslate()        Translate a character string containing
*                       variables.
*
* run()                 RUN | !
*
* say()                 @... SAY
*
* setColorStandard()    set up standard color.
*
* setFunction()         SET FUNCTION
*
* setMouse()            Show or Hide the mouse cursor.
*
* setOutput()           A Complex output set up.
*
* setRptEject()         Used with rpt().
*
* setRptLines()         Used with rpt().
*
* setVerb()             A verbose set()
*
* strAddExtention()     Add extention to a file name.
*
* strCutExtention()     Cut the extention from a file name.
*
* strDrive()            Extract the disk drive name form
*                       a pathname.
*
* strExtention()        Extract the extention form a pathname.
*
* strFile()             Extract the file name form a pathname.
*
* strFileFind()         Try to find a file from a group of
*                       paths.
*
* strGetLen             Calculate the length of a possible get
*                       field from a given picture.
*
* strListAsArray        Returns an array from the given list.
*
* strOccurs()           Pattern occurrence inside a string.
*
* strParent()           Return a parent path.
*
* strPath()             extract the path from a pathname.
*
* strTempPath()         Returns a path for temporary files.
*
* strXToString()        Convert any data into string.
*
* tab()                 Move cursor pointer ( col(), row() ) to
*                       the next tab.
*
* tb()                  Db TBrowse.
*
* text()                Text display on the screen.
*
* tglInsert()           Toggle insert/overwrite.
*
* timeX2N()             Calculate a time number form Hours, Minutes
*                       and Seconds.
*
* timeN2H()             Extract Hours from a time number (also if
*                       more than 24).
*
* timeN2M()             Extract Minutes from a time number.
*
* timeN2S()             Extract Seconds from a time number (Seconds
*                       may contain decimals).
*
* true()                A function that returns ever true.
*
* trueSetKey()          A SetKey() that returns ever ture.
*
* waitFor()             A wait window.
*
* waitFileEval()        A wait bar for file process.
*
* waitProgress()        A wait bar.
*
* waitWheel()           A wait wheel.
*
*=================================================================
*
*=================================================================
* NAME TRANSLATION
*=================================================================




*=================================================================
* FILE INCLUSION
*=================================================================

#include "standard.ch"

*=================================================================
* STATIC
*=================================================================

* Used by CM, ERRORMACRO
static errorChoice

*=================================================================
* ACCEPT
*=================================================================

#define ACCEPT_BUTTON_ESC_CANCEL;
    "[Esc] Cancel"
#define ACCEPT_BUTTON_F1_HELP;
    "[F1] Help"
#define ACCEPT_BUTTON_PGDN_CONFIRM;
    "[Pag] Confirm"

#define ACCEPT_HELP;
    "Accept()" +;
    NL(3) +;
    "This function permits you to insert a simple information." +;
    NL(3) +;
    "[Esc]      Exit without data input;" +;
    NL(2) +;
    "[F1]       this help;" +;
    NL(2) +;
    "[Pag]     confirm input."

*=================================================================
function Accept( xField, cMessage, cHeader )
*
* Accept( <Field>, [<cMessage>], [<cHeader>] ) --> Field|NIL
*
* <Field>      the variable to be edited and returned.
*
* <cMessage>   the message to show.
*
* <cHeader>    the header to show at the top window.
*
* The function shows <cMessage> and waits for input to place
* into <Field>.
*

    local aoGet             := {}
    local aButton           := {}
    local cOldScreen
    local cOldColor         := setcolor()
    local nOldCursor        := setcursor( SETCURSOR_NORMAL )
    local nOldRow           := row()
    local nOldCol           := col()
    local bOld_F1       :=;
        setkey( K_F1, { || Text( ACCEPT_HELP )} )
    local nTop
    local nLeft
    local nBottom
    local nRight
    local nWidth
    local nLines
    local nLine
    local xReturn           := NIL

    default( @cHeader,      "Input" )

    *-------------------------------------------------------------
    * Start a new sequence
    *-------------------------------------------------------------

    begin sequence

        *---------------------------------------------------------
        * Check if <xField> is a editable field. If not, break.
        *---------------------------------------------------------

        if  (;
                valtype( xField ) == "C" .or.;
                valtype( xField ) == "N" .or.;
                valtype( xField ) == "D" .or.;
                valtype( xField ) == "L";
            )
            // Ok, valid xField
        else
            
            break                                       // BREAK

        end

        *---------------------------------------------------------
        * Check if <cMessage> is valid.
        *---------------------------------------------------------

        if valtype( cMessage ) <> "C"
            cMessage := ""
        end

        *---------------------------------------------------------
        * Calculate field dimention.
        *---------------------------------------------------------

        nWidth := maxcol()+1

        *---------------------------------------------------------
        * Calculate how many lines are needed to display
        * <cMessage>.
        *---------------------------------------------------------

        cMessage := alltrim(cMessage)
        nLines := mlcount( cMessage, nWidth-2 )

        *---------------------------------------------------------
        * Create a kind of window.
        *---------------------------------------------------------

        nBottom     := maxrow()
        nTop        := nBottom - ( nLines+6 )
        nLeft       := 0
        nRight      := maxcol()

        cOldScreen  := mouseScrSave( nTop, nLeft, nBottom, nRight )

        setcolor( COLOR_BODY )

        scroll( nTop, nLeft, nBottom, nRight )

        dispBoxShadow(;
            nTop, nLeft, nBottom, nRight,;
            1,;
            dispBoxColor( 1 ),;
            dispBoxColor( 2 );
        )

        say(;
            nTop+1, nLeft+1,;
            padc( cHeader, nWidth-2 ),;
            ,;
            COLOR_HEAD;
        )

        *---------------------------------------------------------
        * Show <cMessage>.
        *---------------------------------------------------------

        setpos( row()+1, col() )

        for nLine := 1 to nLines
            say(;
                row()+1, nLeft+1,;
                memoline(cMessage, nWidth-2, nLine );
            )
        end

        *---------------------------------------------------------
        * Do editing.
        *---------------------------------------------------------

        setpos( row()+1, col() )

        while .T.                                       // FOREVER

            if valtype( xField ) == "C"
                get(;
                    @aoGet, row(), nLeft+1,;
                    { |x| iif( pcount() > 0, xField := x, xField ) },;
                    picChrMax( nLeft+1, nRight-1);
                )
            else
                get(;
                    @aoGet, row(), nLeft+1,;
                    { |x| iif( pcount() > 0, xField := x, xField ) };
                )
            end

            button( @aButton, row()+2, nLeft+1,;
                    ACCEPT_BUTTON_ESC_CANCEL, , {||keyboard(chr(K_ESC))} )
            button( @aButton, row(), col()+1,;
                    ACCEPT_BUTTON_F1_HELP, , {|| eval(setkey(K_F1))} )
            button( @aButton, row(), col()+1,;
                    ACCEPT_BUTTON_PGDN_CONFIRM, , {||keyboard(chr(K_PGDN))} )

            *-----------------------------------------------------
            * Read.
            *-----------------------------------------------------

            read( aoGet, , aButton )
            aoGet := {}
            aButton := {}

            *-----------------------------------------------------
            * Check what was read.
            *-----------------------------------------------------

            do case
            case lastkey() = K_ESC

                *-------------------------------------------------
                * [Esc] means leave.
                *-------------------------------------------------

                xReturn := NIL

                exit                                    // EXIT

            case lastkey() = K_PGDN

                *-------------------------------------------------
                * [PgDn] means confirm.
                * Before, check for valid data.
                *-------------------------------------------------

                xReturn := xField

                exit                                    // EXIT

            otherwise

                *-------------------------------------------------
                * Exit with different keys is not allowed.
                * Loop again.
                *-------------------------------------------------

            end

        end

        *---------------------------------------------------------
        * Restore screen deleting the window.
        *---------------------------------------------------------

        mouseScrRestore( nTop, nLeft, nBottom, nRight, cOldScreen )

    end //sequence

    *-------------------------------------------------------------
    * Restore old values.
    *-------------------------------------------------------------

    setcolor( cOldColor )
    setcursor( nOldCursor )
    setpos( nOldRow, nOldCol )
    setkey( K_F1, bOld_F1 )

    return xReturn

*=================================================================
* ACHOICE()
* ACHOICEWINDOW()
*=================================================================

#define ACHOICE_BUTTONS {;
    { nBottom+1, nLeft+0,  "[Esc] Exit",    {|| keyboard( chr(K_ESC)       ) } },;
    { nBottom+1, nLeft+11, "[Pag]",        {|| keyboard( chr(K_PGUP)      ) } },;
    { nBottom+1, nLeft+18, "[Ctrl]+[Pag]", {|| keyboard( chr(K_CTRL_PGUP) ) } },;
    { nBottom+2, nLeft+0,  "[F1] Help ",    {|| keyboard( chr(K_F1)        ) } },;
    { nBottom+2, nLeft+11, "[Pag]",        {|| keyboard( chr(K_PGDN)      ) } },;
    { nBottom+2, nLeft+18, "[Ctrl]+[Pag]", {|| keyboard( chr(K_CTRL_PGDN) ) } };
    }

#define ACHOICE_WINDOW_HELP;
    "aChoiceWindow()" +;
    NL(3) +;
    "This function lets you select an item form a list." +;
    NL(3) +;
    "[Esc]          Leave without selecting anything." +;
    NL(2) +;
    "[Enter]        Select what appears under the cursor." +;
    NL(2) +;
    "[]/[]        Move the cursor up or down." +;
    NL(1) +;
    "[Pag]/[Pag]  Move the cursor page up or page down." +;
    NL(1) +;
    "[Ctrl]+[Pag]  Move the cursor top." +;
    NL(1) +;
    "[Ctrl]+[Pag]  Move the cursor bottom."

*=================================================================
function aChoice(;
        nTop, nLeft, nBottom, nRight,;
        acMenuItems, alSelectable, nFirst,;
        xButtons;
    )
*
* aChoice(;
*    [<nTop>], [<nLeft>], [<nBottom>], [<nRight>],;
*    <acMenuItems>, [<alSelectable>], [<nFirst>],;
*    [<lButtons> | <aButtons>];
* ) --> nPosition
*
* <nTop>, <nLeft>, <nBottom>, <nRight>
*                      are the display coordinates where the list
*                      may appears.
* <acMenuItems>        is the array of character strings to
*                      display as the menu items.
* <alSelectable>       is a parallel array of logical values
*                      that specify the selectable menu items.
* <nFirst>             is the position in the <acMenuItems> array
*                      of the item that will be higlighted when
*                      the menu is initially displayed.
* <lButtons>           specify if a default array of buttons
*                      should appear at the bottom of the
*                      specified display area. If true, buttons
*                      will appear inside the area.
* <aButtons>           is an optional array of buttons that
*                      should appear outside the aChoice()
*                      display area.
*
*      aButtons[n][1]  N,      the nth button row position.
*      aButtons[n][2]  N,      the nth button column position.
*      aButtons[n][3]  C,      the nth button text.
*      aButtons[n][4]  B,      the nth button code block.
*
*
* -->                  is the selected item number. If no
*                      item is specified, zero is returned.
*
* This function is a substitution of the original CA-Clipper
* aChoice(), to support the mouse.
*

    local lOldSetMouse          := setMouse( .F. )
    local oBrow
    local oCol
    local nKey
    local nSubscript
    local nChoice       := 0
    local lExit
    local nI
    local acMenuClone

    default( @nTop,             0 )
    default( @nLeft,            0 )
    default( @nBottom,          maxrow() )
    default( @nRight,           maxcol() )
    default( @acMenuItems,      {} )
    default( @alSelectable,     afill( array( len( acMenuItems ) ), .T. ) )
    default( @nFirst,           1 )
    default( @xButtons,         .F. )

    *-------------------------------------------------------------
    * Start a new sequence
    *-------------------------------------------------------------

    begin sequence

        *---------------------------------------------------------
        * Check if <acMenuItems> is valid. If not, break.
        *---------------------------------------------------------

        if len( acMenuItems ) == 0

            break                                       // BREAK

        else

            *-----------------------------------------------------
            * Clone <acMenuItem> and correct space differences.
            * As arrays are passed by reference, this array clone
            * must be done to avoid troubles with the original
            * data when the function terminates.
            *-----------------------------------------------------

            acMenuClone := aclone( acMenuItems )
            aeval(;
                acMenuClone,;
                {;
                    |xValue, nI|;
                    acMenuClone[nI] :=;
                        padr( acMenuClone[nI], nRight-nLeft+1 );
                    };
                )
        end

        *---------------------------------------------------------
        * Check <xButtons>.
        *---------------------------------------------------------

        do case
        case valtype( xButtons ) == "L"

            if xButtons

                *-------------------------------------------------
                * As button are placed automatically,
                * the space is taken form the bottom
                * area:
                *-------------------------------------------------

                nBottom := nBottom-2

                *-------------------------------------------------
                * Then, buttons are placed beyond the new nButtom
                * line.
                *-------------------------------------------------

                xButtons := ACHOICE_BUTTONS

            end

        case valtype( xButtons ) == "A"

            if empty( xButtons )

                xButtons := .F.

            end

        otherwise

            xButtons := .F.

        end

        *---------------------------------------------------------
        * If Buttons are ready, it is the time to show them.
        *---------------------------------------------------------

        if valtype( xButtons ) == "A"

            for nI := 1 to len( xButtons )

                say(;
                    xButtons[nI][1], xButtons[nI][2],;
                    xButtons[nI][3],;
                    ,;
                    COLOR_BUTTON;
                )

            next

        end

        *---------------------------------------------------------
        * Prepare the starting position.
        *---------------------------------------------------------

        nSubscript := nFirst

        *---------------------------------------------------------
        * Create the TBrowse object and make the necessary
        * instances.
        *---------------------------------------------------------

        oBrow := tbrowsenew( nTop, nLeft, nBottom, nRight )

        oBrow:skipBlock     :=;
            {|nReq| aChoiceJumpIt(nReq, @nSubscript, acMenuClone)}

        oBrow:goTopBlock    := {|| nSubscript := 1}

        oBrow:goBottomBlock := {|| nSubscript := len(acMenuClone)}

        *---------------------------------------------------------
        * Create the Column object and add to the TBrowse.
        *---------------------------------------------------------

        oCol := tbcolumnnew( , {|| acMenuClone[nSubscript]})
        oBrow:addColumn(oCol)

        *---------------------------------------------------------
        * Start the choice loop.
        *---------------------------------------------------------

        lExit := .F.
        while !lExit

            oBrow:forceStable()

            *-----------------------------------------------------
            * Keyborad loop.
            *-----------------------------------------------------

            while .T.                                   // FOREVER

                *-------------------------------------------------
                * Show the mouse cursor.
                *-------------------------------------------------

                setMouse( .T. )

                *-------------------------------------------------
                * Check for mouse click.
                *-------------------------------------------------

                if !( mouse() == NIL )

                    aChoiceMouseKeyboard(;
                        xButtons,;
                        nTop, nLeft, nBottom, nRight,;
                        nSubscript, len(acMenuClone);
                    )
                    mouse( .T. )
                end

                *-------------------------------------------------
                * Read the keyboard without waiting.
                *-------------------------------------------------

                nKey := inkey()

                *-------------------------------------------------
                * If a key is inside the keyboard buffer, the
                * mouse cursor is hidden before any action is
                * taken.
                *-------------------------------------------------

                if nKey <> 0

                    setMouse( .F. )

                end

                *-------------------------------------------------
                * Do an action depending on the pressed key.
                *-------------------------------------------------

                do case
                case !( setkey( nKey ) == NIL )

                    *---------------------------------------------
                    * The key was redirected, so execute the
                    * code block.
                    *---------------------------------------------

                    eval( setkey( nKey ) )

                    exit                                // EXIT

                case nKey == K_DOWN

                    oBrow:down()

                    exit                                // EXIT

                case nKey == K_UP

                    oBrow:up()

                    exit                                // EXIT

                case nKey == K_PGDN

                    oBrow:pageDown()

                    exit                                // EXIT

                case nKey == K_PGUP

                    oBrow:pageUp()

                    exit                                // EXIT

                case nKey == K_CTRL_PGUP

                    oBrow:goTop()

                    exit                                // EXIT

                case nKey == K_CTRL_PGDN

                    oBrow:goBottom()

                    exit                                // EXIT

                case nKey == K_RIGHT

                    oBrow:right()

                    exit                                // EXIT

                case nKey == K_LEFT

                    oBrow:left()

                    exit                                // EXIT

                case nKey == K_HOME

                    oBrow:home()

                    exit                                // EXIT

                case nKey == K_END

                    oBrow:end()

                    exit                                // EXIT

                case nKey == K_CTRL_LEFT

                    oBrow:panLeft()

                    exit                                // EXIT

                case nKey == K_CTRL_RIGHT

                    oBrow:panRight()

                    exit                                // EXIT

                case nKey == K_CTRL_HOME

                    oBrow:panHome()

                    exit                                // EXIT

                case nKey == K_CTRL_END

                    oBrow:panEnd()

                    exit                                // EXIT

                case nKey == K_ESC

                    *---------------------------------------------
                    * [Esc] means leave.
                    *---------------------------------------------

                    nChoice         := 0

                    lExit           := .T.

                    exit                                // EXIT

                case nKey == K_ENTER

                    *---------------------------------------------
                    * [Enter] means select.
                    *---------------------------------------------

                    nChoice         := nSubscript

                    *---------------------------------------------
                    * If it is a valid choice, exit also previous
                    * loop.
                    *---------------------------------------------

                    if alSelectable[nChoice]

                        lExit       := .T.

                    end

                    exit                                // EXIT

                otherwise

                    *---------------------------------------------
                    * Another key is not allowed, so repeat loop.
                    *---------------------------------------------

                end

            end

            *-----------------------------------------------------
            * If [Esc] or [Enter] with a valid choice was
            * pressed, the previous loop must end.
            *-----------------------------------------------------

            if lExit

                exit                                    // EXIT

            end

        end

    end //sequence

    *-------------------------------------------------------------
    * Restore old data.
    *-------------------------------------------------------------

    setMouse( lOldSetMouse )

    *-------------------------------------------------------------
    * Return the selected item.
    *-------------------------------------------------------------

    return nChoice

*-----------------------------------------------------------------
static function aChoiceJumpIt(nRequest, nSubscript, aArray)
*
*
    local nActually := 0

    do case
    case nRequest == 0

        *---------------------------------------------------------
        * No movement was required: do nothing.
        *---------------------------------------------------------

    case nRequest > 0

        *---------------------------------------------------------
        * Skip forward.
        *
        * The skip cannot go over the end of the array.
        *---------------------------------------------------------

        if nRequest < (len(aArray) - nSubscript)
            nActually := nRequest
        else
            nActually := len(aArray) - nSubscript
        end

    case nRequest < 0

        *---------------------------------------------------------
        * Skip backwards.
        *
        * The skip cannot go back before the top ot the array.
        *---------------------------------------------------------

        if nRequest < (1 - nSubscript)
            nActually := 1 - nSubscript
        else
            nActually := nRequest
        end

    end

    *-------------------------------------------------------------
    * Correct the <nSubscript> value.
    *-------------------------------------------------------------

    nSubscript += nActually

    *-------------------------------------------------------------
    * Return the real skip.
    *-------------------------------------------------------------

    return nActually

*-----------------------------------------------------------------
static function aChoiceMouseKeyboard(;
        aButtons,;
        nTop, nLeft, nBottom, nRight,;
        nSubscript, nMaxElements;
    )
*
*
    local aMouse    := mouse()

    local nMCol     := aMouse[1]-1
    local nMRow     := aMouse[2]-1

    local nTimes    := 0
    local cKeyboard := ""
    local nI

    local nButtRow
    local nButtCol
    local nButtColEnd

    *-------------------------------------------------------------
    * If buttons are there, check if a button was selected.
    *-------------------------------------------------------------

    if valtype( aButtons ) == "A"

        for nI := 1 to len(aButtons)
            nButtRow    := aButtons[nI][1]
            nButtCol    := aButtons[nI][2]
            nButtColEnd := nButtCol + len( aButtons[nI][3] ) -1

            if nButtRow == nMRow;
                .and. nButtCol <= nMCol;
                .and. nButtColEnd >= nMCol

                *-------------------------------------------------
                * Ok, this button was pressed. Do the action.
                *-------------------------------------------------

                eval( aButtons[nI][4] )

                *-------------------------------------------------
                * No more check. Return.
                *-------------------------------------------------

                return NIL                              // RETURN

            end

        next

    end

    *-------------------------------------------------------------
    * If it wasn't a button, 
    * test if the mouse was inside the possible area.
    *-------------------------------------------------------------

    if nMRow >= nTop;
        .and. nMRow <= nBottom;
        .and. nMCol >= nLeft;
        .and. nMCol <= nRight

        *---------------------------------------------------------
        * Ok, the mouse was inside the right area.
        *---------------------------------------------------------

    else

        *---------------------------------------------------------
        * The mouse was outside the right area. Return.
        *---------------------------------------------------------

        return NIL                                      // RETURN

    end

    *-------------------------------------------------------------
    * If still here,
    * Check for a selection.
    *-------------------------------------------------------------

    do case
    case nMRow > row();
        .and.;
        (nMaxElements - nSubscript) >= nMRow - row()
        

        *---------------------------------------------------------
        * The mouse cursor is after the TBrowse cursor and
        * not over the end of the array:
        * the element can be selected.
        * Stuff inside the keyboard buffer as many [Down]
        * as needed plus a final [Enter].
        *---------------------------------------------------------

        nTimes := nMRow - row()
        for nI := 1 to nTimes
            cKeyboard += chr( K_DOWN )
        end
        cKeyboard += chr( K_ENTER )
        keyboard( cKeyboard )

    case nMRow < row()

        *---------------------------------------------------------
        * The mouse cursor is before the TBrowse cursor
        * (it cannot be before the top of the array).
        * The element can be selected.
        * Stuff inside the keyboard buffer as many [Up]
        * as needed plus a final [Enter].
        *---------------------------------------------------------

        nTimes := row() - nMRow
        for nI := 1 to nTimes
            cKeyboard += chr( K_UP )
        end
        cKeyboard += chr( K_ENTER )
        keyboard( cKeyboard )

    case nMRow == row()

        *---------------------------------------------------------
        * The mouse cursor is exactly at the the TBrowse cursor
        * position.
        * The element can be selected.
        * Stuff inside the keyboard buffer a simple [Enter].
        *---------------------------------------------------------

        cKeyboard += chr( K_ENTER )
        keyboard( cKeyboard )

    end

    return NIL

*=================================================================
function achoiceWindow(;
    acMenuItems, cDescription,;
    nTop, nLeft, nBottom, nRight,;
    alSelectable, nFirst;
    )
*
* achoiceWindow(
*    <acMenuItems>, [<cDescription>],
*    [<nTop>], [<nLeft>], [<nBottom>], [<nRight>],
*    [<alSelectable>], [<nFirst>] ) --> nPosition
*
* <acMenuItems>        is the character array containing the list
*                      of choices.
* <cDescription>       is the header to be shown at the top
*                      window.
* <nTop>, <nLeft>, <nBottom>, <nRight>
*                      are the window coordinates.
* <alSelectable>       selectable items.
* <nFirst>             first selected element.
*
* This function is similar to achoice(), but it shows a header
* and it saves the screen, acting like a window.
*

    local cOldColor     := setcolor()
    local lOldSetMouse  := setMouse( .F. )
    local cOldScreen
    local nSetCursor    := setcursor ( SETCURSOR_NORMAL )
    local bOld_F1       :=;
        setkey( K_F1, { || Text( ACHOICE_WINDOW_HELP )} )
    local nOldRow       := row()
    local nOldCol       := col()
    local nChoice       := 0

    default( @cDescription,     "Select" )
    default( @nTop,             0 )
    default( @nLeft,            0 )
    default( @nBottom,          maxrow() )
    default( @nRight,           maxcol() )

    *-------------------------------------------------------------
    * Create a kind of window.
    *-------------------------------------------------------------

    cOldScreen    :=;
        mouseScrSave( nTop, nLeft, nBottom, nRight )
    setcolor( COLOR_BODY )
    scroll( nTop, nLeft, nBottom, nRight )
    dispBoxShadow(;
        nTop, nLeft, nBottom, nRight,;
        1,;
        dispBoxColor( 1 ),;
        dispBoxColor( 2 );
    )
    say(;
        nTop+1, nLeft+1,;
        padc( cDescription, nRight-nLeft-1 ),;
        ,;
        COLOR_HEAD;
    )

    *-------------------------------------------------------------
    * Call aChoice()
    *-------------------------------------------------------------

    nChoice :=;
        achoice(;
            nTop+1, nLeft+1, nBottom-1, nRight-1,;
            acMenuItems, alSelectable, nFirst, .T.;
        )

    *-------------------------------------------------------------
    * Restore old data.
    *-------------------------------------------------------------

    mouseScrRestore( nTop, nLeft, nBottom, nRight, cOldScreen )
    setcursor ( nSetCursor )
    setcolor( cOldColor )
    setkey( K_F1, bOld_F1 )
    setpos( nOldRow, nOldCol )
    setMouse( lOldSetMouse )

    *-------------------------------------------------------------
    * Return the choice made.
    *-------------------------------------------------------------

    return nChoice

*=================================================================
* ALERTBOX()
* ALERTBUTTONS()
*=================================================================
function alertBox( cMessage, aOptions )
*
* alertBox( <cMessage>, [<aOptions>] ) --> nChoice
*
* <cMessage>           is the message text displayed in the
*                      alert box. If the message contais one
*                      ore more semicolons, the text after
*                      the semicolon is displayed in succeeding
*                      lines in the dialog box.
* <aOptions>           defines a list of possible responses to
*                      the dialog box. All options will be
*                      displayed on one display line.
* -->                  a number corresponding to the selected
*                      option will be returned. If no choice
*                      is made, zero is returned.
*

    local aMenu         := {}
    local nOldCursor    := setcursor( SETCURSOR_NONE )
    local cOldColor     := setcolor()
    local nOldRow       := row()
    local nOldCol       := col()
    local lOldSetMouse  := setMouse( .F. )
    local cOldScreen
    local nTop
    local nLeft
    local nBottom
    local nRight
    local nWidth := 0
    local nLines
    local nLine
    local nButtonWidth
    local nI
    local nMnuChoice := 0

    default( @cMessage,     "" )
    default( @aOptions,     {"OK"} )

    *-------------------------------------------------------------
    * Start a sequence.
    *-------------------------------------------------------------

    begin sequence

        *---------------------------------------------------------
        * If there is no message, break.
        *---------------------------------------------------------

        if empty( cMessage )

            break                                       // BREAK

        end

        *---------------------------------------------------------
        * Calculate how may lines are needed for <cMessage>.
        *---------------------------------------------------------

        cMessage    := alltrim(cMessage)
        nLines      := mlcount( cMessage, maxcol()-1 )

        *---------------------------------------------------------
        * Calculate the max width of the lines containde inside
        * <cMessage>
        *---------------------------------------------------------

        for nLine   := 1 to nLines
            nWidth  :=;
                max(;
                    nWidth,;
                    len(;
                        rtrim(;
                            memoline(;
                                cMessage,;
                                maxcol()-1,;
                                nLine;
                            );
                        );
                    );
                )
        next

        *---------------------------------------------------------
        * Calculate buttons width.
        *---------------------------------------------------------

        nButtonWidth := 1
        for nI := 1 to len( aOptions )
            nButtonWidth += len( aOptions[nI] ) +1
        next

        *---------------------------------------------------------
        * Calculate the max.
        *---------------------------------------------------------

        nWidth := max( nButtonWidth, nWidth )

        *---------------------------------------------------------
        * Create a kind of window.
        *---------------------------------------------------------

        nTop        := int( ( (maxrow()+1) - (nLines+3) ) / 2 )
        nLeft       := int( ( (maxcol()+1) - (nWidth+3) ) / 2 )
        nBottom     := nTop + nLines+2
        nRight      := nLeft + nWidth+1
        cOldScreen  := mouseScrSave( nTop, nLeft, nBottom, nRight )
        setcolor( COLOR_ALERT )
        scroll( nTop, nLeft, nBottom, nRight )
        dispBoxShadow(;
            nTop, nLeft, nBottom, nRight,;
            1,;
            dispBoxColor( 1 ),;
            dispBoxColor( 2 );
        )

        *---------------------------------------------------------
        * Display alert message.
        *---------------------------------------------------------

        for nLine := 1 to nLines
            say(;
                nTop+nLine, nLeft+1,;
                memoline(cMessage, nWidth, nLine );
            )
        next

        *---------------------------------------------------------
        * Prepare buttons menu.
        *---------------------------------------------------------

        setpos( nBottom-1, nLeft+1 )
        for nI := 1 to len( aOptions )
            menuPrompt( @aMenu, row(), col()+1,;
                aOptions[nI] )
        next

        *---------------------------------------------------------
        * Read the menu.
        *---------------------------------------------------------

        nMnuChoice := menuTo( @aMenu, nMnuChoice )

    end //sequence

    *-------------------------------------------------------------
    * Restore old data.
    *-------------------------------------------------------------

    setcursor( nOldCursor )
    setcolor( cOldColor )
    setpos( nOldRow, nOldCol )
    mouseScrRestore( nTop, nLeft, nBottom, nRight, cOldScreen )
    setMouse( lOldSetMouse )

    *-------------------------------------------------------------
    * Return the choosen menu item.
    *-------------------------------------------------------------

    return nMnuChoice

*=================================================================
function alertButtons( nRow, nCol, aOptions )
*
* alertButtons( [<nRow>], [<nCol>], [<aOptions>] ) --> nChoice
*
* <nRow> and <nCol>     are the screen coordinates where the
*                       buttons will be displayed.
* <aOptions>            defines a list of possible responses.
*
* -->                   a number corresponding to the selected
*                       option will be returned. If no choice
*                       is made, zero is returned.
*

    local aMenu         := {}
    local nOldCursor    := setcursor( SETCURSOR_NONE )
    local cOldColor     := setcolor( COLOR_BUTTON )
    local nOldRow       := row()
    local nOldCol       := col()
    local lOldSetMouse  := setMouse( .F. )
    local cOldScreen    := mouseScrSave()
    local nI
    local nMnuChoice := 0

    default( @nRow,         row() )
    default( @nCol,         col() )
    default( @aOptions,     {"OK"} )

    *-------------------------------------------------------------
    * Start a sequence.
    *-------------------------------------------------------------

    begin sequence

        *---------------------------------------------------------
        * Prepare buttons menu.
        *---------------------------------------------------------

        setpos( nRow, nCol )
        for nI := 1 to len( aOptions )
            menuPrompt( @aMenu, row(), col(),;
                aOptions[nI] )
            setpos( row(), col()+1 )
        next

        *---------------------------------------------------------
        * Read the menu.
        *---------------------------------------------------------

        nMnuChoice := menuTo( @aMenu, nMnuChoice )

    end //sequence

    *-------------------------------------------------------------
    * Restore old data.
    *-------------------------------------------------------------

    setcursor( nOldCursor )
    setcolor( cOldColor )
    setpos( nOldRow, nOldCol )
    mouseScrRestore(,,,,cOldScreen)
    setMouse( lOldSetMouse )

    *-------------------------------------------------------------
    * Return the choosen menu item.
    *-------------------------------------------------------------

    return nMnuChoice

*=================================================================
* ATB()
*=================================================================

#define ATB_PROMPT_CUT_LINE     "Cut Line?"

#define ATB_PROMPT_DELETE_LINE  "Delete Line?"

#define ATB_ERROR_ARRAY_NOT_VALID;
    "Data sent for tbrowse is not valid."

#define ATB_ERROR_ARRAY_DIFFERENT_DIMENTIONS;
    "This parallel array has a different dimention."

#define ATB_BUTTONS {;
    { nBottom+1, nLeft+0,  "[Esc]",  {|| keyboard( chr(K_ESC) ) }       },;
    { nBottom+1, nLeft+7,  "[-]",   {|| keyboard( chr(K_LEFT) ) }       },;
    { nBottom+1, nLeft+12, "[]",    {|| keyboard( chr(K_UP) ) }         },;
    { nBottom+1, nLeft+16, "[Pag]", {|| keyboard( chr(K_PGUP) ) }       },;
    { nBottom+2, nLeft+0,  "[F1] ",  {|| keyboard( chr(K_F1) ) }      },;
    { nBottom+2, nLeft+7,  "[-]",   {|| keyboard( chr(K_RIGHT) ) }      },;
    { nBottom+2, nLeft+12, "[]",    {|| keyboard( chr(K_DOWN) ) }       },;
    { nBottom+2, nLeft+16, "[Pag]", {|| keyboard( chr(K_PGDN) ) }       };
    }

#define ATB_HELP;
    "ATB() - Array TBrowse" +;
    NL(2) +;
    "This function permits to browse a bidimentional Array." +;
    NL(3) +;
    "Special keys:" +;
    NL(2) +;
    "[Pag]             previous page;" +; 
    NL(1) +;
    "[Pag]             next page;" +; 
    NL(1) +;
    "[Ctrl]+[Pag]      top of table;" +; 
    NL(1) +;
    "[Ctrl]+[Pag]      bottom of table;" +; 
    NL(1) +;
    "[Ctrl]+[Home]      first column;" +; 
    NL(1) +;
    "[Ctrl]+[End]       last column;" +; 
    NL(1) +;
    "[Ctrl]+[Enter]     append;" +; 
    NL(1) +;
    "[CTRL]+[F1]        cut;" +;
    NL(1) +;
    "[CTRL]+[F2]        copy;" +; 
    NL(1) +;
    "[CTRL]+[F3]        paste;" +; 
    NL(1) +;
    "[Ctrl]+[Del]       delete;" +;
    NL(1) +;
    "[Ctrl]+[Y]         delete."


*=================================================================
function ATB(;
        nTop, nLeft, nBottom, nRight,;
        aArray, nSubscript,;
        acColSayPic,;
        acColTopSep, acColBodySep, acColBotSep,;
        acColHead, acColFoot,;
        abColValid,;
        abColMsg,;
        cColor, abColColors,;
        lModify,;
        xButtons;
    )
*
* ATB( <nTop>, <nLeft>, <nBottom>, <nRight>,
*   <aArray>, [<nSubscript>],
*   [<acColSayPic>],
*   [<acColTopSep>], [<acColBodySep>], [<acColBotSep>],
*   [<acColHead>], [<acColFoot>],
*   [<abColValid>],
*   [<abColMsg>],
*   [<cColor>], [<abColColors>],
*   [<lModify>],
*   [lButtons|aButtons]
*   )  --> aArray
*
* <nTop>, <nLeft>, <nBottom>, <nRight>
*                   The display area used to browse.
* <aArray>          Bidimentional array to browse.
* <nSubscript>      Starting array position.
* <acColSayPic>     Picture array.
* <acColTopSep>     Header separation.
* <acColBodySep>    Body separation.
* <acColBotSep>     Footer separation.
* <acColHead>       Column head description array.
* <acColFoot>       Column foot description array.
* <abColValid>      Validation codeblock array.
* <abColMsg>        Message codeblock array. The codeblock
*                   must have a string result.
* <cColor>          Color string. It MAY BE LONGER than
*                   the usual 5 elements.
* <abColColor>      Code blocks for column colors. The
*                   codeblocks must return a array of
*                   a couple of digit that point to two
*                   colors of <cColor>.
* <lModify>         Ability to modify data.
* <lButtons>        If buttons are desired.
* <aButtons>        Buttons array.
*                   aButtons[1] = nRow
*                   aButtons[2] = nCol
*                   aButtons[3] = cText
*                   aButtons[4] = bAction
*

    local bOldErrorHandler
    local nOldCursor
    local nOldRow
    local nOldCol
    local bOldF1
    local lOldReadExit  := set( _SET_EXIT, .T. )
    local lOldSetMouse  := setMouse()

    local nColumns := 0
    local anColWidth    := {}
    local anColShift    := {}

    local oBrowse
    local oColumn
    local nKey
    local lMore := .T.

    local nRow
    local nCol
    local cMsg

    local nI
    local nJ
    local lError := .F.

    local aTemp

    *-------------------------------------------------------------
    * Test for a valid <aArray>.
    * It should contains at least one valid subarray.
    * If not valid, terminate.
    *-------------------------------------------------------------

    if  valtype( aArray ) == "A"        .and.;
        !empty( aArray )

        for nI := 1 to len( aArray )

            if valtype(aArray[nI]) == "A"

                if len( aArray[nI] ) == len( aArray[1] )

                    for nJ := 1 to len( aArray[nI] )

                        if  valtype( aArray[nI][nJ] ) == "C" .or.;
                            valtype( aArray[nI][nJ] ) == "N" .or.;
                            valtype( aArray[nI][nJ] ) == "D" .or.;
                            valtype( aArray[nI][nJ] ) == "L"

                            // Ok

                        else

                            lError := .T.

                            exit                        // EXIT

                        end

                    next

                else

                    lError := .T.

                    exit                                // EXIT

                end

            else

                lError := .T.

                exit                                    // EXIT

            end

        next

    else

        lError := .T.

    end

    if lError

        alertBox( ATB_ERROR_ARRAY_NOT_VALID )

        // Terminate function execution.
        return aArray                                   // RETURN

    end

    *-------------------------------------------------------------
    * If <aArray> is considered valid, set default values.
    * Note that the first subarray, <aArray[1]> is the model
    * for the others.
    *-------------------------------------------------------------

    nColumns := len( aArray[1] )

    default( @nTop,         0 )
    default( @nLeft,        0 )
    default( @nBottom,      maxrow() )
    default( @nRight,       maxcol() )
    default( @nSubscript,   1 )
    default( @cColor,       setcolor() )
    default( @abColColors,  afill( array( nColumns ), {||{1,2}} ) )
    default( @acColSayPic,  atbDefPictures( aArray[1] ) )
    default( @acColHead,    afill( array( nColumns ), "" ) )
    default( @acColFoot,    afill( array( nColumns ), "" ) )
    default( @abColValid,   afill( array( nColumns ), {||.T.} ) )
    default( @abColMsg,     NIL )
    default( @acColTopSep,  afill( array( nColumns ), chr(194)+chr(196) ) )
    default( @acColBodySep, afill( array( nColumns ), chr(179) ) )
    default( @acColBotSep,  afill( array( nColumns ), chr(193)+chr(196) ) )
    default( @lModify,      .T. )

    *-------------------------------------------------------------
    * Check <xButtons>, it may be logical or a button array.
    * If <xButton> is logical and is True, the space for
    * buttons si taken from the bottom of the given area.
    * If <xButton> is an array, it contains the buttons
    * coordinates, so no space for buttons is calculated.
    *-------------------------------------------------------------

    do case
    case valtype( xButtons ) == "L"
    
        if xButtons

            *-----------------------------------------------------
            * As button are placed automatically,
            * the space is taken form the bottom
            * area:
            *-----------------------------------------------------

            nBottom := nBottom-2

            *-----------------------------------------------------
            * Buttons are placed beyond nButtom
            * line.
            *-----------------------------------------------------

            xButtons := ATB_BUTTONS
            
        end
        
    case valtype( xButtons ) == "A"

        *---------------------------------------------------------
        * OK.
        * Don't leave the space, as the caller
        * do what it wants
        *---------------------------------------------------------

    otherwise
    
        xButtons := NIL
        
    end

    *-------------------------------------------------------------
    * If [F1] is already set, no local help will be started.
    *-------------------------------------------------------------

    if setkey( K_F1 ) == NIL

        *---------------------------------------------------------
        * There is no previous help.
        *---------------------------------------------------------

        bOldF1 := setkey( K_F1, { || Text( ATB_HELP ) } )

    else

        bOldF1 := setkey( K_F1 )

    end

    *-------------------------------------------------------------
    * Check if paralel arrays are of the right length.
    * If different lengths are found, terminate.
    *-------------------------------------------------------------

    if !( nColumns == len( acColSayPic ) )

        alertBox(;
            "acColumnSayPicture" +;
            NL(1) +;
            ATB_ERROR_ARRAY_DIFFERENT_DIMENTIONS;
        )

        return aArray                                   // RETURN

    end

    if !( nColumns == len( acColHead ) )

        alertBox(;
            "alColumnHeads" +;
            NL(1) +;
            ATB_ERROR_ARRAY_DIFFERENT_DIMENTIONS;
        )

        return aArray                                   // RETURN

    end

    if !( nColumns == len( acColFoot ) )

        alertBox(;
            "acColumnFooters" +;
            NL(1) +;
            ATB_ERROR_ARRAY_DIFFERENT_DIMENTIONS;
        )

        return aArray                                   // RETURN

    end

    if !( nColumns == len( abColValid ) )

        alertBox(;
            "abColumnValidate" +;
            NL(1) +;
            ATB_ERROR_ARRAY_DIFFERENT_DIMENTIONS;
        )

        return aArray                                   // RETURN

    end

    if !( nColumns == len( abColColors ) )

        alertBox(;
            "abColumnColors" +;
            NL(1) +;
            ATB_ERROR_ARRAY_DIFFERENT_DIMENTIONS;
        )

        return aArray                                   // RETURN

    end

    if !( nColumns == len( acColTopSep ) )

        alertBox(;
            "abColumnTopSeparations" +;
            NL(1) +;
            ATB_ERROR_ARRAY_DIFFERENT_DIMENTIONS;
        )

        return aArray                                   // RETURN

    end

    if !( nColumns == len( acColBodySep ) )

        alertBox(;
            "abColumnBodySeparations" +;
            NL(1) +;
            ATB_ERROR_ARRAY_DIFFERENT_DIMENTIONS;
        )

        return aArray                                   // RETURN

    end

    if !( nColumns == len( acColBotSep ) )

        alertBox(;
            "abColumnBottomSeparations" +;
            NL(1) +;
            ATB_ERROR_ARRAY_DIFFERENT_DIMENTIONS;
        )

        return aArray                                   // RETURN

    end

    *-------------------------------------------------------------
    * The message line needs extra space.
    *-------------------------------------------------------------

    if ( abColMsg == NIL ) .or. len(abColMsg) <> nColumns
        // No messages.
        abColMsg := NIL
    else
        // Leave the space for messages.
        nBottom--
    end

    *-------------------------------------------------------------
    * Prepare an empty array for Cut/Copy/Paste.
    *-------------------------------------------------------------

    aTemp := aClone( atbEmptyLine( aArray ) )

    *-------------------------------------------------------------
    * Prepare the <anColWidth> with the columns width and
    * <anColShift> with the column shift for numeric fields:
    * if numeric fields are shorter than the column field
    * (this may happens only because the column head is greater),
    * these numeric fields appears right aligned inside the
    * column, so the first display column of the field is
    * greater then the first display column of the column.
    *-------------------------------------------------------------

    for nI := 1 to len( aTemp )

        *---------------------------------------------------------
        * First determinate the field length and put it inside
        * the <anColWidth>.
        * At the moment, <anColShift> is filled with zeroes.
        *---------------------------------------------------------
        
        aadd( anColWidth, len( transform( aTemp[nI], acColSayPic[nI] ) ) )

        aadd( anColShift, 0 )

        *---------------------------------------------------------
        * If the column head is longher then the field, the
        * column will be greater.
        *---------------------------------------------------------
        
        if  len( acColHead[nI] ) > anColWidth[nI]

            *-----------------------------------------------------
            * If the field is numeric, it is right aligned
            * inside the column, so, the corresponding
            * <anColShift[nI]> is calculated.
            *-----------------------------------------------------
            
            if  valtype( aTemp[nI] ) == "N"

                anColShift[nI] := len( acColHead[nI] ) - anColWidth[nI]

            end

            *-----------------------------------------------------
            * The column with is updated to the head width.
            *-----------------------------------------------------
            
            anColWidth[nI] := len( acColHead[nI] ) 
            
        end

    next
    
    *-------------------------------------------------------------
    * <nSubscript> is used to browse <aArray> up and down.
    * Check its value.
    *-------------------------------------------------------------

    if nSubscript > nColumns;
        .or. nSubscript < 1
        //
        nSubscript := 1
    end

    *-------------------------------------------------------------
    * Save settings before any display.
    *-------------------------------------------------------------

    nOldCursor          := setcursor()
    nOldRow             := row()
    nOldCol             := col()

    *-------------------------------------------------------------
    * Show buttons.
    *-------------------------------------------------------------

    if !( xButtons == NIL )
        for nI := 1 to len( xButtons )
            say(;
                xButtons[nI][1], xButtons[nI][2],;
                xButtons[nI][3],, COLOR_BUTTON;
                )
        next
    end

    *-------------------------------------------------------------
    * Prepare for browse: save error handler before.
    *-------------------------------------------------------------

    bOldErrorHandler    := errorblock( {|e| ErrorHandler(e)} )
    begin sequence

        *---------------------------------------------------------
        * Create the TBrowse object and set up some instance
        * variables.
        *---------------------------------------------------------

        oBrowse := tbrowsenew( nTop, nLeft, nBottom, nRight )

        oBrowse:skipBlock       :=;
            {|nReq| atbJumpIt(nReq, @nSubscript, aArray)}
        oBrowse:goTopBlock      := {|| nSubscript := 1}
        oBrowse:goBottomBlock   := {|| nSubscript := len(aArray)}
        oBrowse:colorSpec       := cColor

        *---------------------------------------------------------
        * Create Columns Objects and add to the TBrowse object.
        * This _long_ code is made because a fixed subscript
        * must be defined. Add more if more columns may be
        * needed.
        *---------------------------------------------------------

        begin sequence

            *-----------------------------------------------------
            * If the array is bigger then 61 columns, the
            * following list of test must be expanded.
            *-----------------------------------------------------

            if len( aArray[1] ) > 0
                oColumn :=;
                    tbcolumnnew( , {|| aArray[nSubscript][1]})
                oBrowse:addColumn(oColumn)
            else
                break                                    // BREAK
            end
            if len( aArray[1] ) > 1
                oColumn :=;
                    tbcolumnnew( , {|| aArray[nSubscript][2]})
                oBrowse:addColumn(oColumn)
            else
                break                                    // BREAK
            end
            if len( aArray[1] ) > 2
                oColumn :=;
                    tbcolumnnew( , {|| aArray[nSubscript][3]})
                oBrowse:addColumn(oColumn)
            else
                break                                    // BREAK
            end
            if len( aArray[1] ) > 3
                oColumn :=;
                    tbcolumnnew( , {|| aArray[nSubscript][4]})
                oBrowse:addColumn(oColumn)
            else
                break                                    // BREAK
            end
            if len( aArray[1] ) > 4
                oColumn :=;
                    tbcolumnnew( , {|| aArray[nSubscript][5]})
                oBrowse:addColumn(oColumn)
            else
                break                                    // BREAK
            end
            if len( aArray[1] ) > 5
                oColumn :=;
                    tbcolumnnew( , {|| aArray[nSubscript][6]})
                oBrowse:addColumn(oColumn)
            else
                break                                    // BREAK
            end

            if len( aArray[1] ) > 6
                oColumn :=;
                    tbcolumnnew( , {|| aArray[nSubscript][7]})
                oBrowse:addColumn(oColumn)
            else
                break                                    // BREAK
            end
            if len( aArray[1] ) > 7
                oColumn :=;
                    tbcolumnnew( , {|| aArray[nSubscript][8]})
                oBrowse:addColumn(oColumn)
            else
                break                                    // BREAK
            end
            if len( aArray[1] ) > 8
                oColumn :=;
                    tbcolumnnew( , {|| aArray[nSubscript][9]})
                oBrowse:addColumn(oColumn)
            else
                break                                    // BREAK
            end
            if len( aArray[1] ) > 9
                oColumn :=;
                    tbcolumnnew( , {|| aArray[nSubscript][10]})
                oBrowse:addColumn(oColumn)
            else
                break                                    // BREAK
            end
            if len( aArray[1] ) > 10
                oColumn :=;
                    tbcolumnnew( , {|| aArray[nSubscript][11]})
                oBrowse:addColumn(oColumn)
            else
                break                                    // BREAK
            end
            if len( aArray[1] ) > 11
                oColumn :=;
                    tbcolumnnew( , {|| aArray[nSubscript][12]})
                oBrowse:addColumn(oColumn)
            else
                break                                    // BREAK
            end
            if len( aArray[1] ) > 12
                oColumn :=;
                    tbcolumnnew( , {|| aArray[nSubscript][13]})
                oBrowse:addColumn(oColumn)
            else
                break                                    // BREAK
            end
            if len( aArray[1] ) > 13
                oColumn :=;
                    tbcolumnnew( , {|| aArray[nSubscript][14]})
                oBrowse:addColumn(oColumn)
            else
                break                                    // BREAK
            end
            if len( aArray[1] ) > 14
                oColumn :=;
                    tbcolumnnew( , {|| aArray[nSubscript][15]})
                oBrowse:addColumn(oColumn)
            else
                break                                    // BREAK
            end
            if len( aArray[1] ) > 15
                oColumn :=;
                    tbcolumnnew( , {|| aArray[nSubscript][16]})
                oBrowse:addColumn(oColumn)
            else
                break                                    // BREAK
            end
            if len( aArray[1] ) > 16
                oColumn :=;
                    tbcolumnnew( , {|| aArray[nSubscript][17]})
                oBrowse:addColumn(oColumn)
            else
                break                                    // BREAK
            end
            if len( aArray[1] ) > 17
                oColumn :=;
                    tbcolumnnew( , {|| aArray[nSubscript][18]})
                oBrowse:addColumn(oColumn)
            else
                break                                    // BREAK
            end
            if len( aArray[1] ) > 18
                oColumn :=;
                    tbcolumnnew( , {|| aArray[nSubscript][19]})
                oBrowse:addColumn(oColumn)
            else
                break                                    // BREAK
            end
            if len( aArray[1] ) > 19
                oColumn :=;
                    tbcolumnnew( , {|| aArray[nSubscript][20]})
                oBrowse:addColumn(oColumn)
            else
                break                                    // BREAK
            end
            if len( aArray[1] ) > 20
                oColumn :=;
                    tbcolumnnew( , {|| aArray[nSubscript][21]})
                oBrowse:addColumn(oColumn)
            else
                break                                    // BREAK
            end
            if len( aArray[1] ) > 21
                oColumn :=;
                    tbcolumnnew( , {|| aArray[nSubscript][22]})
                oBrowse:addColumn(oColumn)
            else
                break                                    // BREAK
            end
            if len( aArray[1] ) > 22
                oColumn :=;
                    tbcolumnnew( , {|| aArray[nSubscript][23]})
                oBrowse:addColumn(oColumn)
            else
                break                                    // BREAK
            end
            if len( aArray[1] ) > 23
                oColumn :=;
                    tbcolumnnew( , {|| aArray[nSubscript][24]})
                oBrowse:addColumn(oColumn)
            else
                break                                    // BREAK
            end
            if len( aArray[1] ) > 24
                oColumn :=;
                    tbcolumnnew( , {|| aArray[nSubscript][25]})
                oBrowse:addColumn(oColumn)
            else
                break                                    // BREAK
            end
            if len( aArray[1] ) > 25
                oColumn :=;
                    tbcolumnnew( , {|| aArray[nSubscript][26]})
                oBrowse:addColumn(oColumn)
            else
                break                                    // BREAK
            end
            if len( aArray[1] ) > 26
                oColumn :=;
                    tbcolumnnew( , {|| aArray[nSubscript][27]})
                oBrowse:addColumn(oColumn)
            else
                break                                    // BREAK
            end
            if len( aArray[1] ) > 27
                oColumn :=;
                    tbcolumnnew( , {|| aArray[nSubscript][28]})
                oBrowse:addColumn(oColumn)
            else
                break                                    // BREAK
            end
            if len( aArray[1] ) > 28
                oColumn :=;
                    tbcolumnnew( , {|| aArray[nSubscript][29]})
                oBrowse:addColumn(oColumn)
            else
                break                                    // BREAK
            end
            if len( aArray[1] ) > 29
                oColumn :=;
                    tbcolumnnew( , {|| aArray[nSubscript][30]})
                oBrowse:addColumn(oColumn)
            else
                break                                    // BREAK
            end
            if len( aArray[1] ) > 30
                oColumn :=;
                    tbcolumnnew( , {|| aArray[nSubscript][31]})
                oBrowse:addColumn(oColumn)
            else
                break                                    // BREAK
            end
            if len( aArray[1] ) > 31
                oColumn :=;
                    tbcolumnnew( , {|| aArray[nSubscript][32]})
                oBrowse:addColumn(oColumn)
            else
                break                                    // BREAK
            end
            if len( aArray[1] ) > 32
                oColumn :=;
                    tbcolumnnew( , {|| aArray[nSubscript][33]})
                oBrowse:addColumn(oColumn)
            else
                break                                    // BREAK
            end
            if len( aArray[1] ) > 33
                oColumn :=;
                    tbcolumnnew( , {|| aArray[nSubscript][34]})
                oBrowse:addColumn(oColumn)
            else
                break                                    // BREAK
            end
            if len( aArray[1] ) > 34
                oColumn :=;
                    tbcolumnnew( , {|| aArray[nSubscript][35]})
                oBrowse:addColumn(oColumn)
            else
                break                                    // BREAK
            end
            if len( aArray[1] ) > 35
                oColumn :=;
                    tbcolumnnew( , {|| aArray[nSubscript][36]})
                oBrowse:addColumn(oColumn)
            else
                break                                    // BREAK
            end
            if len( aArray[1] ) > 36
                oColumn :=;
                    tbcolumnnew( , {|| aArray[nSubscript][37]})
                oBrowse:addColumn(oColumn)
            else
                break                                    // BREAK
            end
            if len( aArray[1] ) > 37
                oColumn :=;
                    tbcolumnnew( , {|| aArray[nSubscript][38]})
                oBrowse:addColumn(oColumn)
            else
                break                                    // BREAK
            end
            if len( aArray[1] ) > 38
                oColumn :=;
                    tbcolumnnew( , {|| aArray[nSubscript][39]})
                oBrowse:addColumn(oColumn)
            else
                break                                    // BREAK
            end
            if len( aArray[1] ) > 39
                oColumn :=;
                    tbcolumnnew( , {|| aArray[nSubscript][40]})
                oBrowse:addColumn(oColumn)
            else
                break                                    // BREAK
            end
            if len( aArray[1] ) > 40
                oColumn :=;
                    tbcolumnnew( , {|| aArray[nSubscript][41]})
                oBrowse:addColumn(oColumn)
            else
                break                                    // BREAK
            end
            if len( aArray[1] ) > 41
                oColumn :=;
                    tbcolumnnew( , {|| aArray[nSubscript][42]})
                oBrowse:addColumn(oColumn)
            else
                break                                    // BREAK
            end
            if len( aArray[1] ) > 42
                oColumn :=;
                    tbcolumnnew( , {|| aArray[nSubscript][43]})
                oBrowse:addColumn(oColumn)
            else
                break                                    // BREAK
            end
            if len( aArray[1] ) > 43
                oColumn :=;
                    tbcolumnnew( , {|| aArray[nSubscript][44]})
                oBrowse:addColumn(oColumn)
            else
                break                                    // BREAK
            end
            if len( aArray[1] ) > 44
                oColumn :=;
                    tbcolumnnew( , {|| aArray[nSubscript][45]})
                oBrowse:addColumn(oColumn)
            else
                break                                    // BREAK
            end
            if len( aArray[1] ) > 45
                oColumn :=;
                    tbcolumnnew( , {|| aArray[nSubscript][46]})
                oBrowse:addColumn(oColumn)
            else
                break                                    // BREAK
            end
            if len( aArray[1] ) > 46
                oColumn :=;
                    tbcolumnnew( , {|| aArray[nSubscript][47]})
                oBrowse:addColumn(oColumn)
            else
                break                                    // BREAK
            end
            if len( aArray[1] ) > 47
                oColumn :=;
                    tbcolumnnew( , {|| aArray[nSubscript][48]})
                oBrowse:addColumn(oColumn)
            else
                break                                    // BREAK
            end
            if len( aArray[1] ) > 48
                oColumn :=;
                    tbcolumnnew( , {|| aArray[nSubscript][49]})
                oBrowse:addColumn(oColumn)
            else
                break                                    // BREAK
            end
            if len( aArray[1] ) > 49
                oColumn :=;
                    tbcolumnnew( , {|| aArray[nSubscript][50]})
                oBrowse:addColumn(oColumn)
            else
                break                                    // BREAK
            end
            if len( aArray[1] ) > 50
                oColumn :=;
                    tbcolumnnew( , {|| aArray[nSubscript][51]})
                oBrowse:addColumn(oColumn)
            else
                break                                    // BREAK
            end
            if len( aArray[1] ) > 51
                oColumn :=;
                    tbcolumnnew( , {|| aArray[nSubscript][52]})
                oBrowse:addColumn(oColumn)
            else
                break                                    // BREAK
            end
            if len( aArray[1] ) > 52
                oColumn :=;
                    tbcolumnnew( , {|| aArray[nSubscript][53]})
                oBrowse:addColumn(oColumn)
            else
                break                                    // BREAK
            end
            if len( aArray[1] ) > 53
                oColumn :=;
                    tbcolumnnew( , {|| aArray[nSubscript][54]})
                oBrowse:addColumn(oColumn)
            else
                break                                    // BREAK
            end
            if len( aArray[1] ) > 54
                oColumn :=;
                    tbcolumnnew( , {|| aArray[nSubscript][55]})
                oBrowse:addColumn(oColumn)
            else
                break                                    // BREAK
            end
            if len( aArray[1] ) > 55
                oColumn :=;
                    tbcolumnnew( , {|| aArray[nSubscript][56]})
                oBrowse:addColumn(oColumn)
            else
                break                                    // BREAK
            end
            if len( aArray[1] ) > 56
                oColumn :=;
                    tbcolumnnew( , {|| aArray[nSubscript][57]})
                oBrowse:addColumn(oColumn)
            else
                break                                    // BREAK
            end
            if len( aArray[1] ) > 57
                oColumn :=;
                    tbcolumnnew( , {|| aArray[nSubscript][58]})
                oBrowse:addColumn(oColumn)
            else
                break                                    // BREAK
            end
            if len( aArray[1] ) > 58
                oColumn :=;
                    tbcolumnnew( , {|| aArray[nSubscript][59]})
                oBrowse:addColumn(oColumn)
            else
                break                                    // BREAK
            end
            if len( aArray[1] ) > 59
                oColumn :=;
                    tbcolumnnew( , {|| aArray[nSubscript][60]})
                oBrowse:addColumn(oColumn)
            else
                break                                    // BREAK
            end
            if len( aArray[1] ) > 60
                oColumn :=;
                    tbcolumnnew( , {|| aArray[nSubscript][61]})
                oBrowse:addColumn(oColumn)
            else
                break                                    // BREAK
            end

        end //sequence

        *---------------------------------------------------------
        * Give more instances to columns.
        *---------------------------------------------------------

        for nI := 1 to nColumns
            // Get (a reference to) the column
            oColumn := oBrowse:getColumn(nI)
            oColumn:picture := acColSayPic[nI]
            oColumn:heading := acColHead[nI]
            oColumn:footing := acColFoot[nI]
            oColumn:headSep := acColTopSep[nI]
            oColumn:colSep  := acColBodySep[nI]
            oColumn:footSep := acColBotSep[nI]
            oColumn:colorBlock := abColColors[nI]
        next

        *---------------------------------------------------------
        * Prepare before TBrowse show:
        * Turn the cursor off while browsing.
        *---------------------------------------------------------

        setcursor(SETCURSOR_NONE)

        *---------------------------------------------------------
        * TBrowse loop.
        *---------------------------------------------------------

        while lMore

            *-----------------------------------------------------
            * Stabilize the display until it's stable
            * or a key is pressed.
            *-----------------------------------------------------

            oBrowse:forceStable()

            *-----------------------------------------------------
            * Show the bottom message (if the array was given).
            * ( Here is the only possible place inside
            * this program ).
            *-----------------------------------------------------

            nRow := row()
            nCol := col()
            if !(abColMsg == NIL)
                cMsg :=;
                    eval(;
                        abColMsg[oBrowse:colPos],;
                        aArray,;
                        nSubscript,;
                        oBrowse:colPos;
                        )
                say(;
                    nBottom+1, nLeft,;
                    padc( cMsg, nRight+1-nLeft );
                    )
                setpos( nRow, nCol )
            end

            *-----------------------------------------------------
            * Read the mouse or the keyboard inside a loop,
            * as the mouse presence do not permit to pause
            * for a key.
            *-----------------------------------------------------

            while .t.

                *-------------------------------------------------
                * Read the mouse before and than, the keyboard.
                *-------------------------------------------------

                // Show the mouse.
                setMouse( .T. )

                // Was a mouse button?
                if !( mouse() == NIL )
                    atbMouseKeyboard(;
                        xButtons,;
                        nTop, nLeft, nBottom, nRight,;
                        oBrowse, anColWidth, anColShift;
                        )
                    mouse( .T. )
                end

                // Read the keyboard.
                nKey := inkey()

                *-------------------------------------------------
                * If a key was pressed, the "wait windows", or
                * "message line" must be closed!
                * Maybe there is nothing to close, but it is
                * better to do it now!
                *-------------------------------------------------

                if nKey <> 0

                    waitFor()       // Close a waitfor()

                    messageLine()   // Close a messageLine()

                    setMouse( .F. ) // Hide the mouse()

                end

                *-------------------------------------------------
                * Analise now the key pressed.
                *-------------------------------------------------

                do case
                case ( nKey == 0 )

                    *---------------------------------------------
                    * No key was pressed, loop again.
                    *---------------------------------------------

                    loop                                // LOOP

                case ( nKey == K_ESC )

                    *---------------------------------------------
                    * Esc means leave
                    *---------------------------------------------

                    lMore := .F.

                    *---------------------------------------------
                    * Exit wait state loop.
                    *---------------------------------------------

                    exit                                // EXIT

                end

                *-------------------------------------------------
                * Now it may be a function key or whatever else
                * that is "redirected" to a special funciton.
                * After that, it may be a normal key.
                *-------------------------------------------------

                atbApplyKey(;
                    oBrowse,;
                    nKey,;
                    @aTemp,;
                    @aArray, @nSubscript,;
                    abColValid,;
                    lModify;
                    )

                *-------------------------------------------------
                * Finally, exit wait state loop
                *-------------------------------------------------

                exit

            end

            *-----------------------------------------------------
            * Mouse and keyboard loop terminated.
            *-----------------------------------------------------

        end

        *---------------------------------------------------------
        * TBrowse loop termiated.
        *---------------------------------------------------------

    recover

        *---------------------------------------------------------
        * Nothing to recover.
        *---------------------------------------------------------

    end //sequence

    *-------------------------------------------------------------
    * Restore previous values and return the modified array.
    *-------------------------------------------------------------

    errorblock(bOldErrorHandler)
    set( _SET_EXIT, lOldReadExit )
    setcursor(nOldCursor)
    setpos( nOldRow, nOldCol )
    setkey( K_F1, bOldF1 )
    setMouse( lOldSetMouse )

    return aArray

*-----------------------------------------------------------------
static function atbApplyKey(;
    oBrowse,;
    nKey,;
    aTemp,;
    aArray, nSubscript,;
    abColValid,;
    lModify;
    )
*
*
*

    local cKeyString

    do case
    case !empty( setkey( nKey ) )

        eval( setkey( nKey ) )

    case nKey == K_DOWN

        oBrowse:down()

    case nKey == K_PGDN

        oBrowse:pageDown()

    case nKey == K_CTRL_PGDN

        oBrowse:goBottom()

    case nKey == K_UP

        oBrowse:up()

    case nKey == K_PGUP

        oBrowse:pageUp()

    case nKey == K_CTRL_PGUP

        oBrowse:goTop()

    case nKey == K_RIGHT

        oBrowse:right()

    case nKey == K_LEFT

        oBrowse:left()

    case nKey == K_HOME

        oBrowse:home()

    case nKey == K_END

        oBrowse:end()

    case nKey == K_CTRL_LEFT

        oBrowse:panLeft()

    case nKey == K_CTRL_RIGHT

        oBrowse:panRight()

    case nKey == K_CTRL_HOME

        oBrowse:panHome()

    case nKey == K_CTRL_END

        oBrowse:panEnd()

    case nKey == K_RETURN

        *---------------------------------------------------------
        * Edit if you can.
        *---------------------------------------------------------

        do case
        case !lModify

            *-----------------------------------------------------
            * Read only.
            * No edit allowed.
            *-----------------------------------------------------

        otherwise

            atbDoGet(;
                oBrowse,;
                aArray,;
                nSubscript,;
                abColValid;
                )
        end

    case; 
        nKey == K_CTRL_DEL      .or.;
        nKey == K_CTRL_Y

        *---------------------------------------------------------
        * Delete if you can.
        *---------------------------------------------------------

        if  lModify .and.;
            alertBox( ATB_PROMPT_DELETE_LINE,;
                { _MENU_NO, _MENU_YES } ) == 2

            // Delete.

            if len( aArray ) == 1
                aArray[1] :=;
                    aClone( atbEmptyLine( aArray ) )
            else
                adel( aArray, nSubscript )
                asize( aArray, len( aArray )-1 )
                oBrowse:up()
            end

            oBrowse:inValidate()
            oBrowse:refreshAll()
            oBrowse:forceStable()

        end

    case nKey == K_CTRL_F1      // delete/cut

        *---------------------------------------------------------
        * Cut if you can.
        *---------------------------------------------------------

        if  lModify .and.;
            alertBox( ATB_PROMPT_CUT_LINE,;
                { _MENU_NO, _MENU_YES } ) == 2
            // Cut.
            aTemp := aClone( aArray[nSubscript] )
            if len( aArray ) == 1
                aArray[1] :=;
                    aClone( atbEmptyLine( aArray ) )
            else
                adel( aArray, nSubscript )
                asize( aArray, len( aArray )-1 )
                oBrowse:up()
            end

            oBrowse:inValidate()
            oBrowse:refreshAll()
            oBrowse:forceStable()

        end

    case nKey == K_CTRL_F3              // paste

        *---------------------------------------------------------
        * Paste inserting if you can.
        *---------------------------------------------------------

         if lModify
            asize( aArray, len( aArray )+1 )
            ains( aArray, nSubscript )
            aArray[nSubscript] := aClone( aTemp )
            oBrowse:down()
            oBrowse:inValidate()
            oBrowse:refreshAll()
            oBrowse:forceStable()

        end

    case nKey == K_CTRL_ENTER   // append a new line

        *---------------------------------------------------------
        * Append if you can.
        *---------------------------------------------------------

         if lModify

            aadd( aArray, aClone( atbEmptyLine( aArray ) ) )

            oBrowse:goBottom()
            oBrowse:inValidate()
            oBrowse:refreshAll()
            oBrowse:forceStable()

        end

    case nKey == K_CTRL_F2              // copy

        *---------------------------------------------------------
        * Save the actual subarray for future
        * possible insertions.
        *---------------------------------------------------------

         aTemp := aClone( aArray[nSubscript] )

    otherwise

        *---------------------------------------------------------
        * It must be a editing key, so start
        * etiting (if possible) and stuff the
        * key again into the keyboard buffer.
        *---------------------------------------------------------

        do case
        case !lModify

            *-----------------------------------------------------
            * No edit allowed.
            *-----------------------------------------------------

        otherwise

            *-----------------------------------------------------
            * Save all pending keys and then
            * stuff them again into the keyboard.
            *-----------------------------------------------------

            cKeyString := chr( nKey )
            while (nKey := inkey()) > 0
                cKeyString += chr( nKey )
            end

            keyboard( cKeyString )

            *-----------------------------------------------------
            * Start editing.
            *-----------------------------------------------------

            atbDoGet(;
                oBrowse,;
                aArray,;
                nSubscript,;
                abColValid;
                )

        end

    end

    return NIL

*-----------------------------------------------------------------
static function atbJumpIt(nRequest, nSubscript, aArray)
*
*
*

    local nActually := 0

    do case
    case nRequest == 0

        // No change.

    case nRequest > 0

        *---------------------------------------------------------
        * Skip forward, but not beyond the <aArray> lengts.
        *---------------------------------------------------------

        if nRequest < (len(aArray) - nSubscript)
            nActually := nRequest
        else
            nActually := len(aArray) - nSubscript
        end

    case nRequest < 0

        *---------------------------------------------------------
        * Skip backwards, but not before the first position.
        *---------------------------------------------------------

        if nRequest < (1 - nSubscript)
            nActually := 1 - nSubscript
        else
            nActually := nRequest
        end

    end

    *-------------------------------------------------------------
    * Update <nSubscript> to the proper element position
    *-------------------------------------------------------------

    nSubscript += nActually

    *-------------------------------------------------------------
    * Return the number of elements skipped.
    *-------------------------------------------------------------

    return nActually

*-----------------------------------------------------------------
static function atbDoGet(;
    oBrowse, aArray, nSubscript, abColValid;
    )
*
* Do a GET for the current column in the browse.
*

    local oCol
    local aoGet    := {}
    local nKey
    local bSavIns
    local cOldScreen
    local cOldColor
    local nSetCursor
    local nRow
    local nCol
    local xVar

    *-------------------------------------------------------------
    * Make a copy of the active array element.
    *-------------------------------------------------------------

    xVar := aArray[nSubscript][oBrowse:colPos]

    *-------------------------------------------------------------
    * Make sure screen is fully updated.
    *-------------------------------------------------------------

    oBrowse:forceStable()

    *-------------------------------------------------------------
    * Get the current column object from the browse.
    *-------------------------------------------------------------

    oCol := oBrowse:getColumn(oBrowse:colPos)

    *-------------------------------------------------------------
    * Loop to check for valid data.
    *-------------------------------------------------------------

    while .T.

        *---------------------------------------------------------
        * Create a corresponding GET.
        * Before, save cursor position for possible loop.
        *---------------------------------------------------------

        nRow := row()
        nCol := col()

        get(;
            @aoGet, row(), col(),;
            { |x| iif( pcount() > 0, xVar := x, xVar ) },;
            oCol:picture,;
            oBrowse:colorSpec;
            )

        *---------------------------------------------------------
        * Set insert key to toggle insert mode and cursor shape.
        * Prepare cursor shape.
        *---------------------------------------------------------

        bSavIns := setkey(K_INS, { || tglInsert() })

        if set( _SET_INSERT )
            setcursor( SETCURSOR_INSERT )
        else
            setcursor( SETCURSOR_NORMAL )
        end

        *---------------------------------------------------------
        * Read and transfer the new value back into the
        * array element.
        *---------------------------------------------------------

        read( aoGet )
        aoGet := {}

        aArray[nSubscript][oBrowse:colPos] := xVar

        *---------------------------------------------------------
        * Restore cursor, Insert and position: all before
        * check for valid data.
        *---------------------------------------------------------

        setcursor(SETCURSOR_NONE)
        setkey(K_INS, bSavIns)
        setpos( nRow, nCol )

        *---------------------------------------------------------
        * Check for valid data.
        *---------------------------------------------------------

        if eval(;
            abColValid[oBrowse:colPos],;
            aArray,;
            nSubscript,;
            oBrowse:colPos;
            )

            *-----------------------------------------------------
            * Data is valid: exit loop.
            *-----------------------------------------------------

            exit

        else

            *-----------------------------------------------------
            * Data is not valid, loop again.
            *-----------------------------------------------------

        end

    end

    *-------------------------------------------------------------
    * The loop for valid data is terminated: refresh.
    *-------------------------------------------------------------

    oBrowse:refreshCurrent()

    *-------------------------------------------------------------
    * Check the key used to exit the edit loop and stuff it
    * again in the keyboard buffer.
    *-------------------------------------------------------------

    nKey := lastkey()
    if nKey == K_UP   .or.;
       nKey == K_DOWN .or.;
       nKey == K_PGUP .or.;
       nKey == K_PGDN
        keyboard(chr(nKey))
    end

    *-------------------------------------------------------------
    * Return the edited value.
    *-------------------------------------------------------------

    return xVar

*-----------------------------------------------------------------
static function atbDefPictures( aValues )
*
* Default picture definition.
*

    local nLen := 0
    local cColSayPic := ""
    local acPictures := {}
    local nI

    *-------------------------------------------------------------
    * Scan all values.
    *-------------------------------------------------------------

    for nI := 1 to len( aValues )
        do case
        case valtype( aValues[nI] ) == "C"
            nLen := len( aValues[nI] )
            if nLen > 40
                cColSayPic := "@s40"
            else
                cColSayPic := replicate( "x", nLen )
            end
        case valtype( aValues[nI] ) == "N"
            cColSayPic := "9999999999999999.9999999999999"
        case valtype( aValues[nI] ) == "D"
            cColSayPic := "99/99/9999"
        case valtype( aValues[nI] ) == "L"
            cColSayPic := "L"
        end

        aadd( acPictures, cColSayPic )

    next

    *-------------------------------------------------------------
    * Return the array of pictures.
    *-------------------------------------------------------------

    return acPictures

*-----------------------------------------------------------------
static function atbEmptyLine( aArray )
*
* Create an empty array clone of <aArray[1]>
*

    local nI
    local aEmpty    := {}

    for nI := 1 to len( aArray[1] )

        do case
        case valtype( aArray[1][nI] ) == "C"
            aadd( aEmpty, space( len( aArray[1][nI] ) ) )
        case valtype( aArray[1][nI] ) == "N"
            aadd( aEmpty, 0 )
        case valtype( aArray[1][nI] ) == "L"
            aadd( aEmpty, .F. )
        case valtype( aArray[1][nI] ) == "D"
            aadd( aEmpty, ctod("  /  /    ") )
        end
    next

    return aEmpty

*-----------------------------------------------------------------
static function atbMouseKeyboard(;
    aButtons,;
    nTop, nLeft, nBottom, nRight,; 
    oBrowse, anColWidth, anColShift;
    )
*
*
*

    local aMouse    := mouse()

    local nMCol      := aMouse[1]-1
    local nMRow      := aMouse[2]-1

    local nTimes    := 0
    local cKeyboard := ""
    local nI

    local nButtRow
    local nButtCol
    local nButtColEnd

    *-------------------------------------------------------------
    * If <aButtons> is an array, test if the mouse selected
    * a button.
    *-------------------------------------------------------------

    if valtype( aButtons ) == "A"

        for nI := 1 to len(aButtons)

            nButtRow    := aButtons[nI][1]
            nButtCol    := aButtons[nI][2]
            nButtColEnd := nButtCol + len( aButtons[nI][3] ) -1

            if nButtRow == nMRow;
                .and. nButtCol <= nMCol;
                .and. nButtColEnd >= nMCol

                *-------------------------------------------------
                * Ok button selected. Do the action and terminate.
                *-------------------------------------------------

                eval( aButtons[nI][4] )

                return NIL

            end

        next

    end

    *-------------------------------------------------------------
    * If still here, it wasn't a button.
    * If it happened inside the right area, transform into
    * keyboard.
    *-------------------------------------------------------------

    if  nMRow >= nTop           .and.;
        nMRow <= nBottom        .and.;
        nMCol >= nLeft          .and.;
        nMCol <= nRight
        
        // Ok, this is the right place.
    
    else

        return NIL                                      // RETURN

    end

    *-------------------------------------------------------------
    * If still here, the place was right.
    * Transform into keyboard.
    *-------------------------------------------------------------

    *-------------------------------------------------------------
    * First the vertical movement.
    *-------------------------------------------------------------
    
    do case
    case nMRow > row()

        nTimes := nMRow - row()

        for nI := 1 to nTimes

            cKeyboard += chr( K_DOWN )

        end
    
    case nMRow < row()

        nTimes := row() - nMRow

        for nI := 1 to nTimes

            cKeyboard += chr( K_UP )

        end

    end

    *-------------------------------------------------------------
    * Then the horizontal movement.
    *-------------------------------------------------------------
    
    cKeyboard += atbMouseHorizontal( oBrowse, anColWidth, anColShift, nMCol ) 

    *-------------------------------------------------------------
    * If something was done, transfer the keys into the
    * keyboard buffer.
    *-------------------------------------------------------------
    
    if len( cKeyboard ) > 0

        keyboard( cKeyboard )

    end

    return NIL

*-----------------------------------------------------------------
static function atbMouseHorizontal(;
        oBrowse,;
        anColWidth, anColShift,;
        nMCol;
    )
*

    local cHorKey       := ""
    local nCol          := oBrowse:colPos
    local nScreenCol    := col() - anColShift[nCol]

    while .T.                                           // FOREVER

        do case
        case; 
            nMCol >= nScreenCol                         .and.;
            nMCol <= nScreenCol + anColWidth[nCol]

            *-----------------------------------------------------
            * The mouse cursor is inside the area of the current
            * column. 
            *-----------------------------------------------------
            
            exit                                        // EXIT

        case;
            nMCol > nScreenCol + anColWidth[nCol]

            *-----------------------------------------------------
            * The mouse cursor is on the area of a different
            * column on the right
            *-----------------------------------------------------
            
            if nCol < len( anColWidth )

                *-------------------------------------------------
                * It is possible to go right.
                *-------------------------------------------------
                
                nScreenCol += anColWidth[nCol] +1
                nCol++
                cHorKey += chr( K_RIGHT )

            else
                
                *-------------------------------------------------
                * No more columns are available.
                *-------------------------------------------------
            
                exit                                    // EXIT

            end

        case;
            nMCol < nScreenCol

            *-----------------------------------------------------
            * The mouse cursor is on the area of a different
            * column on the left
            *-----------------------------------------------------
            
            if nCol > 1
                
                *-------------------------------------------------
                * It is possible to go left.
                *-------------------------------------------------
                
                nCol--
                nScreenCol -= (anColWidth[nCol] +1)
                cHorKey += chr( K_LEFT )

            else
                *-------------------------------------------------
                * No more columns are available.
                *-------------------------------------------------
            
                exit                                    // EXIT

            end

        end

    end

    *-------------------------------------------------------------
    * The string containing the key to be pressed to reach the
    * column, is returned.
    *-------------------------------------------------------------
    
    return cHorKey


*=================================================================
* BCOMPILE()
*=================================================================
function bCompile( cString )
*
* bCompile( <cString> ) --> bBlock
*
* <bString>       string to translate into code block.
*
* Compiles the string <cString> and result a code block.
*

    local cCodeBlock

    if cString <> NIL
        cCodeBlock :=;
            "{||";
            + alltrim(cString);
            + "}"

        return &(cCodeBlock)
    else
        return NIL
    end

    return NIL

*=================================================================
* BUTTON()
*=================================================================
function button( aButtons, nRow, nCol, cText, cColor, bAction )
*
* button( <aButtons>,
*      [<nRow>], [<nCol>], [<cText>], [<cColor>],
*      [<bAction>] ) --> NIL
*
* <aButtons>           Array of buttons to be incremented
*                      with a new one.
* <nRow>               Row position on where the new button
*                      should be placed.
* <nCol>               Column position where the new button
*                      should be placed.
* <cText>              The button text.
* <cColor>             The color to be used.
* <bAction>            Code block to be executed when the
*                      button will be selected.
*
* This function adds a new button in the <aButtons> array.
*

    default( @nRow,     row() )
    default( @nCol,     col() )
    default( @ctext,    "Button" )
    default( @cColor,   COLOR_BUTTON )
    default( @bAction,  {||NIL} )

    say( nRow, nCol, cText, , cColor )

    aadd( aButtons, { nRow, nCol, cText, bAction } )

    return NIL

*=================================================================
* CM...
*
* Note:
*   CM(), the macro compilation function is not properly a
* standard function, but as it is a delicated work that is
* linked with the macro interpretation, I decided to leave it
* here and not to document the function inside the nB manual.
*=================================================================

#define CM_MAX_LINES                999999
#define CM_MAX_LINES_DIGITS         6

#define CM_STEP00;
    "Reading the file..."
#define CM_STEP01;
    "Deleting comments..."
#define CM_STEP02;
    "Translating multiple line commands;" +;
    NL(1) +;
    "deleting empty lines;" +;
    NL(1) +;
    "packing."

#define CM_STEP03;
    "Calculating PROCEDURE pointers."
#define CM_STEP04;
    "Calculating BEGIN|IF|CASE|WHILE structure pointers."

#define CM_ERROR_MENU_IGNORE        "Ignore"
#define CM_ERROR_MENU_BREAK         "Break"


#define CM_ERROR_FILE_NOT_FOUND;
    "Macro file not found!"

#define CM_ST_MAX_NEST       256

#define CM_ST_PROCEDURE        1
#define CM_ST_RETURN           2
#define CM_ST_ENDPROCEDURE     3
#define CM_ST_DOPROCEDURE      4

#define CM_ST_BEGIN           11
#define CM_ST_BREAK           12

#define CM_ST_IF              21
#define CM_ST_THEN            22
#define CM_ST_ELSE            23

#define CM_ST_WHILE           31

#define CM_ST_DOCASE          41
#define CM_ST_CASE            42
#define CM_ST_CASEMATCHED     43
#define CM_ST_OTHERWISE       44

#define CM_ST_MAIN            81

#define CM_ST_LOOP            97
#define CM_ST_EXIT            98
#define CM_ST_END             99

#define CM_ST_ERROR_NO_END;
    "The macro file is terminated before expected: a END was searched"
#define CM_ST_ERROR_ALONE_BREAK;
    "BREAK: missing BEGIN SEQUENCE."
#define CM_ST_ERROR_ALONE_CASE;
    "CASE: missing DO CASE."
#define CM_ST_ERROR_ALONE_ELSE;
    "ELSE: missing IF."
#define CM_ST_ERROR_ALONE_END;
    "END: missing IF|WHILE|DO CASE|BEGIN SEQUENCE."
#define CM_ST_ERROR_ALONE_ENDPROCEDURE;
    "ENDPROCEDURE: missing PROCEDURE."
#define CM_ST_ERROR_ALONE_EXIT;
    "EXIT: missing WHILE."
#define CM_ST_ERROR_ALONE_LOOP;
    "LOOP: missing WHILE."
#define CM_ST_ERROR_ALONE_OTHERWISE;
    "OTHERWISE: missing DO CASE."

#define CM_ST_ERROR_LINE_TOO_LONG;
    "Command line too long: it exeeds " +;
    ltrim( str( _MAX_STRING_LEN ) ) + " bytes."
#define CM_ST_ERROR_NO_CONDITION;
   "No condition supplied!"
#define CM_ST_ERROR_NO_PROCEDURE_NAME;
    "Missing procedure name."
#define CM_ST_ERROR_PROCEDURE_NOT_FOUND;
    "Procedure not found!"
#define CM_ST_ERROR_SEMICOLON_EOF;
    "The file is terminated before expected: semicolon."
#define CM_ST_ERROR_UNCLOSED_STRUCTURE;
    "Structure unclosed: missing END."
#define CM_ST_ERROR_UNCLOSED_PROCEDURE;
    "Structure unclosed: missing ENDPROCEDURE."

*=================================================================
function cm( cFileMacro, cFileCompiled )
*
* cm( <cFileMacro>, [<cFileCompiled>] ) --> nExitCode
*
* <cFileMacro>     The macro filename with extension.
* <cFileCompiled>  The compiled macro filename with extention.
*
* Compile <cFileMacro> into [<cFileCompiled>].
*

    local bOldErrorHandler
    local cMacro
    local nReturn             := _MACRO_EXIT_NORMAL
    local nOldSelect          := select()
    local aStruct             := {}

    *-------------------------------------------------------------
    * The name must be alltrimed.
    *-------------------------------------------------------------

    cFileMacro := alltrim( cFileMacro )

    *-------------------------------------------------------------
    * Start a new sequence with a different error handler.
    *-------------------------------------------------------------

    bOldErrorHandler := errorblock( {|e| errorHandler(e)} )
    begin sequence

        *---------------------------------------------------------
        * Test for macro file existance.
        *---------------------------------------------------------

        if !file( cFileMacro )

            alertBox( cFileMacro + NL(1) +;
                CM_ERROR_FILE_NOT_FOUND )

            *-----------------------------------------------------
            * Prepare to return and break.
            *-----------------------------------------------------

            nReturn := _MACRO_EXIT_NO_MACRO_FILE

            break                                       // BREAK

        end

        *---------------------------------------------------------
        * Check <cFielCompiled.
        * If this parameter comes from
        * "NB -m <source> <destination>"
        * it may be only char type, but empty.
        *---------------------------------------------------------

        if  valtype(cFileCompiled) == "C";
            .and.;
            !empty(cFileCompiled)

            *-----------------------------------------------------
            * The data is correct; complete with file extention.
            *-----------------------------------------------------

            cFileCompiled :=;
                alltrim( strAddExtention( cFileCompiled,;
                _EXTENTION_MACRO_COMPILED ) )

        else

            *-----------------------------------------------------
            * There is no <cFileCompiled>, so it must be
            * determinated changing the extention of <cFileMacro>.
            *-----------------------------------------------------

            cFileCompiled :=;
                strPath( cFileMacro ) +;
                strFile( cFileMacro )

            cFileCompiled := strCutExtention( cFileCompiled )

            cFileCompiled :=;
                strAddExtention(;
                    cFileCompiled,;
                    _EXTENTION_MACRO_COMPILED;
                )
        end

        *---------------------------------------------------------
        * Check if <cFileCompiled> already exists.
        *---------------------------------------------------------

        if file( cFileCompiled )

            *-----------------------------------------------------
            * Overwrite?
            *-----------------------------------------------------

            if  !(;
                    alertBox( cFileCompiled + NL(1) +;
                    _ERROR_PROMPT_ALREADY_EXIST_OVERWRITE,;
                    { _MENU_NO, _MENU_YES } ) == 2;
                )

                break                                   // BREAK

            end

        end

        *---------------------------------------------------------
        * A structure file is created using the default RDD.
        *---------------------------------------------------------

        aStruct := {}
        aadd( aStruct, { "Line",    "N", CM_MAX_LINES_DIGITS, 0 } )
        aadd( aStruct, { "Macro",   "C", _MAX_STRING_LEN, 0 } )
        aadd( aStruct, { "Command", "N", 2, 0 } )
        aadd( aStruct, { "Goto1",   "N", CM_MAX_LINES_DIGITS, 0 } )
        aadd( aStruct, { "Goto2",   "N", CM_MAX_LINES_DIGITS, 0 } )
        dbcreate( cFileCompiled, aStruct, _DEFAULT_RDD )

        *---------------------------------------------------------
        * The structure file is opened.
        *---------------------------------------------------------

        dbusearea( .T., _DEFAULT_RDD, cFileCompiled,;
            _MACRO_ALIAS, .F., .F. )

        *---------------------------------------------------------
        * Step 00 - Append the source file into the new file.
        *---------------------------------------------------------

        waitFor( cFileMacro + NL(1) + CM_STEP00 )
        dbSdf( .F., cFileMacro, { "Macro" } )

        *---------------------------------------------------------
        * Step 01
        * Save original line numbers;
        * cut comments;
        * left trim.
        *---------------------------------------------------------

        waitFor( cFileCompiled + NL(1) + CM_STEP01 )
        cmStep01()

        *---------------------------------------------------------
        * Step 02
        * Multiple lines translated into single lines;
        * delete empty lines;
        * pack.
        *---------------------------------------------------------

        waitFor( cFileCompiled + NL(1) + CM_STEP02 )
        cmStep02()
        dbPack()


        *---------------------------------------------------------
        * Step 03
        * Procedure pointers: the file is scanned twice.
        *---------------------------------------------------------

        waitFor( cFileCompiled + NL(1) + CM_STEP03 )
        cmStep03()

        *---------------------------------------------------------
        * Step 04
        * BeginIf|Case|While structure pointers.
        *---------------------------------------------------------

        waitFor( cFileCompiled + NL(1) + CM_STEP04 )
        cmStep04()

        *---------------------------------------------------------
        * Terminated.
        *---------------------------------------------------------

        waitfor()

        dbclosearea()

    recover

        *---------------------------------------------------------
        * Try to remedy.
        *---------------------------------------------------------

        if select(_MACRO_ALIAS) > 0
            // the file is still open
            (_MACRO_ALIAS)->(dbclosearea())
        end

    end //sequence

    *-------------------------------------------------------------
    * Restore old data.
    *-------------------------------------------------------------

    select( nOldSelect )

    *-------------------------------------------------------------
    * Return compilation result.
    *-------------------------------------------------------------

    return nReturn

*-----------------------------------------------------------------
static function cmStep01()
*
*

    dbgotop()

    dbeval( {|| Field->Line := recno(),;
            Field->Macro :=;
            ltrim( exCommentCut( Field->Macro ) ) },;
        {||waitFileEval()},;
        ,,, .F. )

    waitFileEval( .T. )

    return NIL

*-----------------------------------------------------------------
static function cmStep02()
*
*

    local nRecord
    local nNext

    dbgotop()

    for nRecord := 1 to lastrec()

        dbgoto( nRecord )
        waitFileEval()

        *---------------------------------------------------------
        * Attach lines together.
        *---------------------------------------------------------

        Field->Macro := cmSemicolon()

        *---------------------------------------------------------
        * If the current record is empty, delete it.
        *---------------------------------------------------------

        if empty( Field->Macro )

            dbdelete()

        end

    next

    waitFileEval(.T.)

    return NIL

*-----------------------------------------------------------------
static function cmSemicolon()
*
*

    local cCommand := rtrim( Field->Macro )

    *-------------------------------------------------------------
    * If a semicolon exists at the end of the line, it is a line
    * that continue on the next record.
    *-------------------------------------------------------------

    if right( cCommand, 1 ) == ";"

        *---------------------------------------------------------
        * The command is cleaned from the semicolon.
        *---------------------------------------------------------

        cCommand := left( cCommand, len( cCommand )-1 )

        *---------------------------------------------------------
        * Skip to next record to attach next line.
        *---------------------------------------------------------

        dbskip(+1)

        *---------------------------------------------------------
        * If the file is terminated before the command
        * is terminated tell it.
        *---------------------------------------------------------

        if eof()

            alertBox( "(End Of File)" +;
                NL(1) +;
                CM_ST_ERROR_SEMICOLON_EOF )

            // Continue.

        end

        *---------------------------------------------------------
        * Recursion: take next peace of command.
        *---------------------------------------------------------

        cCommand += cmSemicolon()

        *---------------------------------------------------------
        * Replace the record with spaces:
        * this because what was here will be copied inside the
        * previous record.
        *---------------------------------------------------------

        Field->Macro := space(_MAX_STRING_LEN)

        *---------------------------------------------------------
        * Back to previous record.
        *---------------------------------------------------------

        dbskip(-1)

    end

    *-------------------------------------------------------------
    * A complete command cannot be longher than
    * the record limit that correspond to the maximum
    * code block string length.
    *-------------------------------------------------------------

    if len( cCommand ) > _MAX_STRING_LEN

        alertBox( "Line(" +;
            ltrim(str(Field->Line)) +;
            ")" +;
            NL(1) +;
            left(cCommand, 80) +;
            NL(1) +;
            CM_ST_ERROR_LINE_TOO_LONG )

    end

    *-------------------------------------------------------------
    * Return the command line (or the peace of command)
    * that started from this record and, eventually, continued
    * on next records.
    *-------------------------------------------------------------

    return cCommand

*-----------------------------------------------------------------
static function cmStep03()
*
* aProcedure[n][1] = Nth Procedure name,
* aProcedure[n][2] = Nth starting line position,
*

    local aProcedure          := {}
    local nProcedure          := 0
    local cProcedure          := ""
    local nRecord
    local cCommand            := ""
    local nEndProc

    local nExitCode           := _MACRO_EXIT_NORMAL

    *-------------------------------------------------------------
    * Start a new sequence.
    *-------------------------------------------------------------

    begin sequence

        *---------------------------------------------------------
        * Scan records
        *---------------------------------------------------------

        for nRecord := 1 to lastrec()

            dbgoto( nRecord )

            waitFileEval()

            cCommand := Field->Macro

            *-----------------------------------------------------
            * Analise the line.
            *-----------------------------------------------------

            do case
            case upper( left( cCommand, 10 ) ) == "PROCEDURE "

                *-------------------------------------------------
                * This the begin of a procedure.
                * First, find the endprocedure point.
                *-------------------------------------------------

                nEndProc := cmEndProc()

                *-------------------------------------------------
                * Save the starting prosition on a local
                * array to resolve procedure calls.
                *-------------------------------------------------

                cProcedure :=;
                    alltrim( substr( ltrim( cCommand ), 10 ) )
                cProcedure := upper( cProcedure )
                aadd( aProcedure,;
                    { cProcedure, recno() } )

                *-------------------------------------------------
                * Update the destination file.
                *-------------------------------------------------

                Field->Command := CM_ST_PROCEDURE
                Field->Goto1   := nEndProc+1

            case upper( left( cCommand, 13 ) ) == "ENDPROCEDURE "

                *-------------------------------------------------
                * This the end of a procedure.
                * Update the destination file.
                *-------------------------------------------------

                Field->Command := CM_ST_ENDPROCEDURE

            case upper( left( cCommand, 7 ) ) == "RETURN "

                *-------------------------------------------------
                * RETURN statement.
                * Update the destination file.
                *-------------------------------------------------

                Field->Command := CM_ST_RETURN

            end

        next

        *---------------------------------------------------------
        * Close wait file eval.
        *---------------------------------------------------------

        waitFileEval( .T. )

        *---------------------------------------------------------
        * Start a new scan to find the procedure calls:
        * DO PROCEDURE can be scanned only when all procedure
        * are listed inside the aProcedure array.
        *---------------------------------------------------------

        for nRecord := 1 to lastrec()

            dbgoto( nRecord )

            waitFileEval()

            cCommand := Field->Macro

            *-----------------------------------------------------
            * Analise the line.
            *-----------------------------------------------------

            do case
            case upper( left( cCommand, 13 ) ) == "DO PROCEDURE "

                *-------------------------------------------------
                * A procedure call was found.
                * Save the procedure name.
                *-------------------------------------------------

                cProcedure :=;
                    alltrim( substr( ltrim( cCommand ), 13 ) )
                cProcedure := upper( cProcedure )

                *-------------------------------------------------
                * Start a loop to find the procedure inside
                * <aProcedure>.
                *-------------------------------------------------

                nProcedure := 1
                while .t.                               // FOREVER

                    *---------------------------------------------
                    * If the procedure name is not found,
                    * tell it and stop this loop.
                    *---------------------------------------------

                    if nProcedure > len( aProcedure )

                        alertBox( ;
                            "Line(" +;
                            ltrim(str(Field->Line)) +;
                            ")" +NL(1) +;
                            cProcedure + NL(1) +;
                            CM_ST_ERROR_PROCEDURE_NOT_FOUND )

                        exit                            // EXIT

                    end

                    *---------------------------------------------
                    * If the procedure name is found, update
                    * pointers inside the destination file.
                    * Terminate loop.
                    *---------------------------------------------

                    if aProcedure[nProcedure][1] == cProcedure

                        Field->Command := CM_ST_DOPROCEDURE
                        Field->Goto1 :=;
                            aProcedure[nProcedure][2]+1

                        exit                            // EXIT

                    end

                    *---------------------------------------------
                    * Next <aProcedure> element.
                    *---------------------------------------------

                    nProcedure++

                end

            end

        next

        *---------------------------------------------------------
        * Scan terminated, close wait bar.
        *---------------------------------------------------------

        waitFileEval( .T. )

    end //sequence

    *-------------------------------------------------------------
    * Return exit code.
    *-------------------------------------------------------------

    return nExitCode

*-----------------------------------------------------------------
static function cmEndProc()
*
*

    local nOldRecord := recno()
    local nEndProc

    *-------------------------------------------------------------
    * Start the search for the next "ENDPROCEDURE"
    *-------------------------------------------------------------

    while .t.

        dbskip()

        do case
        case eof()

            *-----------------------------------------------------
            * The file is terminated before. Tell it and exit
            * the search loop.
            *-----------------------------------------------------

            alertBox( "( EOF )" + NL(1) +;
            CM_ST_ERROR_UNCLOSED_PROCEDURE )

            exit                                        // EXIT

        case upper( left( Field->Macro, 13 ) ) == "ENDPROCEDURE "

            *-----------------------------------------------------
            * "ENDPROCEDURE" found.
            * Stop the search loop.
            *-----------------------------------------------------

            exit                                        // EXIT

        end

    end

    nEndProc := recno()
    dbgoto( nOldRecord )

    *-------------------------------------------------------------
    * Return the "ENDPROCEDURE" pointer or the eof() position.
    *-------------------------------------------------------------

    return nEndProc

*-----------------------------------------------------------------
static function cmStep04()
*
*

    local aNest               := array( CM_ST_MAX_NEST )
    local nNest               := 0
    local nRecord
    local cCommand            := ""
    local cCondition          := ""
    local nPosition
    local nPosition2

    local nExitCode           := _MACRO_EXIT_NORMAL

    *-------------------------------------------------------------
    * Start a new sequence.
    *-------------------------------------------------------------

    begin sequence

        *---------------------------------------------------------
        * The lower nest level is MAIN.
        *---------------------------------------------------------

        nNest := 1
        aNest[nNest]          := { CM_ST_MAIN }
        // [1]MAIN

        *---------------------------------------------------------
        * Scan records to find statement like Begin/If/Case/While.
        *---------------------------------------------------------

        for nRecord := 1 to lastrec()

            waitFileEval()

            dbgoto( nRecord )
            cCommand := Field->Macro

            *-----------------------------------------------------
            * If eof() the scan terminates with break.
            *-----------------------------------------------------

            if eof()

                nExitCode := _MACRO_EXIT_NORMAL

                break                                   // BREAK

            end

            *-----------------------------------------------------
            * Check if the line contains a statement.
            *-----------------------------------------------------

            do case
            case upper( left( cCommand, 4 ) ) == "END "

                *-------------------------------------------------
                * "END" was found.
                * Update the destination file.
                *-------------------------------------------------

                Field->Command  := CM_ST_END

                *-------------------------------------------------
                * Close one nest level.
                *-------------------------------------------------

                do case
                case aNest[nNest][1] == CM_ST_MAIN

                    *---------------------------------------------
                    * Not allowed: it is a surplus "END".
                    *---------------------------------------------

                    alertBox( "(" + ltrim(str(Field->Line)) + ")" +NL(1) +;
                        CM_ST_ERROR_ALONE_END )

                    *---------------------------------------------
                    * The compilation breaks.
                    *---------------------------------------------

                    nExitCode := _MACRO_EXIT_BREAK

                    break                               // BREAK

                case aNest[nNest][1] == CM_ST_DOPROCEDURE

                    *---------------------------------------------
                    * Not allowed: it is a surplus "END".
                    *---------------------------------------------

                    alertBox( "Line(" +;
                        ltrim(str(Field->Line)) +;
                        ")" +;
                        NL(1) +;
                        CM_ST_ERROR_ALONE_END )

                    nExitCode := _MACRO_EXIT_BREAK

                    break                               // BREAK

                case aNest[nNest][1] == CM_ST_IF

                    *---------------------------------------------
                    * One nest level is closed.
                    *---------------------------------------------

                    aNest[nNest] := NIL
                    nNest--

                case aNest[nNest][1] == CM_ST_BEGIN

                    *---------------------------------------------
                    * One nest level is closed.
                    *---------------------------------------------

                    aNest[nNest] := NIL
                    nNest--

                case aNest[nNest][1] == CM_ST_WHILE

                    *---------------------------------------------
                    * One nest level is closed.
                    *---------------------------------------------

                    aNest[nNest] := NIL
                    nNest--

                case aNest[nNest][1] == CM_ST_DOCASE

                    *---------------------------------------------
                    * One nest level is closed.
                    *---------------------------------------------

                    aNest[nNest] := NIL
                    nNest--

                end

            case upper( left( cCommand, 10 ) ) == "PROCEDURE "

                *-------------------------------------------------
                * Procedure are already checked on the previous
                * step. Here simpli trak the nest sequence.
                *-------------------------------------------------

                nNest++
                aNest[nNest] := { CM_ST_PROCEDURE }
                // [1]PROCEDURE

            case upper( left( cCommand, 13 ) ) == "ENDPROCEDURE "

                *-------------------------------------------------
                * Simply go back one nest level.
                *-------------------------------------------------

                do case
                case aNest[nNest][1] == CM_ST_PROCEDURE

                    aNest[nNest] := NIL
                    nNest--

                otherwise

                    *---------------------------------------------
                    * This level wasn't started form a PROCEDURE,
                    * so it is wrong.
                    *---------------------------------------------

                    alertBox( "Line(" +;
                        ltrim(str(Field->Line)) +;
                        ")" +;
                        NL(1) +;
                        CM_ST_ERROR_ALONE_ENDPROCEDURE )


                    *---------------------------------------------
                    * The compilation breaks.
                    *---------------------------------------------

                    nExitCode := _MACRO_EXIT_BREAK

                    break                               // BREAK

                end

            case upper( left( cCommand, 15 ) ) == "BEGIN SEQUENCE "

                *-------------------------------------------------
                * Find End.
                *-------------------------------------------------

                nPosition := cmEndPosition()

                *-------------------------------------------------
                * Save one nest level.
                * [1]BEGIN SEQUENCE  [2]nEndPosition
                *-------------------------------------------------
                nNest++
                aNest[nNest] :=;
                    {CM_ST_BEGIN, nPosition }

                *-------------------------------------------------
                * Save on the destination file.
                *-------------------------------------------------

                Field->Command := CM_ST_BEGIN
                Field->Goto1   := nPosition+1

            case upper( left( cCommand, 6 ) ) == "BREAK "

                *-------------------------------------------------
                * Save on the destination file.
                *-------------------------------------------------

                Field->Command := CM_ST_BREAK

            case upper( left( cCommand, 3 ) ) == "IF "

                *-------------------------------------------------
                * Find End.
                *-------------------------------------------------

                nPosition := cmEndPosition()

                *-------------------------------------------------
                * Find Else.
                *-------------------------------------------------

                nPosition2 := cmElsePosition()

                *-------------------------------------------------
                * Save one nest level.
                * [1]IF  [2]nEndPosition [3]nElsePosition
                *-------------------------------------------------

                nNest++
                aNest[nNest] := { CM_ST_IF, nPosition, nPosition2 }

                *-------------------------------------------------
                * Save on the destination file.
                *-------------------------------------------------

                Field->Command := CM_ST_IF
                Field->Goto1   := nPosition+1
                Field->Goto2   := nPosition2+1

            case upper( left( cCommand, 5 ) ) == "ELSE "

                *-------------------------------------------------
                * Save on the destination file.
                *-------------------------------------------------

                Field->Command := CM_ST_ELSE

            case upper( left( cCommand, 8 ) ) == "DO CASE "

                *-------------------------------------------------
                * Find End.
                *-------------------------------------------------

                nPosition := cmEndPosition()

                *-------------------------------------------------
                * Save one nest level.
                * [1]DO CASE [2]nEndPosition
                *-------------------------------------------------

                nNest++
                aNest[nNest] := {CM_ST_DOCASE, nPosition}

                *-------------------------------------------------
                * Save on the destination file.
                *-------------------------------------------------

                Field->Command := CM_ST_DOCASE
                Field->Goto1   := nPosition+1

            case upper( left( cCommand, 5 ) ) == "CASE "

                *-------------------------------------------------
                * Check if the nest level is right.
                *-------------------------------------------------

                if aNest[nNest][1] == CM_ST_DOCASE

                    *---------------------------------------------
                    * Find next Case.
                    *---------------------------------------------

                    nPosition := cmCasePosition()

                    *---------------------------------------------
                    * Save on the destination file.
                    *---------------------------------------------

                    Field->Command := CM_ST_CASE
                    Field->Goto1   := nPosition

                else

                    *---------------------------------------------
                    * This Case do not follow a DO CASE statement.
                    *---------------------------------------------

                    alertBox( "Line(" +;
                        ltrim( str( Field->Line ) ) +;
                        ")" +;
                        NL(1) +;
                        CM_ST_ERROR_ALONE_CASE )

                    nExitCode := _MACRO_EXIT_BREAK

                    break                               // BREAK

                end

            case upper( left( cCommand, 10 ) ) == "OTHERWISE "

                *-------------------------------------------------
                * Check if before there was a DO CASE statement.
                *-------------------------------------------------

                if aNest[nNest][1] == CM_ST_DOCASE

                    *---------------------------------------------
                    * Save on the destination file.
                    *---------------------------------------------

                    Field->Command := CM_ST_OTHERWISE

                else

                    *---------------------------------------------
                    * There wasn't a DO CASE before. Break.
                    *---------------------------------------------

                    alertBox( "Line(" +;
                        ltrim( str( Field->Line ) ) +;
                        ")" +;
                        NL(1) +;
                        CM_ST_ERROR_ALONE_OTHERWISE )

                    nExitCode := _MACRO_EXIT_BREAK

                    break                               // BREAK

                end

            case upper( left( cCommand, 6 ) ) == "WHILE "

                *-------------------------------------------------
                * Find End.
                *-------------------------------------------------

                nPosition := cmEndPosition()

                *-------------------------------------------------
                * Save one nest level.
                * [1]WHILE  [2]nEndPosition
                *-------------------------------------------------

                nNest++
                aNest[nNest] := {CM_ST_WHILE, nPosition}

                *-------------------------------------------------
                * Save on the destination file.
                *-------------------------------------------------

                Field->Command := CM_ST_WHILE
                Field->Goto1   := nPosition+1

            case upper( left( cCommand, 5 ) ) == "LOOP "

                *-------------------------------------------------
                * Save on the destination file.
                *-------------------------------------------------

                Field->Command := CM_ST_LOOP

            case upper( left( cCommand, 5 ) ) == "EXIT "

                *-------------------------------------------------
                * Save on the destination file.
                *-------------------------------------------------

                Field->Command := CM_ST_EXIT

            end

        end

    end //sequence

    *-------------------------------------------------------------
    * Close the wait bar.
    *-------------------------------------------------------------

    waitFileEval( .T. )

    *-------------------------------------------------------------
    * Return exit code.
    *-------------------------------------------------------------

    return nExitCode

*-----------------------------------------------------------------
static function cmEndPosition()
*
*

    local nOldRecord := recno()
    local nRecord    := recno()
    local nLevel     := 1
    local cLine

    *-------------------------------------------------------------
    * Sart a new sequence.
    *-------------------------------------------------------------

    begin sequence

        *---------------------------------------------------------
        * Scan next records to find the right "END".
        *---------------------------------------------------------

        while .T.                                       // FOREVER

            nRecord++
            dbgoto(nRecord)

            *-----------------------------------------------------
            * If the eof() is reached, it is a mistake.
            *-----------------------------------------------------

            if eof()

                alertBox( "( EOF )" + NL(1) +;
                    CM_ST_ERROR_NO_END )

                break                                   // BREAK

            end

            *-----------------------------------------------------
            * Convert <cLine> to upper case as it is easyer to
            * check.
            *-----------------------------------------------------

            cLine := upper( Field->Macro )

            do case
            case left( cLine, 4 ) == "END "

                *-------------------------------------------------
                * If this is the right end, break.
                *-------------------------------------------------

                nLevel--

                if nLevel == 0

                    *---------------------------------------------
                    * The right END is found.
                    *---------------------------------------------

                    break                               // BREAK

                end

            case left( cLine, 13 ) == "ENDPROCEDURE "

                *-------------------------------------------------
                * This is an error: "ENDPROCEDURE" close all
                * nest, so it cannot apper when we are looking
                * for a normal "END".
                *-------------------------------------------------

                alertBox( "Line(" +;
                    ltrim( str( Field->Line ) ) +;
                    ")" +;
                    NL(1) +;
                    CM_ST_ERROR_UNCLOSED_STRUCTURE )

                break                                   // BREAK

            case;
                left( cLine, 6 ) == "WHILE "            .or.;
                left( cLine, 3 ) == "IF "               .or.;
                left( cLine, 15 ) == "BEGIN SEQUENCE "  .or.;
                left( cLine, 8 ) == "DO CASE "

                *-------------------------------------------------
                * Now a new "END" must be started before.
                * This means that we have a new sub level
                * of search.
                *-------------------------------------------------

                nLevel++

            end

        end

    end //sequence

    *-------------------------------------------------------------
    * Go back to the original record.
    *-------------------------------------------------------------

    dbgoto( nOldRecord )

    *-------------------------------------------------------------
    * Return the record number where the right "END" was found.
    *-------------------------------------------------------------

    return nRecord

*-----------------------------------------------------------------
static function cmElsePosition()
*
*

    local nOldRecord := recno()
    local nRecord    := recno()
    local nLevel     := 1
    local cLine

    *-------------------------------------------------------------
    * Start a new sequence.
    *-------------------------------------------------------------

    begin sequence

        *---------------------------------------------------------
        * Scan next records to find the right "ELSE" or "END"
        * if no ELSE is found.
        *---------------------------------------------------------

        while .T.

            nRecord++
            dbgoto(nRecord)

            *-----------------------------------------------------
            * If the eof() is reached, it is a error.
            *-----------------------------------------------------

            if eof()

                alertBox( "( EOF )" +;
                    NL(1) +;
                    CM_ST_ERROR_NO_END )

                break                                   // BREAK

            end

            *-----------------------------------------------------
            * Convert <cLine> to upper case as it is easyer to
            * check.
            *-----------------------------------------------------

            cLine := upper( Field->Macro )

            do case
            case nLevel == 1;
                .and. left( cLine, 5 ) == "ELSE "

                *-------------------------------------------------
                * The right ELSE is found.
                *-------------------------------------------------

                break                                   // BREAK

            case left( cLine, 4 ) == "END "

                *-------------------------------------------------
                * Go back one sub nest level.
                *-------------------------------------------------

                nLevel--

                *-------------------------------------------------
                * If it was the last level there is no "ELSE".
                *-------------------------------------------------

                if nLevel == 0

                    *---------------------------------------------
                    * The right END is found.
                    *---------------------------------------------

                    break                               //BREAK

                end

            case left( cLine, 13 ) == "ENDPROCEDURE "

                *-------------------------------------------------
                * This is an error: "ENDPROCEDURE" close all
                * nest, so it cannot apper when we are looking
                * for a "ELSE" or a normal "END".
                *-------------------------------------------------

                alertBox( "Line(" +;
                    ltrim( str( Field->Line ) ) +;
                    ")" +;
                    NL(1) +;
                    CM_ST_ERROR_UNCLOSED_STRUCTURE )

                break                                   // BREAK

            case;
                left( cLine, 6 ) == "WHILE "            .or.;
                left( cLine, 3 ) == "IF "               .or.;
                left( cLine, 15 ) == "BEGIN SEQUENCE "  .or.;
                left( cLine, 8 ) == "DO CASE "

                *-------------------------------------------------
                * Now a new "END" search must be started before
                * the search for a "ELSE".
                * This means that we have a new sub level
                * of search.
                *-------------------------------------------------

                nLevel++

            end
        end

    end //sequence

    *-------------------------------------------------------------
    * Go back to the original record.
    *-------------------------------------------------------------

    dbgoto( nOldRecord )

    *-------------------------------------------------------------
    * Return the record number where the right "ELSE" or "END"
    * was found.
    *-------------------------------------------------------------

    return nRecord

*-----------------------------------------------------------------
static function cmCasePosition()
*
*

    local nOldRecord := recno()
    local nRecord    := recno()
    local nLevel     := 1
    local cLine

    *-------------------------------------------------------------
    * Start a new sequence.
    *-------------------------------------------------------------

    begin sequence

        *---------------------------------------------------------
        * Scan next records to find the right "CASE",
        * "OTHERWISE" or "END".
        *---------------------------------------------------------

        while .T.

            nRecord++
            dbgoto(nRecord)

            *-----------------------------------------------------
            * If the eof() is reached, it is a error.
            *-----------------------------------------------------

            if eof()

                alertBox( "( EOF )" +;
                    NL(1) +;
                    CM_ST_ERROR_NO_END )

                break                               // BREAK

            end

            cLine := upper( Field->Macro )

            do case
            case nLevel == 1;
                .and. (;
                     left( cLine, 5 ) == "CASE ";
                     .or. left( cLine, 10 ) == "OTHERWISE ";
                     )

                *-------------------------------------------------
                * The right CASE or OTHERWISE is found.
                *-------------------------------------------------

                break                                   // BREAK

            case left( cLine, 4 ) == "END "

                *-------------------------------------------------
                * Go back one sub nest level.
                *-------------------------------------------------

                nLevel--

                *-------------------------------------------------
                * If it was the last level there is no "CASE"
                * or "OTHERWISE".
                *-------------------------------------------------

                if nLevel == 0

                    *---------------------------------------------
                    * The right END is found.
                    *---------------------------------------------

                    break                               //BREAK

                end

            case left( cLine, 13 ) == "ENDPROCEDURE "

                *-------------------------------------------------
                * This is an error: "ENDPROCEDURE" close all
                * nest, so it cannot apper when we are looking
                * for a "ELSE" or a normal "END".
                *-------------------------------------------------

                alertBox( "Line(" +;
                    ltrim( str( Field->Line ) ) +;
                    ")" +;
                    NL(1) +;
                    CM_ST_ERROR_UNCLOSED_STRUCTURE )

                break                                   // BREAK

            case;
                left( cLine, 6 ) == "WHILE "            .or.;
                left( cLine, 3 ) == "IF "               .or.;
                left( cLine, 15 ) == "BEGIN SEQUENCE "  .or.;
                left( cLine, 8 ) == "DO CASE "

                *-------------------------------------------------
                * Now a new "END" search must be started before
                * the search for a "CASE", "OTHERWISE".
                * This means that we have a new sub level
                * of search.
                *-------------------------------------------------

                nLevel++

            end
        end

    end //sequence

    *-------------------------------------------------------------
    * Go back to the original record.
    *-------------------------------------------------------------

    dbgoto( nOldRecord )

    *-------------------------------------------------------------
    * Return the record number where the right "CASE", or
    * "OTHERWISE" or "END" was found.
    *-------------------------------------------------------------

    return nRecord

*-----------------------------------------------------------------
static function  cmScanBack( aNest, nNest, cName )

    begin sequence

        while nNest > 0
            if aNest[nNest][1] == cName
                // Found.
                break
            else
                nNest--
            end
        end

    end //sequence

return nNest

*-----------------------------------------------------------------
static function cmExecute( cFileName )
*
* cmExecute( <cFileName> ) --> nExitCode
*
* <cFileName>  nB compiled macro file.
*
*

    local bSaveErrorHandler
    local nOldSelect          := select()
    local nMacroArea
    local cMacroAlias
    local aNest               := array( CM_ST_MAX_NEST )
    local nNest               := 0
    local nRecord
    local cCondition          := ""
    local xResult

    local nExitCode           := _MACRO_EXIT_NORMAL

    *-------------------------------------------------------------
    * Start a new sequence.
    *-------------------------------------------------------------

    begin sequence

        *---------------------------------------------------------
        * Test for valid data.
        *---------------------------------------------------------

        if valtype( cFileName ) == "C"
            cFileName := alltrim( cFileName )
        else
            break                                        // BREAK
        end

        *---------------------------------------------------------
        * Test for macro file existance.
        *---------------------------------------------------------

        if !file( cFileName )
            alertBox( cFileName + NL(1) +;
                CM_ERROR_FILE_NOT_FOUND )
            nExitCode := _MACRO_EXIT_NO_MACRO_FILE

            break                                        // BREAK

        end

        *---------------------------------------------------------
        * Open the macro file.
        *---------------------------------------------------------

        nMacroArea         := cmLastArea()
        cMacroAlias     :=;
            _MACRO_ALIAS +;
            padl( ltrim( str( nMacroArea ) ), 3, "0" )

        (nMacroArea)->( dbusearea( .F., _DEFAULT_RDD, cFileName,;
            cMacroAlias, .T., .T. ) )

        *---------------------------------------------------------
        * Main nest level.
        *---------------------------------------------------------

        nNest := 1
        aNest[nNest]          := { CM_ST_MAIN }
        // [1]MAIN

        nRecord := 1
        (cMacroAlias)->(dbgoto( nRecord ))

        while !(cMacroAlias)->(eof())

            do case
            case (cMacroAlias)->Command ==;
                CM_ST_END

                *-------------------------------------------------
                * END was reached.
                * Try to close one nest level.
                *-------------------------------------------------

                do case
                case aNest[nNest][1] == CM_ST_MAIN

                    *---------------------------------------------
                    * Not allowed: this END statement cannot
                    * be here as the main statement is not closed
                    * with a END.
                    *---------------------------------------------

                    alertBox( cFileName +;
                        "(" +;
                        ltrim(str((cMacroAlias)->Line )) +;
                        ")" +;
                        NL(1) +;
                        CM_ST_ERROR_ALONE_END )
                    nExitCode := _MACRO_EXIT_STATEMENT_ERROR

                    break                               // BREAK

                case aNest[nNest][1] == CM_ST_IF;
                    .or. aNest[nNest][1] == CM_ST_THEN;
                    .or. aNest[nNest][1] == CM_ST_ELSE

                    *---------------------------------------------
                    * The current nest level can be closed, and
                    * the record pointer can be incremented.
                    *---------------------------------------------

                    aNest[nNest] := NIL
                    nNest--
                    nRecord++

                case aNest[nNest][1] == CM_ST_BEGIN

                    *---------------------------------------------
                    * The current nest level can be closed, and
                    * the record pointer can be incremented.
                    *---------------------------------------------

                    aNest[nNest] := NIL
                    nNest--
                    nRecord++

                case aNest[nNest][1] == CM_ST_WHILE

                    *---------------------------------------------
                    * The current nest level may be closed
                    * only if the "While condition" is no more
                    * valid.
                    *---------------------------------------------

                    *---------------------------------------------
                    * Test the condition.
                    *---------------------------------------------
                    
                    xResult :=; 
                        exEvalCondition(; 
                            aNest[nNest][4],; 
                            cFileName, (cMacroAlias)->Line; 
                        )

                    *---------------------------------------------
                    * If the condition is not valid, <xReturn>
                    * contains NIL, else it has a logical 
                    * value.
                    *---------------------------------------------
                
                    if xResult == NIL

                        break                               // BREAK

                    end

                    if xResult

                        *-----------------------------------------
                        * Continue While loop.
                        *-----------------------------------------

                        nRecord := aNest[nNest][3]

                    else

                        *-----------------------------------------
                        * Terminate While loop.
                        *-----------------------------------------

                        aNest[nNest] := NIL
                        nNest--
                        nRecord++

                    end

                case aNest[nNest][1] == CM_ST_DOCASE

                    *---------------------------------------------
                    * The current nest level can be closed, and
                    * the record pointer can be incremented.
                    *---------------------------------------------

                    aNest[nNest] := NIL
                    nNest--
                    nRecord++

                case aNest[nNest][1] == CM_ST_CASEMATCHED

                    *---------------------------------------------
                    * The current nest level can be closed, and
                    * the record pointer can be incremented.
                    *---------------------------------------------

                    aNest[nNest] := NIL
                    nNest--
                    nRecord++

                end

            case (cMacroAlias)->Command ==;
                CM_ST_PROCEDURE

                *-------------------------------------------------
                * If the begin of a procedure is found, the
                * lines contained inside the procedure are
                * ignored: they will be executed when a call to
                * the procedure is reached.
                * Jump after procedure.
                *-------------------------------------------------

                nRecord := (cMacroAlias)->Goto1

            case (cMacroAlias)->Command ==;
                CM_ST_DOPROCEDURE

                *-------------------------------------------------
                * When a procedure call is reached, a new nest
                * level is created and the record pointer is
                * moved to the begin of the called procedure.
                *-------------------------------------------------

                nNest++
                aNest[nNest] :=;
                    { CM_ST_DOPROCEDURE, nRecord+1 }
                // [1]DOPROCEDURE  [2]nReturnLine

                *-------------------------------------------------
                * Jump to procedure begin.
                *-------------------------------------------------

                nRecord := (cMacroAlias)->Goto1

            case (cMacroAlias)->Command ==;
                CM_ST_RETURN

                *-------------------------------------------------
                * The RETURN statement closes a procedure or the
                * program execution.
                *-------------------------------------------------

                while .T.
                    do case
                    case aNest[nNest][1] == CM_ST_DOPROCEDURE

                        *-----------------------------------------
                        * Ok, ready to return.
                        *-----------------------------------------

                        nRecord := aNest[nNest][2]
                        aNest[nNest] := NIL
                        nNest--

                        exit                            // EXIT

                    case aNest[nNest][1] == CM_ST_MAIN

                        *-----------------------------------------
                        * This RETURN must be considered
                        * as a "terminate file execution".
                        *-----------------------------------------

                        break                           // BREAK

                    otherwise

                        *-----------------------------------------
                        * This happens when RETURN appears inside
                        * another nest level (if/when/case...).
                        * The nest level is closed, but the
                        * record pointer is not incremented:
                        * the next loop will find again the
                        * RETURN statement.
                        *-----------------------------------------

                        aNest[nNest] := NIL
                        nNest--

                    end

                end

            case (cMacroAlias)->Command ==;
                CM_ST_ENDPROCEDURE

                *-------------------------------------------------
                * END PROCEDURE is reached. If it was reached
                * after a procedure call, it is equivalent to
                * a return; if not, it is a mistake.
                *-------------------------------------------------

                do case
                case aNest[nNest][1] == CM_ST_DOPROCEDURE

                    *---------------------------------------------
                    * Ok, return.
                    *---------------------------------------------

                    nRecord := aNest[nNest][2]
                    aNest[nNest] := NIL
                    nNest--

                otherwise

                    *---------------------------------------------
                    * The END PROCEDURE cannot be encountered
                    * here.
                    *---------------------------------------------

                    alertBox( cFileName +;
                        "(" +;
                        ltrim(str((cMacroAlias)->Line)) +;
                        ")" +;
                        CM_ST_ERROR_ALONE_ENDPROCEDURE )

                    nExitCode := _MACRO_EXIT_STATEMENT_ERROR

                    break                                // BREAK

                end

            case (cMacroAlias)->Command ==;
                CM_ST_BEGIN

                *-------------------------------------------------
                * A new BEGIN statement is found: a new nest
                * level is created.
                *-------------------------------------------------

                nNest++
                aNest[nNest] :=;
                    { CM_ST_BEGIN, (cMacroAlias)->Goto1 }
                // [1]BEGIN  [2] nAfterEnd

                nRecord++

            case (cMacroAlias)->Command ==;
                CM_ST_BREAK

                *-------------------------------------------------
                * This should break the sequence.
                *-------------------------------------------------

                while .T.                                // FOREVER
                    do case
                    case aNest[nNest][1] == CM_ST_BEGIN

                        *-----------------------------------------
                        * This nest level can be closed and the
                        * record pointer placed over the end of
                        * sequence.
                        *-----------------------------------------

                        nRecord := aNest[nNest][2]

                        aNest[nNest] := NIL
                        nNest--

                        exit                                // EXIT

                    case aNest[nNest][1] == CM_ST_MAIN

                        *-----------------------------------------
                        * a BREAK at the main nest level is not
                        * good.
                        *-----------------------------------------

                        alertBox( cFileName +;
                            "(" +;
                            ltrim(str((cMacroAlias)->Line)) +;
                            ")" +;
                           CM_ST_ERROR_ALONE_BREAK )

                        nExitCode := _MACRO_EXIT_STATEMENT_ERROR

                        break                           // BREAK

                    otherwise

                        *-----------------------------------------
                        * This BREAK don't belogns this level,
                        * but probably the previuous one.
                        * So, try to close one nest level,
                        * without moving the record pointer.
                        * The next loop will find again a BREAK.
                        *-----------------------------------------

                        aNest[nNest] := NIL
                        nNest--

                    end

                end

            case (cMacroAlias)->Command ==;
                CM_ST_IF

                *-------------------------------------------------
                * IF was found: try to isolate the condition.
                *-------------------------------------------------

                cCondition :=;
                    substr( (cMacroAlias)->Macro, 3 )

                *-------------------------------------------------
                * Test the condition.
                *-------------------------------------------------
                    
                xResult :=; 
                    exEvalCondition(; 
                        cCondition,; 
                        cFileName, (cMacroAlias)->Line; 
                    )

                *-------------------------------------------------
                * If the condition is not valid, <xReturn>
                * contains NIL, else it has a logical 
                * value.
                *-------------------------------------------------
                
                if xResult == NIL

                    break                               // BREAK

                end

                do case
                case xResult

                    *---------------------------------------------
                    * If the condition gives a True value, a new
                    * nest level is created and the record pointer
                    * is incremented to reach the following lines.
                    *---------------------------------------------

                    nNest++
                    aNest[nNest] :=;
                        { CM_ST_IF,;
                        (cMacroAlias)->Goto1 }
                    // [1]IF  [2]nAfterEnd

                    nRecord++

                otherwise

                    *---------------------------------------------
                    * The condition gives False and a ELSE or
                    * ENDif must be searched for.
                    *---------------------------------------------

                    do case
                    case (cMacroAlias)->Goto1 ==;
                        (cMacroAlias)->Goto2

                        *-----------------------------------------
                        * Then there is no Else.
                        * Jump over End.
                        *-----------------------------------------

                        nRecord := (cMacroAlias)->Goto1

                    case (cMacroAlias)->Goto1 >;
                        (cMacroAlias)->Goto2

                        *-----------------------------------------
                        * There is a Else:
                        * a new nest level is created and the
                        * record pointer is moved at the begin
                        * of statements following the ELSE.
                        *-----------------------------------------

                        nNest++
                        aNest[nNest] :=;
                            { CM_ST_IF,;
                            (cMacroAlias)->Goto1 }
                            // [1]IF  [2]nAfterEnd

                        nRecord := (cMacroAlias)->Goto2

                    otherwise

                        *-----------------------------------------
                        * This should not happen: it is a compiler
                        * error.
                        *-----------------------------------------

                    end
                end

            case (cMacroAlias)->Command ==;
                CM_ST_ELSE

                *-------------------------------------------------
                * ELSE can be reached only if the lines after
                * a IF were executed. In this case, this means
                * that the IF is terminated as the ELSE is not
                * to be executed.
                *-------------------------------------------------

                do case
                case aNest[nNest][1] == CM_ST_IF

                    *---------------------------------------------
                    * A Then was executed, so
                    * jump it and close one nest level.
                    *---------------------------------------------

                    nRecord := aNest[nNest][2]

                    aNest[nNest] := NIL
                    nNest--

                otherwise

                    *---------------------------------------------
                    * This ELSE is not good here.
                    *---------------------------------------------

                    alertBox( cFileName +;
                        "(" +;
                        ltrim(str((cMacroAlias)->Line)) +;
                        ")" +;
                        CM_ST_ERROR_ALONE_ELSE )

                    nExitCode := _MACRO_EXIT_STATEMENT_ERROR

                    break                                // BREAK

                end

            case (cMacroAlias)->Command ==;
                CM_ST_DOCASE

                *-------------------------------------------------
                * A DO CASE is found. A new nest level is created.
                *-------------------------------------------------

                nNest++
                aNest[nNest] :=;
                    { CM_ST_DOCASE, (cMacroAlias)->Goto1 }
                // [1]DOCASE  [2]nAfterEndCase

                nRecord++

            case (cMacroAlias)->Command ==;
                CM_ST_CASE

                *-------------------------------------------------
                * If a CASE is found, it may appear after another
                * successfull CASE or not. If not, the condition
                * is tested, else the record pointer is moved
                * after the ENDcase.
                *-------------------------------------------------

                do case
                case aNest[nNest][1] == CM_ST_CASEMATCHED

                    *---------------------------------------------
                    * A CASE with a condition that was True was
                    * found before. This means that this CASE
                    * is not to be tested. So:
                    * jump after ENDcase and close this nest
                    * level.
                    *---------------------------------------------

                    nRecord := aNest[nNest][2]

                    aNest[nNest] := NIL
                    nNest--

                case aNest[nNest][1] == CM_ST_DOCASE

                    *---------------------------------------------
                    * No previous CASE with a True condition were
                    * found. The condition of this CASE must be
                    * tested.
                    *---------------------------------------------

                    cCondition :=;
                        substr( (cMacroAlias)->Macro, 5 )

                    *---------------------------------------------
                    * Test the condition.
                    *---------------------------------------------
                    
                    xResult :=; 
                        exEvalCondition(; 
                            cCondition,; 
                            cFileName, (cMacroAlias)->Line; 
                        )

                    *---------------------------------------------
                    * If the condition is not valid, <xReturn>
                    * contains NIL, else it has a logical 
                    * value.
                    *---------------------------------------------
                
                    if xResult == NIL

                        break                               // BREAK

                    end

                    do case
                    case xResult

                        *-----------------------------------------
                        * As the condition gives a True value,
                        * the actual nest level is transformed to
                        * tell that a CASE with a true result
                        * was found.
                        *-----------------------------------------

                        aNest[nNest][1] := CM_ST_CASEMATCHED

                        *-----------------------------------------
                        * No more CASEs.
                        * Execute following lines until next
                        * CASE or OTHERWISE or END.
                        *-----------------------------------------

                        nRecord++

                    otherwise

                        *-----------------------------------------
                        * The CASE condition gived a False result:
                        * the next CASE or OTHERWISE or ENDcase
                        * must be reached.
                        *-----------------------------------------

                        nRecord := (cMacroAlias)->Goto1

                    end

                otherwise

                    *---------------------------------------------
                    * If a CASE appears outside a DO CASE..END
                    * it is an error.
                    *---------------------------------------------

                    alertBox( cFileName +;
                        "(" +;
                        ltrim(str((cMacroAlias)->Line)) +;
                        ")" +;
                        CM_ST_ERROR_ALONE_CASE )

                    nExitCode := _MACRO_EXIT_STATEMENT_ERROR

                    break                                // BREAK

                end

            case (cMacroAlias)->Command ==;
                CM_ST_OTHERWISE

                *-------------------------------------------------
                * OTHERWISE was found: if the last CASE was
                * executed the record pointer must be placed
                * over the ENDcase, else the OTHERWISE is to
                * be executed.
                *-------------------------------------------------

                do case
                case aNest[nNest][1] == CM_ST_CASEMATCHED

                    *---------------------------------------------
                    * The last CASE had a True condition and it
                    * was executed. So:
                    * jump after endcase and close the nest level.
                    *---------------------------------------------

                    nRecord := aNest[nNest][2]

                    aNest[nNest] := NIL
                    nNest--

                case aNest[nNest][1] == CM_ST_DOCASE

                    *---------------------------------------------
                    * If no CASE had a True condition, it is
                    * equivalent to find a final CASE with a True
                    * condition. So:
                    * Change the nest level into a CASE with True
                    * condition found and move the record pointer
                    * to the first line inside this OTHERWISE.
                    *---------------------------------------------

                    aNest[nNest][1] := CM_ST_CASEMATCHED

                    nRecord++

                otherwise

                    *---------------------------------------------
                    * This is not the right place for a OTHERWISE.
                    *---------------------------------------------

                    alertBox( cFileName +;
                        "(" +;
                        ltrim( str( (cMacroAlias)->Line ) ) +;
                        ")" +;
                        CM_ST_ERROR_ALONE_OTHERWISE )

                    nExitCode := _MACRO_EXIT_STATEMENT_ERROR

                    break                               // BREAK

                end

            case (cMacroAlias)->Command ==;
                CM_ST_WHILE

                *-------------------------------------------------
                * A WHILE is found.
                * If the condition is not True, the WHILE is
                * simply jumped without the creation of a new
                * next level. If the condition is True at least
                * for the first time, the nest level is created
                * and the loop may begin.
                *-------------------------------------------------

                cCondition :=;
                    substr( (cMacroAlias)->Macro, 6 )

                *-------------------------------------------------
                * Test the condition.
                *-------------------------------------------------
                    
                xResult :=; 
                    exEvalCondition(; 
                        cCondition,; 
                        cFileName, (cMacroAlias)->Line; 
                    )
                *-------------------------------------------------
                * If the condition is not valid, <xReturn>
                * contains NIL, else it has a logical 
                * value.
                *-------------------------------------------------
                
                if xResult == NIL

                    break                               // BREAK

                end

                do case
                case xResult

                    *---------------------------------------------
                    * The nest level is created and the record
                    * pointer is incremented.
                    *---------------------------------------------

                    nNest++
                    aNest[nNest] :=;
                        { CM_ST_WHILE,;
                        (cMacroAlias)->Goto1,;
                        nRecord+1,;
                        cCondition }
                    // [1]WHILE  [2]nAfterEndWhile
                    // [3]nLine1stStatement [4]cCondition

                    nRecord++

                otherwise

                    *---------------------------------------------
                    * No nest level is created and the record
                    * pointer is moved after the ENDwhile.
                    *---------------------------------------------

                    nRecord := (cMacroAlias)->Goto1

                end

            case (cMacroAlias)->Command ==;
                CM_ST_LOOP

                *-------------------------------------------------
                * LOOP should move the record pointer to the
                * begin of the WHILE or over the ENDwhile
                * depending on the original WhileCondition.
                *-------------------------------------------------

                while .T.                                // FOREVER
                    do case
                    case aNest[nNest][1] == CM_ST_WHILE

                        *-----------------------------------------
                        * A LOOP inside a WHILE nest level is OK.
                        *-----------------------------------------
                    
                        *-----------------------------------------
                        * Test the condition.
                        *-----------------------------------------
                    
                        xResult :=; 
                            exEvalCondition(; 
                                aNest[nNest][4],; 
                                cFileName, (cMacroAlias)->Line; 
                            )

                        *-----------------------------------------
                        * If the condition is not valid, <xReturn>
                        * contains NIL, else it has a logical 
                        * value.
                        *-----------------------------------------
                
                        if xResult == NIL

                            break                       // BREAK

                        end

                        do case
                        case xResult

                            *-------------------------------------
                            * The condition is still valid: loop.
                            *-------------------------------------

                            nRecord := aNest[nNest][3]

                        otherwise

                            *-------------------------------------
                            * The condition is not valid:
                            * exit the WHILE and close the nest
                            * level.
                            *-------------------------------------

                            nRecord := aNest[nNest][2]

                            aNest[nNest] := NIL
                            nNest--

                        end

                        exit                                // EXIT

                    case aNest[nNest][1] == CM_ST_MAIN

                        *-----------------------------------------
                        * A LOOP inside a MAIN nest level is not
                        * good.
                        *-----------------------------------------

                        alertBox( cFileName +;
                            "(" +;
                            ltrim( str( (cMacroAlias)->Line ) ) +;
                            ")" +;
                            CM_ST_ERROR_ALONE_LOOP )

                        nExitCode := _MACRO_EXIT_STATEMENT_ERROR

                        break                                // BREAK

                    otherwise

                        *-----------------------------------------
                        * As this is not the right nest level,
                        * close it without moving the record
                        * pointer.
                        *-----------------------------------------

                        aNest[nNest] := NIL
                        nNest--

                    end

                end

            case (cMacroAlias)->Command ==;
                CM_ST_EXIT

                *-------------------------------------------------
                * EXIT should move the record pointer to the
                * line following the ENDwhile.
                *-------------------------------------------------

                while .T.                                // FOREVER
                    do case
                    case aNest[nNest][1] == CM_ST_WHILE

                        *-----------------------------------------
                        * This is the right nest level: move the
                        * record pointer after the ENDwhile and
                        * close the nest level.
                        *-----------------------------------------

                        nRecord := aNest[nNest][2]

                        aNest[nNest] := NIL
                        nNest--

                        exit                                // EXIT

                    case aNest[nNest][1] == CM_ST_MAIN

                        *-----------------------------------------
                        * The statement MAIN cannot be closed with
                        * a EXIT.
                        *-----------------------------------------

                        alertBox( cFileName +;
                            "(" +;
                            ltrim( str( (cMacroAlias)->Line ) ) +;
                            ")" +;
                            CM_ST_ERROR_ALONE_EXIT )

                        nExitCode := _MACRO_EXIT_STATEMENT_ERROR

                        break                                // BREAK

                    otherwise

                        *-----------------------------------------
                        * If EXIT appears inside a different
                        * nest level, the actual nest level must
                        * be closed in the hope to find a WHILE
                        * nest before. The record pointer is not
                        * moved.
                        *-----------------------------------------

                        aNest[nNest] := NIL
                        nNest--

                    end

                end

            otherwise

                *-------------------------------------------------
                * No other statements are known: it must be
                * a function or an assignment.
                * Before trying to executing it with the macro
                * operator "&", it is better to assign a different
                * error handler made for the macro file
                * interpretation.
                *-------------------------------------------------

                bSaveErrorHandler :=;
                    errorblock(;
                        { |e|;
                            errorMacro(;
                                e,;
                                cFileName,;
                                (cMacroAlias)->Line,;
                                rtrim((cMacroAlias)->Macro);
                            );
                         };
                    )
                begin sequence

                    *---------------------------------------------
                    * Execute with macro compiler.
                    *---------------------------------------------

                    xResult := &((cMacroAlias)->Macro)

                recover

                    *---------------------------------------------
                    * errorChoice is a static global variable that
                    * contains the choice made after the error
                    * was intercepted.
                    *---------------------------------------------

                    do case
                    case errorChoice == CM_ERROR_MENU_IGNORE
                        // Ignore error.
                    case errorChoice == CM_ERROR_MENU_BREAK
                        nExitCode := _MACRO_EXIT_BREAK
                    end

                end //sequence
                errorblock(bSaveErrorHandler)

                *-------------------------------------------------
                * If after a error a BREAK was selected, the
                * macro interpretation must be stopped.
                *-------------------------------------------------

                if nExitCode == _MACRO_EXIT_BREAK

                    break                                // BREAK

                end

                *-------------------------------------------------
                * The record pointer is incremented.
                *-------------------------------------------------

                nRecord++

            end

            *-----------------------------------------------------
            * The real record pointer is moved.
            *-----------------------------------------------------

            (cMacroAlias)->(dbgoto( nRecord ))

        end

    end //sequence

    *-------------------------------------------------------------
    * The macro interpretation is terminated. Normally, the macro
    * file is still opened, but maybe it is not. Try to close it.
    *-------------------------------------------------------------

    if select(cMacroAlias) > 0
        // the file is still open
        (cMacroAlias)->(dbclosearea())
    end

    select( nOldSelect )

    return nExitCode

*-----------------------------------------------------------------
static function cmLastArea()
*
* This function returns a free area number with the highest value.
*

    local nSelect

    for nSelect := _MAX_SELECT to 1 step -1

        if !(nSelect)->(used())

            return nSelect

        end

    end

    *-------------------------------------------------------------
    * The normal return is not here.
    *-------------------------------------------------------------

    return 0

*=================================================================
function colorArray( cColor )
*
* colorArray( <cColor> ) --> aColors
*
* <cColor>        is the color string to be translated into an
*               array.
*
* This function returns an array of colors. The array has as many
* elements as the colors contained inside <cColor> string.
*

    local aColors         := {}
    local cRest
    local nCommaPos       := 0

    default( @cColor,   setcolor() )

    cRest := cColor

    *-------------------------------------------------------------
    * The <aColors> array is incremented with the colors
    * contained inside the string <cColor>.
    *-------------------------------------------------------------

    while .T.                                           // FOREVER

        nCommaPos := at( ",", cRest )

        do case
        case nCommaPos > 0

            *-----------------------------------------------------
            * Isolate the color before the comma.
            *-----------------------------------------------------

            aadd( aColors, substr( cRest, 1, nCommaPos-1 ) )

            *-----------------------------------------------------
            * Assign to <cRest> the remaining colors.
            *-----------------------------------------------------

            cRest := substr( cRest, nCommaPos+1 )

        case empty( cRest )

            *-----------------------------------------------------
            * <cRest> contains no more colors.
            *-----------------------------------------------------

            exit                                       // EXIT

        otherwise

            *-----------------------------------------------------
            * <cRest> contains the last color.
            *-----------------------------------------------------

            aadd( aColors, cRest )

            *-----------------------------------------------------
            * Clear <cRest>.
            *-----------------------------------------------------

            cRest := ""

        end

    end

    *-------------------------------------------------------------
    * The color array is returned.
    *-------------------------------------------------------------

    return aColors

*=================================================================
* COORDINATE()
*=================================================================
function coordinate(;
        nTop, nLeft, nBottom, nRight,;
        cHorizontal,;
        cVertical;
    )
*
* <@nTop>, <@Left>, <@Bottom>, <nRight>
*                   are the original coordinates and will be
*                   returned corrected for the new alignment.
* <cHorizontal>     the horizontal alignment: "L", "C", "T",
*                   "l", "c", "t".
* <cVertical>       the vertical alignment: "T", "C", "B",
*                   "t", "c", "b".
*
* Calculate a new position aligned differently.
*

    local nHorDim
    local nVerDim

    default( @nTop,     0 )
    default( @nLeft,    0 )
    default( @nBottom,  maxrow() )
    default( @nRight,   maxcol() )
    default( @cHorizontal,  "C" )
    default( @cVertical,    "C" )

    nHorDim := nRight - nLeft +1
    nVerDim := nBottom - nTop +1

    *-------------------------------------------------------------
    * Align horizontally.
    *-------------------------------------------------------------

    do case
    case cHorizontal == "L"

        *---------------------------------------------------------
        * Align Left.
        *---------------------------------------------------------

        nLeft := ( maxcol()+1 - nHorDim ) / 4
        nRight := nLeft + nHorDim -1

    case cHorizontal == "l"

        *---------------------------------------------------------
        * Align relative Left.
        *---------------------------------------------------------

        nLeft := 0
        nRight := nLeft + nHorDim -1

    case cHorizontal == "R"

        *---------------------------------------------------------
        * Align Right.
        *---------------------------------------------------------

        nRight := maxcol()
        nLeft := nRight - nHorDim +1

    case cHorizontal == "r"

        *---------------------------------------------------------
        * Align relative Right.
        *---------------------------------------------------------

        nRight := maxcol() - ( maxcol()+1 - nHorDim ) / 4
        nLeft := nRight - nHorDim +1

    otherwise

        *---------------------------------------------------------
        * Center horizontally.
        *---------------------------------------------------------

        nLeft := ( maxcol()+1 - nHorDim ) / 2
        nRight := nLeft + nHorDim -1

    end

    *-------------------------------------------------------------
    * Align vertically.
    *-------------------------------------------------------------

    do case
    case cVertical == "T"

        *---------------------------------------------------------
        * Align Top.
        *---------------------------------------------------------

        nTop := 0
        nBottom := nTop + nVerDim -1

    case cVertical == "t"

        *---------------------------------------------------------
        * Align relative Top.
        *---------------------------------------------------------

        nTop := ( maxrow()+1 - nVerDim ) / 4
        nBottom := nTop + nVerDim -1

    case cVertical == "B"

        *---------------------------------------------------------
        * Align Bottom.
        *---------------------------------------------------------

        nBottom := maxrow()
        nTop := nBottom - nVerDim +1

    case cVertical == "b"

        *---------------------------------------------------------
        * Align rlative Bottom.
        *---------------------------------------------------------

        nBottom := maxrow() - ( maxrow()+1 - nVerDim ) / 4
        nTop := nBottom - nVerDim +1

    otherwise

        *---------------------------------------------------------
        * Center Vertically.
        *---------------------------------------------------------

        nTop := ( maxrow()+1 - nVerDim ) / 2
        nBottom := nTop + nVerDim -1

    end

    return NIL

*=================================================================
* COPYFILE()
*=================================================================
function copyFile( cSource, cDestination )
*
* copyFile( <cSource>, <cDestination> ) --> NIL
*
* Copy File command substitute.
*

    // COPY FILE (cSource) TO (cDestination)
    __CopyFile( cSource, cDestination )

    return NIL

*=================================================================
* DBAPP()
* DBCLOSE()
* DBCONTINUE()
* DBCOPY()
* DBCOPYSTRUCT
* DBCOPYXSTRUCT
* DBDELIM()
* DBJOIN()
* DBLABELFORM()
* DBLIST()
* DBLOCATE()
* DBOLDCREATE()
* DBPACK()
&&&&&&&&* DBREPORTFORM()
* DBSDF()
* DBSORT()
* DBTOTAL()
* DBUPDATE()
* DBZAP()
*=================================================================
function dbApp( cFileName, acFields,;
                bForCondition, bWhileCondition,;
                nNextRecords, nRecord, lRest, cDriver )
*
* dbApp( <cFileName>,
*              [<acFields>], [<bForCondition>],
*              [<bWhileCondition>], [<nNextRecords>], [<nRecord>],
*              [<lRest>], [<cDriver>] ) --> NIL
*
* Import records from a (.dbf) file.
*
* <cFileName>          the (.dbf) file name to use to
*                      import data.
* <acFields>           the fields to be involved.
* <bForCondition>      code block FOR condition.
* <bWhileCondition>    code block WHILE condition.
* <nNextRecords>       equivalent to NEXT ...
* <nRecord>            equivalent to RECORD ...
* <lRest>              if true (.T.), then REST records.
* <cDriver>            the RDD name to use.
*

    // APPEND [FROM <xcFile>]
    //     [FIELDS <Fields,...>]
    //     [FOR <forCondition>]
    //     [WHILE <whileCondition>]
    //     [NEXT <next>]
    //     [RECORD <rec>]
    //     [REST]
    //     [ALL]
    __dbApp( cFileName, acFields,;
                bForCondition, bWhileCondition,;
                nNextRecords, nRecord, lRest, cDriver )

    return NIL

*=================================================================
function dbClose()
*
* This function is made in substitution to DBCLOSEALL() to avoid
* the macro file(s), _MACRO_ALIAS to be closed during
* macro execution.
*

    local nSelect

    for nSelect := 1 to _MAX_SELECT
        if  (nSelect)->(used()) .and.;
            left( (nSelect)->(alias()), 7 ) <> _MACRO_ALIAS

            (nSelect)->(dbclosearea())
        end
    end

    return NIL

*=================================================================
function dbContinue()
*
* dbContinue() --> NIL
*
* Continue command substitute.
*

    // CONTINUE
    __dbContinue()

    return NIL

*=================================================================
function dbCopy( cFileName, acFields,;
                bForCondition, bWhileCondition,;
                nNextRecords, nRecord, lRest, cDriver )
*
* dbCopy( <cFileName>,
*              [<acFields>], [<bForCondition>],
*              [<bWhileCondition>], [<nNextRecords>], [<nRecord>],
*              [<lRest>], [<cDriver>] ) --> NIL
*
* Export records to a new (.dbf) file.
*
* <cFileName>          the (.dbf) file name to create.
* <acFields>           the fields to be involved.
* <bForCondition>      code block FOR condition.
* <bWhileCondition>    code block WHILE condition.
* <nNextRecords>       equivalent to NEXT ...
* <nRecord>            equivalent to RECORD ...
* <lRest>              if true (.T.), then REST records.
* <cDriver>            the RDD name to use.
*

    // COPY [TO <xcFile>]
    //     [FIELDS <Fields,...>]
    //     [FOR <forCondition>]
    //     [WHILE <whileCondition>]
    //     [NEXT <next>]
    //     [RECORD <rec>]
    //     [REST]
    //     [ALL]
    __dbCopy( cFileName, acFields,;
                bForCondition, bWhileCondition,;
                nNextRecords, nRecord, lRest, cDriver )

    return NIL

*=================================================================
function dbCopyStruct( cFileName, acFields )
*
* dbCopyStruct( <cFileName>, [<acFields>] )  --> NIL
*
* Copy the current (.dbf) structure to a new (.dbf) file.
*

    // COPY [STRUCTURE] [TO <xcStructureFile>]
    //     [FIELDS <fields,...>]
    __dbCopyStruct( cFileName, acFields )

    return NIL

*=================================================================
function dbCopyXStruct( cFileStructure )
*
* dbCopyXStruct( <cFileStructure> )  --> NIL
*
* Copy field definitions to a (.dbf) file.
*

    // COPY STRUCTURE EXTENDED TO (cFileStructure)
    __dbCopyXStruct( cFileStructure )

    return NIL

*=================================================================
function dbDelim( lCopy, cFileName, cDelimiter, acFields,;
                bForCondition, bWhileCondition,;
                nNextRecords, nRecord, lRest )
*
* dbDelim( <lCopy>, <cFileName>, [<cDelimiter>],
*              [<acFields>], [<bForCondition>],
*              [<bWhileCondition>], [<nNextRecords>], [<nRecord>],
*              [<lRest>] ) --> NIL
*
* Export records to a new ASCII file.
* Import records from a ASCII file.
*
* <lCopy>              if true (.T.) it exports, else it
*                      imports data.
* <cFileName>          the ASCII file name to create or the
*                      ASCII file name to use to import data.
* <cDelimiter>         the delimiter used.
* <acFields>           the fields to be involved.
* <bForCondition>      code block FOR condition.
* <bWhileCondition>    code block WHILE condition.
* <nNextRecords>       equivalent to NEXT ...
* <nRecord>            equivalent to RECORD ...
* <lRest>              if true (.T.), then REST records.
*

    // APPEND [FROM <xcFile>] | COPY [TO <xcFile>]
    //     [DELIMITED [WITH <delim>]]
    //     [FIELDS <Fields,...>]
    //     [FOR <forCondition>]
    //     [WHILE <whileCondition>]
    //     [NEXT <next>]
    //     [RECORD <rec>]
    //     [REST]
    //     [ALL]
    __dbDelim( lCopy, cFileName, cDelimiter, acFields,;
                bForCondition, bWhileCondition,;
                nNextRecords, nRecord, lRest )

    return NIL

*=================================================================
function dbJoin( cAlias, cFileName, acFields, bForCondition )
*
* dbJoin( <cAlias>, <cFileName>, [<acFields>],
*      [<bForCondition>] ) --> NIL
*
* Generates a new (.dbf) file by joining the active Alias
* with a second alias <cAlias> and eliminating records that
* do not meet the condition stated into the code block
* <bForCondition>.
*
* <cAlias>             the other Alias containing the data
*                      to use to join.
* <cFileName>          the (.dbf) file to create.
* <acFields>           fields to be involved.
* <bForCondition>      code block FOR.
*

    // JOIN [WITH <Alias>] [TO <xcFile>]
    //     [FIELDS <fields,...>]
    //     [FOR <forCondition>]
    __dbJoin( cAlias, cFileName, acFields, bForCondition )

    return NIL

*=================================================================
function dbLabelForm( cLabel, lToPrinter, cFileName,;
             lNoConsole, bForCondition, bWhileCondition,;
             nNextRecords, nRecord, lRest,;
             lSample )
*
* dbLabelForm( <cLabel>, [<lToPrinter>], [<cFileName>],
*     [<lNoConsole>], [<bForCondition>], [<bWhileCondition>],
*     [<nNextRecords>], [<nRecord>], [<lRest>],
*     [<lSample> ) --> NIL
*
* Displays or prints lables form a definition held in a
* label (.lbl) file for a range of records in the active
* Alias.
*
* <cLabel>             the label (.lbl) file name.
* <lToPrinter>         if true (.T.) the labels will be
*                      printed.
* <cFileName>          the text file name to create
*                      containing the lables.
* <lNoConsole>         if true (.T.) the labels will not
*                      displayed on the console.
* <bForCondition>      code block FOR condition.
* <bWhileCondition>    code block WHILE condition.
* <nNextRecords>       equivalent to NEXT ...
* <nRecord>            equivalent to RECORD ...
* <lRest>              if true (.T.), then REST records.
* <lSample>            if true (.T.), only test labels
*                      of asterisks are printed.
*

    default( @lToPrinter,   .F. )
    default( @lNoConsole,   .F. )

    // LABEL FORM <xcLabelFile>
    //     [SAMPLE]
    //     [NOCONSOLE]
    //     [TO PRINTER]
    //     [TO FILE <xcAlternateFile>]
    //     [FOR <forCondition>]
    //     [WHILE <whileCondition>]
    //     [NEXT <next>]
    //     [RECORD <rec>]
    //     [REST]
    //     [ALL]
    __LabelForm( cLabel, lToPrinter, cFileName,;
        lNoConsole, bForCondition, bWhileCondition,;
        nNextRecords, nRecord, lRest,;
        lSample )

    return NIL

*=================================================================
function dbList( lToDisplay, abListColumns, lAll,;
                bForCondition, bWhileCondition,;
                nNextRecords, nRecord, lRest,;
                lToPrinter, cFileName )
*
* dbList( [<lToDisplay>], <abListColumns>, [<lAll>],
*     [<bForCondition>], [<bWhileCondition>],
*     [<nNextRecords>], [<nRecord>], [<lRest>],
*     [<lToPrinter>], [<cFileName>] ) --> NIL
*
* Displays or prints the result of one or more expressions
* listed into <abListColumns> for a range of records in the
* active Alias.
*
* <lToDisplay>         if true (.T.) it displays the list.
* <abListColumns>      every element contains the expression
*                      (in code block form) to be listed.
* <lAll>               if true (.T.) it lists all records.
* <bForCondition>      code block FOR condition.
* <bWhileCondition>    code block WHILE condition.
* <nNextRecords>       equivalent to NEXT ...
* <nRecord>            equivalent to RECORD ...
* <lRest>              if true (.T.), then REST records.
* <lToPrinter>         if true (.T.) it prints the list.
* <cFileName>          the text file to create containing
*                      the list.
*

    // LIST | DISPLAY
    //     [<list,...>]
    //     [OFF]
    //     [TO PRINTER]
    //     [TO FILE <xcAlternateFile>]
    //     [FOR <forCondition>]
    //     [WHILE <whileCondition>]
    //     [NEXT <next>]
    //     [RECORD <rec>]
    //     [REST]
    //     [ALL]
    __dbList( lToDisplay, abListColumns, lAll,;
                bForCondition, bWhileCondition,;
                nNextRecords, nRecord, lRest,;
                lToPrinter, cFileName )

    return NIL

*=================================================================
function dbLocate( bForCondition, bWhileCondition,;
    nNextRecords, nRecord, lRest;
                 )
*
* dbLocate( [<bForCondition>], [<bWhileCondition>],
*     [<nNextRecords>], [<nRecord>], [<lRest>] ) --> NIL
*
* Positions the record pointer to the first record
* in the active Alias that matches the specified condition
* within the given scope.
*
* <bForCondition>      code block FOR condition.
* <bWhileCondition>    code block WHILE condition.
* <nNextRecords>       equivalent to NEXT ...
* <nRecord>            equivalent to RECORD ...
* <lRest>              if true (.T.), then REST records.
*

    default( @bForCondition,    { || .T. } )
    default( @bWhileCondition,  { || .T. } )

    // LOCATE
    //     [FOR <forCondition>]
    //     [WHILE <whileCondition>]
    //     [NEXT <next>]
    //     [RECORD <rec>]
    //     [REST]
    //     [ALL]
    __dbLocate( bForCondition, bWhileCondition,;
        nNextRecords, nRecord, lRest;
              )

    return NIL

*=================================================================
function dbOldCreate( cFileName, cFileStructure,;
                      cDriver, lNew, cAlias )
*
* dbOldCreate( <cFileName>, <cFileStructure>,
*      [<cDriver>], [<lNew>], [<cAlias>] ) --> NIL
*
* Creates a new database file with structure defined by
* the specified extended file <cFileStructure>.
*
* The name "dbOldCreate" is so to make no confusion with the
* CA-Clipper function dbCreate that works with an Array for
* the data structue.
*
* <cFileName>          contains the file name to create.
* <cFileStructure>     contains the file containing the
*                      structure.
* <cDriver>            contains the driver name to be used.
* <lNew>               if true (.T.) it opend the new file
*                      into the next free work area, else
*                      it will be opened into the actura
*                      area.
* <cAlias>             it contains the alias to be used
*                      to indentify the new file.
*

    default( @lNew,     .F. )

    // CREATE <xcFile1>
    //     [FROM <xcFile2>]
    //     [VIA <rdd>]
    //     [ALIAS <Alias>]
    //     [NEW]
    __dbCreate( cFileName, cFileStructure, cDriver, lNew, cAlias )

    return NIL

*=================================================================
function dbPack()
*
* dbPack() --> NIL
*
* Pack command substitute.
*

    // PACK
    __dbPack()

    return NIL

/* &&&&&&
*=================================================================
function dbReportForm( cForm, lToPrinter, cFileName,;
             lNoConsole, bForCondition, bWhileCondition,;
             nNextRecords, nRecord, lRest,;
             lPlain, cHeading,;
             lNoEject, lSummary )
*
* dbReportForm( <cForm>, [<lToPrinter>], [<cFileName>],
*     [<lNoConsole>], [<bForCondition>], [<bWhileCondition>],
*     [<nNextRecords>], [<nRecord>], [<lRest>],
*     [<lPlain>], [<cHeading>],
*     [<lNoEject>, [<lSummary>] ) --> NIL
*
* Displays or prints a tabular and optionally grouped
* report with page and column headings for a range of
* records in the active Alias.
* The report definition is held in a report (.frm) file.
*
* <cForm>              the report (.frm) file name.
* <lToPrinter>         if true (.T.) the report will be
*                      printed.
* <cFileName>          the text file name to create
*                      containing the report.
* <lNoConsole>         if true (.T.) the report will not
*                      displayed on the console.
* <bForCondition>      code block FOR condition.
* <bWhileCondition>    code block WHILE condition.
* <nNextRecords>       equivalent to NEXT ...
* <nRecord>            equivalent to RECORD ...
* <lRest>              if true (.T.), then REST records.
* <lPlain>             if true (.T.), the report will
*                      not have page breaks.
* <cHeading>           contains the header printed each
*                      page.
* <lNoEject>           if true (.T.) it suppresses the
*                      initial page eject.
* <lSummary>           if true (.T.), only group, subgroup,
*                      and grand total lines are printed.
*

    default( @lToPrinter,   .F. )
    default( @lNoConsole,   .F. )
    default( @lPlain,       .F. )
    default( @lNoEject,     .F. )
    default( @lSummary,     .F. )

    // REPORT FORM <xcFormFile>
    //     [HEADING <Heading>]
    //     [PLAIN]
    //     [NOEJECT]
    //     [SUMMARY]
    //     [NOCONSOLE]
    //     [TO PRINTER]
    //     [TO FILE <xcAlternateFile>]
    //     [FOR <forCondition>]
    //     [WHILE <whileCondition>]
    //     [NEXT <next>]
    //     [RECORD <rec>]
    //     [REST]
    //     [ALL]
    __ReportForm( cForm, lToPrinter, cFileName,;
        lNoConsole, bForCondition, bWhileCondition,;
        nNextRecords, nRecord, lRest,;
        lPlain, cHeading,;
        lNoEject, lSummary )

    return NIL

*/ &&&&&&&&

*=================================================================
function dbSDF( lCopy, cFileName, acFields,;
                bForCondition, bWhileCondition,;
                nNextRecords, nRecord, lRest )
*
* dbSDF( <lCopy>, <cFileName>,
*              [<acFields>], [<bForCondition>],
*              [<bWhileCondition>], [<nNextRecords>], [<nRecord>],
*              [<lRest>] ) --> NIL
*
* Export records to a new ASCII file.
* Import records from a ASCII file.
*
* <lCopy>              if true (.T.) it exports, else it
*                      imports data.
* <cFileName>          the ASCII file name to create or the
*                      ASCII file name to use to import data.
* <acFields>           the fields to be involved.
* <bForCondition>      code block FOR condition.
* <bWhileCondition>    code block WHILE condition.
* <nNextRecords>       equivalent to NEXT ...
* <nRecord>            equivalent to RECORD ...
* <lRest>              if true (.T.), then REST records.
*

    // APPEND [FROM <xcFile>] | COPY [TO <xcFile>]
    //     [SDF]
    //     [FIELDS <Fields,...>]
    //     [FOR <forCondition>]
    //     [WHILE <whileCondition>]
    //     [NEXT <next>]
    //     [RECORD <rec>]
    //     [REST]
    //     [ALL]
    __dbSDF( lCopy, cFileName, acFields,;
                bForCondition, bWhileCondition,;
                nNextRecords, nRecord, lRest )

    return NIL

*=================================================================
function dbSort( cFileName, acFields,;
                bForCondition, bWhileCondition,;
                nNextRecords, nRecord, lRest )
*
* dbSort( <cFileName>, [<acFields>], [<bForCondition>],
*              [<bWhileCondition>], [<nNextRecords>], [<nRecord>],
*              [<lRest>] ) --> NIL
*
* Copies records within the specified scope and condition
* from the current work area to another database file
* sorted according to the specified fields.
*
* <cFileName>          the (.dbf) file name to create.
* <acFields>           the fields to be involved.
* <bForCondition>      code block FOR condition.
* <bWhileCondition>    code block WHILE condition.
* <nNextRecords>       equivalent to NEXT ...
* <nRecord>            equivalent to RECORD ...
* <lRest>              if true (.T.), then REST records.
*

    // SORT [TO <xcFile>] [ON <Fields,...>]
    //     [FOR <forCondition>]
    //     [WHILE <whileCondition>]
    //     [NEXT <next>]
    //     [RECORD <rec>]
    //     [REST]
    //     [ALL]
    __dbSort( cFileName, acFields,;
                bForCondition, bWhileCondition,;
                nNextRecords, nRecord, lRest )

    return NIL

*=================================================================
function dbTotal( cFileName, bKey, acFields,;
                bForCondition, bWhileCondition,;
                nNextRecords, nRecord, lRest )
*
* dbTotal( <cFileName>, <bKey>, [<acFields>], [<bForCondition>],
*              [<bWhileCondition>], [<nNextRecords>], [<nRecord>],
*              [<lRest>] ) --> NIL
*
* Summerizes records in the current work area by key value,
* summing the specified numeric fields and then copying
* summary records to a second database file.
*
* <cFileName>          the (.dbf) file name to create.
* <bKey>               code block Key expression.
* <acFields>           the fields to be involved.
* <bForCondition>      code block FOR condition.
* <bWhileCondition>    code block WHILE condition.
* <nNextRecords>       equivalent to NEXT ...
* <nRecord>            equivalent to RECORD ...
* <lRest>              if true (.T.), then REST records.
*

    // TOTAL [TO <xcFile>] [ON <expKey>]
    //     [FIELDS <Fields,...>]
    //     [FOR <forCondition>]
    //     [WHILE <whileCondition>]
    //     [NEXT <next>]
    //     [RECORD <rec>]
    //     [REST]
    //     [ALL]
    __dbTotal( cFileName, bKey, acFields,;
                bForCondition, bWhileCondition,;
                nNextRecords, nRecord, lRest )

    return NIL

*=================================================================
function dbUpdate( cAlias, bKey, lRandom, bReplacement )
*
* dbUpdate( <cAlias>, <bKey>, [<lRandom>],
*              [<bReplacement>] ) --> NIL
*
* Updates the active Alias with the data contained into
* <cAlias> using the code block <bReplacement>.
*
* <cAlias>             the other Alias containing the data
*                      to use to update the active Alias.
* <bKey>               code block Key expression.
* <lRandom>            if true (.T.), then RANDOM.
* <bReplacement>       code block to be executed for the
*                      records with a matching key.
*
* Example:
*      dbUpdate( "INVOICE", {|| LAST}, .T.,;
*              {|| FIELD->TOTAL1 := INVOICE->SUM1,;
*                  FIELD->TOTAL2 := INVOICE->SUM2 } )
*

    // UPDATE [FROM <xcAlias>] [ON xckey]
    //     [REPLACE <Field1> WITH <exp1> [, <Fieldn> WITH <expn>]]
    //     [RANDOM]
    __dbUpdate( cAlias, bKey, lRandom, bReplacement )

    return NIL

*=================================================================
function dbZap()
*
* dbZap() --> NIL
*
* Zap command substitute.
*

    // ZAP
    __dbZap()

    return NIL

*=================================================================
* DBIALLSTATUS()
* DBISTATUS()
* DBISTRUCTURE()
*=================================================================

#define DBI_STRUCTURE_TOP;
   "Name     Type Len Dec"

*=================================================================
function dbiAllStatus()
*
* dbiAllStatus() --> cDbInformations
*
* This function returns the information on all the Aliases
* in all areas in text form.
*

    local cMessage := ""
    local nI       := 0

    *-------------------------------------------------------------
    * All Aliases informations.
    *-------------------------------------------------------------

    for nI := 1 to _MAX_SELECT
        if !( alias(nI) == "" )
            cMessage += (nI)->(dbiStatus())
            cMessage += NL(1)
        end
    next

    return cMessage

*=================================================================
function dbiStatus()
*
* dbiStatus() --> cDbInformations
*
* This function returns the information on the active Alias
* in a text form.
*

    local cFileInfo := ""
    local cOrdKey := ""
    local cLinkExpr := ""
    local nOrder    := 0
    local nRelation := 0
    local cFilter   := ""
    local nRelSelect := 0

    *-------------------------------------------------------------
    * If the an active Alias exists, informations may be returned.
    *-------------------------------------------------------------

    if len( alias() ) == 0

        *---------------------------------------------------------
        * No Alias is active now.
        *---------------------------------------------------------

        cFileInfo := _ERROR_NO_ALIAS

    else

        *---------------------------------------------------------
        * Begin with the area number and the alias name.
        *---------------------------------------------------------

        cFileInfo :=;
            "Select n. " + alltrim( str( select() ) ) +;
            NL(1) +;
            "Alias: " + alias() +;
            NL(1)

        *---------------------------------------------------------
        * If indexes are associated, loop to show all orders.
        *---------------------------------------------------------

        nOrder := 1
        while .T.                                       // FOREVER

            cOrdKey := ordkey( nOrder )
            if cOrdKey == ""

                exit                                    // EXIT

            else

                cFileInfo +=;
                    "Order n. " +;
                    alltrim( str( nOrder ) ) +;
                    ": " + ordname(nOrder) + " " + cOrdKey +;
                    NL(1)

            end

            nOrder++

        end

        *---------------------------------------------------------
        * If DBFILTER() returns a filter, it is added to the
        * description.
        *---------------------------------------------------------

        cFilter := dbfilter()
        if cFilter == ""

            *-----------------------------------------------------
            * The filter is empty: don't add the description.
            *-----------------------------------------------------

        else

            *-----------------------------------------------------
            * The filter may be added.
            *-----------------------------------------------------

            cFileInfo +=;
                "Filter : " + cFilter +;
                NL(1)

        end

        *---------------------------------------------------------
        * If relations are present, loop to show all relations.
        *---------------------------------------------------------

        nRelation := 0
        while .T.                                       // FOREVER

            cLinkExpr := dbrelation( ++nRelation )

            if cLinkExpr == ""

                exit                                    // EXIT

            else

                nRelSelect := dbrselect( nRelation )
                cFileInfo +=;
                    "Relation n. " +;
                    alltrim( str( nRelation ) ) +;
                    " " + alias(nRelSelect) +;
                    ": " + cLinkExpr +;
                    NL(1)

            end

        end

    end

    *-------------------------------------------------------------
    * The text containing informations on the active Alias is
    * returned.
    *-------------------------------------------------------------

    return cFileInfo

*=================================================================
function dbiStructure()
*
* dbiStructure() --> cTextStructure
*                --> NIL
*
* It returns a text containing the active Alias structure.
*

    local aStruct
    local cStructure := ""
    local nI

    *-------------------------------------------------------------
    * If the an active Alias exists, informations may be returned.
    *-------------------------------------------------------------

    if alias() == "";

        *---------------------------------------------------------
        * No Alias is active now.
        *---------------------------------------------------------

        alertBox( _ERROR_NO_ALIAS )

        return NIL                                      // RETURN

    end

    *-------------------------------------------------------------
    * Save the structure.
    *-------------------------------------------------------------

    aStruct := dbStruct()

    *-------------------------------------------------------------
    * Write the Header.
    *-------------------------------------------------------------

    cStructure := "Alias: " + alias() +;
         NL(2) +;
         DBI_STRUCTURE_TOP +;
         NL(2)

    *-------------------------------------------------------------
    * Write the structure.
    *-------------------------------------------------------------

    for nI := 1 to len( aStruct )

        cStructure +=;
            padr( aStruct[nI][1], 10 ) + " " +;
            aStruct[nI][2] + " " +;
            str( aStruct[nI][3], 5 ) + " " +;
            str( aStruct[nI][4], 4 ) + NL(1)

    next

    *-------------------------------------------------------------
    * Return the information formatted string.
    *-------------------------------------------------------------

    return cStructure

*=================================================================
* DEFAULT()
*=================================================================
function default( xVar, xDefaultValue )
*
* substitution to "default ... to ..."
*

    if xVar == NIL
        xVar := xDefaultValue
    end

    return xVar

*=================================================================
* DISPBOXCOLOR()
* DISPBOXSHADOW()
*=================================================================
function dispBoxColor( nColorNumber, cBaseColor )
*
* dispBoxColor( [<nColorNumber>], [<cBaseColor>] ) --> cColor
*
* <nColorNumber>        may be 1 or 2 and means:
*                       1 = normal color;
*                       2 = shadow color.
* <cBaseColor>          is the source color. The default value
*                       is setcolor()
*
* This function returns the right color string for the use with
* dispBoxShadow().
*
    local cForeground
    local cBackground
    local nSlashPosition
    local nCommaPosition
    local cBoxColor

    default( @cBaseColor,   setcolor() )
    default( @nColorNumber, 1 )

    *-------------------------------------------------------------
    * The first color contained inside <cColorBase> is analised.
    *-------------------------------------------------------------

    nSlashPosition := at( "/", cBaseColor )
    if nSlashPosition > 0

        *---------------------------------------------------------
        * The color is Ok.
        *---------------------------------------------------------

    else

        *---------------------------------------------------------
        * The color is unknown.
        *---------------------------------------------------------

        return cBaseColor                               // RETURN

    end

    *-------------------------------------------------------------
    * The foreground color from the first color couple is
    * extracted.
    *-------------------------------------------------------------

    cForeGround := upper( substr( cBaseColor, 1, nSlashPosition-1 ) )

    *-------------------------------------------------------------
    * The comma separate the first color from the others inside
    * the <cBaseColor> string.
    *-------------------------------------------------------------

    nCommaPosition := at( ",", cBaseColor )
    if nCommaPosition > 0

        *---------------------------------------------------------
        * The comma is found.
        *---------------------------------------------------------

    else

        *---------------------------------------------------------
        * The comma is not found: the <cBaseColor> string contains
        * only one color couple.
        *---------------------------------------------------------

        nCommaPosition := len( cBaseColor )+1

    end

    *-------------------------------------------------------------
    * The backround color from the first color couple is
    * extracted.
    *-------------------------------------------------------------

    cBackground :=;
        upper(;
            substr( cBaseColor, nSlashPosition+1, nCommaPosition-nSlashPosition-1 );
            )

    *-------------------------------------------------------------
    * If the computer is monocrome, the shadow cannot be
    * shown.
    *-------------------------------------------------------------

    if iscolor()

        if nColorNumber == 1

            *-----------------------------------------------------
            * If the background is Black, the line will be
            * White, otherwise the line is Black
            *-----------------------------------------------------

            if left(cBackground, 1) == "N"
                cBoxColor := "W/"+cBackground
            else
                cBoxColor := "N/"+cBackground
            end

        else

            *-----------------------------------------------------
            * If the background is Black, the line will be
            * White+, otherwise the line is the same as the
            * background, but enhanced (+)
            *-----------------------------------------------------

            if left(cBackground, 1) == "N"
                cBoxColor := "W+/"+cBackground
            else
                cBoxColor := cBackground + "+/"+cBackground
            end

        end

    else

        cBoxColor := cBaseColor

    end

    *-------------------------------------------------------------
    * The new color is returned.
    *-------------------------------------------------------------

    return cBoxColor

*=================================================================
function dispBoxShadow(;
    nTop, nLeft, nBottom, nRight,;
    cBoxString,;
    cColor1, cColor2;
    )
*
* like dispBox() with two colors for borders.
*

    local nI

    default( @cBoxString,   BOX_SINGLE )
    default( @cColor1,      setcolor() )
    default( @cColor2,      cColor1 )

    *-------------------------------------------------------------
    * If <cBoxString> contains a number, it is converted into the
    * right string.
    *-------------------------------------------------------------

    if valtype( cBoxString ) == "N"
        if cBoxString == 1
            cBoxString := BOX_SINGLE
        else
            cBoxString := BOX_DOUBLE
        end
    end

    *-------------------------------------------------------------
    * If <cBoxString> contains only one character, it is
    * replicated.
    *-------------------------------------------------------------

    if len( cBoxString ) == 1
        cBoxString := replicate( cBoxString, 8 )
    end

    *-------------------------------------------------------------
    * If it is only a line and not a box, the <cBoxString> is
    * converted into a single character.
    *-------------------------------------------------------------

    if nTop == nBottom

        *---------------------------------------------------------
        * It is only a horizontal line.
        *---------------------------------------------------------

        cBoxString := stuff( cBoxString, 1, 1, substr( cBoxString, 2, 1 ) )
        cBoxString := stuff( cBoxString, 3, 1, substr( cBoxString, 2, 1 ) )
        cBoxString := stuff( cBoxString, 5, 1, substr( cBoxString, 2, 1 ) )
        cBoxString := stuff( cBoxString, 7, 1, substr( cBoxString, 2, 1 ) )

    end

    if nLeft == nRight

        *---------------------------------------------------------
        * It is a vertical line.
        *---------------------------------------------------------

        cBoxString := stuff( cBoxString, 1, 1, substr( cBoxString, 4, 1 ) )
        cBoxString := stuff( cBoxString, 3, 1, substr( cBoxString, 4, 1 ) )
        cBoxString := stuff( cBoxString, 5, 1, substr( cBoxString, 4, 1 ) )
        cBoxString := stuff( cBoxString, 7, 1, substr( cBoxString, 4, 1 ) )

    end

    *-------------------------------------------------------------
    * Display the horizontal top line.
    *-------------------------------------------------------------

    say( nTop, nLeft, substr( cBoxString, 1, 1 ), NIL, cColor1 )
    dispbox( nTop, nLeft+1, nTop, nRight-1, substr( cBoxString, 2, 1 ), cColor1 )

    *-------------------------------------------------------------
    * Display the vertical Right line.
    *-------------------------------------------------------------

    say( nTop, nRight, substr( cBoxString, 3, 1 ), NIL, cColor1 )
    dispbox( nTop+1, nRight, nBottom-1, nRight, substr( cBoxString, 4, 1 ), cColor2 )

    *-------------------------------------------------------------
    * Display the horizontal bottom line.
    *-------------------------------------------------------------

    say( nBottom, nRight, substr( cBoxString, 5, 1 ), NIL, cColor2 )
    dispbox( nBottom, nRight-1, nBottom, nLeft+1, substr( cBoxString, 6, 1 ), cColor2 )

    *-------------------------------------------------------------
    * Display the vertical Left line.
    *-------------------------------------------------------------

    say( nBottom, nLeft, substr( cBoxString, 7, 1 ), NIL, cColor2 )
    dispbox( nTop+1, nLeft, nBottom-1, nLeft, substr( cBoxString, 8, 1 ), cColor1 )

    *-------------------------------------------------------------
    * Set the cursor position at the same position that the
    * original DISPBOX().
    *-------------------------------------------------------------

    setpos( nTop+1, nLeft+1 )

    return NIL

*=================================================================
* DIR()
*=================================================================

#define DIR_WAIT_READING_DIRECTORY;
    "Reading Directory..."

#define DIR_SAY_DRIVE;
    "<Drive:>"

#define DIR_GET_NEW_DRIVE;
    "New Drive:"

#define DIR_HELP;
    "Dir()" +;
    NL(3) +;
    "This function permits to select:" +;
    NL(2) +;
    "- a disk drive," +;
    NL(1) +;
    "- a directory, " +;
    NL(1) +;
    "- a file," +;
    NL(2) +;
    "depending of the context." +;
    NL(2) +;
    "Use arrow keys, []/[], to move cursor. " +;
    "Press [Enter] to select a drive, a new directory, " +;
    "the current directory or " +;
    "a file." +;
    NL(2) +;
    "If a directory or a new drive is selected, it will " +;
    "appear the new list." +;
    NL(2) +;
    "If a file or the current displayed directory [.] " +;
    "is selected, the selection window disappears " +;
    "automatically and the program continues."

*=================================================================
function dir(;
        cFileSpec,;
        lDrives, lDirs, lFiles,;
        lNoDirReturn, nSortColumn;
    )
*
* dir( [<cFileSpec>], [<lDrives>], [<lDirs>], [<lFiles>],
*      [<lNoDirReturn>], [<nSortColumn>] ) --> cPathname
*
* <cFileSpec>      Filename, Pathname, also with wildcards.
*
* <lDrives>        true (.t.) means: include drives letters.
*
* <lDirs>          true (.t.) means: include directory names.
*
* <lFiles>         true (.t.) means: include file names.
*
* <lNoRirReturn>   true (.t.) means: do not return the
*                  shown directory if [Esc] is used to exit.
*
* <nSortColumn>        the column number to use to sort the
*                      list. The columns are:
*                      Name        = 1,
*                      Size        = 2,
*                      Date        = 3,
*                      Time        = 4,
*                      Attribute   = 5.
*                      It is not possible to sort for extention.
*

    local cOldScreen
    local cOldColor     := setcolor()
    local nOldCursor    := setcursor( SETCURSOR_NONE )
    local nOldRow       := row()
    local nOldCol       := col()
    local bOld_F1       :=;
        setkey( K_F1, {|| Text( DIR_HELP ) } )
    local bOld_F2       := setkey( K_F2, NIL )
    local nTop
    local nLeft
    local nBottom
    local nRight
    local nWidth

    local aDir := {}
    local nI := 0
    local cDrive
    local cPath
    local cWild
    local cFile

    local cOldFileSpec

    local cNewDir
    local cNewDrive

    local cReturn

    default( @cFileSpec,    "*.*" )
    default( @lDrives,      .T. )
    default( @lDirs,        .T. )
    default( @lFiles,       .T. )
    default( @lNoDirReturn, .F. )
    default( @nSortColumn,  1 ) // File Name

    cOldFileSpec := cFileSpec

    *-------------------------------------------------------------
    * Start a new sequence.
    *-------------------------------------------------------------

    begin sequence

        *---------------------------------------------------------
        * Correct possible error on nSortColumn.
        *---------------------------------------------------------

        if nSortColumn > 5    // Attribute
            nSortColumn := 1  // File Name
        end

        *---------------------------------------------------------
        * Distinguish name.
        *---------------------------------------------------------

        cDrive := strDrive( cFileSpec )
        cPath  := strPath( cFileSpec )
        cFile  := strFile( cFileSpec )

        *---------------------------------------------------------
        * Rebuild the pathname with some defaults.
        *---------------------------------------------------------

        do case
        case;
            cDrive == "" .and.;
            cPath == "" .and.;
            curdir() == ""

            *-----------------------------------------------------
            * It must be the root.
            *-----------------------------------------------------

            cPath := "\"

        case;
            cDrive == "" .and.;
            cPath == "" .and.;
            !empty( curdir() )

            *-----------------------------------------------------
            * It must be the current directory of the current
            * drive.
            *-----------------------------------------------------

            cPath := "\" + curdir() + "\"

        case;
            cDrive == "" .and.;
            !empty( cPath )

            *-----------------------------------------------------
            * Ok, no need to determinate defaults.
            *-----------------------------------------------------

        case;
            !empty( cDrive ) .and.;
            cPath == "" .and.;
            curdir( cDrive ) == ""

            *-----------------------------------------------------
            * It must be the root of <cDrive>.
            *-----------------------------------------------------

            cPath := "\"

        case;
            !empty( cDrive ) .and.;
            cPath == "" .and.;
            !empty( curdir( cDrive ) )

            *-----------------------------------------------------
            * It must be the current directory of <cDrive>.
            *-----------------------------------------------------

            cPath := "\" + curdir( cDrive ) + "\"

        case;
            !empty( cDrive ) .and.;
            !empty( cPath )

            *-----------------------------------------------------
            * Ok, no need to determinate defaults.
            *-----------------------------------------------------

        end

        *---------------------------------------------------------
        * The file name (with wildcards) is saved.
        *---------------------------------------------------------

        cWild  := strFile( cFileSpec )

        *---------------------------------------------------------
        * The directory is read.
        *---------------------------------------------------------

        aDir :=;
            dirArray(;
                cDrive+cPath+cWild,;
                lDrives, lDirs, lFiles, lNoDirReturn,;
                nSortColumn;
            )

        *---------------------------------------------------------
        * Create a kind of window.
        *---------------------------------------------------------

        nTop       := 0
        nLeft      := maxcol()-46
        nBottom    := maxrow()
        nRight     := maxcol()
        nWidth     := nRight - nLeft +1

        cOldScreen := mouseScrSave( nTop, nLeft, nBottom, nRight )

        setcolor( COLOR_BODY )

        scroll( nTop, nLeft, nBottom, nRight )

        dispBoxShadow(;
            nTop, nLeft, nBottom, nRight,;
            1,;
            dispBoxColor( 1 ),;
            dispBoxColor( 2 );
        )

        *---------------------------------------------------------
        * The following loop will activate a aChoice() to
        * select a disk, a directory or a file.
        * If a new disk or a new directory is selected, the new
        * directory is shown.
        *---------------------------------------------------------

        while .t.                                       // FOREVER

            say(;
                nTop+1, nLeft+1,;
                padc( cDrive+cPath+cWild, nWidth-2 ),,;
                COLOR_HEAD;
            )

            *-----------------------------------------------------
            * Note that the function aChoiceWindow() cannot be
            * used as here special action are made: see what
            * happens when a new drive is selected.
            *-----------------------------------------------------

            nI :=;
                achoice(;
                    nTop+2, nLeft+1, nBottom-1, nRight-1,;
                    aDir,,,.t.;
                )

            *-----------------------------------------------------
            * Analise the selection and do the appopriate action.
            *-----------------------------------------------------

            do case
            case nI == 0        // [Esc] pressed

                *-------------------------------------------------
                * Abandon: the original is returned.
                *-------------------------------------------------

                cReturn := cOldFileSpec

                exit                                    // EXIT

            case left( aDir[nI], 3 ) == "[.]"   // Directory

                *-------------------------------------------------
                * Return current directory.
                *-------------------------------------------------

                cReturn := cDrive+cPath

                exit                                    // EXIT

            case left( aDir[nI], 1 ) == "["     // Directory

                *-------------------------------------------------
                * Note that the at() function cannot work on "["
                * parentesis !!!
                *-------------------------------------------------

                cNewDir := left( aDir[nI], 15 )

                *-------------------------------------------------
                * Now <cNewDir> contains "[EXAMPLE.XX]".
                *-------------------------------------------------

                cNewDir := alltrim( cNewDir )

                *-------------------------------------------------
                * Now <cNewDir> contains "[EXAMPLE.XX]".
                *-------------------------------------------------

                cNewDir := substr( cNewDir, 2 )

                *-------------------------------------------------
                * Now <cNewDir> contains "EXAMPLE.XX]".
                *-------------------------------------------------

                cNewDir := substr( cNewDir, 1, len( cNewDir ) -1 )

                *-------------------------------------------------
                * Now <cNewDir> contains "EXAMPLE.XX".
                *-------------------------------------------------

                *-------------------------------------------------
                * Analise the selected directory.
                *-------------------------------------------------

                do case
                case cNewDir == "."

                    *---------------------------------------------
                    * It shouldn't happend as it is already
                    * checked before.
                    *---------------------------------------------

                case cNewDir == ".."

                    *---------------------------------------------
                    * Go back one level.
                    *---------------------------------------------

                    cPath := strParent( cPath )

                    *---------------------------------------------
                    * Read the new directory.
                    *---------------------------------------------

                    aDir :=;
                        dirArray(;
                            cDrive+cPath+cWild,;
                            lDrives, lDirs, lFiles, lNoDirReturn,;
                            nSortColumn;
                            )

                otherwise

                    *---------------------------------------------
                    * Change directory.
                    *---------------------------------------------

                    cPath := cPath+cNewDir+"\"

                    *---------------------------------------------
                    * Read the new directory.
                    *---------------------------------------------

                    aDir :=;
                        dirArray(;
                            cDrive+cPath+cWild,;
                            lDrives, lDirs, lFiles, lNoDirReturn,;
                            nSortColumn;
                            )

                end

            case rtrim(aDir[nI]) == DIR_SAY_DRIVE

                *-------------------------------------------------
                * The "new drive" was selected: select the new
                * drive.
                *-------------------------------------------------

                cNewDrive := dirDriveGet()

                *-------------------------------------------------
                * If the selection is valid, read the new drive
                * current directory.
                *-------------------------------------------------

                if  !empty( cNewDrive ) .and.;
                    left( cDrive, 1 ) <> cNewDrive

                    *---------------------------------------------
                    * If a new drive is selected, determinate the
                    * new directory location and read it.
                    *---------------------------------------------

                    cDrive := cNewDrive+":"

                    if curdir(cDrive) == ""
                        cPath := "\"
                    else
                        cPath := "\" + curdir(cDrive) + "\"
                    end

                    aDir :=;
                        dirArray(;
                            cDrive+cPath+cWild,;
                            lDrives, lDirs, lFiles, lNoDirReturn,;
                            nSortColumn;
                        )

                end

            otherwise

                *-------------------------------------------------
                * A file was selected.
                *-------------------------------------------------

                cFile := left( aDir[nI], 16 )
                cFile := alltrim( cFile )
                cReturn := cDrive+cPath+cFile

                exit                                    // EXIT

            end

        end

        *---------------------------------------------------------
        * Restore screen deleting the window.
        *---------------------------------------------------------

        mouseScrRestore( nTop, nLeft, nBottom, nRight, cOldScreen )

    end //sequence

    *-------------------------------------------------------------
    * Restore old values.
    *-------------------------------------------------------------

    setkey( K_F1, bOld_F1 )
    setkey( K_F2, bOld_F2 )
    setcursor( nOldCursor )
    setcolor( cOldColor )
    setpos( nOldRow, nOldCol )

    *-------------------------------------------------------------
    * Return the selected pathname.
    *-------------------------------------------------------------

    return cReturn

*-----------------------------------------------------------------
static function dirArray(;
    cFileSpec,;
    lDrives, lDirs, lFiles,;
    lNoDirReturn, nSortColumn;
    )

    local aDirectory
    local aDir := {}
    local nI := 0

    local cDrive := strDrive( cFileSpec )
    local cPath  := strPath( cFileSpec )

    *-------------------------------------------------------------
    * Correct the path: it must terminate with "\".
    *-------------------------------------------------------------

    if  len( cPath ) > 0 .and.;
        right( cPath, 1 ) <> "\"

        cPath := cPath + "\"

    end

    waitFor( DIR_WAIT_READING_DIRECTORY )

    *-------------------------------------------------------------
    * Disk drives.
    *-------------------------------------------------------------

    if lDrives
        aadd( aDir, DIR_SAY_DRIVE + space( 60 ) )
    end

    *-------------------------------------------------------------
    * Directories.
    *-------------------------------------------------------------

    if lDirs

        *--------------------------------------------------------
        * Read the entire directory.
        *--------------------------------------------------------

        aDirectory :=;
            asort(;
                directory( cDrive+cPath+"*.*", "D" ), NIL, NIL,;
                { |x, y| x[nSortColumn] < y[nSortColumn] };
            )

        *--------------------------------------------------------
        * Filter only directoryes.
        *--------------------------------------------------------

        for nI := 1 to len( aDirectory )

            *-----------------------------------------------------
            * If it isn't a directory, loop again.
            *-----------------------------------------------------

            if aDirectory[nI][5] <> "D"    // Attribute

                loop                                    // LOOP

            end

            *-----------------------------------------------------
            * If it is a directory, we are here.
            *-----------------------------------------------------

            if aDirectory[nI][1] == "."    // File Name

                *-------------------------------------------------
                * The current directory.
                *-------------------------------------------------

                if !lNoDirReturn

                    *---------------------------------------------
                    * Current directory will mean return
                    * directory.
                    *---------------------------------------------

                    aadd( adir, "[.] " + cFileSpec + space(60) )

                end

            else

                *-------------------------------------------------
                * Add the directory to the list.
                *-------------------------------------------------

                aadd(;
                    adir,;
                    left( "[" + aDirectory[nI][1] + "]" +;
                        space(20), 15;
                    ) +;
                    str(aDirectory[nI][2], 10, 0) + " " +;
                    dtoc(aDirectory[nI][3]) + " " +;
                    aDirectory[nI][4] + " " +;
                    aDirectory[nI][5];
               )
            end

        next

    end

    *-------------------------------------------------------------
    * Files.
    *-------------------------------------------------------------

    if lFiles

        *--------------------------------------------------------
        * Read the entire directory.
        *--------------------------------------------------------

        aDirectory := asort( directory( cFileSpec, "" ), NIL, NIL,;
                    { |x, y| x[nSortColumn] < y[nSortColumn] } )

        *--------------------------------------------------------
        * Filter only files.
        *--------------------------------------------------------

        for nI := 1 to len( aDirectory )

            aadd(;
                aDir,;
                left( aDirectory[nI][1] + space(20), 15 )  +;
                str(aDirectory[nI][2],10,0) + " " +;
                dtoc(aDirectory[nI][3]) + " " +;
                aDirectory[nI][4] + " " +;
                aDirectory[nI][5];
            )

        next
    end

    waitFor()

    *-------------------------------------------------------------
    * Return a copy of <aDir>.
    *-------------------------------------------------------------

    return aClone( aDir )

*-----------------------------------------------------------------
static function dirDriveGet()

    local aoGet     := {}

    local cNewDrive := space(1)
    local nRow      := row()
    local nCol      := maxcol() -45

    local nOldCursor := setcursor( SETCURSOR_NORMAL )

    say( nRow, nCol, space(45) )
    say( nRow, nCol, DIR_GET_NEW_DRIVE )
    get( @aoGet, nRow, col()+1,;
        { |x| iif( pcount() > 0, cNewDrive := x, cNewDrive ) },;
        "!";
    )

    read( @aoGet )
    aoGet := {}

    do case
    case lastkey() = K_ESC  // exit
        cNewDrive := NIL
    case cNewDrive >= chr(65) .and. cNewDrive <= chr(90)
        // Ok.
    otherwise
        // Nil.
        cNewDrive := NIL
    end

    // Restore.
    setcursor( nOldCursor )

    return cNewDrive

*=================================================================
* DOTLINE()
*=================================================================

#define DOT_LINE_HELP;
    "dotLine()" +;
    NL(3) +;
    "This function is a dot line useable as a pop up " +;
    "calculator." +;
    NL(2) +;
    "A calculation may be done simply writing on the dot line." +;
    NL(3) +;
    "Examples:" +;
    NL(2) +;
    "123 + 456" +;
    NL(1) +;
    "(256 + 567) * 45" +;
    NL(3) +;
    "+, -, *, /, **, (, ) and matematical functions " +;
    "may be used." +;
    NL(3) +;
    "[Esc]       exit dot line." +;
    NL(1) +;
    "[Enter]     resolve dot line." +;
    NL(1) +;
    "[Pag]      insert the dot line content into the the " +;
    "keyboard buffer."

#define DOT_LINE_TITLE;
   "Dot Line Calculator"

#define DOT_LINE_BUTTON_ESC_CANCEL      "[Esc] exit"
#define DOT_LINE_BUTTON_F1_HELP         "[F1] help"
#define DOT_LINE_BUTTON_ENTER_RESOLVE   "[Enter] resolve"
#define DOT_LINE_BUTTON_PGDN_KEYBOARD   "[Pag] keyboard"

*=================================================================
function dotLine()
*
* dotLine() --> NIL
*
* Mini dot command line to use as a pop up calcultor.
* Keyboard() is used to send a result to a get field.
*

    local aoGet             := {}
    local aButton           := {}
    local bOld_F1           :=;
        setkey( K_F1, { || Text( DOT_LINE_HELP )} )

    local cOldColor         := setcolor()
    local nOldCursor        := setcursor( SETCURSOR_NORMAL )
    local bOldErrorHandler
    local cOldScreen
    local nOldRow           := row()
    local nOldCol           := col()
    local nOldSetDec        := set( _SET_DECIMALS, 15 )

    local nTop
    local nLeft
    local nBottom
    local nRight
    local nWidth
    local nHeight

    local lMore     := .T.
    local cCommand  := space(_MAX_STRING_LEN)
    local xResult

    *-------------------------------------------------------------
    * Start a new sequence.
    *-------------------------------------------------------------

    begin sequence

        *---------------------------------------------------------
        * Create a kind of window placed near the current cursor
        * position.
        *---------------------------------------------------------

        nTop    := row()+1
        nLeft   := col()
        nBottom := nTop+6
        if nBottom > maxrow()
            nBottom := maxrow()
            nTop    := nBottom - 6
        end
        nRight  := nLeft+59
        if nRight > maxcol()
            nRight := maxcol()
            nLeft  := nRight -59
        end
        nWidth  := nRight - nLeft +1
        nHeight := nBottom - nTop +1

        cOldScreen    :=;
            mouseScrSave( nTop, nLeft, nBottom, nRight )
        setcolor( COLOR_BODY )
        scroll( nTop, nLeft, nBottom, nRight )
        dispBoxShadow(;
            nTop, nLeft, nBottom, nRight,;
            1,;
            dispBoxColor( 1 ),;
            dispBoxColor( 2 );
        )
        say( nTop+1, nLeft+1, padc( DOT_LINE_TITLE, nWidth-2 ), , COLOR_HEAD )

        *---------------------------------------------------------
        * Start the read loop.
        *---------------------------------------------------------

        while .T.                               // FOREVER

            get( @aoGet, nTop+3, nLeft+1,;
                { |x| iif( pcount() > 0, cCommand := x, cCommand ) },;
                "@s"+ltrim(str(nWidth-2))+"@",;
                COLOR_BODY;
            )
            button( @aButton, row()+2, nLeft+1,;
                DOT_LINE_BUTTON_ESC_CANCEL, , {||keyboard(chr(K_ESC))} )
            button( @aButton, row(), col()+1,;
                DOT_LINE_BUTTON_F1_HELP, , {|| eval(setkey(K_F1))} )
            button( @aButton, row(), col()+1,;
                DOT_LINE_BUTTON_ENTER_RESOLVE, , {||keyboard(chr(K_ENTER))} )
            button( @aButton, row(), col()+1,;
                DOT_LINE_BUTTON_PGDN_KEYBOARD, , {||keyboard(chr(K_PGDN))} )

            *-----------------------------------------------------
            * Read.
            *-----------------------------------------------------

            read( aoGet, , aButton )
            aoGet := {}
            aButton := {}

            *-----------------------------------------------------
            * Analise the Keybaord buffer.
            *-----------------------------------------------------

            do case
            case lastkey() = K_PGDN

                *-------------------------------------------------
                * Insert the data inside the keyboard buffer.
                *-------------------------------------------------

                keyboard( alltrim(cCommand) )

                exit                                    // EXIT

            case lastkey() = K_ESC

                exit                                    // EXIT

            case lastkey() = K_ENTER

                *-------------------------------------------------
                * Try to resolve.
                *-------------------------------------------------

                if !empty( cCommand )

                    *---------------------------------------------
                    * Before solving the string it is better to
                    * handle the possible error.
                    *---------------------------------------------

                    bOldErrorHandler :=;
                        errorblock( {|e| break(e)} )

                    begin sequence

                        *-----------------------------------------
                        * Resolve.
                        *-----------------------------------------

                        xResult := &(alltrim(cCommand))

                    recover

                        *-----------------------------------------
                        * Handle the error.
                        *-----------------------------------------

                        xResult := padr( "Error", _MAX_STRING_LEN )

                    end

                    errorblock(bOldErrorHandler)

                    *---------------------------------------------
                    * Insert the result (or the error) inside the
                    * <cCommand> line.
                    *---------------------------------------------

                    cCommand :=;
                        padr( alltrim(transform( xResult, "@!" )),;
                            _MAX_STRING_LEN )

                end

            otherwise

                *-------------------------------------------------
                * No other key is considered: loop again.
                *-------------------------------------------------

            end

        end

        *---------------------------------------------------------
        * Restore the previous screen.
        *---------------------------------------------------------

        mouseScrRestore( nTop, nLeft, nBottom, nRight, cOldScreen )

    end //sequence

    *-------------------------------------------------------------
    * Restore previous values.
    *-------------------------------------------------------------

    setpos( nOldRow, nOldCol )
    setcursor(nOldCursor)
    setcolor( cOldColor )
    set( _SET_DECIMALS, nOldSetDec )

    *-------------------------------------------------------------
    * No value is returned.
    *-------------------------------------------------------------

    return NIL

*=================================================================
* DTEMONTH()
* DTEWEEK()
*=================================================================
function dteMonth( nMonth, cLanguage )
*
* dteMonth( <nMonth>, <cLanguage> ) --> cMonth
*
* <nMonth>     the month number
* <cLanguage>  the language name
*
* This function translates the <nMonth> number into the month
* name.
*
    local cMonth := ""

    *-------------------------------------------------------------
    * Start a new sequence.
    *-------------------------------------------------------------

    begin sequence

        *---------------------------------------------------------
        * Check input.
        *---------------------------------------------------------

        if valtype( nMonth ) <> "N"

            break                                       // BREAK

        end

        if valtype( cLanguage ) <> "C"

            break                                       // BREAK

        end

        *---------------------------------------------------------
        * Translate.
        *---------------------------------------------------------

        cLanguage := upper( alltrim( cLanguage ) )

        do case
        case cLanguage == "ITALIANO" .or.;
             cLanguage == "ITALIANA" .or.;
             cLanguage == "ITALIA"
            do case
            case nMonth = 1
                cMonth := "gennaio"
            case nMonth = 2
                cMonth := "febbraio"
            case nMonth = 3
                cMonth := "marzo"
            case nMonth = 4
                cMonth := "aprile"
            case nMonth = 5
                cMonth := "maggio"
            case nMonth = 6
                cMonth := "giugno"
            case nMonth = 7
                cMonth := "luglio"
            case nMonth = 8
                cMonth := "agosto"
            case nMonth = 9
                cMonth := "settembre"
            case nMonth = 10
                cMonth := "ottobre"
            case nMonth = 11
                cMonth := "novembre"
            case nMonth = 12
                cMonth := "dicembre"
            end
        end

    end //sequence

    *-------------------------------------------------------------
    * Translated month is returned.
    *-------------------------------------------------------------

    return cMonth

*=================================================================
function dteWeek( nWeek, cLanguage )
*
* dteWeek( <nWeek>, <cLanguage> ) --> cWeek
*
* <nWeek>      the week number
* <cLanguage>  the language name
*
* This function translates the <nWeek> number into the week
* name.
* nWeek == 1 -> Sunday
* nWeek == 2 -> Monday
* ...
* nWeek == 7 -> Saturday
*
    local cWeek := ""

    *-------------------------------------------------------------
    * Start a new sequence.
    *-------------------------------------------------------------

    begin sequence

        *---------------------------------------------------------
        * Check input.
        *---------------------------------------------------------

        if valtype( nWeek ) <> "N"

            break                                       // BREAK

        end

        if valtype( cLanguage ) <> "C"

            break                                       // BREAK

        end

        *---------------------------------------------------------
        * Translate.
        *---------------------------------------------------------

        cLanguage := upper( alltrim( cLanguage ) )

        do case
        case cLanguage == "ITALIANO" .or.;
             cLanguage == "ITALIANA" .or.;
             cLanguage == "ITALIA"
            do case
            case nWeek = 1
                cWeek := "domenica"
            case nWeek = 2
                cWeek := "lunedi'"
            case nWeek = 3
                cWeek := "martedi'"
            case nWeek = 4
                cWeek := "mercoledi'"
            case nWeek = 5
                cWeek := "giovedi'"
            case nWeek = 6
                cWeek := "venerdi'"
            case nWeek = 7
                cWeek := "sabato"
            end
        end

    end //sequence

    *-------------------------------------------------------------
    * Translated week is returned.
    *-------------------------------------------------------------

    return cWeek

*=================================================================
* ERRORHANDLER()
* ERRORMACRO()
*=================================================================

#define ERROR_MENU_CHOICE_IGNORE        "Ignore"
#define ERROR_MENU_CHOICE_BREAK         "Break"
#define ERROR_MENU_CHOICE_RETRY         "Retry"
#define ERROR_MENU_CHOICE_DEFAULT       "Default"
#define ERROR_MENU_CHOICE_QUIT          "Quit"

*=================================================================
function errorHandler(e)
*
* errorHandler( <oError> ) --> lRetry
*
* <oError>    Error object.
*
* Error handling function. If it don't breaks or quit,
* it returns true if a "retry" choice is made,
* or false if a "default" choice is made by the user.
*
* Example:
*
*  * Save the previous error handler
*  bOldErrorHandler := errorblock( {|e| ErrorHandler(e)} )
*  *
*  begin sequence
*      * Action to be protected from the new error handler
*      ...
*      ...
*  recover
*      * Alternative action to be taken when an error
*      * followed by a break is occurred, for example:
*      dbcloseall()
*      ...
*  end //sequence
*  * Restore previous error handler
*  errorblock( bOldErrorHandler )
*

    local cMessage
    local aOptions
    local nChoice

    *-------------------------------------------------------------
    * Automatically resolved errors.
    *-------------------------------------------------------------

    *-------------------------------------------------------------
    * Zero division --> 0.
    *-------------------------------------------------------------

    if ( e:genCode == ERROR_GENERIC_ZERODIV )

        return (0)                                      // RETURN

    end

    *-------------------------------------------------------------
    * Network open error.
    *-------------------------------------------------------------

    if  e:genCode == ERROR_GENERIC_OPEN .and.;
        e:osCode == 32                  .and.;
        e:canDefault

        neterr(.T.)

        return (.F.)    // DEFAULT                      // RETURN

    end

    *-------------------------------------------------------------
    * Lock error during APPEND BLANK.
    *-------------------------------------------------------------

    if  e:genCode == ERROR_GENERIC_APPENDLOCK .and.;
        e:canDefault

        neterr(.T.)

        return (.F.)    // DEFAULT                      // RETURN

    end

    *-------------------------------------------------------------
    * Other errors.
    *-------------------------------------------------------------

    *-------------------------------------------------------------
    * Build error message.
    *-------------------------------------------------------------

    cMessage := errorMessage(e)

    *-------------------------------------------------------------
    * Build options array.
    *-------------------------------------------------------------

    aOptions := {ERROR_MENU_CHOICE_BREAK}

    if (e:canRetry)
        aadd(aOptions, ERROR_MENU_CHOICE_RETRY)
    end

    if (e:canDefault)
        aadd(aOptions, ERROR_MENU_CHOICE_DEFAULT)
    end

    aadd(aOptions, ERROR_MENU_CHOICE_QUIT)

    *-------------------------------------------------------------
    * Prompt the user.
    *-------------------------------------------------------------

    nChoice := 0

    while ( nChoice == 0 )      // [Esc], 0, is not allowed.
        nChoice := alertBox( cMessage, aOptions )
    end

    *-------------------------------------------------------------
    * Do as instructed.
    *-------------------------------------------------------------

    do case
    case ( aOptions[nChoice] == ERROR_MENU_CHOICE_BREAK )

        break(e)                                        // BREAK

    case ( aOptions[nChoice] == ERROR_MENU_CHOICE_RETRY )

        return (.T.)    // RETRY                        // RETURN

    case ( aOptions[nChoice] == ERROR_MENU_CHOICE_DEFAULT )

        return (.F.)    // DEFAULT                      // RETURN

    otherwise

        *---------------------------------------------------------
        * It must be the QUIT choice. Exit.
        *---------------------------------------------------------

    end

    *-------------------------------------------------------------
    * QUIT was choosen: prepare to show the error informations.
    *-------------------------------------------------------------

    /* &&&
    *-------------------------------------------------------------
    * Translate actual cMessage with ";" into Hard CR.
    *-------------------------------------------------------------

    cMessage := strtran( cMessage, ";", NL(1) )
    */&&&

    *-------------------------------------------------------------
    * Get more informations to understand the mistake.
    *-------------------------------------------------------------

    cMessage += errorMoreInfo(e)

    *-------------------------------------------------------------
    * Show the error informations.
    *-------------------------------------------------------------

    Text( cMessage )

    *-------------------------------------------------------------
    * end of program.
    *-------------------------------------------------------------

    errorlevel(1)
    quit()

    return (.F.)

*----------------------------------------------------------------
static function errorMessage(e)

    local cMessage

    *-------------------------------------------------------------
    * Start error message: Warning/Error.
    *-------------------------------------------------------------

    if e:severity > ERROR_SEVERITY_WARNING
        cMessage := "Error "
    else
        cMessage := "Warning "
    end

    *-------------------------------------------------------------
    * Add subsystem name if available.
    *-------------------------------------------------------------

    if valType(e:subsystem) == "C"
        cMessage += e:subsystem()
    else
        cMessage += "???"
    end

    *-------------------------------------------------------------
    * Add subsystem's error code if available.
    *-------------------------------------------------------------

    if valType(e:subCode) == "N"
        cMessage += ("/" + alltrim( str(e:subCode) ) )
    else
        cMessage += "/???"
    end

    *-------------------------------------------------------------
    * Add error description if available.
    *-------------------------------------------------------------

    if valType(e:description) == "C"
        cMessage += ("  " + e:description)
    end

    *-------------------------------------------------------------
    * Add filename.
    *-------------------------------------------------------------

    if !Empty(e:filename)
        cMessage += ( NL(1) + e:filename )
    end

    *-------------------------------------------------------------
    * Add operation.
    *-------------------------------------------------------------

    if !Empty(e:operation)
        cMessage += ( NL(1) + e:operation )
    end

    *-------------------------------------------------------------
    * Add dos error.
    *-------------------------------------------------------------

    if !( Empty(e:osCode) )
        cMessage += ";(DOS Error " + alltrim( str(e:osCode) ) + ")"
    end

    *-------------------------------------------------------------
    * Return message.
    *-------------------------------------------------------------

    return cMessage

*-----------------------------------------------------------------
static function errorMoreInfo(e)

    local cMessage := ""
    local nI := 0

    cMessage += NL(2)

    *-------------------------------------------------------------
    * Procedure names.
    *-------------------------------------------------------------

    nI := 2 // to jump ErrorHandler() and ErrorMoreInfo()
    while !empty( procname(nI) )
        cMessage += rtrim( procName(nI) ) +;
            "(" + alltrim( str( procLine(nI) ) ) + ")" +;
            NL(1)
        nI++
    end

    *-------------------------------------------------------------
    * Active Alias.
    *-------------------------------------------------------------

    if len( alias() ) == 0
        cMessage +=;
            NL(1) +;
            "No Alias is selected."
    else
        cMessage +=;
             NL(1) +;
             "Actual area n. " +;
             alltrim( str( select() ) ) +;
             " Alias: " + alias() +;
             NL(2)
    end

    *-------------------------------------------------------------
    * All Aliases informations.
    *-------------------------------------------------------------

    cMessage += dbiAllStatus()

    *-------------------------------------------------------------
    * Return informations.
    *-------------------------------------------------------------

    return cMessage

*=================================================================
function errorMacro( e, cName, nLine, cCommand )
*
* Macro executor error handler
*

    local cMessage
    local aOptions
    local nChoice

    *-------------------------------------------------------------
    * Automatically resolved errors.
    *-------------------------------------------------------------

    *-------------------------------------------------------------
    * Zero division --> 0.
    *-------------------------------------------------------------

    if ( e:genCode == ERROR_GENERIC_ZERODIV )

        return (0)                                      // RETURN

    end

    *-------------------------------------------------------------
    * Network open error.
    *-------------------------------------------------------------

    if  e:genCode == ERROR_GENERIC_OPEN .and.;
        e:osCode == 32                  .and.;
        e:canDefault

        neterr(.T.)

        return (.F.)    // DEFAULT                      // RETURN

    end

    *-------------------------------------------------------------
    * Lock error during APPEND BLANK.
    *-------------------------------------------------------------

    if  e:genCode == ERROR_GENERIC_APPENDLOCK .and.;
        e:canDefault

        neterr(.T.)

        return (.F.)    // DEFAULT                      // RETURN

    end

    *-------------------------------------------------------------
    * Other errors.
    *-------------------------------------------------------------

    *-------------------------------------------------------------
    * Build error message.
    *-------------------------------------------------------------

    cMessage :=;
        alltrim( cCommand ) +;
        NL(2) +;
        errorMsgMacro( e, cName, nLine, cCommand )

    *-------------------------------------------------------------
    * Build options array.
    *-------------------------------------------------------------

    aOptions :=;
        {;
        ERROR_MENU_CHOICE_IGNORE,;
        ERROR_MENU_CHOICE_BREAK;
        }

    if e:canRetry
        aadd(aOptions, ERROR_MENU_CHOICE_RETRY)
    end

    if e:canDefault
        aadd(aOptions, ERROR_MENU_CHOICE_DEFAULT)
    end

    aadd(aOptions, ERROR_MENU_CHOICE_QUIT)

    *-------------------------------------------------------------
    * Prompt the user.
    *-------------------------------------------------------------

    nChoice := 0
    while ( nChoice == 0 )      // [Esc] is not allowed.
        nChoice := alertBox( cMessage, aOptions )
    end

    *-------------------------------------------------------------
    * Do as instructed.
    *-------------------------------------------------------------

    do case
    case ( aOptions[nChoice] == ERROR_MENU_CHOICE_IGNORE )

        *---------------------------------------------------------
        * Save status on static variable.
        *---------------------------------------------------------

        errorChoice := ERROR_MENU_CHOICE_IGNORE

        break(e)                                        // BREAK

    case ( aOptions[nChoice] == ERROR_MENU_CHOICE_BREAK )

        *---------------------------------------------------------
        * Save status on static variable.
        *---------------------------------------------------------

        errorChoice := ERROR_MENU_CHOICE_BREAK

        break(e)                                        // BREAK

    case ( aOptions[nChoice] == ERROR_MENU_CHOICE_RETRY )

        return (.T.)    // RETRY                        // RETURN

    case ( aOptions[nChoice] == ERROR_MENU_CHOICE_DEFAULT )

        return (.F.)    // DEFAULT                      // RETURN

    otherwise

        *---------------------------------------------------------
        * If here, it must be a QUIT.
        *---------------------------------------------------------

    end

    *-------------------------------------------------------------
    * QUIT was choosen: prepare message.
    *-------------------------------------------------------------

    /* &&&&
    *-------------------------------------------------------------
    * Translate actual cMessage with ";" into Hard CR.
    *-------------------------------------------------------------

    cMessage := strtran( cMessage, ";", NL(1) )
    */ &&&&

    *-------------------------------------------------------------
    * Take more informations to understand the mistake.
    *-------------------------------------------------------------

    cMessage +=;
        NL(2) +;
        errorMoreInfo( e )

    Text( cMessage )

    *-------------------------------------------------------------
    * End of program.
    *-------------------------------------------------------------

    errorlevel(1)
    quit()

    return (.F.)

*-----------------------------------------------------------------
static function errorMsgMacro( e, cName, nLine, cCommand )

    local cMessage := ""

    *-------------------------------------------------------------
    * Start error message.
    *-------------------------------------------------------------

    cMessage :=;
        upper(alltrim(cName)) +;
        " line " +;
        ltrim( str( nLine ) ) +;
        NL(1)

    *-------------------------------------------------------------
    * Get the standard informations form the normal
    * errorMessage().
    *-------------------------------------------------------------

    cMessage += errorMessage(e)

    *-------------------------------------------------------------
    * Return error message.
    *-------------------------------------------------------------

    return cMessage

*=================================================================
* EX()
*=================================================================

#define EX_ERROR_MENU_CHOICE_IGNORE        "Ignore"
#define EX_ERROR_MENU_CHOICE_BREAK         "Break"


#define EX_ERROR_FILE_NOT_FOUND;
    "Macro file not found!"
#define EX_ERROR_FILE_LOCKED;
    "Macro file is locked!"
#define EX_ERROR_FILE_DIFFERENT_STRUCTURE;
    "This file is not a macro!"

#define EX_ST_MAX_NEST       256

#define EX_ST_PROCEDURE        1
#define EX_ST_RETURN           2
#define EX_ST_ENDPROCEDURE     3
#define EX_ST_DOPROCEDURE      4

#define EX_ST_BEGIN           11
#define EX_ST_BREAK           12

#define EX_ST_IF              21
#define EX_ST_THEN            22
#define EX_ST_ELSE            23

#define EX_ST_WHILE           31

#define EX_ST_DOCASE          41
#define EX_ST_CASE            42
#define EX_ST_CASEMATCHED     43
#define EX_ST_OTHERWISE       44

#define EX_ST_MAIN            81

#define EX_ST_LOOP            97
#define EX_ST_EXIT            98
#define EX_ST_END             99


#define EX_ST_ERROR_ALONE_BREAK;
    "BREAK: missing BEGIN SEQUENCE."
#define EX_ST_ERROR_ALONE_CASE;
    "CASE: missing DO CASE."
#define EX_ST_ERROR_ALONE_ELSE;
    "ELSE: missing IF."
#define EX_ST_ERROR_ALONE_END;
    "END: missing IF|WHILE|DO CASE|BEGIN SEQUENCE."
#define EX_ST_ERROR_ALONE_ENDPROCEDURE;
    "ENDPROCEDURE: missing PROCEDURE."
#define EX_ST_ERROR_ALONE_EXIT;
    "EXIT: missing WHILE."
#define EX_ST_ERROR_ALONE_LOOP;
    "LOOP: missing WHILE."
#define EX_ST_ERROR_ALONE_OTHERWISE;
    "OTHERWISE: missing DO CASE."

#define EX_ST_ERROR_LINE_TOO_LONG;
    "Command line too long: it exeeds " +;
    ltrim( str( _MAX_STRING_LEN ) ) + " bytes."
#define EX_ST_ERROR_NO_CONDITION;
   "No condition supplied!"
#define EX_ST_ERROR_NO_VALID_CONDITION;
   "The condition is not valid: the result is not logical!"
#define EX_ST_ERROR_NO_PROCEDURE_NAME;
    "Missing procedure name."
#define EX_ST_ERROR_PROCEDURE_NOT_FOUND;
    "Procedure not found!"
#define EX_ST_ERROR_SEMICOLON_EOF;
    "The file is terminated before expected: semicolon."
#define EX_ST_ERROR_UNCLOSED_STRUCTURE;
    "Structure unclosed: missing END."
#define EX_ST_ERROR_UNCLOSED_PROCEDURE;
    "Structure unclosed: missing ENDPROCEDURE."

*=================================================================
function ex( cFileMacro )
*
* ex( <cFileMacro> ) --> nExitCode
*
* execute <cFileMacro>.
*
* <cFileMacro> The macro filename with extension.
*
*

    local cProcedure
    local nExitCode

    *-------------------------------------------------------------
    * The name must be alltrimed.
    *-------------------------------------------------------------

    cFileMacro := alltrim( cFileMacro )

    *-------------------------------------------------------------
    * Test the file.
    *-------------------------------------------------------------

    nExitCode := exTestFile( cFileMacro )

    do case
    case nExitCode == _MACRO_EXIT_NORMAL

        *---------------------------------------------------------
        * It is a "compiled" macro: execute it.
        *---------------------------------------------------------

        nExitCode := cmExecute( cFileMacro )

    case nExitCode == _MACRO_EXIT_NO_MACRO_FILE

        *---------------------------------------------------------
        * File not found.
        *---------------------------------------------------------

        alertBox(;
            cFileMacro + NL(1) + EX_ERROR_FILE_NOT_FOUND;
        )

    case nExitCode == _MACRO_EXIT_FILE_LOCKED

        *---------------------------------------------------------
        * File LOCKED.
        *---------------------------------------------------------

        alertBox(;
            cFileMacro + NL(1) + EX_ERROR_FILE_LOCKED;
        )

    case nExitCode == _MACRO_EXIT_DIFFERENT_DBF

        *---------------------------------------------------------
        * Not a right "compiled" macro file.
        *---------------------------------------------------------

        alertBox(;
            cFileMacro + NL(1) + EX_ERROR_FILE_DIFFERENT_STRUCTURE;
        )

    case nExitCode == _MACRO_EXIT_FILE_NOT_DBF

        *---------------------------------------------------------
        * Not a "compiled" macro, maybe a text macro:
        * try to execute it.
        *---------------------------------------------------------

        cProcedure := memoread( cFileMacro )
        nExitCode := execute( cProcedure, cFileMacro )

    otherwise

        *---------------------------------------------------------
        * Unknown result.
        *---------------------------------------------------------

    end

    *-------------------------------------------------------------
    * Return the exit code.
    *-------------------------------------------------------------

    return nExitCode

*-----------------------------------------------------------------
static function exTestFile( cFileName )
*
*

    local nExitCode
    local nOldSelect := select()
    local bOldErrorHandler
    local dummy

    *-------------------------------------------------------------
    * Make some tests on <cFileName>
    *-------------------------------------------------------------

    do case
    case file( cFileName )

        *---------------------------------------------------------
        * The file exists.
        *---------------------------------------------------------

        bOldErrorHandler    := errorblock( {|e| break(e)} )
        begin sequence

            *-----------------------------------------------------
            * Try to open the file: shared and read only.
            *-----------------------------------------------------

            dbusearea(.T., _DEFAULT_RDD, cFileName,;
                "_TEST_", .T., .T.)

            do case
            case neterr()

                *-------------------------------------------------
                * The file is already locked from another user.
                *-------------------------------------------------

                nExitCode := _MACRO_EXIT_FILE_LOCKED

            case;
                fieldpos("Line") > 0       .and.;
                fieldpos("Macro") > 0      .and.;
                fieldpos("Command") > 0    .and.;
                fieldpos("Goto1") > 0      .and.;
                fieldpos("Goto2") > 0

                *-------------------------------------------------
                * The file contains all the right fields.
                *-------------------------------------------------

                nExitCode := _MACRO_EXIT_NORMAL

            otherwise

                *-------------------------------------------------
                * It is a .DBF file but it is different.
                *-------------------------------------------------

                nExitCode := _MACRO_EXIT_DIFFERENT_DBF

            end

            *-----------------------------------------------------
            * Now close it.
            *-----------------------------------------------------

            dbclosearea()

        recover

            *-----------------------------------------------------
            * if opening the file an error occurred, it isn't a
            * .DBF file.
            *-----------------------------------------------------

            nExitCode := _MACRO_EXIT_FILE_NOT_DBF

        end //sequence
        errorblock( bOldErrorHandler )

    otherwise

        *---------------------------------------------------------
        * The file don't exists.
        *---------------------------------------------------------

        nExitCode   := _MACRO_EXIT_NO_MACRO_FILE

    end

    *-------------------------------------------------------------
    * Restore old values.
    *-------------------------------------------------------------

    dbselectarea( nOldSelect )

    *-------------------------------------------------------------
    * Return the exit code.
    *-------------------------------------------------------------

    return nExitCode

*-----------------------------------------------------------------
static function execute( cCommands, cName )
*
* execute( <cCommands>, [<cName>] ) --> nExitCode
*
* <cCommands>  Variable containing many commands separated
*              by NL(1) ( = CR+LF).
*
* <cName>      A name to be used for error report duriing program
*              execution.
*
* This function executes a Ascii macro file.
*

    local aNest               := array( EX_ST_MAX_NEST )
    local nNest               := 0
    local aProcedure          := {}
    local nProcedure          := 0
    local cProcedure          := ""
    local nLine
    local cCommand            := ""
    local cCondition          := ""

    local xResult
    local nExitCode           := _MACRO_EXIT_NORMAL

    *-------------------------------------------------------------
    * Start a new sequence.
    *-------------------------------------------------------------

    begin sequence

        *---------------------------------------------------------
        * Prepare the MAIN nest level.
        *---------------------------------------------------------

        nLine := 1
        nNest := 1
        aNest[nNest]          := { EX_ST_MAIN, .T. }
        // [1]MAIN  [2]lContinue

        *---------------------------------------------------------
        * Start the interpreter loop.
        *---------------------------------------------------------

        while .T.                                       // FOREVER

            *-----------------------------------------------------
            * Get the <nLine> line: it can be distributed on
            * different lines and may contains comments.
            * The function exCommandExtract() get a unique line
            * form the <nLine> initial position and updates
            * <nLine> in the way that it now contains the
            * last line number of the command.
            *-----------------------------------------------------

            cCommand   := exCommandExtract( cCommands, @nLine )

            *-----------------------------------------------------
            * if <cCommand> contains NIL the file is terminated.
            *-----------------------------------------------------

            if cCommand == NIL  // end of file

                nExitCode := _MACRO_EXIT_NORMAL

                break                                   // BREAK

            end

            do case
            case upper( cCommand ) == "END"

                *-------------------------------------------------
                * Try to close one nest level.
                *-------------------------------------------------

                do case
                case aNest[nNest][1] == EX_ST_MAIN

                    *---------------------------------------------
                    * Not allowed: the main level cannot be
                    * closed.
                    *---------------------------------------------

                    alertBox( cName +;
                        "(" +;
                        ltrim(str(nLine)) +;
                        ")" +;
                        NL(1) +;
                        EX_ST_ERROR_ALONE_END;
                    )

                    nExitCode := _MACRO_EXIT_STATEMENT_ERROR

                    break                               // BREAK

                case aNest[nNest][1] == EX_ST_DOPROCEDURE

                    *---------------------------------------------
                    * Not allowed: END cannot close a procedure.
                    *---------------------------------------------

                    alertBox( cName +;
                        "(" +;
                        ltrim(str(nLine)) +;
                        ")" +;
                        NL(1) +;
                        EX_ST_ERROR_ALONE_END;
                    )
                    nExitCode := _MACRO_EXIT_STATEMENT_ERROR

                    break                               // BREAK

                case;
                    aNest[nNest][1] == EX_ST_IF         .or.;
                    aNest[nNest][1] == EX_ST_THEN       .or.;
                    aNest[nNest][1] == EX_ST_ELSE

                    *---------------------------------------------
                    * Close the nest level.
                    *---------------------------------------------

                    aNest[nNest] := NIL
                    nNest--

                    *---------------------------------------------
                    * Prepare to read the next line.
                    *---------------------------------------------

                    nLine++

                case aNest[nNest][1] == EX_ST_BEGIN

                    *---------------------------------------------
                    * Close the nest level.
                    *---------------------------------------------

                    aNest[nNest] := NIL
                    nNest--

                    *---------------------------------------------
                    * Prepare to read the next line.
                    *---------------------------------------------

                    nLine++

                case aNest[nNest][1] == EX_ST_WHILE

                    *---------------------------------------------
                    * Check the while condition.
                    *---------------------------------------------

                    xResult :=; 
                        exEvalCondition(; 
                            aNest[nNest][2], cName, nLine; 
                        )

                    *---------------------------------------------
                    * If the condition is not valid, <xReturn>
                    * contains NIL, else it has a logical value.
                    *---------------------------------------------
                    
                    if xResult == NIL

                        break                           // BREAK

                    end

                    do case
                    case xResult

                        *-----------------------------------------
                        * The condition is true: cContinue the
                        * while loop moving the line counter to
                        * the begin of the while.
                        *-----------------------------------------

                        nLine := aNest[nNest][3]

                    otherwise

                        *-----------------------------------------
                        * The condition is not true: terminate
                        * the while loop.
                        * Close the nest level.
                        *-----------------------------------------

                        aNest[nNest] := NIL
                        nNest--

                        *-----------------------------------------
                        * Prepare to read the next line.
                        *-----------------------------------------

                        nLine++

                    end

                case aNest[nNest][1] == EX_ST_LOOP

                    *---------------------------------------------
                    * This level means that the interpreter is
                    * trying to LOOP a while.
                    * Close the nest level and test the previous
                    * one.
                    *---------------------------------------------

                    aNest[nNest] := NIL
                    nNest--

                    *---------------------------------------------
                    * <nLine> remains on last END.
                    *---------------------------------------------

                    do case
                    case aNest[nNest][1] == EX_ST_WHILE

                        *-----------------------------------------
                        * If the previous level is a while, the
                        * LOOP statement is arrived to the right
                        * level. Now, the while condition must
                        * be checked.
                        *-----------------------------------------

                        xResult :=; 
                            exEvalCondition(; 
                                aNest[nNest][2], cName, nLine; 
                            )

                        *-----------------------------------------
                        * If the condition is not valid, <xReturn>
                        * contains NIL, else it has a logical 
                        * value.
                        *-----------------------------------------
                    
                        if xResult == NIL

                            break                       // BREAK

                        end

                        do case
                        case xResult

                            *-------------------------------------
                            * Repeat While loop.
                            *-------------------------------------

                            nLine := aNest[nNest][3]

                        otherwise

                            *-------------------------------------
                            * Exit While loop: as we are not at
                            * the ENDwhile position, this must
                            * be searched.
                            *-------------------------------------

                            exJumpIt(;
                                cCommands, @nLine,;
                                {|cLine| cLine=="END"};
                            )

                            *-------------------------------------
                            * Now <nLine> should be on next
                            * END.
                            * <nLine> is not advanced and the nest
                            * level is not reduced: if it is
                            * the right ENDwhile, the next loop
                            * will end "naturally".
                            *-------------------------------------

                        end

                    otherwise

                        aNest[nNest][1] := EX_ST_LOOP

                        *-----------------------------------------
                        * The previous level is still a LOOP.
                        * Go to next END: the function exjumpIt()
                        * begins with the <nLine> incremet, this
                        * is the reason why the <nLine> pointer
                        * was not incremented after the nest
                        * level was reduced.
                        *-----------------------------------------

                        exJumpIt(;
                            cCommands, @nLine,;
                            {|cLine| cLine=="END"};
                        )

                        *-----------------------------------------
                        * Now <nLine> should be on next
                        * END.
                        * <nLine> is not advanced and the nest
                        * level is not reduced: if it is
                        * the right ENDwhile, the next loop
                        * will end "naturally".
                        *-----------------------------------------

                    end

                case aNest[nNest][1] == EX_ST_BREAK

                    *---------------------------------------------
                    * This level means that the interpreter is
                    * trying to exit a SEQUENCE.
                    * Close the nest level and test the previous
                    * one.
                    *---------------------------------------------

                    aNest[nNest] := NIL
                    nNest--

                    *---------------------------------------------
                    * <nLine> remains on last END.
                    *---------------------------------------------

                    do case
                    case aNest[nNest][1] == EX_ST_BEGIN

                        *-----------------------------------------
                        * If the previous level is a BEGIN, the
                        * BREAK statement is arrived to the right
                        * level. Now, the ENDsequence must
                        * be reached.
                        *-----------------------------------------

                        exJumpIt(;
                            cCommands, @nLine,;
                            {|cLine| cLine=="END"};
                        )

                        *-----------------------------------------
                        * Now <nLine> should be on next
                        * END.
                        * <nLine> is not advanced and the nest
                        * level is not reduced: if it is
                        * the right ENDsequence, the next loop
                        * will end "naturally".
                        *-----------------------------------------

                    otherwise

                        aNest[nNest][1] := EX_ST_BREAK

                        *-----------------------------------------
                        * The previous level is still a BREAK.
                        * Go to next END: the function exjumpIt()
                        * begins with the <nLine> incremet, this
                        * is the reason why the <nLine> pointer
                        * was not incremented after the nest
                        * level was reduced.
                        *-----------------------------------------

                        exJumpIt( cCommands, @nLine,;
                            {|cLine| cLine=="END"} )

                        *-----------------------------------------
                        * Now <nLine> should be on next
                        * END.
                        * <nLine> is not advanced and the nest
                        * level is not reduced: if it is
                        * the right ENDsequence, the next loop
                        * will end "naturally".
                        *-----------------------------------------

                    end

                case aNest[nNest][1] == EX_ST_EXIT

                    *---------------------------------------------
                    * This level means that the interpreter is
                    * trying to EXIT a while.
                    * Close the nest level and test the previous
                    * one.
                    *---------------------------------------------

                    aNest[nNest] := NIL
                    nNest--

                    *---------------------------------------------
                    * <nLine> remains on last END.
                    *---------------------------------------------

                    do case
                    case aNest[nNest][1] == EX_ST_WHILE

                        *-----------------------------------------
                        * If the previous level is a WHILE, the
                        * EXIT statement is arrived to the right
                        * level. Now, the while condition is
                        * transformed in a way that it will be
                        * shurely false.
                        *-----------------------------------------

                        aNest[nNest][2] := ".F."

                        *-----------------------------------------
                        * Now the END must be found.
                        *-----------------------------------------

                        exJumpIt(;
                            cCommands, @nLine,;
                            {|cLine| cLine=="END"};
                        )

                        *-----------------------------------------
                        * Now <nLine> should be on next
                        * END.
                        * <nLine> is not advanced and the nest
                        * level is not reduced: if it is
                        * the right ENDwhile, the next loop
                        * will end "naturally".
                        *-----------------------------------------

                    otherwise

                        aNest[nNest][1] := EX_ST_EXIT

                        *-----------------------------------------
                        * The previous level is still a EXIT.
                        * Go to next END: the function exjumpIt()
                        * begins with the <nLine> incremet, this
                        * is the reason why the <nLine> pointer
                        * was not incremented after the nest
                        * level was reduced.
                        *-----------------------------------------

                        exJumpIt(;
                            cCommands, @nLine,;
                            {|cLine| cLine=="END"};
                        )

                        *-----------------------------------------
                        * Now <nLine> should be on next
                        * END.
                        * <nLine> is not advanced and the nest
                        * level is not reduced: if it is
                        * the right ENDwhile, the next loop
                        * will end "naturally".
                        *-----------------------------------------

                    end

                case;
                    aNest[nNest][1] == EX_ST_DOCASE     .or.;
                    aNest[nNest][1] == EX_ST_CASEMATCHED

                    *---------------------------------------------
                    * Close the nest level.
                    *---------------------------------------------

                    aNest[nNest] := NIL
                    nNest--

                    *---------------------------------------------
                    * Prepare to read the next line.
                    *---------------------------------------------

                    nLine++

                end

                *-------------------------------------------------
                * The END statement test is terminated.
                *-------------------------------------------------

            case upper( cCommand ) == "PROCEDURE"

                *-------------------------------------------------
                * Error: no procedure name following PROCEDURE.
                *-------------------------------------------------

                alertBox( cName +;
                    "(" +;
                    ltrim(str(nLine)) +;
                    ")" +;
                    NL(1) +;
                    EX_ST_ERROR_NO_PROCEDURE_NAME;
                )

                nExitCode := _MACRO_EXIT_STATEMENT_ERROR

                break                                   // BREAK

            case upper( left( cCommand, 10 ) ) == "PROCEDURE "

                *-------------------------------------------------
                * Save the procedure name inside the
                * procedure array.
                *-------------------------------------------------

                cProcedure :=;
                    alltrim( substr( ltrim( cCommand ), 10 ) )
                cProcedure := upper( cProcedure )
                aadd( aProcedure, { cProcedure, nLine+1 } )

                *-------------------------------------------------
                * Jump to the line following the ENDPROCEDURE
                *-------------------------------------------------

                exEndproc( cCommands, @nLine )
                nLine++

            case upper( cCommand ) == "DO PROCEDURE"

                *-------------------------------------------------
                * Erro: no procedure name to call.
                *-------------------------------------------------

                alertBox( cName +;
                    "(" +;
                    ltrim(str(nLine)) +;
                    ")" +;
                    NL(1) +;
                    EX_ST_ERROR_PROCEDURE_NOT_FOUND;
                )

                nExitCode := _MACRO_EXIT_STATEMENT_ERROR

                break                                   // BREAK

            case upper( left( cCommand, 13 ) ) == "DO PROCEDURE "

                *-------------------------------------------------
                * A new nest is added.
                *-------------------------------------------------

                nNest++
                aNest[nNest] := {EX_ST_DOPROCEDURE, nLine+1}
                // [1]PROCEDURE  [2]nNextLine

                *-------------------------------------------------
                * Read the procedure name.
                *-------------------------------------------------

                cProcedure :=;
                    alltrim( substr( ltrim( cCommand ), 13 ) )
                cProcedure := upper( cProcedure )

                *-------------------------------------------------
                * Scan the procedure array.
                *-------------------------------------------------

                for nProcedure := 1 to len( aProcedure )+1  //!!

                    if nProcedure > len( aProcedure )

                        *-----------------------------------------
                        * Error: the procedure name is not jet
                        * appeared.
                        *-----------------------------------------

                        alertBox(;
                            cName +;
                            "(" +;
                            ltrim(str(nLine)) +;
                            ")" +;
                            NL(1) +;
                            EX_ST_ERROR_PROCEDURE_NOT_FOUND;
                        )

                        nExitCode := _MACRO_EXIT_STATEMENT_ERROR

                        break                           // BREAK

                    end

                    if aProcedure[nProcedure][1] == cProcedure

                        *-----------------------------------------
                        * The procedure name is found: the <nLine>
                        * pointer is moved to the begin of the
                        * called procedure.
                        *-----------------------------------------

                        nLine := aProcedure[nProcedure][2]

                        exit                            // EXIT

                    end

                next

            case upper( cCommand ) == "RETURN"

                *-------------------------------------------------
                * The return line is contained inside
                * <aNest> at the correponding DOPROCEDURE
                * position
                *-------------------------------------------------

                while .T.                               // FOREVER

                    do case
                    case aNest[nNest][1] == EX_ST_DOPROCEDURE

                        *-----------------------------------------
                        * If the current level is a DOPROCEDURE,
                        * the RETURN statement is arrived to the
                        * right level: the <nLine> pointer is
                        * moved back to the position after the
                        * procedure.
                        *-----------------------------------------

                        nLine := aNest[nNest][2]

                        *-----------------------------------------
                        * The nest level is closed.
                        *-----------------------------------------

                        aNest[nNest] := NIL
                        nNest--

                        exit                            // EXIT

                    case aNest[nNest][1] == EX_ST_MAIN

                        *-----------------------------------------
                        * This RETURN must be considered
                        * as a "terminate file execution".
                        *-----------------------------------------

                        break                           // BREAK

                    otherwise

                        *-----------------------------------------
                        * This is not the right level: try to
                        * go back one nest.
                        *-----------------------------------------

                        aNest[nNest] := NIL
                        nNest--

                    end

                end

            case upper( cCommand ) == "ENDPROCEDURE"

                *-------------------------------------------------
                * The return line is contained inside
                * <aNest> at the correponding DOPROCEDURE
                * position
                *-------------------------------------------------

                while .T.                               // FOREVER

                    do case
                    case aNest[nNest][1] == EX_ST_DOPROCEDURE

                        *-----------------------------------------
                        * If the current level is a DOPROCEDURE,
                        * the ENDPROCEDURE statement appears
                        * at the right level: the <nLine> pointer
                        * is moved back to the position after the
                        * procedure.
                        *-----------------------------------------

                        nLine := aNest[nNest][2]

                        *-----------------------------------------
                        * The nest level is closed.
                        *-----------------------------------------

                        aNest[nNest] := NIL
                        nNest--

                        exit                            // EXIT

                    otherwise

                        *-----------------------------------------
                        * Error: ENDPROCEDURE cannot appear
                        * inside a different level then
                        * DO PROCEDURE.
                        *-----------------------------------------

                        alertBox(;
                            cName +;
                            "(" +;
                            ltrim(str(nLine)) +;
                            ")" +;
                            NL(1) +;
                            EX_ST_ERROR_ALONE_ENDPROCEDURE;
                        )

                        nExitCode := _MACRO_EXIT_STATEMENT_ERROR

                        break                           // BREAK

                    end

                end

            case upper( cCommand ) == "BEGIN SEQUENCE"

                *-------------------------------------------------
                * Add a new nest level
                *-------------------------------------------------

                nNest++
                aNest[nNest] := {EX_ST_BEGIN}
                // [1]BEGIN SEQUENCE

                *-------------------------------------------------
                * Prepare the line pointer to the next line.
                *-------------------------------------------------

                nLine++

            case upper( cCommand ) == "BREAK"

                do case
                case aNest[nNest][1] == EX_ST_BEGIN

                    *---------------------------------------------
                    * The BREAK statement is arrived to the right
                    * level. Now, the ENDsequence must
                    * be reached.
                    *---------------------------------------------

                    exJumpIt(;
                        cCommands, @nLine,;
                        {|cLine| cLine=="END"};
                    )

                    *---------------------------------------------
                    * Now <nLine> should be on next
                    * END.
                    * <nLine> is not advanced and the nest
                    * level is not reduced: if it is
                    * the right ENDsequence, the next loop
                    * will end "naturally".
                    *---------------------------------------------

                otherwise

                    *---------------------------------------------
                    * This level is not a BEGIN SEQUENCE.
                    * The right ENDsequence must be found, so
                    * transform this level into a BREAK level.
                    *---------------------------------------------

                    aNest[nNest][1] := EX_ST_BREAK

                    *---------------------------------------------
                    * Go to next END: the function exjumpIt()
                    * begins with the <nLine> incremet, this
                    * is the reason why the <nLine> pointer
                    * was not incremented after the nest
                    * level was reduced.
                    *---------------------------------------------

                    exJumpIt(;
                        cCommands, @nLine,;
                        {|cLine| cLine=="END"};
                    )

                    *---------------------------------------------
                    * Now <nLine> should be on next END.
                    * <nLine> is not advanced and the nest
                    * level is not reduced: if it is
                    * the right ENDsequence, the next loop
                    * will end "naturally".
                    *---------------------------------------------

                end

            case upper( cCommand ) == "IF"

                *-------------------------------------------------
                * Error: no condition following IF.
                *-------------------------------------------------

                alertBox(;
                    cName +;
                    "(" +;
                    ltrim(str(nLine)) +;
                    ")" +;
                    NL(1) +;
                    EX_ST_ERROR_NO_CONDITION;
                )

                nExitCode := _MACRO_EXIT_STATEMENT_ERROR

                break                                   // BREAK

            case upper( left( cCommand, 3 ) ) == "IF "

                *-------------------------------------------------
                * Add a new nest level.
                *-------------------------------------------------

                nNest++
                aNest[nNest] := {EX_ST_IF }
                // [1]IF|THEN|ELSE

                *-------------------------------------------------
                * Isolate the condition.
                *-------------------------------------------------

                cCondition :=;
                    alltrim( substr( ltrim(cCommand), 3 ) )

                *-------------------------------------------------
                * Test the condition.
                *-------------------------------------------------

                xResult :=; 
                    exEvalCondition(; 
                        cCondition, cName, nLine; 
                    )

                *-------------------------------------------------
                * If the condition is not valid, <xReturn>
                * contains NIL, else it has a logical 
                * value.
                *-------------------------------------------------
                
                if xResult == NIL

                    break                               // BREAK

                end

                if xResult

                    *---------------------------------------------
                    * The condition is true: add a new nest level.
                    *---------------------------------------------

                    aNest[nNest][1] := EX_ST_THEN
                    nLine++

                else

                    *---------------------------------------------
                    * The condition is false, so the ELSE or END
                    * shuld be reached: add a new nest level.
                    *---------------------------------------------

                    aNest[nNest][1] := EX_ST_ELSE

                    exJumpIt(;
                        cCommands, @nLine,;
                        {|cLine| cLine=="ELSE"},;
                        {|cLine| cLine=="END"};
                    )

                    *---------------------------------------------
                    * Now <nLine> should be on next END.
                    * <nLine> is not advanced and the nest
                    * level is not reduced: if it is
                    * the right ENDsequence, the next loop
                    * will end "naturally".
                    *---------------------------------------------

                end

            case upper( cCommand ) == "ELSE"

                *-------------------------------------------------
                * Test the current nest level to determinate the
                * correct action.
                *-------------------------------------------------

                do case
                case aNest[nNest][1] == EX_ST_ELSE

                    *---------------------------------------------
                    * This means that the THEN statement
                    * wasn't executed.
                    * Ok execute ELSE: move the <nLine> pointer
                    * to the next line.
                    *---------------------------------------------

                    nLine++

                otherwise

                    *---------------------------------------------
                    * If it wasn't a ELSE it was a THEN.
                    * As THEN was executed, it must be jumped.
                    *---------------------------------------------

                    exJumpIt(;
                        cCommands, @nLine,;
                        {|cLine| cLine=="END"};
                    )

                    *---------------------------------------------
                    * Now <nLine> should be on next END.
                    * <nLine> is not advanced and the nest
                    * level is not reduced: if it is
                    * the right ENDsequence, the next loop
                    * will end "naturally".
                    *---------------------------------------------

                end

            case upper( cCommand ) == "DO CASE"

                *-------------------------------------------------
                * Add a new nest level.
                *-------------------------------------------------

                nNest++
                aNest[nNest] := {EX_ST_DOCASE}
                // [1]DOCASE|CASEMATCHED

                *-------------------------------------------------
                * Prepare the line pointer to the next line.
                *-------------------------------------------------

                nLine++

            case upper( cCommand ) == "CASE"

                *-------------------------------------------------
                * Error: no condition following CASE.
                *-------------------------------------------------

                alertBox(;
                    cName +;
                    "(" +;
                    ltrim(str(nLine)) +;
                    ")" +;
                    NL(1) +;
                    EX_ST_ERROR_NO_CONDITION;
                )

                nExitCode := _MACRO_EXIT_STATEMENT_ERROR

                break                                   // BREAK

            case upper( left( cCommand, 5 ) ) == "CASE "

                *-------------------------------------------------
                * Test the current nest level to determinate the
                * correct action.
                *-------------------------------------------------

                do case
                case aNest[nNest][1] == EX_ST_DOCASE

                    *---------------------------------------------
                    * This level means that a DO CASE is started
                    * and no matching CASE was found: Ok try it.
                    *---------------------------------------------

                    cCondition :=;
                        alltrim( substr( ltrim(cCommand), 5 ) )

                    *---------------------------------------------
                    * Test the condition.
                    *---------------------------------------------
                    
                    xResult :=; 
                        exEvalCondition(; 
                            cCondition, cName, nLine; 
                        )

                    *---------------------------------------------
                    * If the condition is not valid, <xReturn>
                    * contains NIL, else it has a logical 
                    * value.
                    *---------------------------------------------
                
                    if xResult == NIL

                        break                               // BREAK

                    end

                    do case
                    case xResult

                        *-----------------------------------------
                        * The actual nest level is changed to
                        * tell that a CASE matched the condition
                        * and no more CASE should be tested.
                        *-----------------------------------------

                        aNest[nNest][1] := EX_ST_CASEMATCHED

                        *-----------------------------------------
                        * Execute the following lines as the
                        * CASE condition was true.
                        *-----------------------------------------

                        nLine++

                   otherwise

                        *-----------------------------------------
                        * The condition results false: 
                        * try to find next CASE or OTHERWISE.
                        *-----------------------------------------

                        exJumpIt(; 
                            cCommands,; 
                            @nLine,;
                            {|cLine| left(cLine,5)=="CASE "},;
                            {|cLine| cLine=="OTHERWISE"},;
                            {|cLine| cLine=="END"}; 
                        )

                        *-----------------------------------------
                        * Now <nLine> should be on next CASE,
                        * OTHERWISE or END.
                        * <nLine> is not advanced and the nest
                        * level is not reduced.
                        *-----------------------------------------

                    end

                case aNest[nNest][1] == EX_ST_CASEMATCHED

                    *---------------------------------------------
                    * This level means that a DO CASE is started
                    * and the matching CASE is just terminated:
                    * reach the ENDcase.
                    *---------------------------------------------

                    exJumpIt(; 
                        cCommands,; 
                        @nLine,;
                        {|cLine| cLine=="END"}; 
                    )
                        
                    *---------------------------------------------
                    * Now <nLine> should be on next END.
                    * <nLine> is not advanced and the nest
                    * level is not reduced.
                    *---------------------------------------------

                otherwise

                    *---------------------------------------------
                    * Error: CASE cannot appear inside a different
                    * nest level.
                    *---------------------------------------------
                    
                    alertBox( cName +;
                        "(" +;
                        ltrim(str(nLine)) +;
                        ")" +;
                        NL(1) +;
                        EX_ST_ERROR_ALONE_CASE )

                    nExitCode := _MACRO_EXIT_STATEMENT_ERROR

                    break

                end

            case upper( cCommand ) == "OTHERWISE"

                *-------------------------------------------------
                * Test the current nest level to determinate the
                * correct action.
                *-------------------------------------------------
                
                do case
                case aNest[nNest][1] == EX_ST_DOCASE
                    
                    *---------------------------------------------
                    * This nest level means that a DO CASE is 
                    * started and no CASE was found true: Ok do it
                    * simply moving the <nLine> pointer to the
                    * begin of the statements contained inside
                    * the otherwise.
                    *---------------------------------------------
                    
                    nLine++

                case aNest[nNest][1] == EX_ST_CASEMATCHED

                    *---------------------------------------------
                    * This nest level means that a DO CASE is 
                    * started and the last CASE was found true: 
                    * Jump it - fine the ENDcase.
                    *---------------------------------------------
                    
                    exJumpIt(; 
                        cCommands,; 
                        @nLine,;
                        {|cLine| cLine=="END"}; 
                    )

                    *---------------------------------------------
                    * Now <nLine> should be on next END.
                    * <nLine> is not advanced and the nest
                    * level is not reduced.
                    *---------------------------------------------
                
                otherwise

                    *---------------------------------------------
                    * Error: OTHERWISE cannot appear on a 
                    * different nest level.
                    *---------------------------------------------
                    
                    alertBox( cName +;
                        "(" +;
                        ltrim(str(nLine)) +;
                        ")" +;
                        NL(1) +;
                        EX_ST_ERROR_ALONE_OTHERWISE; 
                    )

                    nExitCode := _MACRO_EXIT_STATEMENT_ERROR

                    break                               // BREAK

                end

            case upper( cCommand ) == "WHILE"

                *-------------------------------------------------
                * Error: no condition following WHILE.
                *-------------------------------------------------

                alertBox(;
                    cName +;
                    "(" +;
                    ltrim(str(nLine)) +;
                    ")" +;
                    NL(1) +;
                    EX_ST_ERROR_NO_CONDITION;
                )

                nExitCode := _MACRO_EXIT_STATEMENT_ERROR

                break                                   // BREAK

            case upper( left( cCommand, 6 ) ) == "WHILE "

                *-------------------------------------------------
                * Add a new nest level: WHILE, cWhileCondition,
                * nBeginPoint.
                *-------------------------------------------------

                nNest++
                aNest[nNest] := {EX_ST_WHILE, NIL, NIL }
                // [1]WHILE  [2]cCondition  [3]nLine1stStatement

                *-------------------------------------------------
                * Save the WhileCondition.
                *-------------------------------------------------

                aNest[nNest][2] :=;
                    alltrim( substr( ltrim(cCommand), 6 ) )

                *-------------------------------------------------
                * Save the first statement begin point
                * (it is better).
                *-------------------------------------------------

                aNest[nNest][3] := nLine +1

                *-------------------------------------------------
                * Test the condition.
                *-------------------------------------------------

                xResult :=; 
                    exEvalCondition(; 
                        aNest[nNest][2], cName, nLine; 
                    )

                *-------------------------------------------------
                * If the condition is not valid, <xReturn>
                * contains NIL, else it has a logical 
                * value.
                *-------------------------------------------------
                
                if xResult == NIL

                    break                               // BREAK

                end

                do case
                case xResult

                    *---------------------------------------------
                    * The condition is True: Ok, go on, prepare
                    * the <nLine> pointer to the next line that
                    * is the first line contained inside the
                    * WHILE.
                    *---------------------------------------------

                    nLine++

                otherwise

                    *---------------------------------------------
                    * Tranform the WHILE condition into a fixed
                    * false character condition, then try to
                    * reach the next ENDwhile.
                    *---------------------------------------------

                    aNest[nNest][2] := ".F."

                    exJumpIt(;
                        cCommands, @nLine,;
                        {|cLine| cLine=="END"};
                    )

                    *---------------------------------------------
                    * At this point, <nLine> points to a END
                    * statment that is presumed to be the right
                    * ENDwhile.
                    * The level is not closed because next
                    * interpretation loop will find END and will
                    * close it.
                    *---------------------------------------------

                end

            case upper( cCommand ) == "LOOP"

                *-------------------------------------------------
                * The statement LOOP should close a WHILE.
                *-------------------------------------------------

                do case
                case aNest[nNest][1] == EX_ST_WHILE

                    *---------------------------------------------
                    * This is the right nest level.
                    * Test the condition.
                    *---------------------------------------------
                    
                    xResult :=; 
                        exEvalCondition(; 
                            aNest[nNest][2], cName, nLine; 
                        )

                    *---------------------------------------------
                    * If the condition is not valid, <xReturn>
                    * contains NIL, else it has a logical 
                    * value.
                    *---------------------------------------------
                
                    if xResult == NIL

                        break                               // BREAK

                    end


                    do case
                    case xResult

                        *-----------------------------------------
                        * The condition is still True:
                        * repeat the WHILE loop.
                        *-----------------------------------------

                        nLine := aNest[nNest][3]

                    otherwise

                        *-----------------------------------------
                        * The condition is False:
                        * exit the WHILE loop.
                        * The WhileCondition is changed into a
                        * fixed False condition and the ENDwhile
                        * is to find.
                        *-----------------------------------------

                        aNest[nNest][2] := ".F."

                        exJumpIt(;
                            cCommands, @nLine,;
                            {|cLine| cLine=="END"};
                        )

                        *-----------------------------------------
                        * At this point, <nLine> points to a END
                        * statment that is presumed to be the
                        * right ENDwhile.
                        * The level is not closed because next
                        * interpretation loop will find END and
                        * will close it.
                        *-----------------------------------------

                    end

                otherwise

                    *---------------------------------------------
                    * The nest level is not WHILE. This means
                    * that WHILE should be a previous one.
                    * This level is changed into LOOP to tell
                    * the interpreter that the WHILE nest level
                    * is the target.
                    *---------------------------------------------

                    aNest[nNest][1] := EX_ST_LOOP

                    exJumpIt(;
                        cCommands, @nLine,;
                        {|cLine| cLine=="END"};
                    )

                    *---------------------------------------------
                    * At this point, <nLine> points to a END
                    * statment that is presumed to be the right
                    * END of the current level.
                    * The level is not closed because next
                    * interpretation loop will find END and will
                    * close it.
                    *---------------------------------------------

                end

            case upper( cCommand ) == "EXIT"

                *-------------------------------------------------
                * The EXIT statement should close a WHILE.
                *-------------------------------------------------

                do case
                case aNest[nNest][1] == EX_ST_WHILE

                    *---------------------------------------------
                    * This is the right nest level.
                    * The WhileCondition is changed into a
                    * fixed False condition.
                    *---------------------------------------------

                    aNest[nNest][2] := ".F."

                    *---------------------------------------------
                    * The next problem is to reach the right
                    * ENDwhile.
                    *---------------------------------------------

                    exJumpIt(;
                        cCommands, @nLine,;
                        {|cLine| cLine=="END"};
                    )

                otherwise
                    aNest[nNest][1] := EX_ST_EXIT
                    exJumpIt( cCommands, @nLine,;
                        {|cLine| cLine=="END"} )
                end

                *-------------------------------------------------
                * At this point, <nLine> points to a END
                * statment that is presumed to be the right
                * ENDwhile.
                * The level is not closed because next
                * interpretation loop will find END and will
                * close it.
                *-------------------------------------------------

            otherwise

                *-------------------------------------------------
                * Finally, if no statement is recognised, it must
                * be a function or an expression.
                * The line is executed.
                *-------------------------------------------------

                nExitCode := exOneCommand( cCommand, cName, nLine )

                if nExitCode == _MACRO_EXIT_BREAK

                    break                               // BREAK

                end

                *-------------------------------------------------
                * As the line execution was normally executed,
                * the <nLine> pointer is prepared on the next
                * line.
                *-------------------------------------------------

                nLine++

            end

        end

    end //sequence

    *-------------------------------------------------------------
    * The exit code is returned.
    *-------------------------------------------------------------

    return nExitCode

*-----------------------------------------------------------------
static function exCommandExtract( cCommands, nLine )
*
* exCommandExtract( <cCommands>, @<nLine> ) --> NIL
*                                           --> cCommand
*
* It returns one complete command or NIL if end of file.
*
* <cCommands>  The variable containing the commands.
*
* <nLine>      The actual line position.
*

    local cCommand      := ""
    local nSemicolonPos := 0

    *-------------------------------------------------------------
    * Move the waitWheel.
    *-------------------------------------------------------------

    waitWheel()

    *-------------------------------------------------------------
    * Reach next not-empty line.
    *-------------------------------------------------------------

    while .T.                                           // FOREVER

        cCommand := memoline( cCommands, _MAX_STRING_LEN, nLine )

        *---------------------------------------------------------
        * Analise what was read.
        *---------------------------------------------------------

        do case
        case cCommand == ""

            *-----------------------------------------------------
            * End of file reached.
            *-----------------------------------------------------

            return NIL

        case !( empty( cCommand ) )

            *-----------------------------------------------------
            * Cut eventual comment.
            *-----------------------------------------------------

            cCommand := exCommentCut( cCommand )

            *-----------------------------------------------------
            * Cut unusefull blanks.
            *-----------------------------------------------------

            cCommand := alltrim( cCommand )

            *-----------------------------------------------------
            * Check if there is something.
            *-----------------------------------------------------

            if !( cCommand == "" )

                *-------------------------------------------------
                * The remaining is not empty: exit this loop.
                *-------------------------------------------------

                exit                                    // EXIT

            end

            *-----------------------------------------------------
            * If the <cCommand> contains only spaces: loop.
            *-----------------------------------------------------

        end

        *---------------------------------------------------------
        * If still here, the next line must be read.
        *---------------------------------------------------------

        nLine++

    end

    *-------------------------------------------------------------
    * At this point there is a command line that maybe continiues
    * on the next line with a semicolon.
    *-------------------------------------------------------------

    if right( cCommand, 1 ) == ";"

        *---------------------------------------------------------
        * It follows to the next line.
        *---------------------------------------------------------

        cCommand := left( cCommand, len( cCommand )-1 )

        nLine++

        *---------------------------------------------------------
        * Call the same function recursively to obtain the
        * rest of the command.
        *---------------------------------------------------------

        cCommand += exCommandExtract( cCommands, @nLine )

    end

    *-------------------------------------------------------------
    * A complete command is returned and <nLine> contains the
    * line number that terminates the command.
    *-------------------------------------------------------------

    return cCommand

*-----------------------------------------------------------------
static function exCommentCut( cCommandLine )
*
* exCommentCut( <cCommandLine> ) --> cCommandLine
*
* The function cuts the command line at the first
* occurence of "//" that is used as only comment
* indicator.
*
* The string "//" cannot be used for other purpose
* as the command line will be truncated there.
*
* <cCommandLine>       a line to be cutted.
*

    local nCommentPos

    *-------------------------------------------------------------
    * Locate the comment position.
    *-------------------------------------------------------------

    nCommentPos := at( "//", cCommandLine )

    *-------------------------------------------------------------
    * If a comment is found, cut the comment.
    *-------------------------------------------------------------

    if nCommentPos > 0
        cCommandLine := substr( cCommandLine, 1, nCommentPos -1 )
    end

    *-------------------------------------------------------------
    * Return the cleanded command.
    *-------------------------------------------------------------

    return cCommandLine

*-----------------------------------------------------------------
static function exOneCommand( cCommand, cName, nLine )
*
* exOneCommand( <cCommand>, <cName>, <nLine> ) --> nExitCode
*
* This function execute a command after cleaning.
*
* <cCommand>   The command to execute.
*
* <cName>      The procedure name to be used for error
*              documentation.
*
* <nLine>      Actual command line to be used for error
*              documentation.
*

    local bSaveErrorHandler
    local xResult
    local nExitCode := _MACRO_EXIT_NORMAL

    *-------------------------------------------------------------
    * Empty lines should not be executed.
    *-------------------------------------------------------------

    if cCommand == ""
        return _MACRO_EXIT_NORMAL
    end

    *-------------------------------------------------------------
    * Prepare for command macro execution: handle the possible
    * error.
    *-------------------------------------------------------------

    bSaveErrorHandler :=;
        errorblock(;
            {|e|;
                errorMacro(;
                    e,;
                    cName,;
                    nLine,;
                    cCommand;
                );
            };
        )

    begin sequence

        *---------------------------------------------------------
        * &
        *---------------------------------------------------------

        xResult := &(cCommand)

    recover

        *---------------------------------------------------------
        * Remember that <errorChoice> is a global static variable.
        *---------------------------------------------------------

        do case
        case errorChoice == EX_ERROR_MENU_CHOICE_IGNORE

            *-----------------------------------------------------
            * Ignore error.
            *-----------------------------------------------------

        case errorChoice == EX_ERROR_MENU_CHOICE_BREAK

            *-----------------------------------------------------
            * The macro execution is broken.
            *-----------------------------------------------------

            nExitCode := _MACRO_EXIT_BREAK

        end

    end

    errorblock(bSaveErrorHandler)

    *-------------------------------------------------------------
    * The exit code is returned.
    *-------------------------------------------------------------

    return nExitCode

*-----------------------------------------------------------------
static function exEvalCondition( cCondition, cName, nLine )
*
* exEvalCondition( <cCondition>, <cName>, <nLine> ) --> lResult|NIL
*
* This function evaluate the content of <cCondition>.
* If the result is not a logical value or it generate an error,
* NIL is returned.
*
* <cCondition> The command to execute.
*
* <cName>      The procedure name to be used for error
*              documentation.
*
* <nLine>      Actual command line to be used for error
*              documentation.
*

    local bSaveErrorHandler
    local xResult

    *-------------------------------------------------------------
    * An empty condition is not possible.
    *-------------------------------------------------------------

    if cCondition == ""

        alertBox( EX_ST_ERROR_NO_CONDITION )

        return NIL                                      // RETURN

    end

    *-------------------------------------------------------------
    * Prepare for condition evaluation: handle the possible
    * error.
    *-------------------------------------------------------------

    bSaveErrorHandler :=;
        errorblock(;
            {|e|; 
                errorMacro(;
                    e,;
                    cName,;
                    nLine,;
                    cCondition;
                );
            };
        )

    begin sequence

        *---------------------------------------------------------
        * &
        *---------------------------------------------------------

        xResult := &(cCondition)

        if !(valtype( xResult ) == "L")

            alertBox( EX_ST_ERROR_NO_VALID_CONDITION )

        end

    recover
    
        *---------------------------------------------------------
        * NIL means ERROR.
        *---------------------------------------------------------

        xResult := NIL

    end

    errorblock(bSaveErrorHandler)

    if !(valtype( xResult ) == "L")

        *---------------------------------------------------------
        * If the result of the condition evaluation was not
        * logical, NIL is returned as this will be considered
        * an error.
        *---------------------------------------------------------

        xResult := NIL

    end

    *-------------------------------------------------------------
    * The condition result is returned.
    *-------------------------------------------------------------

    return xResult

*-----------------------------------------------------------------
static function exJumpIt( cCommands, nLine,;
                          bExit1, bExit2, bExit3 )
*
* --> NIL
*
* This function jumps to the statements whitch are not to be
* executed according with the control structure used.
*

    local cLine
    local nLevel := 1

    default( @bExit1,   { | cLine | cLine == "END" } )
    default( @bExit2,   bExit1 )
    default( @bExit3,   bExit2 )

    *-------------------------------------------------------------
    * Start a new sequence.
    *-------------------------------------------------------------

    begin sequence

        while .T.                                       // FOREVER

            *-----------------------------------------------------
            * First: the <nLine> pointer is incremented.
            *-----------------------------------------------------

            nLine++

            *-----------------------------------------------------
            * a new command line is extracted.
            *-----------------------------------------------------

            cLine := exCommandExtract( cCommands, @nLine )

            *-----------------------------------------------------
            * if <cLine> contains NIL, the macro file is
            * terminated.
            *-----------------------------------------------------

            if cLine == NIL

                *-------------------------------------------------
                * As we are looking for something, this file
                * termination is an ERROR.
                *-------------------------------------------------

                alertBox( EX_ST_ERROR_UNCLOSED_STRUCTURE )

                break                                   // BREAK

            end

            *-----------------------------------------------------
            * If still here, a line was obtained.
            *-----------------------------------------------------

            cLine := upper( cLine )

            if nLevel == 1

                *-------------------------------------------------
                * Inside this function, <nLevel> is related
                * to the search we are making.
                * If <nLevel> is 1, this is the right place to
                * test the block conditions.
                *-------------------------------------------------

                if  eval( bExit1, cLine ) .or.;
                    eval( bExit2, cLine ) .or.;
                    eval( bExit3, cLine )
                   
                    *---------------------------------------------
                    * One of the block conditons is true: this
                    * means that the right thing is found and
                    * the search terminates with <nLine> pointed
                    * on this line that validates one of the
                    * conditions.
                    *---------------------------------------------

                    break                               // BREAK

                end

            end

            do case
            case cLine == "END"

                *-------------------------------------------------
                * A END was found, the <nLevel> counter is
                * decremented with the hope that is goes not
                * under 1.
                *-------------------------------------------------

                nLevel--

                if nLevel == 0

                    *---------------------------------------------
                    * The search fails.
                    *---------------------------------------------

                    alertBox( EX_ST_ERROR_ALONE_END )

                    break                               // BREAK

                end

            case;
                cLine == "WHILE"                .or.; // (without)
                left( cLine, 6 ) == "WHILE "    .or.; // (with)
                cLine == "IF"                   .or.; // (without)
                left( cLine, 3 ) == "IF "       .or.; // (with)
                cLine == "BEGIN SEQUENCE"       .or.;
                cLine == "DO CASE"
                
                *-------------------------------------------------
                * (with)/(without) means with/without condition.
                *-------------------------------------------------
                
                *-------------------------------------------------
                * Now we are entering a new level that must be
                * closed before the exit block conditions are
                * tested.
                *-------------------------------------------------

                nLevel++

            end

        end

    end //sequence

    *-------------------------------------------------------------
    * Nothing is returned: <nLine> is called with reference so
    * this is the real returned value.
    * <nLine> is pointed on the line that validates at least
    * one of the code blocks.
    *-------------------------------------------------------------

    return NIL

*-----------------------------------------------------------------
static function exEndProc( cCommands, nLine )
*
* --> NIL
*
* This function jumps with speed to the end of procedure.
*

    local cLine

    *-------------------------------------------------------------
    * Start a new sequence.
    *-------------------------------------------------------------

    begin sequence

        while .T.                                       // FOREVER

            *-----------------------------------------------------
            * First: the <nLine> pointer is incremented.
            *-----------------------------------------------------

            nLine++

            *-----------------------------------------------------
            * Show the wait wheel as it takes time.
            *-----------------------------------------------------

            waitWheel()

            *-----------------------------------------------------
            * Extract the new line to be analised.
            *-----------------------------------------------------

            cLine := upper( memoline( cCommands, _MAX_STRING_LEN, nLine ) )

            do case
            case cLine == ""

                *-------------------------------------------------
                * End of file reached: this is an error as we
                * are looking for a ENDPROCEDURE
                *-------------------------------------------------

                alertBox(;
                    "( End of file );" +;
                    EX_ST_ERROR_UNCLOSED_PROCEDURE;
                )

                break                                   // BREAK

            case left( ltrim( cLine ), 12 ) == "ENDPROCEDURE"

                *-------------------------------------------------
                * End procedure reached.
                *-------------------------------------------------

                break                                   // BREAK

            case left( ltrim( cLine ), 9 ) == "PROCEDURE"

                *-------------------------------------------------
                * This should not happen: we are searchig a
                * ENDPROCEDURE and here starts a new procedure.
                *-------------------------------------------------

                alertBox( "(" + ltrim( str(nLine) ) + ");" +;
                    EX_ST_ERROR_UNCLOSED_PROCEDURE )

                break                                   // BREAK

            end
        end

    end //sequence

    *-------------------------------------------------------------
    * <nLine> is pointed on ENDPROCEDURE or beiond end of file.
    *-------------------------------------------------------------

    return NIL

*=================================================================
* GET()
*=================================================================
function get(;
        aGetList,;
        nRow, nCol,;
        bVar,;
        cGetPicture, cColorString,;
        bPreExpression, bValid;
    )
*
* get(
*   [@<aGetList>],;
*   [<nRow>], [<nCol>],
*   < { |x| iif( pcount() > 0, xVar := x, xVar ) } >,
*   [<cGetPicture>],
*   [<cColorString>],
*   [<bPreExpression>],
*   [<bValid>]
*   ) --> NIL
*
* like @...get
*
*
    memvar getlist

    local oGet
    local acColors
    local cGetColor

    default( @aGetList,         GetList )
    default( @nRow,             row() )
    default( @nCol,             col() )
    default( @cColorString,      setcolor() )
    default( @bPreExpression,    { || .T. } )
    default( @bValid,            { || .T. } )

    *-------------------------------------------------------------
    * There is a bug, so that unselected gets
    * appear at normal color... It should be
    * at unselected color...
    *-------------------------------------------------------------

    *-------------------------------------------------------------
    * Disassemble the color string.
    *-------------------------------------------------------------

    acColors := colorArray( cColorString )

    *-------------------------------------------------------------
    * Reassemble the corrected color string.
    *-------------------------------------------------------------

    if len( acColors ) >= 5

        *---------------------------------------------------------
        * The color string is Ok: it contains the unselected
        * color, so proceed.
        *---------------------------------------------------------

        cGetColor :=;
            acColors[5] + "," +;
            acColors[2] + "," +;
            acColors[3] + "," +;
            acColors[4] + "," +;
            acColors[1]

    else

        *---------------------------------------------------------
        * The color string is too short: don't change it.
        *---------------------------------------------------------
    
        cGetColor := cColorString

    end

    *-------------------------------------------------------------
    * Create the get object.
    *-------------------------------------------------------------

    oGet :=;
        getnew(;
            nRow, nCol, bVar, NIL, cGetPicture, cGetColor;
        )

    oGet:postBlock := bValid
    oGet:preBlock  := bPreExpression
    oGet:display()

    *-------------------------------------------------------------
    * Add the get object to the array of gets.
    *-------------------------------------------------------------

    aadd( aGetList, oGet )

    *-------------------------------------------------------------
    * Nothing is returned.
    *-------------------------------------------------------------

    return NIL

*=================================================================
* GVADD()
* GVDEFAULT()
* GVFILEDIR()
* GVFILEEXIST()
* GVFILEEXTENTION()
* GVSUBST()
*=================================================================
function gvAdd( cField, cAdd )
*
* gvAdd( <cField>, <cAdd> ) --> .T.
*
* <@cField>         the field to fill with more data.
* <cAdd>            is the string to be added to the content of 
*                   <cField>.
*
* This function is to be used inside GETs for pre/post validation,
* when a the content of a field should be added with more data.
*
* <cField> is returned with the same length as before to
* avoid troubles with current and future GETs.
*

    local nLen

    *-------------------------------------------------------------
    * Save the original length.
    *-------------------------------------------------------------

    nLen := len( cField )

    *-------------------------------------------------------------
    * Check <cField>, add and return.
    *-------------------------------------------------------------

    if empty( cField )

        cField := padr( cAdd, nLen )

    else

        cField := rtrim( cField ) + " " + cAdd

        cField := padr( cField, nLen )

    end

    return .T.

*=================================================================
function gvDefault( cField, cDefault )
*
* gvDefault( <cField>, <cDefault> ) --> .T.
*
* <@cField>         the field to check and if empty correct with
*                   <cDefault>.
* <cDefault>        is the default value to be used to replace
*                   <cField>.
*
* This function is to be used inside GETs for pre/post validation,
* when a field should have a default value.
*
* <cField> is returned with the same length as before to
* avoid troubles with current and future GETs.
*

    local nLen

    *-------------------------------------------------------------
    * Save the original length.
    *-------------------------------------------------------------

    nLen := len( cField )

    *-------------------------------------------------------------
    * Check <cField>; correct and return.
    *-------------------------------------------------------------

    if empty( cField )

        cField := padr( cDefault, nLen )

    end

    return .T.

*=================================================================
function gvFileDir( cWildName )
*
* gvFileDir( <cWildName> ) --> .T.
*
* <cWildName>       is the file name taken from the current
*                   get to be used for search with DIR().
*
* This function is to be used inside GETs for pre validation:
* the <cWildName> is a file name with wild cards that can be
* searched with the DIR() function after that a specific key
* is pressed.
*
* <cWildName> is returned with the same length as before to
* avoid troubles with current and future GETs.
*

    local cName
    local nLen

    *-------------------------------------------------------------
    * Save the original length.
    *-------------------------------------------------------------

    nLen := len( cWildName )

    *-------------------------------------------------------------
    * Start search.
    *-------------------------------------------------------------

    cName := dir( cWildName,,,,.T. )

    *-------------------------------------------------------------
    * Padd data and return.
    *-------------------------------------------------------------

    cWildName := padr( cName, nLen )

    return .T.

*=================================================================
function gvFileExist( cNameToTest, cExtention )
*
* gvFileExist( <cNameToTest>, [<cExtention>] ) --> lSuccess
*
* <@cNameToTest>    is the file name taken from the current
*                   get to test for existance.
* <cExtention>      is the normal extention of the file.
*
* This function is to be used inside GETs for post validation: the
* file name have to exist.
*
* <cNameToTest> is returned with the same length as before to
* avoid troubles with current and future GETs.
*

    local cName
    local nLen
    local lReturn := .F.

    default( @cExtention,   "" )

    *-------------------------------------------------------------
    * Save the original length.
    *-------------------------------------------------------------

    nLen := len( cNameToTest )

    *-------------------------------------------------------------
    * Prepare the name to search.
    *-------------------------------------------------------------

    cName := strAddExtention( cNameToTest, cExtention )

    *-------------------------------------------------------------
    * Save the modified name with the rigth dimention.
    * <cNameToText> is returned.
    *-------------------------------------------------------------

    cNameToTest := padr( cName, nLen )

    *-------------------------------------------------------------
    * Check if it exists and return the value.
    *-------------------------------------------------------------

    return isFile( cName )

*=================================================================
function gvFileExtention( cName, cExt )
*
* gvFileExtention( <cName>, <cExt> ) --> .T.
*
* <@cName>          the file name to be eventually corrected
*                   with file extention.
*
* <cExt>            the file extention to use as default.
*
* This function is to use inside GETs for pre/post validation,
* when the content of a field should contain a file name
* that should be corrected adding a default extention if not
* given from the user.
*
*

    local nLen

    *-------------------------------------------------------------
    * Save the original length.
    *-------------------------------------------------------------

    nLen := len( cName )

    *-------------------------------------------------------------
    * Correct with default extention.
    *-------------------------------------------------------------

    cName := strAddExtention( cName, cExt )

    *-------------------------------------------------------------
    * Restore original length.
    *-------------------------------------------------------------

    cName := padr( cName, nLen )

    return .T.
    
*=================================================================
function gvSubst( cField, cSubst )
*
* gvSubst( <cField>, <cSubst> ) --> .T.
*
* <@cField>         the field to be replaced with <cSubst>.
* <cSubst>          is the string to be used to replace the
*                   content of <cField>.
*
* This function is to use inside GETs for pre/post validation,
* when the content of a field should be replaced with other
* data.
*
* <cField> is returned with the same length as before to
* avoid troubles with current and future GETs.
*

    local nLen

    *-------------------------------------------------------------
    * Save the original length.
    *-------------------------------------------------------------

    nLen := len( cField )

    *-------------------------------------------------------------
    * Modify <cField> and return.
    *-------------------------------------------------------------

    cField := padr( cSubst, nLen )

    return .T.

*=================================================================
* HTF()
* HTFGENERATE()
* HTF2HTML()
*=================================================================

#define HTF_INDEX_START         "##"
#define HTF_INDEX_END           "##"
#define HTF_POINTER_START       "<"
#define HTF_POINTER_END         ">"

#define HTF_BUTTON_ESC_CANCEL;
    "[Esc] Cancel"
#define HTF_BUTTON_F1_HELP;
    "[F1] Help"
#define HTF_BUTTON_PGDN_CONFIRM;
    "[Pag] Confirm"

#define HTF_BROWSE_BUTTONS {;
    { maxrow()-1, nLeft+0, "[Esc] Exit",           {|| keyboard( chr(K_ESC) ) } },;
    { maxrow()-1, nLeft+11, "[]",                 {|| keyboard( chr(K_UP) ) } },;
    { maxrow()-1, nLeft+15, "[Pag]",              {|| keyboard( chr(K_PGUP) ) } },;
    { maxrow()-1, nLeft+22, "[Ctrl]+[Pag]",       {|| keyboard( chr(K_CTRL_PGUP) ) } },;
    { maxrow()-1, nLeft+36, "[-] Previous",       {|| keyboard( chr(K_LEFT) ) } },;
    { maxrow()-1, nLeft+51, "[Shift]+[F3] Search", {|| htfSearch(.F.), keyboard( chr(K_END) ) } },;
    { maxrow(), nLeft+0, "[F1] Help ",             {|| keyboard( chr(K_F1) ) } },;
    { maxrow(), nLeft+11, "[]",                   {|| keyboard( chr(K_DOWN) ) } },;
    { maxrow(), nLeft+15, "[Pag]",                {|| keyboard( chr(K_PGDN) ) } },;
    { maxrow(), nLeft+22, "[Ctrl]+[Pag]",         {|| keyboard( chr(K_CTRL_PGDN) ) } },;
    { maxrow(), nLeft+36, "[-] Next    ",         {|| keyboard( chr(K_RIGHT) ) } },;
    { maxrow(), nLeft+51, "[F3] Repeat Search ",   {|| htfSearch(.T.), keyboard( chr(K_END) ) } };
    }


#define HTF_ERROR_FILE_TYPE;
    "This is not a Help Text File."

#define HTF_ERROR_PATTERN_NOT_FOUND;
    "Pattern not found."

#define HTF_DIALOG_BOX_TOP_SEARCH;
    "Search for a pattern."

#define HTF_PROMPT_SEARCH_PATTERN;
    "Insert the pattern to search:"

#define HTF_WAIT_SEARCHING;
    "Searching for "

#define HTF_HELP;
    "htf()"+;
    NL(2) +;
    "Help Text File browser" +;
    NL(3) +;
    "Browse the document using the following keys." +;
    NL(2) +;
    "[Esc]          Exit." + NL(1) +;
    "[]            Move cursor up." + NL(1) +;
    "[]            Move cursor down." + NL(1) +;
    "[Page]        Move cursor PageUp." + NL(1) +;
    "[Page]        Move cursor Pagedown." + NL(1) +;
    "[Ctrl]+[Page] Move cursor Top." + NL(1) +;
    "[Ctrl]+[Page] Move cursor Bottom." + NL(1) +;
    "[Enter]        Select reference." + NL(1) +;
    "[<-]           Previous selected reference." + NL(1) +;
    "[->]           Next selected reference." + NL(1) +;
    "[Shift]+[F3]   Search for a new pattern." + NL(1) +;
    "[F3]           Repeat previous search."

#define HTF_HELP_PATTERN_SEARCH;
    "htf()" +;
    NL(2) +;
    "Help Text File pattern search." +;
    NL(3) +;
    "Insert the pattern to search starting from the current " +;
    "position."

#define HTF_WAIT_APPENDING;
    "Loading "

#define HTF_WAIT_SEARCHING_INDEXES;
    "Searching for indexes..."

#define HTF_WAIT_SEARCHING_POINTERS;
    "Searching for pointers..."

#define HTF_WAIT_TRANSLATING_HTML;
    "Translating into HTML..."

#define HTF_WAIT_COPYING;
    "Saving "

*=================================================================
function htf( xInitial, cHeader )
*
* htf( [<nInitialRecord>] | [<cInitialPattern>], [<cHeader>])
*     --> NIL
*
* <nInitialRecord>   The starting record to reach befor showing
*                    the text.
*
* <cInitialPattern>  The starting pattern to search.
*
* Uses the active Alias as an hypertext help file.
*

    local aButtons
    local bOldErrorHandler
    local cOldScreen          := mouseScrSave()
    local cOldColor           := setcolor()
    local nOldCursor          := setcursor()
    local nOldRow             := row()
    local nOldCol             := col()
    local bOld_F1             :=;
        setkey( K_F1, { || Text( HTF_HELP ) } )
    local bOld_F3             := setkey( K_F3 )
    local bOld_SH_F3          := setkey( K_SH_F3 )
    local oBrowse
    local oColumn
    local nKey
    local lTerminate := .f.

    local nTop                := 0 +1
    local nLeft               := (maxcol()+1-80) /2
    local nBottom             := maxrow() -2
    local nRight              := nLeft+80-1

    local anPointer           := array(254)
    local nI                  := 0
    local nMax                := 0

    default( @cHeader,      "" )

    *-------------------------------------------------------------
    * Check for a valid HTF file.
    *-------------------------------------------------------------

    if  fieldname(1) == "TEXT"          .and.;
        fieldname(2) == "POINTER"       .and.;
        valtype( field->Text ) == "C"   .and.;
        valtype( field->Pointer ) == "N"
        
        *---------------------------------------------------------
        * The active Alias seems to be a valid HTF file.
        *---------------------------------------------------------

    else

        *---------------------------------------------------------
        * The active Alias is not an HTF file.
        *---------------------------------------------------------

        alertBox( HTF_ERROR_FILE_TYPE )

        return NIL                                      // RETURN

    end

    *-------------------------------------------------------------
    * Check the initial record.
    *-------------------------------------------------------------

    do case
    case valtype( xInitial ) == "N"

        *---------------------------------------------------------
        * Ok, go to <xInitial>.
        *---------------------------------------------------------

        if  xInitial < 0            .or.;
            xInitial > reccount()

            *-----------------------------------------------------
            * This is not a valid record: go to the top.
            *-----------------------------------------------------

            dbgotop()

        else

            *-----------------------------------------------------
            * Reach the record
            *-----------------------------------------------------

            dbgoto( xInitial )

        end

    case valtype( xInitial ) == "C"

        *---------------------------------------------------------
        * Ok, search for the string <xInitial>
        *---------------------------------------------------------

        dbgotop()

        waitFor(;
            HTF_WAIT_SEARCHING +;
            upper( alltrim( xInitial ) );
        )

        while !eof()                                    // WHILE

            if  upper( alltrim( xInitial ) )    $;
                upper( Field->Text )

                 exit                                   // EXIT

            end

            *-----------------------------------------------------
            * Skip to next record before next loop.
            *-----------------------------------------------------

            dbskip()

        end

        waitFor()

        if eof()

            alertBox(;
                upper( alltrim( xInitial ) ) +;
                NL(1) +;
                HTF_ERROR_PATTERN_NOT_FOUND;
            )

            dbgotop()

        end

    end

    *-------------------------------------------------------------
    * If eof() or bof() it is better to move the record pointer.
    *-------------------------------------------------------------

    if eof()

        dbgobottom()

    end

    if bof()

        dbgotop()

    end

    *-------------------------------------------------------------
    * Save old data.
    *-------------------------------------------------------------

    nOldCursor          := setcursor()
    nOldRow             := row()
    nOldCol             := col()

    *-------------------------------------------------------------
    * Start a new sequence with a different errorhandler.
    *-------------------------------------------------------------

    bOldErrorHandler    := errorblock( {|e| ErrorHandler(e)} )

    begin sequence

        aButtons := HTF_BROWSE_BUTTONS

        setcolor( COLOR_HEAD )
        scroll( 0, 0, 1, maxcol() )
        scroll( maxrow(), 0, maxrow(), maxcol() )
        setpos( 0, 0 )
        dispout( padc( cHeader, maxcol()+1 ) )
        setcolor( COLOR_BODY )
        scroll( 1, 0, maxrow(), maxcol() )

        *---------------------------------------------------------
        * Show buttons.
        *---------------------------------------------------------

        for nI := 1 to len( aButtons )
            say( aButtons[nI][1], aButtons[nI][2], aButtons[nI][3],, COLOR_BUTTON )
        next

        *---------------------------------------------------------
        * Save the starting position.
        *---------------------------------------------------------

        nI              := 1
        nMax            := 1
        anPointer[nI]   := recno()

        *---------------------------------------------------------
        * Now start the HTF loop. <nI> is used as index of
        * previous/next positions.
        *---------------------------------------------------------

        while .t.                                       // FOREVER

            *-----------------------------------------------------
            * Make a Tbrowse object for the current workarea.
            *-----------------------------------------------------

            oBrowse := tbrowsedb(nTop, nLeft, nBottom, nRight)
            oColumn :=;
                 tbColumnNew( "", fieldwblock( "TEXT", select() ) )
            oColumn:picture := "@s80"
            oBrowse:addColumn( oColumn )

            *-----------------------------------------------------
            * Use a custom skipper.
            *-----------------------------------------------------

            oBrowse:skipBlock :=;
                { | nSkip | htfSkip( nSkip, oBrowse ) }

            *-----------------------------------------------------
            * Turn the cursor off while browsing.
            *-----------------------------------------------------

            setcursor(SETCURSOR_NONE)

            *-----------------------------------------------------
            * Main TBrowse loop.
            *-----------------------------------------------------

            while .t.                                   // FOREVER

                *-------------------------------------------------
                * Stabilize the display until it's stable
                * or a key is pressed.
                *-------------------------------------------------

                oBrowse:forceStable()

                *-------------------------------------------------
                * Show the mouse.
                *-------------------------------------------------

                setMouse( .T. )

                *-------------------------------------------------
                * Was a mouse button?
                *-------------------------------------------------

                if !( mouse() == NIL )

                    *---------------------------------------------
                    * Read mouse and transform into keyboard.
                    *---------------------------------------------

                    htfMouseKeyboard( aButtons )

                    *---------------------------------------------
                    * Terminate mouse.
                    *---------------------------------------------

                    mouse( .T. )

                end

                *-------------------------------------------------
                * Read the keyboard without waiting.
                *-------------------------------------------------

                nKey := inkey()

                *-------------------------------------------------
                * If a key was pressed, hide the mouse.
                *-------------------------------------------------

                if !(nKey == 0)

                    setMouse( .F. )

                end

                *-------------------------------------------------
                * Analise the keyboard.
                *-------------------------------------------------

                do case
                case ( nKey == K_ESC )

                    *---------------------------------------------
                    * Esc means leave.
                    *---------------------------------------------

                    lTerminate := .t.

                    exit                                // EXIT

                case nKey == K_END

                    *---------------------------------------------
                    * It is a dummy key used to exit:
                    * when the mouse do an action and
                    * a TBrowse redraw is needed.
                    *---------------------------------------------

                    exit                                // EXIT

                case nKey == K_DOWN

                    oBrowse:down()

                case nKey == K_PGDN

                    oBrowse:pageDown()

                case nKey == K_CTRL_PGDN

                    oBrowse:goBottom()

                case nKey == K_UP

                    oBrowse:up()

                case nKey == K_PGUP

                    oBrowse:pageUp()

                case nKey == K_CTRL_PGUP

                    oBrowse:goTop()

                case nKey == K_SH_F3

                    *---------------------------------------------
                    * Search a new pattern.
                    *---------------------------------------------

                    htfSearch(.F.)

                    exit                                // EXIT

                case nKey == K_F3

                    *---------------------------------------------
                    * Search again old pattern.
                    *---------------------------------------------

                    htfSearch(.T.)

                    exit                                // EXIT

                case nKey == K_RETURN

                    *---------------------------------------------
                    * If the selected line contains a pointer:
                    * reach the pointed index.
                    *---------------------------------------------

                    if htfGo( @anPointer, @nI, @nMax )

                        exit                            // EXIT

                    end

                case nKey == K_LEFT

                    *---------------------------------------------
                    * Previous.
                    *---------------------------------------------

                    if htfPrevious( @anPointer, @nI, @nMax )

                        exit                            // EXIT

                    end

                case nKey == K_RIGHT

                    *---------------------------------------------
                    * Next.
                    *---------------------------------------------

                    if htfNext( @anPointer, @nI, @nMax )

                        exit                            // EXIT

                    end

                case nKey == K_F1

                    Text( HTF_HELP )

                end

            end

            *-----------------------------------------------------
            * <lTerminate> is True if the HTF loop must be
            * terminated.
            *-----------------------------------------------------

            if lTerminate

                exit                                    // EXIT

            end

        end

    recover

        *---------------------------------------------------------
        * No action for error recovery.
        *---------------------------------------------------------

    end //sequence

    errorblock(bOldErrorHandler)

    *-------------------------------------------------------------
    * Restore old values.
    *-------------------------------------------------------------

    mouseScrRestore( NIL, NIL, NIL, NIL, cOldScreen )
    setcolor( cOldColor )
    setcursor(nOldCursor)
    setpos( nOldRow, nOldCol )
    setkey( K_F1, bOld_F1 )
    setkey( K_F3, bOld_F3 )
    setkey( K_SH_F1, bOld_SH_F3 )

    *-------------------------------------------------------------
    * Nothing is returned.
    *-------------------------------------------------------------

    return NIL

*-----------------------------------------------------------------
static function htfSkip(nSkip, oBrowse)
*
* htfSkip(nSkip, oBrowse) --> nSkippedRecords
*
* Handle record movement requests from the Tbrowse object.
*

    local nI            := 0

    do case
    case ( nSkip == 0 .or. lastrec() == 0 )

        *---------------------------------------------------------
        * Skip 0 (significant on a network).
        *---------------------------------------------------------

        dbSkip(0)

    case ( nSkip > 0 .and. !eof() )

        *---------------------------------------------------------
        * Skip Forward.
        *---------------------------------------------------------

        for nI := 1 to nSkip

            dbskip( 1 )

            if eof()

                *-------------------------------------------------
                * Don't let the cursor go to eof.
                *-------------------------------------------------

                dbgobottom()

                *-------------------------------------------------
                * Correct <nI> to the real skip.
                *-------------------------------------------------

                nI--

                *-------------------------------------------------
                * Exit loop.
                *-------------------------------------------------

                exit                                    // EXIT

            end

        next

    case ( nSkip < 0 )

        *---------------------------------------------------------
        * Skip backward.
        *---------------------------------------------------------

        for nI := -1 to nSkip step -1

            dbskip(-1)

            if bof()

                *-------------------------------------------------
                * Don't let the cursor go to bof.
                *-------------------------------------------------

                dbgotop()

                *-------------------------------------------------
                * Correct <nI> to the real skip.
                *-------------------------------------------------

                nI++

                *-------------------------------------------------
                * Exit loop.
                *-------------------------------------------------

                exit                                    // EXIT

            end

        next

    end

    *-------------------------------------------------------------
    * Return the skip amount
    *-------------------------------------------------------------

    return nI

*-----------------------------------------------------------------
static function htfGo( anPointer, nI, nMax )

    local nPointer := Field->Pointer

    if nPointer > 0

        *---------------------------------------------------------
        * Go to the selected record.
        *---------------------------------------------------------

        dbgoto( nPointer )

        *---------------------------------------------------------
        * Add the new position inside the Pointer array.
        *---------------------------------------------------------

        nI++
        nMax := nI
        anPointer[nI] := nPointer

        *---------------------------------------------------------
        * A valid go to record was done.
        *---------------------------------------------------------

        return .t.

    end

    return .f.

*-----------------------------------------------------------------
static function htfPrevious( anPointer, nI, nMax )

    if nI < 2

        *---------------------------------------------------------
        * can't go back.
        *---------------------------------------------------------

    else

        nI--
        dbgoto( anPointer[nI] )

        *---------------------------------------------------------
        * A valid go to record was done.
        *---------------------------------------------------------

        return .t.

    end

    return .f.

*-----------------------------------------------------------------
static function htfNext( anPointer, nI, nMax )

    if nI < nMax

        nI++

        dbgoto( anPointer[nI] )

        *---------------------------------------------------------
        * A valid go to record was done.
        *---------------------------------------------------------

        return .t.

    else

        *---------------------------------------------------------
        * can't go next.
        *---------------------------------------------------------

    end

    return .f.

*-----------------------------------------------------------------
static function htfSearch( lRepeatSearch )
*
* Search for a text pattern.
*

    local aoGet             := {}
    local aButton           := {}
    local aTab              := TAB_DEFAULT
    local bOldErrorHandler
    local cOldScreen
    local cOldColor         := setcolor()
    local nOldCursor        := setcursor( SETCURSOR_NORMAL )
    local bOld_F1           :=;
        setkey( K_F1, { || Text( HTF_HELP_PATTERN_SEARCH ) } )
    local bOld_F2           := setkey( K_F2, NIL )
    local nOldRow           := row()
    local nOldCol           := col()
    local nTop
    local nLeft
    local nBottom
    local nRight
    local nWidth

    local lGoOn             := .F.

    local nStartRecord      := recno()

    static cPattern

    default( @lRepeatSearch,    .F. )
    default( @cPattern,         space(_MAX_STRING_LEN) )

    if empty( cPattern )
        lRepeatSearch := .F.
    end

    *-------------------------------------------------------------
    * Start a new sequence.
    *-------------------------------------------------------------

    begin sequence

        *---------------------------------------------------------
        * If it is a new search, a dialog box must be shown.
        *---------------------------------------------------------

        if !lRepeatSearch

            *-----------------------------------------------------
            * Create a kind of window.
            *-----------------------------------------------------

            nBottom             := maxrow()
            nTop                := nBottom - 6
            nLeft               := 0
            nRight              := maxcol()
            nWidth              := nRight - nLeft +1

            aTab[TAB_LEFT]      := nLeft
            aTab[TAB_RIGHT]     := nRight
            aTab[TAB_FIELD_LEN] := 10
            aTab[TAB_TAB_ARRAY] := { 8 }

            cOldScreen          :=;
                mouseScrSave( nTop, nLeft, nBottom, nRight )

            setcolor( COLOR_BODY )

            scroll( nTop, nLeft, nBottom, nRight )

            dispBoxShadow(;
                nTop, nLeft, nBottom, nRight,;
                1,;
                dispBoxColor( 1 ),;
                dispBoxColor( 2 );
            )

            say(;
                nTop+1, nLeft+1,;
                padc( HTF_DIALOG_BOX_TOP_SEARCH, nWidth-2 ),,;
                COLOR_HEAD;
            )

            *-----------------------------------------------------
            * Get/Read Loop.
            *-----------------------------------------------------

            while .T.                                   // FOREVER

                say(;
                    nTop+3, nLeft+1,;
                    HTF_PROMPT_SEARCH_PATTERN,,;
                    COLOR_BODY;
                )
                tab( aTab )
                get(;
                    @aoGet, row(), col()+1,;
                    { |x| iif( pcount() > 0, cPattern := x, cPattern ) },;
                    picChrMax( col()+1, nRight-1 ),;
                    COLOR_BODY,;
                    { || trueSetkey( K_F2, {||NIL} ) },;
                    { || cPattern <> space(_MAX_STRING_LEN) };
                )

                button( @aButton, row()+2, 01,;
                    HTF_BUTTON_ESC_CANCEL, , {||keyboard(chr(K_ESC))} )
                button( @aButton, row(), col()+1,;
                    HTF_BUTTON_F1_HELP, , {|| eval(setkey(K_F1))} )
                button( @aButton, row(), col()+1,;
                    HTF_BUTTON_PGDN_CONFIRM, , {||keyboard(chr(K_PGDN))} )

                *-------------------------------------------------
                * Read.
                *-------------------------------------------------

                read( aoGet, 1, aButton )
                aoGet    := {}
                aButton := {}

                do case
                case lastkey() = K_ESC

                    *---------------------------------------------
                    * Exit.
                    *---------------------------------------------

                    lGoOn := .F.

                    exit                                // EXIT

                case lastkey() = K_PGDN

                    *---------------------------------------------
                    * Confirm: check for correct data.
                    *---------------------------------------------

                    if !empty( cPattern )

                        *-----------------------------------------
                        * Data is correct: do search.
                        *-----------------------------------------

                        lGoOn := .T.

                        exit                            // EXIT

                    else

                        *-----------------------------------------
                        * Loop again.
                        *-----------------------------------------

                    end

                otherwise

                    *---------------------------------------------
                    * No other exit key is allowed: Loop again.
                    *---------------------------------------------

                end

            end

            *-----------------------------------------------------
            * Close window.
            *-----------------------------------------------------

            mouseScrRestore( nTop, nLeft, nBottom, nRight, cOldScreen )

        end

        if  lGoOn           .or.;
            lRepeatSearch

            *-----------------------------------------------------
            * A new search pattern was selected <lGoOn>, or
            * a repet search <lRepeatSearch> was selected before.
            *-----------------------------------------------------

            *-----------------------------------------------------
            * Isolate the error hanler with a new sequence.
            *-----------------------------------------------------

            bOldErrorHandler := errorblock( {|e| ErrorHandler(e)} )

            begin sequence

                WaitFor( HTF_WAIT_SEARCHING +;
                    upper( alltrim( cPattern ) ) )

                dbskip()

                *-------------------------------------------------
                * Search loop.
                *-------------------------------------------------

                while !eof()

                    if  upper( alltrim( cPattern ) )    $;
                        upper( Field->Text )

                        *-----------------------------------------
                        * If the pattern is found, exit.
                        *-----------------------------------------

                        exit                            // EXIT

                    end

                    if inkey() == K_ESC

                        *-----------------------------------------
                        * If [Esc] is pressed, stop search
                        * and restore the starting record
                        * position.
                        *-----------------------------------------

                        dbgoto( nStartRecord )

                        exit                            // EXIT

                    end

                    *---------------------------------------------
                    * Skip to next record before next loop.
                    *---------------------------------------------

                    dbskip()

                end

                *-------------------------------------------------
                * If the record pointer si over eof, the pattern
                * was not/no more found.
                *-------------------------------------------------

                if eof()

                    alertBox( upper( alltrim( cPattern ) ) +;
                        NL(1) + HTF_ERROR_PATTERN_NOT_FOUND )

                    *---------------------------------------------
                    * The starting record position is restored.
                    *---------------------------------------------

                    dbgoto( nStartRecord )

                end

            recover

                *-------------------------------------------------
                * If an error occurred, the original record
                * postion is restored.
                *-------------------------------------------------

                dbgoto( nStartRecord )

            end //sequence
            waitfor()
            errorblock(bOldErrorHandler)

        end

    end //sequence

    *-------------------------------------------------------------
    * Restore old values.
    *-------------------------------------------------------------

    setcursor( nOldCursor )
    setcolor( cOldColor )
    setpos( nOldRow, nOldCol )
    setkey( K_F1, bOld_F1 )
    setkey( K_F2, bOld_F2 )

    *-------------------------------------------------------------
    * No value is returned.
    *-------------------------------------------------------------

    return NIL

*-----------------------------------------------------------------
static function htfMouseKeyboard( aButtons )
*

    local aMouse    := mouse()

    local nCol      := aMouse[1]-1
    local nRow      := aMouse[2]-1

    local nTimes    := 0
    local cKeyboard := ""
    local nI

    local nButtRow
    local nButtCol
    local nButtColEnd

    *-------------------------------------------------------------
    * If buttons are passed, test for buttons.
    *-------------------------------------------------------------

    if valtype( aButtons ) == "A"

        for nI := 1 to len(aButtons)

            nButtRow    := aButtons[nI][1]
            nButtCol    := aButtons[nI][2]
            nButtColEnd := nButtCol + len( aButtons[nI][3] ) -1

            if  nButtRow == nRow        .and.;
                nButtCol <= nCol        .and.;
                nButtColEnd >= nCol

                *-------------------------------------------------
                * OK button pressed. Do the action.
                *-------------------------------------------------

                eval( aButtons[nI][4] )

                return NIL                              // RETURN

            end

        next

    end

    *-------------------------------------------------------------
    * Test if mouse is outside the browse area.
    *-------------------------------------------------------------

    if nRow > 1 .and. nRow < maxrow()-2

        *---------------------------------------------------------
        * Ok.
        *---------------------------------------------------------

    else

        *---------------------------------------------------------
        * The mouse click was outside the area and it is not
        * considered.
        *---------------------------------------------------------

        return NIL                                      // RETURN

    end

    *-------------------------------------------------------------
    * Transform the mouse click into a keyboard movement.
    *-------------------------------------------------------------

    do case
    case nRow > row()

        nTimes := nRow - row()

        for nI := 1 to nTimes

            cKeyboard += chr( K_DOWN )

        end

        cKeyboard += chr( K_ENTER )

        keyboard( cKeyboard )

    case nRow < row()

        nTimes := row() - nRow

        for nI := 1 to nTimes

            cKeyboard += chr( K_UP )

        end

        cKeyboard += chr( K_ENTER )

        keyboard( cKeyboard )

    case nRow == row()

        cKeyboard += chr( K_ENTER )

        keyboard( cKeyboard )

    end

    return NIL

*=================================================================
function htfGenerate( cHtfText, cDestName,;
    cIndexStart, cIndexEnd,;
    cPointerStart, cPointerEnd )
*
* htfGenerate( <cHtfText>, [<cDestName>],
*     [<cIndexStart>], [<cIndexEnd>],
*     [<cPointerStart>], [<cPointerEnd>] ) --> NIL
*
* <cHtfText>      the filename used as input.
* <cDestName>     the filename used as output.
* <cIndexStart>   the symbol used as index start.
* <cIndexEnd>     the symbol used as index end.
* <cPointerStart> the symbol used as pointer start.
* <cPointerEnd>   the symbol used as pointer end.
*
* This function generates a .dbf file to use as a kind of
* hypertext.
*

    local bOldErrorHandler
    local nOldSelect          := select()
    local cOldRdd             := rddsetdefault( _DEFAULT_RDD )

    local nIStart             := 0
    local nIEnd               := 0
    local nIStartLen          := 0
    local nIendLen            := 0
    local nPStart             := 0
    local nPEnd               := 0
    local nPStartLen          := 0
    local nPEndLen            := 0
    local cIndex              := ""
    local cPointer            := ""
    local axIndex             := {}
    local aStruct             := {}

    default( @cIndexStart,    HTF_INDEX_START )
    default( @cIndexEnd,      HTF_INDEX_END )
    default( @cPointerStart,  HTF_POINTER_START )
    default( @cPointerEnd,    HTF_POINTER_END )

    *-------------------------------------------------------------
    * Start a new sequence.
    *-------------------------------------------------------------

    begin sequence

        *---------------------------------------------------------
        * If no <cHtfText> was given, stop.
        *---------------------------------------------------------

        if cHtfText == NIL

            break                                       // BREAK

        end

        *---------------------------------------------------------
        * Check and correct <cDestName>.
        *---------------------------------------------------------

        if valtype( cDestName ) == "C"

            cDestName :=;
                strAddExtention( cDestName, _EXTENTION_HTF )

        else

            cDestName :=;
                strAddExtention(;
                    strCutExtention( cHtfText ),;
                    _EXTENTION_HTF;
                )

        end

        *---------------------------------------------------------
        * Check if the cHtfText exists.
        *---------------------------------------------------------

        if file( cHtfText )

            *-----------------------------------------------------
            * Ok.
            *-----------------------------------------------------

        else

            *-----------------------------------------------------
            * The file don't exists: stop.
            *-----------------------------------------------------

            alertBox(;
                cHtfText +;
                NL(1) +;
                _ERROR_NO_FILE;
            )

            break                                       // BREAK

        end


        *---------------------------------------------------------
        * Check if the destination file already exists.
        *---------------------------------------------------------

        if file( cDestName )

            if  alertBox(;
                    cDestName + NL(1) +;
                    _ERROR_PROMPT_ALREADY_EXIST_OVERWRITE,;
                    { _MENU_NO, _MENU_YES };
                ) == 1

                break                                   // BREAK

            end

        end

        *---------------------------------------------------------
        * Calculate the index and pointer length.
        *---------------------------------------------------------

        nIStartLen := len( cIndexStart )
        nIEndLen   := len( cIndexEnd )
        nPStartLen := len( cPointerStart )
        nPEndLen   := len( cPointerEnd )

        *---------------------------------------------------------
        * Create the destination file.
        *---------------------------------------------------------

        aadd( aStruct, { "TEXT", "C", 80, 0 } )
        aadd( aStruct, { "POINTER", "N", 8, 0 } )

        dbcreate( cDestName, aStruct )
        dbusearea( .T., NIL, cDestName,;
            "HlpTxtFile" )

        *---------------------------------------------------------
        * Data transfer.
        *---------------------------------------------------------

        waitFor( HTF_WAIT_APPENDING + alltrim( cHtfText ) )

        dbSdf( .F., cHtfText )

        *---------------------------------------------------------
        * Find "indexes".
        *---------------------------------------------------------

        waitFor( HTF_WAIT_SEARCHING_INDEXES )

        HlpTxtFile->(dbgotop())

        while !eof()

            *-----------------------------------------------------
            * There may be only one index per line.
            *-----------------------------------------------------

            HlpTxtFile->(waitFileEval())

            *-----------------------------------------------------
            * Locate the presence of an "index"
            *-----------------------------------------------------

            nIStart := at( cIndexStart, HlpTxtFile->Text )

            if nIStart > 0

                nIStart += nIStartLen

                nIEnd :=;
                    at(;
                        cIndexEnd,;
                        substr( HlpTxtFile->Text, nIStart );
                    )

                if  nIEnd > 0

                    cIndex :=;
                        substr(;
                            HlpTxtFile->Text,;
                            nIStart,;
                            nIEnd-1;
                        )

                    *---------------------------------------------
                    * An "index" was found: add it to the index
                    * array.
                    *---------------------------------------------

                    cIndex := upper( alltrim( cIndex ) )

                    aadd( axIndex, { cIndex, recno() } )

                end

            end

            *-----------------------------------------------------
            * Prepare to scan next record.
            *-----------------------------------------------------

            HlpTxtFile->(dbskip())

        end

        *---------------------------------------------------------
        * Close Wait bar.
        *---------------------------------------------------------

        waitFileEval( .T. )

        *---------------------------------------------------------
        * Find "pointers".
        *---------------------------------------------------------

        waitFor( HTF_WAIT_SEARCHING_POINTERS )

        HlpTxtFile->(dbgotop())

        while !eof()

            *-----------------------------------------------------
            * There may be only one pointer per line.
            *-----------------------------------------------------

            HlpTxtFile->(waitFileEval())

            *-----------------------------------------------------
            * Locate the presence of a "pointer"
            *-----------------------------------------------------

            nPStart := at( cPointerStart, HlpTxtFile->Text )

            if nPStart > 0

                nPStart += nPStartLen

                nPEnd :=;
                    at(;
                        cPointerEnd,;
                        substr( HlpTxtFile->Text, nPStart );
                    )

                if  nPEnd > 0

                    cPointer :=;
                        substr(;
                            HlpTxtFile->Text,;
                            nPStart, nPEnd-1;
                        )

                    cPointer := upper( alltrim( cPointer ) )

                    *---------------------------------------------
                    * An "pointer" was found: search the "index"
                    * array to find the position.
                    *---------------------------------------------

                    aeval(;
                        axIndex,;
                        { |aValue, nIndex|;
                            iif( aValue[1] == cPointer,;
                            HlpTxtFile->Pointer := aValue[2], NIL);
                        };
                    )

                end

            end

            HlpTxtFile->(dbskip())

        end

        *---------------------------------------------------------
        * Close Wait bar and wait for.
        *---------------------------------------------------------

        waitFor()

        waitFileEval( .T. )

        *---------------------------------------------------------
        * Close file.
        *---------------------------------------------------------

        HlpTxtFile->(dbclosearea())

    end //sequence

    *-------------------------------------------------------------
    * Restore old data.
    *-------------------------------------------------------------

    rddsetdefault( cOldRdd )
    dbselectarea( nOldSelect )

    return NIL

*=================================================================
function htf2Html(;
        cHtfText, cDestName,;
        cIndexStart, cIndexEnd,;
        cPointerStart, cPointerEnd,;
        cTitle;
    )
*
* htf2Html( <cHtfText>, [<cDestName>],
*     [<cIndexStart>], [<cIndexEnd>],
*     [<cPointerStart>], [<cPointerEnd>],
*     [<cTitle>] ) --> NIL
*
* <cHtfText>      the filename used as input.
* <cDestName>     the filename used as output.
* <cIndexStart>   the symbol used as index start.
* <cIndexEnd>     the symbol used as index end.
* <cPointerStart> the symbol used as pointer start.
* <cPointerEnd>   the symbol used as pointer end.
* <cTitle>        the HTML page title.
*
* This function generates a .htm file from a ASCII text file
* destinated to HTF.
*

    local bOldErrorHandler
    local nOldSelect            := select()
    local cOldRdd               := rddsetdefault( _DEFAULT_RDD )
    local aOldOutput

    local nI
    local nIStart               := 0
    local nIEnd                 := 0
    local nIStartLen            := 0
    local nIendLen              := 0
    local nPStart               := 0
    local nPEnd                 := 0
    local nPStartLen            := 0
    local nPEndLen              := 0
    local cIndex                := ""
    local cPointer              := ""
    local cPointerNormal
    local axIndex               := {}
    local aStruct               := {}

    local cTempPath             := strTempPath()

    default( @cIndexStart,    HTF_INDEX_START )
    default( @cIndexEnd,      HTF_INDEX_END )
    default( @cPointerStart,  HTF_POINTER_START )
    default( @cPointerEnd,    HTF_POINTER_END )
    default( @cTitle,         "No Title" )

    *-------------------------------------------------------------
    * Start a new sequence.
    *-------------------------------------------------------------

    begin sequence

        *---------------------------------------------------------
        * If no <cHtfText> was given, stop.
        *---------------------------------------------------------

        if cHtfText == NIL

            break                                       // BREAK

        end

        *---------------------------------------------------------
        * Check and correct <cDestName>.
        *---------------------------------------------------------

        if valtype( cDestName ) == "C"

            cDestName :=;
                strAddExtention( cDestName, _EXTENTION_HTM )

        else

            cDestName :=;
                strAddExtention(;
                    strCutExtention( cHtfText ),;
                    _EXTENTION_HTM;
                )

        end

        *---------------------------------------------------------
        * Check if the cHtfText exists.
        *---------------------------------------------------------

        if file( cHtfText )

            *-----------------------------------------------------
            * Ok.
            *-----------------------------------------------------

        else

            *-----------------------------------------------------
            * The file don't exists: stop.
            *-----------------------------------------------------

            alertBox(;
                cHtfText +;
                NL(1) +;
                _ERROR_NO_FILE;
            )

            break                                       // BREAK

        end


        *---------------------------------------------------------
        * Check if the destination file already exists.
        *---------------------------------------------------------

        if file( cDestName )

            if  alertBox(;
                    cDestName + NL(1) +;
                    _ERROR_PROMPT_ALREADY_EXIST_OVERWRITE,;
                    { _MENU_NO, _MENU_YES };
                ) == 1

                break                                   // BREAK

            else

                *-------------------------------------------------
                * As the file will be written using QOUT()
                * that appends data, the file must be deleted
                * before.
                *-------------------------------------------------

                ferase( cDestName )

            end

        end

        *---------------------------------------------------------
        * Translate Indexes and Pointes into HTML standard
        * (if necessary).
        *---------------------------------------------------------
        
        cIndexStart     := htmlTranslate( cIndexStart )
        cIndexEnd       := htmlTranslate( cIndexEnd )
        cPointerStart   := htmlTranslate( cPointerStart )
        cPointerEnd     := htmlTranslate( cPointerEnd )

        *---------------------------------------------------------
        * Calculate the index and pointer length.
        *---------------------------------------------------------

        nIStartLen := len( cIndexStart )
        nIEndLen   := len( cIndexEnd )
        nPStartLen := len( cPointerStart )
        nPEndLen   := len( cPointerEnd )

        *---------------------------------------------------------
        * Create a temporary file.
        *---------------------------------------------------------

        aadd( aStruct, { "TEXT", "C", 160, 0 } )

        dbcreate( cTempPath + "\TEMPHTML.DBF", aStruct )
        dbusearea( .T., NIL, cTempPath + "\TEMPHTML.DBF",;
            "TEMPHTML" )

        *---------------------------------------------------------
        * Prepare 4 records at top.
        *---------------------------------------------------------
        
        dbappend()
        dbappend()
        dbappend()
        dbappend()

        *---------------------------------------------------------
        * Data transfer.
        *---------------------------------------------------------

        waitFor( HTF_WAIT_APPENDING + alltrim( cHtfText ) )

        dbSdf( .F., cHtfText )

        *---------------------------------------------------------
        * Translate text into HTML standard.
        *---------------------------------------------------------

        waitFor( HTF_WAIT_TRANSLATING_HTML )

        TEMPHTML->(dbgotop())

        dbeval(;
            { ||;
                TEMPHTML->Text := htmlTranslate( TEMPHTML->Text ),;
                TEMPHTML->(waitFileEval());
            };
        )

        waitFileEval( .T. )

        *---------------------------------------------------------
        * Find "indexes".
        *---------------------------------------------------------

        waitFor( HTF_WAIT_SEARCHING_INDEXES )

        TEMPHTML->(dbgotop())

        while !eof()

            *-----------------------------------------------------
            * There may be only one index per line.
            *-----------------------------------------------------

            TEMPHTML->(waitFileEval())

            *-----------------------------------------------------
            * Locate the presence of an "index"
            *-----------------------------------------------------

            nIStart := at( cIndexStart, TEMPHTML->Text )

            if nIStart > 0

                nIStart += nIStartLen

                nIEnd :=;
                    at(;
                        cIndexEnd,;
                        substr( TEMPHTML->Text, nIStart );
                    )

                if  nIEnd > 0

                    cIndex :=;
                        substr(;
                            TEMPHTML->Text,;
                            nIStart,;
                            nIEnd-1;
                        )

                    *---------------------------------------------
                    * An "index" was found: add it to the index
                    * array and to the TEMPHTML file.
                    *---------------------------------------------


                    cIndex := upper( alltrim( cIndex ) )

                    aadd( axIndex, { cIndex, recno() } )

                    TEMPHTML->Text := '<A NAME="' + cIndex + '"></A>' + TEMPHTML->Text

                end

            end

            *-----------------------------------------------------
            * Prepare to scan next record.
            *-----------------------------------------------------

            TEMPHTML->(dbskip())

        end

        *---------------------------------------------------------
        * Close Wait bar.
        *---------------------------------------------------------

        waitFileEval( .T. )

        *---------------------------------------------------------
        * Find "pointers".
        *---------------------------------------------------------

        waitFor( HTF_WAIT_SEARCHING_POINTERS )

        TEMPHTML->(dbgotop())

        while !eof()

            *-----------------------------------------------------
            * There may be only one pointer per line.
            *-----------------------------------------------------

            TEMPHTML->(waitFileEval())

            *-----------------------------------------------------
            * Locate the presence of a "pointer"
            *-----------------------------------------------------

            nPStart := at( cPointerStart, TEMPHTML->Text )

            if nPStart > 0

                nPStart += nPStartLen

                nPEnd :=;
                    at(;
                        cPointerEnd,;
                        substr( TEMPHTML->Text, nPStart );
                    )

                if  nPEnd > 0

                    cPointer :=;
                        substr(;
                            TEMPHTML->Text,;
                            nPStart, nPEnd-1;
                        )

                    cPointerNormal := cPointer
                    cPointer := upper( alltrim( cPointer ) )

                    *---------------------------------------------
                    * A "pointer" was found: search the "index"
                    * array to find the position.
                    *---------------------------------------------

                    for nI := 1 to len( axIndex )

                        if axIndex[nI][1] == cPointer

                            *-------------------------------------
                            * The corresponding index is found
                            * insiede the index array:
                            * replace the text.
                            *-------------------------------------

                            TEMPHTML->Text :=;
                                stuff(;
                                    TEMPHTML->Text,;
                                    nPStart,;
                                    len(cPointerNormal),;
                                    '<A HREF="#' +;
                                        cPointer +;
                                        '">' +;
                                        cPointerNormal +;
                                        '</A>';
                                )

                            *-------------------------------------
                            * Only one pointer can appear on a
                            * line, so exit the loop.
                            *-------------------------------------

                            exit

                        else

                            loop

                        end

                    end

                    /* &&&&&
                    aeval(;
                        axIndex,;
                        { |aValue, nIndex|;
                            iif(;
                                aValue[1] == cPointer,;
                                TEMPHTML->Text := stuff( TEMPHTML->Text, nPStart, len(cPointerNormal), '<A HREF="#' + cPointer + '">' + cPointerNormal + '</A>' ),;
                                NIL;
                            );
                        };
                    )
                    */

                end

            end

            TEMPHTML->(dbskip())

        end

        *---------------------------------------------------------
        * Close Wait bar and wait for.
        *---------------------------------------------------------

        waitFor()

        waitFileEval( .T. )

        *---------------------------------------------------------
        * Add termination.
        *---------------------------------------------------------
        
        dbappend()

        TEMPHTML->Text := "</PRE></BODY></HTML>"
        
        *---------------------------------------------------------
        * Add at Top.
        *---------------------------------------------------------
        
        dbgotop()
        
        TEMPHTML->Text := "<HTML><HEAD><Title>"
        
        dbskip()
        
        TEMPHTML->Text := cTitle
        
        dbskip()
        
        TEMPHTML->Text := "</Title></HEAD><BODY><PRE>"

        dbskip()

        TEMPHTML->Text := "<H1>" + cTitle + "</H1>"
        
        *---------------------------------------------------------
        * Data transfer.
        * - prepare a wait box
        * - direct output only to the destination HTML file
        * - prepare the record pointer to the first position
        * - loop thru records.
        *---------------------------------------------------------

        waitFor( HTF_WAIT_COPYING + alltrim( cDestName ) )
        
        aOldOutput := setOutput( cDestName )

        dbgotop()
        
        while !eof()
        
            *-----------------------------------------------------
            * Write the current record inside the HTML file
            * removing extra space on the right.
            *-----------------------------------------------------

            qout( rtrim( TEMPHTML->Text ) )
            
            *-----------------------------------------------------
            * Increment the wait bar.
            *-----------------------------------------------------

            TEMPHTML->(waitFileEval())

            *-----------------------------------------------------
            * Go to next record.
            *-----------------------------------------------------

            dbskip()
            
        end

        *---------------------------------------------------------
        * Restore old output standards,
        * close the wait bar,
        * close the wait box.
        *---------------------------------------------------------

        setOutput( aOldOutput )

        waitFileEval( .T. )

        waitFor()

        *---------------------------------------------------------
        * Close file.
        *---------------------------------------------------------

        TEMPHTML->(dbclosearea())

        *---------------------------------------------------------
        * Delete the temporary file.
        *---------------------------------------------------------

        ferase( cTempPath + "\TEMPHTML.DBF" )

    end //sequence

    *-------------------------------------------------------------
    * Restore old data.
    *-------------------------------------------------------------

    rddsetdefault( cOldRdd )
    dbselectarea( nOldSelect )
    setOutput( aOldOutput )

    return NIL

*=================================================================
* HTMLTRANSLATE()
*=================================================================

#define HTML_EQUIVALENT;
    {;
        { "<", "&LT;" },;
        { ">", "&GT;" };
    }

*=================================================================
function htmlTranslate( cText )
*
* htmlTranslate( <cText> ) --> cHtmlTranslated
*
* <cText>   Text to be converted into HTML format.
*
* This function should translate <cText> into a text compatible
* with Html.
*
* This function depends on HTML_EQUIVALENT that is not complete
* at the moment.
*

    local aHtmlEq := HTML_EQUIVALENT
    
    aeval(;
        aHtmlEq,; 
        { |aValue, nIndex|;
            cText := strtran( cText, aValue[1], aValue[2] );
        },;
    )
    
    return cText

*=================================================================
* ISFILE()
* ISWILD()
* ISMEMVAR()
* ISCONSOLEON()
* ISPRINTERON()
*=================================================================
function isFile( cName )
*
* isFile( <cName> ) --> lIsFilePresent
*
* <cName>  the name to check for existance.
*
* The function returns true (.T.) if it finds the file.
* Before it checks for wild characters "*" or "?".
*

    *-------------------------------------------------------------
    * Wildcards means that is isn't one specific file.
    *-------------------------------------------------------------

    if isWild( cName )

        return .F.                                      // RETURN

    end

    if !file( alltrim( cName ) )

        return .F.                                      // RETURN

    end

    *-------------------------------------------------------------
    * If we are here, the file exists and it is not undefined.
    *-------------------------------------------------------------

    return .T.

*=================================================================
function isWild( cName )
*
* isWild( <cName> ) --> lIsWild
*
* <cName>  the name to check for wild characters.
*
* The function returns true (.T.) if it finds "*" or "?"
* inside the given string.
*

    *-------------------------------------------------------------
    * Wildcard are "*" and "?".
    *-------------------------------------------------------------

    if "*" $ cName .or. "?" $ cName

        return .T.                                      // RETURN

    end

    return .F.

*=================================================================
function isMemvar( cName )
*
* isMemvar( <cName> ) --> lIsMemvar
*
* <cName>  the memvar name to check for existance.
*
* The function returns true (.T.) if it finds a memvar with the
* name contained inside <cName>.
*

    local bSaveErrorHandler := errorblock( {|e| break(e)} )
    local dummy
    local lReturn := .F.

    *-------------------------------------------------------------
    * Start a new sequence with a simple error handler.
    *-------------------------------------------------------------

    begin sequence

        *---------------------------------------------------------
        * If the following operation do not generate errors,
        * the variable exists.
        *---------------------------------------------------------

        dummy := memvar->&(cName)

        lReturn := .T.

    recover

        *---------------------------------------------------------
        * If an error occurred, the variable don't exists.
        *---------------------------------------------------------

        lReturn := .F.

    end //sequence
    bSaveErrorHandler := errorblock( bSaveErrorHandler )

    *-------------------------------------------------------------
    * The result is returned.
    *-------------------------------------------------------------

    return lReturn

*=================================================================
function isConsoleOn()
*
* isConsoleOn() --> lConsoleIsOn
*
* The function return true if qout() and qqout() print to the
* console.
*

    do case
    case set( _SET_CONSOLE )

        *---------------------------------------------------------
        * Set console is ON.
        *---------------------------------------------------------

        return .T.

    case;
        set( _SET_ALTERNATE )                   .and.;
        upper( set( _SET_ALTFILE ) ) == "CON"

        *---------------------------------------------------------
        * The alternate file is the console.
        *---------------------------------------------------------

        return .T.

    case;
        set( _SET_ALTERNATE )                   .and.;
        upper( set( _SET_ALTFILE ) ) == "CON.TXT"

        *---------------------------------------------------------
        * The alternate file is the console.
        *---------------------------------------------------------

        return .T.

    end

    *-------------------------------------------------------------
    * Otherwise.
    *-------------------------------------------------------------

    return .F.

*=================================================================
function isPrinterOn()
*
* isPrinterOn() --> lPrinterIsOn
*
* The function return true if qout() and qqout() print to a
* printer.
*

    do case
    case set( _SET_PRINTER )

        *---------------------------------------------------------
        * Set printer is ON.
        *---------------------------------------------------------

        return .T.

    case;
        set( _SET_ALTERNATE )                       .and.;
        upper( set( _SET_ALTFILE ) ) == "PRN"

        *---------------------------------------------------------
        * The alternate file is the printer.
        *---------------------------------------------------------

        return .T.

    case;
        set( _SET_ALTERNATE )                       .and.;
        upper( set( _SET_ALTFILE ) ) == "PRN.TXT"

        *---------------------------------------------------------
        * The alternate file is the printer.
        *---------------------------------------------------------

        return .T.
        
    case;
        set( _SET_ALTERNATE )                       .and.;
        upper( set( _SET_ALTFILE ) ) == "LPT1"

        *---------------------------------------------------------
        * The alternate file is the printer.
        *---------------------------------------------------------

        return .T.

    case;
        set( _SET_ALTERNATE )                       .and.;
        upper( set( _SET_ALTFILE ) ) == "LPT1.TXT"

        *---------------------------------------------------------
        * The alternate file is the printer.
        *---------------------------------------------------------

        return .T.

    end

    *-------------------------------------------------------------
    * Otherwise.
    *-------------------------------------------------------------

    return .F.

*=================================================================
* KEYBOARD()
*=================================================================
function keyboard( cString )
*
* Keyboard( [<cString>] ) --> NIL
*
* Keyboard command substitute.
*

    default( @cString,  "" )

    *-------------------------------------------------------------
    * KEYBOARD <cString>
    *-------------------------------------------------------------

    __keyboard( cString )

    return NIL

*=================================================================
* MEMOWINDOW()
*=================================================================

#define MEMO_WINDOW_BOTTOM;
    "[Esc] Cancel  [F1] Help  [Ctrl]+[W] Save"

#define MEMO_WINDOW_HELP;
    "memoWindow()" +;
    NL(3) +;
    "This function lets you modify a long character field " +;
    "just like a little text editor." +;
    NL(3) +;
    "[Esc]           leave editing without modification." + NL(1) +;
    "[Ctrl]+[W]      confirm modifications and exit the function." + NL(1) +;
    "[]             up" + NL(1) +;
    "[]             down" + NL(1) +;
    "[-]            left" + NL(1) +;
    "[-]            right" + NL(1) +;
    "[Ctrl]+[-]     word left" + NL(1) +;
    "[Ctrl]+[-]     word right" + NL(1) +;
    "[Home]          line start" + NL(1) +;
    "[End]           line end" + NL(1) +;
    "[Ctrl]+[Home]   top window" + NL(1) +;
    "[Ctrl]+[End]    bottom window" + NL(1) +;
    "[Pag]          previous window" + NL(1) +;
    "[Pag]          next window" + NL(1) +;
    "[Ctrl]+[Pag]   document start" + NL(1) +;
    "[Ctrl]+[Pag]   end document" + NL(1) +;
    "[Del]           delete character" + NL(1) +;
    "[Backspace]     delete character left" + NL(1) +;
    "[Tab]           insert tab" + NL(1) +;
    "[Ins]           insert/overwrite" + NL(1) +;
    "[Enter]         next line" + NL(1) +;
    "[Ctrl]+[Y]      delete line" + NL(1) +;
    "[Ctrl]+[T]      delete word right"

*=================================================================
function memoWindow(;
        cVar, cDescription,;
        nTop, nLeft, nBottom, nRight,;
        cColorTop, cColorBody,;
        lEditMode, nLineLength, nTabSize;
    )
*
* memoWindow(
*    <cVar>, [<cDescription>],
*    [<nTop>], [<nLeft>], [<nBottom>], [<nRight>],
*    [<cColorTop>], [<cColorBody>],
*    [<lEditMode>], [<nLineLength>], [<nTabSize>] ) --> cVar
*
* <cVar>           is the character field (variable) to be
*                  edited.
* <cDescription>   is the header to be shown at the top
*                  window.
* <nTop>, <nLeft>, <nBottom>, <nRight> are the window
*                  coordinates.
* <cColorTop>      is the color to use for window header
*                  and footer.
* <cColorBody>     is the color to use for the window body
*                  that is the space where the text appears.
* <lEditMode>      is equivalent to memoedit().
* <nLineLength>    is equivalent to memoedit().
* <nTabSize>       is equivalent to memoedit().
*
* This function lets you easyly edit a long character field
* (memo) defining automatically a simple window and providing
* a simple help.
*
*

    local cOldColor     := setcolor()
    local cOldScreen
    local nSetCursor    := setcursor ( SETCURSOR_NORMAL )
    local bOld_F1           :=;
        setkey( K_F1, { || Text( MEMO_WINDOW_HELP )} )
    local nOldRow       := row()
    local nOldCol       := col()

    default( @cDescription, "Edit long character field" )
    default( @nTop,         0 )
    default( @nLeft,        0 )
    default( @nBottom,      maxrow() )
    default( @nRight,       maxcol() )
    default( @cColorTop,    COLOR_HEAD )
    default( @cColorBody,   COLOR_BODY )

    *-------------------------------------------------------------
    * Create a kind of window
    *-------------------------------------------------------------

    cOldScreen    :=;
        mouseScrSave( nTop, nLeft, nBottom, nRight )

    dispBoxShadow(;
        nTop, nLeft, nBottom, nRight,;
        1,;
        dispBoxColor( 1, cColorBody ),;
        dispBoxColor( 2, cColorBody );
    )
    say(;
        nTop+1, nLeft+1,;
        padc( cDescription, nRight-nLeft-1 ),;
        ,;
        cColorTop;
    )
    say(;
        nBottom-1, nLeft+1,;
        padc( MEMO_WINDOW_BOTTOM, nRight-nLeft-1 ),;
        ,;
        cColorTop;
    )
    setcolor( cColorBody )

    *-------------------------------------------------------------
    * Do editing.
    *-------------------------------------------------------------

    cVar :=;
        memoedit(;
            cVar,;
            nTop+2, nLeft+1, nBottom-2, nRight-1,;
            lEditMode, NIL, nLineLength, nTabSize;
        )

    *-------------------------------------------------------------
    * Restore old values.
    *-------------------------------------------------------------

    mouseScrRestore( nTop, nLeft, nBottom, nRight, cOldScreen )
    setcursor ( nSetCursor )
    setcolor( cOldColor )
    setkey( K_F1, bOld_F1 )
    setpos( nOldRow, nOldCol )

    *-------------------------------------------------------------
    * Return the edited value.
    *-------------------------------------------------------------

    return cVar

*=================================================================
* MEMPUBLIC()
* MEMRELEASE()
* MEMRESTORE()
* MEMSAVE()
*=================================================================
function memPublic( VarName )
*
* MemPublic( <cVarName> | <acVarName> ) --> NIL
*
* <cVarName>   Contains the variable name.
* <acVarName>  Contains an array of variable names.
*
* Creates the public variable|variables named into <cVarName>
* or <acVarname>.
*

    local nVar
    local cVar

    *-------------------------------------------------------------
    * Depending on the type of <VarName>, create the variable or
    * the variables.
    *-------------------------------------------------------------

    do case
    case valtype(VarName) == "C"

        *---------------------------------------------------------
        * Only one variable will be created.
        *---------------------------------------------------------

        public &VarName.
        
        &VarName. := NIL

    case valtype(VarName) == "A"

        *---------------------------------------------------------
        * Many variables will be created.
        *---------------------------------------------------------

        for nVar := 1 to len(VarName)

            cVar := VarName[nVar]

            public &cVar.
            
            &cVar. := NIL

        next

    otherwise

        *---------------------------------------------------------
        * The type is not valid: nothing will be created.
        *---------------------------------------------------------

    end

    return NIL

*=================================================================
function memRelease( VarName )
*
* memRelease( <cVarName> | <acVarName> ) --> NIL
*
* <cVarName>   Contains the variable name.
* <acVarName>  Contains an array of variable names.
*
* Releases the public variable|variables named into <cVarName>
* or <acVarname>.
*

    local nVar

    *-------------------------------------------------------------
    * Depending on the type of <VarName>, release the variable or
    * the variables.
    *-------------------------------------------------------------

    do case
    case valtype(VarName) == "C"

        *---------------------------------------------------------
        * RELEASE <idMemvar>
        *---------------------------------------------------------

        __MXRelease( VarName )

    case valtype(VarName) == "A"

        for nVar := 1 to len(VarName)

            *-----------------------------------------------------
            * RELEASE <idMemvar>
            *-----------------------------------------------------

            __MXRelease( VarName[nVar] )

        next

    otherwise

        *---------------------------------------------------------
        * The type is not valid: nothing will be released.
        *---------------------------------------------------------

    end

    return NIL

*=================================================================
function memRestore( cMemFileName, lAdditive )
*
* memRestore( <cMemFileName>, [<lAdditive>] ) --> NIL
*
* Restore From command substitute.
*

    default( @lAdditive,    .F. )

    *-------------------------------------------------------------
    * RESTORE FROM <xcMemFile> [ADDITIVE]
    *-------------------------------------------------------------

    __MRestore( cMemFileName, lAdditive )

    return NIL

*=================================================================
function memSave( cMemFileName, cSkeleton, lLike )
*
* memSave( <cMemFileName>, [<cSkeleton>], [<lLike>] ) --> NIL
*
* Save To command substitute.
*
* <cMemFileName>       (.mem) file name to create.
* <cSkeleton>          memory names skeleton.
* <lLike>              true (.T.) means LIKE SKELETON,
*                      false (.F.) means EXCEPT SKELETON.
*

    default( @cSkeleton,    "*" )
    default( @lLike,        .T. )

    if lLike

        *---------------------------------------------------------
        * SAVE TO <xcMemFile> ALL LIKE <skeleton>
        *---------------------------------------------------------

        __MSave( cMemFileName, cSkeleton, .T. )

    else

        *---------------------------------------------------------
        * SAVE TO <xcMemFile> ALL EXEPT <skeleton>
        *---------------------------------------------------------

        __MSave( cMemFileName, cSkeleton, .F. )

    end

    return NIL

*=================================================================
* MENUPROMPT()
* MENUTO()
*=================================================================
function menuPrompt(;
        aoGet,;
        nRow, nCol,;
        cPrompt,;
        bBlock;
    )
*
* menuPrompt(
*   @<aoGet>,
*   [<nRow>], [<nCol>],
*   [<cPrompt>],
*   [<bBlock>]
*   ) --> NIL
*
* like @...prompt
*
*

    local oGet

    default( @aoGet,            {} )
    default( @nRow,             row() )
    default( @nCol,             col() )
    default( @cPrompt,          "" )
    default( @bBlock,           {||NIL} )

    *-------------------------------------------------------------
    * Create a new get object that will be used in read only mode.
    *-------------------------------------------------------------

    oGet :=;
        getnew(;
            nRow, nCol, {|| cPrompt}, NIL, NIL, NIL;
        )

    *-------------------------------------------------------------
    * The prevalidation block may be used to display some
    * informations when the cursor reach the menu item
    *-------------------------------------------------------------

    oGet:preBlock := bBlock

    *-------------------------------------------------------------
    * Show the get object (show the menu item).
    *-------------------------------------------------------------

    oGet:display()

    *-------------------------------------------------------------
    * Add the get object to the getlist array.
    *-------------------------------------------------------------

    aadd( aoGet, oGet )

    return NIL

*=================================================================
function menuTo( aoGet, nPos )
*
* menuTo( <aoGet>, <nPos> ) --> nChoice
*
* <aoGet>      array of get objects.
*
* <nPos>       starting position to be edited.
*
* Like MENU TO. It works like the READ and return the selected
* Get.
*

    local oGet
    local lOldSetMouse      := setMouse()

    local nI
    local nLen
    local aGetPos       := {}
    local lExit         := .F.

    default( @nPos,     1 )
    if nPos <= 0
        nPos := 1
    end

    *-------------------------------------------------------------
    * If GetList is empty, exit.
    *-------------------------------------------------------------

    if ( empty( aoGet ) )

        return 0                                        // RETURN

    end

    *-------------------------------------------------------------
    * Copy GETs position.
    *-------------------------------------------------------------

    for nI := 1 to len( aoGet )

        oGet := aoGet[nI]

        nLen := len( eval( oGet:block ) )

        aadd( aGetPos, { oGet:row, oGet:col, oGet:row, oGet:col+nLen-1 } )

    next

    *-------------------------------------------------------------
    * Start the main read loop.
    *-------------------------------------------------------------

    while !( nPos == 0 ) .and. !lExit

        *---------------------------------------------------------
        * Get next GET from list.
        *---------------------------------------------------------

        oGet := aoGet[ nPos ]

        *---------------------------------------------------------
        * Eval prevalidation Block.
        *---------------------------------------------------------

        eval( oGet:preBlock )

        *---------------------------------------------------------
        * Read the GET.
        *---------------------------------------------------------

        lExit := menuReader( oGet, aGetPos, @nPos )

        *---------------------------------------------------------
        * Hide mouse cursor before a movement inside the
        * Get list fields. The mouse cursor will be shown
        * inside the GetApplayKey() function.
        *---------------------------------------------------------

        setMouse(.F.)

    end

    *-------------------------------------------------------------
    * Restore the mouse status.
    *-------------------------------------------------------------

    setMouse( lOldSetMouse )

    *-------------------------------------------------------------
    * To clear the menu array.
    *-------------------------------------------------------------

    aoGet := {}

    *-------------------------------------------------------------
    * Return the selected position.
    *-------------------------------------------------------------

    return ( nPos )


*-----------------------------------------------------------------
static function menuReader( oGet, aGetPos, nPos )
*
*

    local lExit := .f.

    *-------------------------------------------------------------
    * Activate the GET for reading.
    *-------------------------------------------------------------

    oGet:setFocus()

    *-------------------------------------------------------------
    * Clear saved mouse status.
    *-------------------------------------------------------------

    mouse( .T. )

    *-------------------------------------------------------------
    * Show the mouse cursor.
    *-------------------------------------------------------------

    setMouse(.T.)

    *-------------------------------------------------------------
    * Applay the key: read the mouse and the keyboard.
    *-------------------------------------------------------------

    lExit := menuApplyKey( oGet, aGetPos, @nPos )

    *-------------------------------------------------------------
    * Clear saved mouse status.
    *-------------------------------------------------------------

    mouse( .T. )

    *-------------------------------------------------------------
    * Hide the mouse cursor.
    *-------------------------------------------------------------

    setMouse( .F. )

    *-------------------------------------------------------------
    * De-activate the GET.
    *-------------------------------------------------------------

    oGet:killFocus()

    return lExit

*-----------------------------------------------------------------
static function menuApplyKey( oGet, aGetPos, nPos )
*
* NOTE: GET must have focus.
*

    local lExit := .F.
    local nKey

    *-------------------------------------------------------------
    * Inkey() loop.
    *-------------------------------------------------------------

    while .T.                                           // FOREVER

        if !( mouse() == NIL )

            *-----------------------------------------------------
            * A mouse button was pressed and released.
            * Get the mouse position relative to the
            * list of GETs or buttons.
            *-----------------------------------------------------

            nPos := menuMousePos( aGetPos )

            lExit := .T.

            *-----------------------------------------------------
            * Exit this loop too.
            *-----------------------------------------------------

            exit                                        // EXIT

        end

        *---------------------------------------------------------
        * Read a key without waiting.
        *---------------------------------------------------------

        nKey := inkey()

        do case
        case ( nKey <> 0 ) .and. valtype( setkey( nKey ) ) == "B"

            *-----------------------------------------------------
            * The key is redirected: Execute a key code block.
            *-----------------------------------------------------

            setMouse(.F.)

            eval( setkey( nKey ) )

            setMouse(.T.)

        case ( nKey == K_UP )

            if nPos > 1
                nPos--
            else
                nPos := 1
            end

            exit                                        // EXIT

        case ( nKey == K_LEFT )

            if nPos > 1
                nPos--
            else
                nPos := 1
            end

            exit                                        // EXIT

        case ( nKey == K_SH_TAB )

            if nPos > 1
                nPos--
            else
                nPos := 1
            end

            exit                                        // EXIT

        case ( nKey == K_DOWN )

            if nPos < len( aGetPos )
                nPos++
            else
                nPos := len( aGetPos )
            end

            exit                                        // EXIT

        case ( nKey == K_RIGHT )

            if nPos < len( aGetPos )
                nPos++
            else
                nPos := len( aGetPos )
            end

            exit                                        // EXIT

        case ( nKey == K_TAB )

            if nPos < len( aGetPos )
                nPos++
            else
                nPos := len( aGetPos )
            end

            exit                                        // EXIT

        case ( nKey == K_ENTER )

            lExit := .T.

            exit                                        // EXIT

        case ( nKey == K_ESC )

            nPos := 0

            exit                                        // EXIT

        case ( nKey == K_HOME )

            nPos := 1

            exit                                        // EXIT

        case ( nKey == K_END )

            nPos := len( aGetPos )

            exit                                        // EXIT

        end

    end

    return lExit

*-----------------------------------------------------------------
static function menuMousePos( aGetPos )
*
*

    local aMouse    := mouse()

    local nCol      := aMouse[1]-1
    local nRow      := aMouse[2]-1

    local lLeft     := aMouse[3]

    local nI
    local nPos  := 0

    local aSameLine := {}

    local nGetRow
    local nGetCol
    local nGetColEnd

    *-------------------------------------------------------------
    * Start a new sequence.
    *-------------------------------------------------------------

    begin sequence

        if lLeft
            // OK
        else
            // nPos is 0
            break                                       // BREAK
        end

        *---------------------------------------------------------
        * Try to find a get.
        *---------------------------------------------------------

        for nI := 1 to len(aGetPos)

            nGetRow    := aGetPos[nI][1]
            nGetCol    := aGetPos[nI][2]
            nGetColEnd := aGetPos[nI][4]

            if  nGetRow == nRow         .and.;
                nGetCol <= nCol         .and.;
                nGetColEnd >= nCol

                *-------------------------------------------------
                * OK selected get.
                *-------------------------------------------------

                nPos := nI

                break                                   // BREAK

            end

        next

    end //sequence

    *-------------------------------------------------------------
    * Return selected position.
    *-------------------------------------------------------------

    return nPos

*=================================================================
* MESSAGELINE()
*=================================================================
function messageLine( cMessage, cColor, nPosTop, nPosLeft )
*
* messageLine( <cMessage>, [<cColor>], [<nPosTop>], [<nPosLeft>] )
*   --> NIL
*
* Message line appearing near the cursor position.
*

    local getlist           := {}
    local cOldColor         := setcolor()
    local nOldRow           := row()
    local nOldCol           := col()

    local nWidth

    static cOldScreen
    static nTop
    static nLeft
    static nBottom
    static nRight

    default( @cColor,       COLOR_ALERT )
    default( @nPosTop,      row() )
    default( @nPosLeft,     col() )

    do case
    case;
        cMessage == NIL         .and.;
        cOldScreen == NIL

        *---------------------------------------------------------
        * Nothing to be closed: return.
        *---------------------------------------------------------

        return NIL                                      // RETURN

    case;
        cMessage == NIL         .and.;
        cOldScreen <> NIL

        *---------------------------------------------------------
        * Close previous line and return.
        *---------------------------------------------------------

        mouseScrRestore( nTop, nLeft, nBottom, nRight, cOldScreen )

        cOldScreen := NIL

        return NIL                                      // RETURN

    case;
        cMessage <> NIL         .and.;
        cOldScreen == NIL

        *---------------------------------------------------------
        * New Message Line.
        *---------------------------------------------------------

    case;
        cMessage <> NIL         .and.;
        cOldScreen <> NIL

        *---------------------------------------------------------
        * Close previous line.
        *---------------------------------------------------------

        mouseScrRestore( nTop, nLeft, nBottom, nRight, cOldScreen )

        cOldScreen := NIL

    end

    *-------------------------------------------------------------
    * If still here, a new message line is displayed.
    *-------------------------------------------------------------

    begin sequence

        *---------------------------------------------------------
        * Prepare cMessage.
        *---------------------------------------------------------

        cMessage := alltrim(cMessage)

        nWidth := len( cMessage )

        do case
        case nWidth == 0

            *-----------------------------------------------------
            * Empty messages are not displayed.
            *-----------------------------------------------------

            break                                       // BREAK

        case nWidth > maxcol()+1

            *-----------------------------------------------------
            * Adjust width.
            *-----------------------------------------------------

            nWidth := maxcol()+1

            cMessage := left( cMessage, nWidth )

        end

        *---------------------------------------------------------
        * Treat it as a kind of window and adjust position.
        *---------------------------------------------------------

        nTop        := nPosTop
        nLeft       := nPosLeft
        nBottom     := nTop

        if nBottom > maxrow()
            nBottom := maxrow() -1
            nTop    := nBottom
        end

        nRight  := nLeft+nWidth-1

        if nRight > maxcol()
            nRight := maxcol()
            nLeft  := nRight -nWidth+1
        end

        cOldScreen    :=;
            mouseScrSave( nTop, nLeft, nBottom, nRight )

        setcolor( cColor )
        setpos( nTop, nLeft)

        *---------------------------------------------------------
        * Show the message.
        *---------------------------------------------------------

        dispout( cMessage )

    end //sequence

    *-------------------------------------------------------------
    * Restore old data.
    *-------------------------------------------------------------

    setpos( nOldRow, nOldCol )
    setcolor( cOldColor )

    return NIL

*=================================================================
* MOUSE()
* MOUSESCRSAVE()
* MOUSESTRRESTORE()
*=================================================================
function mouse( lEnd )
*
* This is a special mouse function to read, save and reset the
* mouse status.
*
*       I       The mouse status is tested:
*               if no button is pressed, nothing is done.
*
*       II      If a mouse button is pressed, <lActive> is set
*               to True and this means that a mouse action is
*               takeing place. A loop is started to wait for
*               mouse button release.
*
*       III     When the mouse button is released, the mouse
*               cursor position is read and saved. Now, <lSave>
*               is set to True and <lActive> is turned OFF.
*
*       IV      When <lSave> is True, the previous saved mouse
*               cursor position is returned and the mouse is not
*               read.
*
*       V       When mouse(.T.) is called (that is that <lEnd>
*               is True), all is resetted and ready for a new
*               mouse test.
*

    static lActive
    static lSave
    static nX
    static nY
    static lLeft
    static lRight

    default( @lActive,  .F. )
    default( @lSave,    .F. )
    default( @nX,       0 )
    default( @nY,       0 )
    default( @lLeft,    .F. )
    default( @lRight,   .F. )

    default( @lEnd,     .F. )

    *-------------------------------------------------------------
    * <lEnd> terminates the mouse cicle.
    *-------------------------------------------------------------
    
    if lEnd

        if lSave

            setMouse( .F. )

        end

        lSave       := .F.

        lActive     := .F.

        return NIL                                      // RETURN

    end

    *-------------------------------------------------------------
    * If not <lEnd>, we are here.
    *-------------------------------------------------------------
    
    while .T.                                           // FOREVER

        do case
        case lSave

            *-----------------------------------------------------
            * Report saved values only.
            *-----------------------------------------------------
            
            exit                                        // EXIT

        case !lSave .and. !lActive

            do case
            case; 
                mBout() == 2            .or.;
                mBout() == 8            .or.;
                mBout() == 10

                *-------------------------------------------------
                * Left or Right or Both buttons pressed and not 
                * jet moved.
                *-------------------------------------------------
                
                lActive := .T.

                *-------------------------------------------------
                * Now loop until a button is released.
                *-------------------------------------------------

            otherwise
                
                *-------------------------------------------------
                * Nothing to do here.
                *-------------------------------------------------
                
                exit                                    // EXIT

            end

        case !lSave .and. lActive

            do case
            case mLast() == 4

                *-------------------------------------------------
                * Left button was released.
                *-------------------------------------------------
                
                lActive := .F.

                lSave   := .T.

                setMouse( .F. )

                lLeft   := .T.
                lRight  := .F.

                *-------------------------------------------------
                * Save mouse cursor position.
                *-------------------------------------------------
                
                nX := mX()
                nY := mY()

                exit                                    // EXIT

            case mLast() == 8

                *-------------------------------------------------
                * Right button was released.
                *-------------------------------------------------
                
                lActive := .F.

                lSave   := .T.

                setMouse( .F. )

                lLeft   := .F.
                lRight  := .T.

                *-------------------------------------------------
                * Save mouse cursor position.
                *-------------------------------------------------
                
                nX := mX()
                nY := mY()
                
                exit                                    // EXIT

            case mLast() == 20

                *-------------------------------------------------
                * Both buttons were released.
                *-------------------------------------------------
                
                lActive := .F.

                lSave   := .T.

                setMouse( .F. )

                lLeft   := .T.
                lRight  := .T.
                
                *-------------------------------------------------
                * Save mouse cursor position.
                *-------------------------------------------------
                
                nX := mX()
                nY := mY()
                
                exit                                    // EXIT

            otherwise

                *-------------------------------------------------
                * Loop.
                *-------------------------------------------------
            
            end
        
        end
    
    end

    *-------------------------------------------------------------
    * If <lSave> is True, we are reading the mouse cursor
    * position previuosly saved.
    *-------------------------------------------------------------
    
    if lSave

        return { nX, nY, lLeft, lRight }

    else

        return NIL

    end

    return NIL

*=================================================================
function mouseScrSave( nTop, nLeft, nBottom, nRight )

    local lOldSetMouse      := setMouse(.F.)
    local cSavedScreen      := savescreen( nTop, nLeft, nBottom, nRight )

    setMouse( lOldSetMouse )

    return cSavedScreen

*=================================================================
function mouseScrRestore( nTop, nLeft, nBottom, nRight, cScreen )

    local lOldSetMouse      := setMouse(.F.)

    restscreen( nTop, nLeft, nBottom, nRight, cScreen )

    setMouse( lOldSetMouse )

    return NIL

*=================================================================
* PICCHRMAX()
*=================================================================
function PicChrMax( nCol, nMaxCol )
*
* <nCol>        start position of the get object.
* <nMaxCol>     end position of the get object.
*
* This funciton returns a character picture string for character
* fields.
*
    local nDim

    default( @nCol,     col() )
    default( @nMaxCol,  maxcol() )

    *-------------------------------------------------------------
    * Calculate dimention.
    *-------------------------------------------------------------

    nDim := nMaxCol - nCol +1

    *-------------------------------------------------------------
    * Return the picture string.
    *-------------------------------------------------------------

    return "@s" + ltrim( str( nDim ) )

*=================================================================
* QUIT()
*=================================================================
function Quit()
*
* Quit() --> NIL
*
* Quit command substitute.
*

    __Quit()

    return NIL

*=================================================================
* read()
* readStop()
*=================================================================

* Get positions.
#define GET_POS_ORDER   1
#define GET_POS_ROW     2
#define GET_POS_COL     3

* Button array.
#define BUTTON_ROW      1
#define BUTTON_COL      2
#define BUTTON_TEXT     3
#define BUTTON_BLOCK    4

*=================================================================
function read( aoGet, nStart, aButtons, lReadOnly )
*
* read( <aoGet>, [<nPos>], [<aButtons>], [<lReadOnly>] ) --> lUpdated
*
* <aoGet>    array of get objects.
*
* <nPos>       starting position to be edited.
*
* <aButtons>   array of buttons.
*
* <lReadOnly>  True means that the editing is not possible
*              (the default is False).
*
*  Modal READ on an array of GETs and an array of buttons.
*

    local oGet

    local lHitTop           := .F.
    local lHitBottom        := .F.
    local nLastExitState    := 0
    local nLastPosition     := 0

    local lOldReadKill      := readStop( .F. )
    local oOldGetActive     := getActive()
    local lOldSetMouse      := setMouse()

    local nI
    local aGetPos       := {}
    local aGetPosSort   := {}

    local nPos          := 0
    local nPrevious
    
    default( @lReadOnly, .F. )

    *-------------------------------------------------------------
    * If aoGet is empty, exit.
    *-------------------------------------------------------------
    
    if ( empty( aoGet ) )

        return (.F.)                                    // RETURN
    
    end

    *-------------------------------------------------------------
    * At the moment nothing is updated.
    * This status will be not restored.
    *-------------------------------------------------------------
    
    readUpdated( .F. )

    *-------------------------------------------------------------
    * Set initial GET to be read.
    *-------------------------------------------------------------
    
    if !( valtype( nStart ) == "N" .AND. nStart > 0 )

        nPos :=;
            getSettle(;
                aoGet, 0,;
                @lHitTop, @lHitBottom,;
                @nLastExitState, @nLastPosition;
            )
        
        nStart := nPos
    
    else
        
        nPos := nStart
    
    end

    *-------------------------------------------------------------
    * Copy GETs position for mouse use.
    *-------------------------------------------------------------
    
    for nI := 1 to len( aoGet )

        oGet := aoGet[nI]

        aadd( aGetPos, { nI, oGet:row, oGet:col } )

    next

    *-------------------------------------------------------------
    * Sort GETs positions by row/col.
    *-------------------------------------------------------------
    
    aGetPosSort :=;
        asort(;
            aGetPos,;
            NIL,;
            NIL,;
            { |x, y|;
                iif(;
                    x[GET_POS_ROW]==y[GET_POS_ROW],;
                    x[GET_POS_COL] < y[GET_POS_COL],;
                    x[GET_POS_ROW] < y[GET_POS_ROW]; 
                );
            };
        )

    *-------------------------------------------------------------
    * Read loop.
    *-------------------------------------------------------------
    
    while !( nPos == 0 )

        *---------------------------------------------------------
        * Get next GET from list and post it as the active GET.
        *---------------------------------------------------------
        
        oGet := aoGet[ nPos ]

        getActive( oGet )

        *---------------------------------------------------------
        * Read the GET.
        *---------------------------------------------------------
        
        if ( valtype( oGet:reader ) == "B" )

            *-----------------------------------------------------
            * Use custom reader block.
            *-----------------------------------------------------
            
            eval( oGet:reader, oGet )

        else

            *-----------------------------------------------------
            * Use standard reader.
            * The standard reader includes the mouse read.
            *-----------------------------------------------------
            
            nPrevious := nPos

            nPos := getReader( oGet, aButtons, aGetPosSort, nPos, lReadOnly )

        end

        if nPos <> nPrevious

            *-----------------------------------------------------
            * Changed by the mouse. Check if the pre validation
            * condition is satisfied, else return to last
            * position.
            *-----------------------------------------------------

            if !eval( aoGet[nPos]:preBlock, aoGet[nPos] )

                *-------------------------------------------------
                * The prevalidation block result False, so, 
                * the previous position must be restored...
                *-------------------------------------------------
                
                nPos := nPrevious

                *-------------------------------------------------
                * ...but using the mouse, the
                * prevalidation condition may be
                * False also for the previous position.
                * In this case, reuse the <nStart> position:
                * it *must* be ever valid!
                *-------------------------------------------------

                if !eval( aoGet[nPos]:preBlock, aoGet[nPos] )

                    nPos := nStart

                end

            end

        else
            
            *-----------------------------------------------------
            * The mouse wasn't used, so use the normal cursor
            * movement system.
            *-----------------------------------------------------
            
            nPos :=;
                getSettle(;
                    aoGet, nPos,;
                    @lHitTop, @lHitBottom,;
                    @nLastExitState, @nLastPosition;
                )
        end

    end

    *-------------------------------------------------------------
    * Restore state variables.
    *-------------------------------------------------------------
    
    readStop( lOldReadKill )
    
    getActive( oOldGetActive )

    setMouse( lOldSetMouse )

    *-------------------------------------------------------------
    * Return True if a update was made or False in the other
    * case.
    *-------------------------------------------------------------
    
    return ( readUpdated() )

*=================================================================
function readStop( lNew )
*
*  Modifyed to hold the readkill status.
*

    local  lPrevious
    static lKill

    default( @lKill, .F. )

    lPrevious := lKill

    if !(lNew == NIL)
        
        *---------------------------------------------------------
        * lNew is not NIL and the value of <lNew> is saved
        * inside the static variable <lKill>.
        *---------------------------------------------------------
        
        lKill := lNew
    
    end

    *-------------------------------------------------------------
    * The Previous value is returned.
    *-------------------------------------------------------------
    
    return lPrevious

*-----------------------------------------------------------------
static function readUpdated( lNew )
*
*  Modifyed to hold the readupdated status.
*

    local  lPrevious
    static lUpdated

    default( @lUpdated, .F. )

    lPrevious := lUpdated

    if !(lNew == NIL)
        
        lUpdated := lNew
    
    end

    return lPrevious

*-----------------------------------------------------------------
static function getActive( oNew )
*
*  Modifyed to hold the previous value.
*

    local  oPrevious
    static oActive

    oPrevious := oActive

    if !(oNew == NIL)
        
        oActive := oNew
    
    end

    return oPrevious

*-----------------------------------------------------------------
static function getApplyKey( oGet, nKey, lReadOnly )
*
* getApplyKey( <oGet>, <nKey>, <lReadOnly> ) --> NIL
*
* <oGet>       GET object to be read.
*
* <nKey>       key pressed.
*
* <lReadOnly>  if True, the fields are read only.
*
* Apply a single INKEY() keystroke to a GET or read the
* mouse action.
* NOTE: GET must have focus.
*

    local cKey

    *-------------------------------------------------------------
    * Check for SET KEY first.
    *-------------------------------------------------------------
    
    if !( setkey( nKey ) == NIL )
        
        *---------------------------------------------------------
        * Evaluate the code block.
        *---------------------------------------------------------
        
        getDoBlock( setkey( nKey ), oGet )
        
        return NIL                                      // RETURN

    end

    do case
    case ( nKey == K_UP )
        
        oGet:exitState := GETEXIT_UP

    case ( nKey == K_SH_TAB )

        oGet:exitState := GETEXIT_UP

    case ( nKey == K_DOWN )

        oGet:exitState := GETEXIT_DOWN

    case ( nKey == K_TAB )

        oGet:exitState := GETEXIT_DOWN

    case ( nKey == K_ENTER )

        oGet:exitState := GETEXIT_ENTER

    case ( nKey == K_ESC )

        *---------------------------------------------------------
        * If set escape is ON the read can terminate.
        *---------------------------------------------------------
        
        if ( set( _SET_ESCAPE ) )
        
            *-----------------------------------------------------
            * Test before if not read only.
            *-----------------------------------------------------

            if !lReadOnly

                *-------------------------------------------------
                * The current get is undone.
                *-------------------------------------------------
            
                oGet:undo()
                
            end

            oGet:exitState := GETEXIT_ESCAPE

        end

    case ( nKey == K_PGUP )

        oGet:exitState := GETEXIT_WRITE

    case ( nKey == K_PGDN )

        oGet:exitState := GETEXIT_WRITE

    case ( nKey == K_CTRL_HOME )

        oGet:exitState := GETEXIT_TOP

    case ( nKey == K_CTRL_W )

        oGet:exitState := GETEXIT_WRITE

    case ( nKey == K_INS )

        *---------------------------------------------------------
        * Toggle insert mode.
        *---------------------------------------------------------
        
        set( _SET_INSERT, !set( _SET_INSERT ) )

    case ( nKey == K_CTRL_U )

        *---------------------------------------------------------
        * Test before if not read only.
        *---------------------------------------------------------

        if !lReadOnly

            *-----------------------------------------------------
            * Undo get editing.
            *-----------------------------------------------------
        
            oGet:undo()
            
        end
    
    case ( nKey == K_HOME )

        oGet:home()

    case ( nKey == K_END )

        oGet:end()

    case ( nKey == K_RIGHT )

        oGet:right()

    case ( nKey == K_LEFT )

        oGet:left()

    case ( nKey == K_CTRL_RIGHT )

        oGet:wordRight()

    case ( nKey == K_CTRL_LEFT )

        oGet:wordLeft()

    case ( nKey == K_BS )
    
        *---------------------------------------------------------
        * Test before if not read only.
        *---------------------------------------------------------

        if !lReadOnly

            oGet:backSpace()
            
        end

    case ( nKey == K_DEL )
    
        *---------------------------------------------------------
        * Test before if not read only.
        *---------------------------------------------------------

        if !lReadOnly

            oGet:delete()
            
        end

    case ( nKey == K_CTRL_T )

        *---------------------------------------------------------
        * Test before if not read only.
        *---------------------------------------------------------

        if !lReadOnly

            oGet:delWordRight()
            
        end

    case ( nKey == K_CTRL_Y )

        *---------------------------------------------------------
        * Test before if not read only.
        *---------------------------------------------------------

        if !lReadOnly
    
            oGet:delEnd()
            
        end

    case ( nKey == K_CTRL_BS )
    
        *---------------------------------------------------------
        * Test before if not read only.
        *---------------------------------------------------------

        if !lReadOnly

            oGet:delWordLeft()
            
        end

    case ( nKey >= 32 .AND. nKey <= 255 )
        
        *---------------------------------------------------------
        * Test before if not read only.
        *---------------------------------------------------------

        if !lReadOnly

            *-----------------------------------------------------
            * A character key was pressed: 
            * Convert the inkey() code into the
            * corresponding character.
            *-----------------------------------------------------

            cKey := CHR( nKey )

            if ( oGet:type == "N" .AND. ( cKey == "." .OR. cKey == "," ) )
            
                *-------------------------------------------------
                * The oGet holds a numeric field and 
                * the character was a point or a comma:
                * jump to decimal position.
                *-------------------------------------------------

                oGet:toDecPos()

            else

                *-------------------------------------------------
                * Write the character.
                * If insert mode is ON, insert the character,
                * else overwrite the oGet text content.
                *-------------------------------------------------
            
                if ( set( _SET_INSERT ) )

                    oGet:insert( cKey )

                else

                    oGet:overstrike( cKey )

                end

                if ( oGet:typeOut )
                
                    *---------------------------------------------
                    * The typing reached the end:
                    * Ring the bell if set bell is ON and exit 
                    * the get if set confirm is OFF.
                    *---------------------------------------------

                    if ( set( _SET_BELL ) )

                        qqout( chr(7) )

                    end

                    if ( !set( _SET_CONFIRM ) )

                        oGet:exitState := GETEXIT_ENTER

                    end

                end

            end
            
        end

    end

    return NIL

*-----------------------------------------------------------------
static function getDoBlock( keyBlock, oGet )
*
* Process SET KEY during editing.
*

    local lSavUpdated

    *-------------------------------------------------------------
    * If editing has occurred, assign variable.
    *-------------------------------------------------------------
    
    if ( oGet:changed )

        oGet:assign()

        readUpdated( .T. )

    end

    *-------------------------------------------------------------
    * Save readUpdated() status
    *-------------------------------------------------------------
    
    lSavUpdated := readUpdated()

    *-------------------------------------------------------------
    * Evaluate code block.
    *-------------------------------------------------------------
    
    eval( keyBlock, oGet )

    oGet:updateBuffer()

    *-------------------------------------------------------------
    * Restore readUpdated() status
    *-------------------------------------------------------------
    
    readUpdated( lSavUpdated )

    *-------------------------------------------------------------
    * If readStop() (the original readkill) is True: stop
    * editing.
    *-------------------------------------------------------------
    
    if readStop()

        oGet:exitState := GETEXIT_ESCAPE   // provokes read() exit

    end

    return NIL

*-----------------------------------------------------------------
static function getMouse( aButtons, oGet, aGetPos )
*
*
    local aMouse    := mouse()

    local nMCol     := aMouse[1]-1
    local nMRow     := aMouse[2]-1

    local nI
    local nPos  := 0

    local aSameLine := {}

    local nButtRow
    local nButtCol
    local nButtColEnd

    *-------------------------------------------------------------
    * Start a new sequence.
    *-------------------------------------------------------------
    
    begin sequence

        *---------------------------------------------------------
        * Try to find a button.
        *---------------------------------------------------------

        if valtype(aButtons) == "A"

            for nI := 1 to len(aButtons)

                nButtRow    := aButtons[nI][BUTTON_ROW]
                nButtCol    := aButtons[nI][BUTTON_COL]
                nButtColEnd := nButtCol + len( aButtons[nI][BUTTON_TEXT] ) -1

                if  nButtRow == nMRow           .and.;
                    nButtCol <= nMCol           .and.;
                    nButtColEnd >= nMCol

                    *---------------------------------------------
                    * OK button pressed: do the correspondent 
                    * block, but before: 
                    * - hide the mouse cursor,;
                    * - stop mouse saveing.
                    * The last action is made to avoid to send
                    * a mouse action to a function called from
                    * the code block contained inside the
                    * button.
                    *---------------------------------------------
                    
                    setMouse(.F.)

                    mouse( .T. )

                    getDoBlock( aButtons[nI][BUTTON_BLOCK], oGet )

                    *---------------------------------------------
                    * Terminate.
                    *---------------------------------------------
                    
                    break                               // BREAK

                end

            next

        end

        *---------------------------------------------------------
        * If it wasn't a button, try to see if it was a get
        * field.
        *---------------------------------------------------------

        *---------------------------------------------------------
        * First: isolate fields on the same screen line.
        *---------------------------------------------------------

        for nI := 1 to len( aGetPos )

            if nMRow == aGetPos[nI][2]

                aadd( aSameLine, aGetPos[nI] )

            end

        next

        *---------------------------------------------------------
        * Note that <aGetPos> and consequently also <aSameLine>
        * are sorted. So, <aSameLine> contains 0, 1 or more
        * fiels starting from the left, ending to the right.
        *---------------------------------------------------------

        do case
        case empty( aSameLine )

            *-----------------------------------------------------
            * No fields on this line.
            *-----------------------------------------------------
            
            nPos := 0

        case len( aSameLine ) == 1

            *-----------------------------------------------------
            * Only one field on this line: select it.
            *-----------------------------------------------------

            nPos := aSameLine[1][1]

        case len( aSameLine ) > 1
            
            *-----------------------------------------------------
            * Search for a field near the mouse cursor.
            *-----------------------------------------------------

            for nI := 1 to len( aSameLine )
                do case
                case (nMCol - aSameLine[nI][3]) < 0

                    *---------------------------------------------
                    * The mouse cursor is BEFORE the field
                    * start. If a field exist before the active
                    * field, the previous is the right.
                    *---------------------------------------------
                    
                    if nI == 1

                        *-----------------------------------------
                        * There is no field before on this line:
                        * select the actual filed.
                        *-----------------------------------------
                        
                        nPos := aSameLine[nI][1]

                    else

                        *-----------------------------------------
                        * The previous field is selected.
                        *-----------------------------------------
                        
                        nPos := aSameLine[nI-1][1]
                    
                    end
                    
                    *---------------------------------------------
                    * Exit the FOR loop as the field was found.
                    *---------------------------------------------
                    
                    exit                                // EXIT

                case (nMCol - aSameLine[nI][3]) == 0

                    *---------------------------------------------
                    * This is exactly the right field.
                    *---------------------------------------------

                    nPos := aSameLine[nI][1]

                    exit                                // EXIT

                case; 
                    (nMCol - aSameLine[nI][3]) > 0      .and.;
                    nI == len( aSameLine )

                    *---------------------------------------------
                    * There are no more fields, so, select this
                    * one.
                    *---------------------------------------------

                    nPos := aSameLine[nI][1]

                    exit                                // EXIT

                otherwise

                    *---------------------------------------------
                    * Try with the next.
                    *---------------------------------------------
                    
                    loop                                // EXIT

                end

            next

        end

    end //sequence

    *-------------------------------------------------------------
    * Conclusion: return the field returned by the mouse.
    *-------------------------------------------------------------

    return nPos

*-----------------------------------------------------------------
static function getPostValidate( oGet )
*
* Test exit condition (VALID clause) for a GET.
*
* -->           True means the post validation condition is
*               true, otherwise, it is false.
*
* NOTE: Bad dates are rejected in such a way as to preserve
* edit buffer
*

    local lSavUpdated
    local lValid := .T.

    *-------------------------------------------------------------
    * If [Esc] or equivalent was used to exit the READ, the post
    * validation code block is not taken in to consideration. 
    *-------------------------------------------------------------
    
    if ( oGet:exitState == GETEXIT_ESCAPE )

        return ( .T. )                                  // RETURN
    
    end

    if ( oGet:badDate() )

        oGet:home()

        return ( .F. )                                  // RETURN

    end

    *-------------------------------------------------------------
    * If editing occurred, assign the new value to the variable.
    *-------------------------------------------------------------
    
    if ( oGet:changed )
        oGet:assign()
        readUpdated( .T. )
    end

    *-------------------------------------------------------------
    * Reform edit buffer, set cursor to home position, redisplay.
    *-------------------------------------------------------------

    oGet:reset()

    *-------------------------------------------------------------
    * Check VALID condition if specified
    *-------------------------------------------------------------

    if !( oGet:postBlock == NIL )

        *---------------------------------------------------------
        * Before the code block evaluation, save the readUpdated()
        * status.
        *---------------------------------------------------------

        lSavUpdated := readUpdated()

        lValid := eval( oGet:postBlock, oGet )

        oGet:updateBuffer()

        *---------------------------------------------------------
        * restore the readUpdated() status.
        *---------------------------------------------------------
        
        readUpdated( lSavUpdated )

        *---------------------------------------------------------
        * If readStop() (readkill) was set to True, the READ
        * must terminate.
        *---------------------------------------------------------
        
        if readStop()

            oGet:exitState := GETEXIT_ESCAPE    // Provokes read() exit

            lValid := .T.

        end

    end

    *-------------------------------------------------------------
    * The evaluation result is returned.
    *-------------------------------------------------------------
    
    return ( lValid )

*-----------------------------------------------------------------
static function getPreValidate( oGet )
*
* Test entry condition for a GET.
*

    local lSavUpdated
    local lWhen := .T.

    local nRow  := row()
    local nCol  := col()

    *-------------------------------------------------------------
    * The cursor is moved on the Get start position to permit
    * prevalidation code block to use this data.
    *-------------------------------------------------------------

    setpos( oGet:row, oGet:col )

    if !( oGet:preBlock == NIL )

        *---------------------------------------------------------
        * Before the code block evaluation, save the readUpdated()
        * status.
        *---------------------------------------------------------
        
        lSavUpdated := readUpdated()

        lWhen := eval( oGet:preBlock, oGet )

        oGet:display()

        *---------------------------------------------------------
        * restore the readUpdated() status.
        *---------------------------------------------------------
        
        readUpdated( lSavUpdated )

    end

    *-------------------------------------------------------------
    * If readStop() (readkill) was set to True, the READ
    * must terminate.
    *-------------------------------------------------------------
    
    if readStop()

        lWhen := .F.

        oGet:exitState := GETEXIT_ESCAPE     // Provokes read() exit

    elseif ( !lWhen )

        oGet:exitState := GETEXIT_WHEN         // Indicates failure

    else

        oGet:exitState := GETEXIT_NOEXIT       // Prepares for editing

    end

    *-------------------------------------------------------------
    * Restore the cursor position.
    *-------------------------------------------------------------
    
    setpos( nRow, nCol )

    *-------------------------------------------------------------
    * Return the prevalidation result.
    *-------------------------------------------------------------
    
    return ( lWhen )

*-----------------------------------------------------------------
static function getReader( oGet, aButtons, aGetPos, nPos, lReadOnly )
*
* getReader( <oGet>, <aButtons>, <aGetPos>, <nPos> ) --> NIL
*
* <oGet>       GET object to be read.
*
* <aGetPos>    sorted array of GET positions.
*
* <aButtons>   array of buttons.
*
* <nPos>       mouse position.
*
* <lReadOnly>  if True the get fields are read only.
*
*  Modal read of a single GET.
*

    local nMPos
    local nKey

    *-------------------------------------------------------------
    * Read the GET if the WHEN condition is satisfied.
    *-------------------------------------------------------------
    
    if ( GetPreValidate( oGet ) )

        *---------------------------------------------------------
        * Activate the GET for reading.
        *---------------------------------------------------------
        
        oGet:setFocus()

        while ( oGet:exitState == GETEXIT_NOEXIT )

            *-----------------------------------------------------
            * Check for initial typeout (no editable positions).
            *-----------------------------------------------------
            
            if ( oGet:typeOut )

                oGet:exitState := GETEXIT_ENTER
                
                *-------------------------------------------------
                * Exit is not needed here as it follows another
                * while ( oGet:exitState == GETEXIT_NOEXIT ) .
                *-------------------------------------------------

            end

            *-----------------------------------------------------
            * Apply keystrokes until exit or mouse action.
            *-----------------------------------------------------
            
            while ( oGet:exitState == GETEXIT_NOEXIT )

                *-------------------------------------------------
                * Show the mouse cursor.
                *-------------------------------------------------
                
                setMouse( .T. )

                *-------------------------------------------------
                * Read the mouse.
                *-------------------------------------------------
                
                nMPos := 0
                if !( mouse() == NIL )
                    nMPos := getMouse( aButtons, oGet, aGetPos )
                    mouse( .T. )
                end

                *-------------------------------------------------
                * nMPos == 0 means nothing or Button pressed,
                * nMPos > 0 means a field selected.
                *-------------------------------------------------

                if  nMPos > 0           .and.;
                    nMPos <> nPos

                    *---------------------------------------------
                    * The mouse has selected a different
                    * field that will be returned later.
                    *---------------------------------------------
                    
                    nPos := nMPos

                    *---------------------------------------------
                    * Now, it is better do hide the mouse.
                    *---------------------------------------------
                    
                    setMouse( .F. )

                    *---------------------------------------------
                    * Erase keyboard buffer.
                    *---------------------------------------------
                    
                    keyboard()

                    *---------------------------------------------
                    * Save changes.
                    *---------------------------------------------

                    if ( oGet:changed )

                        oGet:assign()

                        readUpdated( .T. )

                    end

                    *---------------------------------------------
                    * Prepare to exit loop
                    *---------------------------------------------

                    oGet:exitState := GETEXIT_ENTER

                else
                    
                    *---------------------------------------------
                    * Read the keyboard.
                    *---------------------------------------------

                    nKey := inkey()

                    *---------------------------------------------
                    * Test if a key was pressed or stuffed
                    * from a mouse action.
                    *---------------------------------------------
                    
                    if nKey <> 0

                        *-----------------------------------------
                        * A key was pressed or a mouse action was
                        * done: it is better do hide the mouse.
                        *-----------------------------------------

                        setMouse( .F. )

                        *-----------------------------------------
                        * Applay the key.
                        *-----------------------------------------

                        getApplyKey( oGet, nKey, lReadOnly )

                    end

                end

            end

            *-----------------------------------------------------
            * Disallow exit if the VALID condition
            * is not satisfied.
            *-----------------------------------------------------
            
            if ( !getPostValidate( oGet ) )

                oGet:exitState := GETEXIT_NOEXIT

            end

        end

        *---------------------------------------------------------
        * De-activate the GET.
        *---------------------------------------------------------
        
        oGet:killFocus()

    end

    *-------------------------------------------------------------
    * Return <nPos> that contains the position selecte with the
    * mouse if the mouse was used
    *-------------------------------------------------------------
    
    return nPos

*-----------------------------------------------------------------
static function getSettle(;
    aoGet, nPos,;
    lHitTop, lHitBottom,;
    nLastExitState, nLastPosition;
    )
*
* Returns new position in array of Get objects, based on:
*    - current position
*    - exitState of Get object at current position
*
* NOTES: return value of 0 indicates termination of READ
*        exitState of old Get is transferred to new Get
*

    local nExitState

    if ( nPos == 0 )
        nExitState := GETEXIT_DOWN
    else
        nExitState := aoGet[ nPos ]:exitState
    end

    *-------------------------------------------------------------
    * Terminate read if an exit key was pressed.
    *-------------------------------------------------------------
    
    if  nExitState == GETEXIT_ESCAPE    .or.; 
        nExitState == GETEXIT_WRITE

        return ( 0 )                                    // RETURN

    end

    *-------------------------------------------------------------
    * Tese if prevalidation condition failed.
    *-------------------------------------------------------------
    
    if !( nExitState == GETEXIT_WHEN )
        
        *---------------------------------------------------------
        * Reset state info.
        *---------------------------------------------------------

        nLastPosition := nPos
        lHitTop       := .F.
        lHitBottom    := .F.

    else

        *---------------------------------------------------------
        * Re-use last exitState, do not disturb state info.
        *---------------------------------------------------------

        nExitState := nLastExitState

    end

    *-------------------------------------------------------------
    * Move.
    *-------------------------------------------------------------
    
    do case
    case ( nExitState == GETEXIT_UP )

        nPos--

    case ( nExitState == GETEXIT_DOWN )

        nPos++

    case ( nExitState == GETEXIT_TOP )

        nPos       := 1
        lHitTop  := .T.
        nExitState := GETEXIT_DOWN

    case ( nExitState == GETEXIT_BOTTOM )

        nPos       := LEN( aoGet )
        lHitBottom  := .T.
        nExitState := GETEXIT_UP

    case ( nExitState == GETEXIT_ENTER )

        nPos++

    end

    *-------------------------------------------------------------
    * Bounce.
    *-------------------------------------------------------------
    
    if ( nPos == 0 )                       

        *---------------------------------------------------------
        * Bumped top.
        *---------------------------------------------------------

        if ( !set( _SET_EXIT ) .and. !lHitBottom )

            lHitTop  := .T.
            nPos       := nLastPosition
            nExitState := GETEXIT_DOWN

        end

    elseif ( nPos == len( aoGet ) + 1 )  
        
        *---------------------------------------------------------
        * Bumped bottom.
        *---------------------------------------------------------

        if  !set( _SET_EXIT )                   .and.; 
            !( nExitState == GETEXIT_ENTER )    .and.; 
            !lHitTop

            lHitBottom  := .T.
            nPos       := nLastPosition
            nExitState := GETEXIT_UP

        else

            nPos := 0

        end

    end

    *-------------------------------------------------------------
    * Record exit state.
    *-------------------------------------------------------------
    
    nLastExitState := nExitState

    if !( nPos == 0 )
        aoGet[ nPos ]:exitState := nExitState
    end

    return ( nPos )

*=================================================================
* RF()
*=================================================================

* Report array definitions
#define RF_P_HEADER    1  // Array of header strings
#define RF_P_WIDTH     2  // Numeric, report page width
#define RF_P_LMARGIN   3  // Numeric, report page offset
#define RF_P_RMARGIN   4  // NIL, Not used
#define RF_P_LINES     5  // Numeric, number of lines per page
#define RF_P_SPACING   6  // Numeric, single=1, double=2
#define RF_P_BEJECT    7  // Logical, eject before 1st page
#define RF_P_AEJECT    8  // Logical, eject after last page
#define RF_P_PLAIN     9  // Logical, plain report
#define RF_P_SUMMARY  10  // Logical, no detail lines
#define RF_P_COLUMNS  11  // Array of Column arrays
#define RF_P_GROUPS   12  // Array of Group arrays
#define RF_P_HEADING  13  // Character, heading for the report

#define RF_P_COUNT    13  // Number of elements in the Report array

* Column array definitions ( one array per column definition )
#define RF_C_EXP       1  // Block, contains compiled column exp.
#define RF_C_TEXT      2  // Character, contains text column exp.
#define RF_C_TYPE      3  // Character, type of expression
#define RF_C_HEADER    4  // Array of column heading strings
#define RF_C_WIDTH     5  // Numeric, column width including
                          // decimals and decimal point
#define RF_C_DECIMALS  6  // Numeric, number of decimal places
#define RF_C_TOTAL     7  // Logical, total this column
#define RF_C_PICT      8  // Character, picture string

#define RF_C_COUNT     8  // Number of elements in the Column array

* Group array definitions ( one array per group definition )
#define RF_G_EXP       1  // Block, contains compiled group exp.
#define RF_G_TEXT      2  // Character, contains text group exp.
#define RF_G_TYPE      3  // Character, type of expression
#define RF_G_HEADER    4  // Character, column heading string
#define RF_G_AEJECT    5  // Logical, eject after group

#define RF_G_COUNT     5  // Number of elements in the Group array

#define RF_MSG_PAGENO       1
#define RF_MSG_SUBTOTAL     2
#define RF_MSG_SUBSUBTOTAL  3
#define RF_MSG_TOTAL        4
#define RF_MSG_COLSEP       5
#define RF_MSG_LINESEP      6

*-----------------------------------------------------------------

* Definitions for buffer sizes
#define  RF_SIZE_FILE_BUFF      1990    // Size of report file
#define  RF_SIZE_LENGTHS_BUFF   110
#define  RF_SIZE_OFFSETS_BUFF   110
#define  RF_SIZE_EXPR_BUFF      1440
#define  RF_SIZE_FIELDS_BUFF    300
#define  RF_SIZE_PARAMS_BUFF    24

* Definitions for offsets into the FILE_BUFF string
#define  RF_LENGTHS_OFFSET      5      // Start of expression length array
#define  RF_OFFSETS_OFFSET      115    // Start of expression position array
#define  RF_EXPR_OFFSET         225    // Start of expression data area
#define  RF_FIELDS_OFFSET       1665   // Start of report columns (fields)
#define  RF_PARAMS_OFFSET       1965   // Start of report parameters block

* These are offsets into the FIELDS_BUFF string to actual values
* Values are added to a block offset FLD_OFFSET that is moved in
* increments of 12
#define  RF_FIELD_WIDTH_OFFSET      1
#define  RF_FIELD_TOTALS_OFFSET     6
#define  RF_FIELD_DECIMALS_OFFSET   7

* These are offsets into FIELDS_BUFF which are used to 'point' into
* the EXPR_BUFF string which contains the textual data
#define  RF_FIELD_CONTENT_EXPR_OFFSET   9
#define  RF_FIELD_HEADER_EXPR_OFFSET    11

* These are actual offsets into the PARAMS_BUFF string which
* are used to 'point' into the EXPR_BUFF string
#define  RF_PAGE_HDR_OFFSET     1
#define  RF_GRP_EXPR_OFFSET     3
#define  RF_SUB_EXPR_OFFSET     5
#define  RF_GRP_HDR_OFFSET      7
#define  RF_SUB_HDR_OFFSET      9

* These are actual offsets into the PARAMS_BUFF string to actual values
#define  RF_PAGE_WIDTH_OFFSET   11
#define  RF_LNS_PER_PAGE_OFFSET 13
#define  RF_LEFT_MRGN_OFFSET    15
#define  RF_RIGHT_MGRN_OFFSET   17
#define  RF_COL_COUNT_OFFSET    19
#define  RF_DBL_SPACE_OFFSET    21
#define  RF_SUMMARY_RPT_OFFSET  22
#define  RF_PE_OFFSET           23
#define  RF_OPTION_OFFSET       24

* File error definitions
#define  RF_F_OK                0       // No error
#define  RF_F_EMPTY             -3      // File is empty
#define  RF_F_ERROR             -1      // Some kind of error
#define  RF_F_NOEXIST           2       // File does not exist

*=================================================================
function rf(;
        cFrmName,;
        bForCondition,;
        bWhileCondition,;
        nNext,;
        nRecord,;
        lRest,;
        lPlain,;
        cHeading,;
        lBEject,;
        lSummary,;
        lDate,;
        acExtra;
    )
*
* rf( <cFRMName>,
*      [<bForCondition>],
*      [<bWhileCondition>],
*      [<nNext>],
*      [<nRecord>],
*      [<lRest>],
*      [<lPlain>],
*      [<cHeading>],
*      [<lBeforeEject>],
*      [<lSummary>],
*      [<lDate>],
*      [<acExtra>] )
*
* <cFRMName>           the form (.FRM) file to use to
*                      print the active Alias.
* <bForCondition>      code block for the FOR condition.
* <bWhileCondition>    code block for the WHILE condition.
* <nNext>              see REPORT FORM.
* <nRecord>            see REPORT FORM.
* <lRest>              see REPORT FORM.
* <lPlain>             if true (.T.), force the print
*                      in a simple way.
* <cHeading>           additional header in character or
*                      code block form.
*                      If a code block is sent, the final
*                      result must be a character string.
* <lBeforeEject>       if true (.T.), force a form feed
*                      before the print.
* <lSummary>           if true (.T.), force a summary print only.
* <lDate>              if false (.F.), force the print without
*                      date at the top of page.
* <acExtra>            a character array that may be used for
*                      translating standard printed report form
*                      words and to add vertical and horizontal
*                      separations. The default value of acExtra
*                      is:
*                      acExtra[1]  "Page No."
*                      acExtra[2]  "** Subtotal **"
*                      acExtra[3]  "* Subsubtotal *"
*                      acExtra[4]  "*** Total ***"
*                      acExtra[5]  " "     vertical column
*                                          separation
*                      axExtra[6]  ""      horizontal
*                                          separation: no
*                                          separation.
*
* This function does the same work of REPORT FORM or __ReportForm()
* or dbReportForm(), but it prints where qout() and qqout() print.
*
* RF() is obtained modifying the original system files for .FRM
* file print.
*

    local nCol
    local nGroup
    local xBreakVal
    local lBroke        := .F.
    local err

    local lAnyTotals
    local lAnySubTotals

    local aRepData
    local nPageNum          //@
    local nLinesLeft        //@
    local aReportTotals     //@
    local aGroupTotals      //@
    local lFirstPass        //@
    local nMaxLinesAvail

    local cLineSeparation := ""

    *-------------------------------------------------------------
    * Resolve parameters and set up defaults.
    *-------------------------------------------------------------

    if valtype( cFRMName ) == "C"

        cFRMName := strAddExtention( cFRMName, _EXTENTION_FORM )

    else

        return NIL                                      // RETURN

    end

    default( @bForCondition,    {||.T.} )
    default( @bWhileCondition,  {||.T.} )
    default( @nNext,            NIL )
    default( @nRecord,          NIL )
    default( @lRest,            .F. )
    default( @lPlain,           .F. )
    default( @cHeading,         "" )
    default( @lBEject,          .F. )
    default( @lSummary,         .F. )
    default( @lDate,            .T. )
    default(;
        @acExtra,;
        {;
            "Page No.",;
            "** Subtotal **",;
            "* Subsubtotal *",;
            "*** Total ***",;
            " ",;
            "";
        };
    )

    *-------------------------------------------------------------
    * Start a new sequence.
    *-------------------------------------------------------------

    begin sequence

        *---------------------------------------------------------
        * Load the frm into an array.
        *---------------------------------------------------------

        aRepData := rfFormLoad( cFRMName )

        nMaxLinesAvail := aRepData[RF_P_LINES]

        *---------------------------------------------------------
        * Modify <aRepData> based on the report parameters.
        *---------------------------------------------------------

        if lSummary == .T.

            *-----------------------------------------------------
            * Set the summary only flag.
            *-----------------------------------------------------

            aRepData[RF_P_SUMMARY] := lSummary

        end

        if lBEject == .T.

            *-----------------------------------------------------
            * Set the Before Eject flag.
            *-----------------------------------------------------

            aRepData[RF_P_BEJECT]  := .T.

        end

        if lPlain

            *-----------------------------------------------------
            * Set plain report flag.
            *-----------------------------------------------------

            aRepData[RF_P_PLAIN]    := .T.

            cHeading                := ""

        else

            *-----------------------------------------------------
            * Create the horizontal separation line.
            *-----------------------------------------------------

            if !(acExtra[RF_MSG_LINESEP] == "")

                for nCol := 1 to len(aRepData[RF_P_COLUMNS])

                    cLineSeparation +=;
                        replicate(;
                            acExtra[RF_MSG_LINESEP],;
                            aRepData[RF_P_COLUMNS, nCol, RF_C_WIDTH];
                        )

                    if nCol > 1
                        cLineSeparation +=;
                            acExtra[RF_MSG_LINESEP]

                    end

                next

            end

        end

        *---------------------------------------------------------
        * Copy the heading.
        *---------------------------------------------------------

        aRepData[ RF_P_HEADING ]  := cHeading

        *---------------------------------------------------------
        * Set the initial page number.
        *---------------------------------------------------------

        nPageNum := 1              

        *---------------------------------------------------------
        * Set the first pass flag.
        *---------------------------------------------------------

        lFirstPass  := .T.         

        *---------------------------------------------------------
        * At the beginning the available lines of the current
        * page are equel to the total lines.
        *---------------------------------------------------------

        nLinesLeft  := aRepData[ RF_P_LINES ]

        *---------------------------------------------------------
        * Check to see if a "before report" eject,
        * or TO FILE has been specified
        *---------------------------------------------------------

        if aRepData[RF_P_BEJECT]

            *-----------------------------------------------------
            * Eject Page.
            *-----------------------------------------------------

            if !aRepData[RF_P_PLAIN]

                *-------------------------------------------------
                * Eject page sendig the FF (form feed) code.
                *-------------------------------------------------

                qqout( FF )

            end

        end

        *---------------------------------------------------------
        * Generate the initial report header manually
        * (in case there are no
        * records that match the report scope)
        *---------------------------------------------------------

        rfReportHeader(;
            aRepData,;
            @aReportTotals,;
            @aGroupTotals,;
            @nPageNum,;
            @nLinesLeft,;
            @nMaxLinesAvail,;
            acExtra,;
            lDate,;
            cLineSeparation;
        )

        *---------------------------------------------------------
        * Initialize aReportTotals to track both
        * group and report totals, then
        * set the column total elements to 0
        * if they are to be totaled, otherwise
        * leave them NIL.
        *---------------------------------------------------------

        aReportTotals :=;
            array(;
                len(aRepData[RF_P_GROUPS]) + 1,;
                len(aRepData[RF_P_COLUMNS]);
            )

        *---------------------------------------------------------
        * Column total elements.
        *---------------------------------------------------------

        for nCol := 1 to len(aRepData[RF_P_COLUMNS])

            if aRepData[RF_P_COLUMNS,nCol,RF_C_TOTAL]

                for nGroup := 1 to len(aReportTotals)

                    aReportTotals[nGroup,nCol] := 0

                next

            end

        next

        *---------------------------------------------------------
        * Initialize aGroupTotals as an array.
        *---------------------------------------------------------

        aGroupTotals := ARRAY( LEN(aRepData[RF_P_GROUPS]) )

        *---------------------------------------------------------
        * Execute the actual report based on matching records.
        *---------------------------------------------------------

        dbEval(;
            { ||;
                rfExecuteReport(;
                    aRepData,;
                    @aReportTotals,;
                    @aGroupTotals,;
                    @nPageNum,;
                    @lFirstPass,;
                    @nLinesLeft,;
                    @nMaxLinesAvail,;
                    acExtra,;
                    lDate,;
                    cLineSeparation;
                );
            },;
            bForCondition,;
            bWhileCondition,;
            nNext,;
            nRecord,;
            lRest;
        )

        *---------------------------------------------------------
        * Generate any totals that may have been identified
        * Make a pass through all the groups.
        *---------------------------------------------------------

        for nGroup := len(aRepData[RF_P_GROUPS]) to 1 step -1

            *-----------------------------------------------------
            * Make sure group has subtotals.
            *-----------------------------------------------------

            lAnySubTotals := .F.

            for nCol := 1 to len(aRepData[RF_P_COLUMNS])

                if aRepData[RF_P_COLUMNS,nCol,RF_C_TOTAL]

                    lAnySubTotals := .T.

                    exit                                // EXIT

                end

            next

            if !lAnySubTotals

                loop                                    // LOOP

            end

            *-----------------------------------------------------
            * Check to see if we need to eject the page.
            *-----------------------------------------------------

            if nLinesLeft < 2

                if aRepData[ RF_P_PLAIN ]

                    nLinesLeft := 1000

                else

                    *---------------------------------------------
                    * Eject Page.
                    *---------------------------------------------

                    qqout( FF )

                    *---------------------------------------------
                    * Print header.
                    *---------------------------------------------

                    rfReportHeader(;
                        aRepData,;
                        @aReportTotals,;
                        @aGroupTotals,;
                        @nPageNum,;
                        @nLinesLeft,;
                        @nMaxLinesAvail,;
                        acExtra,;
                        lDate,;
                        cLineSeparation;
                    )

                end

            end

            *-----------------------------------------------------
            * Print the first line.
            *-----------------------------------------------------

            rfPrintIt(;
                space(aRepData[RF_P_LMARGIN]) + ;
                iif(;
                    nGroup==1,;
                    acExtra[RF_MSG_SUBTOTAL],;
                    acExtra[RF_MSG_SUBSUBTOTAL];
                );
            )

            *-----------------------------------------------------
            * Print the second line.
            *-----------------------------------------------------

            qqout( space(aRepData[RF_P_LMARGIN]) )
            
            for nCol := 1 to len(aRepData[RF_P_COLUMNS])

                if nCol > 1

                    *---------------------------------------------
                    * Add a column spacing.
                    *---------------------------------------------

                    qqout( " " )

                end

                if aRepData[RF_P_COLUMNS,nCol,RF_C_TOTAL]

                    qqout(;
                        transform( aReportTotals[nGroup+1,nCol],;
                        aRepData[RF_P_COLUMNS,nCol,RF_C_PICT]);
                    )

                else

                    qqout(;
                        space(;
                            aRepData[RF_P_COLUMNS,nCol,RF_C_WIDTH];
                        );
                    )

                end

            next

            *-----------------------------------------------------
            * Send a cr/lf for the last line.
            *-----------------------------------------------------

            qout()

        next

        *---------------------------------------------------------
        * Any report totals?
        *---------------------------------------------------------

        lAnyTotals := .F.

        for nCol := 1 to len(aRepData[RF_P_COLUMNS])

            if aRepData[RF_P_COLUMNS,nCol,RF_C_TOTAL]

                lAnyTotals := .T.

                exit                                    // EXIT

            end

        next

        if lAnyTotals

            *-----------------------------------------------------
            * Check to see if we need to eject the page.
            *-----------------------------------------------------

            if nLinesLeft < 2

                if aRepData[ RF_P_PLAIN ]

                    nLinesLeft := 1000

                else

                    *---------------------------------------------
                    * Eject Page.
                    *---------------------------------------------

                    qqout( FF )

                    *---------------------------------------------
                    * Print header.
                    *---------------------------------------------

                    rfReportHeader()

                end

            end

            *-----------------------------------------------------
            * Print the first line.
            *-----------------------------------------------------

            rfPrintIt(;
                space(aRepData[RF_P_LMARGIN]);
                    + acExtra[RF_MSG_TOTAL];
            )

            *-----------------------------------------------------
            * Print the second line
            *-----------------------------------------------------

            qqout( space(aRepData[RF_P_LMARGIN]) )

            for nCol := 1 to len(aRepData[RF_P_COLUMNS])

                if nCol > 1

                    *---------------------------------------------
                    * Add a column spacing.
                    *---------------------------------------------

                    qqout( " " )

                end

                if aRepData[RF_P_COLUMNS,nCol,RF_C_TOTAL]

                    qqout(;
                        transform(aReportTotals[1,nCol], ;
                        aRepData[RF_P_COLUMNS,nCol,RF_C_PICT]);
                    )

                else

                    qqout(;
                        space(aRepData[RF_P_COLUMNS,nCol,RF_C_WIDTH]);
                    )

                end

            next

            *-----------------------------------------------------
            * Send a cr/lf for the last line.
            *-----------------------------------------------------

            qout()

        end

        *---------------------------------------------------------
        * Check to see if an "after report" eject,
        * or TO FILE has been specified.
        *---------------------------------------------------------

        if aRepData[RF_P_AEJECT]

            *-----------------------------------------------------
            * Eject Page
            *-----------------------------------------------------

            if !aRepData[RF_P_PLAIN]

                qqout( FF )

            end

        end

    recover using xBreakVal
    
       lBroke := .T.
       
    end //sequence

    if lBroke

        *---------------------------------------------------------
        * Keep the break value going.
        *---------------------------------------------------------

        break xBreakVal

    end

    return NIL

*----------------------------------------------------------------
static function rfExecuteReport(;
        aRepData,;
        aReportTotals,;
        aGroupTotals,;
        nPageNum,;
        lFirstPass,;
        nLinesLeft,;
        nMaxLinesAvail,;
        acExtra,;
        lDate,;
        cLineSeparation;
    )

    local aRecordHeader  := {}
    local aRecordToPrint := {}
    local nCol
    local nGroup
    local lGroupChanged  := .F.
    local lEjectGrp := .F.
    local nMaxLines
    local nLine
    local cLine
    local nLastElement

    local lAnySubTotals

    *-------------------------------------------------------------
    * Add to the main column totals.
    *-------------------------------------------------------------

    for nCol := 1 to len(aRepData[RF_P_COLUMNS])

        if aRepData[RF_P_COLUMNS,nCol,RF_C_TOTAL]

            *-----------------------------------------------------
            * If this column should be totaled, do it.
            *-----------------------------------------------------

            aReportTotals[ 1 ,nCol] += ;
                eval( aRepData[RF_P_COLUMNS,nCol,RF_C_EXP] )

        end

    next

    *-------------------------------------------------------------
    * Determine if any of the groups have changed.
    * If so, add the appropriate
    * line to aRecordHeader for totaling out the previous records.
    *-------------------------------------------------------------

    if !lFirstPass

        *---------------------------------------------------------
        * Don't bother first time through.
        *---------------------------------------------------------

        *---------------------------------------------------------
        * Make a pass through all the groups.
        *---------------------------------------------------------

        for nGroup := len(aRepData[RF_P_GROUPS]) to 1 step -1

            *-----------------------------------------------------
            * Make sure group has subtotals.
            *-----------------------------------------------------

            lAnySubTotals := .F.

            for nCol := 1 to len(aRepData[RF_P_COLUMNS])

                if aRepData[RF_P_COLUMNS,nCol,RF_C_TOTAL]

                    lAnySubTotals := .T.

                    exit                                // EXIT

                end

            next

            *-----------------------------------------------------
            * Retrieve group eject state from report form.
            *-----------------------------------------------------

            if ( nGroup == 1 )

                lEjectGrp :=;
                    aRepData[RF_P_GROUPS,nGroup,RF_G_AEJECT]

            end

            if !lAnySubTotals

                loop                                    // LOOP

            end

            *-----------------------------------------------------
            * For subgroup processing: check if group
            * has been changed.
            *-----------------------------------------------------

            if  strXToString(;
                    eval(aRepData[RF_P_GROUPS, 1, RF_G_EXP]),;
                    aRepData[RF_P_GROUPS, 1, RF_G_TYPE];
                ) <> aGroupTotals[1]

                lGroupChanged  := .T.

            end

            *-----------------------------------------------------
            * If this (sub)group has changed since the last record.
            *-----------------------------------------------------

            if  lGroupChanged                   .or.;
                strXToString(;
                    eval( aRepData[RF_P_GROUPS, nGroup, RF_G_EXP] ),;
                    aRepData[RF_P_GROUPS,nGroup,RF_G_TYPE];
                ) != aGroupTotals[nGroup]

                aadd(;
                    aRecordHeader,;
                    iif( nGroup==1,;
                        acExtra[RF_MSG_SUBTOTAL],;
                        acExtra[RF_MSG_SUBSUBTOTAL];
                        );
                    )
                aadd( aRecordHeader, "" )

                *-------------------------------------------------
                * Cycle through the columns, adding either
                * the group amount from aReportTotals or
                * spaces wide enough for the non-totaled columns.
                *-------------------------------------------------

                for nCol := 1 to len(aRepData[RF_P_COLUMNS])

                    if  aRepData[RF_P_COLUMNS,nCol,RF_C_TOTAL]

                        aRecordHeader[ LEN(aRecordHeader) ] += ;
                            transform(;
                                aReportTotals[nGroup+1,nCol],;
                                aRepData[RF_P_COLUMNS,nCol,RF_C_PICT];
                            )

                        *-----------------------------------------
                        * Zero out the group totals column from
                        * <aReportTotals>.
                        *-----------------------------------------

                        aReportTotals[nGroup+1,nCol] := 0

                    else
                        aRecordHeader[ len(aRecordHeader) ] += ;
                            space( aRepData[RF_P_COLUMNS, nCol, RF_C_WIDTH] )

                    end

                    aRecordHeader[ LEN(aRecordHeader) ] += " "

                next

                *-------------------------------------------------
                * Get rid of the extra space from the last column.
                *-------------------------------------------------

                aRecordHeader[len(aRecordHeader)] := ;
                    left(;
                        aRecordHeader[len(aRecordHeader)], ;
                        len( aRecordHeader[len(aRecordHeader)] ) - 1;
                    )
            end

        next

    end

    if  len( aRecordHeader ) > 0            .and.;
        lEjectGrp                           .and.;
        lGroupChanged

        if len( aRecordHeader ) > nLinesLeft

            if (aRepData[RF_P_PLAIN])

                nLinesLeft := 1000

            else

                *-------------------------------------------------
                * Eject Page.
                *-------------------------------------------------

                qqout( FF )

                *-------------------------------------------------
                * Print header.
                *-------------------------------------------------

                rfReportHeader(;
                    aRepData,;
                    @aReportTotals,;
                    @aGroupTotals,;
                    @nPageNum,;
                    @nLinesLeft,;
                    @nMaxLinesAvail,;
                    acExtra,;
                    lDate,;
                    cLineSeparation;
                )

            end

        end

        aeval(;
            aRecordHeader,;
            { |HeaderLine| ;
                rfPrintIt(;
                    space( aRepData[ RF_P_LMARGIN ] ) +;
                    HeaderLine;
                );
            };
        )

        aRecordHeader := {}

        if ( aRepData[RF_P_PLAIN] )

            nLinesLeft := 1000
        else

            *-----------------------------------------------------
            * Eject Page.
            *-----------------------------------------------------

            qqout( FF )

            *-----------------------------------------------------
            * Print header.
            *-----------------------------------------------------

            rfReportHeader(;
                aRepData,;
                @aReportTotals,;
                @aGroupTotals,;
                @nPageNum,;
                @nLinesLeft,;
                @nMaxLinesAvail,;
                acExtra,;
                lDate,;
                cLineSeparation;
            )

        end

    end

    *-------------------------------------------------------------
    * Add to <aRecordHeader> in the event that the group
    * has changed and new group headers need to be generated.
    *-------------------------------------------------------------

    *-------------------------------------------------------------
    * Cycle through the groups.
    *-------------------------------------------------------------
    for nGroup := 1 to len(aRepData[RF_P_GROUPS])

        *---------------------------------------------------------
        * If the group has changed.
        *---------------------------------------------------------

        if  strXToString(;
                eval( aRepData[RF_P_GROUPS,nGroup,RF_G_EXP] ),;
                aRepData[RF_P_GROUPS,nGroup,RF_G_TYPE];
            ) == aGroupTotals[nGroup]

            *-----------------------------------------------------
            * The group is not changed: do nothing.
            *-----------------------------------------------------

        else

            *-----------------------------------------------------
            * Add a blank line before a new group.
            *-----------------------------------------------------

            aadd( aRecordHeader, "" )

            *-----------------------------------------------------
            * Page eject after group:
            * put CR+FF after group.
            *-----------------------------------------------------

            if  nGroup == 1                     .and.;
                !lFirstPass                     .and.;
                !lAnySubTotals

                if  lEjectGrp :=;
                        aRepData[RF_P_GROUPS,nGroup,RF_G_AEJECT]

                    nLinesLeft  := 0

                end

            end

            aadd(;
                aRecordHeader,;
                rtrim(;
                    aRepData[RF_P_GROUPS,nGroup,RF_G_HEADER];
                ) + " ";
                + strXToString(;
                    eval(;
                        aRepData[RF_P_GROUPS,nGroup,RF_G_EXP];
                    ),;
                    aRepData[RF_P_GROUPS,nGroup,RF_G_TYPE];
                );
            )

            *-----------------------------------------------------
            * Add line separation.
            *-----------------------------------------------------

            if !(cLineSeparation == "")

                aadd( aRecordHeader, cLineSeparation )

            end

        end

    next

    lFirstPass := .F.

    *-------------------------------------------------------------
    * Is there anything in the record header?
    *-------------------------------------------------------------

    if len( aRecordHeader ) > 0

        *---------------------------------------------------------
        * Determine if aRecordHeader will fit
        * on the current page.  If not,
        * start a new header.
        *---------------------------------------------------------

        if len( aRecordHeader ) > nLinesLeft

            if aRepData[RF_P_PLAIN]

                nLinesLeft := 1000

            else

                *-------------------------------------------------
                * Eject Page.
                *-------------------------------------------------

                qqout( FF )

                *-------------------------------------------------
                * Print header.
                *-------------------------------------------------

                rfReportHeader(;
                    aRepData,;
                    @aReportTotals,;
                    @aGroupTotals,;
                    @nPageNum,;
                    @nLinesLeft,;
                    @nMaxLinesAvail,;
                    acExtra,;
                    lDate,;
                    cLineSeparation;
                )

            end

        end

        *---------------------------------------------------------
        * Send aRecordHeader to the output device,
        * resetting <nLinesLeft>.
        *---------------------------------------------------------

        aeval(;
            aRecordHeader,;
            { |HeaderLine|;
                rfPrintIt( space(aRepData[RF_P_LMARGIN]) + HeaderLine );
            };
        )

        nLinesLeft -= len( aRecordHeader )

        *---------------------------------------------------------
        * Make sure it didn't hit the bottom margin.
        *---------------------------------------------------------

        if nLinesLeft == 0

            if aRepData[RF_P_PLAIN]

                nLinesLeft := 1000

            else

                *-------------------------------------------------
                * Eject Page.
                *-------------------------------------------------

                qqout( FF )

                *-------------------------------------------------
                * Print header.
                *-------------------------------------------------

                rfReportHeader(;
                    aRepData,;
                    @aReportTotals,;
                    @aGroupTotals,;
                    @nPageNum,;
                    @nLinesLeft,;
                    @nMaxLinesAvail,;
                    acExtra,;
                    lDate,;
                    cLineSeparation;
                )

            end

        end

    end

    *-------------------------------------------------------------
    * Add to the group totals.
    *-------------------------------------------------------------

    for nCol := 1 to len(aRepData[RF_P_COLUMNS])

        *---------------------------------------------------------
        * If this column should be totaled, do it.
        *---------------------------------------------------------

        if aRepData[RF_P_COLUMNS,nCol,RF_C_TOTAL]

            *-----------------------------------------------------
            * Cycle through the groups.
            *-----------------------------------------------------

            for nGroup := 1 to len( aReportTotals ) - 1

                aReportTotals[nGroup+1,nCol] += ;
                    eval( aRepData[RF_P_COLUMNS,nCol,RF_C_EXP] )

            next

        end

    next

    *-------------------------------------------------------------
    * Reset the group expressions in <aGroupTotals>.
    *-------------------------------------------------------------

    for nGroup := 1 to len(aRepData[RF_P_GROUPS])

        aGroupTotals[nGroup] :=;
            strXToString(;
                eval(aRepData[RF_P_GROUPS,nGroup,RF_G_EXP]),;
                aRepData[RF_P_GROUPS,nGroup,RF_G_TYPE];
            )

    next

    *-------------------------------------------------------------
    * Only run through the record detail if this is NOT
    * a summary report.
    *-------------------------------------------------------------

    if !aRepData[ RF_P_SUMMARY ]

        *---------------------------------------------------------
        * Determine the max number of lines needed
        * by each expression.
        *---------------------------------------------------------

        nMaxLines := 1

        for nCol := 1 to len(aRepData[RF_P_COLUMNS])

            if aRepData[RF_P_COLUMNS,nCol,RF_C_TYPE] $ "M"

                nMaxLines :=;
                    max(;
                        mlcount(;
                            rtrim(;
                                eval( aRepData[RF_P_COLUMNS, nCol, RF_C_EXP]);
                            ),;
                            aRepData[RF_P_COLUMNS,nCol,RF_C_WIDTH];
                        ),;
                        nMaxLines;
                    )

            elseif aRepData[RF_P_COLUMNS,nCol,RF_C_TYPE] $ "C"

                nMaxLines :=;
                   max(;
                       mlcount(;
                           rtrim(;
                               strtran(;
                                   eval(;
                                       aRepData[RF_P_COLUMNS, nCol, RF_C_EXP];
                                   ),;
                                   ";",;
                                   NL(1);
                               );
                           ),;
                           aRepData[RF_P_COLUMNS, nCol, RF_C_WIDTH];
                       ),;
                       nMaxLines;
                   )
            end

        next

        *---------------------------------------------------------
        * Size aRecordToPrint to the maximum number of lines
        * it will need, then fill it with nulls.
        *---------------------------------------------------------

        asize( aRecordToPrint, nMaxLines )

        afill( aRecordToPrint, "" )

        *---------------------------------------------------------
        * Load the current record into <aRecordToPrint>.
        *---------------------------------------------------------

        for nCol := 1 to len(aRepData[RF_P_COLUMNS])

            for nLine := 1 to nMaxLines

                *-------------------------------------------------
                * Check to see if it's a memo or character.
                *-------------------------------------------------

                if aRepData[RF_P_COLUMNS,nCol,RF_C_TYPE] $ "CM"

                    *---------------------------------------------
                    * Load the current line of the current
                    * column into cLine with multi-lines
                    * per record ";"- method.
                    *---------------------------------------------

                    if aRepData[RF_P_COLUMNS,nCol,RF_C_TYPE] $ "C"

                        cLine :=;
                            memoline(;
                                rtrim(;
                                    strtran(;
                                        eval(;
                                            aRepData[RF_P_COLUMNS, nCol, RF_C_EXP];
                                        ),;
                                        ";", NL(1);
                                    );
                                ),;
                                aRepData[RF_P_COLUMNS, nCol, RF_C_WIDTH],;
                                nLine;
                            )

                    else

                        cLine :=;
                            memoline(;
                                rtrim(;
                                    eval(;
                                        aRepData[RF_P_COLUMNS, nCol, RF_C_EXP];
                                    );
                                ),;
                                aRepData[RF_P_COLUMNS, nCol, RF_C_WIDTH],;
                                nLine;
                            )

                    end

                    cLine :=;
                        padr( cLine, aRepData[RF_P_COLUMNS,nCol,RF_C_WIDTH] )

                else

                    if nLine == 1

                        cLine :=;
                            transform(;
                                eval(;
                                    aRepData[RF_P_COLUMNS, nCol, RF_C_EXP];
                                ),;
                                aRepData[RF_P_COLUMNS,nCol,RF_C_PICT];
                            )

                        cLine := padr( cLine, aRepData[RF_P_COLUMNS,nCol,RF_C_WIDTH] )

                    else

                        cLine :=;
                            space(;
                                aRepData[RF_P_COLUMNS,nCol,RF_C_WIDTH];
                            )

                    end

                end

                *-------------------------------------------------
                * Add it to the existing report line.
                *-------------------------------------------------

                if nCol > 1

                    aRecordToPrint[ nLine ] += acExtra[RF_MSG_COLSEP]

                end

                aRecordToPrint[ nLine ] += cLine

            next

        next

        *---------------------------------------------------------
        * Add line separation.
        *---------------------------------------------------------

        if !(cLineSeparation == "")
            aadd( aRecordToPrint, cLineSeparation )
        end

        *---------------------------------------------------------
        * Determine if aRecordToPrint will fit on the current page.
        *---------------------------------------------------------

        if len( aRecordToPrint ) > nLinesLeft

            *-----------------------------------------------------
            * The record will not fit on the current page.
            * Will it fit on a full page?
            * If not, break it up and print it.
            *-----------------------------------------------------

            if len( aRecordToPrint ) > nMaxLinesAvail

                *-------------------------------------------------
                * This record is HUGE!  Break it up...
                *-------------------------------------------------

                nLine := 1

                while nLine < len( aRecordToPrint )

                    rfPrintIt(;
                        space(aRepData[RF_P_LMARGIN]) +;
                        aRecordToPrint[nLine];
                    )

                    nLine++

                    nLinesLeft--

                    if nLinesLeft == 0

                        if aRepData[RF_P_PLAIN]

                            nLinesLeft := 1000

                        else

                        *-----------------------------------------
                        * Eject Page.
                        *-----------------------------------------

                        qqout( FF )

                        *-----------------------------------------
                        * Print header.
                        *-----------------------------------------

                        rfReportHeader(;
                            aRepData,;
                            @aReportTotals,;
                            @aGroupTotals,;
                            @nPageNum,;
                            @nLinesLeft,;
                            @nMaxLinesAvail,;
                            acExtra,;
                            lDate,;
                            cLineSeparation;
                        )

                        end

                    end

                end

            else

                if aRepData[RF_P_PLAIN]

                    nLinesLeft := 1000

                else

                    *---------------------------------------------
                    * Eject Page.
                    *---------------------------------------------

                    qqout( FF )

                    *---------------------------------------------
                    * Print header.
                    *---------------------------------------------

                    rfReportHeader(;
                        aRepData,;
                        @aReportTotals,;
                        @aGroupTotals,;
                        @nPageNum,;
                        @nLinesLeft,;
                        @nMaxLinesAvail,;
                        acExtra,;
                        lDate,;
                        cLineSeparation;
                    )

                end

                aeval(;
                    aRecordToPrint,;
                    { |RecordLine| ;
                        rfPrintIt(;
                            space(aRepData[RF_P_LMARGIN]) + RecordLine;
                        );
                    };
                )

                nLinesLeft -= LEN( aRecordToPrint )

            end

        else

            *-----------------------------------------------------
            * Send aRecordToPrint to the output device,
            * resetting nLinesLeft.
            *-----------------------------------------------------

            aeval(;
                aRecordToPrint, ;
                { |RecordLine| ;
                    rfPrintIt(;
                        space(aRepData[RF_P_LMARGIN]) + RecordLine );
                };
            )

            nLinesLeft -= len( aRecordToPrint )

        end

        *---------------------------------------------------------
        * Tack on the spacing for double/triple/etc.
        *---------------------------------------------------------

        if aRepData[ RF_P_SPACING ] > 1

            *-----------------------------------------------------
            * Double space problem in REPORT FORM
            * at the bottom of the page.
            *-----------------------------------------------------

            if nLinesLeft >= aRepData[ RF_P_SPACING ] - 1

                for nLine := 2 to aRepData[ RF_P_SPACING ]

                    rfPrintIt()

                    nLinesLeft--

                next

            end

        end


    end    // Was this a summary report?

    return NIL

*-----------------------------------------------------------------
static function rfReportHeader(;
        aRepData,;
        aReportTotals,;
        aGroupTotals,;
        nPageNum,;
        nLinesLeft,;
        nMaxLinesAvail,;
        acExtra,;
        lDate,;
        cLineSeparation;
    )

    local nLinesInHeader := 0
    local aPageHeader    := {}
    local nHeadingLength :=;
        aRepData[RF_P_WIDTH] - aRepData[RF_P_LMARGIN] - 30

    local nCol
    local nLine
    local nMaxColLength
    local nGroup
    local cHeader
    local cPage
    local cDate
    local cTop
    local cMoreHeading

    local nHeadLine     // lines in a single heading
    local nRPageSize    // width of report after subtracting right margin
    local aTempPgHeader // temporary page header array
    local nHeadSize

    nRPageSize := aRepData[RF_P_WIDTH] - aRepData[RF_P_RMARGIN]

    *-------------------------------------------------------------
    * Header width should be less then 255 characters.
    *-------------------------------------------------------------

    nHeadSize := min(nRPageSize, 254)

    *-------------------------------------------------------------
    * Create the header and drop it into <aPageHeader>.
    *-------------------------------------------------------------

    *-------------------------------------------------------------
    * Start with the heading.
    *-------------------------------------------------------------

    if !aRepData[RF_P_PLAIN]

        *---------------------------------------------------------
        * If not a plain paper report, build.
        *---------------------------------------------------------

        cPage :=;
            acExtra[RF_MSG_PAGENO] + " " + ltrim( str(nPageNum,6) )

        cDate := iif( lDate, dtoc( date() ), "" )

        cTop :=;
           padr(;
               cDate,;
               nHeadSize - len(cPage) - aRepData[RF_P_LMARGIN];
           );
           + cPage

        aadd( aPageHeader, cTop )

        *---------------------------------------------------------
        * Test if the header is a code block.
        *---------------------------------------------------------

        do case
        case valtype( aRepData[RF_P_HEADING] ) == "B"

            cMoreHeading := eval( aRepData[RF_P_HEADING] )

        case valtype( aRepData[RF_P_HEADING] ) == "C"

            cMoreHeading := aRepData[RF_P_HEADING]

        otherwise

            cMoreHeading := ""

        end

        if !empty( cMoreHeading )

            aTempPgHeader :=;
                rfParseHeader(;
                    cMoreHeading, ;
                    strOccurs( ";", cMoreHeading ) + 1;
                )

            for nLine := 1 to len( aTempPgHeader )

                *-------------------------------------------------
                * Determine number of lines in header given
                * current report dimensions.
                *-------------------------------------------------

                nLinesInHeader :=;
                    max(;
                        mlcount(;
                            alltrim( aTempPgHeader[ nLine ] ),;
                            nHeadSize - aRepData[RF_P_LMARGIN];
                        ),;
                        1;
                    )

                *-------------------------------------------------
                * Extract lines and add to array.
                *-------------------------------------------------

                for nHeadLine := 1 to nLinesInHeader

                    aadd(;
                        aPageHeader,;
                        padc(;
                            rtrim(;
                                memoline(;
                                    ltrim( aTempPgHeader[nLine] ),;
                                    nHeadSize - aRepData[RF_P_LMARGIN],;
                                    nHeadLine;
                                );
                            ),;
                            nHeadSize - aRepData[RF_P_LMARGIN];
                        );
                    )

                next

            next

        end

    end

    *-------------------------------------------------------------
    * Tack on the actual header from the FRM.
    *-------------------------------------------------------------

    for nLine := 1 to len( aRepData[RF_P_HEADER] )

        *---------------------------------------------------------
        * Determine number of lines in header given current
        * report dimensions.
        *---------------------------------------------------------

        nLinesInHeader :=;
            max(;
                mlcount(;
                    alltrim( aRepData[RF_P_HEADER, nLine] ),;
                    nHeadSize - aRepData[RF_P_LMARGIN];
                ),;
                1;
            )

        *---------------------------------------------------------
        * Extract lines and add to array.
        *---------------------------------------------------------

        for nHeadLine := 1 to nLinesInHeader
            cHeader :=;
                rtrim(;
                    memoline(;
                        ltrim( aRepData[RF_P_HEADER, nLine] ),;
                        nHeadSize - aRepData[RF_P_LMARGIN], nHeadLine;
                    );
                )

            aadd(;
                aPageHeader,;
                padc(;
                    cHeader,;
                    nHeadSize - aRepData[RF_P_LMARGIN], nHeadLine;
                );
            )

        next

    next

    *-------------------------------------------------------------
    * Add a blank line between the .FRM header
    * and the columns.
    *-------------------------------------------------------------

    aadd( aPageHeader, "" )

    *-------------------------------------------------------------
    * Now add the column headings.
    *-------------------------------------------------------------

    nLinesInHeader := len( aPageHeader )

    *-------------------------------------------------------------
    * Determine the longest column header.
    *-------------------------------------------------------------

    nMaxColLength := 0

    for nCol := 1 to len( aRepData[ RF_P_COLUMNS ] )

        nMaxColLength :=;
            max(;
                len(aRepData[RF_P_COLUMNS,nCol,RF_C_HEADER]),;
                nMaxColLength;
            )

    next

    for nCol := 1 to len( aRepData[RF_P_COLUMNS] )

        asize(;
            aRepData[RF_P_COLUMNS,nCol,RF_C_HEADER],;
            nMaxColLength;
        )

    next

    for nLine := 1 to (nMaxColLength)

        aadd( aPageHeader, "" )

    next

    for nCol := 1 to len(aRepData[RF_P_COLUMNS])

        *---------------------------------------------------------
        * Cycle through the columns.
        *---------------------------------------------------------

        for nLine := 1 to nMaxColLength

            if nCol > 1

                aPageHeader[ nLinesInHeader + nLine ] +=;
                    acExtra[RF_MSG_COLSEP]

            end

            if aRepData[RF_P_COLUMNS,nCol,RF_C_HEADER,nLine] == NIL

                aPageHeader[ nLinesInHeader + nLine ] += ;
                    space( aRepData[RF_P_COLUMNS,nCol,RF_C_WIDTH] )

            else

                if aRepData[RF_P_COLUMNS,nCol,RF_C_TYPE] == "N"

                    aPageHeader[ nLinesInHeader + nLine ] += ;
                        padl(;
                            aRepData[RF_P_COLUMNS, nCol, RF_C_HEADER, nLine],;
                            aRepData[RF_P_COLUMNS,nCol,RF_C_WIDTH];
                        )

                else

                    aPageHeader[ nLinesInHeader + nLine ] += ;
                           padr(;
                               aRepData[RF_P_COLUMNS, nCol, RF_C_HEADER, nLine],;
                               aRepData[RF_P_COLUMNS, nCol, RF_C_WIDTH];
                           )

                end

            end

        next

    next

    *-------------------------------------------------------------
    * Add line separation.
    *-------------------------------------------------------------

    if !(cLineSeparation == "")

        aadd( aPageHeader, cLineSeparation )

    end

    *-------------------------------------------------------------
    * At this position two blank lines are normally inserted
    * between the heading and the actual data. - I don't like it!
    *   aadd( aPageHeader, "" )
    *   aadd( aPageHeader, "" )
    *-------------------------------------------------------------

    aeval(;
        aPageHeader,;
        { | HeaderLine | ;
            rfPrintIt( space(aRepData[RF_P_LMARGIN])+ HeaderLine );
        };
    )

    *-------------------------------------------------------------
    * Set the page number and number of available lines.
    *-------------------------------------------------------------

    nPageNum++

    nLinesLeft := aRepData[RF_P_LINES] - len( aPageHeader )

    nMaxLinesAvail := aRepData[RF_P_LINES] - len( aPageHeader )

    return NIL

*-----------------------------------------------------------------
static function rfPrintIt( cString )

    default( @cString,  "" )

    *-------------------------------------------------------------
    * Print <cString> starting from the actual position and then
    * go to a new line.
    *-------------------------------------------------------------

    qqout( cString )

    qout()

    return NIL

*-----------------------------------------------------------------
static function rfParseHeader( cHeaderString, nFields )

    local cItem
    local nItemCount := 0
    local aPageHeader := {}
    local nHeaderLen := 254
    local nPos

    while ( ++nItemCount <= nFields )

        cItem := substr( cHeaderString, 1, nHeaderLen )

        *---------------------------------------------------------
        * Check for explicit delimiter.
        *---------------------------------------------------------

        nPos := at( ";", cItem )

        if !empty( nPos )

            *-----------------------------------------------------
            * Delimiter present.
            *-----------------------------------------------------

            aadd( aPageHeader, substr( cItem, 1, nPos - 1 ) )

        else

            if empty( cItem )

                aadd( aPageHeader, "" )

            else

                aadd( aPageHeader, cItem )

            end

            *-----------------------------------------------------
            * Empty or not, we jump past the field
            *-----------------------------------------------------

            nPos := nHeaderLen

        end

        cHeaderString := substr( cHeaderString, nPos + 1 )

    end

    return aPageHeader


*-----------------------------------------------------------------
static function rfFormLoad( cFrmFile )
*
*  rfFormLoad( <cFrmFile> ) --> aReport
*
*  Create a report array from a (.frm) file
*
*  Reads a report (.frm) file and creates a report array
*
*  Notes:
*
*   Report file name has extension.
*
*   File error number placed in nFileError
*
*   Offsets start at 1. Offsets are into a Clipper string, 1 to 1990
*
*   The offsets mentioned in these notes are actual DOS FILE offsets,
*   not like the offsets declared in the body of FrmLoad()
*   which are Clipper STRING offsets.
*
*   Report file length is 7C6h (1990d) bytes.
*
*   Expression length array starts at 04h (4d) and can
*   contain upto 55 short (2 byte) numbers.
*
*   Expression offset index array starts at 72h (114d) and
*   can contain upto 55 short (2 byte) numbers.
*
*   Expression area starts at offset E0h (224d).
*
*   Expression area length is 5A0h (1440d).
*
*   Expressions in expression area are null terminated.
*
*   Field expression area starts at offset 680h (1664d).
*
*   Field expressions (column definition) are null terminated.
*
*   Field expression area can contain upto 25 12-byte blocks.
*
*
*

    local cFieldsBuff
    local cParamsBuff
    local nFieldOffset      := 0
    local cFileBuff         := SPACE(RF_SIZE_FILE_BUFF)
    local cGroupExp         := SPACE(200)
    local cSubGroupExp      := SPACE(200)
    local nColCount         := 0    // Number of columns in report
    local nCount
    local nFrmHandle                // (.frm) file handle
    local nBytesRead                // Read/write and content record counter
    local nPointer          := 0    // Points to an offset into EXPR_BUFF string
    local nFileError                // Contains current file error
    local cOptionByte               // Contains option byte

    local aReport[ RF_P_COUNT ]     // Create report array
    local err                       // error object

    local cDefPath                  // contents of SET DEFAULT string
    local aPaths                    // array of paths
    local nPathIndex := 0           // iteration counter

    local s
    local paths
    local i
	local aHeader				    // temporary storage for report form headings
	local nHeaderIndex		        // index into temporary header array

    // The following where originally filewide STATIC buffers.
    local cLengthsBuff  := ""       //@
    local cOffsetsBuff  := ""       //@
    local cExprBuff     := ""       //@

    *-------------------------------------------------------------
    * Default report values.
    *-------------------------------------------------------------

    aReport[ RF_P_HEADER ]    := {}
    aReport[ RF_P_WIDTH ]     := 80
    aReport[ RF_P_LMARGIN ]   := 8
    aReport[ RF_P_RMARGIN ]   := 0
    aReport[ RF_P_LINES ]     := 58
    aReport[ RF_P_SPACING ]   := 1
    aReport[ RF_P_BEJECT ]    := .T.
    aReport[ RF_P_AEJECT ]    := .F.
    aReport[ RF_P_PLAIN ]     := .F.
    aReport[ RF_P_SUMMARY ]   := .F.
    aReport[ RF_P_COLUMNS ]   := {}
    aReport[ RF_P_GROUPS ]    := {}
    aReport[ RF_P_HEADING ]   := ""

    *-------------------------------------------------------------
    * Open the report file.
    *-------------------------------------------------------------

    nFrmHandle := fopen( cFrmFile )

    if  !empty( nFileError := ferror() )            .and.;
        !( "\" $ cFrmFile .or. ":" $ cFrmFile )

        *---------------------------------------------------------
        * Search through default path; attempt to open report file.
        *---------------------------------------------------------

        cDefPath := set( _SET_DEFAULT ) + ";" + set( _SET_PATH )

        cDefPath := strtran( cDefPath, ",", ";" )

        aPaths := strListAsArray( cDefPath, ";" )

        for nPathIndex := 1 to len( aPaths )

            nFrmHandle :=;
                fopen( aPaths[ nPathIndex ] + "\" + cFrmFile )

            *-----------------------------------------------------
            * If no error is reported, we have our report file.
            *-----------------------------------------------------

            if empty( nFileError := ferror() )

                exit                                        // EXIT

            end

        next

    end

    *-------------------------------------------------------------
    * File error.
    *-------------------------------------------------------------

    if nFileError != RF_F_OK

        err := ErrorNew()

        err:severity := ERROR_SEVERITY_ERROR
        err:genCode := ERROR_GENERIC_OPEN
        err:subSystem := "FRMLBL"
        err:osCode := nFileError
        err:filename := cFrmFile

        eval(errorBlock(), err)

    end

    *-------------------------------------------------------------
    * OPEN ok?
    *-------------------------------------------------------------

    if nFileError == RF_F_OK

        *---------------------------------------------------------
        * Go to START of report file.
        *---------------------------------------------------------

        fseek( nFrmHandle, 0 )

        *---------------------------------------------------------
        * SEEK ok?
        *---------------------------------------------------------

        nFileError := ferror()

        if nFileError == RF_F_OK

            *-----------------------------------------------------
            * Read entire file into process buffer.
            *-----------------------------------------------------

            nBytesRead :=;
                fread( nFrmHandle, @cFileBuff, RF_SIZE_FILE_BUFF )

            *-----------------------------------------------------
            * READ ok?
            *-----------------------------------------------------

            if nBytesRead == 0

                *-------------------------------------------------
                * The file is empty.
                *-------------------------------------------------

                nFileError := RF_F_EMPTY        

            else

                *-------------------------------------------------
                * Check for DOS errors
                *-------------------------------------------------

                nFileError := ferror()

            end

            if nFileError == RF_F_OK

                *-------------------------------------------------
                * Is this a .FRM type file (2 at start and end of file)
                *-------------------------------------------------

                if  bin2w( substr( cFileBuff, 1, 2 ) ) == 2;
                    .and.;
                    bin2w( substr( cFileBuff, RF_SIZE_FILE_BUFF -1, 2) ) == 2

                    nFileError := RF_F_OK

                else

                   nFileError := RF_F_ERROR

                end

            end

        end

        *---------------------------------------------------------
        * Close file.
        *---------------------------------------------------------

        if !FCLOSE(nFrmHandle)

            nFileError := FERROR()

        end

    end

    *-------------------------------------------------------------
    * File existed, was opened and read ok and is a .FRM file
    *-------------------------------------------------------------

    if nFileError == RF_F_OK

        *---------------------------------------------------------
        * Fill processing buffers
        *---------------------------------------------------------

        cLengthsBuff := SUBSTR(cFileBuff, RF_LENGTHS_OFFSET, RF_SIZE_LENGTHS_BUFF)
        cOffsetsBuff := SUBSTR(cFileBuff, RF_OFFSETS_OFFSET, RF_SIZE_OFFSETS_BUFF)
        cExprBuff    := SUBSTR(cFileBuff, RF_EXPR_OFFSET, RF_SIZE_EXPR_BUFF)
        cFieldsBuff  := SUBSTR(cFileBuff, RF_FIELDS_OFFSET, RF_SIZE_FIELDS_BUFF)
        cParamsBuff  := SUBSTR(cFileBuff, RF_PARAMS_OFFSET, RF_SIZE_PARAMS_BUFF)


        *---------------------------------------------------------
        * Process report attributes
        * Report width
        *---------------------------------------------------------

        aReport[ RF_P_WIDTH ]   := bin2w(SUBSTR(cParamsBuff, RF_PAGE_WIDTH_OFFSET, 2))

        *---------------------------------------------------------
        * Lines per page
        *---------------------------------------------------------

        aReport[ RF_P_LINES ]   := bin2w(SUBSTR(cParamsBuff, RF_LNS_PER_PAGE_OFFSET, 2))

        *---------------------------------------------------------
        * Page offset (left margin)
        *---------------------------------------------------------

        aReport[ RF_P_LMARGIN ] := bin2w(SUBSTR(cParamsBuff, RF_LEFT_MRGN_OFFSET, 2))

        *---------------------------------------------------------
        * Page right margin (not used)
        *---------------------------------------------------------

        aReport[ RF_P_RMARGIN ] := bin2w(substr(cParamsBuff, RF_RIGHT_MGRN_OFFSET, 2))

        nColCount  := bin2w(substr(cParamsBuff, RF_COL_COUNT_OFFSET, 2))

        *---------------------------------------------------------
        * Line spacing
        * Spacing is 1, 2, or 3
        *---------------------------------------------------------

        aReport[ RF_P_SPACING ] :=;
            iif(;
                IsAffirm( substr(cParamsBuff, RF_DBL_SPACE_OFFSET, 1) ),;
                2,;
                1;
            )

        *---------------------------------------------------------
        * Summary report flag
        *---------------------------------------------------------

        aReport[ RF_P_SUMMARY ] :=;
            iif( IsAffirm( substr(cParamsBuff, RF_SUMMARY_RPT_OFFSET, 1) ),;
                .T.,;
                .F.;
            )

        *---------------------------------------------------------
        * Process report eject and plain attributes option byte.
        *---------------------------------------------------------

        cOptionByte := ASC(substr(cParamsBuff, RF_OPTION_OFFSET, 1))

        if int( cOptionByte / 4 ) == 1

            *-----------------------------------------------------
            * Plain page.
            *-----------------------------------------------------

            aReport[ RF_P_PLAIN ] := .T.

            cOptionByte -= 4

        end

        if int( cOptionByte / 2 ) == 1

            *-----------------------------------------------------
            * Page eject after report.
            *-----------------------------------------------------

            aReport[ RF_P_AEJECT ] := .T.

            cOptionByte -= 2
        end

        if int( cOptionByte / 1 ) == 1

            *-----------------------------------------------------
            * Page eject before report.
            *-----------------------------------------------------

            aReport[ RF_P_BEJECT ] := .F.

            cOptionByte -= 1

        end

        *---------------------------------------------------------
        * Page heading, report title.
        *---------------------------------------------------------

        nPointer := bin2w(substr(cParamsBuff, RF_PAGE_HDR_OFFSET, 2))

        *---------------------------------------------------------
	    * Retrieve the header stored in the .FRM file.
        *---------------------------------------------------------

        nHeaderIndex := 4

        aHeader := rfParseHeader( rfGetExpr( nPointer, @cLengthsBuff, @cOffsetsBuff, @cExprBuff ), nHeaderIndex )

        *---------------------------------------------------------
	    * Certain that we have retrieved all heading entries from
	    * the .FRM file, we now retract the empty headings.
        *---------------------------------------------------------

        while ( nHeaderIndex > 0 )

            if !empty( aHeader[ nHeaderIndex ] )

                exit                                    // EXIT

            end

            nHeaderIndex--

        end

        aReport[ RF_P_HEADER ] :=;
            iif(;
                empty( nHeaderIndex ),;
                {},;
                asize( aHeader, nHeaderIndex );
            )

        *---------------------------------------------------------
        * Process Groups
        * Group
        *---------------------------------------------------------

        nPointer := bin2w(substr(cParamsBuff, RF_GRP_EXPR_OFFSET, 2))

        if !empty(cGroupExp := rfGetExpr( nPointer, @cLengthsBuff, @cOffsetsBuff, @cExprBuff ))

            *-----------------------------------------------------
            * Add a new group array.
            *-----------------------------------------------------

            aadd( aReport[ RF_P_GROUPS ], ARRAY( RF_G_COUNT ))

            *-----------------------------------------------------
            * Group expression.
            *-----------------------------------------------------

            aReport[ RF_P_GROUPS ][1][ RF_G_TEXT ] := cGroupExp

            aReport[ RF_P_GROUPS ][1][ RF_G_EXP ] := &( "{ || " + cGroupExp + "}" )

            if used()

                aReport[ RF_P_GROUPS ][1][ RF_G_TYPE ] := ;
                    valtype( eval( aReport[ RF_P_GROUPS ][1][ RF_G_EXP ] ) )

            end

            *-----------------------------------------------------
            * Group header
            *-----------------------------------------------------

            nPointer = bin2w(substr(cParamsBuff, RF_GRP_HDR_OFFSET, 2))

            aReport[ RF_P_GROUPS ][1][ RF_G_HEADER ] := rfGetExpr( nPointer, @cLengthsBuff, @cOffsetsBuff, @cExprBuff )

            *-----------------------------------------------------
            * Page eject after group.
            *-----------------------------------------------------

            aReport[ RF_P_GROUPS ][1][ RF_G_AEJECT ] :=;
                iif(IsAffirm(substr(cParamsBuff, RF_PE_OFFSET, 1)), .T., .F.)

        end

        *---------------------------------------------------------
        * Subgroup
        *---------------------------------------------------------

        nPointer = bin2w(substr(cParamsBuff, RF_SUB_EXPR_OFFSET, 2))

        if !empty(cSubGroupExp := rfGetExpr( nPointer, @cLengthsBuff, @cOffsetsBuff, @cExprBuff ))

            *-----------------------------------------------------
            * Add new group array.
            *-----------------------------------------------------

            aadd( aReport[ RF_P_GROUPS ], ARRAY( RF_G_COUNT ))

            *-----------------------------------------------------
            * Subgroup expression.
            *-----------------------------------------------------

            aReport[ RF_P_GROUPS ][2][ RF_G_TEXT ] := cSubGroupExp

            aReport[ RF_P_GROUPS ][2][ RF_G_EXP ] := &( "{ || " + cSubGroupExp + "}" )

            if used()

                aReport[ RF_P_GROUPS ][2][ RF_G_TYPE ] := ;
                    valtype( eval( aReport[ RF_P_GROUPS ][2][ RF_G_EXP ] ) )

            end

            *-----------------------------------------------------
            * Subgroup header.
            *-----------------------------------------------------

            nPointer = bin2w(substr(cParamsBuff, RF_SUB_HDR_OFFSET, 2))

            aReport[ RF_P_GROUPS ][2][ RF_G_HEADER ] := rfGetExpr( nPointer, @cLengthsBuff, @cOffsetsBuff, @cExprBuff )

            *-----------------------------------------------------
            * Page eject after subgroup
            *-----------------------------------------------------

            aReport[ RF_P_GROUPS ][2][ RF_G_AEJECT ] := .F.

        end

        *---------------------------------------------------------
        * Process columns.
        *---------------------------------------------------------

        nFieldOffset := 12      // dBASE skips first 12 byte fields block.

        for nCount := 1 to nColCount

            aadd( aReport[ RF_P_COLUMNS ], rfGetColumn( cFieldsBuff, @nFieldOffset, @cLengthsBuff, @cOffsetsBuff, @cExprBuff ) )

        next

    end

    return aReport

*-----------------------------------------------------------------
static function rfGetExpr( nPointer, cLengthsBuff, cOffsetsBuff, cExprBuff )
*
* rfGetExpr( <nPointer>, @<cLengthsBuff>, @<cOffsetsBuff>, @<cExprBuff> ) --> cString
*
* Reads an expression from EXPR_BUFF via the OFFSETS_BUFF
* and returns a pointer to offset contained in OFFSETS_BUFF
* that in turn points to an expression located in the EXPR_BUFF
* string.
*
*  Notes:
*
*   The expression is empty if:
*
*       Passed pointer is equal to 65535
*
*       Character following character pointed to by pointer is CHR(0)
*

    local nExprOffset   := 0
    local nExprLength   := 0
    local nOffsetOffset := 0
    local cString := ""

    *-------------------------------------------------------------
    * Stuff for dBASE compatability.
    *-------------------------------------------------------------

    if nPointer != 65535

        *---------------------------------------------------------
        * Convert DOS FILE offset to CLIPPER string offset.
        *---------------------------------------------------------

        nPointer++

        *---------------------------------------------------------
        * Calculate offset into OFFSETS_BUFF.
        *---------------------------------------------------------

        if nPointer > 1

            nOffsetOffset = (nPointer * 2) - 1

        end

        nExprOffset = bin2w(substr(cOffsetsBuff, nOffsetOffset, 2))

        nExprLength = bin2w(substr(cLengthsBuff, nOffsetOffset, 2))

        *---------------------------------------------------------
        * RF_EXPR_OFFSET points to a NULL, so add one (+1) to get
        * the string and subtract one (-1) from EXPR_LENGTH for
        * correct length
        *---------------------------------------------------------

        nExprOffset++

        nExprLength--

        *---------------------------------------------------------
        * Extract string.
        *---------------------------------------------------------

        cString := substr(cExprBuff, nExprOffset, nExprLength)

        *---------------------------------------------------------
        * dBASE does this so we must do it too
        * Character following character pointed to by pointer is
        * NULL
        *---------------------------------------------------------

        if  chr(0) == substr(cString, 1, 1)         .and.;
            len(substr(cString,1,1)) == 1

            cString = ""

        end

    end

    return cString


*-----------------------------------------------------------------
static function rfGetColumn( cFieldsBuffer, nOffset, cLengthsBuff, cOffsetsBuff, cExprBuff )
*
* rfGetColumn( <cFieldBuffer>, @<nOffset>, @<cLengthsBuff>, @<cOffsetsBuff>, @<cExprBuff> ) --> aColumn
*
* Get a COLUMN element from FIELDS_BUFF string using <nOffset>
* to point to the current RF_FIELDS_OFFSET block.
*
* Notes:
*
*   The Header or Contents expressions are empty if:
*
*       Passed pointer is equal to 65535
*
*       Character following character pointed to by pointer
*       is CHR(0)
*
*

    local nPointer  := 0
    local nNumber   := 0
    local aColumn[ RF_C_COUNT ]
    local cType

    *-------------------------------------------------------------
    * Column width.
    *-------------------------------------------------------------

    aColumn[ RF_C_WIDTH ] :=;
        bin2w( substr(cFieldsBuffer, nOffset + RF_FIELD_WIDTH_OFFSET, 2) )

    *-------------------------------------------------------------
    * Total column?
    *-------------------------------------------------------------

    aColumn[ RF_C_TOTAL ] :=;
        iif(;
            IsAffirm(substr(cFieldsBuffer, nOffset + RF_FIELD_TOTALS_OFFSET, 1)),;
            .T.,;
            .F.;
        )

    *-------------------------------------------------------------
    * Decimals width.
    *-------------------------------------------------------------

    aColumn[ RF_C_DECIMALS ] :=;
        bin2w(substr(cFieldsBuffer, nOffset + RF_FIELD_DECIMALS_OFFSET, 2))

    *-------------------------------------------------------------
    * Offset (relative to RF_FIELDS_OFFSET), 'point' to
    * expression area via array OFFSETS[].
    *-------------------------------------------------------------

    *-------------------------------------------------------------
    * Content expression.
    *-------------------------------------------------------------

    nPointer :=;
        bin2w(substr(cFieldsBuffer, nOffset + RF_FIELD_CONTENT_EXPR_OFFSET, 2))

    aColumn[ RF_C_TEXT ] := rfGetExpr( nPointer, @cLengthsBuff, @cOffsetsBuff, @cExprBuff )

    aColumn[ RF_C_EXP ] := &( "{ || " + rfGetExpr( nPointer, @cLengthsBuff, @cOffsetsBuff, @cExprBuff ) + "}" )

    *-------------------------------------------------------------
    * Header expression.
    *-------------------------------------------------------------

    nPointer :=;
        bin2w(substr(cFieldsBuffer, nOffset + RF_FIELD_HEADER_EXPR_OFFSET, 2))

    aColumn[ RF_C_HEADER ] :=;
        strListAsArray(rfGetExpr( nPointer, @cLengthsBuff, @cOffsetsBuff, @cExprBuff ), ";")

    *-------------------------------------------------------------
    * Column picture.
    * Setup picture only if a database file is open.
    *-------------------------------------------------------------

    if used()

        cType := VALTYPE( EVAL(aColumn[ RF_C_EXP ]) )

        aColumn[ RF_C_TYPE ] := cType

        do case
        case cType = "C" .OR. cType = "M"

            aColumn[ RF_C_PICT ] := REPLICATE("X", aColumn[ RF_C_WIDTH ])

        case cType = "D"

            aColumn[ RF_C_PICT ] := "@D"

        case cType = "N"

            if  aColumn[ RF_C_DECIMALS ] != 0

                aColumn[ RF_C_PICT ] :=;
                    replicate( "9", aColumn[ RF_C_WIDTH ] - aColumn[ RF_C_DECIMALS ] -1) +;
                    "." + ;
                    replicate( "9", aColumn[ RF_C_DECIMALS ] )

            else

                aColumn[ RF_C_PICT ] := REPLICATE("9", aColumn[ RF_C_WIDTH ])

            end

        case cType = "L"

            aColumn[ RF_C_PICT ] := "@L" + REPLICATE("X",aColumn[ RF_C_WIDTH ]-1)

        end

    end

    *-------------------------------------------------------------
    * Update offset into ?_buffer.
    *-------------------------------------------------------------

    nOffset += 12

    return aColumn

*=================================================================
* RPT()
* RPTMANY()
* RPTTRANSLATE()
*=================================================================

#define RPT_DEFAULT_COMMAND_SYMBOL      "*"

#define RPT_DEFAULT_LPP                 60

#define RPT_ERROR_PARAMETER_EVAL_EMPTY;
    "The required parameter is missing."

#define RPT_ERROR_PRINT_COMMAND_UNCOMPLETED;
    "Print command uncompleted."

#define RPT_ERROR_PRINT_FILE_NOT_FOUND;          
    "File not found."

#define RPT_ERROR_PRINT_ISOLATED_END;
    RPT_DEFAULT_COMMAND_SYMBOL + "END isolated."

#define RPT_ERROR_PARENTESES;    
    "Error with " + chr(174) + chr(175) + " delimiters."

*=================================================================
function rpt(; 
        cText,; 
        nTotalLines, lMain, lHeadFoot,;
        nLineLength, nTabSize, lWrap, cSymbol; 
    )
*
* rpt( <cText>, [<nTotalLines>], [<lMain>], [<lHeadFoot>],
*      [<nLineLength>], [<nTabSize>], [<lWrap>], [<cSymbol>] )
*      --> NIL
*
* Prints with the nB report standard the text contained into
* <cText> once.
*
* As this function is recursive, it is important to transfer the
* exact line dimention to the recursive call.
*
* <cText>              the text to be printed.
*
* <nTotalLines>        <cText> lines length, used with recursive
*                      call.
* <lMain>              true (.t.) means that it is the first
*                      call and not recursive. True is the
*                      default value.
* <lHeadFoot>          true (.t.) means that <cText> is a header
*                      or footer.
* <nLineLength>        the maximal line length.
* <nTabSize>           tab length.
* <lWrap>              true (.t.) means that the text will be
*                      wrapped if necessary.
* <cSymbol>            the prefix simbol for report commands.
*

    local nLine               := 0
    local cLine               := ""

    local cSubText            := ""
    local nSubTextLines       := 0
    local cCommand            := ""
    local cCondition          := ""
    local nParameter          := 0
    local cParameter          := ""
    local xParameter
    local nNeeded             := 0

    local nOldRow             := row()
    local nOldCol             := col()

    static nLPP
    static nPrintedLine
    static cHeader
    static nHeadLines
    static cFooter
    static nFootLines
    static cLeft
    static lNewPage

    *-------------------------------------------------------------
    * <nLineLength> must be set befor line count.
    *-------------------------------------------------------------
    
    default( @nLineLength,  _MAX_STRING_LEN )

    default(; 
        @nTotalLines,;
        mlcount( cText, nLineLength, nTabSize, lWrap ); 
    )

    *-------------------------------------------------------------
    * <lMain> main call for the entire document      
    *-------------------------------------------------------------
    
    default(; 
        @lMain,;
        .T.; 
    ) 

    *-------------------------------------------------------------
    * <lHeadFoot> if .T. it is a head or foot print.
    *-------------------------------------------------------------
    
    default(; 
        @lHeadFoot,;
        .F.; 
    )

    default( @cSymbol,      RPT_DEFAULT_COMMAND_SYMBOL )

    *-------------------------------------------------------------
    * Start a new sequence.
    *-------------------------------------------------------------
    
    begin sequence

        *---------------------------------------------------------
        * If it is the first call.
        *---------------------------------------------------------
        
        if lMain == .T. .or. lMain == NIL

            *-----------------------------------------------------
            * Static variable initialization.
            *-----------------------------------------------------
            
            nLPP         := NIL
            nPrintedLine := NIL
            cHeader      := NIL
            nHeadLines   := NIL
            cFooter      := NIL
            nFootLines   := NIL
            cLeft        := NIL
            lNewPage     := NIL

        end

        *---------------------------------------------------------
        * Static defaults.
        *---------------------------------------------------------
        
        default( @nLPP,                 RPT_DEFAULT_LPP )

        default( @nPrintedLine,         0 )

        default( @cHeader,              "" )

        default( @nHeadLines,           0 )

        default( @cFooter,              "" )

        default( @nFootLines,           0 ) 
        
        default( @cLeft,                "" ) 
        
        default( @lNewPage,             .T. )   

        *---------------------------------------------------------
        * If there are no more lines to print.
        *---------------------------------------------------------
        
        if nTotalLines <= 0

            break                                       // BREAK

        end

        *---------------------------------------------------------
        * Main loop.
        *---------------------------------------------------------
        
        for nLine := 1 to nTotalLines

            *-----------------------------------------------------
            * Line conuter correction.
            *-----------------------------------------------------

            if nPrintedLine >= nLPP

                nPrintedLine := 0

            end

            *-----------------------------------------------------
            * Remaining lines calculation.
            *-----------------------------------------------------
            
            setRptLines( nLPP - nFootLines - nPrintedLines )

            *-----------------------------------------------------
            * Next line.
            *-----------------------------------------------------
            
            cLine :=;
                memoline(; 
                    cText, nLineLength, nLine, nTabSize, lWrap; 
                )

            *-----------------------------------------------------
            * Line analisys.
            *-----------------------------------------------------
            
            do case
            case upper( alltrim(cLine) ) = cSymbol+"REM"

                *-------------------------------------------------
                * Comment, the line is ignored.
                *-------------------------------------------------
            
            case upper( alltrim(cLine) ) = cSymbol+"COMMENT"

                *-------------------------------------------------
                * Comment, the line is ignored.
                *-------------------------------------------------
            
            case upper( alltrim(cLine) ) = cSymbol+"END"

                *-------------------------------------------------
                * If it appears here, it is an error,
                * the line is ignored.
                *-------------------------------------------------

                alertBox( RPT_ERROR_PRINT_ISOLATED_END )

            case upper( alltrim(cLine) ) = cSymbol+"COMMAND"

                *-------------------------------------------------
                * Start a command line.
                * The command lines are isolated.
                *-------------------------------------------------
                
                cCommand :=;
                    rptReadToEnd(; 
                        cText, @nLine, NIL, nLineLength,;
                        nTabSize, lWrap, cSymbol; 
                    )

                execute( cCommand, "REPORT COMMAND")

            case upper( alltrim(cLine) ) = cSymbol+"IF"

                cCondition :=;
                    alltrim(; 
                        substr( ltrim(cLine), 3+len(cSymbol) ); 
                    )

                cSubText :=;
                    rptReadToEnd(; 
                        cText, @nLine, @nSubTextLines,;
                        nLineLength, nTabSize, lWrap, cSymbol; 
                    )

                if cCondition == ""

                    *---------------------------------------------
                    * No condition.
                    *---------------------------------------------
                    
                    alertBox(; 
                        alltrim(cLine) + NL(1) +;
                        RPT_ERROR_PRINT_COMMAND_UNCOMPLETED; 
                        )

                    *---------------------------------------------
                    * The line is ignored.
                    *---------------------------------------------

                else

                    *---------------------------------------------
                    * Test condition.
                    *---------------------------------------------
                    
                    if rptEval( cCondition, nLine )

                        rpt(; 
                            cSubText, nSubTextLines, .F., .F.,;
                            nLineLength, nTabSize, lWrap, cSymbol; 
                        )

                    end

                end

            case upper( alltrim(cLine) ) = cSymbol+"WHILE"

                cCondition :=;
                    alltrim(; 
                        substr( ltrim(cLine), 6+len(cSymbol) ); 
                    )

                cSubText :=;
                    rptReadToEnd(; 
                        cText, @nLine, @nSubTextLines,;
                        nLineLength, nTabSize, lWrap, cSymbol; 
                    )

                if cCondition == ""

                    *---------------------------------------------
                    * No condition.
                    *---------------------------------------------
                    
                    alertBox(; 
                        alltrim(cLine) + NL(1) +;
                        RPT_ERROR_PRINT_COMMAND_UNCOMPLETED; 
                        )

                    *---------------------------------------------
                    * The line is ignored.
                    *---------------------------------------------
                
                else

                    *---------------------------------------------
                    * Test condition.
                    *---------------------------------------------
                    
                    while rptEval( cCondition, nLine ) 

                        RPT(; 
                            cSubText, nSubTextLines, .F., .F.,;
                            nLineLength, nTabSize, lWrap, cSymbol; 
                        )
                    
                    end

                end

            case upper( alltrim(cLine) ) = cSymbol+"DBSKIP"

                *-------------------------------------------------
                * dbskip()
                * the eof() and bof() check MUST be done inside
                * the document!!!
                *-------------------------------------------------
                
                cParameter :=;
                    alltrim(; 
                        substr( ltrim(cLine), 7+len(cSymbol) ); 
                    )

                *-------------------------------------------------
                * Default skip is +1.
                *-------------------------------------------------
                
                if cParameter == ""

                    dbskip(+1)

                else

                    nParameter := rptEval( cParameter, nLine ) 

                    dbskip( nParameter )

                end

            case upper( alltrim(cLine) ) = cSymbol+"LPP"

                *-------------------------------------------------
                * Lines per page.
                *-------------------------------------------------
                
                cParameter :=;
                    alltrim(; 
                        substr( ltrim(cLine), 4+len(cSymbol) ); 
                    )

                if cParameter == ""

                    *---------------------------------------------
                    * No parameter is given.
                    *---------------------------------------------
                    
                    alertBox(; 
                        alltrim(cLine) + NL(1) +;
                        RPT_ERROR_PRINT_COMMAND_UNCOMPLETED; 
                    )

                    *---------------------------------------------
                    * Line ignored.
                    *---------------------------------------------
                
                else

                    nParameter := rptEval( cParameter, nLine ) 

                    nLPP := nParameter

                end

            case upper( alltrim(cLine) ) = cSymbol+"PA"

                if  nPrintedLine > nHeadLines           .and.;
                    nPrintedLine < nLPP

                    *---------------------------------------------
                    * It is inside a page,
                    * then, the footer must be printed.
                    *---------------------------------------------
                    
                    while nPrintedLine < (nLPP - nFootLines)

                        *-----------------------------------------
                        * Print an empty line each time.
                        *-----------------------------------------
                        
                        rpt(; 
                            NL(1), 1, .F., .F., nLineLength,;
                            nTabSize, lWrap, cSymbol; 
                        )

                    end

                    *---------------------------------------------
                    * The footer should have been printed
                    * automatically.
                    *---------------------------------------------

                end

            case upper( alltrim(cLine) ) = cSymbol+"INSERT"

                cParameter :=;
                    alltrim(; 
                        substr( ltrim(cLine),;
                        7+len(cSymbol) ); 
                    )

                if cParameter == ""

                    *---------------------------------------------
                    * No parameter is given.
                    *---------------------------------------------
                    
                    alertBox(; 
                        alltrim(cLine) + NL(1) +;
                        RPT_ERROR_PRINT_COMMAND_UNCOMPLETED; 
                    )

                    *---------------------------------------------
                    * Line ignored.
                    *---------------------------------------------
                
                else

                    *---------------------------------------------
                    * If the file exists (complete with
                    * extention), it is included.
                    *---------------------------------------------
                    
                    if file( cParameter )

                        rpt(; 
                            memoread(cParameter), NIL, .F., NIL,;
                            _MAX_STRING_LEN, NIL, NIL, cSymbol; 
                        )

                    else

                        alertBox(; 
                            alltrim(cLine) + NL(1) +;
                            RPT_ERROR_PRINT_FILE_NOT_FOUND; 
                        )

                    end

                end

            case upper( alltrim(cLine) ) = cSymbol+"NEED"

                cParameter :=;
                    alltrim(; 
                        substr( ltrim(cLine),;
                        5+len(cSymbol) ); 
                    )

                if cParameter == ""

                    *---------------------------------------------
                    * No parameter is given.
                    *---------------------------------------------
                    
                    alertBox(; 
                        alltrim(cLine) + NL(1) +;
                        RPT_ERROR_PRINT_COMMAND_UNCOMPLETED; 
                    )

                    *---------------------------------------------
                    * Line ignored.
                    *---------------------------------------------

                else

                    nParameter := rptEval( cParameter, nLine ) 

                    nNeeded := nParameter

                    if nPrintedLine + nNeeded > nLPP - nFootLines

                        *-----------------------------------------
                        * Also when equal is ok.
                        *
                        * Jump to next page.
                        *-----------------------------------------

                        rpt(; 
                            cSymbol+"PA", 1, .F., .F.,;
                            nLineLength, nTabSize, lWrap, cSymbol; 
                        )

                    end

                end

            case upper( alltrim(cLine) ) = cSymbol+"HEAD"

                cHeader :=;
                    rptReadToEnd(; 
                        cText, @nLine, @nHeadLines,;
                        nLineLength, nTabSize, lWrap, cSymbol; 
                    )

            case upper( alltrim(cLine) ) = cSymbol+"FOOT"

                cFooter :=;
                    rptReadToEnd(; 
                        cText, @nLine, @nFootLines,;
                        nLineLength, nTabSize, lWrap, cSymbol; 
                    )

            case upper( alltrim(cLine) ) = cSymbol+"LEFT"

                cParameter :=;
                    alltrim(; 
                        substr( ltrim(cLine), 5+len(cSymbol) ); 
                    )

                if cParameter == ""

                    *---------------------------------------------
                    * No parameter is given.
                    *---------------------------------------------
                    
                    alertBox(; 
                        alltrim(cLine) + NL(1) +;
                        RPT_ERROR_PRINT_COMMAND_UNCOMPLETED; 
                    )

                    *---------------------------------------------
                    * Line ignored.
                    *---------------------------------------------
                
                else

                    xParameter := rptEval( cParameter, nLine ) 

                    do case
                    case; 
                        valtype(xParameter) == "N"      .and.;
                        xParameter >= 0

                        cLeft := space(xParameter)

                    case valtype(xParameter) == "C"

                        cLeft := xParameter

                    otherwise

                        alertBox(; 
                            alltrim(cLine) + NL(1) +;
                            RPT_ERROR_PRINT_COMMAND_UNCOMPLETED; 
                        )

                        *-----------------------------------------
                        * Line ignored.
                        *-----------------------------------------
                    
                    end
                end

            otherwise

                *-------------------------------------------------
                * Normal line to be printed.
                * Check if the head must be printed.
                *-------------------------------------------------
                
                if  nPrintedLine == 0           .and.;
                    !lHeadFoot

                    *---------------------------------------------
                    * Print the header.
                    *---------------------------------------------
                    
                    rpt(; 
                        cHeader, nHeadLines, .F., .T.,;
                        nLineLength, nTabSize, lWrap, cSymbol; 
                    )

                end

                *-------------------------------------------------
                * It prints the line deleting extra characters on
                * the right. This because the extra space 
                * (the line read from memoline() is ever
                * _MAX_STRING_LEN long) produce extra white lines
                * on the printed text.
                *-------------------------------------------------
                
                qqout( rtrim( cLeft + RPTLineTrans(cLine) ) )

                *-------------------------------------------------
                * New Line.
                *-------------------------------------------------
                
                qout("")

                *-------------------------------------------------
                * The printing interpretation may be long: so
                * show the waitwheel().
                *-------------------------------------------------
                
                waitWheel()

                *-------------------------------------------------
                * A new line was printed: increment 
                * <nPrintedLine>.
                *-------------------------------------------------
                
                nPrintedLine++

                *-------------------------------------------------
                * As a new line is printed: check if the footer 
                * must be printed.
                *-------------------------------------------------
                
                if  nPrintedLine >= ( nLPP - nFootLines );       
                    .and.;
                    lHeadFoot == .F.

                    rpt(; 
                        cFooter, nFootLines, .F., .T.,;
                        nLineLength, nTabSize, lWrap, cSymbol; 
                    )

                    *---------------------------------------------
                    * After a footer there is ever a eject.
                    * The Eject command works only with printers
                    *---------------------------------------------
                    
                    if setRptEject()

                        qqout( FF )  // form feed - chr(12)

                    end

                end

            end

        next

        *---------------------------------------------------------
        * The print process is terminated: check if a final
        * footer is to be printed.
        *---------------------------------------------------------
        
        if  lMain                                       .and.;
            (nPrintedLine > 0 .and. nPrintedLine < nLPP)

            *-----------------------------------------------------
            * This is the main call and the print process is
            * terminated in the middle of the page (no automatic
            * footer was printed):
            * the footer must be printed.
            *-----------------------------------------------------

            while nPrintedLine < (nLPP - nFootLines)

                *-------------------------------------------------
                * It prints an empty line.
                *-------------------------------------------------
                
                rpt(; 
                    "", 1, .F., .F., nLineLength, nTabSize,;
                    lWrap, cSymbol; 
                )

            end

            *-----------------------------------------------------
            * The footer should be automatically printed.
            *-----------------------------------------------------
        
        end

    end //sequence

    *-------------------------------------------------------------
    * Restore old data.
    *-------------------------------------------------------------
    
    if lMain

        setpos( nOldRow, nOldCol )

    end

    return NIL

*=================================================================
function rptMany( cText, bWhileCondition, bForCondition, cSymbol )
*
* rptMany( <cText>, [<bWhileCondition>], [<bForCondition>],
*          [<cSymbol>] ) --> NIL
*
* Prints with the nB report standard the text contained into
* <cText> for every record in the active Alias matching the
* conditions.
*
* <cText>              the text to print.
* <bWhileCondition>    code block WHILE condition for record
*                      to be included.
* <bForCondition>      code block FOR condition for record
*                      to be included.
* <cSymbol>            the symbol for print commands contained
*                      into the text file.
*

    default( @bWhileCondition,  {|| .T. } )
    default( @bForCondition,    {|| .T. } )
    default( @cSymbol,          RPT_DEFAULT_COMMAND_SYMBOL )

    *-------------------------------------------------------------
    * Check for active Alias.
    *-------------------------------------------------------------
    
    if alias() == ""
        
        alertBox( _ERROR_NO_ALIAS )
        
        return NIL                                      // RETURN
    
    end

    *-------------------------------------------------------------
    * Move record pointer to the top: all the records with the
    * active filter will be printed.
    *-------------------------------------------------------------

    dbgotop()

    *-------------------------------------------------------------
    * Start print until eof() is reached.
    *-------------------------------------------------------------
    
    while !eof() .and. eval(bWhileCondition)

        waitFileEval()

        if eval(bForCondition)

            rpt(; 
                cText,; 
                NIL, NIL, NIL, _MAX_STRING_LEN, NIL, NIL,;
                cSymbol; 
            )

        end

        dbskip()

    end

    *-------------------------------------------------------------
    * Close wait bar and move the record pointer to the top.
    *-------------------------------------------------------------
    
    waitFileEval( .T. )

    dbgotop()

    return NIL

*=================================================================
function rptTranslate( cText )
*
* rptTranslate( <cText> ) --> cTranslatedText
*
* This function translates the text <cText> if it contains
* variables delimited with chr(174) and chr(175).
*

    return rptLineTrans( cText )

*-----------------------------------------------------------------
static function rptEval( cEval, nLine )
*
* rptEval( <cEval>, <nLine> ) --> xResult|NIL
*
* This function evaluate the content of <cEval> and returns it.
* If an error occurs, NIL is returned.
*
* <cEval>      The command to execute.
*
* <nLine>      Actual command line to be used for error
*              documentation.
*

    local bSaveErrorHandler
    local xResult               := NIL

    *-------------------------------------------------------------
    * An empty command is not possible.
    *-------------------------------------------------------------

    if cEval == ""

        alertBox( RPT_ERROR_PARAMETER_EVAL_EMPTY )

        return NIL                                      // RETURN

    end

    *-------------------------------------------------------------
    * Prepare for evaluation: handle the possible error.
    *-------------------------------------------------------------

    bSaveErrorHandler :=;
        errorblock(;
            {|e|; 
                errorMacro(;
                    e,;
                    "REPORT COMMAND",;
                    nLine,;
                    cEval;
                );
            };
        )

    begin sequence

        *---------------------------------------------------------
        * &
        *---------------------------------------------------------

        xResult := &(cEval)

    recover
    
        *---------------------------------------------------------
        * NIL means ERROR.
        *---------------------------------------------------------

        xResult := NIL

    end

    errorblock(bSaveErrorHandler)

    *-------------------------------------------------------------
    * The condition result is returned.
    *-------------------------------------------------------------

    return xResult

*-----------------------------------------------------------------
static function rptReadToEnd(; 
    cText, nLine, nLines,;
    nLineLength, nTabSize,;
    lWrap, cSymbol; 
    )
*
* --> cBlock
*
* Returns a block of lines from <cText>.
* The block is delimited with:
*      *...Command
*              ...
*              ...
*      *end
*

    local cLine  := ""
    local cBlock := ""
    local nStartLine := nLine
    local nSubLines := 0
    local lStart    := .T.

    while .T.                                           // FOREVER

        nLine++

        cLine := memoline( cText, nLineLength, nLine, nTabSize, lWrap  )

        do case
        case cLine == "" // end of block

            exit                                        // EXIT

        case upper( alltrim(cLine) ) = cSymbol+"END"

            exit                                        // EXIT

        case; 
            upper( alltrim(cLine) ) = cSymbol+"COMMAND"     .or.;
            upper( alltrim(cLine) ) = cSymbol+"IF"          .or.;
            upper( alltrim(cLine) ) = cSymbol+"WHILE"       .or.;
            upper( alltrim(cLine) ) = cSymbol+"HEAD"        .or.;
            upper( alltrim(cLine) ) = cSymbol+"FOOT"

            *-----------------------------------------------------
            * The next *END must be jumped as it don't belongs
            * to this level.
            *-----------------------------------------------------
            
            if lStart

                *-------------------------------------------------
                * <cBlock> will contain the block of code that
                * we are rebuilding. As it is the lowest level,
                * the new read line <cLine> will be added to
                * it without any CR+LF code.
                *-------------------------------------------------
                
                cBlock += rtrim(cLine)

            else

                *-------------------------------------------------
                * <cBlock> contains already some lines:
                * start with a new line, then add <cLine>
                *-------------------------------------------------
                
                cBlock += NL(1) + rtrim(cLine)
            
            end

            *-----------------------------------------------------
            * Add to <cBlock> all the lines reached until the
            * *END of this level. 
            *-----------------------------------------------------
            
            cBlock +=;
                NL(1) +;
                rptReadToEnd(; 
                    cText, @nLine, @nSubLines,;
                    nLineLength, nTabSize, lWrap, cSymbol; 
                ) +;
                NL(1) + cSymbol+"END"

        otherwise

            *-----------------------------------------------------
            * This is a normal line.
            *-----------------------------------------------------
            
            if lStart

                *-------------------------------------------------
                * <cBlock> will contain the block of code that
                * we are rebuilding. As it is the lowest level,
                * the new read line <cLine> will be added to
                * it without any CR+LF code.
                *-------------------------------------------------
                
                cBlock += rtrim(cLine)

            else

                *-------------------------------------------------
                * <cBlock> contains already some lines:
                * start with a new line, then add <cLine>
                *-------------------------------------------------
                
                cBlock += NL(1) + rtrim(cLine)

            end

        end

        lStart := .F.

    end

    *-------------------------------------------------------------
    * At now, <nLine> is pointed to *END.
    *-------------------------------------------------------------

    *-------------------------------------------------------------
    * <nLines> must count only the content not the commands:
    * *... and *end.
    *-------------------------------------------------------------
    
    nLines := (nLine - nStartLine) -1

    *-------------------------------------------------------------
    * <nLine> and <nLines> are sent back
    *-------------------------------------------------------------
    
    return cBlock

*-----------------------------------------------------------------
static function rptLineTrans( cLine )
*
* --> cTranslatedLine
*
* Translates the <cLine> content replacing what is into
* chr(174) and chr(175), using Memvars and Fields.
*
* This function is recursive.
*

    local bSaveErrorHandler
    local bSav2ErrorHandler
    local nPosLeft
    local nPosRight
    local cLineLeft
    local cLineCenter
    local cLineRight
    local cString
    local xContent

    *-------------------------------------------------------------
    * Find the characters.
    *-------------------------------------------------------------
    
    nPosLeft := at( chr(174), cLine )

    nPosRight := at( chr(175), cLine )

    *-------------------------------------------------------------
    * If the line contains variables.
    *-------------------------------------------------------------
    
    do case
    case nPosLeft == 0 .and. nPosRight == 0

        *---------------------------------------------------------
        * The line contains no variables.
        *---------------------------------------------------------
        
        return cLine                                    // RETURN

    case nPosLeft == 0 .or. nPosRight == 0

        *---------------------------------------------------------
        * The line contains a mistake with symbols: no variables
        * are found.
        *---------------------------------------------------------
        
        alertBox( RPT_ERROR_PARENTESES )

        return cLine                                    // RETURN

    end

    *-------------------------------------------------------------
    * A variable string exists.
    *-------------------------------------------------------------
    
    *-------------------------------------------------------------
    * Extract the string inside the delimitation symbols.
    *-------------------------------------------------------------
    
    cString := substr( cLine, nPosLeft+1 )
    cString := substr( cString, 1, at( chr(175), cString ) - 1 )
    cString := alltrim( cString )

    *-------------------------------------------------------------
    * Extract the peace of line on the left.
    *-------------------------------------------------------------
    
    cLineLeft := substr( cLine, 1, nPosLeft-1 )

    *-------------------------------------------------------------
    * Variable transfer.
    *-------------------------------------------------------------
    
    bSaveErrorHandler := errorblock( {|| break(NIL)} )

    begin sequence

        xContent := &(cString)

    recover

        *---------------------------------------------------------
        * If the <cString> content is not recognised,
        * it generates an error.
        *---------------------------------------------------------
        
        bSaveErrorHandler :=;
            errorblock( {|| break(NIL)} )  // good so

        begin sequence

            *-----------------------------------------------------
            * Maybe, <cString> is a Memvar not jet defined.
            *-----------------------------------------------------
            
            public &cString. := space(_MAX_STRING_LEN)

            &cString. :=;
                 alltrim(; 
                     accept( &cString., "Insert " + cString ); 
                 )

            xContent := alltrim( &cString. )

        recover

            *-----------------------------------------------------
            * If another error comes,
            * It wasn't a new variable.
            * So, it is printed as it is.
            *-----------------------------------------------------
            
            xContent := chr(174) + cString + chr(175)

        end
        errorblock(bSav2ErrorHandler)

    end
    errorblock(bSaveErrorHandler)

    *-------------------------------------------------------------
    * Adapt <xContent>.
    *-------------------------------------------------------------
    
    do case
    case valtype(xContent) == "C"

        cLineCenter := xContent     // without alltrim

    case valtype(xContent) == "N"

        cLineCenter := alltrim( str( xContent ) )

    case valtype(xContent) == "D"

        cLineCenter := dtoc( xContent )

    case valtype(xContent) == "M"

        cLineCenter := xContent     // without trim

    end

    *-------------------------------------------------------------
    * Right extraction with recursion.
    *-------------------------------------------------------------
    
    cLineRight := rptLineTrans( substr( cLine, nPosRight+1 ) )

    return cLineLeft + cLineCenter + cLineRight

*=================================================================
* RUN()
*=================================================================
function run( cCommand )
*
* run( <cCommand> ) --> NIL
*
* <cCommand>   String to execute by OS.
*
* Run command substitute.
*

   if cCommand <> NIL
      
      __Run( cCommand )

   end

   return NIL

*=================================================================
* SAY()
*=================================================================
function say( nRow, nCol, cExpr, cSayPicture, cColorString )
*
* like @...say
*

   default( @cColorString,  setcolor() )

   setpos( nRow, nCol )

   if cSayPicture == NIL

      dispout( cExpr, cColorString )

   else

      devoutpict( cExpr, cSayPicture, cColorString )

   end

   return NIL

*=================================================================
* SETCOLORSTANDARD()
* SETFUNCTION()
* SETMOUSE()
* SETOUTPUT()
* SETRPTEJECT()
* SETRPTLINES()
* SETVERBOSE()
*=================================================================

#define SET_VERBOSE_LIST {;
    "EXACT",;
    "FIXED",;
    "DECIMALS",;
    "DATEFORMAT",;
    "EPOCH",;
    "PATH",;
    "DEFAULT",;
    "EXCLUSIVE",;
    "SOFTSEEK",;
    "UNIQUE",;
    "DELETED",;
    "CANCEL",;
    "DEBUG",;
    "TYPEAHEAD",;
    "COLOR",;
    "CURSOR",;
    "CONSOLE",;
    "ALTERNATE",;
    "ALTFILE",;
    "DEVICE",;
    "EXTRA",;
    "EXTRAFILE",;
    "PRINTER",;
    "PRINTFILE",;
    "MARGIN",;
    "BELL",;
    "CONFIRM",;
    "ESCAPE",;
    "INSERT",;
    "EXIT",;
    "INTENSITY",;
    "SCOREBOARD",;
    "DELIMITERS",;
    "DELIMCHARS",;
    "WRAP",;
    "MESSAGE",;
    "MCENTER",;
    "SCROLLBREAK";
    }

*=================================================================
function setColorStandard( nColor, acColor )
*
* setColorStandard( [<nColor>], [<cColor>|<acColor>] ) 
*    --> cPreviousColor|acPreviousColor
*
* <nColor>              The color number:
*                       0 = All colors,
*                       1 = Base,
*                       2 = Menu,
*                       3 = Head,
*                       4 = Body (Say - Get),
*                       5 = Button (Mouse buttons),
*                       6 = Message,
*                       7 = Alert.
*
* <cColor>              Color string associated to the
*                       <nColor>.
*
* <acColor>             Color array.
*
* Defines standard colors used inside the program.
*
*

    static aColorStandard
    local xPreviousColors

    default( @aColorStandard,;
        {;
            COLOR_STANDARD_BASE,;
            COLOR_STANDARD_MENU,;
            COLOR_STANDARD_HEAD,;
            COLOR_STANDARD_BODY,;
            COLOR_STANDARD_BUTTON,;
            COLOR_STANDARD_MESSAGE,;
            COLOR_STANDARD_ALERT;
        };
    )

    do case
    case nColor == NIL .or. nColor == 0

        *---------------------------------------------------------
        * The standard color set is not updated, but the
        * previuos array is returned.
        *---------------------------------------------------------
        
        xPreviousColors := aClone( aColorStandard )

    case nColor == 0 .and. valtype( acColor ) == "A"
        
        *---------------------------------------------------------
        * The color array is stored and becomes the new standard
        * set of colors.
        *---------------------------------------------------------
        
        xPreviousColors := aClone( aColorStandard )

        aColorStandard := acColor

    case nColor > 0 .and. valtype( acColor ) == "C"

        *---------------------------------------------------------
        * Only one set of colors is received:
        * The color set specified from <nColor> is updated and
        * the previous one (not the entire array) is returned.
        *---------------------------------------------------------
        
        xPreviousColors := aColorStandard[nColor]
        aColorStandard[nColor] := acColor

    case nColor > 0

        *---------------------------------------------------------
        * The standard color set is not updated, but the
        * previuos <nColor> color is returned.
        *---------------------------------------------------------
        
        xPreviousColors := aColorStandard[nColor]

    end

    *-------------------------------------------------------------
    * The previous color array (or the previous color string) is
    * returned.
    *-------------------------------------------------------------
    
    return xPreviousColors

*=================================================================
function setfunction( nFunctionKey, cString )
*
* setfunction( <nFunctionKey>, [<cString>] ) --> NIL
*
* Set Function command substitute.
*

    default( @cString,  "" )

    *-------------------------------------------------------------
    * SET FUNCTION <nFunctionKey> [ TO <cString> ]
    *-------------------------------------------------------------
    
    __setFunction( nFunctionKey, cString )

    return NIL

*=================================================================
function setMouse( lShow )
*
* it uses CLIPMOUS.LIB
*
* setMouse( [<lShow>] ) --> lPrevious
*
* <lShow>      .T. - it shows the mouse cursor,
*              .F. - it hides the mouse cursor,
*              NIL - it reports only the status.
*
* It shows, hide or report the mouse cursor.
*

    local  lPrevious

    static lMouseShow

    default( @lMouseShow,   .F. )

    lPrevious := lMouseShow

    do case
    case lShow == NIL
        
        *---------------------------------------------------------
        * Report only.
        *---------------------------------------------------------

    case lShow

        *---------------------------------------------------------
        * mShow() and mHide() from the mouse library, work good 
        * only if they are made when necessary.
        * Before the mShow() a test is made to verify that the
        * mouse is not already shown.
        *---------------------------------------------------------
        
        if !lMouseShow

            *-----------------------------------------------------
            * Ok, show the mouse cursor.
            *-----------------------------------------------------
            
            mShow()

            lMouseShow := .T.

        else

            *-----------------------------------------------------
            * The mouse cursor is already shown.
            *-----------------------------------------------------

        end

    case !lShow

        *---------------------------------------------------------
        * mShow() and mHide() from the mouse library, work good 
        * only if they are made when necessary.
        * Before the mHide() a test is made to verify that the
        * mouse is not already hidden.
        *---------------------------------------------------------

        if lMouseShow

            *-----------------------------------------------------
            * Ok, hide the mouse cursor.
            *-----------------------------------------------------

            mHide()

            lMouseShow := .F.

        else

            *-----------------------------------------------------
            * The mouse is already hidden.
            *-----------------------------------------------------

        end

    end

    *-------------------------------------------------------------
    * The previous mouse status cursor status is returned.
    *-------------------------------------------------------------
    
    return lPrevious

*=================================================================
function setOutput( caPeripheral )
*
*
* aOldOutput[1]    = _SET_CONSOLE
* aOldOutput[2]    = _SET_PRINTER
* aOldOutput[3]    = _SET_ALTERNATE
* aOldOutput[4]    = _SET_ALTFILE
* aOldOutput[5]    = _SET_EXTRA
* aOldOutput[6]    = _SET_EXTRAFILE
*

    local aOldOutput :=; 
        {;
            .F.,;
            .F.,;
            .F.,;
            NIL,;
            .F.,;
            NIL;
        }

    local nLen

    *-------------------------------------------------------------
    * Fill the <aOldOutput> array with actual data.
    *-------------------------------------------------------------
    
    if isConsoleOn()
        aOldOutput[1] := .T.
    end

    if isPrinterOn()
        aOldOutput[2] := .T.
    end

    if set( _SET_ALTERNATE )
        aOldOutput[3] := .T.
        aOldOutput[4] := set( _SET_ALTFILE )
    end

    if set( _SET_EXTRA )
        aOldOutput[5] := .T.
        aOldOutput[6] := set( _SET_EXTRAFILE )
    end

    do case
    case valtype( caPeripheral ) == "C"

        *---------------------------------------------------------
        * If the parameter <caPeripheral> is a character string,
        * it must be interpreded.
        * The peripheral contained inside <caPeripheral> is set
        * as active exclusively: the others are set to OFF.
        *---------------------------------------------------------
        
        caPeripheral := upper( caPeripheral )

        do case
        case; 
            caPeripheral == "CON"       .or.;
            caPeripheral == "CON.TXT"

            set( _SET_ALTERNATE, .F.)
            set( _SET_PRINTER, "OFF" )
            set( _SET_CONSOLE, "ON" )

        case; 
            caPeripheral == "PRN"       .or.;
            caPeripheral == "PRN.TXT"   .or.;
            caPeripheral == "LPT1"      .or.;
            caPeripheral == "LPT1.TXT"

            *-----------------------------------------------------
            * Other LPT than the first one are not taken into
            * consideration.
            *-----------------------------------------------------
            
            set( _SET_ALTERNATE, .F.)
            set( _SET_PRINTER, "ON" )
            set( _SET_CONSOLE, "OFF" )

        otherwise

            *-----------------------------------------------------
            * The peripheral must be a file.
            *-----------------------------------------------------
            
            set( _SET_CONSOLE, "OFF" )
            set( _SET_PRINTER, "OFF" )
            set( _SET_ALTERNATE, .T.)
            set( _SET_ALTFILE, caPeripheral, .T. )

        end

    case valtype( caPeripheral ) == "A"

        *---------------------------------------------------------
        * If the parameter <caPeripheral> is an array,
        * many peripherals may be ON at the same time.
        *---------------------------------------------------------

        nLen := len( caPeripheral )

        if nLen > 0
            set( _SET_CONSOLE, caPeripheral[1] )
        end

        if nLen > 1
            set( _SET_PRINTER, caPeripheral[2] )
        end

        if nLen > 3
            set( _SET_ALTERNATE, caPeripheral[3] )
            set( _SET_ALTFILE, caPeripheral[4], .T. )
        end

        if nLen > 5
            set( _SET_EXTRA, caPeripheral[5] )
            set( _SET_EXTRAFILE, caPeripheral[6],  )
        end

    end

    *-------------------------------------------------------------
    * The previous array of peripheral configuration is
    * returned.
    *-------------------------------------------------------------
    
    return aOldOutput

*=================================================================
function setRptEject( lbEject )
*
* setRptEject( [<lbEject>] ) --> lPreviousEjectSet
*
* <lbEject>    logical or code block; if .t. it sets the eject
*              after the footer print with RPT() function.
*              Default is no change, starting value is .T..
*
* It sets the eject mode for the RPT() function.
*

    local  lPrevious
    static lEject

    default( @lEject,   .T. )

    lPrevious := lEject

    do case
    case lbEject == NIL

        *---------------------------------------------------------
        * <lEject> is not modified: it is only returned.
        *---------------------------------------------------------
    
    case valtype( lbEject ) == "L"

        lEject := lbEject

    case valtype( lbEject ) == "B"

        lEject := eval( lbEject )

    otherwise
        
        *---------------------------------------------------------
        * <lEject> is not modified: it is only returned.
        *---------------------------------------------------------

    end

    *-------------------------------------------------------------
    * The previous value is returned.
    *-------------------------------------------------------------
    
    return lPrevious

*=================================================================
function setRptLines( nLines )
*
* setRptLines( [<nLines>] ) --> nPreviousLines
*
* <nLines>     remaining lines: if given, it sets the
*              RPT() remaining line counter to the new value.
*              Default is no change.
*
* It sets the remaining line counter and returns the previous
* value.
*

    local  nPrevious := 0

    static nRemaining

    default( @nRemaining,       0 )

    nPrevious := nRemaining

    do case
    case nLines == NIL
        
        *---------------------------------------------------------
        * <nRemaining> is not modified: it is only returned.
        *---------------------------------------------------------

    case nLines >= 0

        nRemaining := nLines

    otherwise
        
        *---------------------------------------------------------
        * <nRemaining> is not modified: it is only returned.
        *---------------------------------------------------------

    end

    *-------------------------------------------------------------
    * The previous value is returned.
    *-------------------------------------------------------------
    
    return nPrevious

*=================================================================
function setVerb( cSpecifier, xNewSetting, lOpenMode )
*
* This function is the equivalent to set(), but it
* accept a char parameter in place of the original
* number.

    local nI
    local acSet := SET_VERBOSE_LIST

    if !( valtype( cSpecifier ) == "C" )
        
        *---------------------------------------------------------
        * The data is not valid, so exit.
        *---------------------------------------------------------
        
        return NIL                                      // RETURN

    end

    *-------------------------------------------------------------
    * Convert to uppercase.
    *-------------------------------------------------------------
    
    cSpecifier := upper( alltrim( cSpecifier ) )

    *-------------------------------------------------------------
    * Scan the array of names.
    *-------------------------------------------------------------
    
    for nI := 1 to _SET_COUNT

        if cSpecifier == acSet[nI]

            *-----------------------------------------------------
            * The name is found: set it and return.
            *-----------------------------------------------------
            
            return set( nI, xNewSetting, lOpenMode )    // RETURN

        end

    next

    *-------------------------------------------------------------
    * If nothing is found, NIL is returned.
    *-------------------------------------------------------------
    
    return NIL

*=================================================================
* STRADDEXTENTION()
* STRCUTEXTENTION()
* STRDRIVE()
* STREXTENTION()
* STRFILE()
* STRFILEFIND()
* STRGETLEN()
* STRLISTASARRAY()
* STROCCURS()
* STRPARENT()
* STRPATH()
* STRTEMPPATH()
* STRXTOSTRING()
*=================================================================
function strAddExtention( cName, cExt )
*
* StrAddExtention( <cName>, <cExt> ) --> cCompleteName
*
* <cName>      the filename or pathname without extention.
*
* <cExt>       the extention to be added.
*
* The function tries to add the extention <cExt> to <cName> if
* <cName> has no extention jet.
*

    local cLastFour
    local cNewName

    default( @cExt,     "" )

    cName := alltrim( cName )

    *-------------------------------------------------------------
    * Remove the extention point from the given extention.
    *-------------------------------------------------------------
    
    if left( cExt, 1 ) == "."

        cExt := substr( cExt, 2 )

    end

    *-------------------------------------------------------------
    * Isolate last four characters.
    *-------------------------------------------------------------
    
    cLastFour := right( cName, 4 )

    *-------------------------------------------------------------
    * If last four characters contains the point ".", there is
    * an extention, else there isn't.
    *-------------------------------------------------------------
    
    if at( ".", cLastFour ) == 0 
        
        *---------------------------------------------------------
        * There is no extention: add the extention.
        *---------------------------------------------------------
        
        cNewName := cName + "." + cExt

    else

        *---------------------------------------------------------
        * The name contains already an extention.
        *---------------------------------------------------------
        
        cNewName := cName

    end

    *-------------------------------------------------------------
    * The name with extention is returned.
    *-------------------------------------------------------------
    
    return cNewName

*=================================================================
function strCutExtention( cName )
*
* StrCutExtention( <cName> ) --> cName
*
* <cName>      the file name or pathname with extention.
*
* The funcion tries to cut the extention from <cName>.
*

    local cLastFour
    local cNewName

    cName := alltrim( cName )

    *-------------------------------------------------------------
    * Isolate last four characters.
    *-------------------------------------------------------------
    
    cLastFour := right( cName, 4 )

    *-------------------------------------------------------------
    * If last four characters contains the point ".", there is
    * an extention, else there isn't.
    *-------------------------------------------------------------
    
    if at( ".", cLastFour ) == 0 
        
        *---------------------------------------------------------
        * There is no extention to cut.
        *---------------------------------------------------------
        
        cNewName := cName

    else

        *---------------------------------------------------------
        * There is the extention and it must be cutted.
        *---------------------------------------------------------
        
        cNewName := left( cName, rat( ".", cName )-1 )

    end

    *-------------------------------------------------------------
    * The name without extention is returned.
    *-------------------------------------------------------------

    return cNewName

*=================================================================
function strDrive( cName )
*
* StrDrive( <cName> ) --> cDrive
*
* <cName>      the pathname where is to extract the drive.
*
* The function tries to extract the drive name from <cName>.
*

    local cDrive

    cName := ltrim( cName )

    *-------------------------------------------------------------
    * If the second character is a colon, the first should be
    * the drive letter.
    *-------------------------------------------------------------
    
    if subst( cName, 2, 1 ) == ":"

        cDrive := left( cName, 2 )

    else

        cDrive := ""

    end

    *-------------------------------------------------------------
    * The drive letter with semicolon is returned.
    *-------------------------------------------------------------
    
    return cDrive

*=================================================================
function strExtention( cName )
*
* StrExtention( <cName> ) --> cExtension
*
* <cName>      the filename or pathname where is to extract
*              the file extention.
*
* The function tries to extract the file extention from
* <cName>.
*

    local cLastFour
    local cExt

    cName := alltrim( cName )

    *-------------------------------------------------------------
    * Isolate last four characters.
    *-------------------------------------------------------------
    
    cLastFour := right( cName, 4 )

    *-------------------------------------------------------------
    * If last four characters contains the point ".", there is
    * an extention, else there isn't.
    *-------------------------------------------------------------
    
    if at( ".", cLastFour ) == 0

        *---------------------------------------------------------
        * There is no extention to extract.
        *---------------------------------------------------------
        
        cExt := ""

    else

        cExt := right( cName, ( len(cName) - rat( ".", cName ) ) )

    end

    return cExt

*=================================================================
function strFile( cName )
*
* StrFile( <cName> ) --> cFileName
*
* <cName>      the filename or pathname where is to extract
*              the file name.
*
* The function tries to extract the file name without path
* from <cName>.
*

    local nLastSlash
    local nLastColon

    cName := alltrim( cName )

    *-------------------------------------------------------------
    * Find the last "\".
    *-------------------------------------------------------------
    
    nLastSlash := rat( "\", cName )

    *-------------------------------------------------------------
    * If a slash is contained inside <cName>, the filename
    * starts after the last slash.
    *-------------------------------------------------------------
    
    if nLastSlash > 0

        cName := substr( cName, nLastSlash+1 )

    end

    *-------------------------------------------------------------
    * Find the last ":".
    *-------------------------------------------------------------
    
    nLastColon := rat( ":", cName )

    *-------------------------------------------------------------
    * If a semicolon is contained inside <cName>, the filename
    * starts after the semicolon as this indicates the disk
    * drive letter.
    *-------------------------------------------------------------
    
    if nLastColon > 0

        cName := substr( cName, nLastColon+1 )

    end

    *-------------------------------------------------------------
    * The file name is returned.
    *-------------------------------------------------------------
    
    return cName

*=================================================================
function strFileFind( cName, cPath )
*
* strFileFind( <cName>, <cPath> ) --> cPathName
*
* <cName>      the filename or pathname containig the file name
*              to search inside the <cPath> list.
*
* <cPath>      a list of paths separated with ";" semicolon,
*              where <cFile> should be sarched.
*
* The function tries to find the file contained inside <cName>
* from the paths contained in <cPath>.
*

    local nPath
    local nI
    local aPath    := {}
    local cAddpath
    local xReturn  := cName

    *-------------------------------------------------------------
    * Clean the name.
    *-------------------------------------------------------------
    
    cName := strFile( cName )

    *-------------------------------------------------------------
    * Translate <cPath>.
    *-------------------------------------------------------------
    
    cPath := strtran( cPath, ";", NL(1) )

    *-------------------------------------------------------------
    * Trasform <cPath> into an array.
    *-------------------------------------------------------------
    
    nPath := mlcount( cPath )

    for nI := 1 to nPath

        cAddPath := memoline( cPath, _MAX_STRING_LEN, nI )

        cAddPath := alltrim( cAddPath )

        *---------------------------------------------------------
        * If the path contains a final backslash it must be 
        * cutted.
        *---------------------------------------------------------
        
        if right( cAddPath, 1 ) == "\"

            cAddPath := left( cAddPath, len( cAddPath )-1 )

        end

        aadd( aPath, cAddPath )

    next

    *-------------------------------------------------------------
    * Search the file.
    *-------------------------------------------------------------

    for nI := 1 to len( aPath )

        if file( aPath[nI] + "\" + cName )

            *-----------------------------------------------------
            * If the file is found, the complete pathname
            * is stored for future return.
            *-----------------------------------------------------
            
            xReturn := aPath[nI] + "\" + cName

            exit                                        // EXIT

        end

    next

    *-------------------------------------------------------------
    * <xReturn> contains the right pathname or the original name
    * if no pathname was right.
    *-------------------------------------------------------------
    
    return xReturn

*=================================================================
function strGetLen( xExpr, cPicture )
*
* This function returns the length of a possible get field
* when using the given picture.
*

    return len( transform( xExpr, cPicture ) )

*=================================================================
function strListAsArray( cList, cDelimiter )
*
* strListAsArray( <cList>, [<cDelimiter>] ) --> aList
*
* <cList>       Delimited string.
*
* <cDelimiter>  List delimiter. The default is comma (,).
*
*  Convert a delimited string to an array
*

    local nPos
    local aList          := {}
    local lDelimLast     := .F.

    default( @cDelimiter,   "," )


    while len(cList) <> 0

        nPos := at(cDelimiter, cList)

        if ( nPos == 0 )

            nPos := len(cList)

        end

        if ( substr( cList, nPos, 1 ) == cDelimiter )

            lDelimLast := .T.

            *-----------------------------------------------------
            * Add last element.
            *-----------------------------------------------------

            aadd( aList, substr(cList, 1, nPos - 1) )

        else

            lDelimLast := .F.

            *-----------------------------------------------------
            * Add a new element.
            *-----------------------------------------------------

            aadd( aList, substr(cList, 1, nPos) )

        end

        cList := substr(cList, nPos + 1)

    end

    if ( lDelimLast )

        aadd(aList, "")

    end

    *-------------------------------------------------------------
    * Return the array.
    *-------------------------------------------------------------

    return aList

*=================================================================
function strOccurs( cSearch, cTarget )
*
* strOccurs( <cSearch>, <cTarget> ) --> nOccurrence
*
* <cSearch>        is the search string to find.
* <cTarget>        is the string to be searched for the
*                  presence of <cSearch>.
*
* The function returns the number of occurrence of <cSearch>
* inside <cTarget>.
*

    local nPos
    local nCount := 0

    while !empty( cTarget )

        if (nPos := at( cSearch, cTarget )) != 0

            nCount++

            cTarget := substr( cTarget, nPos + 1 )

        else

            *-----------------------------------------------------
            * End of string.
            *-----------------------------------------------------

            cTarget := ""

        end

    end

    return nCount

*=================================================================
function strParent( cPathName )
*
* strParent( <cPathName> ) --> cParentPath
*
* <cPath>      the path to transform.
*
* The function tries to return a parent path.
*
    
    local cDrive := strDrive( cPathName )
    local cPath  := strPath( cPathName )
    local cFile  := strFile( cPathName )

    if len( cPath ) > 0

        do case
        case cPath == "\"

            *-----------------------------------------------------
            * Root; it remains as it is.
            *-----------------------------------------------------
        
        case right( cPath, 1 ) == "\"

            *-----------------------------------------------------
            * Normal condition.
            *-----------------------------------------------------
            
            cPath := strPath( left( cPath, len( cPath )-1 ) )

        otherwise

            *-----------------------------------------------------
            * Unknown situation.
            *-----------------------------------------------------

        end

    end

    *-------------------------------------------------------------
    * The "drive - path - file" is rebuilded.
    *-------------------------------------------------------------
    
    return cDrive+cPath+cFile

*=================================================================
function strPath( cName )
*
* strPath( <cName> ) --> cPath
*
* <cName>      the filename or pathname where is to extract
*              the path.
*
* The function tries to extract the path from <cName>.
*

    local cPath
    local nLastSlash

    cName := alltrim( cName )

    *-------------------------------------------------------------
    * First find the last backslash then extract the drive
    * letter if it exists.
    *-------------------------------------------------------------
    
    nLastSlash := rat( "\", cName )

    if nLastSlash > 0

        if substr( cName, 2, 1 ) == ":"  // drive

            cPath := strPath( substr( cName, 3 ) )  // recursion

        else

            cPath := substr( cName, 1, nLastSlash )

        end

    else

        *---------------------------------------------------------
        * If no backslash exists, there is no path.
        *---------------------------------------------------------
        
        cPath := ""

    end

    return cPath

*=================================================================
function strTempPath()
*
* strTempPath() --> cTempPath
*
* The function tries to find if a temporary directory is defined
* with enviromental variables.
*

    local cTempPath := ""

    do case

    case; 
        getenv( "TEMP" ) == ""          .and.;
        getenv( "TMP" ) == ""

        cTempPath := "."

    case !( getenv( "TEMP" ) == "" )

        cTempPath := getenv( "TEMP" )

    case !( getenv( "TMP" ) == "" )

        cTempPath := getenv( "TMP" )

    end

    return cTempPath

*=================================================================
function strXToString( xVar, cType )
*
* strXToString( <xVar>, [<cType>] ) --> cTrasformed_to_string
*
* <xVar>   is the data of any type to be converted into string.
* <cType>  is the type of the data contained inside <xVar>.
*
* The function return <xVar> transformed into a character string.
*

    local cString

    default( @cType,    valtype( xVar ) )

    cType := upper( cType )

    do case
    case cType == "D"

        cString := dtoc( xVar )

    case cType == "L"

        cString := iif( xVar, "T", "F" )

    case cType == "N"

        cString := str( xVar )

    case cType == "C"

        cString := xVar

    case cType == "M"

        cString := xVar

    case xVar == NIL

        cString := "NIL"

    otherwise

       cString := "*******"

    end

    return( cString )

*=================================================================
* TAB()
*=================================================================

#define TAB_ERROR_END_WINDOW;
    "TAB() - Error: you are trying to display or editing " +;
    "a field over the window area!"

*=================================================================
function Tab( aTab )
*
* This function helps on creating easyly defining some screen
* tabs.
*
* All informations are contained inside <aTab> that can be seen
* like a sort of object that is passed to this function.
*
* <aTab> structure: 
*
* { <TAB_LEFT>, <TAB_RIGHT>, <TAB_FIELD_LEN>, <TAB_TAB_ARRAY> }
*
* TAB_LEFT      contains the left screen border;               
* TAB_RIGHT     contains the right screen border;        
* TAB_FIELD_LEN contains the len of the field to be displayed;
* TAB_TAB_ARRAY contains an array of tabs positions.          
*
* Default values are:
*
* TAB_DEFAULT   { 0, maxcol(), 1, { 4 } }
*
*
    local aTabTotals
    local nLastTab
    local nI
    local nSpace

    *-------------------------------------------------------------
    * If <aTab> is not an array there is no sense to continue.
    *-------------------------------------------------------------
    
    if !(valtype( aTab ) == "A")

        return NIL                                  // RETURN

    end

    *-------------------------------------------------------------
    * Make a copy of the tab positions.
    *-------------------------------------------------------------
    
    aTabTotals := aclone( aTab[TAB_TAB_ARRAY] )

    *-------------------------------------------------------------
    * The tab position copy is transformed into an array that
    * contains the progressive relative screen position.
    *-------------------------------------------------------------
    
    for nI := 2 to len( aTabTotals )

        aTabTotals[nI] := aTabTotals[nI] + aTabTotals[nI-1]

    next

    *-------------------------------------------------------------
    * point <nI> to the last element number of tab positions.
    *-------------------------------------------------------------
    
    nI := len( aTab[TAB_TAB_ARRAY] )

    *-------------------------------------------------------------
    * If the tab positions are not enough to reach the right
    * limit, add other tab position equal to the last one.
    *-------------------------------------------------------------
    
    while .T.                                           // FOREVER

        if (aTabTotals[nI] + aTab[TAB_LEFT]) > aTab[TAB_RIGHT]

            exit                                        // EXIT

        end

        nLastTab := aTab[TAB_TAB_ARRAY][nI]

        aadd( aTab[TAB_TAB_ARRAY], nLastTab )

        aadd( aTabTotals, aTabTotals[nI]+nLastTab )

        nI++

    end

    *-------------------------------------------------------------
    * Calculate the space to add to the actual col() to reach
    * the next tab position.
    *-------------------------------------------------------------
    
    for nI := 1 to len( aTab[TAB_TAB_ARRAY] )

        nSpace := ( (aTab[TAB_LEFT] + aTabTotals[nI] -1) - col() )

        if nSpace > 0

            exit                                        // EXIT

        end

    end

    if nSpace < 0

        alert( TAB_ERROR_END_WINDOW )

        return NIL

    end

    *-------------------------------------------------------------
    * If the field may not stay inside the bounders starting
    * from the calculated position, it is better to move it
    * on the next screen line.
    *-------------------------------------------------------------
    
    if col() + nSpace + aTab[TAB_FIELD_LEN]-1 > aTab[TAB_RIGHT]

        *---------------------------------------------------------
        * Change the screen cursor position to the begin of a
        * new screen line.
        *---------------------------------------------------------
        
        setpos( row()+1, aTab[TAB_LEFT] )

    else

        *---------------------------------------------------------
        * Change the screen cursor position to col()+nSpace.
        *---------------------------------------------------------
        
        setpos( row(), col()+nSpace)

    end

    return NIL

*=================================================================
* TB()
*=================================================================

#define TB_ERROR_ARRAY_DIFFERENT_DIMENTION_TOP_SEP;
    "TB() - The array of top separations has a different " +;
    "number of elements as expected."

#define TB_ERROR_ARRAY_DIFFERENT_DIMENTION_BODY_SEP;
    "TB() - The array of body separations has a different " +;
    "number of elements as expected."

#define TB_ERROR_ARRAY_DIFFERENT_DIMENTION_BOT_SEP;
    "TB() - The array of bottom separations has a different " +;
    "number of elements as expected."

#define TB_ERROR_ARRAY_DIFFERENT_DIMENTION_SAY_PIC;
    "TB() - The array of pictures has a different " +;
    "number of elements as expected."

#define TB_ERROR_ARRAY_DIFFERENT_DIMENTION_CALC;
    "TB() - The array of calculated columns (not editable) has a different " +;
    "number of elements as expected."

#define TB_ERROR_ARRAY_DIFFERENT_DIMENTION_HEAD;
    "TB() - The array of headers has a different " +;
    "number of elements as expected."

#define TB_ERROR_ARRAY_DIFFERENT_DIMENTION_FOOT;
    "TB() - The array of footers has a different " +;
    "number of elements as expected."

#define TB_ERROR_ARRAY_DIFFERENT_DIMENTION_VALID;
    "TB() - The array of post validation code blocks has a different " +;
    "number of elements as expected."

#define TB_ERROR_ARRAY_DIFFERENT_DIMENTION_MSG;
    "TB() - The array of messages has a different " +;
    "number of elements as expected."

#define TB_ERROR_ARRAY_DIFFERENT_DIMENTION_COLORS;
    "TB() - The array of color code blocks has a different " +;
    "number of elements as expected."

#define TB_ERROR_NO_COLUMNS;
    "TB() - No Columns."

#define TB_ERROR_NO_RECORDS;
    "TB() - The file contains no records."

#define TB_ERROR_NO_PICTURES;
    "TB() - No Pictures."

#define TB_ERROR_RECORD_LOCKED;
    "TB() - Record locked." +;
    NL(1) +;
    "Try again later."

#define TB_ERROR_FILE_LOCKED;
    "TB() - File locked." +;
    NL(1) +;
    "Try again later."

#define TB_PROMPT_DELETE_RECORD "Delete Record?"

#define TB_PROMPT_NO_RECORDS_APPEND;
    "The active Alias contains no records: do you want to try to append an empty record?"


#define TB_WINDOW_BOTTOM_MEMO   "[Esc] cancel  [Ctrl]+[W] save"

#define TB_BUTTONS {;
    { nBottom+1, nLeft+0,  "[Esc]",         {|| keyboard( chr(K_ESC) ) }        },;
    { nBottom+1, nLeft+6,  "[-]",          {|| keyboard( chr(K_LEFT) ) }       },;
    { nBottom+1, nLeft+11, "[]",           {|| keyboard( chr(K_UP) ) }         },;
    { nBottom+1, nLeft+15, "[Ctrl]+[-]",   {|| keyboard( chr(K_CTRL_LEFT) ) }  },;
    { nBottom+1, nLeft+27, "[Pag]",        {|| keyboard( chr(K_PGUP) ) }       },;
    { nBottom+1, nLeft+34, "[Ctrl]+[Home]", {|| keyboard( chr(K_CTRL_HOME) ) }  },;
    { nBottom+1, nLeft+48, "[Ctrl]+[Pag]", {|| keyboard( chr(K_CTRL_PGUP) ) }  },;
    { nBottom+2, nLeft+0,  "[F1] ",         {|| keyboard( chr(K_F1) ) }         },;
    { nBottom+2, nLeft+6,  "[-]",          {|| keyboard( chr(K_RIGHT) ) }      },;
    { nBottom+2, nLeft+11, "[]",           {|| keyboard( chr(K_DOWN) ) }       },;
    { nBottom+2, nLeft+15, "[Ctrl]+[-]",   {|| keyboard( chr(K_CTRL_RIGHT) ) } },;
    { nBottom+2, nLeft+27, "[Pag]",        {|| keyboard( chr(K_PGDN) ) }       },;
    { nBottom+2, nLeft+34, "[Ctrl]+[End] ", {|| keyboard( chr(K_CTRL_END) ) }   },;
    { nBottom+2, nLeft+48, "[Ctrl]+[Pag]", {|| keyboard( chr(K_CTRL_PGDN) ) }  };
    }

#define TB_HELP;
    "tb()" +;
    NL(3) +;
    "Standard .DBF browse." +;
    NL(2) +;
    "This function permits to browse the active Alias, " +;
    "eventually with relations." +;
    NL(3) +;
    "Special keys:" +;
    NL(2) +;
    "[Enter]            start field editing;" +;
    NL(1) +;
    "[Pag]             previous page;" +; 
    NL(1) +;
    "[Pag]             next page;" +; 
    NL(1) +;
    "[Ctrl]+[Pag]      top of file;" +; 
    NL(1) +;
    "[Ctrl]+[Pag]      bottom of file;" +; 
    NL(1) +;
    "[Ctrl]+[Home]      first column;" +; 
    NL(1) +;
    "[Ctrl]+[End]       last column;" +; 
    NL(1) +;
    "[Ctrl]+[Enter]     append a new empty record;" +;
    NL(1) +;
    "[Ctrl]+[F2]        copy the current record;" +;
    NL(1) +;
    "[Ctrl]+[F3]        append and paste a record;" +;
    NL(1) +;
    "[Ctrl]+[F4]        paste a record deleting the content of the current one;" +;
    NL(1) +;
    "[Ctrl]+[Y]         delete/recall record;" +;
    NL(1) +;
    "[Ctrl]+[Del]       delete/recall record." +;
    NL(3) +;
    "When a memo field is edited:" +;
    NL(2) +;
    "[Esc]         cancel and close window;" + NL(1) +;
    "[Ctrl]+[Y]    line delete;" + NL(1) +;
    "[Ctrl]+[W]    save and close window."

*=================================================================
function TB(;
        nTop, nLeft, nBottom, nRight,;
        acCol, acColSayPic,;
        acColTopSep, acColBodySep, acColBotSep,;
        acColHead, acColFoot,;
        alColCalc,;
        abColValid,;
        abColMsg,;
        cColor, abColColors,;
        nFreeze,;
        lModify, lAppend, lDelete,;
        xButtons;
    )
*
* TB( <nTop>, <nLeft>, <nBottom>, <nRight>,
*      <acCol>, [<acColSayPic>],
*      [<acColTopSep>], [<acColBodySep>], [<acColBotSep>],;
*      [<acColHead>], [<acColFoot>],
*      [<alColCalc>],
*      [<abColValid>],
*      [<abColMsg>],
*      [<cColor>], [<abColColors>],;
*      [<nFreeze>],
*      [<lModify>], [<lAppend>], [<lDelete>],
*      [lButtons|aButtons] )  --> NIL
*
* <nTop>, <nLeft>, <nBottom>, <nRight>         The display area
*                                              used to browse.
* <acCol>              Column array.
* <acColSayPic>        Picture array.
* <acColTopSep>        Header separation.
* <acColBodySep>       Body separation.
* <acColBotSep>        Footer separation.
* <acColHead>          Column head description array.
* <acColFoot>          Column foot description array.
* <alColCalc>          Calculated column array: .T. menas
*                      calculated, .F. means editable.
* <abColValid>         Validation codeblock array.
* <abColMsg>           Message codeblock array. The codeblock
*                      must have a string result.
* <cColor>             Color string. It MAY BE LONGER than
*                      the usual 5 elements.
* <abColColor>         Code blocks for column colors. The
*                      codeblocks must return a array of
*                      a couple of digit that point to two
*                      colors of <cColor>.
* <nFreeze>            Number of columns to be left visible.
* <lModify>            Ability to modify the file.
* <lAppend>            Ability to append data.
* <lDelete>            Ability to delete records.
* <lButtons>           If buttons are desired.
* <aButtons>           Buttons array.
*
* This function starts a Browse table.
*
* Calculated fields are given as character string and not as
* code blocks as the editable fields are also sent this way:
* character string containing the field name.
*

    local bOldErrorHandler
    local nOldCursor
    local nOldRow
    local nOldCol
    local bOldF1
    local lOldReadExit      := set( _SET_EXIT, .T. )
    local lOldSetMouse      := setMouse()

    local nColumns          := 0

    local oBrowse
    local oColumn
    local cCol
    local cColType
    local nNumLen
    local nKey
    local lMore             := .T.

    local alColMemo
    local anColWidth        := {}
    local anColShift        := {}
    local axColCopy

    local nRow
    local nCol
    local cMsg
    local nLockedCol        := 0
    local nI

    *-------------------------------------------------------------
    * If the <acCol> array is not given (NIL), create a default
    * information to create a tBrowse of the active Alias.
    *-------------------------------------------------------------

    if acCol == NIL
    
        tbDefault(;
            @acCol, @acColSayPic, @alColCalc,;
            @acColHead, @acColFoot,;
            @abColValid, @abColMsg;
        )

        nFreeze := 1

    end

    *-------------------------------------------------------------
    * If the <acCol> array is still NIL, there is maybe no active
    * Alias.
    *-------------------------------------------------------------

    if acCol == NIL

        if empty( alias() )

            alertBox( _ERROR_NO_ALIAS )

        else

            alertBox( TB_ERROR_NO_COLUMNS )

        end

        return NIL                                      // RETURN

    end

    *-------------------------------------------------------------
    * <nColumns> will contain the number of columns calculating 
    * the <acCol> length. The other column arrays should have the
    * same length.
    *-------------------------------------------------------------

    nColumns := len( acCol )

    *-------------------------------------------------------------
    * Set default data.
    *-------------------------------------------------------------

    default( @nTop,         0 )
    
    default( @nLeft,        0 )
    
    default( @nBottom,      maxrow() )
    
    default( @nRight,       maxcol() )

    default( @cColor,       setcolor() )
    
    default( @abColColors,  afill( array( nColumns ), {||{1,2}} ) )

    default( @acColSayPic,  array( nColumns ) )

    default( @acColHead,    aclone( acCol ) )

    default( @acColFoot,    afill( array( nColumns ), "" ) )

    default( @alColCalc,    afill( array( nColumns ), .F. ) )

    default( @abColValid,   afill( array( nColumns ), {||.T.} ) )
    
    default( @abColMsg,     NIL )

    default(;
        @acColTopSep,;
        afill( array( nColumns ), chr(194)+chr(196) );
    )

    default(;
        @acColBodySep,;
        afill( array( nColumns ), chr(179) );
    )

    default(;
        @acColBotSep,;
        afill( array( nColumns ), chr(193)+chr(196) );
    )

    default( @nFreeze,      0 )
    
    default( @lModify,      .T. )
    
    default( @lAppend,      .T. )
    
    default( @lDelete,      .T. )

    *-------------------------------------------------------------
    * Clean the screen.
    *-------------------------------------------------------------
    
    scroll( nTop, nLeft, nBottom, nRight )

    *-------------------------------------------------------------
    * Check <xButtons>, it may be logical or a button array.
    * If <xButton> is logical and is True, the space for
    * buttons si taken from the bottom of the given area.
    * If <xButton> is an array, it contains the buttons
    * coordinates, so no space for buttons is calculated.
    *-------------------------------------------------------------

    do case
    case valtype( xButtons ) == "L"
    
        if xButtons

            *-----------------------------------------------------
            * As button are placed automatically,
            * the space is taken form the bottom
            * area:
            *-----------------------------------------------------

            nBottom := nBottom-2

            *-----------------------------------------------------
            * Buttons are placed beyond nButtom
            * line.
            *-----------------------------------------------------

            xButtons := TB_BUTTONS

        else
        
            *-----------------------------------------------------
            * xButtons cannot remain logical.
            *-----------------------------------------------------

            xButtons := NIL

        end

    case valtype( xButtons ) == "A"

        *---------------------------------------------------------
        * OK.
        * Don't leave the space, as the caller
        * do what it wants
        *---------------------------------------------------------

    otherwise
    
        xButtons := NIL
        
    end

    *-------------------------------------------------------------
    * If [F1] is already set, no local help will be started.
    *-------------------------------------------------------------

    if setkey( K_F1 ) == NIL

        *---------------------------------------------------------
        * There is no previous help.
        *---------------------------------------------------------
        
        bOldF1 := setkey( K_F1, { || Text( TB_HELP ) } )

    else

        bOldF1 := setkey( K_F1 )

    end

    *-------------------------------------------------------------
    * Prepare <alColMemo> empty.
    *-------------------------------------------------------------

    alColMemo := array( len( acCol ) )

    *-------------------------------------------------------------
    * Check arrays length and report errors.
    *-------------------------------------------------------------

    nColumns := len( acCol )

    if !( nColumns == len( acColTopSep ) )
    
        alertBox( TB_ERROR_ARRAY_DIFFERENT_DIMENTION_TOP_SEP )

        return NIL                                      // RETURN

    end
    
    if !( nColumns == len( acColBodySep ) )
    
        alertBox( TB_ERROR_ARRAY_DIFFERENT_DIMENTION_BODY_SEP )

        return NIL                                      // RETURN

    end

    if !( nColumns == len( acColBotSep ) )
    
        alertBox( TB_ERROR_ARRAY_DIFFERENT_DIMENTION_BOT_SEP )

        return NIL                                      // RETURN

    end

    if !( nColumns == len( acColSayPic ) )
    
        alertBox( TB_ERROR_ARRAY_DIFFERENT_DIMENTION_SAY_PIC )

        return NIL                                      // RETURN

    end

    if !( nColumns == len( alColCalc ) )
    
        alertBox( TB_ERROR_ARRAY_DIFFERENT_DIMENTION_CALC )

        return NIL                                      // RETURN

    end

    if !( nColumns == len( acColHead ) )
    
        alertBox( TB_ERROR_ARRAY_DIFFERENT_DIMENTION_HEAD )

        return NIL                                      // RETURN

    end

    if !( nColumns == len( acColFoot ) )
    
        alertBox( TB_ERROR_ARRAY_DIFFERENT_DIMENTION_FOOT )

        return NIL                                      // RETURN

    end

    if !( nColumns == len( abColValid ) )
    
        alertBox( TB_ERROR_ARRAY_DIFFERENT_DIMENTION_VALID )

        return NIL                                      // RETURN

    end

    if  valtype( abColMsg ) == "A"          .and.;
        !( nColumns == len( abColMsg ) )
    
        alertBox( TB_ERROR_ARRAY_DIFFERENT_DIMENTION_MSG )

        return NIL                                      // RETURN

    end

    if !( nColumns == len( abColColors ) )
    
        alertBox( TB_ERROR_ARRAY_DIFFERENT_DIMENTION_COLORS )

        return NIL                                      // RETURN

    end

    *-------------------------------------------------------------
    * If it is not possible to modify correct possible errors.
    *-------------------------------------------------------------

    if  !lModify

        lAppend := .F.

        lDelete := .F.

    end

    *-------------------------------------------------------------
    * If the file is empty ask what to do or inform the user.
    *-------------------------------------------------------------
    
    if reccount() == 0
    
        *---------------------------------------------------------
        * The file is empty.
        *---------------------------------------------------------
        
        if lAppend
    
            *-----------------------------------------------------
            * Record appending is allowed.
            *-----------------------------------------------------
            
            if  alertbox(;
                    TB_PROMPT_NO_RECORDS_APPEND,;
                    { " Yes ", " No " };
                ) == 1
                
                dbappend()
                
                if neterr()
                
                    alertbox( TB_ERROR_FILE_LOCKED )
                    
                end
                
            else

                return NIL                              // RETURN

            end
            
        else
        
            *-----------------------------------------------------
            * Record appending is not allowed.
            *-----------------------------------------------------

            alertbox( TB_ERROR_NO_RECORDS )

            return NIL                                  // RETURN

        end
        
    end

    *-------------------------------------------------------------
    * If exist <abColMsg>, an extra line at bottom is needed.
    *-------------------------------------------------------------

    if  !( abColMsg == NIL )

        nBottom--

    end

    *-------------------------------------------------------------
    * Calculate the number of column on the left
    * where the cursor cannot enter.
    *-------------------------------------------------------------

    if nFreeze > 0

        nLockedCol := 0

        while (++nLockedCol) <= nFreeze
        
            if !(alColCalc[nLockedCol])

                *-------------------------------------------------
                * This column is not read only, so it cannot be
                * locked.
                *-------------------------------------------------

                nLockedCol--

                exit                                    // EXIT

            end

            *-----------------------------------------------------
            * If the loop '' WHILE (++nLockedCol) <= nFreeze ''
            * will terminate with <nLockedCol> greater than
            * <nFreeze> because only then the condition is False.
            * This is the reason for the following test.
            *-----------------------------------------------------

            if nLockedCol == nFreeze

                exit                                    // EXIT

            end


        end

    else

        nLockedCol := 0

    end

    *-------------------------------------------------------------
    * If eof() or bof() it is better to move the record pointer.
    *-------------------------------------------------------------

    if eof()

        dbgobottom()

    end

    if bof()

        dbgotop()

    end

    *-------------------------------------------------------------
    * Save old data.
    *-------------------------------------------------------------

    nOldCursor          := setcursor()
    nOldRow             := row()
    nOldCol             := col()

    *-------------------------------------------------------------
    * Show buttons.
    *-------------------------------------------------------------

    if valtype( xButtons ) == "A"

        tbButtonSay( xButtons )

    end

    *-------------------------------------------------------------
    * Prepare for browse: save error handler before.
    *-------------------------------------------------------------

    bOldErrorHandler    := errorblock( {|e| ErrorHandler(e)} )
    begin sequence

        *---------------------------------------------------------
        * Create the TBrowse object and set up some instance
        * variables.
        *---------------------------------------------------------

        oBrowse := tbrowsedb(nTop, nLeft, nBottom, nRight)

        oBrowse:skipBlock   := { |nRequest| tbSkip( nRequest, oBrowse ) }
        
        oBrowse:colorSpec   := cColor

        *---------------------------------------------------------
        * Add columns to the <oBrowse> object.
        *---------------------------------------------------------

        for nI := 1 to nColumns

            *-----------------------------------------------------
            * Extract some data of this column.
            *-----------------------------------------------------

            cCol        := acCol[nI]
            cColType    := valtype( &(cCol) )

            do case
            case alColCalc[nI]

                *-------------------------------------------------
                * Make a calculated field.
                *-------------------------------------------------

                oColumn :=; 
                    tbColumnNew( acColHead[nI], tbBlockCalc( cCol ) )
            
                alColMemo[nI] := .F.

            case cColType == "M"

                *-------------------------------------------------
                * Make a calculated field to activate the
                * macro window
                *-------------------------------------------------

                if  valtype( acColSayPic[nI] ) == "C"   .and.;
                    !empty( acColSayPic[nI] )

                    cCol :=; 
                        "transform(" +; 
                        cCol +; 
                        ",'" +; 
                        acColSayPic[nI] +; 
                        "')"

                else

                    cCol := "padr(" + cCol + ", 30 )"

                end
                
                oColumn :=; 
                    tbColumnNew( acColHead[nI], tbBlockCalc( cCol ) )

                alColMemo[nI] := .T.

            case;
                ( cColType == "C")                      .and.;
                ( len( &(cCol) ) > _MAX_STRING_LEN )
                
                *-------------------------------------------------
                * Make a calculated field to activate the
                * macro window
                *-------------------------------------------------

                if  valtype( acColSayPic[nI] ) == "C"   .and.;
                    !empty( acColSayPic[nI] )

                    cCol :=; 
                        "transform(" +; 
                        cCol +; 
                        ",'" +; 
                        acColSayPic[nI] +; 
                        "')"

                else

                    cCol := "padr(" + cCol + ", 30 )"

                end
                
                oColumn :=; 
                    tbColumnNew( acColHead[nI], tbBlockCalc( cCol ) )

                alColMemo[nI] := .T.

            otherwise

                *-------------------------------------------------
                * Make a new normal field column.
                *-------------------------------------------------

                oColumn :=;
                    tbColumnNew(;
                        acColHead[nI],;
                        tbBlockField( cCol );
                    )

                alColMemo[nI] := .F.

            end

            *-----------------------------------------------------
            * Other attribute.
            *-----------------------------------------------------

            oColumn:picture     := acColSayPic[nI]
            oColumn:headSep     := acColTopSep[nI]
            oColumn:colSep      := acColBodySep[nI]
            oColumn:footSep     := acColBotSep[nI]
            oColumn:colorBlock  := abColColors[nI]
            oColumn:footing     := acColFoot[nI]            

            *-----------------------------------------------------
            * Prepare the <anColWidth> with the columns width and
            * <anColShift> with the column shift for numeric 
            * fields:
            * if numeric fields are shorter than the column field
            * (this may happens only because the column head is 
            * greater), these numeric fields appears right 
            * aligned inside the column, so the first display 
            * column of the field is greater then the first 
            * display column of the column.
            *-----------------------------------------------------

            *-----------------------------------------------------
            * First determinate the field length and put it inside
            * the <anColWidth>.
            * At the moment, <anColShift> is filled with zeroes.
            *-----------------------------------------------------
        
            aadd(; 
                anColWidth,; 
                len( transform( &(cCol), acColSayPic[nI] ) ); 
            )

            aadd( anColShift, 0 )

            *-----------------------------------------------------
            * If the column head is longher then the field, the
            * column will be greater.
            *-----------------------------------------------------
        
            if  len( acColHead[nI] ) > anColWidth[nI]

                *-------------------------------------------------
                * If the field is numeric, it is right aligned
                * inside the column, so, the corresponding
                * <anColShift[nI]> is calculated.
                *-------------------------------------------------
            
                if  cColType == "N"

                    anColShift[nI] :=; 
                        len( acColHead[nI] ) - anColWidth[nI]

                end

                *-------------------------------------------------
                * The column with is updated to the head width.
                *-------------------------------------------------
            
                anColWidth[nI] := len( acColHead[nI] ) 
            
            end

            
            *-----------------------------------------------------
            * Finally add the column to the <oBrowse> object.
            *-----------------------------------------------------

            oBrowse:addColumn( oColumn )

        next

        *---------------------------------------------------------
        * More tBrowse object instances to make after columns
        * definition.
        *---------------------------------------------------------
        
        oBrowse:freeze      := nFreeze

        *---------------------------------------------------------
        * Prepare before TBrowse show:
        * Turn the cursor off while browsing.
        *---------------------------------------------------------

        setcursor(SETCURSOR_NONE)

        *---------------------------------------------------------
        * TBrowse loop.
        *---------------------------------------------------------

        while lMore

            *-----------------------------------------------------
            * Don't let the cursor move into locked columns
            * on the left.
            *-----------------------------------------------------

            if ( oBrowse:colPos <= nLockedCol )

                oBrowse:colPos := nLockedCol + 1

            end

            *-----------------------------------------------------
            * Stabilize the display until it's stable
            * or a key is pressed.
            *-----------------------------------------------------

            oBrowse:forceStable()

            *-----------------------------------------------------
            * Show the bottom message (if the array was given).
            * ( Here is the only possible place inside
            * this program ).
            *-----------------------------------------------------

            nRow := row()
            nCol := col()

            if !(abColMsg == NIL)

                cMsg := eval( abColMsg[oBrowse:colPos] )

                say(;
                    nBottom+1, nLeft,;
                    padc( cMsg, nRight+1-nLeft );
                )
                
                setpos( nRow, nCol )

            end

            *-----------------------------------------------------
            * Make sure that the current record is showing
            * up-to-date data in case we are on a network.
            *-----------------------------------------------------

            oBrowse:refreshCurrent():forceStable()

            *-----------------------------------------------------
            * Read the mouse or the keyboard inside a loop,
            * as the mouse presence do not permit to pause
            * for a key.
            *-----------------------------------------------------

            while .t.

                *-------------------------------------------------
                * Read the mouse before and then, the keyboard.
                *-------------------------------------------------

                // Show the mouse.
                setMouse( .T. )

                // Was a mouse button?
                if !( mouse() == NIL )

                    tbMouseKeyboard(; 
                        xButtons,;
                        nTop, nLeft, nBottom, nRight,;
                        oBrowse, anColWidth, anColShift; 
                    ) 

                    mouse( .T. )

                end

                // Read the keyboard.
                nKey := inkey()

                *-------------------------------------------------
                * If a key was pressed, the "wait windows", or
                * "message line" must be closed!
                * Maybe there is nothing to close, but it is
                * better to do it now!
                *-------------------------------------------------

                if nKey <> 0

                    waitFor()       // Close a waitfor()

                    messageLine()   // Close a messageLine()

                    setMouse( .F. ) // Hide the mouse cursor

                end

                *-------------------------------------------------
                * Analise now the key pressed.
                *-------------------------------------------------

                do case
                case ( nKey == 0 )

                    *---------------------------------------------
                    * No key was pressed, loop again.
                    *---------------------------------------------

                    loop                                // LOOP

                case ( nKey == K_ESC )

                    *---------------------------------------------
                    * Esc means leave
                    *---------------------------------------------

                    lMore := .F.

                    *---------------------------------------------
                    * Exit wait state loop.
                    *---------------------------------------------

                    exit                                // EXIT

                end

                *-------------------------------------------------
                * Now it may be a function key or whatever else
                * that is "redirected" to a special funciton.
                * After that, it may be a normal key.
                *-------------------------------------------------

                tbApplyKey(;
                    oBrowse, nKey, acCol, @axColCopy, alColCalc,;
                    acColHead, abColValid,;
                    alColMemo, lModify, lAppend, lDelete;
                )

                *-------------------------------------------------
                * Finally, exit wait state loop
                *-------------------------------------------------

                exit

            end

            *-----------------------------------------------------
            * Mouse and keyboard loop terminated.
            *-----------------------------------------------------

        end

        *---------------------------------------------------------
        * TBrowse loop termiated.
        *---------------------------------------------------------

    recover

        *---------------------------------------------------------
        * Nothing to recover.
        *---------------------------------------------------------

    end //sequence

    *-------------------------------------------------------------
    * Restore previous values and return.
    *-------------------------------------------------------------

    errorblock(bOldErrorHandler)
    setcursor(nOldCursor)
    setpos( nOldRow, nOldCol )
    setkey( K_F1, bOldF1 )
    setMouse( lOldSetMouse )

    return NIL

*-----------------------------------------------------------------
static function tbSkip(nRequest, oBrowse)
*
* tbSkip(nRequest, oBrowse) --> nSkippedRecords
*
* Handle record movement requests from the Tbrowse object.
*

    local nActually    := 0

    do case
    case;
        nRequest == 0              .or.;
        lastrec() == 0

        *---------------------------------------------------------
        * Skip 0 (significant on a network).
        *---------------------------------------------------------

        dbSkip(0)

        nActually := 0

    case;
        nRequest > 0               .and.;
        !eof()

        *---------------------------------------------------------
        * Skip Forward.
        *---------------------------------------------------------

        while (++nActually) <= nRequest

            dbskip( 1 )

            if eof()
            
                dbgobottom()

                nActually--

                exit                                    // EXIT

            end

            *-----------------------------------------------------
            * The following test is made because <nActually>
            * should not be greater then <nRequest>, but the
            * WHILE loop will terminate only then the <nActually>
            * will be greter.
            *-----------------------------------------------------
            
            if nActually == nRequest

                exit                                    // EXIT

            end

        end

    case;
        nRequest < 0               .and.;
        !bof()

        *---------------------------------------------------------
        * Skip backward.
        *---------------------------------------------------------

        while (--nActually) >= nRequest

            dbskip(-1)

            if bof()

                dbgotop()

                nActually++

                exit                                    // EXIT

            end
            
            *-----------------------------------------------------
            * The following test is made because <nActually>
            * should not be less then <nRequest>, but the
            * WHILE loop will terminate only then the <nActually>
            * will be less.
            *-----------------------------------------------------
            
            if nActually == nRequest

                exit                                    // EXIT

            end

        end

    otherwise

        *---------------------------------------------------------
        * No skip allowed.
        *---------------------------------------------------------

        nActually := 0

    end

    return nActually

*-----------------------------------------------------------------
static function tbApplyKey(;
    oBrowse, nKey,;
    acCol, axColCopy, alColCalc, acColHead,;
    abColValid, alColMemo,;
    lModify, lAppend, lDelete;
    )
*
* Apply one keystroke to the oBrowse.
*

    local bOldErrorHandler
    local lMemo                 := .F.
    local cKeyString
    local xKey

    *-------------------------------------------------------------
    * Is it a memo field?
    *-------------------------------------------------------------

    lMemo := alColMemo[oBrowse:colPos]

    do case
    case !empty( setkey( nKey ) )

        eval( setkey( nKey ) )

    case nKey == K_DOWN

        oBrowse:down()

    case nKey == K_PGDN

        oBrowse:pageDown()

    case nKey == K_CTRL_PGDN

        oBrowse:goBottom()

    case nKey == K_UP

        oBrowse:up()

    case nKey == K_PGUP

        oBrowse:pageUp()

    case nKey == K_CTRL_PGUP

        oBrowse:goTop()

    case nKey == K_RIGHT

        oBrowse:right()

    case nKey == K_LEFT

        oBrowse:left()

    case nKey == K_HOME

        oBrowse:home()

    case nKey == K_END

        oBrowse:end()

    case nKey == K_CTRL_LEFT

        oBrowse:panLeft()

    case nKey == K_CTRL_RIGHT

        oBrowse:panRight()

    case nKey == K_CTRL_HOME

        oBrowse:panHome()

    case nKey == K_CTRL_END

        oBrowse:panEnd()

    case nKey == K_RETURN

        *---------------------------------------------------------
        * Edit if you can.
        *---------------------------------------------------------

        do case
        case !lModify

            *-----------------------------------------------------
            * Read only.
            * No edit allowed.
            *-----------------------------------------------------

        case alColCalc[oBrowse:colPos]

            *-----------------------------------------------------
            * Calculated field.
            * No edit allowed.
            *-----------------------------------------------------

        otherwise

            *-----------------------------------------------------
            * Try to lock the record to edit.
            *-----------------------------------------------------

            if ( tbAlias(oBrowse, acCol, alColCalc) )->(rlock() )

                *-------------------------------------------------
                * Try to handle error that may comes.
                *-------------------------------------------------

                bOldErrorHandler :=;
                    errorblock( {|e| ErrorHandler(e)} )

                begin sequence

                    tbDoGet(oBrowse, acCol, abColValid, lMemo)

                    dbcommit()

                recover

                    *---------------------------------------------
                    * No recovery procedure.
                    *---------------------------------------------

                end
                errorblock(bOldErrorHandler)

                dbunlock()

            else

                alertBox( TB_ERROR_RECORD_LOCKED )

            end

        end

    case;
        nKey == K_CTRL_Y        .or.;
        nKey == K_CTRL_DEL

        *---------------------------------------------------------
        * Delete/Undelete if you can.
        *---------------------------------------------------------

        do case
        case;
            lModify         .and.;
            lDelete         .and.;
            !deleted()

            *-----------------------------------------------------
            * Try to lock the record to edit.
            *-----------------------------------------------------

            if  ( tbAlias(oBrowse, acCol, alColCalc) )->(rlock());   
                                                        .and.;
                alertBox(;
                    TB_PROMPT_DELETE_RECORD,;
                    { _MENU_NO, _MENU_YES };
                ) == 2

                dbdelete()

            else

                alertBox( TB_ERROR_RECORD_LOCKED )

            end

            dbcommit()

            dbunlock()

        case;
            lModify         .and.;
            lDelete         .and.;
            deleted()

            *-----------------------------------------------------
            * Try to lock the record to edit.
            *-----------------------------------------------------

            if ( tbAlias(oBrowse, acCol, alColCalc) )->(rlock())

                dbRecall()

                dbcommit()

                dbunlock()

            else

                alertBox( TB_ERROR_RECORD_LOCKED )

            end

        end

    case nKey == K_CTRL_ENTER

        *---------------------------------------------------------
        * Append if you can.
        *---------------------------------------------------------

        if lAppend

            *-----------------------------------------------------
            * Try to lock the file to append.
            *-----------------------------------------------------

            if  ( tbAlias(oBrowse, acCol, alColCalc) )->(flock() )

                dbappend()
                
                *-------------------------------------------------
                * Try to get the record's key value after the GET.
                *-------------------------------------------------

                xKey := if(empty(indexkey()), NIL, &(indexkey()))
                
                if  xKey == NIL
                
                    *---------------------------------------------
                    * No Index key is available.
                    *---------------------------------------------
                
                    oBrowse:inValidate()
                    oBrowse:refreshAll()
                    oBrowse:forceStable()
                    oBrowse:goBottom()

                else
                
                    oBrowse:inValidate()
                    oBrowse:refreshAll()
                    oBrowse:forceStable()
                    
                    *---------------------------------------------
                    * Make sure we're still on the right record
                    * after stabilizing.
                    *---------------------------------------------

                    while; 
                        &(indexkey()) > xKey            .and.; 
                        !oBrowse:hitTop()

                        oBrowse:up()
                        oBrowse:forceStable()

                    end

                end

                *-------------------------------------------------
                * Release file lock.
                *-------------------------------------------------

                dbunlock()

            end

        end

    case nKey == K_CTRL_F3

        *---------------------------------------------------------
        * Append if you can.
        *---------------------------------------------------------

        if  lAppend                     .and.;
            !( axColCopy == NIL )

            *-----------------------------------------------------
            * Try to lock the file to append.
            *-----------------------------------------------------

            if  ( tbAlias(oBrowse, acCol, alColCalc) )->(flock() )

                dbappend()

                *-------------------------------------------------
                * Transfer data from the temporary <axColCopy>.
                *-------------------------------------------------

                tbPaste( axColCopy, acCol )
                
                *-------------------------------------------------
                * Try to get the record's key value after the GET.
                *-------------------------------------------------

                xKey := if(empty(indexkey()), NIL, &(indexkey()))
                
                if  xKey == NIL
                
                    *---------------------------------------------
                    * No Index key is available.
                    *---------------------------------------------
                
                    oBrowse:inValidate()
                    oBrowse:refreshAll()
                    oBrowse:forceStable()
                    oBrowse:goBottom()

                else
                
                    oBrowse:inValidate()
                    oBrowse:refreshAll()
                    oBrowse:forceStable()
                    
                    *---------------------------------------------
                    * Make sure we're still on the right record
                    * after stabilizing.
                    *---------------------------------------------

                    while; 
                        &(indexkey()) > xKey            .and.; 
                        !oBrowse:hitTop()

                        oBrowse:up()
                        oBrowse:forceStable()

                    end

                end

                *-------------------------------------------------
                * Release file lock.
                *-------------------------------------------------

                dbunlock()

            end

        end

    case nKey == K_CTRL_F4

        *---------------------------------------------------------
        * Paste if you can.
        *---------------------------------------------------------

        if  lModify                     .and.;
            !( axColCopy == NIL )

            *-----------------------------------------------------
            * Try to lock the record.
            *-----------------------------------------------------

            if  ( tbAlias(oBrowse, acCol, alColCalc) )->(rlock() )

                *-------------------------------------------------
                * Transfer data from the temporary <axColCopy>.
                *-------------------------------------------------

                tbPaste( axColCopy, acCol )
                
                *-------------------------------------------------
                * Try to get the record's key value after the GET.
                *-------------------------------------------------

                xKey := if(empty(indexkey()), NIL, &(indexkey()))
                
                if  xKey == NIL
                
                    *---------------------------------------------
                    * No Index key is available.
                    *---------------------------------------------
                
                    oBrowse:inValidate()
                    oBrowse:refreshAll()
                    oBrowse:forceStable()
                    oBrowse:goBottom()

                else
                
                    oBrowse:inValidate()
                    oBrowse:refreshAll()
                    oBrowse:forceStable()
                    
                    *---------------------------------------------
                    * Make sure we're still on the right record
                    * after stabilizing.
                    *---------------------------------------------

                    while; 
                        &(indexkey()) > xKey            .and.; 
                        !oBrowse:hitTop()

                        oBrowse:up()
                        oBrowse:forceStable()

                    end

                end

                *-------------------------------------------------
                * Release record lock.
                *-------------------------------------------------

                dbunlock()

            end

        end

    case nKey == K_CTRL_F2

        *---------------------------------------------------------
        * Copy current record.
        *---------------------------------------------------------

        tbCopy( @axColCopy, acCol, alColCalc )

    otherwise

        *---------------------------------------------------------
        * It must be a editing key, so start
        * etiting (if possible) and stuff the
        * key again into the keyboard buffer.
        *---------------------------------------------------------

        do case
        case !lModify

            *-----------------------------------------------------
            * No edit allowed.
            *-----------------------------------------------------

        case alColCalc[oBrowse:colPos]

            *-----------------------------------------------------
            * No edit allowed.
            *-----------------------------------------------------

        case lMemo

            *-----------------------------------------------------
            * No edit allowed.
            *-----------------------------------------------------

        otherwise

            *-----------------------------------------------------
            * Save all pending keys.
            *-----------------------------------------------------

            cKeyString := chr( nKey )

            while (nKey := inkey()) > 0

                cKeyString += chr( nKey )

            end

            *-----------------------------------------------------
            * Try to lock the record before editing.
            *-----------------------------------------------------

            if ( tbAlias(oBrowse, acCol, alColCalc) )->(rlock() )

                *-------------------------------------------------
                * Try to handle error that may comes.
                *-------------------------------------------------

                bOldErrorHandler :=;
                    errorblock( {|e| ErrorHandler(e)} )

                begin sequence

                    *---------------------------------------------
                    * Repeat the keys for the following tbDoGet().
                    *---------------------------------------------

                    keyboard( cKeyString )

                    *---------------------------------------------
                    * Do editing.
                    *---------------------------------------------

                    tbDoGet(oBrowse, acCol, abColValid, lMemo)

                    dbcommit()

                recover

                    *---------------------------------------------
                    * No recovery procedure.
                    *---------------------------------------------

                end
                errorblock(bOldErrorHandler)

                dbunlock()

            else

                alertBox( TB_ERROR_RECORD_LOCKED )

            end

        end

    end

    return NIL

*-----------------------------------------------------------------
static function tbDoGet(oBrowse, acCol, abColValid, lMemo)
*
* Do a GET for the current column in the browse.
*

    local lFlag := .T.
    local oCol
    local aoGet             := {}
    local nKey
    local bSavIns
    local nSavRecNo := recno()
    local xNewKey
    local xSavKey
    local cOldScreen
    local cOldColor
    local nSetCursor
    local cMemoOld
    local cMemoNew
    local nRow
    local nCol
    
    *-------------------------------------------------------------
    * Make sure screen is fully updated,
    * dbf position is correct, etc.
    *-------------------------------------------------------------

    oBrowse:forceStable()

    *-------------------------------------------------------------
    * Save the current record's key value (or NIL)
    *-------------------------------------------------------------

    xSavKey := iif( empty( indexkey() ), NIL, &( indexkey() ) )

    *-------------------------------------------------------------
    * Get the current column object from the browse.
    *-------------------------------------------------------------

    oCol := oBrowse:getColumn(oBrowse:colPos)

    *-------------------------------------------------------------
    * Memo fields are edited in a different way.
    *-------------------------------------------------------------

    if lMemo

        *---------------------------------------------------------
        * Copy the memo field.
        *---------------------------------------------------------

        cMemoOld := rtrim( &(acCol[oBrowse:colPos]) )

        *---------------------------------------------------------
        * Edit the memo field.
        *---------------------------------------------------------

        cMemoNew :=;
            memoWindow(;
                cMemoOld, oCol:heading,;
                maxrow()/2, 00, maxrow(), maxcol(),;
                COLOR_HEAD, COLOR_BODY,;
                .T., _MAX_STRING_LEN;
            )

        *---------------------------------------------------------
        * If data is changed, update the memo field.
        *---------------------------------------------------------

        if !( cMemoOld == cMemoNew )
        
            *-----------------------------------------------------
            * Attention: the following assignment is very
            * delicate: it works only if <acCol[oBrowse:colPos]>
            * return a field name complete with alias
            * ( ALIAS->NAME ).
            * I don't know what to do to resolve the problem.
            *-----------------------------------------------------

            &(acCol[oBrowse:colPos]) := cMemoNew

            dbcommit()

        end

    else

        *---------------------------------------------------------
        * Loop to check for valid data.
        *---------------------------------------------------------

        while .T.

            *-----------------------------------------------------
            * Save cursor position.
            *-----------------------------------------------------

            nRow := row()

            nCol := col()

            *-----------------------------------------------------
            * Create a corresponding GET.
            *-----------------------------------------------------

            aoGet :=;
                {;
                    getnew(;
                        row(), col(),;
                        oCol:block,;
                        oCol:heading,;
                        oCol:picture,;
                        oBrowse:colorSpec;
                    );
                }

            *-----------------------------------------------------
            * Set insert key to toggle insert mode and cursor
            * shape.
            *-----------------------------------------------------

            bSavIns := setkey(K_INS, { || tglInsert() })

            *-----------------------------------------------------
            * Set initial cursor shape.
            *-----------------------------------------------------

            if set( _SET_INSERT )

                setcursor( SETCURSOR_INSERT )

            else

                setcursor( SETCURSOR_NORMAL )

            end

            *-----------------------------------------------------
            * Read.
            *-----------------------------------------------------

            read(aoGet)
            aoGet := {}

            *-----------------------------------------------------
            * Hide cursor.
            *-----------------------------------------------------

            setcursor(SETCURSOR_NONE)

            *-----------------------------------------------------
            * Restore [Ins] key.
            *-----------------------------------------------------

            setkey(K_INS, bSavIns)

            *-----------------------------------------------------
            * Restore cursor position before data Validation.
            *-----------------------------------------------------

            setpos( nRow, nCol )

            *-----------------------------------------------------
            * Check for valid data.
            *-----------------------------------------------------

            if eval( abColValid[oBrowse:colPos] )

                *-------------------------------------------------
                * Data is valid: exit.
                *-------------------------------------------------

                exit                                    // EXIT

            else

                *-------------------------------------------------
                * Repeat Get/Read loop.
                *-------------------------------------------------

            end

        end

    end

    *-------------------------------------------------------------
    * Try to get the record's key value after the GET.
    *-------------------------------------------------------------

    xNewKey := if(empty(indexkey()), NIL, &(indexkey()))

    oBrowse:inValidate()
    oBrowse:refreshAll():forceStable()

    *-------------------------------------------------------------
    * if the key has changed (or if this is a new record).
    *-------------------------------------------------------------

    if  !(xNewKey == xSavKey)       .and.;          
        !(xNewKey == NIL)

        *---------------------------------------------------------
        * Do a complete refresh.
        *---------------------------------------------------------

        oBrowse:refreshAll():forceStable()

        *---------------------------------------------------------
        * Make sure we're still on the right record
        * after stabilizing.
        *---------------------------------------------------------

        while &(indexkey()) > xNewKey .and. !oBrowse:hitTop()

            oBrowse:up():forceStable()

        end

    end

    *-------------------------------------------------------------
    * Check exit key from get.
    *-------------------------------------------------------------

    nKey := lastkey()

    if  nKey == K_UP        .or.;
        nKey == K_DOWN      .or. ;
        nKey == K_PGUP      .or.;
        nKey == K_PGDN

        keyboard(chr(nKey))

    end

    return NIL

*-----------------------------------------------------------------
static function tbBlockCalc( cCalc )
*
* Return the column code block.
*
* In fact this function seems to do nothing important, but
* this action cannot be written where this function is called.
* In that case:
* as we are using the macro operator to resolve the content
* of <cCalc>, it happens that the calculated column will contain
* the content of the last column.
*

    return bCompile( cCalc )

*-----------------------------------------------------------------
static function tbBlockField( cFieldName )
*
* return the column code block
*
    local bFieldWBlock
    local cAlias
    local cField
    local nArrowStart
    local nSelect

    *-------------------------------------------------------------
    * Locate the "->" symbol inside the field name.
    *-------------------------------------------------------------

    nArrowStart := at( "->", cFieldName )

    do case
    case nArrowStart == 0

        *---------------------------------------------------------
        * There is no arrow, so, no alias was specified.
        *---------------------------------------------------------

        cAlias := NIL

        cField := alltrim( cFieldName )

    case nArrowStart > 0

        *---------------------------------------------------------
        * The arrow symbol exists: divide the Alias name and the
        * Field name.
        *---------------------------------------------------------

        cAlias := substr( cFieldName, 1, nArrowStart-1 )

        cField := substr( cFieldName, nArrowStart+2 )

    end

    *-------------------------------------------------------------
    * Create the Code Block.
    *-------------------------------------------------------------

    do case
    case cAlias == NIL

        bFieldWblock := fieldwblock( cField, select() )

    case upper(cAlias) == "FIELD"

        bFieldWblock := fieldwblock( cField, select() )

    case;
        upper(cAlias) == "MEMVAR"       .or. ;
        upper(cAlias) == "M"

        bFieldWblock := memvarblock( cField )

    otherwise

        nSelect := select(cAlias)

        bFieldWblock := fieldwblock( cField, nSelect )

    end

    *-------------------------------------------------------------
    * Return the code block.
    *-------------------------------------------------------------

    return bFieldWBlock

*-----------------------------------------------------------------
static function tbAlias( oBrowse, acCol, alColCalc )
*
* return the alias name for the actual column
*
    local cAlias
    local nArrowStart

    *-------------------------------------------------------------
    * Locate the "->" symbol inside the field name.
    *-------------------------------------------------------------

    nArrowStart := at( "->", acCol[oBrowse:colPos] )

    *-------------------------------------------------------------
    * Extract Alias name.
    *-------------------------------------------------------------

    do case
    case alColCalc[oBrowse:colPos]

        *---------------------------------------------------------
        * Calculated fields have no Alias.
        *---------------------------------------------------------

        cAlias := alias()

    case nArrowStart == 0

        *---------------------------------------------------------
        * There is no arrow, so, no alias was specified.
        *---------------------------------------------------------

        cAlias := alias()

    case nArrowStart > 0

        *---------------------------------------------------------
        * The Alias name may be extracted.
        *---------------------------------------------------------

        cAlias := substr( acCol[oBrowse:colPos], 1, nArrowStart-1 )

    end

    *-------------------------------------------------------------
    * Return the Alias name.
    *-------------------------------------------------------------

    return alltrim(cAlias)

*-----------------------------------------------------------------
static function tbFieldAdd( cName )
*
* Return <cName> preceded form "Field->" if it hasn't one.
*
    
    local nArrowStart

    *-------------------------------------------------------------
    * Locate the "->" symbol inside the field name.
    *-------------------------------------------------------------

    nArrowStart := at( "->", cName )

    *-------------------------------------------------------------
    * Add "Field->".
    *-------------------------------------------------------------

    if  nArrowStart == 0

        cName := "Field->" + alltrim( cName )
    
    end

    *-------------------------------------------------------------
    * Return the corrected <cName>.
    *-------------------------------------------------------------
    
    return cName

*-----------------------------------------------------------------
static function tbDefault( acCol, acColSayPic, alColCalc,;
    acColHead, acColFoot, abColValid, abColMsg )
*

    *-------------------------------------------------------------
    * If there is no active Alias, no default is created.
    *-------------------------------------------------------------

    if ( alias() == "" )

         return NIL                                     // RETURN

    end

    *-------------------------------------------------------------
    * prepare arrays
    *-------------------------------------------------------------

    acCol       := {}
    acColSayPic := {}
    alColCalc   := {}
    acColHead   := {}
    acColFoot   := {}
    abColValid  := {}
    abColMsg    := NIL

    *-------------------------------------------------------------
    * The first default column.
    *-------------------------------------------------------------

    aadd(;
        acCol,;
        "iif( deleted(), '*', ' ') +" + " str(recno(), 5, 0)";
    )

    aadd( alColCalc, .T. )

    aadd( acColSayPic, "!!!!!!" )

    aadd( acColHead, "Rec. #" )

    aadd( acColFoot, "" )

    aadd( abColValid, {||.T.} )

    *-------------------------------------------------------------
    * Add the other columns.
    *-------------------------------------------------------------

    tbDefColAdd( @acCol, @acColSayPic, @alColCalc,;
        @acColHead, @acColFoot,;
        @abColValid, @abColMsg )

    return NIL

*-----------------------------------------------------------------
static function tbDefColAdd( acCol, acColSayPic, alColCalc,;
    acColHead, acColFoot, abColValid, abColMsg )
*

    local aStruct   := {}
    local nFields   := 0
    local nColIndex := 0
    local nRelations := 0
    local nIndRel := 0
    local cChildAlias := ""

    default( @acCol,        {} )
    default( @acColSayPic,  {} )
    default( @alColCalc,    {} )
    default( @acColHead,    {} )
    default( @acColFoot,    {} )
    default( @abColValid,   {} )
    default( @abColMsg,     NIL )

    *-------------------------------------------------------------
    * Relation number determination.
    *-------------------------------------------------------------

    nRelations := 0

    while .T.                                           // FOREVER

        nRelations++

        if dbrselect( nRelations ) > 0

            *-----------------------------------------------------
            * Go on.
            *-----------------------------------------------------

        else

            *-----------------------------------------------------
            * nRelation must be reduced as the last is not true.
            *-----------------------------------------------------

            nRelations--

            exit                                        // EXIT

        end

    end

    *-------------------------------------------------------------
    * Now <nRelations> contains the number of relations
    * established with the original Alias.
    *-------------------------------------------------------------

    *-------------------------------------------------------------
    * File analisys.
    *-------------------------------------------------------------

    aStruct := dbstruct()

    nFields := len( aStruct )

    *-------------------------------------------------------------
    * Arrays compilation.
    *-------------------------------------------------------------

    for nColIndex := 1 to nFields

        aadd( acCol, Alias()+"->"+aStruct[nColIndex, 1] )

        aadd( alColCalc, .F. )

        aadd(;
            acColSayPic,;
            tbDefPicture(;
                aStruct[nColIndex, 1],;
                aStruct[nColIndex, 2],;
                aStruct[nColIndex, 3],;
                aStruct[nColIndex, 4];
            );
        )

        aadd( acColHead, Alias()+"->"+aStruct[nColIndex, 1] )

        aadd( acColFoot, "" )

        aadd( abColValid, {||.T.} )

    end

    *-------------------------------------------------------------
    * Fields contained inside related Alias are added.
    *-------------------------------------------------------------

    if nRelations > 0

        for nIndRel := 1 to nRelations

            *-----------------------------------------------------
            * Child Alias name.
            *-----------------------------------------------------

            cChildAlias := alias( dbrselect( nIndRel ) )

            *-----------------------------------------------------
            * Recursivelly collection of the fields contained
            * inside <cChildAlias>.
            *-----------------------------------------------------

            (cChildAlias)->(tbDefColAdd( @acCol,;
                @acColSayPic, @alColCalc,;
                @acColHead, @acColFoot,;
                @abColValid, @abColMsg ) )

        next                                            // EXIT

    end

    return NIL

*-----------------------------------------------------------------
static function tbDefPicture( cColumn, cColumnType, nColumnLen,;
                    nColumnDec )

    local nLen := 0
    local cColSayPic        := ""
    local nMaxWidth         := ( maxcol() + 1 ) /2

    *-------------------------------------------------------------
    * The column picture depends on the data type.
    *-------------------------------------------------------------

    do case
    case cColumnType == "C"

        nLen := len( &(cColumn) )

        if nLen > nMaxWidth

            cColSayPic := "@s" + ltrim( str( nMaxWidth ) )

        else

            cColSayPic := replicate( "x", nLen )

        end

    case cColumnType == "N"

        if nColumnDec > 0

            *---------------------------------------------------------
            * It contains the decima point and decima value.
            *---------------------------------------------------------

            cColSayPic := ;
                replicate ( "9", nColumnLen-1-nColumnDec ) +;
                "." +;
                replicate ( "9", nColumnDec )

        else

            *---------------------------------------------------------
            * Integer.
            *---------------------------------------------------------

            cColSayPic := ;
                replicate ( "9", nColumnLen )
        end

    case cColumnType == "D"

        cColSayPic := "99/99/9999"

    case cColumnType == "L"

        cColSayPic := "L"

    case cColumnType == "M"

        cColSayPic := "@s40" // <Memo>

    end

    *-------------------------------------------------------------
    * The picture is returned.
    *-------------------------------------------------------------

    return cColSayPic

*-----------------------------------------------------------------
static function tbCopy( axColCopy, acCol, alColCalc )
*
* <@axColCopy>          the array of copied data.
*
* <acCol>               the array of character fields containing
*                       the columns content.
*
* <alColCalc>           the array of calculated fields.
*
*
* The function copies the data of a line for future paste.
*

    local nI

    *-------------------------------------------------------------
    * Substitutes <axColCopy> with an empty array.
    *-------------------------------------------------------------
    
    axColCopy := array( len( alColCalc ) )

    for nI := 1 to len( alColCalc )

        *---------------------------------------------------------
        * If the field is not a calculated one, make a copy
        * of it inside <axColCopy>.
        *---------------------------------------------------------
        
        if !alColCalc[nI]

            axColCopy[nI] := &(acCol[nI])

        end
    
    next

    return NIL

*-----------------------------------------------------------------
static function tbPaste( axColCopy, acCol )
*
* <axColCopy>           the array of copied data.
*
* <acCol>               the array of character fields containing
*                       the columns content.
*
* The function pastes the data of a previous copied line
* inside the current record.
*

    local nI
    local cAssign
    
    memPublic( "___dummy" )

    *-------------------------------------------------------------
    * If <axColCopy> is NIL and not an array, no paste can be
    * done.
    *-------------------------------------------------------------
    
    if axColCopy == NIL

        return NIL                                      // RETURN

    end

    *-------------------------------------------------------------
    * Paste the fields for every element contained inside
    * <axColCopy> that is not NIL. When <axColCopy> contains
    * NIL inside the elements, these correspond to calculated
    * fields.
    *-------------------------------------------------------------
    
    for nI := 1 to len( axColCopy )

        if !( axColCopy[nI] == NIL )

            *-----------------------------------------------------
            * The assignment of the Alias fields is a little bit
            * compilcated as the fields name are contained inside
            * <acCol> as character string.
            * A code block will be created starting from a
            * string command.
            * In that case, all variables contained inside this
            * kind of code block must be MEMVARs or FIELDs.
            *-----------------------------------------------------
            
            Memvar->___dummy := axColCopy[nI]

            *-----------------------------------------------------
            * The assignment will not work if the Field name
            * is not preceded form the alias name or the word
            * FIELS with the "->" symbol.
            *-----------------------------------------------------
            
            cAssign := tbFieldAdd( acCol[nI] ) + ":= ___dummy" 
            
            eval( bCompile( cAssign ) )
        
        end
    
    next

    memRelease( "___dummy" )

    return NIL

*-----------------------------------------------------------------
static function tbMouseKeyboard(;
    aButtons,;
    nTop, nLeft, nBottom, nRight,; 
    oBrowse, anColWidth, anColShift;
    )
*
*
*

    local aMouse    := mouse()

    local nMCol      := aMouse[1]-1
    local nMRow      := aMouse[2]-1

    local nTimes    := 0
    local cKeyboard := ""
    local nI

    local nButtRow
    local nButtCol
    local nButtColEnd

    *-------------------------------------------------------------
    * If <aButtons> is an array, test if the mouse selected
    * a button.
    *-------------------------------------------------------------

    if valtype( aButtons ) == "A"

        for nI := 1 to len(aButtons)

            nButtRow    := aButtons[nI][1]
            nButtCol    := aButtons[nI][2]
            nButtColEnd := nButtCol + len( aButtons[nI][3] ) -1

            if nButtRow == nMRow;
                .and. nButtCol <= nMCol;
                .and. nButtColEnd >= nMCol

                *-------------------------------------------------
                * Ok button selected. Do the action and terminate.
                *-------------------------------------------------

                eval( aButtons[nI][4] )

                return NIL

            end

        next

    end

    *-------------------------------------------------------------
    * If still here, it wasn't a button.
    * If it happened inside the right area, transform into
    * keyboard.
    *-------------------------------------------------------------

    if  nMRow >= nTop           .and.;
        nMRow <= nBottom        .and.;
        nMCol >= nLeft          .and.;
        nMCol <= nRight
        
        // Ok, this is the right place.
    
    else

        return NIL                                      // RETURN

    end

    *-------------------------------------------------------------
    * If still here, the place was right.
    * Transform into keyboard.
    *-------------------------------------------------------------

    *-------------------------------------------------------------
    * First the vertical movement.
    *-------------------------------------------------------------
    
    do case
    case nMRow > row()

        nTimes := nMRow - row()

        for nI := 1 to nTimes

            cKeyboard += chr( K_DOWN )

        end
    
    case nMRow < row()

        nTimes := row() - nMRow

        for nI := 1 to nTimes

            cKeyboard += chr( K_UP )

        end

    end

    *-------------------------------------------------------------
    * Then the horizontal movement.
    *-------------------------------------------------------------
    
    cKeyboard += tbMouseHorizontal( oBrowse, anColWidth, anColShift, nMCol ) 

    *-------------------------------------------------------------
    * If something was done, transfer the keys into the
    * keyboard buffer.
    *-------------------------------------------------------------
    
    if len( cKeyboard ) > 0

        keyboard( cKeyboard )

    end

    return NIL

*-----------------------------------------------------------------
static function tbMouseHorizontal(;
        oBrowse,;
        anColWidth, anColShift,;
        nMCol;
    )
*

    local cHorKey       := ""
    local nCol          := oBrowse:colPos
    local nScreenCol    := col() - anColShift[nCol]

    while .T.                                           // FOREVER

        do case
        case; 
            nMCol >= nScreenCol                         .and.;
            nMCol <= nScreenCol + anColWidth[nCol]

            *-----------------------------------------------------
            * The mouse cursor is inside the area of the current
            * column. 
            *-----------------------------------------------------
            
            exit                                        // EXIT

        case;
            nMCol > nScreenCol + anColWidth[nCol]

            *-----------------------------------------------------
            * The mouse cursor is on the area of a different
            * column on the right
            *-----------------------------------------------------
            
            if nCol < len( anColWidth )

                *-------------------------------------------------
                * It is possible to go right.
                *-------------------------------------------------
                
                nScreenCol += anColWidth[nCol] +1
                nCol++
                cHorKey += chr( K_RIGHT )

            else
                
                *-------------------------------------------------
                * No more columns are available.
                *-------------------------------------------------
            
                exit                                    // EXIT

            end

        case;
            nMCol < nScreenCol

            *-----------------------------------------------------
            * The mouse cursor is on the area of a different
            * column on the left
            *-----------------------------------------------------
            
            if nCol > 1
                
                *-------------------------------------------------
                * It is possible to go left.
                *-------------------------------------------------
                
                nCol--
                nScreenCol -= (anColWidth[nCol] +1)
                cHorKey += chr( K_LEFT )

            else
                *-------------------------------------------------
                * No more columns are available.
                *-------------------------------------------------
            
                exit                                    // EXIT

            end

        end

    end

    *-------------------------------------------------------------
    * The string containing the key to be pressed to reach the
    * column, is returned.
    *-------------------------------------------------------------
    
    return cHorKey

*-----------------------------------------------------------------
static function tbButtonSay( aButtons )
*
* aButtons[1] = nRow
* aButtons[2] = nCol
* aButtons[3] = cText
* aButtons[4] = cColor
* aButtons[5] = bAction

    local nI

    for nI := 1 to len( aButtons )

        say(;
            aButtons[nI][1], aButtons[nI][2],;
            aButtons[nI][3],, COLOR_BUTTON )
    next

    return NIL

*=================================================================
* TEXT()
*=================================================================

#define TEXT_KEY_REMINDER;
   "[Esc] Exit  [F7] Print  []/[]/[Pag]/[Pag] Move text."

#define TEXT_END    NL(1) + ""

*=================================================================
function Text( cText )
*
* Text( <cText> ) --> NIL
*
*
* <cText>      Text to display.
*

    local nOldCursor := setcursor( SETCURSOR_NONE )
    local cOldScreen := mouseScrSave()
    local cOldColor  := setcolor()
    local bOld_F1    := setkey( K_F1, NIL ) // to avoid recursion
                                            // when usign it as
                                            // help
    local bOld_F7    :=;
        setkey( K_F7, {|| textPrint( cText ) } )
    local nOldRow    := row()
    local nOldCol    := col()

    *-------------------------------------------------------------
    * Start a sequence.
    *-------------------------------------------------------------

    begin sequence

        *---------------------------------------------------------
        * If <cText> was not given, there is nothing to show.
        *---------------------------------------------------------

        if empty( cText )

            break                                       // BREAK

        end

        *---------------------------------------------------------
        * Create a window, but use all the availabel space.
        *---------------------------------------------------------

        setcolor( COLOR_BODY )
        scroll( 0, 0, maxrow(), maxcol() )
        dispBoxShadow(;
            0, 0, maxrow(), maxcol(),;
            1,;
            dispBoxColor( 1 ),;
            dispBoxColor( 2 );
            )
        say(;
            maxrow()-1, 1,;
            padc( TEXT_KEY_REMINDER, maxcol()-1 ),;
            ,;
            COLOR_HEAD;
            )

        *---------------------------------------------------------
        * Show the text.
        * Sorry: no mouse support.
        *---------------------------------------------------------

        memoedit( cText + TEXT_END, 01, 01,maxrow()-2,maxcol()-1, .F. )

    end

    *-------------------------------------------------------------
    * Restore previous data.
    *-------------------------------------------------------------

    setcursor( nOldCursor )
    setcolor( cOldColor )
    mouseScrRestore( NIL, NIL, NIL, NIL, cOldScreen )
    setkey( K_F1, bOld_F1 )
    setkey( K_F7, bOld_F7 )
    setpos( nOldRow, nOldCol )

    return NIL

*-----------------------------------------------------------------
static function textPrint( cText )
*
*

    local nLine
    local nLines    := mlcount( cText, 80 )

    *-------------------------------------------------------------
    * It prints on the output peripheral, no matter if it is a
    * screen, a file, a printer...
    *-------------------------------------------------------------

    for nLine := 1 to nLines
        qout( memoline( cText, 80, nLine ) )
    next

    *-------------------------------------------------------------
    * Form Feed in the end.
    *-------------------------------------------------------------

    qqout( FF )

    return NIL

*=================================================================
* TGLINSERT()
*=================================================================
function tglInsert()
*
* TglInsert() -->  NIL
*
* Toggle the global insert mode and the cursor shape.
*
*
    if set( _SET_INSERT )
        set( _SET_INSERT, .F. )
        setcursor(SETCURSOR_NORMAL)
    else
        set( _SET_INSERT, .T. )
        setcursor(SETCURSOR_INSERT)
    end

    return NIL

*=================================================================
* TRUE()
* TRUESETKEY()
*=================================================================
function true( p1, p2, p3, p4, p5, p6, p7, p8, p9, p10, p11, p12, p13, p14, p15 )
*
* true( , , , , , , , , , ) --> true
*
* It does nothing, it returns true.
*

    return .t.

*=================================================================
function trueSetKey( nInkeyCode, bAction )
*
* trueSetKey( <nInkeyCode>, [<bAction>] ) --> true
*
* Just like SetKey() but returns ever true (.t.) and not the
* previous bAction.
*

    return true( setkey( nInkeyCode, bAction ) )

*=================================================================
* WAITFOR()
* WAITFILEEVAL()
* WAITPROGRESS()
* WAITWHEEL()
*=================================================================
#define WAIT_DO_YOU_WANT_TO_BREAK;
    "[Esc] was pressed.;" +;
    "Do you want to break or continue the process?;" +;
    "As 'break' is not ever safe, if you select 'break', " +;
    "you accept all the consequences of this choice!"

#define WAIT_MENU_CHOICE_BREAK         "Break"
#define WAIT_MENU_CHOICE_CONTINUE      "Continue"

*=================================================================
function waitFor( cMessage )
*
* This function was originally named "Wait()", but the name is
* now changed as the compiler confuses it with the command
* WAIT
*
* WaitFor( [<cMessage>] ) --> .T.
*
* <cMessage>   Text message to show. If it is NIL, it closes
*              the message.
*
* It shows a wait message at the center of the screen.
*
* WaitFor() without argument, closes the message.
*
* This function must be closed.
*

    static cOldScreen
    static nTop
    static nLeft
    static nBottom
    static nRight

    local nOldCursor := setcursor( SETCURSOR_NONE )
    local cOldColor  := setcolor()
    local nOldRow    := row()
    local nOldCol    := col()

    local nWidth := 0

    local nLines
    local nLine

    *-------------------------------------------------------------
    * Start a new sequence.
    *-------------------------------------------------------------
    
    begin sequence

        *---------------------------------------------------------
        * Is it a Wait close?
        *---------------------------------------------------------
        
        if valtype( cMessage ) <> "C"

            if cOldScreen <> NIL

                mouseScrRestore( nTop,nLeft,nBottom,nRight, cOldScreen )

                cOldScreen := NIL

            end

            break                                       // BREAK

        end

        *---------------------------------------------------------
        * Is it an already opened Wait?
        *---------------------------------------------------------
        
        if cOldScreen <> NIL

            mouseScrRestore( nTop,nLeft,nBottom,nRight, cOldScreen )

            cOldScreen := NIL

        end

        *---------------------------------------------------------
        * Prepare to display the wait message.
        * Lines count.
        *---------------------------------------------------------
        
        cMessage := alltrim(cMessage)

        nLines := mlcount( cMessage, maxcol()-1 )

        *---------------------------------------------------------
        * Determinate the message width.
        *---------------------------------------------------------
        
        for nLine := 1 to nLines
            
            nWidth :=;
                max(; 
                    nWidth,;
                    len(; 
                        rtrim(; 
                            memoline( cMessage, maxcol()-1, nLine ); 
                        ); 
                    ); 
                )
        
        next

        *---------------------------------------------------------
        * Calculate the window coordinates.
        *---------------------------------------------------------
        
        nTop        := int( ( (maxrow()+1) - (nLines+2) ) / 2 )
        nLeft       := int( ( (maxcol()+1) - (nWidth+2) ) / 2 )
        nBottom     := nTop + nLines+1
        nRight      := nLeft + nWidth+1

        *---------------------------------------------------------
        * Save window screen.
        *---------------------------------------------------------
        
        cOldScreen := mouseScrSave( nTop, nLeft, nBottom, nRight )

        *---------------------------------------------------------
        * Use the right color.
        *---------------------------------------------------------
        
        setcolor( COLOR_MESSAGE )

        *---------------------------------------------------------
        * Display the box.
        *---------------------------------------------------------
        
        dispBoxShadow(;
            nTop, nLeft, nBottom, nRight,;
            1,;
            dispBoxColor( 1 ),;
            dispBoxColor( 2 );
        )

        *---------------------------------------------------------
        * Display the message.
        *---------------------------------------------------------
        
        for nLine := 1 to nLines
            
            say(;
                nTop+nLine, nLeft+1,;
                memoline(cMessage, nWidth, nLine );
            )
        
        next

    end //sequence

    *-------------------------------------------------------------
    * Restore old values.
    *-------------------------------------------------------------
    
    setcursor( nOldCursor )
    setcolor( cOldColor )
    setpos( nOldRow, nOldCol )

    return NIL

*=================================================================
function waitFileEval( lClose )
*
* WaitFileEval( [<lClose>] ) --> .T.
*
* Shows a wait bar calculated with recno()/lastrec(),
*
* This function must be "closed".
*

    static nI
    static lIndex

    default( @lClose,   .F. )

    do case
    case lClose

        nI      := NIL
        lIndex  := NIL

        *---------------------------------------------------------
        * Close WaitProgress().
        *---------------------------------------------------------
        
        waitProgress(2) // 2 > 100%

        return .T.                                      // RETURN

    case nI == NIL

        nI      := 0

        if  !(alias() == "")                    .and.;
            !( ordsetfocus() == "" )

            *-----------------------------------------------------
            * The active Alias is controlled by an index key:
            * the wait bar progress cannot be calculated on the
            * record position ( recno()/lastrec() ).
            *-----------------------------------------------------
            
            lIndex := .T.

        else

            lIndex := .F.

        end

    end

    *-------------------------------------------------------------
    * If [Esc] is pressed, a break() is possible.
    *-------------------------------------------------------------
    
    if inkey() == K_ESC

        *---------------------------------------------------------
        * Do you want to break?
        *---------------------------------------------------------
        
        if  alertBox(; 
                WAIT_DO_YOU_WANT_TO_BREAK,;
                {; 
                    WAIT_MENU_CHOICE_CONTINUE,;
                    WAIT_MENU_CHOICE_BREAK;
                }; 
            ) == 2

            break                                       // BREAK

        end

    end

    if lIndex

        nI++

        waitProgress( nI/lastrec() )

    else

        if recno() == lastrec()

            *-----------------------------------------------------
            * Close WaitProgress().
            *-----------------------------------------------------
            
            waitProgress(2) // 2 > 100%

        else

            waitProgress( recno()/lastrec() )

        end

    end

    *-------------------------------------------------------------
    * This function returns ever True.
    *-------------------------------------------------------------
    
    return .T.

*=================================================================
function waitProgress( nPercent )
*
* WaitProgress ( [<nPercent>] ) --> .t.
*
* <nPercent>   the actual percent value (1 = 100%).
*              When nPercent > 1 the wait bar is closed.
*              When nPercent == 1 the wait bar is displayed
*              and closed.
*
* Shows a wait bar at the display top.
* The wait bar appears at the top as so it will not be in
* conflict with qout()|qqout() functions, as the natural
* display scroll is bottom-up.
*
* This function must be "closed".
*

    static cOldScreen

    local nElements
    local nBar
    local nOldRow    := row()
    local nOldCol    := col()
    local cOldColor  := setcolor( COLOR_BASE )

    *-------------------------------------------------------------
    * Start a new sequence.
    *-------------------------------------------------------------
    
    begin sequence

        *---------------------------------------------------------
        * Save the screen if necessary.
        * Position 00,maxcol() is reserver for the wait wheel.
        *---------------------------------------------------------
        
        if cOldScreen == NIL

            cOldScreen :=;
                mouseScrSave( 00,00,00,maxcol()-1 )

        end

        *---------------------------------------------------------
        * Move the wait wheel.
        *---------------------------------------------------------
        
        waitWheel()

        if  nPercent == NIL             .or.;
            nPercent > 1

            *-----------------------------------------------------
            * Restore the screen.
            * Position 00,maxcol() is reserver for the wait wheel.
            *-----------------------------------------------------
            
            mouseScrRestore( 00,00,00,maxcol()-1, cOldScreen )

            cOldScreen := NIL

            break                                       // BREAK

        end

        *---------------------------------------------------------
        * Calculate the bar length.
        * Position 00,maxcol() is reserver for the wait wheel.
        *---------------------------------------------------------
        
        nElements := maxcol()

        nBar := nElements*nPercent

        if nBar < 0

            nBar := 0

        end

        *---------------------------------------------------------
        * Show the wait bar.
        * Position 00,maxcol() is reserver for the wait wheel.
        *---------------------------------------------------------
        
        setpos( 00,00 )

        dispout( replicate( chr(219), nBar ) )

        dispout( replicate( chr(254), nElements-int(nBar) ) )

        *---------------------------------------------------------
        * If 100% or more restore the screen.
        *---------------------------------------------------------

        if nPercent >= 1

            mouseScrRestore( 00,00,00,maxcol()-1, cOldScreen )

            cOldScreen := NIL

        end

    end

    *-------------------------------------------------------------
    * Restore old data.
    *-------------------------------------------------------------
    
    setpos( nOldRow, nOldCol )
    setcolor( cOldColor )

    *-------------------------------------------------------------
    * This function returns ever True.
    *-------------------------------------------------------------
    
    return .T.

*=================================================================
function waitWheel( lWheel )
*
* waitWheel( [<lWheel>] ) --> lWheelNow
*
* <lWheel>      If True activate the Wait Wheel visualisation,
*               if False, the Wait Wheel is not shown when
*               called.
*
* Shows a wait wheel at 00,maxcol() position.
*
* The screen is not saved.
*

    static nCounter
    static lWork

    local lPrevious

    local nOldRow    := row()
    local nOldCol    := col()
    local cOldColor  := setcolor( COLOR_BASE )

    *-------------------------------------------------------------
    * <lWork> is the static variable that holds the wait wheel
    * status: show it or not.
    *-------------------------------------------------------------
    
    default( @lWork,    .T. ) 
    
    lPrevious := lWork

    *-------------------------------------------------------------
    * Turn ON/OFF waitWheel()
    *-------------------------------------------------------------
    
    if valtype( lWheel ) == "L"

        lWork := lWheel

    end

    *-------------------------------------------------------------
    * If the wait wheel is not to be shown: return.
    *-------------------------------------------------------------
    
    if !lWork

        return NIL                                      // RETURN

    end

    *-------------------------------------------------------------
    * <nCounter> is the static variable used to determinate the
    * actual wheel shape.
    * <nCounter> range from 1 to 8.
    *-------------------------------------------------------------
    
    if nCounter == NIL

        nCounter := 1

    else

        nCounter++

        if nCounter > 8

            nCounter := 1

        end

    end

    *-------------------------------------------------------------
    * Display the wait wheel.
    *-------------------------------------------------------------
    
    setpos( 0,maxcol() )
    do case
    case nCounter == 1
        dispout( "|" )
    case nCounter == 2
        dispout( "/" )
    case nCounter == 3
        dispout( "-" )
    case nCounter == 4
        dispout( "\" )
    case nCounter == 5
        dispout( "|" )
    case nCounter == 6
        dispout( "/" )
    case nCounter == 7
        dispout( "-" )
    case nCounter == 8
        dispout( "\" )
    end

    *-------------------------------------------------------------
    * Restore old data.
    *-------------------------------------------------------------
    
    setpos( nOldRow, nOldCol )
    setcolor( cOldColor )

    *-------------------------------------------------------------
    * Returns waitWheel previous status.
    *-------------------------------------------------------------
    
    return lPrevious

*=================================================================
* END
*=================================================================


*=================================================================
* TIMEX2N()
* TIMEN2H()
* TIMEN2M()
* TIMEN2S()
*=================================================================
function timeX2N( nHH, nMM, nSS )
*
* timeC2N( [<nHH>], [<nMM>], [<nSS>] ) --> nTime
*
* <nHH>     Hours
* <nMM>     Minutes
* <nSS>     Seconds (it may contains decimals)
*
* This function returns a time number where 1 is equivalent to
* 24 Hours (1 day), so, 0.5 is equivalent to 12:00:00 am, and so
* on.
*

    local nTime
    
    default( @nHH,  0 )
    default( @nMM,  0 )
    default( @nSS,  0 )

    nTime := ( ( ( (nSS / 60) + nMM ) / 60 ) + nHH ) / 24
    
    return nTime
    
*=================================================================
function timeN2H( nTime )
*
* timeN2H( <nTime> ) --> nHours
*
* <nTime>   Time number where 1 is equivalent to
*           24 Hours (1 day), so, 0.5 is equivalent to 12:00:00 am,
*           and so on.
*
* This function returns a the hours contained inside <nTime>.
*

    return int( nTime * 24 )
    
*=================================================================
function timeN2M( nTime )
*
* timeN2M( <nTime> ) --> nMinutes
*
* <nTime>   Time number where 1 is equivalent to
*           24 Hours (1 day), so, 0.5 is equivalent to 12:00:00 am,
*           and so on.
*
* This function returns a the minutes contained inside <nTime>.
*

    local nHours
    
    local nRestTime
    
    nHours := int( nTime * 24 )
    
    nRestTime := nTime - (nHours / 24)
    
    return int( nRestTime * 24 * 60 ) 
    
*=================================================================
function timeN2S( nTime )
*
* timeN2S( <nTime> ) --> nSeconds
*
* <nTime>   Time number where 1 is equivalent to
*           24 Hours (1 day), so, 0.5 is equivalent to 12:00:00 am,
*           and so on.
*
* This function returns a the seconds contained inside <nTime>.
*

    local nHours
    
    local nMinutes
    
    local nRestTime
    
    nHours := int( nTime * 24 )
    
    nRestTime := nTime - (nHours / 24)
    
    nMinutes := int( nRestTime * 24 * 60 )
    
    nRestTime := nRestTime - (nMinutes / 60 / 24)
    
    return nRestTime * 24 * 60 * 60
