/******************************************************************************
* Module    :   Split --- Calculate the number of parts needed, and do
*               even split.
*
* Author    :   John W. M. Stevens
******************************************************************************/

#include    <stdio.h>
#include    <stdlib.h>
#include    <string.h>
#include    <time.h>
#include    <errno.h>

#include    "post.h"

extern  char    *sys_errlist[];

/*-----------------------------------------------------------------------------
| Routine   :   GetHdrValues() --- Get the header line values from the user.
-----------------------------------------------------------------------------*/

/*  Define flags for processing the posting header lines.   */
typedef enum    {
    HDR_LINE,
    HDR_SUBJECT,
	HDR_DATE,
    HDR_LAST
} HDR_TYPES;

typedef	struct  {
    HDR_TYPES   HdrLnType;
    char        *Line;
    char        Bfr[BFR_SZ];
} HDR_STRCT;

HDR_STRCT	NewsHdr[] = {
    {   HDR_LINE,       "Newsgroups: ",		""  },
    {   HDR_SUBJECT,    "Subject: ",		""  },
    {   HDR_LINE,       "Summary: ",		""  },
    {   HDR_LINE,       "Expires: ",		""  },
    {   HDR_LINE,       "Sender: ",			""  },
    {   HDR_LINE,       "Followup-To: ",	""  },
    {   HDR_LINE,       "Distribution: ",	""	},
    {   HDR_LINE,       "Keywords: ",		""  },
    {   HDR_LAST,       NULL,				""  }
};

HDR_STRCT	MailHdr[] = {
	{	HDR_LINE,		"From: ",			""	},
	{	HDR_LINE,		"To: ",				""	},
	{	HDR_DATE,		"Date: ",			""	},
	{	HDR_SUBJECT,	"Subject: ",		""	},
    {   HDR_LAST,       NULL,				""  }
};

HDR_STRCT	*Hdr = NewsHdr;

static
void    GetHdrValues(FILE   *HdrFl)
{
    register    int     i;
    register    int     j;

    /*  Get values for the header.  */
    if (HdrFl == stdin)
        printf("Enter values for header lines.\n");

    /*  Read header info from file or user. */
    for (i = 0; Hdr[i].HdrLnType != HDR_LAST; i++)
    {
		/*	Skip getting stuff that is totally system supplied.	*/
		if (Hdr[i].HdrLnType == HDR_DATE)
			continue;

        /*  Get a line from standard input. */
        if (HdrFl == stdin)
        {
            /*  Print a prompt. */
            printf(Hdr[i].Line);

            /*  Get the information from the user.  */
            if (gets(Hdr[i].Bfr) == NULL)
                *Hdr[i].Bfr = '\0';
        }
        else
        {
            /*  Get the information from the information file.  */
            if (fgets(Hdr[i].Bfr, BFR_SZ, HdrFl) == NULL)
                *Hdr[i].Bfr = '\0';
            else
            {
                /*  Remove end of line character.   */
                j = strlen( Hdr[i].Bfr ) - 1;
                Hdr[i].Bfr[j] = '\0';
            }
        }
    }
}

/*-----------------------------------------------------------------------------
| Routine   :   WrtArtHdr() --- Write the article header.
|
| Inputs    :   PartFlPtr   - Pointer to part file.
|               FlName      - Name of file we are posting.
|               PartNo      - Part number.
|               Parts       - Total number of parts.
|               BinData     - Configuration data.
-----------------------------------------------------------------------------*/

/*  Picture action types.   */
static  char    *PictTypeStrs[] =
{
    "",
    " [female]",
    " [male]",
    " [lesbian]",
    " [gay]",
    " [beastiality]",
    " [both]",
    " [bondage]"
};

static
void    WrtArtHdr(FILE      *PartFlPtr,
                  char      *FlName,
                  int       PartNo,
                  long      Parts,
                  BIN_DATA  *BinData)
{
    register    int     i;
    auto        char    SubBfr[BFR_SZ];
	auto		time_t	time_val;
	auto		char	WeekDay[5];
	struct		tm		*TmPtr;
	auto		char	Month[5];
	auto		char	Day[4];
	auto		char	TmStr[10];
	auto		int		Year;
	extern		char	*Version;

    /*  Loop through header string list, and write it (with mods) to
    *   the Part file.
    */
    for (i = 0; Hdr[i].HdrLnType != HDR_LAST; i++)
    {
        /*  Modify the line, if it needs it.    */
        switch ( Hdr[i].HdrLnType )
        {
        case HDR_LINE:
            fprintf(PartFlPtr,
                    "%s%s\n",
                    Hdr[i].Line,
                    Hdr[i].Bfr);
            break;
		case HDR_DATE:
			time_val = time( NULL );
			TmPtr = localtime( &time_val );
			sscanf(ctime( &time_val ),
				   "%s %s %s %s %d\n",
				   WeekDay, Month, Day, TmStr, &Year);
			fprintf(PartFlPtr,
					"%s%s, %s-%s-%02d %s %s\n",
					Hdr[i].Line,
					WeekDay,
					Day,
					Month,
					Year % 100,
					TmStr,
#if	! defined(SYSTEM_MS_DOS)
					TmPtr->tm_zone);
#else
					tzname[TmPtr->tm_isdst]);
#endif
			break;
        case HDR_SUBJECT:
            sprintf(SubBfr,
                    "%s %s(Part %d/%ld)%s %s",
                    FlName,
                    ( BinData->RepostFlag )
                        ?   "REPOST "
                        :   "",
                    PartNo,
                    Parts,
                    PictTypeStrs[ BinData->PictAction ],
                    Hdr[i].Bfr);
            fprintf(PartFlPtr,
                    "%s%s\n",
                    Hdr[i].Line,
                    SubBfr);
            break;
        }
    }

	/*	Write POST signature.	*/
	fprintf(PartFlPtr,
			"\n%s %s %s(Part %d/%ld)%s\n\n",
			Version,
			FlName,
			( BinData->RepostFlag )
				?   "REPOST "
				:   "",
			PartNo,
			Parts,
			PictTypeStrs[ BinData->PictAction ]);
}

/*-----------------------------------------------------------------------------
| Routine   :   Split() --- Determine number of parts, and create a split,
|               uuencoded file.
|
| Inputs    :   FlName      - Name of binary file to prepare to post.
|               HdrFl       - Header information file.
|               DescFl      - Description file.
|               BinData     - Data for construction of proper subject line.
|				CmdBfr		- Command string buffer.
-----------------------------------------------------------------------------*/

void    Split(char      *FlName,
              FILE      *HdrFl,
              FILE      *DescFl,
              BIN_DATA  *BinData,
			  char		*CmdBfr)
{
    register    int         i;
    register    int         j;
    register    int         k;
    auto        char        Exten[5];
    auto        char        OutFlNm[MAX_PATH];
    auto        char        OutFlBase[MAX_PATH];
    auto        char        UUFlName[MAX_PATH];
    auto        char        Bfr[BFR_SZ];
    auto        FILE        *BinFlPtr;
    auto        FILE        *PartFlPtr;
    auto        long        Ofs;
    auto        long        Lines;
    auto        long        Parts;
    auto        BIN_TYPES   BinFlType;
	auto		int			FileNmLen;
	static		char		SysCmd[BFR_SZ * 2];

    extern      int         MsDosFileNms;

    /*  Open file as a binary read-only file.   */
    if ((BinFlPtr = fopen(FlName, "rb")) == NULL)
    {
        fprintf(stderr,
                "%s %d : Error - Could not open file '%s' for reading.\n",
                __FILE__,
                __LINE__,
                FlName);
        exit( 1 );
    }

    /*  Determine binary file type. */
    BinFlType = IdBinFile(BinFlPtr, Exten);

    /*  Position file pointer to end of file.   */
    if (fseek(BinFlPtr, 0L, 2) != 0)
    {
        fprintf(stderr,
                "%s %d : Error - %s\n",
                __FILE__,
                __LINE__,
                sys_errlist[errno]);
        exit( 1 );
    }
    Ofs = ftell( BinFlPtr );

    /*  Position file pointer to begining of file.  */
    if (fseek(BinFlPtr, 0L, 0) != 0)
    {
        fprintf(stderr,
                "%s %d : Error - %s\n",
                __FILE__,
                __LINE__,
                sys_errlist[errno]);
        exit( 1 );
    }

    /*  Calculate the number of parts.  */
    Lines = Ofs / BYTES_PER_LINE;
    if (Ofs % BYTES_PER_LINE)
        Lines++;
    Parts = Lines / LINES_PER_PART;
    if (Lines % LINES_PER_PART)
        Parts++;
	if (Lines % Parts)
		Lines = (Lines / Parts) + 1;
	else
		Lines /= Parts;

    /*  Get header line values. */
    GetHdrValues( HdrFl );

    /*  Get base file name. */
    BaseName(FlName, OutFlBase);

    /*  Check for MS-DOS and USENET compatibility.  */
    if ( MsDosFileNms )
        ModifyFlNm(OutFlBase, BinFlType, Exten, UUFlName);
    else
        strcpy(UUFlName, OutFlBase);

    /*  Strip off extension for part base name. */
    StripExt(FlName, OutFlBase);

    /*  Now do split.   */
    for (i = 0; i <= Parts; i++)
    {
        /*  Create the part file name.  */
        sprintf(OutFlNm, "%s.%d", OutFlBase, i);
        FileNmLen = strlen( OutFlNm );

        /*  Open the file to write part to. */
        if ((PartFlPtr = fopen(OutFlNm, "w")) == NULL)
        {
            fprintf(stderr,
                    "%s %d : Error - Could not open file '%s' for writing.\n",
                    __FILE__,
                    __LINE__,
                    OutFlNm);
            exit( 1 );
        }

        /*  Write article header to file.   */
        WrtArtHdr(PartFlPtr, UUFlName, i, Parts, BinData);

        /*  Get and write the description in part 0.	*/
        if (i == 0)
        {
            /*  Print prompt to get description information.    */
            if (DescFl == stdin)
            {
                printf("\nEnter description.  To exit, enter an "
                       "end of file\ncharacter as the first character "
                       "on a blank line.\n");
            }

            /*  Get description information from the description file
            *   pointer.
            */
            while (fgets(Bfr, BFR_SZ, DescFl) != NULL)
                fprintf(PartFlPtr, "  %s", Bfr);
            fprintf(PartFlPtr, "\n");
        }
		else
		{
			/*	Begin uuencode section.	*/
			if (i == 1)
				fprintf(PartFlPtr, "begin 600 %s\n", UUFlName);

			/*  Encode a part.  */
			Encode(BinFlPtr, PartFlPtr, Lines);

			/*	Write end.	*/
			if (i == Parts)
				fprintf(PartFlPtr, "`\nend\n");
		}

        /*  Close the part file.    */
        fprintf(PartFlPtr, "\n");
        fclose( PartFlPtr );

		/*	If there is a command to execute, put part number string
		*	into appropriate place in the command, then execute.
		*/
		if ( *CmdBfr )
		{
			/*	Start copying characters.  If we see a '\', this
			*	is the escape character.  If we see '#', replace
			*	this with the part file name.
			*/
			for (j = k = 0;
				 k < (2 * BFR_SZ) - 1;
				 j++)
			{
				/*	Check current character type.	*/
				if (CmdBfr[j] == '\\')
					j++;
				else if (CmdBfr[j] == '#')
				{
					/*	Check for buffer overflow.	*/
					if (k + FileNmLen < (2 * BFR_SZ) - 1)
					{
						/*	Copy in part file name string.	*/
						strcpy(SysCmd + k, OutFlNm);
						k += FileNmLen;
					}
					else
					{
						fprintf(stderr,
								"%s %d : Error - Command buffer string ",
								__FILE__,
								__LINE__);
						fprintf(stderr,
								"overflow.\n");
						exit( 1 );
					}
					continue;
				}

				/*	Copy character to system command buffer.	*/
				SysCmd[k++] = CmdBfr[j];
			}

			/*	Execute system command.	*/
			system( SysCmd );
		}
    }

    /*  Close the binary file.  */
    fclose( BinFlPtr );
}

/*-----------------------------------------------------------------------------
| Routine   :   HdrType() --- Set header type.
|
| Inputs    :   SwChar	- Type of header to use, Email or USENET news.
-----------------------------------------------------------------------------*/

void	HdrType(char	SwChar)
{
	switch ( SwChar )
	{
	case 'm':
		Hdr = MailHdr;
		break;
	case 'n':
		Hdr = NewsHdr;
		break;
	}
}
