#define MSDOS     1  /* DELETE if not MSDOS */
#define USE_COLOR 1  /* DELETE if not MSDOS nor OS/2, or not using colors */
                     /* USE_COLOR assumes that either OS2 or MSDOS is used */

/*===========================================================================
 * cal.c - print calendar for one month or one year
 * compatible with unix cal command
 * version 3.3
 *
 * cal [-options] [[month] year]    (numerical month argument)
 * or
 * cal [-options] [month] [year]    (verbal month argument)
 *
 * cal (C) 1992 by Unicorn Research Corporation
 * Inspired by the unix `cal' command for an Amiga, ported by Gary L. Brant.
 * Compile with MSDOS defined (above) to activate the MSDOS features.
 *
 * Borland users: PLEASE try not to Borland-ize this code.  Use
 * #ifdef __TURBOC__ on any Borland-specific sections you create.
 *===========================================================================

29 March 1993, Darrel Hankerson hankedr@mail.auburn.edu
 Modifications and enhancements for OS/2 (and others).
 New command line options:
  [--nodata] [--data-file=data-file]
 and color options (if compiling with -DUSE_COLOR for OS/2 or MSDOS)
  [--nocolor] [--color-file=color-file]

05 June 1993, Don Retzlaff  University of North Texas  donr@gab.unt.edu
 Modifications and enhancements (re-done by Alex Matulich for
 ANSI compatibility):
 Variable-number of events for month display (previously only allowed 8)
 New command line option:
   [--maxappts=num]
 Allow multiple --datafile= command line parameters


 BASIC DATA STRUCTURE:

 char *buf  ====>   line[0] -> [......LineWid (default 80).......]
                    line[1] -> [.................................]
                    line[2] -> [.................................]
                          ...  [.................................]
           line[numappts-3] -> [.................................]
            str (work area) -> [.................................]
                spcldesc[0] -> [....DAY_DESC (56).....]
                spcldesc[1] -> [....DAY_DESC (56).....]

               line_attr[0] -> [........LineWid (80).............]
               line_attr[1] -> [.................................] allocated
               line_attr[2] -> [.................................] only if
                          ...  [.................................] colors
      line_attr[numappts-3] -> [.................................] are used
                   mon_attr -> [.................................]
                    wk_attr -> [.................................]
                   day_attr -> [......YEARWID (72)...........]
               spclday_attr -> [....DAY_DESC (56).....]
*/

#define MAXFILES 8   /* maximum number of data file paths */
#define MAXAPPTS 50  /* maximum number of appointment description lines */

#ifdef USE_COLOR
#ifdef OS2
#define INCL_SUB
#include <os2.h>
#else
#include <dos.h> /* for int86() */
#endif /* OS2 */
#endif /* USE_COLOR */

#include <stdio.h>
#include <io.h>   /* for isatty() */
#include <stdlib.h>
#include <time.h>
#include <ctype.h>
#include <string.h>
#include <sys/types.h>

#ifndef max
#define max(a,b) (((a)>(b))?(a):(b))
#endif /* max */

#ifdef __ZTC__                    /* make Zortech compatible with stricmp() */
#define stricmp(a,b) strcmpl(a,b) /* case-insensitive string comparison */
#endif /* __ZTC__ */

#ifdef MSDOS
#ifndef MK_FP
#define MK_FP(seg,off) (void far *)(((unsigned long)(seg)<<16)|(unsigned)(off))
#endif /* MK_FP */
union REGS reg;
char far *video, far *tmpvideo;
#endif /* MSDOS */

#define FUDGE1  1          /* needed to make day of week come out right */
#define FUDGE2  6          /* for old style (Julian) calendar */
#define MONWID  24         /* width of month */
#define YEARWID (3*MONWID-2) /* width of yearly calendar */
#define DAY_DESC  (LineWid-MONWID)  /* width of special day descriptions */

#ifdef USE_COLOR           /* define display attribute possibilities */
#define FG_BLACK     0x00
#define FG_BLUE      0x01
#define FG_GREEN     0x02
#define FG_CYAN      0x03
#define FG_RED       0x04
#define FG_VIOLET    0x05
#define FG_ORANGE    0x06
#define FG_LTGRAY    0x07
#define FG_DKGRAY    0x08
#define FG_BRTBLUE   0x09
#define FG_BRTGREEN  0x0a
#define FG_BRTCYAN   0x0b
#define FG_BRTRED    0x0c
#define FG_BRTVIOLET 0x0d
#define FG_YELLOW    0x0e
#define FG_WHITE     0x0f
#define FG_FLASH     0x80
#define BG_BLACK     0x00
#define BG_BLUE      0x10
#define BG_GREEN     0x20
#define BG_CYAN      0x30
#define BG_RED       0x40
#define BG_VIOLET    0x50
#define BG_ORANGE    0x60
#define BG_WHITE     0x70

unsigned
char dfmon_attr   = FG_BLACK|BG_GREEN,   /* video attribute for month name */
     dfwk_attr    = FG_BLUE|BG_CYAN,     /* attribute for weekday header */
     dfdy_attr    = FG_BLACK|BG_WHITE,   /* attribute for days */
     dfsun_attr   = FG_VIOLET|BG_WHITE,  /* attribute for sundays */
     dftoday_attr = FG_YELLOW|BG_BLUE,   /* attribute for current day */
     dfbk_attr    = FG_LTGRAY|BG_BLACK,  /* year calendar background */
     dfspday_attr = FG_BRTCYAN|BG_BLACK, /* special day description attr. */
     dfspdfl_attr = FG_BRTRED|FG_FLASH,  /* special day flasher */
     dftoday_attr2;                      /* this must be initialized later */

char *mon_attr, *wk_attr, *day_attr, *spclday_attr, *line_attr[MAXAPPTS-2];

int crt;
char *color_file = "cal.col";
#endif /* USE_COLOR */

/* LineWid may change depending on terminal (MSDOS only).             */
/* This variable determines memory allocation size and display width. */
short LineWid = 80;                /* set to 80 for non-MSDOS systems */

char *data_file = "cal.dat";

short days[] = { 31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31 },
    mdays[] = { 0, 31, 59, 90, 120, 151, 181, 212, 243, 273, 304, 334, 365 },
    multi = 0, thisday, thismonth, thisyear, discard_old = 0,
    europe = 0, numfiles = 0,
    numappts = 24;  /* default maximum number of appointments to display */

char *months[] = {
       "January ", "February ", "March ", "April ", "May ", "June ", "July ",
       "August " , "September ", "October ", "November ", "December " },
     monthline[] = "         ---                     ---                     ---",
     *monthpos[] = { monthline+9, monthline+33, monthline+57 },
     USdays[] =     " Su Mo Tu We Th Fr Sa",
     Europedays[] = " Mo Tu We Th Fr Sa Su",
     *dayline = USdays,
     *spcldesc[2],
     *filename[MAXFILES],
     *homedir,        /* home directory from argv[0], may not work in unix */
     *line[MAXAPPTS-2], /* line buffer for body of calendar */
     *buf = NULL,       /* workspace memory into which pointers are assigned */
     *str;            /* temporary string workspace (LineWid characters) */

void printmonth (short, short);
short putd(register short);
void fixtab(register short);
short weekday(register short, register short);
void putattr(char *, register short);
short specialdays(short, short);
FILE *efopen(const char *, const char *);
void usage(void);

/* commandline argument list */
static struct arglist {
   char *opt;         /* full argument string */
   short needsvalue;  /* 1=argument needs a value, 0=stand-alone */
   } opt[] = {
      "nodata",     0,
      "data-file",  1,
      "future",     0,
      "europe",     0,
      "maxappts",   1,
#ifdef USE_COLOR
      "nocolor",    0,
      "color-file", 1,
#endif
      NULL,         0
   };


/*===========================================================================
 * GETARG
 * Return the index of the argument in the option array.  val is set if the
 * argument needs a value.  The passed arg to test does not have to be
 * exactly like one of the arguments in o; it just has to match one of them
 * enough to be uniquely identified.  Any arg that is passed here should
 * begin with a '-' (or also a '/' if compiling for MSDOS or OS/2).
 *=========================================================================*/
int getarg(struct arglist *o, char *arg, char **val)
{
int len = -1, i = -1, found = -1;
*val = NULL;

while (*arg == '-'
       #if (defined(MSDOS) || defined(OS2))
              || *arg == '/'
       #endif
                            ) ++arg;
while (arg[++len]) {
   arg[len] = tolower(arg[len]);
   if (arg[len] == '=') {
      arg[len] = '\0';
      *val = &arg[len+1];
      break;
      }
   }
while (o[++i].opt) {
   if (!memcmp(o[i].opt, arg, len)) {
      if (found >= 0) return -1;
      found = i;
      }
   }
if (found < 0) return -1;
if (o[found].needsvalue && !(*val)) return -1;
return found;
}


/*===========================================================================
 * MAIN
 * Process command line, allocate and initialize, etc.
 *=========================================================================*/
void main(int argc, char *argv[])
{
short bufsiz, i, m, y;
time_t t;
struct tm *lt;
int c, option_index = 0;
char *optarg;

#ifdef USE_COLOR
FILE *colorfile;
short fg[8], bg[8];
crt = isatty(fileno(stdout));  /* check if stdout is being redirected */
#endif /* USE_COLOR */

filename[0] = data_file;
for (i = 1; i < argc; ++i) {
   if (*argv[i] != '-'
      #if (defined(MSDOS) || defined(OS2))
                     && *argv[i] != '/'
      #endif
                                       ) break;
   if ((c = getarg(opt, argv[i], &optarg)) < 0) { usage(); goto freestuff; }

   switch (c) {
      case 0:
         data_file = NULL; break;
      case 1:
         if (numfiles < MAXFILES) filename[numfiles++] = optarg;
         break;
      case 2:
         discard_old = 1; break;
      case 3:
         dayline = Europedays;  europe = 1;  break;
      case 4:
         numappts = atoi(optarg); break;
#ifdef USE_COLOR
      case 5:
         crt = 0; break;
      case 6:
         color_file = optarg; break;
#endif /* USE_COLOR */
      }
   }
if (numappts < 8 || numappts > MAXAPPTS) { usage(); goto freestuff; }
if (!numfiles) ++numfiles;  /* keep the default data file in any case */

/* make the argument list compatible with unix cal */
argc = argc - i + 1;
for (m = 1; m < argc; ++m) argv[m] = argv[i++];

#ifdef MSDOS
reg.h.ah = 0x0f;            /* function 0x0F of INT 0x10 - get display mode */
int86(0x10, &reg, &reg);
LineWid = max(80, reg.h.ah);  /* number of columns returned in AH */
  /* The above should work for all displays.
   * The following can be used instead, but will work in EGA and VGA only: */
  /* Linewid = *((unsigned far *)0x004004a); */
#endif /* MSDOS */

/* allocate workspace buffer */

bufsiz = (numappts+1)*LineWid + 2*DAY_DESC;
#ifdef USE_COLOR
/* adjust workspace for color enhancements */
bufsiz += (numappts+3)*LineWid + YEARWID + DAY_DESC;
#endif /* USE_COLOR */
if ((buf = (char *)calloc(bufsiz,1)) == NULL) {
   puts("Memory overflow");
   goto freestuff;
   }

/* assign pointers into buf */

line[0] = buf;
for (i = 1; i < numappts-2; i++)
   line[i] = line[i-1] + LineWid;                /* calendar body */
str = line[i-1] + LineWid;                       /* string workspace */
*(spcldesc[0] = str + LineWid) = '\0';           /* description 0 */
*(spcldesc[1] = spcldesc[0] + DAY_DESC) = '\0';  /* description 1 */
                  /* descriptions 2,3,... point into line[] */

/* find originating directory */

homedir = argv[0];
i = strlen(homedir);
while (i >= 0 && homedir[i] != '/' && homedir[i] != '\\' && homedir[i] != ':')
   homedir[i--] = '\0';

#ifdef USE_COLOR                     /* color attributes */
line_attr[0] = spcldesc[1] + DAY_DESC;
for (i = 1; i < numappts-2; i++) line_attr[i] = line_attr[i-1] + LineWid;
mon_attr = line_attr[i-1] + LineWid;
wk_attr = mon_attr + LineWid;
day_attr = wk_attr + LineWid;
spclday_attr = day_attr + YEARWID;

/* attempt to read in file of colors */
if (!crt || (colorfile = efopen(color_file, "r")) == NULL) goto contsetup;
for (i = 0; i < 8; i++) {
   if (fgets(str, LineWid, colorfile) == NULL) break;
   if ((fg[i] = atoi(str)) == (bg[i] = atoi(&str[3]))) break;
   if (fg[i] > 15 || bg[i] > 15) break;
   }
fclose(colorfile);
if (i < 8) goto contsetup;
dfmon_attr   = fg[0] | (bg[0] << 4);
dfwk_attr    = fg[1] | (bg[1] << 4);
dfdy_attr    = fg[2] | (bg[2] << 4);
dfsun_attr   = fg[3] | (bg[3] << 4);
dftoday_attr = fg[4] | (bg[4] << 4);
dfbk_attr    = fg[5] | (bg[5] << 4);
dfspday_attr = fg[6] | (bg[6] << 4);
dfspdfl_attr = fg[7] | (bg[7] << 4);

/* initialize color attribute strings */
contsetup:
memset (mon_attr, dfmon_attr, LineWid);
memset (wk_attr, dfwk_attr, LineWid);
memset (day_attr, dfdy_attr, YEARWID);
memset (spclday_attr, dfspday_attr, DAY_DESC);
spclday_attr[0] = dfspdfl_attr;
for (i = MONWID-2; i <= MONWID+MONWID; i += MONWID) {
   memset (&mon_attr[i], dfbk_attr, 2);
   memset (&wk_attr[i], dfbk_attr, 2);
   memset (&day_attr[i], dfbk_attr, 2);
   memset (&day_attr[i - MONWID + 3 + (18*europe)], dfsun_attr, 2);
   }
memset (&day_attr[i - MONWID + 3 + (18*europe)], dfsun_attr, 2);
dftoday_attr2 = ((dfdy_attr & '\x70')|(dftoday_attr >> 4));

#ifdef MSDOS
int86(0x11, &reg, &reg);  /* read the equipment list into reg. */
video = (char far *)MK_FP(((reg.x.ax & 48) == 48) ? 0xb000 : 0xb800, 0);
#endif /* MSDOS */

#endif /* USE_COLOR */

/* now that all the setup is done, we can begin with the program proper */

t = time(NULL);
lt = localtime(&t);
thisyear = y = lt->tm_year + 1900;
thismonth = m = lt->tm_mon + 1;
thisday = lt->tm_mday;
puts("");           /* first display a blank line */

switch (argc) {
   case 1:             /* display current month if no arguments */
      fixtab (y);
      printmonth (m, y);
      break;

   case 2:
      if (isdigit(argv[1][0]) || (isdigit(argv[1][1]) && argv[1][0] == '-')) {
         multi = 1;
         fixtab (y = atoi(argv[1]));
         fputs("\t\t\t\t ", stdout);
         putd(y);         /* put year */
         puts("\n");
         for (m = 1; m < 13; m++) printmonth (m, y);
         break;
         }
      /* drop through to next case */

   case 3:
      m = atoi(argv[1]);
      if (strlen(argv[1]) >= 3) {
         for (i = 0; i < 12; i++) {
            strcpy(str, months[i]);
            str[3] = argv[1][3] = '\0';
            if (!stricmp(argv[1], str)) {
               m = i + 1;
               break;
               }
            }
         }
      if (!m) { usage(); break; }
      if (argc == 3) {
         if (isdigit(argv[2][0]) || (isdigit(argv[1][1]) && argv[2][0] == '-'))
            y = atoi(argv[2]);
         else { usage(); break; }
         }
      fixtab(y);
      printmonth(m, y);
      break;

   default: usage();
   }

freestuff:
free(buf);
}


/*===========================================================================
 * printmonth()
 * either prints an entire month at a time or multiplexes
 * a month into a buffer, dumping the buffer after the third call.
 *=========================================================================*/
void printmonth(short m, short y)
{
register short first, last;
register short index, p = 0;
register char *ll;
static short q = 0;
short l, hilite = ((m == thismonth) && (y == thisyear)), num_appts = 0;

--m;
if (multi) {
   q++;
   if (q > 3) q = 1;
   p = MONWID * (q - 1);        /* character position of line in buffer */
   (void) memcpy(monthpos[q-1], months[m], 3);       /* copy month name */
   if (q == 3) {
      puts(monthline);
      #ifdef USE_COLOR
      putattr(mon_attr, YEARWID);
      #endif /* USE_COLOR */
      for (index = 0; index < 2; ++index) {
         fputs(dayline, stdout);
         fputs("   ", stdout);
         }
      puts(dayline);
      #ifdef USE_COLOR
      putattr(wk_attr, YEARWID);
      for (l = 0; l < 6; l++) memcpy(line_attr[l], day_attr, LineWid);
      #endif /* USE_COLOR */
      }
   }
else {
   q = 1;
   #ifdef USE_COLOR   /* initialize attribute buffers for month display */
   for (l = 0; l < numappts-2; l++) {
      if (l<6)
         memcpy(line_attr[l], day_attr, MONWID-2);
      else
         memset(line_attr[l], 0, MONWID-2);
      line_attr[l][MONWID-2] = dfspdfl_attr;
      memcpy(&line_attr[l][MONWID-1], spclday_attr, DAY_DESC);
      }
   memcpy(&mon_attr[MONWID-1], spclday_attr, DAY_DESC);
   memcpy(&wk_attr[MONWID-1], spclday_attr, DAY_DESC);
   mon_attr[MONWID-2] = wk_attr[MONWID-2] = dfspdfl_attr;
   #endif /* USE_COLOR */
   }

if (!p) memset(line[0], ' ', (numappts-2)*LineWid);

if (y == 1752 && m == 8) {      /* special case Sep. 1752 */
   line[0][p + 8] = '1';
   line[0][p + 11] = '2';
   first = 14;
   last = 16;
   index = 12;
   }
else {
   short dow = weekday(m, y);   /* day of week for first day of month */
   first = 1;
   last = 7 - dow;
   index = 3 * dow;
   }

for (l = 0; l < 6; ++l) {       /* loop thru month one week per line */
   ll = line[l] + p + 1;
   while (first <= last) {      /* for each day in week encode day of month */
      if (first >= 10) ll[index] = '0' + first / 10;
      ll[index+1] = '0' + first % 10;
      if (!multi && hilite && first == thisday) {  /* hilight today */
         #ifdef USE_COLOR
         line_attr[l][index+1] = line_attr[l][index+2] = dftoday_attr;
         line_attr[l][index] = line_attr[l][index+3] = dftoday_attr2;
         ll[index-1] = crt ? '\xde' : '>';
         ll[index+2] = crt ? '\xdd' : '<';
         #else
         ll[index-1] = '>';
         ll[index+2] = '<';
         #endif /* USE_COLOR */
         }
      index += 3;
      ++first;
      }
   last = (last + 7) > days[m] ? days[m] : last + 7;
   index = 0;
   }

if (!multi) {
   short i, yw;
   /* the next 2 lines assume that the shortest-length month name
    * in months[], such as "June ", will be 5 characters long. */
   char mtmp[] = "             ";     /* 13 spaces */
   mtmp[l = (18 - (i = strlen(months[m]))) >> 1] = '\0';
   fputs(mtmp, stdout);
   fputs(months[m], stdout);          /* put month name */
   yw = putd(y);                      /* put year */
   num_appts = specialdays(m+1, y);
   if (num_appts) {
      mtmp[l] = ' ';
      mtmp[MONWID - l - i - yw - 1] = '\0';
      fputs(mtmp, stdout);
      puts(spcldesc[0]);              /* put first description */
      }
   else puts("");
   #ifdef USE_COLOR
   putattr(mon_attr, LineWid-1);
   #endif /* USE_COLOR */
   fputs(dayline, stdout);
   if (num_appts > 1) {
      fputs("  ", stdout);
      puts(spcldesc[1]);              /* put next description */
      }
   else puts("");
   #ifdef USE_COLOR
   putattr(wk_attr, LineWid-1);
   #endif /* USE_COLOR */
   }
num_appts = max(6, num_appts-2);
for (l = 0; l < num_appts; l++)
   if (!multi || q == 3) {
      index = LineWid-1;
      ll = line[l];
      while (index >= 0 && ll[index] == ' ') --index;
      ll[index+1] = '\0';
      puts(ll);
      #ifdef USE_COLOR
      putattr(line_attr[l], multi ? YEARWID : LineWid-1);
      #endif /* USE_COLOR */
      }
}


/*===========================================================================
 * putd() - put year to standard output.
 *=========================================================================*/
short putd(register short n)
{
register char *p = str+10;
short yw = 0, neg = (n < 0);
*p = '\0';
if (neg) n = -n;
do {
   --p;
   ++yw;
   *p = '0' + n % 10;
   n = n / 10;
   } while (n);
if (neg) *(--p) = '-';
fputs(p, stdout);
return yw;
}


/*===========================================================================
 * fixtab() - correct for leapyears.
 *=========================================================================*/
void fixtab(register short y)
{
register short i;
if (!(y % 4)) {
   if ((y % 100) || !(y % 400) || (y < 1753)) {
      days[1] = 29;
      for (i = 2; i < 13; i++) mdays[i]++;
      }
   }
}


/*===========================================================================
 * weekday() - return day-of-week for first day of month.
 *=========================================================================*/
short weekday(register short m, register short y)
{
--y;
if (y > 1752-1 || (y == 1752-1 && m > 8))
   return (mdays[m] + y + y / 4 - y / 100 + y / 400 + FUDGE1 - europe) % 7;
else
   return (mdays[m] + y + y / 4                     + FUDGE2 - europe) % 7;
}


#ifdef USE_COLOR
/*===========================================================================
 * putattr() - MSDOS or OS/2 only
 * modify the video memory text attribute areas with the attributes passed
 * in attr of lengh len, starting at the current cursor location.
 *=========================================================================*/
void putattr(char *attr, register short len)  /* write video attr. string */
{
#ifdef OS2
USHORT row, col;
int i;
#endif /* OS2 */

if (!crt) return;

#ifdef OS2
VioGetCurPos(&row, &col,0);
--row;
while (len) {        /* minimize the number of Vio calls */
   for (i = 1; i < len; i++)
   if (attr[i] != *attr) break;
   VioWrtNAttr((PBYTE) attr, i, row, col, 0);
   attr += i; col += i; len -= i;
   }

#else /* if not OS2 */

reg.h.ah = 3;
reg.h.bh = 0;
int86(0x10, &reg, &reg);
tmpvideo = video + ((LineWid << 1) * (reg.h.dh - 1) + reg.h.dl + 1);
while (len--) {
   *tmpvideo = *(attr++);
   tmpvideo += 2;
   }
#endif /* OS2 */
}
#endif /* USE_COLOR */


/*===========================================================================
 * specialdays()
 * find the file CAL.DAT and attempt to read in special date descriptions.
 *=========================================================================*/
short specialdays(short m, short y)
{
FILE *fp;
short mo, dy, yr, w, n, filenum, wn, weeklyflag, k, j, i = 1, *item = NULL,
    hilight = (m == thismonth && y == thisyear);
char **lln = NULL, *ll;

if (!data_file) return 0;
if ((item = (short *)calloc(numappts+1, sizeof(short)))==NULL) return 0;
if ((lln = (char **)calloc(numappts+1, sizeof(char *)))==NULL) goto spcldone;
memset(item, 0, numappts*sizeof(short));

if (m != thismonth) discard_old = 0;
lln[0] = spcldesc[0];
lln[1] = spcldesc[1];
for (j = 2; j < numappts; j++) lln[j] = &line[j-2][MONWID-1];

*lln[0] = ' ';
lln[0][1] = '\0';
item[0] = 100;  /* day code for new month delimeter in descriptions */

for (filenum = 0; filenum < numfiles; filenum++) {
   if ((fp = efopen(filename[filenum], "r")) == NULL) continue;

   while (fgets(str, LineWid, fp) != NULL) {  /* go until EOF is hit */
      if (!isdigit(*str) && *str != '-') continue;       /* a data line? */
      if ((yr = atoi(str)) == 0) continue;               /* get year */
      if ((mo = atoi(&str[5])) == 0) continue;           /* get month */
      if ((dy = atoi(&str[8])) == 0)                     /* get day */
          { if ((wn = atoi(&str[11])) == 0) continue; }  /* get nth wkday */
      else wn = 0;                   /* don't need wn if dy is valid */
      if (!(((yr == -999 || yr == y) && (mo == m || mo == m+1)) /* got it all? */
          || ((yr == -999 || yr == y+1) && (mo == 1 && m == 12)) || mo == -9))
         continue;
      if (dy == -9)
         if (m == thismonth && y == thisyear)
            dy = thisday;            /* set daily reminder */
         else
            continue;
      weeklyflag = 5*(wn<0 && wn!=-9); /* flag to dup. weekly dates */
      if (wn < 0) wn = 50-wn;
      weeklyduplicate:
      w = weeklyflag ? weeklyflag : wn/10;
      if (((wn >= 11 && wn <= 57) || w == 9) && (mo == m || mo == -9)) {
         /* find nth weekday */
         n = wn%10 - 1;              /* day of week (0=sun, 1=mon, etc.) */
         if (n < 0 || n > 6) continue;
         if (w == 9)                 /* special case to find last weekday */
            for (j = 5; j > 0; j--)
               { if (line[j][3*n+2] != ' ') break; }
         else {                      /* find nth weekday */
            for (j = 0; j < 6; j++)
               if (line[j][3*n+2] != ' ') if (!(--w)) break;
            if (w)
               if (--weeklyflag > 0) goto weeklyduplicate;
               else continue;        /* nth weekday didn't exist */
            }
         dy = atoi(&line[j][3*n+1]); /* date of nth weekday */
         }
      if (!dy) continue;             /* next loop if wn>57 or wn<11 or dy=0 */
      if ((mo == m+1 && (yr == y || yr == -999))
          || (mo == 1 && m == 12 && (yr == y+1 || yr == -999)))
         dy += 100;                  /* make sure 'next month' days are last */
      if (discard_old && dy < thisday) continue;  /* next loop if old day */
      for (j = 0; j < i; j++) {      /* insert in sorted order */
         if (dy < item[j]) {
            k = i;
            if (k == numappts) --k;
            for (; k > j; k--) {
               item[k] = item[k-1];
               strcpy(lln[k], lln[k-1]);
               }
            break;
            }
         }
      ll = lln[j];                   /* calendar line to modify */
      item[j] = dy;                  /* keep track of the dates so far */
      ll[0] = (hilight && dy == thisday && (mo == m || mo == -9)) ? '*' : ' ';  /* 'today' flag */
      ll[1] = (dy%100 < 10) ? ' ' : '0' + (dy%100)/10;
      ll[2] = '0' + dy%10;
      ll[3] = ':';
      j = strlen(str) - 1;           /* get rid of trailing white space */
      while (j >= 13 && isspace(str[j])) str[j--] = '\0';

/**************************************************************************/
/*             Check for special case of Aniversary/Birthdays             */
/*   Change added by R. Scott, and generalized a bit by Alex Matulich     */
/* A field indicated by [] will be displayed with [], but a field using {}*/
/* will be displayed without the braces.                                  */
/**************************************************************************/
/* These changes replace the next two lines: */
/*    str[j = 14+DAY_DESC-6] = '\0';         */
/*    strcpy(&ll[4], &str[13]);              */
/*********************************************/

      j=13; k=4;  /* init pointers into strings */
      while (k < DAY_DESC - 1 && str[j]) {
         ll[k++] = str[j];
         if (str[j] == '[' || str[j] == '{') {   /* start of [field] */
            short suffix = -1;   /* <0 indicates no ordinal suffix */
            long ipower = 1,
                 by = atoi(&str[++j]); /* year of aniversary */
            if (by <= y) {             /* will year be modified? */
               by = y - by + (dy>100 && mo==1); /* years since anniversary */
               suffix = by;            /* ordinal suffix will be attached */
               if (str[j-1] == '{') --k;  /* don't display curly brace */
               }
            while (by / ipower) ipower *= 10L;
            ipower /= 10L;      /* ipower is now the magnitude of by */
            if (!by) ll[k++] = '0';
            else while (ipower && k < DAY_DESC - 1) {
               ll[k++] = by / ipower + '0';
               by -= (by / ipower) * ipower;  /* remove largest digit */
               ipower /= 10L;
               }
           /* optionally add the ordinal suffixes st, nd, rd and th
            * to the end of value for better readability (for example,
            * 31 will be 31st, 23 will be 23rd, etc.) */
            if (suffix >= 0 && k < DAY_DESC - 3) {
               suffix %= 100;  /* only use last two digits */
               if (suffix >= 11 && suffix <= 13) suffix = 4;
               else suffix %= 10; /* last digit */
               switch (suffix) {
                  case 1:  ll[k++]='s'; ll[k++]='t';  break;
                  case 2:  ll[k++]='n'; ll[k++]='d';  break;
                  case 3:  ll[k++]='r'; ll[k++]='d';  break;
                  default: ll[k++]='t'; ll[k++]='h';
                  }
               }
            while (isdigit(str[j])) ++j;      /* advance to end of field */
            if (suffix >= 0 && str[j] == '}') ++j; /* skip curly brace */
            }
         else
            ++j;      /* advance to next char */
         }
      ll[k] = '\0';   /* ensure string is terminated */
/**************************************************************************/
/* End of anniversary changes                                             */
/**************************************************************************/

      if (++i > numappts) i = numappts;
      if (--weeklyflag > 0) goto weeklyduplicate; /* dup. weekly reminders */
      }
   for (j = 0; j < i; j++) if (item[j] == 100) break;
   if (j == i-1)
      lln[--i][0] = '\0';
   else if (j < i) {
      strcat(lln[j], months[m%12]);
      strcat(lln[j], "--");
      }
   fclose(fp);
   }

spcldone:
free(lln);
free(item);
return i;
}


/*===========================================================================
 * efopen() - fopen() replacement that does path searches
 *=========================================================================*/
FILE *efopen(const char *file, const char *mode)
{
FILE  *fp;

#ifdef OS2
char  path[_MAX_PATH];
#endif /* OS2 */

if ((fp = fopen(file, mode)) == NULL) {
   if (*homedir) {
      strcpy(str, homedir);
      strcat(str, file);
      fp = fopen(str, mode);
      }

#ifdef OS2
   if (fp == NULL) {
      _searchenv(file, "PATH", path);
      if (*path == '\0')
         _searchenv(file, "DPATH", path);
      if (*path)
         fp = fopen(path, mode);
      }
#endif /* OS2 */
   }
return fp;
}


/*===========================================================================
 * usage() - called from main() - display commandline usage
 *=========================================================================*/
void usage()
{
fputs("\n\
CAL 3.3 -- unix-like 'cal' by Unicorn Research Corporation\n\
Display a monthly or yearly calendar, with optional appointments\n\n\
Usages:\tcal [options] [[month (1-12)] year]\n\
\tcal [options] [month (jan-dec)] [year]\n\n\
Options:\n\
  -nod[ata]          Ignore appointment descriptions file\n\
  -d[ata-file]=file  Load appointments from `file' (CAL.DAT)\n\
  -f[uture]          Show only future appointments on current month\n\
  -e[urope]          European format (1st day is Monday)\n\
  -m[axappts]=n      Display maximum of n (8-50) appointments (24)\n", stderr);

#ifdef USE_COLOR
fputs("\
  -noc[olor]         Inhibit use of colors\n\
  -c[olor-file]=file Load color definitions from `file' (CAL.COL)\n", stderr);
#endif /* USE_COLOR */
}
