/*
 *  Cheshiresoft Calendar-Almanac
 *  Copyright (c) 2003 by Andrew Ziem.  All rights reserved.
 *  http://cday.sourceforge.net 
 *  http://chesire.freeservers.com
 *
 *     CDAY.C -- command-line code, configures and displays
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */



#include <fcntl.h>
#include <sys/ioctl.h>

#define INC_DATABASE
#define INC_CALENDAR

#include "cday.h"

#include "lls.h"
#include "ctype.h"
#include "scaldate.h"
#include "datetime.h"

#if defined(INC_DATABASE)
#include "library.h"
#endif /* INC_CALENDAR */
#if defined(INC_CALENDAR)
#include "cal.h"
#endif /* INC_CALENDAR  */


OPTS opts;

MOPTS mopts;

GREG greg;



int
get_terminal_width (void)
{
  int ttyfd;
#ifdef TIOCGWINSZ
  struct winsize ws;
#endif


  /* to do: implement command line override (for pipe to file) */

  if (!isatty (0))
    {
      /* no tty */
      return 80;
    }

#ifdef TIOCGWINSZ
  /* if stdout tty, use TIOCGWINSZ ioctl */

  ttyfd = open ((char *) ttyname (0), O_RDWR);

  if (-1 != ttyfd)
    {
      ioctl (ttyfd, TIOCGWINSZ, &ws);
      close (ttyfd);
      return (ws.ws_col);
    }
#endif

  /* check COLUMNS env var */

  if (NULL != getenv ("COLUMNS"))
    {
      return (atoi (getenv ("COLUMNS")));
    }


  /* if TERM set, retrive info from termcap or terminfo */

  /* to do: implement termcap / terminfo */

  /* assume 80 */

  return 80;
}


void
show_calendar (char *pszLongLabel, char *pszShortLabel, char *pszDescription)
{

  if (NULL == pszShortLabel)
    {
      pszShortLabel = pszLongLabel;
    }
  if (DISPLAY_HTML == mopts.display_mode)
    {
      printf ("<TR>\n<TD>%s</TD>\n", pszLongLabel);
      printf ("<TD>%s</TD>\n</TR>", pszDescription);
    }
  else
    {
      printf ("%-15s", pszShortLabel);
      printf ("%s\n", pszDescription);
    }
}


/* show_calendars()
 * Displays each calendar that is enabled.
 */
void
show_calendars (void)
{
  if ((opts.showfrench || opts.showgregorian || opts.showmaya
       || opts.showjewish || opts.showjulian || opts.showjdn
       || opts.showshire || opts.showgue || opts.showmoon)
      && DISPLAY_HTML == mopts.display_mode)

    {
      printf ("<TABLE>\n");
    }



  if (opts.showfrench)

    {
      int fyear, fmonth, fday;

      char szDescription[30];

      SdnToFrench (greg.jdn, &fyear, &fmonth, &fday);

      if (fyear != 0)

	{
	  sprintf (szDescription, "%s, %s %i, %i",
		   FrenchWeekDays[french_dow (fday)],
		   FrenchMonthName[fmonth], fday, fyear);
	  show_calendar ("French Republican", "French Rep.", szDescription);
	}

      /*
         1. Jour de la vertu (Virtue Day)
         2. Jour du genie (Genius Day)
         3. Jour du travail (Labour Day)
         4. Jour de l'opinion (Reason Day)
         5. Jour des recompenses (Rewards Day)
         6. Jour de la revolution (Revolution Day)  (the leap day)
       */
    }				/* French Republican */
  if (opts.showgue)

    {

      char szDescription[30];

      sprintf (szDescription, "%s, %s %i, %i", gue_days[greg.wday],
	       gue_months[greg.month - 1], greg.mday, abs (greg.year - 1027));
      show_calendar ("Great Undergrond Empire", "Great Under.",
		     szDescription);

    }				/* Great Underground Empire */
  if (opts.showgregorian)

    {
      char szDescription[30];

      sprintf (szDescription, "%s, %s %i, %s%i%s",
	       gregorian_days[greg.wday], gregorian_months[greg.month - 1],
	       greg.mday, greg.year > 0 ? "AD " : "", abs (greg.year),
	       greg.year < 0 ? " BC" : "");
      show_calendar ("Gregorian", NULL, szDescription);
    }				/* Gregorian */
  if (opts.showjewish)

    {
      int jday, jmonth, jyear;
      char szDescription[30];
      SdnToJewish (greg.jdn, &jyear, &jmonth, &jday);

      sprintf (szDescription, "%s %i, AM %i", JewishMonthName[jmonth],
	       jday, jyear);
      show_calendar ("Hebrew", NULL, szDescription);
    }

  /* The Julian calendar is not related to JDNs. */
  if (opts.showjulian)
    {
      int jday, jmonth, jyear;
      char szDescription[30];

      SdnToJulian (greg.jdn, &jyear, &jmonth, &jday);
      sprintf (szDescription, "%s %i, %i",
	       gregorian_months[jmonth - 1], jday, jyear);

      show_calendar ("Julian", NULL, szDescription);

    }				/* Julian */

  /* JDNs are not related to the Julian calendar. */
  if (opts.showjdn)

    {
      char szJDN[10];

      commafmt (szJDN, 10, greg.jdn);
      show_calendar ("JDN", NULL, szJDN);

    }				/* JDN */



  if (opts.showmaya)

    {
      char lc[30];
      int hm, hd;
      char szDescription[30];

      SdnToHaab (greg.jdn, &hm, &hd);
      SdnToMayaLongcountStr (greg.jdn, lc);

      sprintf (szDescription, "%s, %i %s, %i %s", lc,
	       SdnToTzolkin13 (greg.jdn),
	       MayaTzolkinDays[SdnToTzolkin20 (greg.jdn)], hd,
	       MayaHaabMonths[hm]);
      show_calendar ("Mayan", NULL, szDescription);
    }
  if (opts.showmoon)

    {

      show_calendar ("Moon", NULL,
		     MoonPhaseText[gregorian_moon_phase
				   (greg.year, greg.yday)]);
    }				/* lunar calendar */
  if (opts.showshire)

    {
      struct shire_tm *my_shire_tm;
      char szDescription[30];


      my_shire_tm =
	shire_date (greg.year, greg.month, greg.mday, greg.yday - 1);

      szDescription[0] = '\0';

      if (my_shire_tm->sh_holiday > 0)
	{
	  /* no months and days for holidays */
	  if (my_shire_tm->sh_wday)
	    {
	      sprintf (szDescription, "%s ",
		       shire_days_archaic[my_shire_tm->sh_wday]);
	    }
	  sprintf (szDescription, "%s%s %i",
		   szDescription, shire_holidays[my_shire_tm->sh_holiday],
		   my_shire_tm->sh_year);
	}
      else
	{
	  sprintf (szDescription, "%s %i %s %i",
		   shire_days_archaic[my_shire_tm->sh_wday],
		   my_shire_tm->sh_mday, shire_months[my_shire_tm->sh_mon],
		   my_shire_tm->sh_year);
	}

      show_calendar ("Shire", NULL, szDescription);
    }
  /* shire */

  if (DISPLAY_HTML == mopts.display_mode)

    {
      printf ("</TABLE>\n");
    }
}				/* show_calendars() */


void
criterr (char const *errmsg, ...)
{
  va_list args;
  va_start (args, errmsg);
  fprintf (stderr, "cday: ");
  vfprintf (stderr, errmsg, args);
  fputc ('\n', stderr);
  va_end (args);
  exit (EXIT_FAILURE);
}				/* criterr() */


#if defined(INC_DATABASE)
void
exitfunc (void)
{
  if (events && LLSnodePtr2First (ListEvents))

    {
      EVENT event;

      do

	{
	  LLSnodeDataTo (ListEvents, &event);
	  FREE (event.text);
	}
      while (LLSnodePtr2Next (ListEvents));
      events = 0;
    }

  if (libraries && LLSnodePtr2First (ListLibraries))

    {
      LIBRARY library;

      do

	{
	  LLSnodeDataTo (ListLibraries, &library);
	  FREE (library.fname);
	}
      while (LLSnodePtr2Next (ListLibraries));
      libraries = 0;
    }
}				/* exitfunc() */
#endif /* INC_DATABASE */


void
show_banner (int force)
{
  static int banner_sent;

  if (!banner_sent)
    {
      printf (PROG_NAME " version " PROG_VER " -- " PROG_DATE "\n");
      printf ("Copyright (C) 2003 by Andrew Ziem.  All Rights Reserved.\n");
    }

}
void
show_usage (int err)
{
  show_banner (TRUE);

  printf ("\n");
  printf ("usage: cday [arguments] [library files]\n");
  printf ("\n");
  printf ("Arguments:\n");
  printf ("(+/-)c(x)       Add/remove calendar x\n");
  printf ("-fdd[mm[yyyy]]  Force date in format ddmmyyyy\n");
  printf ("-h              Print this usage information\n");
  printf ("-l(+/-)         Enable/disable library support\n");
/*  printf ("-lib=x          Scan library directory\n"); */
  printf ("-v              Print version information\n");
  printf ("\n");
  printf ("Who is Jesus Christ?\n");
  printf ("   http://ChristianAnswers.Net/jesus/\n");
  printf ("\n");
  printf ("For further information, see the enclosed documentation or:\n");
  printf ("   http://cday.sourceforge.net\n");
  if (err)
    exit (EXIT_FAILURE);
  exit (EXIT_SUCCESS);
}


void
show_version ()
{
  show_banner (TRUE);

/*                        printf("\nCompiled on " __DATE__ " at " __TIME__ " using " COMPILER_NAME " (runtime revision %X).\n", _emx_rev); */
/*                          printf("\nCompiled on " __DATE__ " at " __TIME__ " using " COMPILER_NAME " (runtime version %X.%02X).\n", (_emx_rev >> 16) >> 8, (_emx_rev & 0xFFFF) & 0xFF); */
  printf ("\nCompiled on " __DATE__ " at " __TIME__ " using " COMPILER_NAME
	  ".\n");


  if (NULL == getenv ("TZ"))
    {
#if (defined __USE_SVID) || defined (__USE_XOPEN)
      if (0 == timezone)
	{
	  printf ("Timezone set to 0:00.\n");
	}
      else
	{
	  printf ("Timezone set to %02i:%02i.\n", timezone / (60 * 60),
		  timezone / (60 * 60 * 60));
	}
#else
      printf ("Unknown timezone.\n");
#endif
    }
  else
    {
      printf ("Environment variable 'TZ' set to '%s'.\n", getenv ("TZ"));
    }

  printf ("\n");
  printf ("CDAY\n");
  printf ("   http://cday.sourceforge.net\n");
  printf ("   http://chesire.freeservers.com\n");
  printf ("\n");
  printf ("Who is Jesus Christ?\n");
  printf ("   http://ChristianAnswers.Net/jesus/\n");
  printf ("\n");
  printf ("Portions public domain and copyright other authors.\n");
  printf ("For further information, see the enclosed documentation.\n");
  exit (EXIT_SUCCESS);
}

int
main (int argc, char *argv[])
{
  time_t t;
  struct tm *my_tm;
  int i;


  opts.showfrench = opts.showgregorian = opts.showmaya =
    opts.showgue = opts.showjewish = opts.showjulian =
    opts.showjdn = opts.showshire = opts.showmoon = TRUE;

  opts.showevents = opts.showbirthdays = opts.showreminders = TRUE;

  mopts.debug = mopts.quiet = mopts.is_cgi = FALSE;
  mopts.display_mode = DISPLAY_TEXT;

  if (NULL != getenv ("GATEWAY_INTERFACE"))
    {
      if (NULL != strstr (getenv ("GATEWAY_INTERFACE"), "CGI"))
	{
	  mopts.is_cgi = TRUE;
	  mopts.display_mode = DISPLAY_HTML;
	}
    }

  if (mopts.is_cgi)
    {
#ifdef THIS_NOT_TRUE
      printf ("HTTP/1.1 200 OK\r\n");
      printf ("Date: %s\n", asctime (localtime (&t)));
      printf ("Connection: close\r\n");	/* to do: check on this */
#endif
      printf ("Content-Type: text/html\n");	/* to do: check on charset */
      printf ("\n");
    }

  if (DISPLAY_HTML == mopts.display_mode)
    {
      printf ("<HTML>\n");
      printf ("<BODY>\n");
    }

#if defined(INC_DATABASE)
  /* initilize clean-up */
  atexit (exitfunc);

  /* initilize linked list */
  LLSsystemInit (3);
  ListEvents = LLScreate (sizeof (EVENT));
  ListLibraries = LLScreate (sizeof (LIBRARY));
  if (ERR_MEMORY == ListEvents)
    criterr ("unable to initilize primary linked list (memory error)\n");
#endif /* INC_DATABASE */

  /* set current time */
  tzset ();
  t = time (NULL);
  my_tm = localtime (&t);
  greg.year = my_tm->tm_year + 1900;	/* calendar year */
  greg.month = my_tm->tm_mon + 1;	/* month of year (1..12) */
  greg.mday = my_tm->tm_mday;	/* day of month (1..31) */

  /* initilize command line */
  for (i = 1; i < argc; i++)

    {
      char opt[100], ss[10];

      /* printf("debug: command line: %s\n",argv[i]); */
      strcpy (opt, argv[i]);
      if (opt[0] == '/' || opt[0] == '-' || opt[0] == '+')

	{
	  switch (toupper (opt[1]))

	    {
	    case 'C':

	      /* calendars */
	      switch (toupper (opt[2]))

		{
		case 'F':
		  opts.showfrench = (opt[0] == '+');
		  break;
		case 'G':
		  switch (toupper (opt[3]))

		    {
		    case 'R':
		      opts.showgregorian = (opt[0] == '+');
		      break;
		    case 'U':
		      opts.showgue = (opt[0] == '+');
		      break;
		    default:
		      fprintf (stderr, "bad switch: %s\n", opt);
		      break;
		    }
		case 'J':
		  switch (toupper (opt[3]))

		    {
		    case 'E':	/* cje */
		      opts.showjewish = (opt[0] == '+');
		      break;
		    case 'D':	/* cjd */
		      opts.showjdn = (opt[0] == '+');
		      break;
		    case 'U':
		      opts.showjulian = (opt[0] == '+');
		      break;


		    default:
		      fprintf (stderr, "bad switch: %s\n", opt);
		      break;
		    }
		case 'M':
		  switch (toupper (opt[3]))

		    {
		    case 'A':	/* cma */
		      opts.showmaya = (opt[0] == '+');
		      break;
		    case 'O':	/* cmo */
		      opts.showmoon = (opt[0] == '+');
		      break;
		    default:
		      fprintf (stderr, "bad switch: %s\n", opt);
		    }
		case 'S':	/* cs */
		  opts.showshire = (opt[0] == '+');
		  break;
		default:
		  opts.showfrench = opts.showgregorian = opts.showmaya =
		    opts.showgue = opts.showjewish = opts.showjulian =
		    opts.showjdn = opts.showshire = opts.showmoon =
		    (opt[0] == '+');
		  break;
		}		/* 'C' (calendars */
	      break;
	    case 'D':
	      mopts.debug = (opt[0] == '+');
	      break;
	    case 'H':

	      show_usage (FALSE);

	    case 'L':

#if defined(INC_DATABASE)
	      /* 0123456 */
	      /* -lib=dirname */
	      opt[4] = NUL;
	      if (0 == strcasecmp (&opt[1], "lib"))

		{
		  library_file_add (&opt[5]);
		}

	      else
		switch (toupper (opt[2]))

		  {
		  case 'E':
		    opts.showevents = (opt[0] == '+');
		    break;
		  case 'B':
		    opts.showbirthdays = (opt[0] == '+');
		    break;
		  case 'R':
		    opts.showreminders = (opt[0] == '+');
		    break;
		  default:
		    opts.showevents = opts.showbirthdays =
		      opts.showreminders = (opt[0] == '+');
		  }

#else /*  */
	      printf ("Sorry, library functions not linked.\n");

#endif /*  */
	      break;
	    case 'F':
	      if (mopts.debug)

		{
		  printf ("debug: raw date (pre-modified): %i-%i-%i\n",
			  greg.year, greg.month, greg.mday);
		  printf ("debug: f.opt = `%s`\n", opt);
		}
	      strncpy (ss, opt + 2, 2);
	      ss[2] = NUL;
	      greg.mday = atoi (ss);
	      if (strlen (opt + 2) >= 4)

		{
		  strncpy (ss, opt + 4, 2);
		  ss[2] = NUL;
		  greg.month = atoi (strncpy (ss, opt + 4, 2));
		}
	      if (strlen (opt + 2) >= 7)
		{
		  strncpy (ss, opt + 6, 7);
		  ss[7] = NUL;
		  greg.year = atoi (ss);
		}
	      if (mopts.debug)

		{
		  printf ("debug: raw date (post-modified): %i-%i-%i\n",
			  greg.year, greg.month, greg.mday);
		}
	      break;
	    case 'V':
	      show_version ();
	      break;
	    case 'S':
	      opts.sort = (opt[0] == '+');
	      break;
	    default:

	      show_usage (TRUE);
	    }			/* switch */
	}			/* if... */

      else

	{
#if defined(INC_DATABASE)
	  if (file_exists (opt))
	    {
	      library_file_add (opt);
	    }
	  else
	    {
	      if (!resource_exists (opt))
		{
#endif
		  show_usage (TRUE);
#if defined(INC_DATABASE)
		}
	    }
#endif
	}
    }				/* for */
  if (!gregorian_check_date (greg.year, greg.month, greg.mday))
    criterr ("\ninvalid date:\n day=%d\n month=%d\n year=%d", greg.mday,
	     greg.month, greg.year);
  if (mopts.debug)

    {
      printf ("debug: raw date: %i-%i-%i\n", greg.year, greg.month,
	      greg.mday);
    }
  greg.yday = daynum (greg.year, greg.month, greg.mday);
  greg.wday = dow (greg.year, greg.month, greg.mday);
  greg.jdn = ymd_to_jdnl (greg.year, greg.month, greg.mday, 0);

#if defined(INC_CALENDAR)
  show_calendars ();

#endif /* show calendars */


#if defined(INC_DATABASE)
  if (opts.showevents || opts.showbirthdays)
    {
      library_file_read_queued ();

      if (0 == events)
	{

	  /* if no libraries added, scan current working directory */
	  library_file_add ("./*");
	  library_file_add ("./lib/*");
	  if (0 == events)
	    {
	      /* try universal directories */
#if defined(__DOS__) || defined(__WIN32__)
	      library_file_add ("c:\\program files\\cday\\lib\\*");
#else
	      library_file_add ("~/cday/lib/*");
	      library_file_add ("/var/lib/cday/*");
#endif
	      library_file_read_queued ();
	    }
	}
      /* show */

      if ((opts.showbirthdays || opts.showevents) && events == 0)
	{
	  if (DISPLAY_HTML == mopts.display_mode)
	    printf ("<P>No birthdays or other events found for today.</P>\n");
	  else
	    printf ("\nNo birthdays or other events found for today.\n");
	}

      else if (opts.showbirthdays || opts.showevents)

	{

	  /* sort */
	  library_sort ();

	  /* show */
	  if (opts.showbirthdays)

	    {
	      library_show (T_BIRTHDAY);
	    }
	  if (opts.showevents)

	    {
	      library_show (T_EVENT);
	    }
	  if (mopts.debug)

	    {
	      printf ("\ndebug: %i total events\n", events);
	    }
	}
    }
#else
  if (mopts.debug)
    {
      printf ("debug: not displaying database because not linked\n");
    }

#endif /* INC_DATABASE */
  if (DISPLAY_HTML == mopts.display_mode)
    printf ("<P>That's all for today.</P>\n");
  else

    printf ("\nThat's all for today.\n");

  if (DISPLAY_HTML == mopts.display_mode)
    {
      printf ("</BODY>\n");
      printf ("</HTML>\n");
    }

  return 0;
}				/* main() */
