/*
 *
 *   lc - List Contents of a directory
 *
 *   SYNOPSYS:
 *	lc [pathspec]
 *         - where pathspec is any valid dos path
 *           with or without trailing backspace and
 *           optional drive specifier.
 *
 *   Written by Paul Frattaroli
 *   copyright(c) 1990 by Paul Frattaroli
 *
 *   Adapted from the UNIX version written by the University
 *   of Waterloo, Math Faculty Computing Facility.
 *   This version contains none of the code from the MFCF version
 *   but its soul and inspiration is attributed there.
 *
 *   If the UW MFCF was not the original designer/creator of this
 *   program, then be it known that...
 *
 *   I have written this code completely on my own, and have taken
 *   nothing from other sources.  While I connot say that the idea
 *   for this program is mine, the implementation is.
 *
 *   Note: UNIX is a trademark of AT&T.
 *
 *   This program is written in Turbo C version 2.0
 *   Compile with:   tcc -G -O -a -d -mt -Z lc
 *   and convert to com file with:   exe2com lc
 *
 */

#include <stdio.h>
#include <dir.h>
#include <dos.h>
#include <string.h>
#include <errno.h>

/* just a couple of convienient defines */

#define TRUE 1;
#define FALSE 0;

/* Function prototypes */

find_dirs( char path_spec[] );
find_files( char path_spec[], int SEARCHMASK );
print_usage( void );

/* Global variable definitions */

struct ffblk fblk;  /* DOS file search block, see dir.h */
char path_spec[256];  /* holds pathspec from command line if present */
char filename[18];  /* holds current filename */
int PRINT_COLS;  /* number of columns to print output in */

main( int argc, char *argv[] )
{
   int DODIRS, DOFILES;  /* variables to tell whether to print files */
                         /* and or directories */
   int PATHLOOP;  /* flag to indicate if a pathspec was in command line */
   int SEARCHMASK;  /* mask for indicating attribute of files to find */
   int i,j;  /* your general purpose counters */
   char *search;  /* intermediate search variable */

   DODIRS = TRUE;  /* want to print directories by default */
   DOFILES = TRUE;  /* want to print files by default */
   PRINT_COLS = 5;  /* want to print 5 column output by default */
   SEARCHMASK = 0; /* find just "normal" files */
   PATHLOOP = FALSE;  /* assume no pathspec is given */

   /* parse arguments and set path spec */

   for( i = 1; i < argc; i++ )  /* parse entire command line */
   {
      if( argv[i][0] == '-' )  /* is this an option */
      {
         for( j = 1; j < strlen( argv[i] ); j++ )
         /* must parse entire argv[i], can have -f1a type options */
         {
            switch( argv[i][j] )
            {
               case '?': print_usage();
                         exit(0);
               /* specify both cases in following because DOS is */
               /* case insensitive.  some people like CapsLock */
               case 'a': SEARCHMASK = FA_SYSTEM + FA_HIDDEN + FA_RDONLY;
                         break;
               case 'A': SEARCHMASK = FA_SYSTEM + FA_HIDDEN + FA_RDONLY;
                         break;
               case 'f': DODIRS = FALSE; /* directories won't be done */
                         break;
               case 'F': DODIRS = FALSE; /* directories won't be done */
                         break;
               case 'd': DOFILES = FALSE; /* files won't be done */
                         break;
               case 'D': DOFILES = FALSE; /* files won't be done */
                         break;
               case '1': PRINT_COLS = 1;
                         break;
               default:  fprintf( stderr, "Invalid argument!" );
                         print_usage();
                         exit(0);
            }
         }
      }
      else  /* not an option, must be a pathspec */
      {
      /* currently will not die if more than one pathspec is given */
      /* can make check for this using PATHLOOP and give error or */
      /* can easily expand program to search and list multiple paths */
         PATHLOOP = TRUE;
         if( argv[i][ strlen( argv[i] ) - 1 ] == '\\' )
            /* if user enters a trailing backslash on path spec */
            strncat( path_spec, argv[i], strlen( argv[i] ) - 1 );
         else
            /* user has not added trailing backslash */
            strcpy( path_spec, argv[i] );
         strcat( path_spec, "\\*.*" );
         if( argv[i][ strlen( argv[i] ) - 1 ] == ':' )
         {
            strcpy( path_spec, argv[i] );    /* for arg of type c: */
            strcat( path_spec, "*.*" );
         }
         search = strchr( argv[i], '*' );  /* for arg where wildcard */
         if( ! search == '' )              /* is specified */
            strcpy( path_spec, argv[i] );
      }
   }

   if( !PATHLOOP )
      strcpy( path_spec, "*.*" );  /* simple case, no args to lc */

   if( DODIRS || ( !DODIRS && !DOFILES ) )
      find_dirs( path_spec );  /* print directories in path spec */
   if( DOFILES || ( !DODIRS && !DOFILES ) )
      find_files( path_spec, SEARCHMASK );  /* print files in pathspec */
}


print_usage()   /* user has screwed up or is asking for advice */
{
   fprintf( stderr, "\nlc - List Contents of a directory.\n"
                    "copyright(c) 1990 Paul Frattaroli\n" );
   fprintf( stderr, "\nUsage:\tlc [-?afd1] [pathspec]\n\twhere pathspec is "
                    "any valid DOS path with\n\tor without trailing backslash"
                    "\n\teg.  lc c:\\path1\\path2\n" );
   fprintf( stderr, "\n\t-?: Show this message." );
   fprintf( stderr, "\n\t-a: List all files, including system files." );
   fprintf( stderr, "\n\t-f: List files only, no directories." );
   fprintf( stderr, "\n\t-d: List directories only, no files." );
   fprintf( stderr, "\n\t-1: Print output in one column, no multi-column." );
   fprintf( stderr, "\n\n" );
}


find_dirs( char path_spec[] )
{
   int done_dirs;
   int i, flag, no_dirs;

   no_dirs = TRUE;
   done_dirs = FALSE;  /* not done finding all directories yet */
   done_dirs = findfirst( path_spec, &fblk, FA_DIREC );  /* find first dir */
   if( _doserrno == ENMFILE )
      exit(0); /* Absolutely no files were found, not even . or .. */
               /* Don't print anything, and exit, for consistency */
   if( _doserrno == ENOPATH )  /* path or filename not found */
   {
      fprintf( stderr, "Directory not found.  Path specifier may be"
                       " invalid!\n" );
      exit(0);
   }
   flag = TRUE;  /* still first time.. haven't printed Directories: yet */
   while( ! done_dirs )
   {
      for( i = 1; i <= PRINT_COLS; i++ )  /* loop to count columns for output */
      {
	 strcpy( filename, fblk.ff_name );  /* get file out of struct */
         if( filename[0] == '.' || fblk.ff_attrib != FA_DIREC )
            --i;  /* don't update column count if not directory */
	 if( fblk.ff_attrib == FA_DIREC && filename[0] != '.' )
                  /* don't want to print . and .. */
         {
            if( flag )  /* if haven't printed Directories: yet */ 
            {
               printf("Directories:\n");
               flag = FALSE;  /* printed Directories: */
               no_dirs = FALSE;
            }
            /* print found files with appended spaces to fill up columns */
	    printf("%s", strlwr( strncat( filename, "                 ",
		   15 - strlen( fblk.ff_name ) ) ) );
         }
         /* finished columns yet? */
	 if( i == PRINT_COLS && fblk.ff_attrib == FA_DIREC )
	    printf( "\n" );
	 done_dirs = findnext( &fblk );  /* find next file or dir */
	 if( done_dirs )  /* any more? */
         {
            if( !no_dirs )
               printf( "\n" );
	    break;
         }
      }
   }
   if( !flag && PRINT_COLS != 1 )
      printf( "\n" );
}


find_files( char path_spec[], int SEARCHMASK )  /* find files in pathspec */
{
   int done_files;
   int i;

   done_files = FALSE;  /* not done finding files yet */
   done_files = findfirst( path_spec, &fblk, SEARCHMASK ); /* find first file */
   if( errno == ENMFILE )  /* no files found in path.  user make boo-boo */
   {
      fprintf( stderr, "No files found.  Path specifier may be invalid!\n" );
      exit(0);
   }
   if( ! done_files )  /* if not finished finding files */
      printf("Files:\n");
   while( ! done_files )
   {
	for( i = 1; i <= PRINT_COLS; i++ ) /* loop to count for column output */
	{
	   strcpy( filename, fblk.ff_name );  /* get file from struct */
           /* print filename with appended spaces to fill up column */
	   printf("%s", strlwr( strncat( filename, "                  ",
		  15 - strlen( fblk.ff_name ) ) ) );
	   done_files = findnext( &fblk );  /* find next file in path spec */
	   if( done_files )  /* no more files? */
	      break;
	}
	printf( "\n" );
   }
}
/* That's the end, that's it. no more... */
/* This has been lc by Paul Frattaroli */
/* written for Turbo C version 2.0 */

