/*
 * This file contains the file i/o stuff.  These functions get stuff from
 * from the outside world into a form for TDE.  The form TDE uses is a
 * double linked list.  Each node in the list points to the prev and
 * the next nodes, if they exist.  Also in each node is a pointer to a
 * line of text, a line length variable, and a node type indicator.  In
 * earlier versions of TDE, a '\n' was used to terminate a line of text.
 * In this version, we must keep an accurate count of characters in
 * each line of text, as no character is used to terminate the line.
 *
 * Each file must contain at least one node.  That node is called the
 * EOF node.  The EOF node terminates the double linked list for each
 * file.  The EOF node has a NULL pointer in the line field, a NULL
 * pointer in the next field, and an EOF in the len field.  Here's
 * a crude picture of the double linked list structure:
 *
 *              Regular node                             EOF node
 *     --------                                 --------
 *     | prev | ---> pointer to prev node       | prev | ---> unknown
 *     | line | ---> "Hello world"              | line | ---> NULL
 *     | len  | ---> 11                         | len  | ---> EOF
 *     | type | ---> DIRTY | syntax flags       | type | ---> 0
 *     | next | ---> pointer to next node       | next | ---> NULL
 *     --------                                 --------
 *
 * Implicitly, I am assuming that EOF is defined as (-1) in stdio.h.
 *
 * jmh 980701: I've added a top of file marker, so now each file contains
 *             at least two nodes and the first line in the file becomes
 *             line_list->next rather than just line_list, which is the
 *             zeroth line and remains constant. It also causes changes in
 *             other files, which are not documented.
 *
 * The load_file function is probably more complicated than expected, but
 * I was trying to read chunks of text that match the disk cluster size
 * and/or some multiple of the cache in the disk controller.
 *
 *
 * New editor name:  TDE, the Thomson-Davis Editor.
 * Author:           Frank Davis
 * Date:             June 5, 1991, version 1.0
 * Date:             July 29, 1991, version 1.1
 * Date:             October 5, 1991, version 1.2
 * Date:             January 20, 1992, version 1.3
 * Date:             February 17, 1992, version 1.4
 * Date:             April 1, 1992, version 1.5
 * Date:             June 5, 1992, version 2.0
 * Date:             October 31, 1992, version 2.1
 * Date:             April 1, 1993, version 2.2
 * Date:             June 5, 1993, version 3.0
 * Date:             August 29, 1993, version 3.1
 * Date:             November 13, 1993, version 3.2
 * Date:             June 5, 1994, version 4.0
 * Date:             December 5, 1998, version 5.0 (jmh)
 *
 * This code is released into the public domain, Frank Davis.
 * You may distribute it freely.
 */


#include "tdestr.h"             /* tde types */
#include "common.h"
#include "define.h"
#include "tdefunc.h"


/*
 * Name:    write_file
 * Purpose: To write text to a file
 *           way.
 * Date:    June 5, 1991
 * Passed:  name:  name of disk file or device
 *          open_mode:  fopen flags to be used in open
 *          file:  pointer to file structure to write
 *          start: first node to write
 *          end:   last node to write
 *          block: write a file or a marked block
 * Returns: OK, or ERROR if anything went wrong
 *
 * jmh:     August 27, 1997 - if name is empty (ie. "") then write to stdout.
 *          This allows TDE to be used in a pipe.
 */
int  write_file( char *name, int open_mode, file_infos *file, long start,
                 long end, int block )
{
FILE *fp;       /* file to be written */
char *p;
char *z = "\x1a";
register int rc;
int  bc;
int  ec;
int  len;
int  write_z;
int  write_eol;
long number;
line_list_ptr ll;
char *open_string;
char *eol;
size_t eol_count;

   if (block == LINE || block == BOX || block == STREAM) {
      if (g_status.marked_file == NULL)
         return( ERROR );
      file = g_status.marked_file;
   }

   write_z = mode.control_z;
   switch (open_mode) {
      case APPEND :
         open_string = "ab";
         break;
      case OVERWRITE :
      default :
         open_string = "wb";
         break;
   }
   switch (file->crlf) {
      case BINARY :
         eol_count = 0;
         eol = "";
         write_z = FALSE;
         break;
      case CRLF   :
         eol_count = 2;
         eol = "\r\n";
         break;
      case LF     :
         eol_count = 1;
         eol = "\n";
         break;
      default     :
         eol_count = 0;
         eol = "";
         assert( FALSE );
   }
   rc = OK;
   if (*name == '\0') {
      fp = stdout;
#if !defined( __UNIX__ )
#if defined( __MSC__ )
      _setmode( fileno( stdout ), _O_BINARY );
#else
      setmode( fileno( stdout ), O_BINARY );
#endif
#endif
   }
   else if ((fp = fopen( name, open_string )) == NULL || ceh.flag == ERROR)
      rc = ERROR;
   if (rc == OK) {
      ll = file->line_list->next;
      for (number=1; number<start && ll->next != NULL; number++)
         ll = ll->next;
      ec = bc = len = 0;
      if (block == BOX || block == STREAM) {
         bc  = file->block_bc;
         ec  = file->block_ec;
         len = ec + 1 - bc;
      }
      if (block == STREAM) {
         if (start == end )
            block = BOX;
      }
      p = g_status.line_buff;
      if (block == BOX) {

         assert( len >= 0 );
         assert( len < MAX_LINE_LENGTH );

         for (;start <= end  &&  ll->len != EOF && rc == OK; start++) {
            g_status.copied = FALSE;
            load_box_buff( p, ll, bc, ec, ' ' );
            if (fwrite( p, sizeof( char ), len, fp ) < (unsigned)len ||
                       ceh.flag == ERROR)
               rc = ERROR;
            if ((rc != ERROR  &&
                 fwrite( eol, sizeof( char ), eol_count, fp ) < eol_count)
                           || ceh.flag == ERROR)
               rc = ERROR;
            ll = ll->next;
            if (ll == NULL)
               rc = ERROR;
         }
      } else {
         for (number=start; number <= end && rc == OK && ll->len != EOF;
                   number++) {
            g_status.copied = FALSE;
            copy_line( ll );
            len = g_status.line_buff_len;
            if (block == STREAM) {
               if (number == start) {
                  bc = bc > len ? len : bc;
                  len -= bc;

                  assert( len >= 0 );

                  memmove( p, p + bc, len );
               } else if (number == end) {
                  ++ec;
                  len =  ec > len ? len : ec;
               }
            }

            assert( len >= 0 );
            assert( len < MAX_LINE_LENGTH );

            if (fwrite( p, sizeof( char ), len, fp ) < (unsigned)len ||
                    ceh.flag == ERROR)
               rc = ERROR;

            /*
             * if a Control-Z is already at EOF, don't write another one.
             */
            write_eol = TRUE;
            if (number == end) {
               if (file->crlf == CRLF ||  file->crlf == LF) {
                  if (len > 0  &&  *(p + len - 1) == '\x1a') {
                     write_eol = FALSE;
                     write_z = FALSE;
                  }
               }
            }

            if ((write_eol == TRUE  &&  rc != ERROR  &&
                  fwrite( eol, sizeof( char ), eol_count, fp ) < eol_count)
                  || ceh.flag == ERROR)
               rc = ERROR;
            ll = ll->next;
            if (ll == NULL)
               rc = ERROR;
         }
      }
      if (rc != ERROR  &&  write_z) {
         if (fwrite( z, sizeof( char ), 1, fp ) < 1 || ceh.flag == ERROR)
            rc = ERROR;
      }
      g_status.copied = FALSE;
      if (ceh.flag != ERROR) {
         if (fclose( fp ) != 0)
            rc = ERROR;
      }
   }
   return( rc );
}


/*
 * Name:    hw_save
 * Purpose: To save text to a file
 * Date:    November 11, 1989
 * Passed:  name:  name of disk file
 *          file:  pointer to file structure
 *          start: first character in text buffer
 *          end:   last character (+1) in text buffer
 *          block: type of block defined
 * Returns: OK, or ERROR if anything went wrong
 */
int hw_save( char *name, file_infos *file, long start, long end, int block )
{
   return( write_file( name, OVERWRITE, file, start, end, block ) );
}


/*
 * Name:    hw_append
 * Purpose: To append text to a file.
 * Date:    November 11, 1989
 * Passed:  name:  name of disk file
 *          file:  pointer to file structure
 *          start: first character in text buffer
 *          end:   last character (+1) in text buffer
 *          block: type of defined block
 * Returns: OK, or ERROR if anything went wrong
 */
int hw_append( char *name, file_infos *file, long start, long end, int block )
{
   return( write_file( name, APPEND, file, start, end, block ) );
}


/*
 * Name:    load_file
 * Purpose: To load a file into the array of text pointers.
 * Date:    December 1, 1992
 * Passed:  name:       name of disk file
 *          fp:         pointer to file structure
 *          file_mode:  BINARY or TEXT
 *          bin_len:    if opened in BINARY mode, length of node line
 *          insert:     NULL for new file, pointer to line to insert file
 * Returns: OK, or ERROR if anything went wrong
 *
 * jmh:     August 27, 1997 - if name is empty (ie. "") then read from stdin.
 *           This allows TDE to be used in a pipe.
 *          January 24, 1998 - added insert line pointer to allow a file to be
 *           inserted into another file.
 */
int  load_file( char *name, file_infos *fp, int *file_mode, int bin_len,
                line_list_ptr insert )
{
FILE *stream;                           /* stream to read */
int  rc;
text_ptr l;
line_list_ptr ll;
line_list_ptr temp_ll;
unsigned long line_count;
char *e;
char *residue;
int  len;
int  res;
size_t t1, t2;
int  crlf;
int  prompt_line;
char temp[MAX_COLS+2];

   /*
    * initialize the counters and pointers
    */
   rc = OK;
   len = 1;
   res = 0;
   residue = g_status.line_buff;
   prompt_line = g_display.nlines;
   if (insert == NULL) {
      fp->length  = 0;
      fp->undo_count = 0;
      fp->undo_top = fp->undo_bot = NULL;
      fp->line_list_end = fp->line_list = NULL;

      ll = (line_list_ptr)my_malloc( sizeof(line_list_struc), &rc );
      if (ll != NULL) {
         ll->type = 0;
         ll->len  = EOF;
         ll->line = NULL;
         ll->next = ll->prev = NULL;
         fp->undo_top = fp->undo_bot = ll;
      }

      ll = (line_list_ptr)my_malloc( sizeof(line_list_struc), &rc );
      if (ll != NULL) {
         temp_ll = (line_list_ptr)my_malloc( sizeof(line_list_struc), &rc );
         if (temp_ll != NULL) {
            ll->type = temp_ll->type = 0;
            ll->len  = temp_ll->len  = EOF;
            ll->line = temp_ll->line = NULL;
            ll->prev = temp_ll->next = NULL;
            ll->next = temp_ll;
            temp_ll->prev = ll;
            fp->line_list = ll;
            fp->line_list_end = temp_ll;
         }
      }
   }
   else
      ll = insert;
   line_count = fp->length;

   if (*name == '\0') {
      stream = stdin;
#if !defined( __UNIX__ )
#if defined( __MSC__ )
      _setmode( fileno( stdin ), _O_BINARY );
#else
      setmode( fileno( stdin ), O_BINARY );
#endif
#endif
   } else if ((stream = fopen( name, "rb" )) == NULL || ceh.flag == ERROR ||
         rc == ERROR) {
      /*
       * file not found or error loading file
       */
      combine_strings( temp, main7a, name, main7b );
      error( INFO, prompt_line, temp );
      if (insert == NULL) {
         if (fp->line_list_end != NULL)
            my_free( fp->line_list_end );
         if (fp->line_list != NULL)
            my_free( fp->line_list );
         if (fp->undo_top != NULL)
            my_free( fp->undo_top );
         rc = ERROR;
      }
   }
   if (rc == OK) {

#if defined( __UNIX__ )
      g_status.line_buff[0] = g_status.line_buff[1] == '\0';
      t1 = fread( g_status.line_buff, sizeof(char), 2, stream );
      if (t1 == 2) {
         if (g_status.line_buff[0] == '#' &&  g_status.line_buff[1] == '!')
            *file_mode = TEXT;
      }
      fseek( stream, 0, SEEK_SET );
#endif

      if (*file_mode == BINARY) {
         mode.trailing = FALSE;
         crlf = BINARY;
         if (bin_len < 0  ||  bin_len > READ_LENGTH)
            bin_len = DEFAULT_BIN_LENGTH;
         for (; rc == OK;) {
            t1 = fread( g_status.line_buff, sizeof(char), bin_len, stream );
            if (ferror( stream )  ||  ceh.flag == ERROR) {
               combine_strings( temp, main9, name, "'" );
               error( WARNING, prompt_line, temp );
               rc = ERROR;
            } else if (t1) {

               assert( t1 < MAX_LINE_LENGTH );

               l = (text_ptr)my_malloc( t1 * sizeof(char), &rc );
               temp_ll = (line_list_ptr)my_malloc(sizeof(line_list_struc), &rc);

               if (rc != ERROR) {

                  /*
                   * if everything is everything, copy from io buff to
                   *   dynamic mem.
                   */
                  if (t1 > 0)
                     my_memcpy( l, g_status.line_buff, t1 );

                  ++line_count;
                  temp_ll->line = l;
                  temp_ll->type = 0;
                  temp_ll->len  = t1;
                  insert_node( fp, ll, temp_ll );
                  ll = temp_ll;
               } else
                  rc = show_file_2big( name, prompt_line, temp_ll, l );
            } else
               break;
         }
      } else {
         crlf = LF;
         for (; rc == OK;) {
            t1 = fread( g_status.line_buff, sizeof(char), READ_LENGTH, stream );
            if (ferror( stream )  ||  ceh.flag == ERROR) {
               combine_strings( temp, main9, name, "'" );
               error( WARNING, prompt_line, temp );
               rc = ERROR;
            } else {

               /*
                * "e" walks down io buffer 1 looking for end of lines.  "t1"
                *   keeps count of number of characters in io buffer 1.
                */
               e = g_status.line_buff;
               while (t1 && rc == OK) {

                  /*
                   * while "t1" is not zero and "len" is less than max line
                   *   length, let "e" walk down the buffer until it finds <LF>.
                   */
                  for (; t1 && len < READ_LENGTH &&  *e != '\n';
                       len++, e++, t1--);

                  /*
                   * if "t1" is not zero, we either found a <LF> or the line
                   *   length max'd out.
                   */
                  if (t1  ||  len >= READ_LENGTH) {

                     if (len > 1 && *e == '\n') {
                        if (len - res == 1) {
                           if (*(residue + res - 1) == '\r') {
                              --len;
                              --res;
                              crlf = CRLF;
                           }
                        } else {
                           if (*(e - 1) == '\r') {
                              --len;
                              crlf = CRLF;
                           }
                        }
                     }
                     if (len > 0)
                        --len;

                     assert( len >= 0 );
                     assert( len < MAX_LINE_LENGTH );

                     /*
                      * allocate space for relocatable array of line pointers
                      *   and allocate space for the line we just read.
                      */
                     l = (text_ptr)my_malloc( len * sizeof(char), &rc );
                     temp_ll =
                       (line_list_ptr)my_malloc( sizeof(line_list_struc), &rc );

                     if (rc != ERROR) {

                        /*
                         * if everything is everything, copy from io buff to
                         *  dynamic mem.  "residue" keeps up with the beginning
                         *  of line in io buffer.
                         */
                        if (res > 0) {

                           assert( res >= 0 );
                           assert( len - res >= 0 );

                           if (res > 0)
                              my_memcpy( l, residue, res );
                           if (len - res > 0)
                              my_memcpy( l+res, g_status.line_buff, len-res );
                           res = 0;
                        } else
                           if (len > 0)
                              my_memcpy( l, residue, len );

                        ++line_count;
                        temp_ll->line = l;
                        temp_ll->type = 0;
                        temp_ll->len  = len;
                        insert_node( fp, ll, temp_ll );
                        ll = temp_ll;

                        /*
                         * reset io buffer pointers and counters.
                         */
                        if (t1 == 0)
                           residue = g_status.tabout_buff;
                        else {
                           t1--;
                           if (len + 1 < READ_LENGTH)
                              residue =  t1 == 0 ? g_status.tabout_buff : ++e;
                           else {
                              if (*e != '\n') {
                                 t1++;
                                 residue = e;
                              } else
                                 residue = t1 == 0 ? g_status.tabout_buff : ++e;
                           }
                        }
                        len = 1;
                     } else
                        rc = show_file_2big( name, prompt_line, temp_ll, l );
                  } else if (len < READ_LENGTH ) {
                     if (!feof( stream ))
                        res = len - 1;
                  } else {
                     error( WARNING, prompt_line, main9a );
                     rc = ERROR;
                  }
               }
            }

            if (rc != OK)
               break;

            /*
             * we may have read all lines that end in '\n', but there may
             *   be some residue after the last '\n'.  ^Z is a good example.
             */
            if (feof( stream )) {
               if (len > 1) {
                  --len;
                  if (t1 == 0)
                     --e;

                  assert( len >= 0 );
                  assert( len < MAX_LINE_LENGTH );

                  /*
                   * allocate space for relocatable array of line pointers and
                   *   allocate space for the line we just read.
                   */
                  l = (text_ptr)my_malloc( len * sizeof(char), &rc );
                  temp_ll =
                      (line_list_ptr)my_malloc( sizeof(line_list_struc), &rc );

                  if (rc != ERROR) {

                     /*
                      * if everything is everything, copy from io buff to
                      *   dynamic mem.  "residue" keeps up with the beginning
                      *   of line in io buffer.
                      */
                     if (res > 0) {

                        assert( res >= 0 );
                        assert( res < MAX_LINE_LENGTH);
                        assert( len - res >= 0 );
                        assert( len - res < MAX_LINE_LENGTH);

                        if (res > 0 )
                           my_memcpy( l, residue, res );
                        if (len - res > 0)
                           my_memcpy( l + res, g_status.line_buff, len - res );
                     } else
                        if (len > 0)
                           my_memcpy( l, residue, len );
                     ++line_count;
                     temp_ll->line = l;
                     temp_ll->type = 0;
                     temp_ll->len  = len;
                     insert_node( fp, ll, temp_ll );
                  } else
                     rc = show_file_2big( name, prompt_line, temp_ll, l );
               }
               break;
            }

            t2 = fread(g_status.tabout_buff, sizeof(char), READ_LENGTH, stream);
            if (ferror( stream )  ||  ceh.flag == ERROR) {
               combine_strings( temp, main9, name, "'" );
               error( WARNING, prompt_line, temp );
               rc = ERROR;
            } else if (rc == OK) {
               e = g_status.tabout_buff;
               while (t2 && rc == OK) {
                  for (; t2 && len < READ_LENGTH &&  *e != '\n';
                       len++, e++, t2--);
                  if (t2  ||  len >= READ_LENGTH) {

                     if (len > 1 && *e == '\n') {
                        if (len - res == 1) {
                           if (*(residue + res - 1) == '\r') {
                              --len;
                              --res;
                              crlf = CRLF;
                           }
                        } else {
                           if (*(e - 1) == '\r') {
                              --len;
                              crlf = CRLF;
                           }
                        }
                     }
                     if (len > 0)
                        --len;

                     assert( len >= 0 );
                     assert( len < MAX_LINE_LENGTH );

                     l = (text_ptr)my_malloc( len * sizeof(char), &rc );
                     temp_ll =
                      (line_list_ptr)my_malloc( sizeof(line_list_struc), &rc );

                     if (rc != ERROR) {
                        if (res > 0) {

                           assert( res >= 0 );
                           assert( res < MAX_LINE_LENGTH);
                           assert( len - res >= 0 );
                           assert( len - res < MAX_LINE_LENGTH);

                           if (res > 0)
                              my_memcpy( l, residue, res );
                           if (len - res > 0)
                              my_memcpy( l+res, g_status.tabout_buff, len-res );
                           res = 0;
                        } else
                           if (len > 0)
                              my_memcpy( l, residue, len );

                        ++line_count;
                        temp_ll->line = l;
                        temp_ll->type = 0;
                        temp_ll->len  = len;
                        insert_node( fp, ll, temp_ll );
                        ll = temp_ll;

                        if (t2 == 0)
                           residue = g_status.line_buff;
                        else {
                           t2--;
                           if (len + 1 < READ_LENGTH)
                              residue =  t2 == 0 ? g_status.line_buff : ++e;
                           else {
                              if (*e != '\n') {
                                 t2++;
                                 residue = e;
                              } else
                                 residue =  t2 == 0 ? g_status.line_buff : ++e;
                           }
                        }
                        len = 1;
                     } else
                        rc = show_file_2big( name, prompt_line, temp_ll, l );
                  } else if (len < READ_LENGTH) {
                     if (!feof( stream ))
                        res = len - 1;
                  } else {
                     error( WARNING, prompt_line, main9a );
                     rc = ERROR;
                  }
               }
            }

            if (rc != ERROR  &&  feof( stream )) {
               if (len > 1) {
                  --len;
                  if (t2 == 0)
                     --e;

                  assert( len >= 0 );
                  assert( len < MAX_LINE_LENGTH );

                  l = (text_ptr)my_malloc( len * sizeof(char), &rc );
                  temp_ll =
                      (line_list_ptr)my_malloc( sizeof(line_list_struc), &rc );

                  if (rc != ERROR) {
                     if (res > 0) {

                        assert( res >= 0 );
                        assert( res < MAX_LINE_LENGTH);
                        assert( len - res >= 0 );
                        assert( len - res < MAX_LINE_LENGTH);

                        if (res > 0)
                           my_memcpy( l, residue, res );
                        if (len - res > 0)
                           my_memcpy( l+res, g_status.tabout_buff, len - res );
                     } else
                        if (len > 0)
                           my_memcpy( l, residue, len );

                     ++line_count;
                     temp_ll->line = l;
                     temp_ll->type = 0;
                     temp_ll->len  = len;
                     insert_node( fp, ll, temp_ll );
                  } else
                     rc = show_file_2big( name, prompt_line, temp_ll, l );
               }
               break;
            }
         }
         *file_mode = crlf;
      }

      /*
       * close the file
       */
      fp->length = line_count;
   }
   /*
    * Restore input back to the keyboard for shelling.
    */
   if (stream == stdin) {
      freopen( STDFILE, "r", stdin );
      g_status.input_redir = FALSE;
   } else if (stream != NULL)
      fclose( stream );
   return( rc );
}


/*
 * Name:    insert_node
 * Purpose: To insert a node into a double linked list
 * Date:    December 1, 1992
 * Passed:  fp:  pointer to file structure that owns the double linked list
 *          current: pointer to current node in double linked list
 *          new:     pointer to new node to insert into double linked list
 * Notes:   if the current list pointer is the last node in the list, insert
 *            new node behind current node.
 */
void insert_node( file_infos *fp, line_list_ptr current, line_list_ptr new )
{

   /*
    * standard double linked list insert
    */
   if (current->next != NULL) {
      current->next->prev = new;
      new->next = current->next;
      current->next = new;
      new->prev = current;
   /*
    * if the current node is the NULL node, insert the new node behind current
    */
   } else {
      new->next = current;
      current->prev->next = new;
      new->prev = current->prev;
      current->prev = new;
   }
}


/*
 * Name:    show_file_2big
 * Purpose: tell user we ran out of room loading file
 * Date:    December 1, 1992
 * Passed:  name:  name of disk file
 *          line:  line to display messages
 *          ll:    double linked list pointer
 *          t:     text line pointer
 * Returns: WARNING
 * Notes:   one or both of the malloc requests overflowed the heap.  free the
 *            dynamic if allocated.
 */
int  show_file_2big( char *name, int prompt_line, line_list_ptr ll, text_ptr t )
{
char temp[MAX_COLS+2];

   combine_strings( temp, main10a, name, main10b );
   error( WARNING, prompt_line, temp );
   if (t != NULL)
      my_free( t );
   if (ll != NULL)
      my_free( ll );
   return( WARNING );
}


/*
 * Name:    backup_file
 * Purpose: To make a back-up copy of current file.
 * Date:    June 5, 1991
 * Passed:  window:  current window pointer
 */
int  backup_file( TDE_WIN *window )
{
char *old_line_buff;
char *old_tabout_buff;
int  old_line_buff_len;
int  old_tabout_buff_len;
int  old_copied;
int  rc;
file_infos *file;

   rc = OK;
   file = window->file_info;
   if (file->backed_up == FALSE  &&  file->modified == TRUE) {
      old_copied = g_status.copied;
      old_line_buff_len = g_status.line_buff_len;
      old_line_buff = calloc( MAX_LINE_LENGTH, sizeof(char) );
      old_tabout_buff_len = g_status.tabout_buff_len;
      old_tabout_buff = calloc( MAX_LINE_LENGTH, sizeof(char) );

      if (old_line_buff != NULL  &&  old_tabout_buff != NULL) {
         memcpy( old_line_buff, g_status.line_buff, MAX_LINE_LENGTH );
         memcpy( old_tabout_buff, g_status.tabout_buff, MAX_LINE_LENGTH );
         if ((rc = save_backup( window )) != ERROR)
            file->backed_up = TRUE;
         else
            rc = ERROR;
         memcpy( g_status.line_buff, old_line_buff, MAX_LINE_LENGTH );
         memcpy( g_status.tabout_buff, old_tabout_buff, MAX_LINE_LENGTH );
         g_status.line_buff_len = old_line_buff_len;
         g_status.tabout_buff_len = old_tabout_buff_len;
         g_status.copied = old_copied;
      } else {
         error( WARNING, window->bottom_line, main4 );
         rc = ERROR;
      }
      if (old_line_buff != NULL)
         free( old_line_buff );
      if (old_tabout_buff != NULL)
         free( old_tabout_buff );
   }
   return( rc );
}


/*
 * Name:    edit_file
 * Purpose: To allocate space for a new file structure and set up some
 *           of the relevant fields.
 * Date:    June 5, 1991
 * Modified: November 13, 1993, Frank Davis per Byrial Jensen
 * Passed:  name:  name of the file to edit
 *          file_mode:  BINARY or TEXT
 *          bin_length: if opened in BINARY mode, length of binary lines
 * Returns: OK if file structure could be created
 *          ERROR if out of memory
 *
 * Change:  file->next_letter is initialized to 0 instead 'a'
 *
 * jmh:     August 27, 1997 - if name is empty (ie. "") then read from stdin.
 *          This allows TDE to be used in a pipe.
 *
 * 980503:  Use of get_full_path to store the filename.
 */
int  edit_file( char *name, int file_mode, int bin_length )
{
int  rc;        /* return code */
int  existing;
int  line;
int  rcol;
register file_infos *file; /* file structure for file belonging to new window */
file_infos *fp;
long found_line;
line_list_ptr ll;
line_list_ptr temp_ll;

   line = g_display.nlines;
   rc = OK;
   /*
    * allocate a file structure for the new file
    */
   file = (file_infos *)calloc( 1, sizeof(file_infos) );
   if (file == NULL) {
      error( WARNING, line, main4 );
      return( ERROR );
   }
   existing = TRUE;
   if (*name == '\0')
      rc = load_file( name, file, &file_mode, bin_length, NULL );

   else if (hw_fattrib( name ) == OK) {

      if (g_status.command != DefineGrep  &&
          g_status.command != DefineRegXGrep  &&
          g_status.command != RepeatGrep)
         rc = load_file( name, file, &file_mode, bin_length, NULL );
      else {
         if (g_status.sas_defined) {
            rc = load_file( name, file, &file_mode, bin_length, NULL );
            if (rc != ERROR) {
               found_line = 1L;
               rcol = 0;
               if (g_status.sas_search_type == BOYER_MOORE)
                  ll = search_forward( file->line_list->next, &found_line,
                                       (size_t *)&rcol, -1 );
               else
                  ll = regx_search_forward( file->line_list->next, &found_line,
                                            &rcol, -1 );
               if (ll == NULL)
                  rc = ERROR;
               else {
                  g_status.sas_rline = found_line;
                  g_status.sas_rcol  = rcol;
                  g_status.sas_ll    = ll;
               }
            }
         } else
            rc = ERROR;
      }
   } else {
      if (ceh.flag == ERROR)
         rc = ERROR;
      else {
         existing = FALSE;
         file->length = 0l;
         file->undo_top = file->undo_bot = NULL;
         file->line_list_end = file->line_list = NULL;
         file->undo_count = 0;
         ll = (line_list_ptr)my_malloc( sizeof(line_list_struc), &rc );
         if (ll != NULL) {
            ll->line = NULL;
            ll->next = ll->prev = NULL;
            ll->type = 0;
            ll->len  = EOF;
            file->undo_top = file->undo_bot = ll;
         } else
            rc = ERROR;

         ll = (line_list_ptr)my_malloc( sizeof(line_list_struc), &rc );
         if (ll != NULL) {
            temp_ll = (line_list_ptr)my_malloc( sizeof(line_list_struc), &rc );
            if (temp_ll != NULL) {
               ll->type = temp_ll->type = 0;
               ll->len  = temp_ll->len  = EOF;
               ll->line = temp_ll->line = NULL;
               ll->prev = temp_ll->next = NULL;
               ll->next = temp_ll;
               temp_ll->prev = ll;
               file->line_list = ll;
               file->line_list_end = temp_ll;
            } else
               rc = ERROR;
         } else
            rc = ERROR;
         if (rc == ERROR) {
            if (file->undo_top != NULL)
               my_free( file->undo_top );
            if (file->line_list_end != NULL)
               my_free( file->line_list_end );
            if (file->line_list != NULL)
               my_free( file->line_list );
         } else
            if (file_mode == TEXT)
#if defined( __UNIX__ )
               file_mode = LF;
#else
               file_mode = CRLF;
#endif
      }
   }

   if (rc != ERROR) {
      /*
       * add file into list
       */
      file->prev = NULL;
      file->next = NULL;
      if (g_status.file_list == NULL)
         g_status.file_list = file;
      else {
         fp = g_status.current_file;
         file->prev = fp;
         if (fp->next)
            fp->next->prev = file;
         file->next = fp->next;
         fp->next = file;
      }

      /*
       * set up all the info we need to know about a file.
       */

      assert( file_mode == CRLF  ||  file_mode == LF  ||  file_mode == BINARY );
      assert( strlen( name ) < PATH_MAX );

      if (*name != '\0') {
         get_full_path( name, file->file_name );
         get_fattr( name, &file->file_attrib );
      }
      file->block_type  = NOTMARKED;
      file->block_br    = file->block_er = 0l;
      file->block_bc    = file->block_ec = 0;
      file->ref_count   = 0;
      file->modified    = FALSE;
      file->backed_up   = FALSE;
      file->new_file    = !existing;
      file->next_letter = 0;
      file->file_no     = ++g_status.file_count;
      file->crlf        = file_mode;
      g_status.current_file = file;
      make_backup_fname( file );
   } else if (file != NULL) {
      /*
       * free the line pointers and linked list of line pointers.
       */
      ll = file->undo_top;
      while (ll != NULL) {
         temp_ll = ll->next;
         if (ll->line != NULL)
            my_free( ll->line );
         my_free( ll );
         ll = temp_ll;
      }

      ll = file->line_list;
      while (ll != NULL) {
         temp_ll = ll->next;
         if (ll->line != NULL)
            my_free( ll->line );
         my_free( ll );
         ll = temp_ll;
      }

#if defined( __MSC__ )
      _fheapmin( );
#endif

      free( file );
   }
   return( rc );
}


/*
 * Name:    edit_another_file
 * Purpose: Bring in another file to editor.
 * Date:    June 5, 1991
 * Passed:  window:  pointer to current window
 * Notes:   New window replaces old window.  Old window becomes invisible.
 *
 * jmh:     January 24, 1998 - can insert file into current.
 * jmh 981127: if the name is a wildcard, bring up the directory list.
 */
int  edit_another_file( TDE_WIN *window )
{
char fname[PATH_MAX+2];         /* new name for file */
register TDE_WIN *win;          /* put window pointer in a register */
int  rc;
int  file_mode;
int  bin_length;
char *old_language;

   win = window;
   entab_linebuff( );
   if (un_copy_line( win->ll, win, TRUE ) == ERROR)
      return( ERROR );

   if (g_status.command == InsertFile && win->ll->next == NULL)
      return( ERROR );

   /*
    * read in name, no default
    */
   fname[0] = '\0';
   /*
    * file name to edit
    */
   if ((rc = get_name( (g_status.command == InsertFile) ? ed15a : ed15,
                       win->bottom_line, fname )) == OK  &&  *fname != '\0') {

      assert( strlen( fname ) < PATH_MAX );

      if (is_pattern( fname )) {
         rc = list_and_pick( fname, win );
         if (rc != OK)
            return( ERROR );
      }

      if (g_status.command == InsertFile) {
         long change = win->file_info->length;
         file_mode  = TEXT;
         bin_length = 0;
         if (binary_file( fname ) == TRUE && g_status.file_chunk != 0) {
            file_mode  = BINARY;
            bin_length = g_status.file_chunk;
         }
         rc = load_file(fname, win->file_info, &file_mode, bin_length, win->ll);
         if (rc == OK) {
            win->file_info->modified = TRUE;
            syntax_check_block( change, win->file_info->length, win->ll->next,
                                win->file_info->syntax );
            redraw_screen( win );
         }
         change = win->file_info->length - change;
         adjust_windows_cursor( win, change );
         restore_marked_block( win, (int)change );
      }
      else {
         /*
          * Don't use the command line language.
          */
         old_language = g_status.language;
         g_status.language = NULL;
         rc = attempt_edit_display( fname, LOCAL );
         g_status.language = old_language;
      }
   }
   return( rc );
}


/*
 * Name:    attempt_edit_display
 * Purpose: try to load then display a file
 * Date:    June 5, 1991
 * Passed:  fname:       file name to load
 *          update_type: update current window or entire screen
 * Notes:   When we first start the editor, we need to update the entire
 *          screen.  When we load in a new file, we only need to update
 *          the current window.
 *
 * jmh:     September 8, 1997 - call init_syntax() to setup syntax highlighting
 * jmh 980801: moved the above from create_window (why didn't I think to put
 *              it here in the first place?)
 * jmh 981111: set deflate tabs when using binary mode.
 * jmh 981127: determine if a file should be loaded in binary directly here,
 *              rather than in all the calling routines.
 */
int  attempt_edit_display( char *fname, int update_type )
{
register int rc;
TDE_WIN *win;
int file_mode;
int bin_length;

   file_mode  = TEXT;
   bin_length = 0;

   if (g_status.file_mode == BINARY || binary_file( fname ) == TRUE) {
      bin_length = g_status.file_chunk;
      if (bin_length != 0)
         file_mode = BINARY;
   }

   rc = edit_file( fname, file_mode, bin_length );
   if (rc != ERROR) {
      rc = initialize_window( );
      if (rc != ERROR) {
         win = g_status.current_window;
         if (file_mode == TEXT)
            init_syntax( win );

         if (file_mode == BINARY && mode.inflate_tabs) {
            /*
             * tabs are just a normal character in binary mode.
             * This is just a quick hack; one day I'll makes tabs a file option.
             */
            mode.inflate_tabs = 0;
            update_type = GLOBAL;
         }

         if (update_type == LOCAL) {
            if (g_status.command != DefineGrep  &&
                g_status.command != DefineRegXGrep  &&
                g_status.command != RepeatGrep)
               redraw_current_window( win );
            show_file_count( g_status.file_count );
            show_window_count( g_status.window_count );
            show_avail_mem( );
         } else if (update_type == GLOBAL)
            redraw_screen( win );
         if (win->file_info->new_file) {
            g_status.command = AddLine;
            insert_newline( win );
            win->file_info->modified = FALSE;
         }
      }
   }
   return( rc );
}


/*
 * Name:     file_file
 * Purpose:  To file the current file to disk.
 * Date:     September 17, 1991
 * Modified: August 27, 1997, Jason Hood
 * Passed:   window:  pointer to current window
 *
 * Change:   If output has been redirected then send the file to stdout.
 *           If a block has been marked, that will be sent, not the whole file.
 */
int  file_file( TDE_WIN *window )
{
register file_infos *file;
long first, last;

   if (!g_status.output_redir) {
      if (save_file( window ) == OK)
         finish( window );
   } else {
      entab_linebuff( );
      if (un_copy_line( window->ll, window, TRUE ) == ERROR)
         return( ERROR );
      file = window->file_info;
      if (file->block_type <= NOTMARKED) {
         first = 1;
         last  = file->length;
      } else {
         first = file->block_br;
         last  = file->block_er;
      }
      hw_save( "", file, first, last, file->block_type );
      finish( window );
   }
   return( OK );
}


/*
 * Name:    file_all
 * Purpose: To save all windows and exit the editor.
 * Date:    August 21, 1997
 * Author:  Jason Hood
 * Passed:  window:  pointer to current window
 * Notes:   if any window couldn't be saved, all windows will remain open.
 */
int  file_all( TDE_WIN *window )
{
register TDE_WIN *wp;
int  rc = OK;

   if (g_status.output_redir) {
      error( WARNING, g_display.nlines, main22 );
      return( ERROR );
   }

   entab_linebuff( );
   if (un_copy_line( window->ll, window, TRUE ) == ERROR)
      return( ERROR );

   for (wp=g_status.window_list; wp != NULL; wp=wp->next) {
      if (save_file( wp ) == ERROR) rc = ERROR;
   }
   if (rc == OK)
      g_status.stop = TRUE;
   return( OK );
}


/*
 * Name:    save_file
 * Purpose: To save the current file to disk.
 * Date:    June 5, 1991
 * Passed:  window:  pointer to current window
 * Notes:   If anything goes wrong, then the modified flag is set.
 *          If the file is saved successfully, then modified flag is
 *           cleared.
 */
int  save_file( TDE_WIN *window )
{
register file_infos *file;
int  rc;
line_list_ptr temp_ll;
char temp[PATH_MAX+2];

   entab_linebuff( );
   if (un_copy_line( window->ll, window, TRUE ) == ERROR)
      return( ERROR );
   file = window->file_info;
   if (file->modified == FALSE)
      return( OK );
   /*
    * set up file name
    */
   assert( strlen( file->file_name ) < PATH_MAX );

   strcpy( temp, file->file_name );

   /*
    * see if there was a file name - if not, then make the user
    *  supply one.
    */
   if (strlen( temp ) == 0)
      rc = save_as_file( window );
   else {
      /*
       * save the file
       */
      rc = write_to_disk( window, temp );
      if (rc != ERROR) {
         file->modified = FALSE;
         file->new_file = FALSE;
      }
   }

   /*
    * clear the dirty flags
    */
   if (rc == OK) {
      temp_ll = file->line_list->next;
      for (; temp_ll->len != EOF; temp_ll=temp_ll->next)
         temp_ll->type &= ~DIRTY;
      file->dirty = GLOBAL;
   }

   return( rc );
}


/*
 * Name:    save_backup
 * Purpose: To save a backup copy of the current file to disk.
 * Date:    June 5, 1991
 * Passed:  window:  pointer to current window
 */
int  save_backup( TDE_WIN *window )
{
   /*
    * set up file name
    */
   return( write_to_disk( window, window->file_info->backup_fname ) );
}


/*
 * Name:    save_as_file
 * Purpose: To save the current file to disk, but under a new name.
 * Date:    June 5, 1991
 * Passed:  window:  pointer to current window
 *
 * jmh:     April 29, 1998 - renames and saves the current file.
 * jmh 980803: default to current name.
 */
int  save_as_file( TDE_WIN *window )
{
int  prompt_line;
int  rc;
register TDE_WIN *win;           /* put window pointer in a register */
register file_infos *file;
char answer[PATH_MAX+2];
fattr_t fattr;
line_list_ptr temp_ll;

   win = window;
   entab_linebuff( );
   if (un_copy_line( win->ll, win, TRUE ) == ERROR)
      return( ERROR );
   /*
    * read in name
    */
   prompt_line = win->bottom_line;

   strcpy( answer, win->file_info->file_name );
   /*
    * new file name:
    */
   if ((rc = get_name( utils9, prompt_line, answer )) == OK  &&
       *answer != '\0') {

       /*
        * make sure it is OK to overwrite any existing file
        */
      rc = get_fattr( answer, &fattr );
      if (rc == OK) {

         /*
          * file exists. make sure this is a regular file, first
          */
#if defined( __UNIX__ )
         if (S_ISREG( fattr ))
#else
         if (!(fattr & SUBDIRECTORY))
#endif
         {  /*
             * overwrite existing file?
             */
            if (get_yn( utils10, prompt_line, R_PROMPT | R_ABORT ) != A_YES  ||
                change_mode( answer, prompt_line ) == ERROR)
               rc = ERROR;
         } else
            rc = ERROR;
      }
#if defined( __UNIX__ ) || defined( __DJGPP__ )
      else
         /*
          * file name does not exist. take a chance on a valid file name.
          */
         rc = OK;
#endif
      if (rc != ERROR)
         rc = write_to_disk( win, answer );

      if (rc == OK) {
         file = window->file_info;
         strcpy( file->file_name, answer );
         show_window_fname( win );
         file->modified = FALSE;
         file->new_file = FALSE;

         temp_ll = file->line_list->next;
         for (; temp_ll->len != EOF; temp_ll=temp_ll->next)
            temp_ll->type &= ~DIRTY;
         file->dirty = GLOBAL;
      }

   }
   return( rc );
}


/*
 * Name:    make_backup_fname
 * Purpose: change the file name's extension to "bak"
 * Date:    January 6, 1992
 * Passed:  file: information allowing access to the current file
 */
void make_backup_fname( file_infos *file )
{
char *p;
int  i;
int  len;
char temp[PATH_MAX+2];

   /*
    * if this is a new file or pipe then don't create a backup - can't backup
    *   a nonexisting file.
    */
   if (file->new_file || file->file_name == '\0')
      file->backed_up = TRUE;

   /*
    * find the file name extension if it exists
    */
   else {
      assert( strlen( file->file_name ) < PATH_MAX );

      strcpy( temp, file->file_name );
      len = strlen( temp );
      for (i=len, p=temp + len; i>=0; i--) {

         /*
          * we found the '.' extension character.  get out
          */
         if (*p == '.')
            break;

         /*
          * we found the drive or directory character.  no extension so
          *  set the pointer to the end of file name string.
          */
         else if (*p == '/'
#if !defined( __UNIX__ )
                            || *p == ':'
#endif
                                        ) {
            p = temp + len;
            break;

         /*
          * we're at the beginning of the string - no '.', drive, or directory
          *  char was found.  set the pointer to the end of file name string.
          */
         } else if (i == 0) {
            p = temp + len;
            break;
         }
         --p;
      }
      strcpy( p, ".bak" );
      strcpy( file->backup_fname, temp );
   }
}


/*
 * Name:    write_to_disk
 * Purpose: To write file from memory to disk
 * Date:    June 5, 1991
 * Passed:  window:  pointer to current window
 *          fname:   file name to save on disk
 */
int  write_to_disk( TDE_WIN *window, char *fname )
{
register file_infos *file;
int  rc;
int  prompt_line;
char temp[MAX_COLS+PATH_MAX+2];
DISPLAY_BUFF;
fattr_t fattr;

   file = window->file_info;
   prompt_line = window->bottom_line;
   save_screen_line( 0, prompt_line, display_buff );
   /*
    * saving
    */
   combine_strings( temp, utils6, fname, "'" );
   set_prompt( temp, prompt_line );
   if ((rc = hw_save( fname, file, 1L, file->length, NOTMARKED )) == ERROR) {
      if (ceh.flag != ERROR) {
         if (get_fattr( fname, &fattr ) == OK &&
#if defined( __UNIX__ )
             !(fattr & (S_IWUSR | S_IWGRP | S_IWOTH)))
#else
             fattr & READ_ONLY)
#endif
            /*
             * file is read only
             */
            combine_strings( temp, utils7a, fname, utils7b );
         else
            /*
             * cannot write to
             */
            combine_strings( temp, utils8, fname, "'" );
         error( WARNING, prompt_line, temp );
      }
   }
   restore_screen_line( 0, prompt_line, display_buff );
   return( rc );
}


/*
 * Name:    search_and_seize
 * Purpose: search files for a pattern
 * Date:    October 31, 1992
 * Passed:  window:  pointer to current window
 * Notes:   New window replaces old window.  Old window becomes invisible.
 */
int  search_and_seize( TDE_WIN *window )
{
char   name[PATH_MAX];          /* new name for file */
char   temp[PATH_MAX];          /* temporary for long file names */
char   searching[MAX_COLS];     /* buffer for displaying file name */
FTYPE  *ffind;                  /* wildcard filename */
fattr_t fattr;
int    i;
int    update_type;
char   *tokens;
register int rc = ERROR;
register TDE_WIN *win;           /* put window pointer in a register */
int    bottom_line;
DISPLAY_BUFF;

   win = window;
   update_type = win == NULL ? GLOBAL : LOCAL;
   if (update_type == LOCAL) {
      if (!g_status.sas_defined ||  g_status.command == DefineGrep ||
                              g_status.command == DefineRegXGrep) {

         /*
          * prompt for the search pattern and the seize path.
          *   initialize all this stuff.
          */
         if (g_status.command == DefineGrep)
            g_status.sas_search_type = BOYER_MOORE;
         else
            g_status.sas_search_type = REG_EXPRESSION;

         if (g_status.sas_search_type == BOYER_MOORE) {
            *sas_bm.pattern = '\0';
            if (get_name( win16a, win->bottom_line, (char *)sas_bm.pattern )
                         == ERROR)
               return( ERROR );
            if (*sas_bm.pattern == '\0')
               return( ERROR );
         } else {
            *sas_regx.pattern = '\0';
            if (get_name( win16b, win->bottom_line, (char *)sas_regx.pattern )
                         == ERROR)
               return( ERROR );
            if (*sas_regx.pattern == '\0')
               return( ERROR );
            else
               strcpy( (char *)regx.pattern, (char *)sas_regx.pattern );
         }
         *g_status.sas_tokens = '\0';
         if (get_name( win17, win->bottom_line, g_status.sas_tokens ) == ERROR)
            return( ERROR );
         i = 0;
         tokens = strtok( g_status.sas_tokens, SAS_DELIMITERS );
         while (tokens != NULL) {
            g_status.sas_arg_pointers[i++] = tokens;
            tokens = strtok( NULL, SAS_DELIMITERS );
         }
         if (i == 0)
            return( ERROR );
         g_status.sas_arg_pointers[i] = NULL;
         g_status.sas_argc = i;
         g_status.sas_arg = 0;
         g_status.sas_argv = g_status.sas_arg_pointers;
         g_status.sas_found_first = FALSE;
         if (g_status.command == DefineGrep) {
            g_status.sas_defined = TRUE;
            bm.search_defined = sas_bm.search_defined = OK;
            build_boyer_array( );
         } else {
            i = build_nfa( );
            if (i == OK) {
               g_status.sas_defined = TRUE;
               regx.search_defined = sas_regx.search_defined = OK;
            } else
               g_status.sas_defined = FALSE;
         }
      }
      bottom_line = win->bottom_line;
   } else
      bottom_line = g_display.nlines;
   if (g_status.sas_defined && g_status.sas_arg < g_status.sas_argc) {
      if (win != NULL) {
         entab_linebuff( );
         un_copy_line( win->ll, win, TRUE );
      }

      /*
       * while we haven't found a valid file, search thru the command
       * line path.
       * we may have an invalid file name when we finish matching all
       * files according to a pattern.  then, we need to go to the next
       * command line argument if it exists.
       */
      while (rc == ERROR && g_status.sas_arg < g_status.sas_argc) {

         /*
          * if we haven't starting searching for a file, check to see if
          * the file is a valid file name.  if no file is found, then let's
          * see if we can find according to a search pattern.
          */
         if (g_status.sas_found_first == FALSE) {

            assert( strlen( g_status.sas_argv[g_status.sas_arg] )
                        < PATH_MAX );

            strcpy( name, g_status.sas_argv[g_status.sas_arg] );
            if (is_pattern( name )) {
               ffind = my_findfirst( name, &g_status.sas_dta, FALSE );

               /*
                * if we found a file using wildcard characters,
                * set the g_status.sas_found_first flag to true so we can
                * find the next matching file.  we need to save the
                * pathname stem so we know which directory we are working in.
                */
               if (ffind != NULL) {
                  rc = OK;
                  g_status.sas_found_first = TRUE;
                  strcpy( g_status.sas_path, g_status.sas_dta.stem );
                  strcpy( name, g_status.sas_path );
                  strcat( name, ffind->fname );
               } else {
                  ++g_status.sas_arg;
                  if (win != NULL)
                     /*
                      * invalid path or file name
                      */
                     error( WARNING, win->bottom_line, win8 );
               }
            } else {
               rc = get_fattr( name, &fattr );
               ++g_status.sas_arg;
            }
         } else {

            /*
             * we already found one file with wild card characters,
             * find the next matching file.
             */
            ffind = my_findnext( &g_status.sas_dta );
            rc = (ffind == NULL);
            if (rc == OK) {
               strcpy( name, g_status.sas_path );
               strcat( name, ffind->fname );
            } else {
               g_status.sas_found_first = FALSE;
               ++g_status.sas_arg;
            }
         }

         /*
          * if everything is everything so far, set up the file
          * and window structures and bring the file into the editor.
          */
         if (rc == OK) {

            strcpy( searching, win19 );
            strcpy( temp, name );
            if (strlen( temp ) > 50)
               temp[50] = '\0';
            strcat( searching, temp );
            save_screen_line( 0, bottom_line, display_buff );
            set_prompt( searching, bottom_line );
#if defined( __UNIX__ )
            refresh( );
#endif
            rc = attempt_edit_display( name, update_type );
            restore_screen_line( 0, bottom_line, display_buff );

            if (rc == OK) {
               win = g_status.current_window;
               bin_offset_adjust( win, g_status.sas_rline );
               find_adjust( win, g_status.sas_ll, g_status.sas_rline,
                            g_status.sas_rcol );
               make_ruler( win );
               show_ruler( win );
               show_ruler_pointer( win );
               show_window_header( win );
               if (win->vertical)
                  show_vertical_separator( win );
               win->file_info->dirty = LOCAL;
            }
         }

         /*
          * either there are no more matching files or we had an
          * invalid file name, set rc to ERROR and let's look at the
          * next file name or pattern on the command line.
          */
         else
            rc = ERROR;
      }
   }
   if (rc == ERROR &&  g_status.sas_arg >= g_status.sas_argc  && win != NULL)
      /*
       * no more files to load
       */
      error( WARNING, win->bottom_line, win9 );
   return( rc );
}


/*
 * Name:     edit_next_file
 * Purpose:  edit next file on command line.
 * Date:     January 6, 1992
 * Modified: December 29, 1996 by Jason Hood
 * Passed:   window:  pointer to current window
 * Notes:    New window replaces old window.  Old window becomes invisible.
 *
 * jmh 981107: pipe now recognizes binary option.
 */
int  edit_next_file( TDE_WIN *window )
{
char name[PATH_MAX+2];          /* new name for file */
int  i;
FTYPE *ffind;
int  update_type;
register int rc = ERROR;
register TDE_WIN *win;           /* put window pointer in a register */
#if defined( __DOS16__ )
fattr_t fattr;
#endif

   win = window;
   update_type = win == NULL ? GLOBAL : LOCAL;

   /* jmh - added !g_status.sas_defined to prevent loading next file */
   if (!g_status.sas_defined && g_status.arg < g_status.argc) {
      if (win != NULL) {
         entab_linebuff( );
         if (un_copy_line( win->ll, win, TRUE ) == ERROR)
            return( ERROR );
      }

      /*
       * while we haven't found a valid file, search thru the command
       * line path.
       * we may have an invalid file name when we finish matching all
       * files according to a pattern.  then, we need to go to the next
       * command line argument if it exists.
       */
      while (rc == ERROR && g_status.arg < g_status.argc) {

         /*
          * if we haven't started searching for a file, check to see if
          * the file is a valid file name.  if no file is found, then let's
          * see if we can find according to a search pattern.
          */
         if (g_status.found_first == FALSE) {

            assert( strlen( g_status.argv[g_status.arg] ) < PATH_MAX );
            /* jmh - moved this from ed.c */
            i = *g_status.argv[g_status.arg];
            if (i == '-'  ||  i == '+') {
               i = *(g_status.argv[g_status.arg] + 1);
               if (i == 'l' || i == 'L') {
                  if (*(g_status.argv[g_status.arg] + 2) == '\0')
                     g_status.language = NULL;
                  else if (*(short*)(g_status.argv[g_status.arg] + 2) == '-')
                     g_status.language = "";
                  else
                     g_status.language = g_status.argv[g_status.arg] + 2;
               } else if (i == 'b' || i == 'B') {
                  g_status.file_mode = BINARY;
                  if (*(g_status.argv[g_status.arg] + 2) != '\0') {
                     i = atoi( g_status.argv[g_status.arg] + 2 );
                     if (i < 0 || i >= MAX_LINE_LENGTH)
                        i = DEFAULT_BIN_LENGTH;
                     g_status.file_chunk = i;
                  }
               } else {
                  if (i == '-')
                     i = *(g_status.argv[g_status.arg] + 2);
                  if (i >= '0' && i <= '9') {
                     i = atoi( g_status.argv[g_status.arg] + 1 );
                     g_status.jump_to = i;
                  }
               }
               ++g_status.arg;
               continue;
            }
            /*
             * Read the pipe before any other files.
             */
            if (g_status.input_redir) {
               *name = '\0';
               rc = OK;
            } else {
               strcpy( name, g_status.argv[g_status.arg] );
               if (is_pattern( name )) {
                  ffind = my_findfirst( name, &g_status.dta, FALSE );

                  /*
                   * if we found a file using wildcard characters,
                   * set the g_status.found_first flag to true so we can
                   * find the next matching file.  we need to save the
                   * pathname stem so we know which directory we are working in.
                   */
                  if (ffind != NULL) {
                     rc = OK;
                     g_status.found_first = TRUE;
                     strcpy( g_status.path, g_status.dta.stem );
                     strcpy( name, g_status.path );
                     strcat( name, ffind->fname );
                  } else {
                     ++g_status.arg;
                     if (win != NULL)
                        /*
                         * invalid path or file name
                         */
                        error( WARNING, win->bottom_line, win8 );
                  }
               } else {
#if defined( __DOS16__ )
                  rc = get_fattr( name, &fattr );
                  /*
                   * a new or blank (unfound) file generates a return code of 2.
                   * an invalid file (bad name) generates a return code of 3.
                   */
                  if (rc == 2)
                     rc = OK;
#else
                  /*
                   * Open a new or existing file, trusting it to be a valid name
                   */
                  rc = OK;
#endif
                  ++g_status.arg;
               }
            }
         } else {

            /*
             * we already found one file with wild card characters,
             * find the next matching file.
             */
            ffind = my_findnext( &g_status.dta );
            rc = (ffind == NULL);
            if (rc == OK) {
               strcpy( name, g_status.path );
               strcat( name, ffind->fname );
            } else {
               g_status.found_first = FALSE;
               ++g_status.arg;
            }
         }

         /*
          * if everything is everything so far, set up the file
          * and window structures and bring the file into the editor.
          */
         if (rc == OK) {

            assert( strlen( name ) < PATH_MAX );

            rc = attempt_edit_display(name, update_type );

            g_status.file_mode = TEXT;
         }

         /*
          * either there are no more matching files or we had an
          * invalid file name, set rc to ERROR and let's look at the
          * next file name or pattern on the command line.
          */
         else
            rc = ERROR;
      }
   }
   if (rc == ERROR)
      if (g_status.sas_defined)
        /*
         * use <Grep> key to load files (jmh)
         */
         error( WARNING, win->bottom_line, win9a );
      else if (g_status.arg >= g_status.argc)
         if (win == NULL) {
            /*
             * There were options, but no file, so see if we're reading a pipe.
             */
            if (g_status.input_redir)
               rc = attempt_edit_display( "", GLOBAL );
         } else
            /*
             * no more files to load
             */
            error( WARNING, win->bottom_line, win9 );

   return( rc );
}


/*
 * Name:    binary_file
 * Purpose: To determine if a file is binary
 * Date:    November 27, 1998
 * Passed:  name: name of the file to test
 * Notes:   Returns TRUE for a binary file, FALSE for a text file.
 *          Examines the first 256 bytes for "binary" characters. If one is
 *           found, ask for the line length (record size).
 *          If there's an error reading the file, return FALSE and let the
 *           caller handle the problem.
 */
int  binary_file( char *name )
{
FILE *file;
char data[256];
int j;
int rc = FALSE;

   if ((file = fopen( name, "rb" )) != NULL) {
      j = fread( data, sizeof(char), sizeof(data), file );
      for (--j; j >= 0; --j)
         if (!(bj_isalnum( data[j] ) || bj_ispunct( data[j] ) ||
               bj_isspace( data[j] ))) {
            rc = TRUE;
            break;
         }
      fclose( file );
   }

   if (rc == TRUE) {
      /*
       * BINARY line length (0 for text) :
       */
      my_ltoa( g_status.file_chunk, data, 10 );
      get_name( ed18, (g_status.current_window == NULL) ? g_display.nlines :
                       g_status.current_window->bottom_line, data );
      j = atoi( data );
      if (j < 0 || j >= MAX_LINE_LENGTH)
         j = DEFAULT_BIN_LENGTH;
      g_status.file_chunk = j;
   }

   return( rc );
}
