/*
 * All video and keyboard functions were gathered into one file.
 *
 * In version 3.2, this file was split into 3 sections:  1) unix curses
 * made to look and feel like a PC, 2) our PC hardware specific stuff,
 * and 3) hardware independent stuff.
 *
 * This file contains the keyboard and video i/o stuff.  Most of this stuff
 * is specific to the PC hardware, but it should be easily modified when
 * porting to other platforms.  With a few key functions written in
 * assembly, we have fairly fast video performance and good keyboard control.
 * Incidentally, the commented C code in the functions perform the same
 * function as the assembly code.  In earlier versions of TDE, I didn't
 * update the commented C code when I changed the assembly code.  The
 * C and assembly should be equivalent in version 2.2.
 *
 * Although using a curses type package would be more portable, curses
 * can be slow on PCs.  Let's keep our video and keyboard routines in
 * assembly.  I feel the need for speed.
 *
 * Being that TDE 2.2 keeps an accurate count of characters in each line, we
 * can allow the user to enter any ASCII or Extended ASCII key.
 *
 * Determining the video adapter type on the PC requires a function.  In
 *  TDE, that function is:
 *
 *                void video_config( struct vcfg *cfg )
 *
 * video_config( ) is based on Appendix C in _Programmer's Guide to
 *  PC & PS/2 Video Systems_ by Richard Wilton.
 *
 * See:
 *
 *   Richard Wilton, _Programmer's Guide to PC & PS/2 Video Systems_,
 *    Microsoft Press, Redmond, Washington, 1987, Appendix C, pp 511-521.
 *    ISBN 1-55615-103-9.
 *
 *
 * New editor name:  TDE, the Thomson-Davis Editor.
 * Author:           Frank Davis
 * Date:             June 5, 1991, version 1.0
 * Date:             July 29, 1991, version 1.1
 * Date:             October 5, 1991, version 1.2
 * Date:             January 20, 1992, version 1.3
 * Date:             February 17, 1992, version 1.4
 * Date:             April 1, 1992, version 1.5
 * Date:             June 5, 1992, version 2.0
 * Date:             October 31, 1992, version 2.1
 * Date:             April 1, 1993, version 2.2
 * Date:             June 5, 1993, version 3.0
 * Date:             August 29, 1993 version 3.1
 * Date:             November 13, version 3.2
 * Date:             June 5, 1994, version 4.0
 * Date:             December 5, 1998, version 5.0 (jmh)
 *
 * This code is released into the public domain, Frank Davis.
 * You may distribute it freely.
 */


#include "tdestr.h"
#include "common.h"
#include "define.h"
#include "tdefunc.h"


/*
 * Name:    video_config
 * Purpose: simulate a call the BIOS keyboard status subfunction
 * Date:    November 13, 1993
 * Passed:  cfg:  not used
 * Notes:   curses has three (?) cursor shapes:  invisible, normal, large.
 */
void video_config( struct vcfg *cfg )
{
/*
   g_display.insert_cursor =
                 mode.cursor_size == SMALL_INS ? CURSES_SMALL : CURSES_LARGE;
   g_display.overw_cursor =
                 mode.cursor_size == SMALL_INS ? CURSES_LARGE : CURSES_SMALL;
*/

   g_display.insert_cursor = g_display.overw_cursor = CURSES_SMALL;
   if (mode.insert)
      g_display.curses_cursor = g_display.insert_cursor;
   else
      g_display.curses_cursor = g_display.overw_cursor;
}


/*
 * Name:    getkey
 * Purpose: use unix curses to get keyboard input
 * Date:    November 13, 1993
 * Passed:  None
 * Notes:   the getch( ) function is not part of the ANSI C standard.
 *            most (if not all) PC C compilers include getch( ) with the
 *            conio stuff.  in unix, getch( ) is in the curses package.
 *          in TDE, lets try to map the curses function keys to their
 *            "natural" positions on PC hardware.  most of the key mapping
 *            is at the bottom of default.c.
 *          this function is based on the ncurses package in Linux.  It
 *            probably needs to be modified for curses in other unices.
 *
 * jmh 980725: translate keys.
 */
long getkey( void )
{
unsigned key;
long new_key;

go_again:
   key = getch( );

   /*
    * map the Control keys to their natural place in TDE.
    */
   if (key < 32)
      key += 430;
   else if (key > 255) {

      /*
       * map ncurses keys to their natural place in TDE.
       * ncurses defines function keys in the range 256-408.
       */
      if (key > 408)
         key = 256;
      key = curses_to_tde[key - 256];
   }

   new_key = translate_key( key );
   if (new_key == ERROR)
      goto go_again;

   return( new_key );
}


/*
 * Name:    waitkey
 * Purpose: nothing right now
 * Date:    November 13, 1993
 * Passed:  enh_keyboard:  boolean - TRUE if 101 keyboard, FALSE otherwise
 * Returns: 1 if no key ready, 0 if key is waiting
 * Notes:   we might do something with this function later...
 */
int  waitkey( int enh_keyboard )
{
   return( 0 );
}


/*
 * Name:    capslock_active
 * Purpose: To find out if Caps Lock is active
 * Date:    September 1, 1993
 * Author:  Byrial Jensen
 * Returns: Non zero if Caps Lock is active,
 *          Zero if Caps Lock is not active.
 * Note:    How is capslock tested with Linux? - jmh
 */
int  capslock_active( void )
{
   return( 0 );
}


/*
 * Name:    flush_keyboard
 * Purpose: flush keys from the keyboard buffer.  flushinp is a curses func.
 * Date:    November 13, 1993
 * Passed:  none
 */
void flush_keyboard( void )
{
   flushinp( );
}


/*
 * Name:    xygoto
 * Purpose: To move the cursor to the required column and line.
 * Date:    November 13, 1993
 * Passed:  col:    desired column (0 up to max)
 *          line:   desired line (0 up to max)
 * Notes;   on the PC, we use col = -1 and line = -1 to turn the cursor off.
 *            in unix, we try to trap the -1's.
 *          curs_set is a curses function.
 */
void xygoto( int col, int line )
{
   if (col < 0 || line < 0) {
      curs_set( CURSES_INVISBL );
      g_display.curses_cursor = CURSES_INVISBL;
   } else {

      /*
       * if the unix cursor is invisible, lets turn it back on.
       */
      if (g_display.curses_cursor == CURSES_INVISBL) {
/*
         if (mode.insert == TRUE) {
            if (g_display.insert_cursor == SMALL_INS)
               set_cursor_size( CURSES_SMALL );
            else
               set_cursor_size( CURSES_LARGE );
         } else {
            if (g_display.insert_cursor == SMALL_INS)
               set_cursor_size( CURSES_LARGE );
            else
               set_cursor_size( CURSES_SMALL );
         }
*/
         set_cursor_size( CURSES_SMALL );
         g_display.curses_cursor = CURSES_SMALL;
      }
      move( line, col );
   }
}


/*
 * Name:    update_line
 * Purpose: Display the current line in window
 * Date:    November 13, 1993
 * Passed:  window:  pointer to current window
 * Notes:   use the curses mvaddch( ) function to put char + attribute
 *           on the screen.
 *          let's check for control characters before they get to curses.
 *            unix and/or curses is brain-damaged when it comes to
 *            control characters and extended ASCII characters.
 * jmh:     September 12, 1997 - totally replaced the old update_line() code
 *            with the djgpp version, using c_output() instead of _farnspokew()
 *            and other appropriate changes. I hope it works.
 * jmh 980725: added not-eol display (ie. line extends beyond window edge)
 * jmh 980816: display break point in the hilited file color.
 */
void update_line( TDE_WIN *window )
{
line_list_ptr ll;   /* current line being displayed */
text_ptr text;      /* current character of orig begin considered */
int  attr;
int  line;
int  col;
int  bcol;
int  bc;
int  ec;
unsigned char line_attr[MAX_LINE_LENGTH]; /* color of every char. in the line */
unsigned char *normal;                  /* color of every char. on-screen */
int  blank = 0;                         /* color to blank out the screen line */
int  block;
int  max_col;
int  block_line;
int  len;
int  show_eol;
int  show_neol;
long rline;
file_infos *file;

   if (window->rline > window->file_info->length || window->ll->len == EOF
             || !g_status.screen_display)
      return;

   ll       = window->ll;
   file     = window->file_info;
   max_col  = window->end_col + 1 - window->start_col;
   line     = window->cline;
   rline    = window->rline;
   show_eol = (mode.show_eol == 1);
   block    = g_display.block_color;

   /*
    * figure which line to display.
    * actually, we could be displaying any line in any file.  we display
    *  the line_buffer only if window->ll == g_status.buff_node
    *  and window->ll-line has been copied to g_status.line_buff.
    */
   text = ll->line;
   len  = ll->len;
   if (g_status.copied && ptoul( ll ) == ptoul( g_status.buff_node )) {
      text = (text_ptr)g_status.line_buff;
      len  = g_status.line_buff_len;
   }
   if (mode.inflate_tabs)
      text = tabout( text, &len );

   normal = line_attr;
   if (window->syntax)
      blank = syntax_attr( text, len, ll->type, line_attr, file->syntax );

   /*
    * lets look at the base column.  if the line to display is shorter
    *  than the base column, then set text to eol and we can't see the
    *  eol either.
    */
   bc = window->bcol;
   if (bc > 0) {
      if (text == NULL) {
         show_eol = FALSE;
         len = 0;
      } else {
         if (len < bc) {
            bc = len;
            show_eol = FALSE;
         }
         text   += bc;
         len    -= bc;
         normal += bc;
      }
   }
   /*
    * for display purposes, set the line length to screen width if line
    *  is longer than screen.
    */
   show_neol = (mode.show_eol == 2 && len > max_col);
   if (len > max_col)
      len = max_col;

   if (rline == file->break_point) {
      blank = (mode.cursor_cross && g_status.cur_line) ? g_display.cross_color :
              g_display.hilited_file;
      memset( normal, blank, len );
   } else if (window->syntax) {
      if (mode.cursor_cross && g_status.cur_line) {
         blank = g_display.cross_color;
         memset( normal, blank, len );
      }
   } else {
      if (g_status.cur_line)
         blank = (mode.cursor_cross) ? g_display.cross_color :
                                       g_display.curl_color;
      else
         blank = (ll->type & DIRTY) ? g_display.dirty_color :
                                      g_display.text_color;
      memset( normal, blank, len );
   }

   bcol  = window->bcol;
   block_line = (file->block_type > NOTMARKED &&
                 rline >= file->block_br && rline <= file->block_er) ?
                TRUE : FALSE;

   /*
    * do this if 1) a box block is marked, or 2) a stream block begins
    *  and ends on the same line.
    */
   if (block_line == TRUE &&
       (file->block_type == BOX ||
        (file->block_type == STREAM &&
         rline == file->block_br && rline == file->block_er))) {

      /*
       * start with the bc and ec equal to physical block marker.
       */
      bc = file->block_bc;
      ec = file->block_ec;
      if (ec < bcol || bc >= bcol + max_col)
         /*
          * we can't see block if ending column is less than the base col or
          *  the beginning column is greater than max_col.
          */
         ec = bc = max_col + 1;
      else if (ec < bcol + max_col) {
         /*
          * if the ec is less than the max column, make ec relative to
          *  base column then figure the bc.
          */
         ec -= bcol;
         if (bc < bcol)
            bc = 0;
         else
            bc -= bcol;
      } else if (bc < bcol + max_col) {
         /*
          * if the bc is less than the max column, make bc relative to
          *  base column then figure the ec.
          */
         bc -= bcol;
         if (ec > bcol + max_col)
            ec = max_col;
         else
            ec -= bcol;
      } else if (bc < bcol  &&  ec >= bcol + max_col) {
         /*
          * if the block is wider than the screen, make bc start at the
          *  logical begin and make ec end at the logical end of the
          *  window.
          */
         bc = 0;
         ec = max_col;
      }

      bcol = window->start_col;
      for (col = 0; col < len; ++bcol, ++col) {
         attr = (col < bc || col > ec) ? normal[col] : block;
         c_output( *text++, bcol, line, attr );
      }
      if (show_eol && col < max_col) {
         attr = (col < bc || col > ec) ? blank : block;
         c_output( EOL_CHAR, bcol++, line, attr );
         ++col;
      }
      for (; col < max_col; ++bcol, ++col) {
         attr = (col < bc || col > ec) ? blank : block;
         c_output( ' ', bcol, line, attr );
      }

   } else if (block_line == TRUE && file->block_type == STREAM &&
              (rline == file->block_br || rline == file->block_er)) {
      if (rline == file->block_br)
         bc = file->block_bc;
      else {
         bc = file->block_ec + 1;
         ec = blank;
         blank = block;
         block = ec;
      }
      if (bc < bcol)
         bc = 0;
      else if (bc < bcol + max_col)
         bc -= bcol;
      else
         bc = max_col + 1;

      if (bc < len) {
         if (rline == file->block_br)
            memset( normal+bc, block, len - bc );
         else
            memset( normal, blank, bc );
      } else if (rline == file->block_er)
         memset( normal, blank, len );

      bcol = window->start_col;
      for (col = 0; col < len; ++bcol, ++col) {
         c_output( *text++, bcol, line, normal[col] );
      }
      if (show_eol && col < max_col) {
         attr = (col < bc) ? blank : block;
         c_output( EOL_CHAR, bcol++, line, attr );
         ++col;
      }
      for (; col < max_col; ++bcol, ++col) {
         attr = (col < bc) ? blank : block;
         c_output( ' ', bcol, line, attr );
      }

   } else {
      attr = (block_line) ? block : blank;
      if (block_line)
         memset( normal, attr, len );

      assert( len >= 0 );
      assert( len <= g_display.ncols );

      bcol = window->start_col;
      for (col = 0; col < len; ++bcol, ++col) {
         c_output( *text++, bcol, line, normal[col] );
      }
      len = max_col - len;
      if (show_eol && len) {
         c_output( EOL_CHAR, bcol++, line, attr );
         --len;
      }
      for (; len; ++bcol, --len) {
         c_output( ' ', bcol, line, attr );
      }

   }

   /*
    * Display the line extension character.
    */
   if (show_neol)
      c_output( NEOL_CHAR, bcol-1, line, attr );

   /*
    * Display the cursor cross (vertical component).
    * This isn't as nice as the DOS versions and currently
    * doesn't display the block color.
    */
   if (mode.cursor_cross) {
      move( line, window->ccol );
      attr = (!g_status.curl_line) ? g_display.cross_color   :
             (window->syntax)      ? line_attr[window->rcol] :
             (ll->type & DIRTY)    ? g_display.dirty_color   :
                                     g_display.curl_color;
      addch( (inch( ) & A_CHARTEXT) | tde_color_table[attr] );
   }
}


/*
 * Name:    c_output
 * Purpose: Output one character on prompt lines
 * Date:    November 13, 1993
 * Passed:  c:     character to output to screen
 *          col:   col to display character
 *          line:  line number to display character
 *          attr:  attribute of character
 * Returns: none
 */
void c_output( int c, int col, int line, int attr )
{
register unsigned int uc;

   uc = c;

   if (uc < 32)
      mvaddch( line, col, (uc+64) | tde_color_table[g_display.curl_color] );
   else if (uc >= 127)
      mvaddch( line, col, '.' | tde_color_table[g_display.curl_color] );
   else
      mvaddch( line, col, uc | tde_color_table[attr] );
}


/*
 * Name:    s_output
 * Purpose: To output a string
 * Date:    November 13, 1993
 * Passed:  s:     string to output
 *          line:  line to display
 *          col:   column to begin display
 *          attr:  color to display string
 * Notes:   This function is used to output most strings not part of file text.
 */
void s_output( char *s, int line, int col, int attr )
{
int max;
register unsigned int uc;

   max = g_display.ncols;
   if (line == g_display.mode_line  &&  (strlen( s ) + col >= g_display.ncols))
      max--;

   while ((uc = (unsigned int)*s)  &&  col < max) {
      if (uc < 32)
         mvaddch( line, col, (uc+64) | tde_color_table[g_display.curl_color] );
      else if (uc >= 127)
         mvaddch( line, col, '.' | tde_color_table[g_display.curl_color] );
      else
         mvaddch( line, col, uc | tde_color_table[attr] );
      col++;
      s++;
   }
}


/*
 * Name:    eol_clear
 * Purpose: To clear the line from col to max columns
 * Date:    November 13, 1993
 * Passed:  col:   column to begin clear
 *          line:  line to clear
 *          attr:  color to clear
 * Notes:   use unix curses clrtoeol( ) to clear the line.
 */
void eol_clear( int col, int line, int attr )
{
   move( line, col );
   clrtoeol( );
}


/*
 * Name:    window_eol_clear
 * Purpose: To clear the line from start_col to end_col
 * Date:    November 13, 1993
 * Passed:  window:  pointer to window
 *          attr:  color to clear
 * Notes:   don't use unix curses clrtoeol, because we could be in a
 *            vertical window.  lines don't always extend to end of screen
 *            in a vertical window.
 */
void window_eol_clear( TDE_WIN *window, int attr )
{
int max_col;
chtype c;       /* chtype is defined in curses.h */

   if (!g_status.screen_display)
      return;

   assert( attr >= 0 && attr < 128 );

   c = ' ' | tde_color_table[attr];
   max_col = window->start_col;
   while (max_col <= window->end_col)
      mvaddch( window->cline, max_col++, c );
}


/*
 * Name:    hlight_line
 * Date:    November 13, 1993
 * Passed:  x:     column to begin hi lite
 *          y:     line to begin hi lite
 *          lgth:  number of characters to hi lite
 *          attr:  attribute color
 */
void hlight_line( int x, int y, int lgth, int attr )
{
int max;

   max = g_display.ncols;
   if (y == g_display.mode_line)
      max--;

   for (; lgth > 0 && x < max; x++, lgth--) {
      move( y, x );
      addch( (inch( ) & A_CHARTEXT) | tde_color_table[attr] );
   }
   refresh( );
}


/*
 * Name:    cls
 * Purpose: clear screen
 * Date:    November 13, 1993
 * Notes:   Call the curses clear screen function.
 */
void cls( void )
{
   touchwin( curscr );
   clear( );
   refresh( );
}


/*
 * Name:    set_cursor_size
 * Purpose: To set cursor size according to insert mode.
 * Date:    November 13, 1993
 * Passed:  csize:  not used in unix curses
 * Notes:   use the global display structures to set the cursor size
 *          curs_set( ) is a curses function.
 */
void set_cursor_size( int csize )
{
/*
   if (mode.insert == TRUE) {
      if (g_display.insert_cursor == SMALL_INS)
         curs_set( CURSES_SMALL );
      else
         curs_set( CURSES_LARGE );
   } else {
      if (g_display.insert_cursor == SMALL_INS)
         curs_set( CURSES_LARGE );
      else
         curs_set( CURSES_SMALL );
   }
*/
   curs_set( CURSES_SMALL );
   g_display.curses_cursor = CURSES_SMALL;
}


/*
 * Name:    set_overscan_color
 * Purpose: To set overscan color
 * Date:    November 13, 1993
 * Passed:  color:  overscan color
 * Notes:   i'm not sure how to set the overscan in Linux (yet?).
 */
void set_overscan_color( int color )
{
}


/*
 * Name:    save_screen_line
 * Purpose: To save the characters and attributes of a line on screen.
 * Date:    November 13, 1993
 * Passed:  col:    desired column, usually always zero
 *          line:   line on screen to save (0 up to max)
 *          screen_buffer:  buffer for screen contents, must be >= 80 chtype
 * Notes:   Save the contents of the line on screen where prompt is
 *           to be displayed.
 */
void save_screen_line( int col, int line, chtype *screen_buffer )
{
int i;

   for (i=0; col < g_display.ncols; col++, i++)
      screen_buffer[i] = mvinch( line, col );
}


/*
 * Name:    restore_screen_line
 * Purpose: To restore the characters and attributes of a line on screen.
 * Date:    November 13, 1993
 * Passed:  col:    usually always zero
 *          line:   line to restore (0 up to max)
 *          screen_buffer:  buffer for screen contents, must be >= 80 chtype
 * Notes:   Restore the contents of the line on screen where the prompt
 *           was displayed
 */
void restore_screen_line( int col, int line, chtype *screen_buffer )
{
int i;

   for (i=0; col < g_display.ncols; col++, i++)
      mvaddch( line, col, screen_buffer[i] );
   refresh( );
}



/*
 * Name:    save_area
 * Purpose: save text and attribute under the pulled menu
 * Date:    November 13, 1993
 * Passed:  buffer: storage for text and attribute
 *          wid: width of the pulled menu
 *          len: length of pulled menu
 *          row: starting row of pulled menu
 *          col: starting column of pulled menu
 * Notes:   use curses to get char and attr
 */
void save_area( chtype *buffer, int wid, int len, int row, int col )
{
int i;
int j;

   wid--;
   len--;
   for (i=0; len >= 0; len--) {
      for (j=wid; j >= 0; j--, i++)
         buffer[i] = mvinch( row+len, col+j );
   }
}


/*
 * Name:    retore_area
 * Purpose: restore text and attribute under the pulled menu
 * Date:    November 13, 1993
 * Passed:  buffer: storage for text and attribute
 *          wid: width of the pulled menu
 *          len: length of pulled menu
 *          row: starting row of pulled menu
 *          col: starting column of pulled menu
 * Notes:   use curses to set char and attr
 */
void restore_area( chtype *buffer, int wid, int len, int row, int col )
{
int i;
int j;

   wid--;
   len--;
   for (i=0; len >= 0; len--) {
      for (j=wid; j >= 0; j--, i++)
         mvaddch( row+len, col+j, buffer[i] );
   }
   refresh( );
}
