/*
 * Now that I own both MSC 7.0 and BC 3.1 and have linux, lets rearrange stuff
 * so many compilers can compile TDE.  Several implementation specific
 * functions needed for several environments were gathered into this file.
 *
 * In version 3.2, these functions changed to support unix.
 *
 * Incidentally, there is a difference between a NULL line pointer and
 * a pointer to a line that contains no characters.  For example, calling
 *
 *       line = malloc( 0 );
 *
 * will return a valid pointer to an item of 0 length in some compilers
 * and a NULL pointer in other compilers.  malloc( 0 ) will return a valid
 * pointer to an object of zero length in MSC.  malloc( 0 ) will return a
 * NULL pointer in BC.  The problem with returning a NULL pointer for
 * malloc( 0 ) is that it's a little harder to tell if the heap is out of
 * memory or if we have a valid NULL pointer.  On the other hand, the good
 * part about returning a NULL pointer for malloc( 0 ) is that extra space
 * is not wasted for an object of 0 length.  In TDE, we will test for 0
 * before calling my_malloc( ) and set an ERROR code if out of memory.
 *
 *
 * New editor name:  TDE, the Thomson-Davis Editor.
 * Author:           Frank Davis
 * Date:             June 5, 1991, version 1.0
 * Date:             July 29, 1991, version 1.1
 * Date:             October 5, 1991, version 1.2
 * Date:             January 20, 1992, version 1.3
 * Date:             February 17, 1992, version 1.4
 * Date:             April 1, 1992, version 1.5
 * Date:             June 5, 1992, version 2.0
 * Date:             October 31, 1992, version 2.1
 * Date:             April 1, 1993, version 2.2
 * Date:             June 5, 1993, version 3.0
 * Date:             August 29, 1993, version 3.1
 * Date:             November 13, 1993, version 3.2
 * Date:             June 5, 1994, version 4.0
 * Date:             December 5, 1998, version 5.0 (jmh)
 *
 * This code is released into the public domain, Frank Davis.
 * You may use and distribute it freely.
 */

#include "tdestr.h"
#include "common.h"
#include "tdefunc.h"
#include "define.h"


/*
 * Name:    my_malloc
 * Purpose: malloc from the far heap
 * Date:    November 13, 1993
 * Passed:  size:  memory needed heap
 *          rc:   pointer to return code
 * Notes:   set the return code only if an ERROR occured with malloc.
 *           returning a NULL pointer is not neccessarily an ERROR.
 */
void *my_malloc( size_t size, int *rc )
{
void *mem;

   assert( size < MAX_LINE_LENGTH );

   if (size == 0)

      /*
       * if 0 bytes are requested, return NULL
       */
      mem = NULL;
   else {

      mem = malloc( size );

      /*
       * if malloc failed, return NULL and an ERROR.
       */
      if (mem == NULL)
         *rc = ERROR;
   }
   return( mem );
}


/*
 * Name:    my_free
 * Purpose: free memory from the far heap
 * Date:    November 13, 1993
 * Passed:  mem:  pointer to memory to free in far heap
 */
void my_free( void *mem )
{
   assert( mem != NULL );
   free( mem );
}


/*
 * Name:    my_heapavail
 * Purpose: available free memory from the far heap
 * Date:    June 5, 1994
 * Notes:   use /proc/meminfo to find available free memory.  let's count
 *            the free memory in the swap space(s), too.
 */
long my_heapavail( void )
{
FILE *f;                /* file pointer */
char buf[82];           /* let's hope lines in /proc/meminfo are < 80 chars */
char temp[82];          /* space to ascii free mem */
char *p;                /* pointer walks down line in buf */
char *q;                /* pointer for temp */
long mem;               /* free memory */
int  no;                /* current line number in /proc/meminfo */

   if ((f = fopen( "/proc/meminfo", "r" )) == NULL)
      mem = 0;
   else {
      for (no=1,mem=0; !feof( f ); no++) {
         fgets( buf, 80, f );
         if (feof( f ))
            break;
         if (no >= 2) {

            /*
             * assume line looks like:
             *   "mem:   total used free buffers"
             */
            p = buf;
            q = temp;
            while (*p++ == ' ');
            while (*p++ != ' ');
            while (*p++ == ' ');
            while (*p++ != ' ');
            while (*p++ == ' ');
            while (*p++ != ' ');
            while (*p++ == ' ');
            --p;
            while ((*q++ = *p++) && *p != ' ');
            *q = '\0';
            mem += atol( temp );
         }
      }
      fclose( f );
   }
   return( mem );
}


/*
 * Name:    my_memcpy
 * Purpose: copy memory
 * Date:    November 13, 1993
 * Passed:  dest: pointer to destination
 *          src:  pointer to source
 *          size: number of bytes to copy
 */
void my_memcpy( void *dest, void *src, size_t size )
{
   if (size > 0) {
      assert( dest != NULL );
      assert( src  != NULL );
      memcpy( dest, src, size );
   }
}


/*
 * Name:    my_memmove
 * Purpose: move memory
 * Date:    November 13, 1993
 * Passed:  dest: pointer to destination
 *          src:  pointer to source
 *          size: number of bytes to copy
 */
void my_memmove( void *dest, void *src, size_t size )
{
   if (size > 0) {
      assert( dest != NULL );
      assert( src  != NULL );
      memmove( dest, src, size );
   }
}


static FTYPE found;

/*
 * Name:    my_findfirst
 * Purpose: find the first file matching a pattern
 * Date:    August 5, 1997
 * Passed:  path: path and pattern to search for files
 *          dta:  file finding info
 *          dirs: TRUE to search for directories and return file sizes
 * Notes:   Returns NULL for no matching files or bad pattern;
 *          otherwise a pointer to a static FTYPE, with fname holding the
 *          filename that was found.
 *          If dirs is TRUE fsize will hold the size of the file, and the
 *          name will have a trailing slash ('/') if it's a directory.
 *          Only files with read access will be returned.
 */
FTYPE * my_findfirst( char *path, FFIND *dta, int dirs )
{
int i;

   dta->dirs = dirs;
   /*
    * Separate the path and pattern
    */
   i = strlen( path ) - 1;
   if (i == -1) {
      get_current_directory( dta->stem );
      strcpy( dta->pattern, "*" );
   }
   else if (path[i] == '/') {
      strcpy( dta->stem, path );
      strcpy( dta->pattern, "*" );
   } else if (access( path, D_OK ) == OK) {
      strcpy( dta->stem, path );
      strcat( dta->stem, "/" );
      strcpy( dta->pattern, "*" );
   } else {
      for (--i; i >= 0; --i) {
         if (path[i] == '/')
            break;
      }
      if (i >= 0) {
         strncpy( dta->stem, path, ++i );
         dta->stem[i] = '\0';
         strcpy( dta->pattern, path+i );
      } else {
         get_current_directory( dta->stem );
         strcpy( dta->pattern, path );
      }
      if (!is_valid_pattern( dta->pattern, &i ))
         return NULL;
   }
   /*
    * Start scanning the directory
    */
   dta->find_info = opendir( dta->stem );
   if (i == NULL) return NULL;
   return my_findnext( dta );
}


/*
 * Name:    my_findnext
 * Purpose: find the next file matching a pattern
 * Date:    August 5, 1997
 * Passed:  dta: file finding info
 * Notes:   my_findfirst() MUST be called before calling this function.
 *          Returns NULL if no more matching names;
 *          otherwise same as my_findfirst.
 *          Only files with read access will be returned.
 */
FTYPE * my_findnext( FFIND *dta )
{
struct stat fstat;
struct dirent *dir;
char temp[PATH_MAX];
char *name;

   strcpy( temp, dta->stem );
   name = temp + strlen( temp );

   for (;;) {
      dir = readdir( dta->find_info );
      if (dir == NULL) {
         closedir( dta->find_info );
         return NULL;
      }
      strcpy( name, dir->d_name );
      if (access( temp, R_OK ) != OK) continue;
      stat( temp, &fstat );
      if (S_ISDIR( fstat.st_mode ))
         /* ignore the "." entry */
         if (dta->dirs && (dir->d_name[0] != '.' || dir->d_name[1] != '\0')
            break;
         else continue;
      if (S_ISREG( fstat.st_mode ) &&
          wildcard( dta->pattern, temp->d_name ) == TRUE)
         break;
   }
   strcpy( found.fname, dir->d_name );
   if (dta->dirs) {
      found.fsize = fstat.st_size;
      found.fattr = fstat.st_mode;
      if (S_ISDIR( fstat.st_mode ))
         strcat( found.fname, "/" );
   }
   return &found;
}


/*
 * Name:    hw_fattrib
 * Purpose: To determine the current file attributes.
 * Date:    November 13, 1993
 * Passed:  name: name of file to be checked
 * Returns: use the function in the tdeasm file to get the DOS file
 *          attributes.  get_fattr() returns 0 or OK if no error.
 */
int  hw_fattrib( char *name )
{
register int rc;
fattr_t fattr;

   rc = get_fattr( name, &fattr );
   return( rc == OK ? rc : ERROR );
}


/*
 * Name:    change_mode
 * Purpose: To prompt for file access mode.
 * Date:    November 13, 1993
 * Passed:  name:  name of file
 *          line:  line to display message
 * Returns: OK if file could be changed
 *          ERROR otherwise
 * Notes:   function is used to change file attributes for save_as function.
 */
int  change_mode( char *name, int line )
{
fattr_t  fattr;
register int rc;

   rc = get_fattr( name, &fattr );
   if (rc == OK  &&  !(fattr & S_IWUSR)) {
      /*
       * file is write protected. overwrite anyway?
       */
      if (get_yn( main6, line, R_PROMPT | R_ABORT ) != A_YES)
         rc = ERROR;
      else if (set_fattr( name, fattr & S_IWUSR & S_IWGRP & S_IWOTH ) != OK)
         rc = ERROR;
   }
   return( rc );
}


/*
 * Name:    change_fattr
 * Purpose: To change the file attributes
 * Date:    December 31, 1991
 * Passed:  window:  pointer to current window
 */
int  change_fattr( TDE_WIN *window )
{
file_infos   *file;
TDE_WIN      *wp;
int          prompt_line;
register int ok;
fattr_t      fattr;
char         *s;
int          rc;
char         answer[MAX_COLS+2];

   assert( window != NULL );

   prompt_line = window->bottom_line;

   answer[0] = '\0';
   /*
    * enter new file attributes
    */
   rc = OK;
   if ((ok = get_name( utils14, prompt_line, answer )) == OK) {
      if (*answer != '\0') {
         fattr = window->file_info->file_attrib;

         /*
          * turn off rwx for user, group, and other.
          */
         fattr &= ~S_IRWXU;
         fattr &= ~S_IRWXG;
         fattr &= ~S_IRWXO;
         s = answer;

         while (ok = bj_toupper( *(s++) ),ok) {
            switch (ok) {
               case L_UNIX_READ :
                  fattr |= S_IRUSR;
                  fattr |= S_IRGRP;
                  fattr |= S_IROTH;
                  break;
               case L_UNIX_WRITE :
                  fattr |= S_IWUSR;
                  fattr |= S_IWGRP;
                  fattr |= S_IWOTH;
                  break;
               case L_UNIX_EXECUTE :
                  fattr |= S_IXUSR;
                  fattr |= S_IXGRP;
                  fattr |= S_IXOTH;
                  break;
               default :
                  break;
            }
            s++;
         }
         file = window->file_info;
         if (set_fattr( file->file_name, fattr ) == ERROR) {
            /*
             * new file attributes not set
             */
            error( WARNING, prompt_line, utils15 );
            rc = ERROR;
         } else {
            file->file_attrib = fattr;
            for (wp=g_status.window_list; wp!=NULL; wp=wp->next) {
               if (wp->file_info == file && wp->visible)
                  show_window_fname( wp );
            }
         }
      }
   } else
      rc = ERROR;
   return( rc );
}


/*
 * Name:    get_fattr
 * Purpose: To get unix file attributes
 * Date:    November 13, 1993
 * Passed:  fname: ASCIIZ file name.  Null terminated file name
 *          fattr: pointer to file attributes
 * Returns: OK if successful, ERROR if error
 * Notes:   Use stat( ) to get file attributes.
 */
int  get_fattr( char *fname, fattr_t *fattr )
{
int  rc;                /* return code */
struct stat fstat;

   assert( fname != NULL  &&  fattr != NULL);

   rc = OK;
   ceh.flag = OK;
   if (stat( fname, &fstat) != ERROR)
      *fattr = fstat.st_mode;
   else
      rc = ERROR;
   if (ceh.flag == ERROR)
      rc = ERROR;
   return( rc );
}


/*
 * Name:    set_fattr
 * Purpose: To set unix file attributes
 * Date:    November 13, 1993
 * Passed:  fname: ASCIIZ file name.  Null terminated file name
 *          fattr: mode_t file attributes
 * Returns: 0 if successfull, non zero if not
 * Notes:   Use chmod( ) function to set file attributes.  To change a
 *           file mode, the effective user ID of the process must equal the
 *           owner of the file, or be a superuser.
 */
int  set_fattr( char *fname, fattr_t fattr )
{
int  rc;                /* return code */

   assert( fname != NULL );

   ceh.flag = OK;
   rc = OK;
   if (chmod( fname, fattr ) == ERROR)
      rc = ERROR;
   if (ceh.flag == ERROR)
      rc = ERROR;
   return( rc );
}


/*
 * Name:    get_current_directory
 * Purpose: get current directory
 * Date:    November 13, 1993
 * Passed:  path:  pointer to buffer to store path
 *          drive: drive to get current directory (not used)
 * Notes:   path is expected to be at least PATH_MAX long
 *          append a trailing slash if not present
 */
int  get_current_directory( char FAR *path, int drive )
{
register int  rc;
int  i;

   assert( path != NULL );

   rc = OK;
   ceh.flag = OK;
   if (getcwd( path, PATH_MAX ) == NULL)
      rc = ERROR;
   if (ceh.flag == ERROR)
      rc = ERROR;
   if (rc != ERROR) {
      i = strlen( path );
      if (path[i-1] != '/') {
        path[i] = '/';
        path[i+1] = '\0';
      }
   }
   return( rc );
}


/*
 * Name:    set_current_directory
 * Purpose: set current directory
 * Date:    November 13, 1993
 * Passed:  new_path: directory path
 */
int  set_current_directory( char FAR *new_path )
{
register int  rc;

   assert( new_path != NULL );

   rc = OK;
   ceh.flag = OK;
   if( chdir( new_path ) == ERROR)
      rc = ERROR;
   if (ceh.flag == ERROR)
      rc = ERROR;
   return( rc );
}


/*
 * Name:    get_full_path
 * Purpose: retrieve the fully-qualified path name for a file
 * Date:    May 3, 1998
 * Passed:  in_path:  path to be canonicalized
 *          out_path: canonicalized path
 * Notes:   out_path is assumed to be PATH_MAX characters.
 *          I don't know if UNIX/gcc provides a function to do this (djgpp has
 *          _fixpath, but is that DJ or gcc?), so I'll just copy it.
 */
void get_full_path( char *in_path, char *out_path )
{
   strcpy( out_path, in_path );
}
