/***********************************************************************/
/* COMM4.C - Commands P-S                                              */
/* This file contains all commands that can be assigned to function    */
/* keys or typed on the command line.                                  */
/***********************************************************************/
/*
 * THE - The Hessling Editor. A text editor similar to VM/CMS xedit.
 * Copyright (C) 1991-1996 Mark Hessling
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to:
 *
 *    The Free Software Foundation, Inc.
 *    675 Mass Ave,
 *    Cambridge, MA 02139 USA.
 *
 *
 * If you make modifications to this software that you feel increases
 * it usefulness for the rest of the community, please email the
 * changes, enhancements, bug fixes as well as any and all ideas to me.
 * This software is going to be maintained and enhanced as deemed
 * necessary by the community.
 *
 * Mark Hessling                 Email:             M.Hessling@qut.edu.au
 * PO Box 203                    Phone:                    +617 3849 7731
 * Bellara                       http://www.gu.edu.au/gwis/the/markh.html
 * QLD 4507                      **** Maintainer PDCurses & REXX/SQL ****
 * Australia                     ************* Author of THE ************
 */

/*
$Id: comm4.c 2.1 1995/06/24 16:28:56 MH Rel MH $
*/

#include <the.h>
#include <proto.h>

/*#define DEBUG 1*/

/*man-start*********************************************************************
COMMAND
     preserve - save various editor settings

SYNTAX
     PREServe

DESCRIPTION
     The PRESERVE command saves various editing settings at the time
     the command is issued.  These settings can then be restored by
     using the <RESTORE> command.

COMPATIBILITY
     XEDIT: Compatible.
     KEDIT: Compatible.

SEE ALSO
     <RESTORE>

STATUS
     Incomplete.
**man-end**********************************************************************/
#ifdef HAVE_PROTO
short Preserve(CHARTYPE *params)
#else
short Preserve(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm4.c:   Preserve");
#endif
/*---------------------------------------------------------------------*/
/* Split parameters up...                                              */
/*---------------------------------------------------------------------*/
#if 0
 num_params = param_split(params,word,PRT_PARAMS,WORD_DELIMS,TEMP_PARAM,strip);
 if (num_params == 0)
   {
    num_params = 1;
    word[0] = (CHARTYPE *)"1";
   }
#endif
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     print - send text to default printer or print spooler

SYNTAX
     PRint [target] [n]
     PRint LINE [text]
     PRint STRING [text]
     PRint FORMfeed
     PRint CLOSE

DESCRIPTION
     The PRINT command writes a portion of the current file to the default
     printer or print spooler, or text entered on the command line.
  
     PRINT [<'target'>] ['n']
        Sends text from the file contents up to the <'target'> to the printer
        followed by a CR/LF (DOS) or LF(UNIX) after each line.
        When ['n'] is specified, this sends a formfeed after ['n'] successive
        lines of text.
     PRINT 'LINE' ['text']
        Sends the remainder of the 'text' on the command line to the printer
        followed by a CR/LF (DOS) or LF(UNIX).
     PRINT 'STRING' ['text']
        Sends the remainder of the 'text' on the command line to the printer
        without any trailing line terminator.
     PRINT 'FORMfeed'
         Sends a formfeed (^L) character to the printer.
     PRINT 'CLOSE'
         Closes the printer spooler.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Compatible.

SEE ALSO
     <SET PRINTER>

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef HAVE_PROTO
short Print(CHARTYPE *params)
#else
short Print(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
#define PRT_PARAMS  2
 CHARTYPE *word[PRT_PARAMS+1];
 CHARTYPE *strip=(CHARTYPE *)"10";
 unsigned short num_params=0;
 short page_break=0;
 short rc=RC_OK;
 short target_type=TARGET_NORMAL|TARGET_ALL|TARGET_BLOCK_CURRENT|TARGET_SPARE;
 TARGET target;
#if defined(UNIX)
 CHARTYPE *line_term = (CHARTYPE *)"\n";
#else
 CHARTYPE *line_term = (CHARTYPE *)"\n\r";
#endif
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm4.c:   Print");
#endif
/*---------------------------------------------------------------------*/
/* Split parameters up...                                              */
/*---------------------------------------------------------------------*/
 num_params = param_split(params,word,PRT_PARAMS,WORD_DELIMS,TEMP_PARAM,strip);
 if (num_params == 0)
   {
    num_params = 1;
    word[0] = (CHARTYPE *)"1";
   }
/*---------------------------------------------------------------------*/
/* If first argument is LINE...                                        */
/*---------------------------------------------------------------------*/
 if (equal((CHARTYPE *)"line",word[0],4))
   {
    print_line(FALSE,0L,0L,0,(CHARTYPE *)word[1],line_term);
#ifdef TRACE
    trace_return();
#endif
    return(RC_OK);
   }
/*---------------------------------------------------------------------*/
/* If first argument is STRING...                                      */
/*---------------------------------------------------------------------*/
 if (equal((CHARTYPE *)"string",word[0],5))
   {
    print_line(FALSE,0L,0L,0,(CHARTYPE *)word[1],(CHARTYPE *)"");
#ifdef TRACE
    trace_return();
#endif
    return(RC_OK);
   }
/*---------------------------------------------------------------------*/
/* If first argument is FORMFEED...                                    */
/*---------------------------------------------------------------------*/
 if (equal((CHARTYPE *)"formfeed",word[0],4))
   {
    if (num_params > 1)
      {
       display_error(1,word[1],FALSE);
#ifdef TRACE
       trace_return();
#endif
       return(RC_INVALID_OPERAND);
      }
    print_line(FALSE,0L,0L,0,(CHARTYPE *)"",(CHARTYPE *)"\f");
#ifdef TRACE
    trace_return();
#endif
    return(RC_OK);
   }
/*---------------------------------------------------------------------*/
/* If first argument is CLOSE...                                       */
/*---------------------------------------------------------------------*/
 if (equal((CHARTYPE *)"close",word[0],5))
   {
    if (num_params > 1)
      {
       display_error(1,word[1],FALSE);
#ifdef TRACE
       trace_return();
#endif
       return(RC_INVALID_OPERAND);
      }
    print_line(TRUE,0L,0L,0,(CHARTYPE *)"",(CHARTYPE *)"");
#ifdef TRACE
    trace_return();
#endif
    return(RC_OK);
   }
/*---------------------------------------------------------------------*/
/* ...treat all other options as targets...                            */
/*---------------------------------------------------------------------*/
 initialise_target(&target);
 if ((rc = validate_target(params,&target,target_type,get_true_line(),TRUE,TRUE)) != RC_OK)
   {
    free_target(&target);
#ifdef TRACE
    trace_return();
#endif
    return(rc);
   }
 if (target.spare == (-1))
    page_break = 0;
 else
   {
    if (!valid_positive_integer(strtrunc(target.rt[target.spare].string)))
      {
       display_error(4,word[0],FALSE);
#ifdef TRACE
       trace_return();
#endif
       return(RC_INVALID_OPERAND);
      }
    page_break = atoi((DEFCHAR *)strtrunc(target.rt[target.spare].string));
   }
 print_line(FALSE,target.true_line,target.num_lines,page_break,(CHARTYPE *)"",line_term);
 free_target(&target);
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     put - write part of a file to another

SYNTAX
     PUT [target] [filename]

DESCRIPTION
     The PUT command writes a portion of the current file, defined by
     <'target'> to another file, either explicit or temporary.

     When no 'filename' is supplied the temporary file used for <PUT>
     and <GET> commands is overwritten.

     When a 'filename' is supplied the portion of the file written out
     is appended to the specified file.

COMPATIBILITY
     XEDIT: Compatible.
     KEDIT: Compatible.

SEE ALSO
     <PUTD>, <GET>

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef HAVE_PROTO
short Put(CHARTYPE *params)
#else
short Put(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm4.c:   Put");
#endif
 rc = execute_put(params,FALSE);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     putd - write part of a file to another and delete

SYNTAX
     PUTD [target] [filename]

DESCRIPTION
     The PUTD command writes a portion of the current file, defined by
     <'target'> to another file, either explicit or temporary, and then
     deletes the lines written.

     When no 'filename' is supplied the temporary file used for <PUT>
     and <GET> commands is overwritten.

     When a 'filename' is supplied the portion of the file written out
     is appended to the specified file.

COMPATIBILITY
     XEDIT: Compatible.
     KEDIT: Compatible.

SEE ALSO
     <PUT>, <GET>

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef HAVE_PROTO
short Putd(CHARTYPE *params)
#else
short Putd(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm4.c:   Putd");
#endif
 rc = execute_put(params,TRUE);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     qquit - exit from the current file without saving changes

SYNTAX
     QQuit

DESCRIPTION
     The QQUIT command exits the user from the current file, whether
     changes made to the file have been saved or not.

     The previous file in the <ring> then becomes the current file.

     If the current file is the only file in the <ring>, THE terminates.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Compatible.

SEE ALSO
     <QUIT>

STATUS
     Complete
**man-end**********************************************************************/
#ifdef HAVE_PROTO
short Qquit(CHARTYPE *params)
#else
short Qquit(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm4.c:   Qquit");
#endif
/*---------------------------------------------------------------------*/
/* No arguments are allowed; error if any are present.                 */
/*---------------------------------------------------------------------*/
 if (strcmp((DEFCHAR *)params,"") != 0)
   {
    display_error(1,(CHARTYPE *)params,FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 post_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line,(LINE *)NULL);
 free_view_memory(TRUE);
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     query - display various option settings

SYNTAX
     Query item

DESCRIPTION
     The QUERY command displays the various settings for options set
     by THE.

     For a complete list of 'item's that can be extracted, see the section;
     <QUERY, EXTRACT and STATUS>.

COMPATIBILITY
     XEDIT: Compatible functionality, but not all options.
     KEDIT: Compatible functionality, but not all options.

SEE ALSO
     <STATUS>, <MODIFY>

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef HAVE_PROTO
short Query(CHARTYPE *params)
#else
short Query(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern VALUE item_values[18];
 extern CHARTYPE *temp_cmd;
/*--------------------------- local data ------------------------------*/
 register short i=0;
 short itemno=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm4.c:   Query");
#endif
 if ((itemno = find_item(params,QUERY_QUERY)) == (-1))
    {
     display_error(1,params,FALSE);
#ifdef TRACE
     trace_return();
#endif
     return(RC_INVALID_OPERAND);
    }

 itemno = get_item_values(itemno,(CHARTYPE *)"",QUERY_QUERY,0L,NULL,0L);
 if (itemno != EXTRACT_ARG_ERROR
 &&  itemno != EXTRACT_VARIABLES_SET)
   {
    strcpy((DEFCHAR *)temp_cmd,"");
    for (i=0;i<itemno+1;i++)
      {
       strcat((DEFCHAR *)temp_cmd,(DEFCHAR *)item_values[i].value);
       strcat((DEFCHAR *)temp_cmd," ");
      }
    display_error(0,temp_cmd,TRUE);
   }
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     quit - exit from the current file if no changes made

SYNTAX
     QUIT

DESCRIPTION
     The QUIT command exits the user from the current file, provided
     that any changes made to the file have been saved, otherwise an
     error message is displayed.

     The previous file in the <ring> then becomes the current file.

     If the current file is the only file in the <ring>, THE terminates.

COMPATIBILITY
     XEDIT: Does not support return code option.
     KEDIT: Compatible.

SEE ALSO
     <QQUIT>

STATUS
     Complete
**man-end**********************************************************************/
#ifdef HAVE_PROTO
short Quit(CHARTYPE *params)
#else
short Quit(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm4.c:   Quit");
#endif
/*---------------------------------------------------------------------*/
/* No arguments are allowed; error if any are present.                 */
/*---------------------------------------------------------------------*/
 if (strcmp((DEFCHAR *)params,"") != 0)
   {
    display_error(1,(CHARTYPE *)params,FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 post_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line,(LINE *)NULL);
 if (CURRENT_FILE->save_alt > 0)
   {
    display_error(22,(CHARTYPE *)"",FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_FILE_CHANGED);
   }
 free_view_memory(TRUE);
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     readv - read keystrokes and pass to macro

SYNTAX
     READV Cmdline [initial text]
     READV KEY

DESCRIPTION
     The READV command allows a REXX macro to interact with the user 
     by accepting either individual keystrokes ('KEY') or a complete
     line of text ('Cmdline').

     The READV 'Cmdline' can take optional 'initial text' to be
     displayed on the command line.

     The 'macro' obtains the entered information by setting REXX
     variables. These are set as follows.

     'KEY' option:

          readv.0 = 3
          readv.1 = name of key (empty if unknown)
          readv.2 = ASCII value of key (null if not an ASCII code)
          readv.3 = curses key value (or ASCII code if an ASCII code)

     'CMDLINE' option:

          readv.0 = 1
          readv.1 = contents of command line

     While editting the command in READV 'Cmdline', any key redefinitions
     you have made will be in effect.  Therefore you can use your
     "normal" editting keys to edit the line.  THE will allow the 
     following commands to be executed while in READV 'Cmdline':

          <cursor> left, <cursor> right, <cursor> down, <cursor> up,
          <sos firstchar>, <sos endchar>, <sos startendchar>,
          <sos delend>, <sos delchar>, <sos delchar>,
          <sos tabb>, <sos tabf>, <sos tabwordb>, <sos tabwordf>,
          <sos undo>, <sos delword>, <set insertmode>, <text>

     Either of the keys, ENTER, RETURN and NUMENTER will terminate
     READV 'Cmdline', irrespective of what THE commands have been
     assigned.

COMPATIBILITY
     XEDIT: Similar to READ CMDLINE option.
     KEDIT: Compatible.

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef HAVE_PROTO
short Readv(CHARTYPE *params)
#else
short Readv(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern bool in_macro;
 extern bool initial;
 extern bool rexx_support;
 extern bool error_on_screen;
/*--------------------------- local data ------------------------------*/
#define REA_PARAMS  2
 CHARTYPE *word[REA_PARAMS+1];
 CHARTYPE *strip=(CHARTYPE *)"10";
 unsigned short num_params=0;
 short rc=RC_OK,itemno=0,num_values=0;
 unsigned short y=0,x=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm4.c:   Readv");
#endif
 if (!in_macro
 ||  !rexx_support)
   {
    display_error(53,(CHARTYPE *)"",FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_ENVIRON);
   }
 num_params = param_split(params,word,REA_PARAMS,WORD_DELIMS,TEMP_PARAM,strip);
 if (num_params == 0)
   {
    display_error(1,params,FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }

 getyx(CURRENT_WINDOW,y,x);
 (void)THERefresh((CHARTYPE *)"");
#if defined(USE_EXTCURSES)
 getyx(CURRENT_WINDOW,y,x);
 wmove(CURRENT_WINDOW,y,x);
 wrefresh(CURRENT_WINDOW);
#endif
 if (equal((CHARTYPE *)"key",word[0],3))
   {
/*---------------------------------------------------------------------*/
/* Find the item in the list of valid extract options...               */
/*---------------------------------------------------------------------*/
    if ((itemno = find_item((CHARTYPE *)"READV",QUERY_READV)) == (-1))
      {
       display_error(1,params,FALSE);
#ifdef TRACE
       trace_return();
#endif
       return(RC_INVALID_OPERAND);
      }
/*---------------------------------------------------------------------*/
/* Get the current settings for the valid item...                      */
/*---------------------------------------------------------------------*/
    num_values = get_item_values(itemno,NULL,QUERY_READV,0L,NULL,0L);
/*---------------------------------------------------------------------*/
/* If the arguments to the item are invalid, return with an error.     */
/*---------------------------------------------------------------------*/
    if (num_values == EXTRACT_ARG_ERROR)
      {
#ifdef TRACE
       trace_return();
#endif
       return(RC_INVALID_OPERAND);
      }
/*---------------------------------------------------------------------*/
/* If the REXX variables have already been set, don't try to set them. */
/*---------------------------------------------------------------------*/
    if (num_values != EXTRACT_VARIABLES_SET)
       rc = set_extract_variables(itemno);
    if (error_on_screen)
       clear_msgline();
   }
 else
   {
    if (equal((CHARTYPE *)"cmdline",word[0],1))
      {
       rc = readv_cmdline(word[1]);
      }
    else
      {
       display_error(1,word[0],FALSE);
#ifdef TRACE
       trace_return();
#endif
       return(RC_INVALID_OPERAND);
      }
   }
 initial = FALSE;
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     recover - recover changed or deleted lines

SYNTAX
     RECover [n|*]

DESCRIPTION
     The RECOVER command restores the last 'n', or all '*' changed or 
     deleted lines back into the body of the file.

COMPATIBILITY
     XEDIT: Also recovers changes to lines, not just lines deleted.
     KEDIT: Compatible.

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef HAVE_PROTO
short Recover(CHARTYPE *params)
#else
short Recover(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
#define REC_PARAMS  1
 CHARTYPE *strip=(CHARTYPE *)"11";
 CHARTYPE *word[REC_PARAMS+1];
 unsigned short num_params=0;
 short num=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm4.c:   Recover");
#endif
/*---------------------------------------------------------------------*/
/* Validate the parameters that have been supplied. The one and only   */
/* parameter should be a positive integer greater than zero or '*'.    */
/* If no parameter is supplied, 1 is assumed.                          */
/*---------------------------------------------------------------------*/
 num_params = param_split(params,word,REC_PARAMS,WORD_DELIMS,TEMP_PARAM,strip);
 if (num_params == 0)
    {
     num_params = 1;
     word[0] = (CHARTYPE *)"1";
    }
 if (num_params != 1)
    {
     display_error(1,word[1],FALSE);
#ifdef TRACE
     trace_return();
#endif
     return(RC_INVALID_OPERAND);
    }
 if (strcmp((DEFCHAR *)word[0],"*") == 0)
   {
    num = 99;
   }
 else
   {
    if (!valid_positive_integer(word[0]))
       {
        display_error(4,word[0],FALSE);
#ifdef TRACE
        trace_return();
#endif
        return(RC_INVALID_OPERAND);
       }
    num = atoi((DEFCHAR *)word[0]);
   }

 get_from_recovery_list(num);

#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     redraw - redraw the current screen

SYNTAX
     REDRAW

DESCRIPTION
     The REDRAW command redraws the current contents of the screen.
     This is usually used when some outside influence has affected 
     the display.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: N/A

SEE ALSO
     <REFRESH>

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef HAVE_PROTO
short Redraw(CHARTYPE *params)
#else
short Redraw(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm4.c:   Redraw");
#endif
 if (strcmp((DEFCHAR *)params,"") != 0)
   {
    display_error(1,params,FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 erase();
 refresh();
 restore_THE();
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     refresh - refresh the contents of the current screen

SYNTAX
     REFRESH

DESCRIPTION
     The REFRESH command refreshes what is being displayed on the screen.
     This is usually used from within a <macro> to indicate the progress
     of the <macro>.

COMPATIBILITY
     XEDIT: Compatible.
     KEDIT: Compatible.

SEE ALSO
     <REDRAW>

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef HAVE_PROTO
short THERefresh(CHARTYPE *params)
#else
short THERefresh(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern bool in_macro;
 extern CHARTYPE display_screens;
 extern bool error_on_screen;
 extern bool curses_started;
 extern bool horizontal;
 extern WINDOW *divider;
/*--------------------------- local data ------------------------------*/
 bool save_in_macro=in_macro;
 unsigned short y=0,x=0;
 LINETYPE new_focus_line=0L;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm4.c:   THERefresh");
#endif
 if (strcmp((DEFCHAR *)params,"") != 0)
   {
    display_error(1,params,FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 if (!curses_started)
   {
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_ENVIRON);
   }
 getyx(CURRENT_WINDOW,y,x);
 in_macro = FALSE;
 if (display_screens > 1)
   {
    prepare_view(other_screen);
    display_screen(other_screen);
    if (!horizontal)
      {
       touchwin(divider);
       wnoutrefresh(divider);
      }
   }
 show_footing();
 CURRENT_VIEW->current_row = calculate_actual_row(CURRENT_VIEW->current_base,
                                                  CURRENT_VIEW->current_off,
                                                  CURRENT_SCREEN.rows[WINDOW_MAIN],TRUE);
 build_screen(current_screen); 
 if (CURRENT_VIEW->current_window != WINDOW_COMMAND)
   {
    if (curses_started)
       getyx(CURRENT_WINDOW,y,x);
    if (!line_in_view(current_screen,CURRENT_VIEW->focus_line))
      {
       new_focus_line = get_focus_line_in_view(current_screen,CURRENT_VIEW->focus_line,y);
       post_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line,(LINE *)NULL);
       CURRENT_VIEW->focus_line = new_focus_line;
       pre_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line,(LINE *)NULL);
      }
    y = get_row_for_focus_line(current_screen,CURRENT_VIEW->focus_line,CURRENT_VIEW->current_row);
    if (curses_started)
       wmove(CURRENT_WINDOW,y,x);
   }
 display_screen(current_screen);
 if (error_on_screen)
   expose_msgline();
 wmove(CURRENT_WINDOW,y,x);
 wrefresh(CURRENT_WINDOW);
 in_macro = save_in_macro;
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     repeat - repeat the last command

SYNTAX
     REPEat [target]

DESCRIPTION
     The REPEAT command advances the current line and executes the
     last command. It is equivalent to <NEXT> 1 (or <UP> 1) and <=> for
     the specified number of times specified by <'target'>.

COMPATIBILITY
     XEDIT: Compatible.
     KEDIT: Compatible.

STATUS
     Incomplete
**man-end**********************************************************************/
#ifdef HAVE_PROTO
short Repeat(CHARTYPE *params)
#else
short Repeat(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm4.c:   Repeat");
#endif

 display_error(0,(CHARTYPE *)"This command has yet to be implemented",FALSE);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     replace - replace the current line with supplied text

SYNTAX
     Replace [text]

DESCRIPTION
     The REPLACE command replaces the <focus line> with the supplied
     'text'.

COMPATIBILITY
     XEDIT: Compatible.
     KEDIT: Compatible.

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef HAVE_PROTO
short Replace(CHARTYPE *params)
#else
short Replace(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 short len_params=0,rc=RC_OK;
 LINETYPE true_line=0L;
 LINE *curr=NULL;
 SELECTTYPE current_select=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm4.c:   Replace");
#endif

 post_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line,(LINE *)NULL);
 if (CURRENT_VIEW->hex)
   {
    if ((len_params = convert_hex_strings(params)) == (-1))
      {
       display_error(32,(CHARTYPE *)"",FALSE);
#ifdef TRACE
       trace_return();
#endif
       return(RC_INVALID_OPERAND);
      }
   }
 else
   len_params = strlen((DEFCHAR *)params);
 true_line = get_true_line();
 if (TOF(true_line)
 ||  BOF(true_line))
   {
    display_error(38,(CHARTYPE *)"",FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_OUT_OF_MEMORY); /* ?? */
   }
 curr = lll_find(CURRENT_FILE->first_line,CURRENT_FILE->last_line,true_line,CURRENT_FILE->number_lines);
 current_select = curr->select;
 add_to_recovery_list(curr->line,curr->length);
 curr = delete_line(CURRENT_FILE->first_line,curr,DIRECTION_FORWARD);
 curr = curr->prev;
 if ((curr = add_line(CURRENT_FILE->first_line,curr,
                          params,len_params,current_select)) == NULL)
   {
    display_error(30,(CHARTYPE *)"",FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_OUT_OF_MEMORY);
   }
 if ((rc = increment_alt(CURRENT_FILE)) != RC_OK)
   {
#ifdef TRACE
    trace_return();
#endif
    return(rc);
   }
 pre_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line,(LINE *)NULL);

 build_screen(current_screen); 
 display_screen(current_screen);

#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     reset - cancel the marked block or prefix commands or both

SYNTAX
     RESet ALL|Block|Prefix

DESCRIPTION
     The RESET command unmarks any marked <block> or outstanding prefix
     commands or both.

COMPATIBILITY
     XEDIT: Adds Block and All options.
     KEDIT: Compatible.

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef HAVE_PROTO
short Reset(CHARTYPE *params)
#else
short Reset(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*------------------------- external data -----------------------------*/
 extern VIEW_DETAILS *vd_mark;
 extern CHARTYPE *pre_rec;
 extern unsigned short pre_rec_len;
/*--------------------------- local data ------------------------------*/
#define RES_PARAMS  1
 CHARTYPE *word[RES_PARAMS+1];
 CHARTYPE *strip=(CHARTYPE *)"1";
 unsigned short num_params=0;
 PPC *curr_ppc=NULL;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm4.c:   Reset");
#endif
 num_params = param_split(params,word,RES_PARAMS,WORD_DELIMS,TEMP_PARAM,strip);
 if (num_params > 1)
   {
    display_error(1,word[1],FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
/*---------------------------------------------------------------------*/
/* Reset the marked block, if any.                                     */
/*---------------------------------------------------------------------*/
 if (equal((CHARTYPE *)"block",word[0],1)
 ||  equal((CHARTYPE *)"all",word[0],3)
 ||  num_params == 0)
   {
    if (MARK_VIEW != (VIEW_DETAILS *)NULL)
      {
       MARK_VIEW->marked_line = MARK_VIEW->marked_col = FALSE;
       MARK_VIEW = (VIEW_DETAILS *)NULL;
       build_screen(current_screen); 
       display_screen(current_screen);
      }
   }
/*---------------------------------------------------------------------*/
/* Reset the pending prefix commands, if any.                          */
/*---------------------------------------------------------------------*/
 if (equal((CHARTYPE *)"prefix",word[0],1)
 ||  equal((CHARTYPE *)"all",word[0],3)
 ||  num_params == 0)
   {
    curr_ppc = CURRENT_FILE->first_ppc;
    while(curr_ppc != NULL)
       curr_ppc = delete_pending_prefix_command(curr_ppc,CURRENT_FILE,(LINE *)NULL);
    memset(pre_rec,' ',PREFIX_WIDTH+1);
    pre_rec_len = 0;
    build_screen(current_screen); 
    display_screen(current_screen);
   }
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     restore - restore various editor settings

SYNTAX
     REStore

DESCRIPTION
     The RESTORE command restores various editing settings at the time
     the command is issued.  These settings must have been saved with
     the <PRESERVE> command.

COMPATIBILITY
     XEDIT: Compatible.
     KEDIT: Compatible.

SEE ALSO
     <PRESERVE>

STATUS
     Incomplete.
**man-end**********************************************************************/
#ifdef HAVE_PROTO
short Restore(CHARTYPE *params)
#else
short Restore(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm4.c:   Restore");
#endif
/*---------------------------------------------------------------------*/
/* Split parameters up...                                              */
/*---------------------------------------------------------------------*/
#if 0
 num_params = param_split(params,word,PRT_PARAMS,WORD_DELIMS,TEMP_PARAM,strip);
 if (num_params == 0)
   {
    num_params = 1;
    word[0] = (CHARTYPE *)"1";
   }
#endif
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     rgtleft - scroll the screen to the left or right

SYNTAX
     RGTLEFT [n]

DESCRIPTION
     The RGTLEFT command scrolls the screen 'n' columns to the right
     if the value of <vershift> is less than or equal to 0, or if
     the value of <vershift> is greater than 0, the screen is
     scrolled 'n' columns to the left.

     If 'n' is not specified, the screen scrolls by three quarters the
     number of columns displayed.

COMPATIBILITY
     XEDIT: Compatible.
     KEDIT: Compatible.

SEE ALSO
     <LEFT>, <RIGHT>

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef HAVE_PROTO
short Rgtleft(CHARTYPE *params)
#else
short Rgtleft(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
 LINETYPE shift_val=0L;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm4.c:   Rgtleft");
#endif
/*---------------------------------------------------------------------*/
/* Validate only parameter, a positive integer. 3/4 if no argument.    */
/*---------------------------------------------------------------------*/
 if (blank_field(params))
    shift_val = min(CURRENT_SCREEN.cols[WINDOW_MAIN],
                  1 + CURRENT_VIEW->verify_end - CURRENT_VIEW->verify_start)*3/4;
 else
   {
    if (valid_positive_integer(params))
       shift_val = atol((DEFCHAR *)params);
   }
 if (shift_val == (-1))                            /* invalid argument */
   {
    display_error(1,params,FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 if ((LINETYPE)CURRENT_VIEW->verify_col - (LINETYPE)CURRENT_VIEW->verify_start > 0)
    shift_val = -shift_val;
 CURRENT_VIEW->verify_col = max(1,CURRENT_VIEW->verify_col+shift_val);
 build_screen(current_screen);
 display_screen(current_screen);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     right - scroll the screen to the right

SYNTAX
     RIght [n|HALF]

DESCRIPTION
     The RIGHT command scrolls the screen to the right.

     If 'n' is supplied, the screen scrolls by that many columns.

     If 'HALF' is specified the screen is scrolled by half the number
     of columns in the <filearea>.

     If no parameter is supplied, the screen is scrolled by one
     column. 

COMPATIBILITY
     XEDIT: Compatible.
     KEDIT: Compatible.

SEE ALSO
     <LEFT>, <RGTLEFT>

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef HAVE_PROTO
short Right(CHARTYPE *params)
#else
short Right(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
 LINETYPE shift_val=-1L;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm4.c:   Right");
#endif
/*---------------------------------------------------------------------*/
/* Validate only parameter, HALF or positive integer. 1 if no argument.*/
/*---------------------------------------------------------------------*/
 if (equal((CHARTYPE *)"half",params,4))
    shift_val = CURRENT_SCREEN.cols[WINDOW_MAIN]/2;
 if (blank_field(params))
    shift_val = 1L;
 if (shift_val == (-1))              /* argument not HALF or empty ... */
   {
    if (valid_positive_integer(params))
      {
       shift_val = atol((DEFCHAR *)params);
       if (shift_val != 0)
          shift_val = shift_val;
      }
   }
 if (shift_val == (-1))                            /* invalid argument */
   {
    display_error(1,params,FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
/*---------------------------------------------------------------------*/
/* If the argument is 0, restore the original verify columns display.  */
/*---------------------------------------------------------------------*/
 if (shift_val == 0L)
    CURRENT_VIEW->verify_col = CURRENT_VIEW->verify_start;
 else
    CURRENT_VIEW->verify_col = max(1,CURRENT_VIEW->verify_col+shift_val);
#ifdef MSWIN
 Win31HScroll(CURRENT_VIEW->verify_col);
#endif
 build_screen(current_screen);
 display_screen(current_screen);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     save - save changes to current file

SYNTAX
     SAVE [filename]

DESCRIPTION
     The SAVE command writes the current file to disk. If a 'filename' is
     supplied, the current file is saved in that file, unless the file
     already exists which will result in an error message being
     displayed. The 'Alterations' counter on the <idline> is
     reset to zero.

COMPATIBILITY
     XEDIT: Compatible.
     KEDIT: Compatible.

SEE ALSO
     <SSAVE>, <FILE>, <FFILE>

STATUS
     Complete
**man-end**********************************************************************/
#ifdef HAVE_PROTO
short Save(CHARTYPE *params)
#else
short Save(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm4.c:   Save");
#endif
 post_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line,(LINE *)NULL);
 if ((rc = save_file(CURRENT_FILE,params,FALSE,CURRENT_FILE->number_lines,1L,FALSE,0,max_line_length,TRUE,FALSE)) != RC_OK)
   {
#ifdef TRACE
    trace_return();
#endif
    return(rc);
   }
/*---------------------------------------------------------------------*/
/* Only set the alteration count to zero if save was successful.       */
/*---------------------------------------------------------------------*/
 CURRENT_FILE->autosave_alt = CURRENT_FILE->save_alt = 0;
/*---------------------------------------------------------------------*/
/* If autosave is on at the time of Saving, remove the .aus file...    */
/*---------------------------------------------------------------------*/
 if (CURRENT_FILE->autosave > 0)
    rc = remove_aus_file(CURRENT_FILE);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     schange - selectively change strings

SYNTAX
     SCHange /string1/string2/ [target] [n] [m]

DESCRIPTION
     The SCHANGE command changes one string of text to another only
     after confirming each individual change with the user.

     The first parameter to the change command is the old and new
     string values, seperated by delimiters.
     The allowable delimiters are '/' '\' and '@'.

     The second parameter is the <target>; how many lines are to be
     searched for occurrences of 'string1' to be changed.

     'n' determines how many occurrences of 'string1' are to be 
     changed to 'string2' on each line.

     'm' determines from which occurrence of 'string1' on the line 
     changes are to commence.

COMPATIBILITY
     XEDIT: Functionally compatible, but syntax different.
     KEDIT: Compatible.

DEFAULT
     1 1 1

SEE ALSO
     <CHANGE>

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef HAVE_PROTO
short Schange(CHARTYPE *params)
#else
short Schange(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm4.c:   Schange");
#endif
 rc = execute_change_command(params,TRUE);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     set - execute various set commands

SYNTAX
     SET set_command [set_command parameter(s) ...]

DESCRIPTION
     The SET command is a front end to existing <SET> commands. It treats
     the first parameter it receives as a command and executes it.

COMPATIBILITY
     XEDIT: Compatible.
     KEDIT: Compatible.

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef HAVE_PROTO
short Set(CHARTYPE *params)
#else
short Set(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm4.c:   Set");
#endif
 rc = execute_set_sos_command(TRUE,params);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     shift - move text left or right

SYNTAX
     SHift Left|Right [n] [target]

DESCRIPTION
     The SHIFT command moves text in the direction specified the number
     of columns 'n' for the specified <'target'> lines.

COMPATIBILITY
     XEDIT: Compatible.
     KEDIT: Compatible.

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef HAVE_PROTO
short Shift(CHARTYPE *params)
#else
short Shift(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
#define SHI_PARAMS  3
 CHARTYPE *word[SHI_PARAMS+1];
 CHARTYPE *strip=(CHARTYPE *)"100";
 short shift_left=(-1);
 LINETYPE num_lines=0L,true_line=0L;
 short num_cols=0,num_params=0;
 short rc=RC_OK;
 short target_type=TARGET_NORMAL|TARGET_BLOCK_CURRENT|TARGET_ALL;
 TARGET target;
 bool num_lines_based_on_scope=FALSE;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm4.c:   Shift");
#endif
 num_params = param_split(params,word,SHI_PARAMS,WORD_DELIMS,TEMP_PARAM,strip);
 if (num_params == 0)                                     /* no params */
   {
    display_error(3,(CHARTYPE *)"",FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
/*---------------------------------------------------------------------*/
/* Validate first parameter:                                           */
/*    must be Left or Right                                            */
/*---------------------------------------------------------------------*/
 if (equal((CHARTYPE *)"left",word[0],1))
     shift_left = TRUE;
 if (equal((CHARTYPE *)"right",word[0],1))
     shift_left = FALSE;
 if (shift_left == (-1))
   {
    display_error(1,word[0],FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
/*---------------------------------------------------------------------*/
/* Validate second parameter (if there is one)                         */
/*    If present, must be valid positive integer.                      */
/*    If not present, default to 1.                                    */
/*---------------------------------------------------------------------*/
 if (num_params < 2)
    num_cols = 1;
 else
   {
    if (!valid_positive_integer(strtrunc(word[1])))
      {
       display_error(4,word[1],FALSE);
#ifdef TRACE
       trace_return();
#endif
       return(RC_INVALID_OPERAND);
      }
    num_cols = atoi((DEFCHAR *)word[1]);
   }
/*---------------------------------------------------------------------*/
/* Validate third  parameter (if there is one)                         */
/*    If present, must be valid target.                                */
/*    If not present, default to 1.                                    */
/*---------------------------------------------------------------------*/
 if (num_params < 3)                                      /* no target */
   {
    num_lines = 1L;
    true_line = get_true_line();
   }
 else
   {
    initialise_target(&target);
    if ((rc = validate_target(word[2],&target,target_type,get_true_line(),TRUE,TRUE)) != RC_OK)
      {
       free_target(&target);
#ifdef TRACE
       trace_return();
#endif
       return(rc);
      }
    num_lines = target.num_lines;
    true_line = target.true_line;
    num_lines_based_on_scope = (target.rt[0].target_type == TARGET_BLOCK_CURRENT) ? FALSE : TRUE;
    free_target(&target);
   }
/*---------------------------------------------------------------------*/
/* Now we are here, everything's OK, do the actual shift...            */
/*---------------------------------------------------------------------*/
 rc = execute_shift_command(shift_left,num_cols,true_line,num_lines,num_lines_based_on_scope);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     showkey - display current key value and command assignation

SYNTAX
     SHOWkey [ALL]

DESCRIPTION

     With no parameter, the SHOWKEY command prompts the user to enter 
     a key and responds with the key name and associated command 
     (if applicable).
     To exit from SHOWKEY, press the space bar.

     With 'ALL' specified, a new file is added to the <ring> with all
     default key mappings and any key mappings assigned with the <DEFINE> 
     command shown.  The key mappings are displayed as <DEFINE> commands.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: N/A

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef HAVE_PROTO
short ShowKey(CHARTYPE *params)
#else
short ShowKey(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 int key=0;
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm4.c:   ShowKey");
#endif
/*---------------------------------------------------------------------*/
/* If no arguments, show key definitions as prompted.                  */
/*---------------------------------------------------------------------*/
 if (strcmp((DEFCHAR *)params,"") == 0)
   {
    display_prompt((CHARTYPE *)"Press the key to be translated...spacebar to exit");
    key = 0;
    while(key != ' ')
      {
       while(1)
         {
#ifdef CAN_RESIZE
          if (is_termresized())
            {
             (void)THE_Resize();
             (void)THERefresh((CHARTYPE *)"");
            }
#endif
          key = my_getch(stdscr);
#ifdef CAN_RESIZE
          if (is_termresized())
             continue;
#endif     
          break;
         }
       clear_msgline();
       display_prompt(get_key_definition(key,FALSE,TRUE));
      }
    clear_msgline();
   }
 else
/*---------------------------------------------------------------------*/
/* If an argument, it must be ALL.                                     */
/*---------------------------------------------------------------------*/
   {
    if (equal((CHARTYPE *)"all",params,3))
       rc = display_all_keys();
    else
      {
       display_error(1,(CHARTYPE *)params,FALSE);
       rc = RC_INVALID_OPERAND;
      }
   }
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     sort - sort selected lines in a file

SYNTAX
     SORT target [[sort field 1] [...] [sort field 10]]

DESCRIPTION
     The SORT command sort a portion of a file based on the 'sort field'
     specifications.

     A 'sort field' specification consists of:

          order flag   - [Ascending|Descending]
          left column  - left column of field to sort on
          right column - right column of field to sort on

     The right column MUST be >= left column.

     Only 10 sort fields are allowed.

     <'target'> can be any valid target including ALL, *, -*, and BLOCK.

COMPATIBILITY
     XEDIT: XEDIT only allows ordering flag for all fields
     KEDIT: Compatible.

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef HAVE_PROTO
short Sort(CHARTYPE *params)
#else
short Sort(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm4.c:   Sort");
#endif
 post_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line,(LINE *)NULL);
 rc = execute_sort(params);
 pre_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line,(LINE *)NULL);
 build_screen(current_screen); 
 display_screen(current_screen);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     sos - execute various sos commands

SYNTAX
     SOS sos_command [sos_command ...]

DESCRIPTION
     The SOS command is a front end to existing <SOS> commands. It treats
     each parameter it receives as a command and executes it.

COMPATIBILITY
     XEDIT: XEDIT only permits 1 command
     KEDIT: Compatible.

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef HAVE_PROTO
short Sos(CHARTYPE *params)
#else
short Sos(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 register short i=0;
#define SOS_PARAMS  10
 CHARTYPE *strip=(CHARTYPE *)"1111111111";
 CHARTYPE *word[SOS_PARAMS+1];
 short num_params=0;
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm4.c:   Sos");
#endif
 num_params = param_split(params,word,SOS_PARAMS,WORD_DELIMS,TEMP_PARAM,strip);
 if (num_params == 0)                                     /* no params */
   {
    display_error(3,(CHARTYPE *)"",FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
/*---------------------------------------------------------------------*/
/* For each "command" go an execute it.                                */
/*---------------------------------------------------------------------*/
 for (i=0;i<num_params;i++)
   {
    if ((rc = execute_set_sos_command(FALSE,word[i])) != RC_OK)
       break;
   }
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     split - split a line into two lines

SYNTAX
     SPlit [ALigned]

DESCRIPTION
     The SPLIT command splits the <focus line> into two lines.

     If 'Aligned' is specified, the first non-blank character of the new
     line is positioned under the first non-blank character of the
     <focus line>. 

     If 'Aligned' is not specified, the text of the new line starts in 
     column 1.

COMPATIBILITY
     XEDIT: Compatible.
     KEDIT: Compatible.

SEE ALSO
     <JOIN>, <SPLTJOIN>

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef HAVE_PROTO
short Split(CHARTYPE *params)
#else
short Split(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
 bool aligned=FALSE;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm4.c:   Split");
#endif
/*---------------------------------------------------------------------*/
/* Check here for parameter value of 'Aligned'.                        */
/*---------------------------------------------------------------------*/
 if (equal((CHARTYPE *)"aligned",params,2))
    aligned = TRUE;
 else
    if (strcmp((DEFCHAR *)params,"") == 0)
       aligned = FALSE;
    else
      {
       display_error(1,(CHARTYPE *)params,FALSE);
#ifdef TRACE
       trace_return();
#endif
       return(RC_INVALID_ENVIRON);
      }
 rc = execute_split_join(SPLTJOIN_SPLIT,aligned);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     spltjoin - split/join two lines

SYNTAX
     spltjoin

DESCRIPTION
     The SPLTJOIN command splits the <focus line> into two or joins the
     <focus line> with the next line depending on the position of the
     cursor. 

     If the cursor is after the last column of a line, the <JOIN>
     command is executed, otherwise the <SPLIT> command is executed.

     The text in the new line is aligned with the text in the <focus line>.

     This command can only be used by assigning it to a function key.

COMPATIBILITY
     XEDIT: Compatible.
     KEDIT: Compatible.

SEE ALSO
     <JOIN>, <SPLIT>

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef HAVE_PROTO
short Spltjoin(CHARTYPE *params)
#else
short Spltjoin(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm4.c:   Spltjoin");
#endif
 rc = execute_split_join(SPLTJOIN_SPLTJOIN,TRUE);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     ssave - force SAVE to specified file

SYNTAX
     SSave [filename]

DESCRIPTION
     The SSAVE command writes the current file to disk. If a 'filename' is
     supplied, the current file is saved in that file, otherwise the
     current name of the file is used.

     If a 'filename' is supplied and that 'filename' already exists, 
     the previous contents of that 'filename' will be replaced with the 
     current file.

     The 'Alterations' counter on the <idline> is reset to zero.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Compatible.

SEE ALSO
     <SAVE>, <FILE>, <FFILE>

STATUS
     Complete
**man-end**********************************************************************/
#ifdef HAVE_PROTO
short Ssave(CHARTYPE *params)
#else
short Ssave(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm4.c:   Ssave");
#endif
 post_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line,(LINE *)NULL);
 if ((rc = save_file(CURRENT_FILE,params,TRUE,CURRENT_FILE->number_lines,1L,FALSE,0,max_line_length,TRUE,FALSE)) != RC_OK)
   {
#ifdef TRACE
    trace_return();
#endif
    return(rc);
   }
/*---------------------------------------------------------------------*/
/* Only set the alteration count to zero if save was successful.       */
/*---------------------------------------------------------------------*/
 CURRENT_FILE->autosave_alt = CURRENT_FILE->save_alt = 0;
/*---------------------------------------------------------------------*/
/* If autosave is on at the time of SSaving, remove the .aus file...   */
/*---------------------------------------------------------------------*/
 if (CURRENT_FILE->autosave > 0)
    rc = remove_aus_file(CURRENT_FILE);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     status - display current settings of various variables

SYNTAX
     STATus [filename]

DESCRIPTION
     The STATUS command, without the optional 'filename', displays a full
     screen of current settings for various variables. 

     With the 'filename', the STATUS command creates a file containing a
     series of <SET> commands with the current values of these settings.

COMPATIBILITY
     XEDIT: Compatible.
     KEDIT: Compatible. KEDIT does not support ['filename'] option.

SEE ALSO
     <QUERY>, <EXTRACT>, <MODIFY>

STATUS
     Complete
**man-end**********************************************************************/
#ifdef HAVE_PROTO
short Status(CHARTYPE *params)
#else
short Status(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern bool batch_only;
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
 int key=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm4.c:   Status");
#endif
 if (strcmp((DEFCHAR *)params,"") == 0)
   {
    if (batch_only)
      {
       display_error(24,(CHARTYPE *)"status",FALSE);
       rc = RC_INVALID_ENVIRON;
      }
    else
      {
       rc = show_status();
       while(1)
         {
#ifdef CAN_RESIZE
          if (is_termresized())
            {
             (void)THE_Resize();
             (void)show_status();
            }
#endif
          key = my_getch(stdscr);
#ifdef PDCURSES
          if (key == KEY_MOUSE)
            {
             int b,ba,bm;
             if (get_mouse_info(&b,&ba,&bm) != RC_OK
             ||  (BUTTON_STATUS(b) & BUTTON_ACTION_MASK) == BUTTON_RELEASED)
                continue;
            }
#endif
#ifdef CAN_RESIZE
          if (is_termresized())
             continue;
#endif     
          break;
         }
       THERefresh((CHARTYPE *)"");
       restore_THE();
      }
   }
 else
    rc = save_status(params);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     suspend - suspend THE and return to operating system

SYNTAX
     SUSPend

DESCRIPTION
     The SUSPEND command suspends the current editing session and 
     returns control to the operating system. Under DOS and OS/2 this
     is the equivalent of <OSNOWAIT>. Under UNIX, the process gets placed
     in the background until it is brought to the foreground.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: N/A

SEE ALSO
     <OSNOWAIT>

STATUS
     Complete
**man-end**********************************************************************/
#ifdef HAVE_PROTO
short Suspend(CHARTYPE *params)
#else
short Suspend(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
#if defined(UNIX) && !defined(XCURSES)
 void (*func)();
#endif
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm4.c:   Suspend");
#endif
 if (strcmp((DEFCHAR *)params,"") != 0)
   {
    display_error(2,params,FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
#if defined(UNIX) && !defined(XCURSES)
 if (strcmp("/bin/sh",getenv("SHELL")) == 0)
   {
    display_error(40,(CHARTYPE *)"",FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 suspend_curses();
 func = signal(SIGTSTP,SIG_DFL);
 kill(0,SIGTSTP);
 signal(SIGTSTP,func);
 resume_curses();
 Redraw((CHARTYPE *)"");
#else
 rc = execute_os_command(params,FALSE,FALSE);
#endif
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
