/***********************************************************************/
/* COMMSET1.C - SET commands A-N                                       */
/***********************************************************************/
/*
 * THE - The Hessling Editor. A text editor similar to VM/CMS xedit.
 * Copyright (C) 1991-1996 Mark Hessling
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to:
 *
 *    The Free Software Foundation, Inc.
 *    675 Mass Ave,
 *    Cambridge, MA 02139 USA.
 *
 *
 * If you make modifications to this software that you feel increases
 * it usefulness for the rest of the community, please email the
 * changes, enhancements, bug fixes as well as any and all ideas to me.
 * This software is going to be maintained and enhanced as deemed
 * necessary by the community.
 *
 * Mark Hessling                 Email:             M.Hessling@qut.edu.au
 * PO Box 203                    Phone:                    +617 3849 7731
 * Bellara                       http://www.gu.edu.au/gwis/the/markh.html
 * QLD 4507                      **** Maintainer PDCurses & REXX/SQL ****
 * Australia                     ************* Author of THE ************
 */

/*
$Id: commset1.c 2.1 1995/06/24 16:29:05 MH Rel MH $
*/

#include <the.h>
#include <proto.h>

/*#define DEBUG 1*/
/*man-start*********************************************************************


========================================================================
SET COMMAND REFERENCE
========================================================================
**man-end**********************************************************************/

/*man-start*********************************************************************
COMMAND
     set arbchar - set arbitrary character(s) for targets

SYNTAX
     [SET] ARBchar ON|OFF [char1] [char2]

DESCRIPTION
     Set the character to use as an 'arbitrary character' in string
     targets. The first arbitrary character matches a group of zero
     or more characters, the second will match exactly one character.

COMPATIBILITY
     XEDIT: Compatible.
            Single arbitrary character not supported.
     KEDIT: Compatible.

DEFAULT
     Off $ ?

STATUS
     Incomplete. Multiple arbitrary character matching is not implemented.
**man-end**********************************************************************/
#ifdef HAVE_PROTO
short Arbchar(CHARTYPE *params)
#else
short Arbchar(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*------------------------- external data -----------------------------*/
/*--------------------------- local data ------------------------------*/
#define ARB_PARAMS  4
 CHARTYPE *word[ARB_PARAMS+1];
 CHARTYPE *strip=(CHARTYPE *)"1111";
 unsigned short num_params=0;
 short rc=RC_INVALID_OPERAND;
 bool arbsts=CURRENT_VIEW->arbchar_status;
 CHARTYPE arbchr_single=CURRENT_VIEW->arbchar_single;
 CHARTYPE arbchr_multiple=CURRENT_VIEW->arbchar_multiple;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset1.c:Arbchar");
#endif
/*---------------------------------------------------------------------*/
/* Validate the parameters that have been supplied.                    */
/*---------------------------------------------------------------------*/
 num_params = param_split(params,word,ARB_PARAMS,WORD_DELIMS,TEMP_PARAM,strip);
 switch(num_params)
   {
/*---------------------------------------------------------------------*/
/* No parameters, error.                                               */
/*---------------------------------------------------------------------*/
    case 0: 
         display_error(3,(CHARTYPE *)"",FALSE); 
         break;
/*---------------------------------------------------------------------*/
/* 1 or 2 parameters, validate them...                                 */
/*---------------------------------------------------------------------*/
    case 1: 
         rc = execute_set_on_off(word[0],&arbsts);
         if (rc != RC_OK)
            break;
    case 2: 
    case 3:
         rc = execute_set_on_off(word[0],&arbsts);
         if (rc != RC_OK)
            break;
         rc = RC_INVALID_OPERAND;
/*---------------------------------------------------------------------*/
/* For 2 parameters, check that a single character has been supplied...*/
/*---------------------------------------------------------------------*/
         if (strlen((DEFCHAR *)word[1]) != 1)
           {
            display_error(1,word[1],FALSE);
            break;
           }
         arbchr_multiple = word[1][0];
         rc = RC_OK;
/*---------------------------------------------------------------------*/
/* For 2 parameters, don't check any more.                             */
/*---------------------------------------------------------------------*/
         if (num_params == 2)
            break;
         rc = RC_INVALID_OPERAND;
/*---------------------------------------------------------------------*/
/* For 3 parameters, check that a single character has been supplied...*/
/*---------------------------------------------------------------------*/
         if (strlen((DEFCHAR *)word[2]) != 1)
           {
            display_error(1,word[2],FALSE);
            break;
           }
         arbchr_single = word[2][0];
         rc = RC_OK;
         break;
/*---------------------------------------------------------------------*/
/* Too many parameters...                                              */
/*---------------------------------------------------------------------*/
    default:
         display_error(2,(CHARTYPE *)"",FALSE); 
         break;
   }
/*---------------------------------------------------------------------*/
/* If valid parameters, change the settings...                         */
/*---------------------------------------------------------------------*/
 if (rc == RC_OK)
   {
    CURRENT_VIEW->arbchar_single = arbchr_single;
    CURRENT_VIEW->arbchar_multiple = arbchr_multiple;
    CURRENT_VIEW->arbchar_status = arbsts;
   }
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     set autosave - set autosave period

SYNTAX
     [SET] AUtosave n|OFF

DESCRIPTION
     The SET AUTOSAVE command sets the interval between automatic saves
     of the file, or turns it off altogether.  The interval 'n' refers
     to the number of alterations made to the file.  Hence a value of
     10 for 'n' would result in the file being automatically saved after
     each 10 alterations have been made to the file.

COMPATIBILITY
     XEDIT: Does not support [mode] option.
     KEDIT: Compatible.

DEFAULT
     OFF

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef HAVE_PROTO
short Autosave(CHARTYPE *params)
#else
short Autosave(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*------------------------- external data -----------------------------*/
/*--------------------------- local data ------------------------------*/
#define AUS_PARAMS  1
 CHARTYPE *strip=(CHARTYPE *)"1";
 CHARTYPE *word[AUS_PARAMS+1];
 unsigned short num_params=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset1.c:Autosave");
#endif
 num_params = param_split(params,word,AUS_PARAMS,WORD_DELIMS,TEMP_PARAM,strip);
 if (num_params == 0)
   {
    display_error(3,(CHARTYPE *)"",FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 if (num_params != 1)
   {
    display_error(2,(CHARTYPE *)"",FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 if (equal((CHARTYPE *)"off",word[0],3))
   {
    CURRENT_FILE->autosave = 0;
#ifdef TRACE
    trace_return();
#endif
    return(RC_OK);
   }
 if (!valid_positive_integer(word[0]))
   {
    display_error(4,(CHARTYPE *)word[0],FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 CURRENT_FILE->autosave = (CHARTYPE)atoi((DEFCHAR *)word[0]);
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     set backup - indicate if a backup copy of the file is to be kept

SYNTAX
     [SET] BACKup OFF|TEMP|KEEP|ON

DESCRIPTION
     The SET BACKUP command allows the user to determine if a backup copy
     of the original file is to be kept when the file being edited is
     saved or filed.

     'KEEP' and 'ON' options are the same. 'ON' is
     kept for compatability with previous versions of THE.

     With 'OFF', the file being written to disk will replace an
     existing file. There is a chance that you will end up with neither
     the old version of the file or the new one if problems occur
     while the file is being written.

     With 'TEMP' or 'KEEP' options, the file being written is first 
     renamed to the filename with a .bak extension. The file in memory 
     is then written to disk. If 'TEMP' is in effect, the backup
     file is then deleted.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Compatible.

DEFAULT
     KEEP

SEE ALSO
     <FILE>, <FFILE>, <SAVE>, <SSAVE>

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef HAVE_PROTO
short Backup(CHARTYPE *params)
#else
short Backup(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*------------------------- external data -----------------------------*/
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
 short backup_type=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset1.c:Backup");
#endif
 if (equal((CHARTYPE *)"off",params,3))
    backup_type = BACKUP_OFF;
 if (equal((CHARTYPE *)"on",params,2))
    backup_type = BACKUP_ON;
 if (equal((CHARTYPE *)"keep",params,4))
    backup_type = BACKUP_KEEP;
 if (equal((CHARTYPE *)"temp",params,4))
    backup_type = BACKUP_TEMP;
 if (backup_type == 0)
   {
    display_error(1,params,FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 CURRENT_FILE->backup = backup_type;
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     set beep - turn on or off the audible alarm when displaying errors

SYNTAX
     [SET] BEEP ON|OFF

DESCRIPTION
     The SET BEEP command allows the user to determine if an audible 
     alarm is sounded when an error is displayed.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Compatible.

DEFAULT
     OFF

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef HAVE_PROTO
short BeepSound(CHARTYPE *params)
#else
short BeepSound(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern bool BEEPx;
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset1.c:BeepSound");
#endif
 rc = execute_set_on_off(params,&BEEPx);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     set case - set case sensitivity parameters

SYNTAX
     [SET] CASE Mixed|Lower|Upper [Respect|Ignore] [Respect|Ignore] [Respect|Ignore]

DESCRIPTION
     The CASE command sets the editor's handling of the case of text.

     The first option (which is mandatory) controls how text is entered
     by the user. When 'LOWER' or 'UPPER' are in effect, the shift or caps
     lock keys have no effect on the text being entered. When 'MIXED' is
     in effect, text is entered in the case set by the use of the shift
     and caps lock keys.

     The second option determines how the editor determines if a string
     target matches text in the file when the target is used in a <LOCATE>
     command.  With 'IGNORE' in effect, a match is
     found irrespective of the case of the target or the found text.
     The following strings are treated as equivalent: the THE The ThE...
     With 'RESPECT' in effect, the target and text must be the same case.
     Therefore a target of 'The' only matches text containing 'The', not
     'THE' or 'ThE' etc.

     The third option determines how the editor determines if a string
     target matches text in the file when the target is used in a <CHANGE>
     command.  With 'IGNORE' in effect, a match is
     found irrespective of the case of the target or the found text.
     The following strings are treated as equivalent: the THE The ThE...
     With 'RESPECT' in effect, the target and text must be the same case.
     Therefore a target of 'The' only matches text containing 'The', not
     'THE' or 'ThE' etc.

     The fourth option determines how the editor determines the sort 
     order of upper and lower case with the <SORT> command.
     With 'IGNORE' in effect, upper and lower case letters are treated as
     equivalent.
     With 'RESPECT' in effect, upper and lower case letters are treated as
     different values and uppercase characters will sort before lowercase
     characters.

COMPATIBILITY
     XEDIT: Adds support for case significance in CHANGE commands.
     KEDIT: Adds support for LOWER option.
     Both:  Adds support for case significance in SORT command.

DEFAULT
     Mixed Ignore Respect Respect

STATUS
     Complete
**man-end**********************************************************************/
#ifdef HAVE_PROTO
short Case(CHARTYPE *params)
#else
short Case(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*------------------------- external data -----------------------------*/
/*--------------------------- local data ------------------------------*/
#define CAS_PARAMS  4
 CHARTYPE parm[CAS_PARAMS];
 CHARTYPE *word[CAS_PARAMS+1];
 CHARTYPE *strip=(CHARTYPE *)"1111";
 register short i=0;
 unsigned short num_params=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset1.c:Case");
#endif
 num_params = param_split(params,word,CAS_PARAMS,WORD_DELIMS,TEMP_PARAM,strip);
/*---------------------------------------------------------------------*/
/* Validate the first parameter: must be Mixed, Upper or Lower         */
/*---------------------------------------------------------------------*/
 parm[0] = (CHARTYPE)UNDEFINED_OPERAND;
 if (equal((CHARTYPE *)"mixed",word[0],1))
    parm[0] = CASE_MIXED;
 if (equal((CHARTYPE *)"upper",word[0],1))
    parm[0] = CASE_UPPER;
 if (equal((CHARTYPE *)"lower",word[0],1))
    parm[0] = CASE_LOWER;
 if (parm[0] == (CHARTYPE)UNDEFINED_OPERAND)
    {
     display_error(1,(CHARTYPE *)word[0],FALSE);
#ifdef TRACE
     trace_return();
#endif
     return(RC_INVALID_OPERAND);
    }
/*---------------------------------------------------------------------*/
/* Save the current values of each remaining case setting.             */
/*---------------------------------------------------------------------*/
 parm[1] = CURRENT_VIEW->case_locate;
 parm[2] = CURRENT_VIEW->case_change;
 parm[3] = CURRENT_VIEW->case_sort;
/*---------------------------------------------------------------------*/
/* Validate the remainder of the arguments.                            */
/* Each must be Respect or Ignore, if present.                         */
/*---------------------------------------------------------------------*/
 for (i=1;i<num_params;i++)
   {
    if (strcmp((DEFCHAR *)word[1],"") != 0)
      {
       if (equal((CHARTYPE *)"respect",word[i],1))
          parm[i] = CASE_RESPECT;
       else
          if (equal((CHARTYPE *)"ignore",word[i],1))
             parm[i] = CASE_IGNORE;
          else
            {
             display_error(1,(CHARTYPE *)word[i],FALSE);
#ifdef TRACE
             trace_return();
#endif
             return(RC_INVALID_OPERAND);
            }
      }
   }
/*---------------------------------------------------------------------*/
/* Set the new values of case settings for the view.                   */
/*---------------------------------------------------------------------*/
 CURRENT_VIEW->case_enter  = parm[0];
 CURRENT_VIEW->case_locate = parm[1];
 CURRENT_VIEW->case_change = parm[2];
 CURRENT_VIEW->case_sort   = parm[3];

#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     set clearscreen - indicate if the screen is to be cleared on exit

SYNTAX
     [SET] CLEARScreen ON|OFF

DESCRIPTION
     The SET CLEARSCREEN command allows the user to request that the
     screen be cleared on exit from THE.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: N/A

DEFAULT
     OFF

STATUS
     Complete
**man-end**********************************************************************/
#ifdef HAVE_PROTO
short Clearscreen(CHARTYPE *params)
#else
short Clearscreen(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern bool CLEARSCREENx;
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset1.c:Clearscreen");
#endif
 rc = execute_set_on_off(params,&CLEARSCREENx);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     set clock - turn on or off display of time on status line

SYNTAX
     [SET] CLOCK ON|OFF

DESCRIPTION
     The SET CLOCK command turns on or off the display of the time on the
     <status line>.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Compatible.

DEFAULT
     ON

STATUS
     Complete
**man-end**********************************************************************/
#ifdef HAVE_PROTO
short Clock(CHARTYPE *params)
#else
short Clock(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern bool CLOCKx;
 extern bool curses_started;
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset1.c:Clock");
#endif
 rc = execute_set_on_off(params,&CLOCKx);
 if (rc == RC_OK
 &&  curses_started)
    clear_footing();
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     set cmdarrows - sets the behaviour of the up and down arrow keys

SYNTAX
     [SET] CMDArrows Retrieve|Tab 

DESCRIPTION
     The SET CMDARROWS command determines the action that occurs when the
     up and down arrows keys are hit while on the <command line>.

     'RETRIEVE' will set the up and down arrows to retrieve the last or 
     next command entered on the <command line>.

     'TAB' will set the up and down arrows to move to the last
     or first line respectively of the main window.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: N/A

DEFAULT
     RETREIVE

SEE ALSO
     <CURSOR>, <?>

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef HAVE_PROTO
short Cmdarrows(CHARTYPE *params)
#else
short Cmdarrows(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*------------------------- external data -----------------------------*/
 extern CHARTYPE CMDARROWSTABCMDx;
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset1.c:Cmdarrows");
#endif
/*---------------------------------------------------------------------*/
/* Determine values for first parameter; command line behaviour        */
/*---------------------------------------------------------------------*/
 if (equal((CHARTYPE *)"tab",params,1))
    CMDARROWSTABCMDx = TRUE;
 else
   if (equal((CHARTYPE *)"retrieve",params,1))
      CMDARROWSTABCMDx = FALSE;
   else
     {
      display_error(1,params,FALSE);
      rc = RC_INVALID_OPERAND;
     }
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     set cmdline - sets the position of the command line.

SYNTAX
     [SET] CMDline ON|OFF|Top|Bottom

DESCRIPTION
     The SET CMDLINE command sets the position of the <command line>, 
     either at the top of the screen, the bottom of the screen or off.

COMPATIBILITY
     XEDIT: Compatible.
            CMDLINE ON is equivalent to CMDLINE Bottom
     KEDIT: Compatible. 

DEFAULT
     BOTTOM

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef HAVE_PROTO
short Cmdline(CHARTYPE *params)
#else
short Cmdline(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*------------------------- external data -----------------------------*/
 extern bool curses_started;
/*--------------------------- local data ------------------------------*/
 CHARTYPE cmd_place='?';
 short off=0;
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset1.c:Cmdline");
#endif
 if (equal((CHARTYPE *)"top",params,1))
   {
    cmd_place='T';
    off = 1;
   }
 if (equal((CHARTYPE *)"bottom",params,1)
 ||  equal((CHARTYPE *)"on",params,2))
   {
    cmd_place='B';
    off = (-1);
   }
 if (equal((CHARTYPE *)"off",params,3))
   {
    cmd_place='O';
    off = 0;
   }
 if (cmd_place=='?')
   {
    display_error(1,(CHARTYPE *)params,FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
/*---------------------------------------------------------------------*/
/* If the setting supplied is the same as the current setting, just    */
/* return without doing anything.                                      */
/*---------------------------------------------------------------------*/
 if (cmd_place == CURRENT_VIEW->cmd_line)
   {
#ifdef TRACE
    trace_return();
#endif
    return(RC_OK);
   }
/*---------------------------------------------------------------------*/
/* Now we need to move the windows around.                             */
/*---------------------------------------------------------------------*/
 CURRENT_VIEW->cmd_line = cmd_place;
/*---------------------------------------------------------------------*/
/* Rebuild the windows and display...                                  */
/*---------------------------------------------------------------------*/
 set_screen_defaults();
 if (curses_started)
   {
    if (set_up_windows(current_screen) != RC_OK)
      {
#ifdef TRACE
       trace_return();
#endif
       return(rc);
      }
   }
 if (CURRENT_VIEW->cmd_line == 'O')
    CURRENT_VIEW->current_window = WINDOW_MAIN;
 build_screen(current_screen);
 if (curses_started)
    display_screen(current_screen);

#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     set colour - set colours for display

SYNTAX
     [SET] COLOUR area [modifier[...]] [foreground] [on] [background]
     [SET] COLOR  area [modifier[...]] [foreground] [on] [background]

DESCRIPTION
     The SET COLOUR command changes the colours or display attributes of
     various display areas in THE.

     Valid values for 'area':

          Arrow    - command line prompt
          Block    - marked <block>
          CBlock   - <current line> if in marked <block>
          Cmdline  - <command line>
          CTofeof  - as for TOfeof if the same as <current line>
          CUrline  - the <current line>
          Divider  - dividing line between vertical split screens
          Filearea - area containing file lines
          Idline   - line containing file specific info
          Msgline  - error messages
          Nondisp  - Non-display characters (<SET ETMODE> OFF)
          Pending  - pending commands in <prefix area>
          PRefix   - <prefix area>
          Reserved - default for <reserved line>
          Scale    - line showing <scale line>
          SHadow   - hidden line marker lines
          STatarea - line showing status of editing session
          Tabline  - line showing tab positions
          TOfeof   - <Top-of-File line> and <Bottom-of-File line>

     Valid values for 'foreground' and 'background':

          BLAck
          BLUe
          Brown
          Green
          GRAy
          GREy
          Cyan
          RED
          Magenta
          Pink
          Turquoise
          Yellow
          White

     Valid values for 'modifier':

          NORmal
          BLInk
          BOld
          BRIght
          High
          REVerse
          Underline

     It is an error to attempt to set a colour on a mono display.

COMPATIBILITY
     XEDIT: Functionally compatible. See below.
     KEDIT: Functionally compatible. See below.
     Does not implement all modifiers.

DEFAULT
     Depends on compatibility mode setting and monitor type.

SEE ALSO
     <SET COMPAT>

STATUS  
     Complete.
**man-end**********************************************************************/
#ifdef HAVE_PROTO
short Colour(CHARTYPE *params)
#else
short Colour(params)
CHARTYPE *params;
#endif

/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern WINDOW *foot;
 extern WINDOW *divider;
 extern bool curses_started;
 extern bool horizontal;
 extern CHARTYPE display_screens;
 extern chtype etmode_table[256];
 extern bool   etmode_flag[256];
/*--------------------------- local data ------------------------------*/
 struct areas
 {
  CHARTYPE *area;
  short area_min_len;
  short area_window;
 };
 typedef struct areas AREAS;
 static AREAS valid_areas[ATTR_MAX]=
 {
  {(CHARTYPE *)"filearea",1,WINDOW_MAIN},
  {(CHARTYPE *)"curline",2,WINDOW_MAIN},
  {(CHARTYPE *)"block",1,WINDOW_MAIN},
  {(CHARTYPE *)"cblock",2,WINDOW_MAIN},
  {(CHARTYPE *)"cmdline",1,WINDOW_COMMAND},
  {(CHARTYPE *)"idline",1,WINDOW_IDLINE},
  {(CHARTYPE *)"msgline",1,WINDOW_ERROR},
  {(CHARTYPE *)"arrow",1,WINDOW_ARROW},
  {(CHARTYPE *)"prefix",2,WINDOW_PREFIX},
  {(CHARTYPE *)"pending",1,WINDOW_PREFIX},
  {(CHARTYPE *)"scale",1,WINDOW_MAIN},
  {(CHARTYPE *)"tofeof",2,WINDOW_MAIN},
  {(CHARTYPE *)"ctofeof",2,WINDOW_MAIN},
  {(CHARTYPE *)"tabline",1,WINDOW_MAIN},
  {(CHARTYPE *)"shadow",2,WINDOW_MAIN},
  {(CHARTYPE *)"statarea",2,WINDOW_FOOTING},
  {(CHARTYPE *)"divider",1,WINDOW_DIVIDER},
  {(CHARTYPE *)"reserved",1,WINDOW_RESERVED},
  {(CHARTYPE *)"nondisp",1,WINDOW_MAIN}
 };
#define COL_PARAMS 2
 CHARTYPE *word[COL_PARAMS+1];
 CHARTYPE *strip=(CHARTYPE *)"11";
 CHARTYPE parm[COL_PARAMS];
 register short i=0;
 unsigned short num_params=0;
 short area=0;
 COLOUR_ATTR attr;
 CHARTYPE *dummy=NULL;
 bool any_colours=FALSE;
 chtype ch=0L,nondisp_attr=0L;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset1.c:Colour");
#endif
 num_params = param_split(params,word,COL_PARAMS,WORD_DELIMS,TEMP_PARAM,strip);
 if (num_params < 2 )
    {
     display_error(3,(CHARTYPE *)"",FALSE);
#ifdef TRACE
     trace_return();
#endif
     return(RC_INVALID_OPERAND);
    }
/*---------------------------------------------------------------------*/
/* Check that the supplied area matches one of the values in the area  */
/* array and that the length is at least as long as the minimum.       */
/*---------------------------------------------------------------------*/
 parm[0] = FALSE;
 for (i=0;i<ATTR_MAX;i++)
    {
     if (equal(valid_areas[i].area,word[0],valid_areas[i].area_min_len))
       {
        parm[0] = TRUE;
        area = i;
        break;
       }
    }
 if (parm[0] == FALSE)
    {
     display_error(1,(CHARTYPE *)word[0],FALSE);
#ifdef TRACE
     trace_return();
#endif
     return(RC_INVALID_OPERAND);
    }
 memcpy(&attr,CURRENT_FILE->attr+area,sizeof(COLOUR_ATTR));
/*---------------------------------------------------------------------*/
/* Determine colours and modifiers.                                    */
/*---------------------------------------------------------------------*/
 if (parse_colours(word[1],&attr,&dummy,FALSE,&any_colours) != RC_OK)
   {
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
/*---------------------------------------------------------------------*/
/* Special handling required for NONDISP...                            */
/*---------------------------------------------------------------------*/
 if (equal(valid_areas[area].area,word[0],valid_areas[area].area_min_len))
   {
    nondisp_attr = set_colour(&attr);
    for (i=0;i<256;i++)
      {
       if (etmode_flag[i])
         {
          ch = etmode_table[i] & A_CHARTEXT;
          etmode_table[i] = ch | nondisp_attr;
         }
      }
   }
/*---------------------------------------------------------------------*/
/* Now we have the new colours, save them with the current file...     */
/*---------------------------------------------------------------------*/
 memcpy(CURRENT_FILE->attr+area,&attr,sizeof(COLOUR_ATTR));
/*---------------------------------------------------------------------*/
/* If we haven't started curses (in profile first time) exit now...    */
/*---------------------------------------------------------------------*/
 if (!curses_started)
   {
#ifdef TRACE
    trace_return();
#endif
    return(RC_OK);
   }
/*---------------------------------------------------------------------*/
/* Update the appropriate window with the new colour combination...    */
/*---------------------------------------------------------------------*/
 switch (valid_areas[area].area_window)
   {
    case WINDOW_MAIN:
                        if (area == ATTR_FILEAREA)
                           wattrset(CURRENT_WINDOW_MAIN,set_colour(CURRENT_FILE->attr+area));
                        build_screen(current_screen); 
                        display_screen(current_screen);
                        break;
    case WINDOW_PREFIX:
                        if (CURRENT_WINDOW_PREFIX != NULL)
                          {
                           wattrset(CURRENT_WINDOW_PREFIX,set_colour(CURRENT_FILE->attr+area));
                           build_screen(current_screen);
                           display_screen(current_screen);
                          }
                        break;
    case WINDOW_COMMAND:
                        if (CURRENT_WINDOW_COMMAND != NULL)
                          {
                           wattrset(CURRENT_WINDOW_COMMAND,set_colour(CURRENT_FILE->attr+area));
                           redraw_window(CURRENT_WINDOW_COMMAND);
                           touchwin(CURRENT_WINDOW_COMMAND);
                           wnoutrefresh(CURRENT_WINDOW_COMMAND);
                          }
                        break;
    case WINDOW_ARROW:
                        if (CURRENT_WINDOW_ARROW != NULL)
                          {
                           wattrset(CURRENT_WINDOW_ARROW,set_colour(CURRENT_FILE->attr+area));
                           redraw_window(CURRENT_WINDOW_ARROW);
                           touchwin(CURRENT_WINDOW_ARROW);
                           wnoutrefresh(CURRENT_WINDOW_ARROW);
                          }
                        break;
    case WINDOW_IDLINE:
                        if (CURRENT_WINDOW_IDLINE != NULL)
                          {
                           wattrset(CURRENT_WINDOW_IDLINE,set_colour(CURRENT_FILE->attr+area));
                           redraw_window(CURRENT_WINDOW_IDLINE);
                           touchwin(CURRENT_WINDOW_IDLINE);
                           wnoutrefresh(CURRENT_WINDOW_IDLINE);
                          }
                        break;
    case WINDOW_FOOTING:
                        if (foot != NULL)
                          {
                           wattrset(foot,set_colour(CURRENT_FILE->attr+area));
                           redraw_window(foot);
                           touchwin(foot);
                           wnoutrefresh(foot);
                          }
                        break;
    case WINDOW_DIVIDER:
                        if (divider != (WINDOW *)NULL)
                          {
/*#ifdef A_ALTCHARSET*/
#if 0
                           wattrset(divider,A_ALTCHARSET|set_colour(CURRENT_FILE->attr+area));
#else
                           wattrset(divider,set_colour(CURRENT_FILE->attr+area));
#endif
                           if (display_screens > 1
                           &&  !horizontal)
                             {
/*                              redraw_window(divider);*/
                              draw_divider();
                              touchwin(divider);
                              wnoutrefresh(divider);
                             }
                          }
                        break;
    default:
                        break;
   }
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     set compat - set compatibility mode

SYNTAX
     [SET] COMPat The|Xedit|Kedit

DESCRIPTION
     The SET COMPAT command changes some settings of THE to make it
     more compatible with the behaviour of XEDIT and KEDIT.

     This command is most useful as the first <SET> command in a
     profile file. It will change the default settings of THE to
     initially look like the chosen editor. You can then make any
     additional changes in THE by issuing other <SET> commands.

     It is recommended that this command NOT be executed from the
     command line, particularly if you have 2 files being displayed
     at the same time.  Although the command works, things may look
     and behave strangely :-)

COMPATIBILITY
     XEDIT: N/A
     KEDIT: N/A

DEFAULT
     THE

STATUS  
     Complete.
**man-end**********************************************************************/
#ifdef HAVE_PROTO
short Compat(CHARTYPE *params)
#else
short Compat(params)
CHARTYPE *params;
#endif

/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern WINDOW *foot;
 extern WINDOW *divider;
 extern bool curses_started;
 extern bool horizontal;
 extern short compatible;
 extern CHARTYPE display_screens;
#ifdef FOR_ALL_VIEWS
 extern VIEW_DETAILS *vd_first;
#endif
/*--------------------------- local data ------------------------------*/
 int rc=RC_OK;
 int prey=0,prex=0;
#ifdef FOR_ALL_VIEWS
 VIEW_DETAILS *viewp=NULL;
#endif
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset1.c:Compat");
#endif
/*---------------------------------------------------------------------*/
/* Parse the parameter...                                              */
/*---------------------------------------------------------------------*/
 if (equal((CHARTYPE *)"THE",params,1))
    compatible = COMPAT_THE;
 else
    if (equal((CHARTYPE *)"XEDIT",params,1))
       compatible = COMPAT_XEDIT;
    else
       if (equal((CHARTYPE *)"KEDIT",params,1))
          compatible = COMPAT_KEDIT;
       else
         {
          display_error(1,params,FALSE);
#ifdef TRACE
          trace_return();
#endif
          return(RC_INVALID_OPERAND);
         }
 if (curses_started)
   {
    if (CURRENT_WINDOW_PREFIX != NULL)
       getyx(CURRENT_WINDOW_PREFIX,prey,prex);
   }
 post_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line,(LINE *)NULL);
/*---------------------------------------------------------------------*/
/* Reset common settings to defaults for THE...                        */
/*---------------------------------------------------------------------*/
 set_global_defaults();

#ifdef FOR_ALL_VIEWS
/*---------------------------------------------------------------------*/
/* Change the settings for all views.                                  */
/*---------------------------------------------------------------------*/
 viewp = vd_first;
 while(viewp != NULL)
   {
    set_file_defaults(viewp->file_for_view);
    set_view_defaults(viewp);
    viewp = viewp->next;
   }
#else
 set_file_defaults(CURRENT_FILE);
 set_view_defaults(CURRENT_VIEW);
#endif
/*---------------------------------------------------------------------*/
/* Set different default for the other compatibility modes...          */
/*---------------------------------------------------------------------*/
 switch(compatible)
   {
    case COMPAT_THE:
         rc = set_THE_defaults(prey,prex);
         break;
    case COMPAT_XEDIT:
         rc = set_XEDIT_defaults(prey,prex);
         break;
    case COMPAT_KEDIT:
         rc = set_KEDIT_defaults(prey,prex);
         break;
   }
 if (rc != RC_OK)
   {
#ifdef TRACE
    trace_return();
#endif
    return(rc);
   }
/*---------------------------------------------------------------------*/
/* Determine the size of each window in each screen in case any changes*/
/* in defaults caused some settings to include/exclude some windows... */
/*---------------------------------------------------------------------*/
 set_screen_defaults();
/*---------------------------------------------------------------------*/
/* For the common windows, set their attributes to match the new values*/
/*---------------------------------------------------------------------*/
 if (curses_started
 &&  foot != NULL)
   {
    wattrset(foot,set_colour(CURRENT_FILE->attr+ATTR_STATAREA));
    clear_footing();
   }
/*---------------------------------------------------------------------*/
/* If more than one screen displayed, redisplay the 'other' screen...  */
/*---------------------------------------------------------------------*/
 if (display_screens > 1)
   {
    OTHER_SCREEN.screen_view->current_row = calculate_actual_row(OTHER_SCREEN.screen_view->current_base,
                                                  OTHER_SCREEN.screen_view->current_off,
                                                  OTHER_SCREEN.rows[WINDOW_MAIN],TRUE);
    pre_process_line(OTHER_SCREEN.screen_view,OTHER_SCREEN.screen_view->focus_line,(LINE *)NULL);
    if (OTHER_SCREEN.screen_view->cmd_line == 'O')
       OTHER_SCREEN.screen_view->current_window = WINDOW_MAIN;
    if (curses_started)
      {
       if (set_up_windows(current_screen) != RC_OK)
         {
#ifdef TRACE
          trace_return();
#endif
          return(rc);
         }
       if (!horizontal)
         {
/*#ifdef A_ALTCHARSET*/
#if 0
          wattrset(divider,A_ALTCHARSET|set_colour(OTHER_SCREEN.screen_view->file_for_view->attr+ATTR_DIVIDER));
#else
          wattrset(divider,set_colour(OTHER_SCREEN.screen_view->file_for_view->attr+ATTR_DIVIDER));
#endif
/*          redraw_window(divider);*/
          touchwin(divider);
          wnoutrefresh(divider);
         }
      }
    redraw_screen((current_screen == 0)?1:0);
    build_screen(other_screen);
    display_screen(other_screen);
   }
/*---------------------------------------------------------------------*/
/* Redisplay the current screen...                                     */
/*---------------------------------------------------------------------*/
 CURRENT_VIEW->current_row = calculate_actual_row(CURRENT_VIEW->current_base,
                                                  CURRENT_VIEW->current_off,
                                                  CURRENT_SCREEN.rows[WINDOW_MAIN],TRUE);
 pre_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line,(LINE *)NULL);
 if (CURRENT_VIEW->cmd_line == 'O')
    CURRENT_VIEW->current_window = WINDOW_MAIN;
 if (curses_started)
   {
    if (set_up_windows(current_screen) != RC_OK)
      {
#ifdef TRACE
       trace_return();
#endif
       return(rc);
      }
   }
 redraw_screen(current_screen);
 build_screen(current_screen); 
 display_screen(current_screen);

#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     set curline - set position of current line on screen

SYNTAX
     [SET] CURLine M[+n|-n] | [+|-]n

DESCRIPTION
     The SET CURLINE command sets the position of the <current line> to
     the physical screen line specified by supplied arguments.

     The first form of parameters is:

     M[+n|-n] 
     this sets the <current line> to be relative to the middle of
     the screen. A positive value adds to the middle line number, 
     a negative subtracts from it.
     eg. M+3 on a 24 line screen will be line 15
         M-5 on a 24 line screen will be line 7

     The second form of parameters is:

     [+|-]n
     this sets the <current line> to be relative to the top of the
     screen (if positive or no sign) or relative to the bottom 
     of the screen if negative.
     eg. +3 or 3 will set current line to line 3
         -3 on a 24 line screen will be line 21

     If the resulting line is outside the bounds of the screen
     the position of the current line will become the middle line
     on the screen.

     It is an error to try to position the CURLINE on the same
     line as a line already allocated by one of <SET HEXSHOW>,
     <SET RESERVED>, <SET SCALE> or <SET TABLINE>.

COMPATIBILITY
     XEDIT: Compatible.
     KEDIT: Compatible.

DEFAULT
     M

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef HAVE_PROTO
short Curline(CHARTYPE *params)
#else
short Curline(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
#define CUR_PARAMS  1
 CHARTYPE *word[CUR_PARAMS+1];
 CHARTYPE *strip=(CHARTYPE *)"1";
 short num_params=0;
 short rc=0;
 short base = (short)CURRENT_VIEW->current_base;
 short off = CURRENT_VIEW->current_off;
 short hexshow_row=0,curline_row=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset1.c:Curline");
#endif
 num_params = param_split(params,word,CUR_PARAMS,WORD_DELIMS,TEMP_PARAM,strip);
 if (num_params < 1)
   {
    display_error(3,(CHARTYPE *)"",FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 if (num_params > 1)
   {
    display_error(2,(CHARTYPE *)"",FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
/*---------------------------------------------------------------------*/
/* Parse the parameter...                                              */
/*---------------------------------------------------------------------*/
 rc = execute_set_row_position(params,&base,&off);
 if (rc != RC_OK)
   {
#ifdef TRACE
    trace_return();
#endif
    return(rc);
   }
/*---------------------------------------------------------------------*/
/* If the CURLINE is the same line as HEXSHOW, SCALE, TABLE or has a   */
/* RESERVED line on it, return ERROR.                                  */
/*---------------------------------------------------------------------*/
 curline_row = calculate_actual_row(base,off,CURRENT_SCREEN.rows[WINDOW_MAIN],TRUE);
 if (calculate_actual_row(CURRENT_VIEW->scale_base,
                          CURRENT_VIEW->scale_off,
                          CURRENT_SCREEN.rows[WINDOW_MAIN],TRUE) == curline_row
 && CURRENT_VIEW->scale_on)
   {
    display_error(64,(CHARTYPE *)"- same as SCALE",FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(rc);
   }
 if (calculate_actual_row(CURRENT_VIEW->tab_base,
                          CURRENT_VIEW->tab_off,
                          CURRENT_SCREEN.rows[WINDOW_MAIN],TRUE) == curline_row
 && CURRENT_VIEW->tab_on)
   {
    display_error(64,(CHARTYPE *)"- same as TABLINE",FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(rc);
   }
 hexshow_row = calculate_actual_row(CURRENT_VIEW->hexshow_base,
                                    CURRENT_VIEW->hexshow_off,
                                    CURRENT_SCREEN.rows[WINDOW_MAIN],TRUE);
 if ((hexshow_row == curline_row
    ||  hexshow_row + 1 == curline_row)
 && CURRENT_VIEW->hexshow_on)
   {
    display_error(64,(CHARTYPE *)"- same as HEXSHOW",FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(rc);
   }
 if (find_reserved_line(current_screen,TRUE,curline_row,0,0) != NULL)
   {
    display_error(64,(CHARTYPE *)"- same as RESERVED line",FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(rc);
   }
/*---------------------------------------------------------------------*/
/* If the "real" row for CURLINE is not the same as the generated one, */
/* set the base and offset to reflect the generated row.               */
/*---------------------------------------------------------------------*/
 if (calculate_actual_row(base,off,CURRENT_SCREEN.rows[WINDOW_MAIN],FALSE) != curline_row)
   {
    CURRENT_VIEW->current_base = (CHARTYPE)POSITION_MIDDLE;
    CURRENT_VIEW->current_off = 0;
   }
 else
   {
    CURRENT_VIEW->current_base = (CHARTYPE)base;
    CURRENT_VIEW->current_off = off;
   }
 post_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line,(LINE *)NULL);
 CURRENT_VIEW->current_row = curline_row;
 build_screen(current_screen); 
 display_screen(current_screen);
 
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     set cursorstay - set on or off the behaviour of the cursor on a scroll

SYNTAX
     [SET] CURSORSTay ON|OFF

DESCRIPTION
     The SETCURSORSTAY command allows the user to set the behaviour of
     the cursor when the file is scrolled with a <FORWARD> or <BACKWARD>
     command.

     Before this command was introduced, the position of the cursor
     after the file was scrolled depended on <SET COMPAT>; for
     THE, the cursor moved to the current line, for XEDIT and KEDIT
     modes the cursor stayed on the same screen line.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: N/A

DEFAULT
     ON

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef HAVE_PROTO
short CursorStay(CHARTYPE *params)
#else
short CursorStay(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern bool scroll_cursor_stay;
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset1.c:CursorStay");
#endif
 rc = execute_set_on_off(params,&scroll_cursor_stay);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     set dirinclude - set the file mask for directory command

SYNTAX
     [SET] DIRInclude * 
     [SET] DIRInclude [Normal] [Readonly] [System] [Hidden] [Directory]

DESCRIPTION
     The DIRINCLUDE command sets the file mask for files that will be
     displayed on subsequent DIRECTORY commands. The operand "*" will
     set the mask to all files, the other options will set the
     mask to include those options specified together with "normal"
     files eg.

        DIRINCLUDE R S

     will display readonly and system files together with "normal" files
     the next time the DIRECTORY command is issued.

     The effects of DIRINCLUDE are ignored in the Unix version.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: N/A

DEFAULT
     *

SEE ALSO
     <DIRECTORY>, <LS>

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef HAVE_PROTO
short Dirinclude(CHARTYPE *params)
#else
short Dirinclude(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset1.c:Dirinclude");
#endif
 rc = set_dirtype(params);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     set display - specify which level of lines to display

SYNTAX
     [SET] DISPlay n [m|*]

DESCRIPTION
     The SET DISPLAY command sets the selection level for lines to be
     displayed on the screen.

COMPATIBILITY
     XEDIT: Compatible.
     KEDIT: Compatible.

DEFAULT
     0 0

SEE ALSO
     <SET SCOPE>, <SET SELECT>, <ALL>

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef HAVE_PROTO
short Display(CHARTYPE *params)
#else
short Display(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
#define DIS_PARAMS  2
 CHARTYPE *word[DIS_PARAMS+1];
 CHARTYPE *strip=(CHARTYPE *)"11";
 unsigned short num_params=0;
 short col1=0,col2=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset1.c:Display");
#endif
/*---------------------------------------------------------------------*/
/* Validate the parameters that have been supplied. One only           */
/* parameter MUST be supplied. The first parameter MUST be a positive  */
/* integer. The second can be a positive integer or '*'. If no second  */
/* parameter is supplied, defaults to p1. The second parameter MUST be */
/* >= first parameter. '*' is regarded as the biggest number and is    */
/* literally 255.                                                      */
/*---------------------------------------------------------------------*/
 num_params = param_split(params,word,DIS_PARAMS,WORD_DELIMS,TEMP_PARAM,strip);
 if (num_params < 1)
   {
    display_error(3,(CHARTYPE *)"",FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 if (num_params > 2)
   {
    display_error(2,(CHARTYPE *)"",FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 if (!valid_positive_integer(word[0]))
   {
    display_error(4,word[0],FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 col1 = atoi((DEFCHAR *)word[0]);
 if (strcmp((DEFCHAR *)word[1],"*") == 0)
    col2 = 255;
 else
    if (num_params == 1)      /* no second parameter, default to first */
       col2 = col1;
    else
       if (!valid_positive_integer(word[1]))
         {
          display_error(4,word[1],FALSE);
#ifdef TRACE
          trace_return();
#endif
          return(RC_INVALID_OPERAND);
         }
       else
          col2 = atoi((DEFCHAR *)word[1]);

 if (col2 > 255)
    col2 = 255;
 if (col1 > col2)
   {
    display_error(6,word[0],FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 CURRENT_VIEW->display_low = col1;
 CURRENT_VIEW->display_high = col2;
/*---------------------------------------------------------------------*/
/* If we are on the command line and the result of this statement means*/
/* that the current line is no longer in scope, we need to make the    */
/* current line and possibly the focus line the next line in scope.    */
/*---------------------------------------------------------------------*/
 if (CURRENT_VIEW->current_window == WINDOW_COMMAND)
   {
    CURRENT_VIEW->current_line = find_next_in_scope(CURRENT_VIEW,NULL,get_true_line(),DIRECTION_FORWARD);
    build_screen(current_screen); 
    if (!line_in_view(current_screen,CURRENT_VIEW->focus_line))
      {
       CURRENT_VIEW->focus_line = CURRENT_VIEW->current_line;
       pre_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line,(LINE *)NULL);
      }
   }
 pre_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line,(LINE *)NULL);
 build_screen(current_screen); 
 display_screen(current_screen);

#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     set eolout - set end of line terminating character(s)

SYNTAX
     [SET] EOLout CRLF|LF|CR|NONE

DESCRIPTION
     The EOLOUT command allows the user to specify the combination of
     characters that terminate a line. Lines of text in Unix files are
     usually terminated with a 'LF', DOS file usually end with a 'CR' and
     'LF' combination. Files on the Apple Macintosh are usually terminated
     with a 'CR'.

     The 'NONE' option can be used to specifiy that no end of line
     character is written.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: N/A

DEFAULT
     LF - Unix, 
     CRLF - DOS/OS2
     NONE - if THE started with -u option

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef HAVE_PROTO
short Eolout(CHARTYPE *params)
#else
short Eolout(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern CHARTYPE EOLx;
/*--------------------------- local data ------------------------------*/
 CHARTYPE eolchar=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset1.c:Eolout");
#endif
 if (equal((CHARTYPE *)"lf",params,2))
    eolchar = EOLOUT_LF;
 else
    if (equal((CHARTYPE *)"cr",params,2))
       eolchar = EOLOUT_CR;
    else
      {
       if (equal((CHARTYPE *)"crlf",params,4))
          eolchar = EOLOUT_CRLF;
       else
          if (equal((CHARTYPE *)"none",params,4))
             eolchar = EOLOUT_NONE;
          else
            {
             display_error(1,(CHARTYPE *)params,FALSE);
#ifdef TRACE
             trace_return();
#endif
             return(RC_INVALID_OPERAND);
            }
      }
 EOLx = CURRENT_FILE->eolout = eolchar;
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     set etmode - indicate if extended display mode is possible

SYNTAX
     [SET] ETMODE ON|OFF [character list]

DESCRIPTION
     The SET ETMODE command allows the user to specify which characters
     in a character set are to be displayed as their actual representation.

     Those characters not explicitly specified to be displayed as they are
     represented, will be displayed as the <SET NONDISP> character in the 
     colour specified by <SET COLOUR> NONDISP. Characters below 32, will 
     be displayed with an alphabetic character representing the "control" 
     code. 

     eg.
     character code with a value of 7, will display as "G" in the colour
     specified by <SET COLOUR> NONDISP.

     'ON' with no optional 'character list' will display ALL
     characters as their actual representation.

     'OFF' with no optional 'character list' will display
     characters below 32, as a "control" character; characters greater 
     than 126 will be displayed as the <SET NONDISP> characters.

     The 'character list' is a list of positive numbers between 0 and
     255 (inclusive).  The format of this character list can be either
     a single number; eg. 124, or a range of numbers specified; eg.
     32-126. (The first number must be less than or equal to the second 
     number).

     As an example; ETMODE ON 32-127 160-250  would result in the
     characters with a decimal value between 32 and 127 inclusive
     and 160 and 250 inclusive being displayed as their actual
     representation (depending on the current font), and the
     characters between 0 and 31 inclusive, being displayed as
     an equivalent "control" character; characters between 128 and 
     159 inculsive and 250 to 255 being displayed with the <SET NONDISP>
     character.

     Up to 20 character specifiers (single number or range) can be 
     specified.

COMPATIBILITY
     XEDIT: Similar function but deals with Double-Byte characters
     KEDIT: N/A

DEFAULT
     DOS/OS2
         ON
     X11
         ON 32-255
     UNIX
         OFF

SEE ALSO
     <SET NONDISP>, <SET COLOUR>

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef HAVE_PROTO
short Etmode(CHARTYPE *params)
#else
short Etmode(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern bool NONDISPx;
 extern chtype etmode_table[256];
 extern bool   etmode_flag[256];
 extern CHARTYPE number_of_files;
/*--------------------------- local data ------------------------------*/
#define ETM_PARAMS  21
 CHARTYPE *word[ETM_PARAMS+1];
 CHARTYPE *strip=(CHARTYPE *)"111111111111111111111";
 unsigned short num_params=0;
 register int i=0,j=0;
 short rc=RC_OK;
 bool tmp_mode=FALSE;
 chtype attr=0L;
 COLOUR_ATTR curr_attr;
 bool flags[256];
 int num=0,num1=0;
 CHARTYPE *wptr=NULL,*wptr1=NULL;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset1.c:Etmode");
#endif
 num_params = param_split(params,word,ETM_PARAMS,WORD_DELIMS,TEMP_PARAM,strip);
 if (num_params < 1)
   {
    display_error(3,(CHARTYPE *)"",FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 rc = execute_set_on_off(word[0],&tmp_mode);
 if (rc != RC_OK)
   {
    display_error(1,word[0],FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 if (CURRENT_VIEW == NULL
 ||  CURRENT_FILE == NULL)
    set_up_default_colours((FILE_DETAILS *)NULL,&curr_attr,ATTR_NONDISP);
 else
    memcpy(&curr_attr,CURRENT_FILE->attr+ATTR_NONDISP,sizeof(COLOUR_ATTR));
 attr = set_colour(&curr_attr);
 if (num_params == 1)  /* absolute ON or OFF */
   {
    if (tmp_mode)  /* ETMODE ON */
      {
       for (i=0;i<256;i++)
         {
          etmode_table[i] = i;
          etmode_flag[i] = FALSE;
         }
      }
    else
      {
       for (i=0;i<256;i++)
         {
          if (i < 32)
            {
             etmode_table[i] = ('@' + i) | attr;
             etmode_flag[i] = (attr)?TRUE:FALSE;
            }
          else
             if (i > 126)
               {
                etmode_table[i] = NONDISPx | attr;
                etmode_flag[i] = (attr)?TRUE:FALSE;
               }
             else
               {
                etmode_table[i] = i;
                etmode_flag[i] = FALSE;
               }
         }
      }
    if (number_of_files != 0)
      {
       build_screen(current_screen); 
       display_screen(current_screen);
      }
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 memset(flags,FALSE,sizeof(flags));
 for (i=1;i<num_params;i++)
   {
    if (valid_positive_integer(word[i]))
      {
       num = atoi((DEFCHAR *)word[i]);
       if (num > 255)
         {
          display_error(6,word[i],FALSE);
#ifdef TRACE
          trace_return();
#endif
          return(RC_INVALID_OPERAND);
         }
       flags[num] = TRUE;
       continue;
      }
    num = strzeq(word[i],(CHARTYPE)'-');
    num1 = strzreveq(word[i],(CHARTYPE)'-');
    if (num != num1
    || num == (-1))
      {
       display_error(1,word[i],FALSE);
#ifdef TRACE
       trace_return();
#endif
       return(RC_INVALID_OPERAND);
      }
    wptr = word[i];
    *(wptr+num) = '\0';
    wptr1 = wptr+num+1;
    if (!valid_positive_integer(wptr))
      {
       display_error(1,wptr,FALSE);
#ifdef TRACE
       trace_return();
#endif
       return(RC_INVALID_OPERAND);
      }
    if (!valid_positive_integer(wptr))
      {
       display_error(1,wptr1,FALSE);
#ifdef TRACE
       trace_return();
#endif
       return(RC_INVALID_OPERAND);
      }
    num = atoi((DEFCHAR *)wptr);
    num1 = atoi((DEFCHAR *)wptr1);
    if (num > num1)
      {
       display_error(1,word[i],FALSE);
#ifdef TRACE
       trace_return();
#endif
       return(RC_INVALID_OPERAND);
      }
    if (num > 255)
      {
       display_error(6,wptr,FALSE);
#ifdef TRACE
       trace_return();
#endif
       return(RC_INVALID_OPERAND);
      }
    if (num1 > 255)
      {
       display_error(6,wptr1,FALSE);
#ifdef TRACE
       trace_return();
#endif
       return(RC_INVALID_OPERAND);
      }
    for (j=num;j<num1+1;j++)
      {
       flags[j] = TRUE;
      }
   }
 for (i=0;i<256;i++)
   {
    if (flags[i])
      {
       etmode_table[i] = i;
       etmode_flag[i] = FALSE;
      }
    else
      {
       if (i < 32)
         {
          etmode_table[i] = ('@' + i) | attr;
          etmode_flag[i] = TRUE;
         }
       else
         {
          etmode_table[i] = NONDISPx | attr;
          etmode_flag[i] = TRUE;
         }
      }
   }
 if (number_of_files != 0)
   {
    build_screen(current_screen); 
    display_screen(current_screen);
   }
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     set hex - set how hexidecimal strings are treated in string operands

SYNTAX
     [SET] HEX ON|OFF

DESCRIPTION
     The SET HEX set command determines whether hexidecimal strings are
     treated as such in string operands.

     With the 'ON' option, any string operand of the form
        /x'31 32 33'/ or
        /d'49 50 51'/
     will be converted to /123/ before the command is executed.

     With the 'OFF' option, no conversion is done.

     This conversion should work wherever a string operand is used
     in any command.

COMPATIBILITY
     XEDIT: Adds support for decimal representation. See below.
     KEDIT: Compatible. See below.
     Spaces must seperate each character representation.

DEFAULT
     OFF

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef HAVE_PROTO
short Hex(CHARTYPE *params)
#else
short Hex(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset1.c:Hex");
#endif
 rc = execute_set_on_off(params,&CURRENT_VIEW->hex);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     set hexdisplay - turn on or off display of character under cursor

SYNTAX
     [SET] HEXDISPlay ON|OFF

DESCRIPTION
     The SET HEXDISPLAY command turns on or off the display of the 
     character under the cursor on the <status line>.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Compatible.

DEFAULT
     ON

STATUS
     Complete
**man-end**********************************************************************/
#ifdef HAVE_PROTO
short Hexdisplay(CHARTYPE *params)
#else
short Hexdisplay(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern bool HEXDISPLAYx;
 extern bool curses_started;
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset1.c:Hexdisplay");
#endif
 rc = execute_set_on_off(params,&HEXDISPLAYx);
 if (rc == RC_OK
 &&  curses_started)
    clear_footing();
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     set hexshow - turn on or off hex display of current line

SYNTAX
     [SET] HEXShow ON|OFF [M[+n|-n]|[+|-]n]

DESCRIPTION
     The SET HEXShow command indicates if and where a hexidecimal
     representation of the <current line> will be displayed.

     The first form of parameters is:

     M[+n|-n] 
     this sets the hexshow line to be relative to the middle of
     the screen. A positive value adds to the middle line number, 
     a negative subtracts from it.
     eg. M+3 on a 24 line screen will be line 15
         M-5 on a 24 line screen will be line 7

     The second form of parameters is:

     [+|-]n
     this sets the hexshow line to be relative to the top of the
     screen (if positive or no sign) or relative to the bottom 
     of the screen if negative.
     eg. +3 or 3 will set current line to line 3
         -3 on a 24 line screen will be line 21

     If the resulting line is outside the bounds of the screen
     the position of the hexshow line will become the middle line
     on the screen.

     The position argument specifies the position of the first line
     of the hexidecimal display.

     It is an error to try to position the HEXSHOW lines on the same
     line as <SET CURLINE>.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: N/A

DEFAULT
     OFF 7

STATUS
     Complete
**man-end**********************************************************************/
#ifdef HAVE_PROTO
short Hexshow(CHARTYPE *params)
#else
short Hexshow(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
#define HEXS_PARAMS  2
 CHARTYPE *word[HEXS_PARAMS+1];
 CHARTYPE *strip=(CHARTYPE *)"11";
 short num_params=0;
 short rc=RC_OK;
 short base=(short)CURRENT_VIEW->hexshow_base;
 short off=CURRENT_VIEW->hexshow_off;
 bool hexshowsts=FALSE;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset1.c:Hexshow");
#endif
 num_params = param_split(params,word,HEXS_PARAMS,WORD_DELIMS,TEMP_PARAM,strip);
 if (num_params < 1)
   {
    display_error(3,(CHARTYPE *)"",FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
/*---------------------------------------------------------------------*/
/* Parse the status parameter...                                       */
/*---------------------------------------------------------------------*/
 rc = execute_set_on_off(word[0],&hexshowsts);
 if (rc != RC_OK)
   {
#ifdef TRACE
    trace_return();
#endif
    return(rc);
   }
/*---------------------------------------------------------------------*/
/* Parse the position parameter...                                     */
/*---------------------------------------------------------------------*/
 if (num_params > 1)
   {
    rc = execute_set_row_position(word[1],&base,&off);
    if (rc != RC_OK)
      {
#ifdef TRACE
       trace_return();
#endif
       return(rc);
      }
   }
/*---------------------------------------------------------------------*/
/* If the HEXSHOW row (or the next row) is the same row as CURLINE and */
/* it is being turned on, return ERROR.                                */
/*---------------------------------------------------------------------*/
 if ((calculate_actual_row(CURRENT_VIEW->current_base,
                          CURRENT_VIEW->current_off,
                          CURRENT_SCREEN.rows[WINDOW_MAIN],TRUE) ==
     calculate_actual_row(base,off,CURRENT_SCREEN.rows[WINDOW_MAIN],TRUE)
     || calculate_actual_row(CURRENT_VIEW->current_base,
                             CURRENT_VIEW->current_off,
                             CURRENT_SCREEN.rows[WINDOW_MAIN],TRUE) ==
        calculate_actual_row(base,off,CURRENT_SCREEN.rows[WINDOW_MAIN],TRUE) + 1)
 && hexshowsts)
   {
    display_error(64,(CHARTYPE *)"- same line as CURLINE",FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_ENVIRON);
   }
 CURRENT_VIEW->hexshow_base = (CHARTYPE)base;
 CURRENT_VIEW->hexshow_off = off;
 CURRENT_VIEW->hexshow_on = hexshowsts;
 post_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line,(LINE *)NULL);
 build_screen(current_screen); 
 display_screen(current_screen);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     set idline - specify if IDLINE is displayed

SYNTAX
     [SET] IDline ON|OFF

DESCRIPTION
     The SET IDLINE set command determines if the <idline> for a file is
     displayed or not.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Compatible.

DEFAULT
     ON

STATUS
     Complete
**man-end**********************************************************************/
#ifdef HAVE_PROTO
short Idline(CHARTYPE *params)
#else
short Idline(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern bool curses_started;
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
 bool save_id_line=FALSE;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset1.c:Idline");
#endif
 save_id_line = CURRENT_VIEW->id_line;
 rc = execute_set_on_off(params,&CURRENT_VIEW->id_line);
 if (rc != RC_OK)
   {
#ifdef TRACE
    trace_return();
#endif
    return(rc);
   }
/*---------------------------------------------------------------------*/
/* If the new value of id_line is the same as before, exit now.        */
/*---------------------------------------------------------------------*/
 if (save_id_line == CURRENT_VIEW->id_line)
   {
#ifdef TRACE
    trace_return();
#endif
    return(rc);
   }
/*---------------------------------------------------------------------*/
/* Redefine the screen sizes...                                        */
/*---------------------------------------------------------------------*/
 set_screen_defaults();
/*---------------------------------------------------------------------*/
/* Recreate windows for the current screen...                          */
/*---------------------------------------------------------------------*/
 if (curses_started)
   {
    if (set_up_windows(current_screen) != RC_OK)
      {
#ifdef TRACE
       trace_return();
#endif
       return(rc);
      }
   }
 build_screen(current_screen);
 display_screen(current_screen);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     set impcmscp - set implied operating system command processing

SYNTAX
     [SET] IMPcmscp ON|OFF

DESCRIPTION
     The SET IMPCMSCP command is used to toggle implied operating system
     command processing from the command line. By turning this feature 
     on you can then issue an operating system command without the need 
     to prefix the operating system command with the <OS> command.

COMPATIBILITY
     XEDIT: Compatible.
     KEDIT: N/A

DEFAULT
     ON

SEE ALSO
     <SET IMPOS>

STATUS
     Complete.
**man-end**********************************************************************/

/*man-start*********************************************************************
COMMAND
     set impmacro - set implied macro command processing

SYNTAX
     [SET] IMPMACro ON|OFF

DESCRIPTION
     The SET IMPMACRO command is used to toggle implied macro processing
     from the command line. By turning this feature on you can then
     issue a <macro> command without the need to prefix the macro name
     with the <MACRO> command.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Compatible.

DEFAULT
     ON

SEE ALSO
     <MACRO>, <SET MACROPATH>

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef HAVE_PROTO
short Impmacro(CHARTYPE *params)
#else
short Impmacro(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset1.c:Impmacro");
#endif
 rc = execute_set_on_off(params,&CURRENT_VIEW->imp_macro);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     set impos - set implied operating system command processing

SYNTAX
     [SET] IMPOS ON|OFF

DESCRIPTION
     The SET IMPOS command is used to toggle implied operating system
     command processing from the command line. By turning this feature 
     on you can then issue an operating system command without the need 
     to prefix the operating system command with the <OS> command.

COMPATIBILITY
     XEDIT: Compatible.
     KEDIT: N/A

DEFAULT
     ON

SEE ALSO
     <SET IMPCMSCP>

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef HAVE_PROTO
short Impos(CHARTYPE *params)
#else
short Impos(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset1.c:Impos");
#endif
 rc = execute_set_on_off(params,&CURRENT_VIEW->imp_os);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     set insertmode - put editor into or out of insert mode

SYNTAX
     [SET] INSERTMode ON|OFF|TOGGLE

DESCRIPTION
     The SET INSERTMODE command enable the user to set the insert mode 
     within THE.

     The 'TOGGLE' option turns insert mode 'ON' if it is currently
     'OFF' and vice versa.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Compatible.

DEFAULT
     OFF

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef HAVE_PROTO
short Insertmode(CHARTYPE *params)
#else
short Insertmode(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern bool INSERTMODEx;
 extern bool curses_started;
/*--------------------------- local data ------------------------------*/
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset1.c:Insertmode");
#endif
 if (equal((CHARTYPE *)"off",params,3))
    INSERTMODEx = FALSE;
 else
    if (equal((CHARTYPE *)"on",params,2))
       INSERTMODEx = TRUE;
    else
       if (equal((CHARTYPE *)"toggle",params,6))
          INSERTMODEx = (INSERTMODEx) ? FALSE : TRUE;
       else
         {
          display_error(1,(CHARTYPE *)params,FALSE);
#ifdef TRACE
          trace_return();
#endif
          return(RC_INVALID_OPERAND);
         }
 if (curses_started)
    draw_cursor(TRUE);
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     set linend - allow/disallow multiple commands on command line

SYNTAX
     [SET] LINENd ON|OFF [character]

DESCRIPTION
     The SET LINEND command allows or disallows the execution of multiple
     commands on the <command line>. When setting LINEND ON, a 'character'
     is specified as the LINEND character which delimits each command.

COMPATIBILITY
     XEDIT: Compatible.
     KEDIT: Compatible.

DEFAULT
     OFF #

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef HAVE_PROTO
short Linend(CHARTYPE *params)
#else
short Linend(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
#define LE_PARAMS  2
 CHARTYPE *word[LE_PARAMS+1];
 CHARTYPE *strip=(CHARTYPE *)"11";
 unsigned short num_params=0;
 bool le_status=CURRENT_VIEW->linend_status;
 CHARTYPE le_value=CURRENT_VIEW->linend_value;
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset1.c:Linend");
#endif
 num_params = param_split(params,word,LE_PARAMS,WORD_DELIMS,TEMP_PARAM,strip);
 switch(num_params)
   {
    case 1:
    case 2:
           rc = execute_set_on_off(word[0],&le_status);
           if (rc != RC_OK)
             {
              display_error(1,word[0],FALSE);
              rc = RC_INVALID_OPERAND;
              break;
             }
           if (num_params == 1)
              break;
           if (strlen((DEFCHAR *)word[1]) > 1)
             {
              display_error(1,word[1],FALSE);
              break;
             }
           le_value = word[1][0];
           break;
    case 0:
           display_error(3,(CHARTYPE *)"",FALSE);
           rc = RC_INVALID_OPERAND;
           break;
    default:
           display_error(2,(CHARTYPE *)"",FALSE);
           rc = RC_INVALID_OPERAND;
           break;
   }
 if (rc == RC_OK)
   {
    CURRENT_VIEW->linend_status = le_status;
    CURRENT_VIEW->linend_value = le_value;
   }
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     set macroext - set default macro extension value

SYNTAX
     [SET] MACROExt [ext]

DESCRIPTION
     The SET MACROEXT command sets the value of the file extension to be
     used for <macro> files. When a macro file name is specified on the
     <command line>, a period '.', then this value will be appended.
     If no value is specified for 'ext', then THE assumes that the
     supplied macro file name is the fully specified name for a macro.

     The length of 'ext' must be 10 characters or less.

     The macro extension is only appended to a file if that file does
     not include any path specifiers.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: N/A

DEFAULT
     the

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef HAVE_PROTO
short Macroext(CHARTYPE *params)
#else
short Macroext(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern CHARTYPE macro_suffix[12];
/*--------------------------- local data ------------------------------*/
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset1.c:Macroext");
#endif
/*---------------------------------------------------------------------*/
/* If no value is specified for ext, set the value of macro_suffix to  */
/* "", otherwise set it to the supplied value, prefixed with '.'       */
/*---------------------------------------------------------------------*/
 if (strlen((DEFCHAR *)params) == 0)
    strcpy((DEFCHAR *)macro_suffix,"");
 else
   {
    if (strlen((DEFCHAR *)params) > 10)
      {
       display_error(7,(CHARTYPE *)params,FALSE);
#ifdef TRACE
       trace_return();
#endif
       return(RC_INVALID_OPERAND);
      }
    strcpy((DEFCHAR *)macro_suffix,".");
    strcat((DEFCHAR *)macro_suffix,(DEFCHAR *)params);
   }
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     set macropath - set default path for macro commands

SYNTAX
     [SET] MACROPath path[s]

DESCRIPTION
     The SET MACROPATH command sets up the search path from which macro
     command files are executed. Each directory is seperated by a
     colon (Unix) or semi-colon (DOS & OS/2).
     No check is done at this stage to validate the supplied path.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Incompatible.

DEFAULT
     Path specified by env variable THE_MACRO_PATH

SEE ALSO
     <MACRO>, <SET IMPMACRO>

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef HAVE_PROTO
short Macropath(CHARTYPE *params)
#else
short Macropath(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern CHARTYPE the_macro_path[MAX_FILE_NAME+1];
/*--------------------------- local data ------------------------------*/
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset1.c:Macropath");
#endif
/*---------------------------------------------------------------------*/
/* No checking is done on macro path supplied other than it contains a */
/* value. Path delimiters are translated if necessary.                 */
/*---------------------------------------------------------------------*/
 if (strlen((DEFCHAR *)params) == 0)
   {
    display_error(3,(CHARTYPE *)"",FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 strcpy((DEFCHAR *)the_macro_path,(DEFCHAR *)params);
 (void *)strtrans(the_macro_path,OSLASH,ISLASH);
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     set margins - set left and right margins for wordwrap

SYNTAX
     [SET] MARgins left right [[+|-]indent]

DESCRIPTION
     The SET MARGINS command sets the 'left' and 'right' margins and the
     number of columns to 'indent' a paragraph.

     These values are used with the <SET WORDWRAP> option.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Compatible.

DEFAULT
     1 72 +0

SEE ALSO
     <SET WORDWRAP>

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef HAVE_PROTO
short Margins(CHARTYPE *params)
#else
short Margins(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern CHARTYPE *temp_cmd;
/*--------------------------- local data ------------------------------*/
#define MAR_PARAMS  3
 CHARTYPE *word[MAR_PARAMS+1];
 CHARTYPE *strip=(CHARTYPE *)"111";
 short num_params=0;
 short left=0,right=0,indent=0;
 bool offset=FALSE,consistancy_error=FALSE;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset1.c:Margins");
#endif
/*---------------------------------------------------------------------*/
/* Two parameters are mandatory, the third is optional.                */
/*---------------------------------------------------------------------*/
 num_params = param_split(params,word,MAR_PARAMS,WORD_DELIMS,TEMP_PARAM,strip);
 if (num_params < 2)
   {
    display_error(3,(CHARTYPE *)"",FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 if (num_params > 3)
   {
    display_error(2,(CHARTYPE *)"",FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
/*---------------------------------------------------------------------*/
/* Parse the parameters...                                             */
/*---------------------------------------------------------------------*/
 left = atoi((DEFCHAR *)word[0]);
 if (left < 1)
   {
    display_error(5,word[0],FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
/*---------------------------------------------------------------------*/
/* Right margin value can be *, set to maximum line length.            */
/*---------------------------------------------------------------------*/
 if (*(word[1]+1) == '*')
   {
    right = max_line_length;
   }
 else
   {
    right = atoi((DEFCHAR *)word[1]);
    if (right < 1)
      {
       display_error(5,word[1],FALSE);
#ifdef TRACE
       trace_return();
#endif
       return(RC_INVALID_OPERAND);
      }
   }
/*---------------------------------------------------------------------*/
/* Left margin must be less than right margin.                         */
/*---------------------------------------------------------------------*/
 if (right < left)
   {
    display_error(5,word[1],FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
/*---------------------------------------------------------------------*/
/* Obtain current values for indent, in case they aren't changed by    */
/* the current command. (ie. no third parameter)                       */
/*---------------------------------------------------------------------*/
 indent = CURRENT_VIEW->margin_indent;
 offset = CURRENT_VIEW->margin_indent_offset_status;
/*---------------------------------------------------------------------*/
/* Determine the type of offset for the indent value. If a sign is     */
/* specified, then the number supplied is relative to the left margin  */
/* otherwise it is an absolute column value.                           */
/*---------------------------------------------------------------------*/
 if (num_params == 3)
   {
    if (*(word[2]) == '-'
    ||  *(word[2]) == '+')
      {
       offset = TRUE;
       if ((indent = atoi((DEFCHAR *)word[2])) == 0)
         {
          if (strcmp((DEFCHAR *)word[2],"+0") != 0)
            {
             display_error(1,word[2],FALSE);
#ifdef TRACE
             trace_return();
#endif
             return(RC_INVALID_OPERAND);
            }
         }
      }
    else
      {
       offset = FALSE;
/*---------------------------------------------------------------------*/
/* Absolute indent cannot be negative.                                 */
/*---------------------------------------------------------------------*/
       if ((indent = atoi((DEFCHAR *)word[2])) < 0)
         {
          display_error(1,word[2],FALSE);
#ifdef TRACE
          trace_return();
#endif
          return(RC_INVALID_OPERAND);
         }
      }
   }
/*---------------------------------------------------------------------*/
/* Once all values are determined, validate the relationship between   */
/* the margins and the indent values.                                  */
/* Rules:                                                              */
/*       o If indent is a negative offset, the resultant column value  */
/*         cannot be negative.                                         */
/*       o If indent is a positive offset, the resultant column value  */
/*         cannot be > max_line_length or right margin                 */
/*       o If indent is an absolute value, it cannot be > right margin */
/*---------------------------------------------------------------------*/
 consistancy_error = FALSE;
 if (offset
 && indent < 0
 && indent + left < 0)
    consistancy_error = TRUE;
 if (offset
 && indent > 0
 && indent + left > right)
    consistancy_error = TRUE;
 if (offset
 && indent > 0
 && indent + left > max_line_length)
    consistancy_error = TRUE;
 if (!offset
 && indent > right)
    consistancy_error = TRUE;
 if (consistancy_error)
   {
    if (offset)
       sprintf((DEFCHAR *)temp_cmd,"%d %d %+d",left,right,indent);
    else
       sprintf((DEFCHAR *)temp_cmd,"%d %d %d",left,right,indent);
    display_error(12,temp_cmd,FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
/*---------------------------------------------------------------------*/
/* All OK, so save the values...                                       */
/*---------------------------------------------------------------------*/
 CURRENT_VIEW->margin_left = left;
 CURRENT_VIEW->margin_right = right;
 CURRENT_VIEW->margin_indent = indent;
 CURRENT_VIEW->margin_indent_offset_status = offset;
/*---------------------------------------------------------------------*/
/* If the SCALE line is currently displayed, display the page so that  */
/* any changes are reflected in the SCALE line.                        */
/*---------------------------------------------------------------------*/
 if (CURRENT_VIEW->scale_on)
   {
    build_screen(current_screen); 
    display_screen(current_screen);
   }
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     set msgline - set position and size of message line

SYNTAX
     [SET] MSGLine ON M[+n|-n]|[+|-]n [lines] [Overlay]

DESCRIPTION
     The SET MSGLINE set command specifies the position of the 
     <message line> and the size of the message line window.

     The first form of parameters is:

     M[+n|-n] 
     this sets the first line to be relative to the middle of
     the screen. A positive value adds to the middle line number, 
     a negative subtracts from it.
     eg. M+3 on a 24 line screen will be line 15
         M-5 on a 24 line screen will be line 7

     The second form of parameters is:

     [+|-]n
     this sets the first line to be relative to the top of the
     screen (if positive or no sign) or relative to the bottom 
     of the screen if negative.
     eg. +3 or 3 will set current line to line 3
         -3 on a 24 line screen will be line 21

     If the resulting line is outside the bounds of the screen
     the position of the message line will become the middle line
     on the screen.

COMPATIBILITY
     XEDIT: Compatible.
            The OVERLAY option is the default but ignored.
     KEDIT: Compatible
            The OVERLAY option is the default but ignored.

DEFAULT
     ON 2 5 Overlay

STATUS
     Complete
**man-end**********************************************************************/
#ifdef HAVE_PROTO
short Msgline(CHARTYPE *params)
#else
short Msgline(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
#define MSG_PARAMS  5
 CHARTYPE *word[MSG_PARAMS+1];
 CHARTYPE *strip=(CHARTYPE *)"1111";
 short num_params=0;
 short rc=RC_OK;
 short base=(short)CURRENT_VIEW->msgline_base;
 short off=CURRENT_VIEW->msgline_off;
 bool msgsts=FALSE;
 ROWTYPE num_lines=CURRENT_VIEW->msgline_rows;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset1.c:Msgline");
#endif
 num_params = param_split(params,word,MSG_PARAMS,WORD_DELIMS,TEMP_PARAM,strip);
 if (num_params < 2)
   {
    display_error(3,(CHARTYPE *)"",FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 if (num_params > 4)
   {
    display_error(2,(CHARTYPE *)"",FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
/*---------------------------------------------------------------------*/
/* Parse the status parameter...                                       */
/*---------------------------------------------------------------------*/
 rc = execute_set_on_off(word[0],&msgsts);
 if (rc != RC_OK)
   {
#ifdef TRACE
    trace_return();
#endif
    return(rc);
   }
/*---------------------------------------------------------------------*/
/* ... only "ON" is allowed...                                         */
/*---------------------------------------------------------------------*/
 if (!msgsts)
   {
    display_error(1,word[0],FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
/*---------------------------------------------------------------------*/
/* Parse the position parameter...                                     */
/*---------------------------------------------------------------------*/
 if (num_params > 1)
   {
    rc = execute_set_row_position(word[1],&base,&off);
    if (rc != RC_OK)
      {
#ifdef TRACE
       trace_return();
#endif
       return(rc);
      }
   }
/*---------------------------------------------------------------------*/
/* To get here we have either two arguments or one. If two, the first  */
/* is the number of lines, and the second MUST be Overlay.             */
/* If one argument, it is either Overlay or number of lines.           */
/*---------------------------------------------------------------------*/
 switch(num_params)
   {
    case 3:
         if (equal((CHARTYPE *)"overlay",word[2],1))
            num_lines = 1;
         else
           {
            num_lines = atoi((DEFCHAR *)word[2]);
            if (num_lines < 1)
              {
               display_error(5,word[2],FALSE);
#ifdef TRACE
               trace_return();
#endif
               return(rc);
              }
           }
         break;
    case 4:
         num_lines = atoi((DEFCHAR *)word[2]);
         if (num_lines < 1)
           {
            display_error(5,word[2],FALSE);
#ifdef TRACE
            trace_return();
#endif
            return(rc);
           }
         if (!equal((CHARTYPE *)"overlay",word[3],1))
           {
            display_error(1,word[3],FALSE);
#ifdef TRACE
            trace_return();
#endif
            return(rc);
           }
         break;
    default:
         num_lines = 1;
         break;
   }
 CURRENT_VIEW->msgline_base = (CHARTYPE)base;
 CURRENT_VIEW->msgline_off = off;
 CURRENT_VIEW->msgline_rows = num_lines;
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     set msgmode - set display of messages on or off

SYNTAX
     [SET] MSGMode ON|OFF

DESCRIPTION
     The SET MSGMODE set command determines whether error messages will
     be displayed or suppressed.

COMPATIBILITY
     XEDIT: Does not support [Short|Long] options.
     KEDIT: Compatible

DEFAULT
     ON

STATUS
     Complete
**man-end**********************************************************************/
#ifdef HAVE_PROTO
short Msgmode(CHARTYPE *params)
#else
short Msgmode(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset1.c:Msgmode");
#endif
 rc = execute_set_on_off(params,&CURRENT_VIEW->msgmode_status);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     set newlines - set position of cursor after adding blank line

SYNTAX
     [SET] NEWLines Aligned|Left

DESCRIPTION
     The SET NEWLINES set command determines where the cursor displays
     after a new line is added to the file.

     With 'ALIGNED', the cursor will display in the column of the new line
     immediately underneath the first non-blank character in the line
     above.
     With 'LEFT', the cursor will display in the first column of the new
     line.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Same command, different functionality.

DEFAULT
     Aligned

STATUS
     Complete
**man-end**********************************************************************/
#ifdef HAVE_PROTO
short Newlines(CHARTYPE *params)
#else
short Newlines(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
#define NEW_PARAMS  1
 CHARTYPE parm[NEW_PARAMS];
 CHARTYPE *word[NEW_PARAMS+1];
 CHARTYPE *strip=(CHARTYPE *)"1";
 unsigned short num_params=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset1.c:Newlines");
#endif
 num_params = param_split(params,word,NEW_PARAMS,WORD_DELIMS,TEMP_PARAM,strip);
 if (num_params > 1)
   {
    display_error(2,(CHARTYPE *)"",FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 if (num_params < 1)
   {
    display_error(3,(CHARTYPE *)"",FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }

 parm[0] = (CHARTYPE)UNDEFINED_OPERAND;
 if (equal((CHARTYPE *)"aligned",word[0],1))
    parm[0] = TRUE;
 if (equal((CHARTYPE *)"left",word[0],1))
    parm[0] = FALSE;
 if (parm[0] == (CHARTYPE)UNDEFINED_OPERAND)
   {
    display_error(1,word[0],FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 CURRENT_VIEW->newline_aligned = parm[0];
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     set nondisp - specify character to display for non-displaying characters

SYNTAX
     [SET] NONDisp character

DESCRIPTION
     The SET NONDISP command allows the user to change the 'character' 
     that is displayed for non-displaying commands when <SET ETMODE> 
     is OFF.

COMPATIBILITY
     XEDIT: Compatible.
     KEDIT: N/A

DEFAULT
     #

SEE ALSO
     <SET ETMODE>

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef HAVE_PROTO
short Nondisp(CHARTYPE *params)
#else
short Nondisp(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern bool NONDISPx;
/*--------------------------- local data ------------------------------*/
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset1.c:Nondisp");
#endif
 if (strlen((DEFCHAR *)params) != 1)
   {
    display_error(1,params,FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 NONDISPx = *params;
 build_screen(current_screen); 
 display_screen(current_screen);
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     set number - turn prefix numbers on or off

SYNTAX
     [SET] NUMber ON|OFF

DESCRIPTION
     The SET NUMBER command allows the user to toggle the display of 
     numbers in the <prefix area>.

COMPATIBILITY
     XEDIT: Compatible.
     KEDIT: Compatible.

DEFAULT
     ON

SEE ALSO
     <SET PREFIX>

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef HAVE_PROTO
short Number(CHARTYPE *params)
#else
short Number(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset1.c:Number");
#endif
 rc = execute_set_on_off(params,&CURRENT_VIEW->number);
 if (rc == RC_OK)
   {
    build_screen(current_screen); 
    display_screen(current_screen);
   }
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
