/*
   lfn.c - Long filename functions for WRen.

   Jason Hood, 14 June, 2000 & 25 November, 2002.

   v1.01 - 6 April, 2003:
     Win32 port;
     use the DOS interrupt for renaming and existing.

   v1.02 - 9 October, 2003:
     added my_chmod and my_unlink functions.
*/

#include "lfn.h"

#ifdef _WIN32

static long find_handle;
static int  find_attrib;

int my_findfirst( const char* pathname, struct my_ffblk* ffblk, int attrib )
{
  find_handle = _findfirst( pathname, ffblk );
  if (find_handle != -1)
  {
    find_attrib = attrib;
    if (((ffblk->attrib & (FA_HIDDEN | FA_SYSTEM)) &&
	 !(attrib & (FA_HIDDEN | FA_SYSTEM)))  ||
	(ffblk->attrib & FA_DIREC) && !(attrib & FA_DIREC))
      return my_findnext( ffblk );
  }

  return (find_handle == -1) ? -1 : 0;
}


int my_findnext( struct my_ffblk* ffblk )
{
  int rc;

  do
  {
    rc = _findnext( find_handle, ffblk );
    if (rc == -1)
    {
      _findclose( find_handle );
      break;
    }
  } while (((ffblk->attrib & (FA_HIDDEN | FA_SYSTEM)) &&
	    !(find_attrib & (FA_HIDDEN | FA_SYSTEM)))  ||
	   (ffblk->attrib & FA_DIREC) && !(find_attrib & FA_DIREC));
  return rc;
}

#else // DOS

#include <dos.h>
#include <string.h>
#include <stdio.h>


#define CF 1                 // Carry flag


int use_lfn;		     // Non-zero if LFN is available


/*
   Determine if LFN is available on the current drive. If so, assume it is
   (not) available on all drives.
*/
int init_lfn( void )
{
  struct REGPACK regs;
  static char* buf = "?:\\";

  *buf = getdisk() + 'A';
  regs.r_ax = 0x71a0;		// Get Volume Information
  regs.r_ds = FP_SEG( buf );
  regs.r_dx = FP_OFF( buf );
  regs.r_cx = 0;		// Ignore the volume name (zero-size buffer)
  regs.r_bx = 0;		// Initialise, for older DOS compatibility
  intr( 0x21, &regs );
  use_lfn = (regs.r_bx & 0x4000);

  return use_lfn;
}


int lfn_findfirst( const char* spec, struct lfn_ffblk* ffblk, int attrib )
{
  struct REGPACK regs;

  regs.r_ax = 0x714e;		// LFN Find First Matching File
  regs.r_cx = attrib;
  regs.r_si = 1;                // DOS time format
  regs.r_ds = FP_SEG( spec );
  regs.r_dx = FP_OFF( spec );
  regs.r_es = FP_SEG( ffblk );
  regs.r_di = FP_OFF( ffblk );
  intr( 0x21, &regs );

  if (regs.r_flags & CF)
    return -1;

  ffblk->handle = regs.r_ax;
  return 0;
}


int lfn_findnext( struct lfn_ffblk* ffblk )
{
  struct REGPACK regs;

  regs.r_ax = 0x714f;		// LFN Find Next Matching File
  regs.r_bx = ffblk->handle;
  regs.r_si = 1;
  regs.r_es = FP_SEG( ffblk );
  regs.r_di = FP_OFF( ffblk );
  intr( 0x21, &regs );

  if (regs.r_flags & CF)
  {
    regs.r_ax = 0x71a1; 	// LFN FindClose
    regs.r_bx = ffblk->handle;
    intr( 0x21, &regs );
    return -1;
  }
  return 0;
}


int my_findfirst( const char* pathname, struct my_ffblk* ffblk, int attrib )
{
  int rc;

  if (use_lfn)
  {
    ffblk->name = ffblk->ff.lf.fname;
    rc = lfn_findfirst( pathname, &ffblk->ff.lf, attrib );
  }
  else
  {
    ffblk->name = ffblk->ff.df.ff_name;
    rc = findfirst( pathname, &ffblk->ff.df, attrib );
  }

  return rc;
}


int my_findnext( struct my_ffblk* ffblk )
{
  return (use_lfn) ? lfn_findnext( &ffblk->ff.lf )
		   :	 findnext( &ffblk->ff.df );
}


int my_rename( const char* old, const char* new )
{
  struct REGPACK regs;

  regs.r_ax = (use_lfn) ? 0x7156	// LFN Rename File
			: 0x5600;	// DOS Rename File
  regs.r_ds = FP_SEG( old );
  regs.r_dx = FP_OFF( old );
  regs.r_es = FP_SEG( new );
  regs.r_di = FP_OFF( new );
  intr( 0x21, &regs );

  return (regs.r_flags & CF) ? -1 : 0;
}


int file_exists( const char* file )
{
  struct REGPACK regs;

  if (use_lfn)
  {
    regs.r_ax = 0x7143; 	// Extended Get/Set File Attributes
    regs.r_bx = 0;		// Get attributes
  }
  else
  {
    regs.r_ax = 0x4300; 	// DOS Get File Attributes
  }
  regs.r_ds = FP_SEG( file );
  regs.r_dx = FP_OFF( file );
  intr( 0x21, &regs );

  return (regs.r_flags & CF) ? 0 : 1;
}


int my_chmod( const char* file, int mode )
{
  struct REGPACK regs;

  if (use_lfn)
  {
    regs.r_ax = 0x7143; 	// Extended Get/Set File Attributes
    regs.r_bx = 1;		// Set attributes
  }
  else
  {
    regs.r_ax = 0x4301; 	// DOS Set File Attributes
  }
  regs.r_ds = FP_SEG( file );
  regs.r_dx = FP_OFF( file );
  regs.r_cx = mode;
  intr( 0x21, &regs );

  return (regs.r_flags & CF) ? 0 : 1;
}


int my_unlink( const char* file )
{
  struct REGPACK regs;

  if (use_lfn)
  {
    regs.r_ax = 0x7141; 	// LFN Delete File
    regs.r_si = 0;		// Ignore wildcards/attributes
  }
  else
  {
    regs.r_ax = 0x41;		// DOS Delete File
  }
  regs.r_ds = FP_SEG( file );
  regs.r_dx = FP_OFF( file );
  intr( 0x21, &regs );

  return (regs.r_flags & CF) ? -1 : 0;
}

#endif // ?_WIN32
