/******************************************************************************
Fractal Infinity - USER CONFIGURATION FUNCTIONS - fr_user.c.
This file Copyright (C) 1999 Graeme Fenwick- see documentation for details.
******************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include "fr_user.h"
#include "fr_hware.h"

#define  MAG_MAX   1000.0                           /* max/min magnification */
#define  MAG_MIN   0.001
#define  MAG_F     1.1                                        /* zoom factor */

void xhair_adjust(long double *xc, long double *yc, long double range,
                  int xhair, int key, long double oxc, long double oyc);

/********** SAVE_NEXT: Save screen using next available filename */

int save_next(void)
{
   FILE         *fpoint;
   char         fname[13];
   int          diskwr = 1;                 /* file/disk writeability status */
   static int   num = 0;                         /* numeric part of filename */

   if (verify_imagespace() < 0) {
      message_dialog("Not enough space to save screen", "on current disk.");
      return 0;
   }
   if (num >= MAXFILES) {                        /* filenames *all* used up? */
      message_dialog("Too many \"fr------.bmp\" files",
                     "in current directory to save screen.");
      return 0;
   }
   do {
      sprintf(fname, "fr%06d.bmp", num++);         /* generate next filename */
      if ((fpoint = fopen(fname, "rb")) != NULL)
         ;                                         /* name in use- try again */
      else {
         fclose(fpoint);
         fpoint = fopen(fname, "wb");
         diskwr = (fpoint == NULL) ? 0 : 2; /* rough writeability check only */
      }
   } while (num < MAXFILES && fpoint != NULL && diskwr == 1);
   if (fpoint == NULL || (num >= MAXFILES && diskwr != 2)) {  
      message_dialog("Cannot write current screen to disk.",
                     "(Please check write-protect if saving to floppy.)");
      return 0;
   }
   fclose (fpoint);
   if (save_display(fname) == NULL) {             /* attempt to save display */
      message_dialog("Could not save to disk", "for unknown reason.");
      return 0;
   }
   else
      return 1; 
}

/********** ADJUSTMENT: Allows user to use keyboard to alter params */

int adjustment(long double *mag, long double *xc, long double *yc,
               long double *range)
{
   long double   oxc = *xc, oyc = *yc;         /* pre-adjustment xc/yc copies*/
   int           key, xhair = 0;

   detect_key(1);                                        /* clear key buffer */
   do {
      key = detect_key(0);                                     /* detect key */
      if (key == PLUS)                                     /* increase magnf */
 	 info_display(*mag= ((*mag * MAG_F)<=MAG_MAX ? *mag * MAG_F : MAG_MAX),
            *range, oxc, oyc);
      else if (key == MINUS)                               /* decrease magnf */
	 info_display(*mag= ((*mag / MAG_F)>=MAG_MIN ? *mag / MAG_F : MAG_MIN),
            *range, oxc, oyc);
      else if (key == SPACE)
         info_display(*mag = 1.0, *range, oxc, oyc);
      else if (key == UP || key == DOWN || key == LEFT || key == RIGHT)
         xhair_adjust(xc, yc, *range, xhair++, key, oxc, oyc);
      else if (key == SAVE) {                         /* save current screen */
         if (xhair > 0) {                             /* erase current xhair */
            toggle_draw_mode();
            draw_xhair((*xc - oxc) / (*range / 2) * (sxdim / 2) + (sxdim / 2),
               (*yc - oyc) / (*range / 2) * (sxdim / 2) + ((sydim - 16) / 2));
            toggle_draw_mode();
            xhair = 0;
         }
         save_next(); 
      }
   } while (key != ENTER && key != QUIT);
   *range /= *mag;
   return key;                                             /* has user quit? */
}

/********** XHAIR_ADJUST: Allows user to choose center of next image */

void xhair_adjust(long double *xc, long double *yc, long double range, 
                  int xhair, int key, long double oxc, long double oyc)
{
   static int    scnt_x, scnt_y;
   long double   y_lim = (range / 2) * ((long double) (sydim - 16) / sxdim);
                                           /* fractal coord lower y boundary */

   toggle_draw_mode();                                       /* exor-draw on */
   if (xhair == 0) {                      /* only calculated on first access */
      scnt_x = sxdim / 2;
      scnt_y = (sydim - 16) / 2;
   }
   else                                              /* erase previous xhair */
      draw_xhair((*xc - oxc) / (range / 2) * scnt_x + scnt_x,
                 (*yc - oyc) / (range / 2) * scnt_x + scnt_y);
   *yc += ((key == DOWN) - (key == UP)) * (range / 128);
   *xc += ((key == RIGHT) - (key == LEFT)) * (range / 128);
   if (*xc - oxc < -range / 2)            /* keep xhair within screen limits */
      *xc = oxc - range / 2;
   else if (*xc - oxc >= range / 2)
      *xc = oxc + range / 2 - (range / sxdim);
   if ((*yc - oyc) < (-y_lim))                                      /* ditto */
      *yc = oyc - y_lim;
   else if ((*yc - oyc) >= y_lim)
      *yc = oyc + y_lim - (range / sxdim);
   draw_xhair((*xc - oxc) / (range / 2) * scnt_x + scnt_x,
              (*yc - oyc) / (range / 2) * scnt_x + scnt_y);  
   toggle_draw_mode();                                      /* exor-draw off */
}

/********** AUTO_SETUP: Tries to set up screen (etc.) with requested res./depth
                        but will automatically adjust, and report errors */

int auto_setup(int xr, int yr, int depth)
{
   int   status;

   if (depth != 32 && depth != 24 && depth !=16 && depth != 15 && depth != 8) {
      printf("\nOnly 32, 24, 16, 15 and 8 bit color depths supported.\n");
      return 0;
   }
   if (yr == -1)                                  /* y-res automatically set */
      yr = 480 * (xr == 640) + 600 * (xr == 800) + 768 * (xr == 1024);
   if (yr == 0) {                                    /* can't auto-set y-res */
      printf("\n%s%s\n",
         "Vertical (y) resolution only automatically set if x = ",
         "640, 800 or 1024.");
      return 0;
   }
   do {                                  /* attempt to open requested screen */
      status = hardware_setup(xr, yr, depth);
      if (status == 0) {                  /* can't open at that res. & depth */
         printf("Can't set up screen at %d x %d x %d.\n\n", xr, yr, depth);
         depth -= 8 * (depth == 32) + 8 * (depth == 24) + (depth == 16) +
                  7 * (depth == 15) + (depth == 8);        /* decrease depth */
         if (depth >= 8) {
            printf("Trying a lower depth (%d bits)...\n", depth);
            sleep_wrap(3);
         }
      }
   } while (status == 0 && depth >= 8);
   if (status == 0) {                      /* unable to set up screen at all */
      printf("%s\n%s\n",
         "Unable to open screen at requested depth (or lower).",
         "Try running program with a different resolution and/or depth.");
      return 0;
   }
   return 1;
}
