/*******************************************************************
 *
 *  ttextend.c                                                   1.0
 *
 *    Extensions Default Body 
 *
 *  Copyright 1996, 1997 by
 *  David Turner, Robert Wilhelm, and Werner Lemberg.
 *
 *  This file is part of the FreeType project, and may only be used
 *  modified and distributed under the terms of the FreeType project
 *  license, LICENSE.TXT. By continuing to use, modify or distribute
 *  this file you indicate that you have read the license and
 *  understand and accept it fully.
 *
 *  It is now possible to 'extend' the FreeType engine, in order
 *  to be able to access additionnal tables, without having to
 *  to change the core library's code.
 *
 *  Only extensions of the face object class are supported.
 *  Each face object has now a typeless pointer called 'extension'
 *  that can be used as an anchor to the data specific to a given
 *  library extension.
 *
 *  The extension component can also allocate memory on engine
 *  initialisation, which calls the TTExtension_Init function.
 *
 *  NOTE :
 *
 *  The ttextend.h and ttextend.c are used to extend the core           
 *  library. They do not affect the high-level API.  
 *
 *  It is possible to extend the high-level API by modifying the
 *  files 'apiext.h' and 'apiext.c' found in this same directory
 *  read their source for more information on how to do that.
 *
 *  ===============================================================
 *
 *  This is the first default extension that came to my mind,
 *  it provides support for the kerning tables. Read 'ftxapi.h'
 *  to see how the user application can access them.
 *
 *                                                    - DavidT
 *
 ******************************************************************/

#include "freetype.h"
#include "ttextend.h"
#include "tterror.h"
#include "ttmemory.h"

/*******************************************************************
 *
 *  Function    :  TTExtension_Init
 *
 *  Description :  Initialize the extensions component.
 *                 Called by the engine initializer FreeType_Init
 *
 ******************************************************************/

  TT_Error  TTExtension_Init()
  {
    /* No supplemental memory required to support these extensions */
    return TT_Err_Ok;
  }


/*******************************************************************
 *
 *  Function    :  TTExtension_Done
 *
 *  Description :  Finalizes the extensions component.
 *                 Called by the engine finalizer FreeType_Done
 *
 ******************************************************************/

  TT_Error  TTExtension_Done()
  {
    /* No memory was required to support these extensions */
    return TT_Err_Ok;
  }

/*******************************************************************
 *
 *  Function    :  Extension_Destroy
 *
 *  Description :  Destroy a previously allocated extensions
 *
 *  NOTE : This is the last function called by the face destructor.
 *         as a consequence, you should consider all the fields in
 *         the face object invalid. You're not allowed to access
 *         them with the exception of the face->extension field
 *         here anyway.
 *
 *         The error code is currently ignored by the destructor
 *
 ******************************************************************/

  TT_Error  Extension_Destroy( PFace  face )
  {
    TExtension*  ext;

    /* by convention, exit when facing null pointers */
    if ( !face )
      return TT_Err_Ok;

    ext = (TExtension*)face->extension;
    if ( !ext )
      return TT_Err_Ok;

    /* ----------------- kerning support --------------------*/

    Kerning_Destroy( &ext->kerning );

    /* --------- other optional extensions ----------------- */

    /* ----------------------------------------------------- */

    /* then, destroy the extension block itself */

    FREE( face->extension );
    return TT_Err_Ok;
  }

/*******************************************************************
 *
 *  Function    :  Extension_Create
 *
 *  Description :  Create
 *
 ******************************************************************/

  TT_Error  Extension_Create( PFace  face )
  {
    TT_Error     error;
    TExtension*  ext;

    /* First of all, create the extension block */

    if ( ALLOC( face->extension, sizeof(TExtension) ) )
      return error;

    ext = (TExtension*)face->extension;

    /* ----------------- kerning support --------------------*/

    error = Kerning_Create( face, &ext->kerning );
    if (error)
      return error;

    /* --------- other optional extensions ----------------- */

    /* ----------------------------------------------------- */

    return error;
  }

