typedef float   RtColor[3];
typedef float   RtPoint[3];
typedef float   RtMatrix[4][4];

typedef char   *RtPointer;

#define SCN_PRIMITIVE    '0'
#define SCN_UNION        '1'
#define SCN_DIFFERENCE   '2'
#define SCN_INTERSECTION '3'
#define SCN_LIST         '4'

/**********/
#include <stdio.h>
#include <math.h>

#ifdef M_PI
#define _PI M_PI
#endif
#define PI _PI

static FILE    *_ScnStream = stdout;

void
ScnBegin(name)
  char           *name;
{
  if (name != NULL)
  {
    if ((_ScnStream = fopen(name, "w")) == NULL)
      _ScnStream = stdout;
  }
  fprintf(_ScnStream, "%% Converted to SCN\n");
  fprintf(_ScnStream, "%% Copyright Antonio Costa INESC-Norte 1992\n\n");
}

void
ScnEnd()
{
  fflush(_ScnStream);
  if (_ScnStream != stdout)
    fclose(_ScnStream);
}

void
ScnDistantLightSource(dir)
  RtPoint	  dir;
{
  fprintf(_ScnStream, "light directional %g %g %g white\n",
          dir[0], dir[1], dir[2]);
}

void
ScnTranslate(x, y, z)
  float           x, y, z;
{
  fprintf(_ScnStream, "transform translate %g %g %g\n", x, y, z);
}

void
ScnRotate(angle, x, y, z)
  float           angle, x, y, z;
{
  fprintf(_ScnStream, "transform rotate %g %g %g %g\n", x, y, z, angle);
}

void
ScnWorldBegin(eye, at, up, fov)
  RtPoint	  eye, at, up;
  float		  fov;
{
  fprintf(_ScnStream, "eye %g %g %g\n", eye[0], eye[1], eye[2]);
  fprintf(_ScnStream, "at %g %g %g\n", at[0], at[1], at[2]);
  fprintf(_ScnStream, "up %g %g %g\n", up[0], up[1], up[2]);
  fprintf(_ScnStream, "fov %g\n", fov);
  fprintf(_ScnStream, "%% geometry\n");
}

void
ScnWorldEnd()
{
  fprintf(_ScnStream, "%% end of geometry\n");
}

void
ScnTransformBegin()
{
  fprintf(_ScnStream, "group %% transform\n");
}

void
ScnTransformEnd()
{
  fprintf(_ScnStream, "ungroup %% transform\n");
}

#define CSG_LEVEL_MAX 256

static int	csg_level = -1;
static char	csg[CSG_LEVEL_MAX];

void
ScnSolidBegin(type)
  char            type;
{
  csg_level++;
  csg[csg_level] = type;
  switch (type)
  {
  case SCN_PRIMITIVE:
    fprintf(_ScnStream, "group %% solid\n");
    return;
  case SCN_UNION:
    fprintf(_ScnStream, "csg union begin\n");
    return;
  case SCN_DIFFERENCE:
    fprintf(_ScnStream, "csg subtraction begin\n");
    return;
  case SCN_INTERSECTION:
    fprintf(_ScnStream, "csg intersection begin\n");
    return;
  case SCN_LIST:
    fprintf(_ScnStream, "list begin group\n");
    return;
  }
}

void
ScnSolidNext()
{
  fprintf(_ScnStream, "csg next\n");
}

void
ScnSolidEnd()
{
  if (csg[csg_level] == SCN_PRIMITIVE)
    fprintf(_ScnStream, "ungroup %% solid\n");
  else
  if (csg[csg_level] == SCN_LIST)
    fprintf(_ScnStream, "ungroup list end\n");
  else
    fprintf(_ScnStream, "csg end\n");
  csg_level--;
}

void
ScnColor(color)
  RtColor         color;
{
  fprintf(_ScnStream, "surface matte %g %g %g\n",
	  color[0], color[1], color[2]);
}

void
ScnSphere(r)
  float           r;
{
  fprintf(_ScnStream, "sphere 0 0 0 %g\n", r);
}

void
ScnClosedCylinder(h, r)
  float           h, r;
{
  fprintf(_ScnStream, "cylinder 0 0 0 0 0 %g %g\n", h, r);
}

void
ScnClosedCone(h, r)
  float           h, r;
{
  fprintf(_ScnStream, "cone 0 0 0 0 0 %g %g\n", h, r);
}

void
ScnBox(x, y, z)
  float           x, y, z;
{
  fprintf(_ScnStream, "box %g %g %g %g %g %g\n",
          x / 2.0, y / 2.0, z / 2.0,
          fabs(x) / 2.0, fabs(y) / 2.0, fabs(z) / 2.0);
}

void
ScnPolygon(n, p)
  int             n;
  char           *p;
{
  RtPoint        *pol = (RtPoint *) p;
  int             i;

  fprintf(_ScnStream, "polygon %d\n", n);
  for (i = 0; i < n; i++)
  {
    fprintf(_ScnStream, "%g %g %g\n", (*pol)[0], (*pol)[1], (*pol)[2]);
    pol++;
  }
}

void
ScnTorus(maj, min)
  float           maj, min;
{
  fprintf(_ScnStream, "torus %g %g 0 360 16 8\n",
          maj, min);
}
#define EQ(a, b) (fabs((a) - (b)) < 1.0e-5)

void
ScnHyperboloid(l1, l2, angle)
  RtPoint         l1, l2;
  float           angle;
{
  float           ang, r1, r2, angle1, angle2;
  RtPoint         p1, p2, o1, o2, n1, n2;
  extern double	  _hypot(), sin(), cos();

  if (l2[2] > l1[2])
  {
    p1[0] = l1[0];
    p1[1] = l1[1];
    p1[2] = l1[2];
    p2[0] = l2[0];
    p2[1] = l2[1];
    p2[2] = l2[2];
  } else
  {
    p2[0] = l1[0];
    p2[1] = l1[1];
    p2[2] = l1[2];
    p1[0] = l2[0];
    p1[1] = l2[1];
    p1[2] = l2[2];
  }
  while (angle < 0.0)
    angle += 360.0;
  if (angle > 360.0)
    angle -= 360.0;
  r1 = _hypot(p1[0], p1[1]);
  r2 = _hypot(p2[0], p2[1]);
  angle1 = atan2(p1[1], p1[0]);
  angle2 = atan2(p2[1], p2[0]);
  if (EQ(r1, 0.0))
    r1 = -1.0;
  if (EQ(r2, 0.0))
    r2 = -1.0;
  o1[0] = p1[0];
  o1[1] = p1[1];
  o1[2] = p1[2];
  o2[0] = p2[0];
  o2[1] = p2[1];
  o2[2] = p2[2];
  n1[2] = p1[2];
  n2[2] = p2[2];
  ang = 3.0;
  while (ang <= angle)
  {
    register double a;

    a = ang / 180.0 * PI;
    if (r1 > 0.0)
    {
      n1[0] = r1 * cos(a + angle1);
      n1[1] = r1 * sin(a + angle1);
    }
    if (r2 > 0.0)
    {
      n2[0] = r2 * cos(a + angle2);
      n2[1] = r2 * sin(a + angle2);
    }
    if (r1 < 0.0)
      fprintf(_ScnStream,
	      "triangle\n %g %g %g\n %g %g %g\n %g %g %g\n",
	      o2[0], o2[1], o2[2],
	      o1[0], o1[1], o1[2],
	      n2[0], n2[1], n2[2]);
    else if (r2 < 0.0)
      fprintf(_ScnStream,
	      "triangle\n %g %g %g\n %g %g %g\n %g %g %g\n",
	      o2[0], o2[1], o2[2],
	      o1[0], o1[1], o1[2],
	      n1[0], n1[1], n1[2]);
    else
      fprintf(_ScnStream,
	      "quadrangle\n %g %g %g\n %g %g %g\n %g %g %g\n %g %g %g\n",
	      o2[0], o2[1], o2[2],
	      o1[0], o1[1], o1[2],
	      n1[0], n1[1], n1[2],
	      n2[0], n2[1], n2[2]);
    if (r1 > 0.0)
    {
      o1[0] = n1[0];
      o1[1] = n1[1];
    }
    if (r2 > 0.0)
    {
      o2[0] = n2[0];
      o2[1] = n2[1];
    }
    ang += 3.0;
  }
  if (EQ(ang, angle))
  {
    register double a;

    a = angle / 180.0 * PI;
    if (r1 > 0.0)
    {
      n1[0] = r1 * cos(a + angle1);
      n1[1] = r1 * sin(a + angle1);
    }
    if (r2 > 0.0)
    {
      n2[0] = r2 * cos(a + angle2);
      n2[1] = r2 * sin(a + angle1);
    }
    if (r1 < 0.0)
      fprintf(_ScnStream,
	      "triangle\n %g %g %g\n %g %g %g\n %g %g %g\n",
	      o2[0], o2[1], o2[2],
	      o1[0], o1[1], o1[2],
	      n2[0], n2[1], n2[2]);
    else if (r2 < 0.0)
      fprintf(_ScnStream,
	      "triangle\n %g %g %g\n %g %g %g\n %g %g %g\n",
	      o2[0], o2[1], o2[2],
	      o1[0], o1[1], o1[2],
	      n1[0], n1[1], n1[2]);
    else
      fprintf(_ScnStream,
	      "quadrangle\n %g %g %g\n %g %g %g\n %g %g %g\n %g %g %g\n",
	      o2[0], o2[1], o2[2],
	      o1[0], o1[1], o1[2],
	      n1[0], n1[1], n1[2],
	      n2[0], n2[1], n2[2]);
  }
}

void
ScnTransform(mat)
  RtMatrix        mat;
{
  if (EQ(mat[0][0], 1.0) && EQ(mat[0][1], 0.0) && EQ(mat[0][2], 0.0) &&
      EQ(mat[1][0], 0.0) && EQ(mat[1][1], 1.0) && EQ(mat[1][2], 0.0) &&
      EQ(mat[2][0], 0.0) && EQ(mat[2][1], 0.0) && EQ(mat[2][2], 1.0))
    {
      if (EQ(mat[0][3], 0.0) && EQ(mat[1][3], 0.0) && EQ(mat[2][3], 0.0))
        return;
      fprintf(_ScnStream, "transform translate %g %g %g\n",
              mat[0][3], mat[1][3], mat[2][3]);
      return;
    }
  fprintf(_ScnStream,
          "transform general\n %g %g %g\n %g %g %g\n %g %g %g\n %g %g %g\n",
	  mat[0][0], mat[0][1], mat[0][2],
	  mat[1][0], mat[1][1], mat[1][2],
	  mat[2][0], mat[2][1], mat[2][2],
          mat[0][3], mat[1][3], mat[2][3]);
}
