/************************************************************************
** MODULE INFORMATION*
**********************
**     FILE     NAME:       dpapps.c
**     SYSTEM   NAME:       PACKET DISPATCHER
**     ORIGINAL AUTHOR(S):  Ling Thio
**     VERSION  NUMBER:     v1.00
**     CREATION DATE:       1989/11/30
**
** DESCRIPTION: Dispatcher's application functions
**              
*************************************************************************
** CHANGES INFORMATION **
*************************
** REVISION:    $Revision:   1.1  $
** WORKFILE:    $Workfile:   DPAPPS.C  $
** LOGINFO:     $Log:   I:/ETSTJAN/CPROG/BEHOLDER/NPD/DP/VCS/DPAPPS.C_V  $
**              
**                 Rev 1.1   01 Feb 1991 14:21:10   etstjan
**              
**                 Rev 1.0   26 Oct 1990 12:27:58   etstjan
**              Initial revision.
*************************************************************************/
#if ! defined(PRD)
static char _pvcs_hdr[] =
"$Header:   I:/ETSTJAN/CPROG/BEHOLDER/NPD/DP/VCS/DPAPPS.C_V   1.1   01 Feb 1991 14:21:10   etstjan  $";
#endif
#define PROFILE

#include <string.h>                         /* for strncpy() */
#include <stdlib.h>                          /* NULL */
#include <stdio.h>
#include <error.h>
#include <futil.h>
#include "dpinc.h"                          /* main include file */

static int DpFreeTimeApps;
static int DpAppsCount;

int DpAppsInit (void)
{
    int Ret = 0;
    FILE *Cfg;
    char Variable[16];
    int Apps;
    DPAPPS *pApps;
    
    ERR_DEB(ERR_NPD, 4, "DpAppsInit");
    DpFreeTimeApps = 0;
    if ((Cfg = fopen (ConfigFile,"r")) == NULL)
    {
        ERR_ERR (ERR_NPD, ERR_CRITICAL, "Cannot open configfile\n");
        return(-1);
    }
    for (DpAppsCount=0; DpAppsActive[DpAppsCount]; DpAppsCount++);
    for (Apps=0; Apps<DpAppsCount; Apps++)
    {
        pApps=DpAppsActive[Apps];
        pApps->Status = DPE_END + DPE_STOP + DPE_HIDE;
        if (FFindSection (Cfg, pApps->Name) >= 0)
        {
            if (FGetVar (Cfg, "EVENTMASK", Variable, 15, UPC) >= 0)
                sscanf(Variable,"%i",&(pApps->EventMask));
            if (FGetVar (Cfg, "STARTMASK", Variable, 15, UPC) >= 0)
                sscanf(Variable,"%i",&(pApps->StartMask));
            if (FGetVar (Cfg, "TIMERMASK", Variable, 15, UPC) >= 0)
                sscanf(Variable,"%i",&(pApps->TimerMask));
            if (FGetVar (Cfg, "TIMERVALUE", Variable, 15, UPC) >= 0)
                sscanf(Variable,"%lu",&(pApps->TimerValue));
        }
        if (pApps->pProcessEventFunction!=NULL)
        {
            Ret |= DpAppsEvent(Apps, DPE_INIT);
        }
        else
        {
            pApps->StartMask = 0;   /* Don't start it */
            pApps->EventMask = 0;   /* Disable it */
            ERR_ERR(ERR_NPD,ERR_CRITICAL,"Application %s has no event function!",
                DpAppsActive[Apps]->Name);
        }
    }
    fclose (Cfg);
    return (Ret);
}


int DpAppsEnd (void)
{
    int Ret = 0;
    int Apps;
    
    ERR_DEB(ERR_NPD, 4, "DpAppsEnd");
    for (Apps=0; Apps<DpAppsCount; Apps++)
        Ret |= DpAppsEvent(Apps, DPE_END);
    return (Ret);
}


int DpAppsStart (void)
{
    int Ret = 0;
    int Apps;
    DPAPPS *pApps;

    ERR_DEB(ERR_NPD, 4, "DpAppsStart");
    for (Apps=0; Apps<DpAppsCount; Apps++)
    {
        pApps = DpAppsActive[Apps];
        if (pApps->StartMask & DPE_START)
            Ret |= DpAppsEvent (Apps, DPE_START);
        if (pApps->StartMask & DPE_RESET)
            Ret |= DpAppsEvent (Apps, DPE_RESET);
        if (pApps->StartMask & DPE_SHOW)
            Ret |= DpAppsEvent (Apps, DPE_SHOW);
        if (pApps->StartMask & DPE_FREETIME)
            Ret |= DpAppsEvent (Apps, DPE_FREETIME);
    }
    return (Ret);
}



int DpAppsStop (void)
{
    int Ret = 0;
    int Apps;
    DPAPPS *pApps;

    ERR_DEB(ERR_NPD, 4, "DpAppsStop");
    for (Apps=0; Apps<DpAppsCount; Apps++)
    {
        pApps = DpAppsActive[Apps];
        if (pApps->Status & DPE_SHOW)
            Ret |= DpAppsEvent(Apps, DPE_HIDE);
        if (pApps->Status & DPE_START)
            Ret |= DpAppsEvent(Apps, DPE_STOP);
    }
    return (Ret);
}

int DpAppsReceivePkt (void)
{
    int Ret = 0;
    int Apps;
    int Filt;
    DPAPPS *pApps;

    for (Apps=0; Apps<DpAppsCount; Apps++)
    {
        pApps=DpAppsActive[Apps];
        if ( (pApps->EventMask & DPE_RECEIVEPKT)
          && (pApps->Status & DPE_START) )
        {
            for (Filt=0;
                 Filt<pApps->FiltCount &&
                 DpFiltFilter (pApps->FiltActive[Filt]);
                 Filt++);
            if (Filt==pApps->FiltCount)
                Ret |= DpAppsEvent(Apps, DPE_RECEIVEPKT);
        }
    }
    return (Ret);
}


int DpAppsEverySecond (void)
{
    int Ret = 0;
    int Apps;
    DPAPPS *pApps;

    TRACECOUNT(1,79);
    for (Apps=0; Apps<DpAppsCount; Apps++)
    {
        pApps=DpAppsActive[Apps];
        if (pApps->Status & DPE_START)
        {
            Ret |= DpAppsEvent(Apps, DPE_EVERYSECOND);
        }
    }
    return (Ret);
}


int DpAppsFreeTime (void)
{
    int Ret = 0;
    DPAPPS *pApps;
    
    if (++DpFreeTimeApps==DpAppsCount)
        DpFreeTimeApps = 0;                        /* wrap around */
    pApps = DpAppsActive[DpFreeTimeApps];
    if (pApps->Status & DPE_START)
    {
        if (clock_ms()-pApps->Timer>pApps->TimerValue)
        {
            DpAppsTimer(DpFreeTimeApps);
            pApps->Timer=clock_ms();
        }
        Ret = DpAppsEvent(DpFreeTimeApps, DPE_FREETIME);
    }
    return (Ret);
}

int DpAppsTimer (int Apps)
{
    int Ret = 0;
    DPAPPS *pApps;

    pApps = DpAppsActive[Apps];
    ERR_DEB(ERR_NPD, 4, "DpAppsTimer for %s",pApps->Name);
    if (pApps->TimerMask & DPE_TIMER)
        Ret |= DpAppsEvent (Apps, DPE_TIMER);
    if (pApps->TimerMask & DPE_STOP)
        Ret |= DpAppsEvent (Apps, DPE_STOP);
    if (pApps->TimerMask & DPE_START)
        Ret |= DpAppsEvent (Apps, DPE_START);
    if (pApps->TimerMask & DPE_RESET)
        Ret |= DpAppsEvent (Apps, DPE_RESET);
    if (pApps->TimerMask & DPE_SHOW)
        Ret |= DpAppsEvent (Apps, DPE_SHOW);
    if (pApps->TimerMask & DPE_FREETIME)
        Ret |= DpAppsEvent (Apps, DPE_FREETIME);
    return (Ret);
}

int DpAppsEvent (int Apps, DPEVENT Event)
{
    int Ret = 0;
    DPAPPS *pApps = DpAppsActive[Apps];

    if (!(pApps->EventMask & Event) || (pApps->Status & Event))
        return (Ret);
#ifdef PROFILE
{
    unsigned long Spoiled;
    unsigned int  Index, Mask;
    
    for (Mask = Event, Index = 0; Mask >>= 1; Index++);
    TRACESCREEN(1+Apps,60+Index,'X');
    Spoiled = clock_ms ();      
    Ret = (*(pApps->pProcessEventFunction))(Event);
    pApps->Profile[Index] += clock_ms() - Spoiled;
    TRACESCREEN(1+Apps,60+Index,'O');
}
#else
    Ret = (*(pApps->pProcessEventFunction))(Event);
#endif
    switch (Event)    
    {    
            case DPE_START:
                pApps->Status |= DPE_START;
                pApps->Status &= ~DPE_STOP;
                break;
            case DPE_STOP:
                pApps->Status |= DPE_STOP;
                pApps->Status &= ~DPE_START;
                break;
            case DPE_SHOW:     
                pApps->Status |= DPE_SHOW;
                pApps->Status &= ~DPE_HIDE;
                break;
            case DPE_HIDE:      
                pApps->Status |= DPE_HIDE;
                pApps->Status &= ~DPE_SHOW;
                break;
            case DPE_INIT:
                pApps->Status |= DPE_INIT;
                pApps->Status &= ~DPE_END;
                break;
            case DPE_END:
                pApps->Status |= DPE_END;
                pApps->Status &= ~DPE_INIT;
                break;
    }
    if (Event & (DPE_INIT | DPE_START | DPE_RESET))
        pApps->TimerStart=clock_ms();
    if (Event & (DPE_STOP | DPE_END))
        pApps->TimerStop=clock_ms();
    return Ret;
}


/**************************************************************
** NAME:        DpAppsGetCount    
** SYNOPSIS:    int DpAppsGetCount (void)
** DESCRIPTION: Returns number of active applications.
** RETURNS:     number of active applications
**************************************************************/
int DpAppsGetCount (void)
{
    return (DpAppsCount);
}

/**************************************************************
** NAME:        DpAppsGetApps    
** SYNOPSIS:    int DpAppsGetApps (char *Name)
** DESCRIPTION: Returns the application id of application.
** RETURNS:     Id of application
**              -1 no application
**************************************************************/
int DpAppsGetApps (char *Name)
{
    int Apps;
    
    for (Apps=0; Apps<DpAppsCount; Apps++)
    {
        if (strcmp(DpAppsActive[Apps]->Name, Name)==0)
            return (Apps);
    }
    return (-1);
}

/**************************************************************
** NAME:        DpAppsGetName    
** SYNOPSIS:    char *DpAppsGetName (int Apps)
** DESCRIPTION: Returns name of application.
** RETURNS:     Name of application
**              NULL  no application
**************************************************************/
char *DpAppsGetName (int Apps)
{
    if (Apps<0 || Apps>=DpAppsCount)
        return (NULL);
    return (DpAppsActive[Apps]->Name);
}

/**************************************************************
** NAME:        DpAppsGetStatus    
** SYNOPSIS:    int *DpAppsGetStatus (int Apps)
** DESCRIPTION: Returns status of application.
** RETURNS:     Status of application
**              NULL no application
**************************************************************/
int *DpAppsGetStatus (int Apps)
{
    if (Apps<0 || Apps>=DpAppsCount)
        return (NULL);
    return (&(DpAppsActive[Apps]->Status));
}

/**************************************************************
** NAME:        DpAppsGetTimer    
** SYNOPSIS:    unsigned long DpAppsGetTimer (int Apps)
** DESCRIPTION: Returns mili seconds sinds last INIT, START or RESET
**              event for this application.
** RETURNS:     Timer of application
**              0 no application
**************************************************************/
unsigned long DpAppsGetTimer (int Apps)
{
    DPAPPS *pApps;

    if (Apps<0 || Apps>=DpAppsCount)
        return (0);
    pApps=DpAppsActive[Apps];
    if (pApps->Status & DPE_START)
        return (clock_ms()-pApps->TimerStart);
    else
        return (pApps->TimerStop-pApps->TimerStart);
}

/**************************************************************
** NAME:        DpAppsGetProfile    
** SYNOPSIS:    long *DpAppsGetProfile (int Apps)
** DESCRIPTION: Returns time spoiled by application for events.
**              All times are in musec.
** RETURNS:     Time spoiled by application for different events.
**              NULL  no application
**************************************************************/
long *DpAppsGetProfile (int Apps)
{
    if (Apps<0 || Apps>=DpAppsCount)
        return (NULL);
    return (DpAppsActive[Apps]->Profile);
}
