/************************************************************************
** MODULE INFORMATION*
**********************
**     FILE     NAME:       dpbuf.c
**     SYSTEM   NAME:       PACKET DISPATCHER
**     ORIGINAL AUTHOR(S):  Ling Thio
**     VERSION  NUMBER:     v1.00
**     CREATION DATE:       1989/11/30
**
** DESCRIPTION: Dispatcher's packet buffering functions
**              
*************************************************************************
** CHANGES INFORMATION **
*************************
** REVISION:    $Revision:   1.1  $
** WORKFILE:    $Workfile:   DPBUF.C  $
** LOGINFO:     $Log:   I:/ETSTJAN/CPROG/BEHOLDER/NPD/DP/VCS/DPBUF.C_V  $
**              
**                 Rev 1.1   01 Feb 1991 14:21:00   etstjan
**              
**                 Rev 1.0   26 Oct 1990 12:27:48   etstjan
**              Initial revision.
*************************************************************************/
#if ! defined(PRD)
static char _pvcs_hdr[] =
"$Header:   I:/ETSTJAN/CPROG/BEHOLDER/NPD/DP/VCS/DPBUF.C_V   1.1   01 Feb 1991 14:21:00   etstjan  $";
#endif
/*
    The buffer functions for the Packet DP
*/
#include <stdio.h>                          /* for NULL */
#include <dos.h>                            /* for _enable(), __disable() */
#include <error.h>
#include <bufm.h>
#include "dpinc.h"                          /* main include file */

#define STAT_SHORT  0x00
#define STAT_LONG   0x80
#define STAT_MALLOC 0x40

static DPBUF *AllocQueue(int n, int size, int mode);
static void FreeQueue(DPBUF *pB);

static DPBUF HeadQueue;

DPBUF *pTailQueue;        /* used in dprec.c   JvO */
DPBUF *pFirstFreeSmall;   /* used in dprec.c   JvO */
DPBUF *pFirstFreeLarge;   /* used in dprec.c   JvO */


/*************************************************************************
** NAME:        DpBufInit
** SYNOPSIS:    int DpBufInit(nSmall,SmallSize,nLarge,LargeSize)
**              unsigned nSmall     # of small buffers
**              unsigned SmallSize  Size of small buffer
**              unsigned nLarge     # of large buffers
**              unsigned LargeSize  Size of large buffer
** DESCRIPTION: Initializes the Packet DP's buffers.
**              Links the 'nSmall' buffers of size 'SmallSize'
**                into the 'BufFreeSmall' list.
**              Links the 'nLarge' buffers of size 'LargeSize'
**                into the 'BufFreeLarge' list.
** RETURNS:     0:              Successful call
**              DPERR_NOMEM:    Insufficient memory
** SEE ALSO:    DpBufEnd.
*************************************************************************/
int DpBufInit(nSmall, SmallSize, nLarge, LargeSize)
unsigned nSmall, SmallSize;
unsigned nLarge, LargeSize;
{
    ERR_DEB(ERR_NPD, 4, "DpBufInit");
    pFirstFreeSmall = AllocQueue(nSmall,SmallSize,STAT_SHORT);
    if (pFirstFreeSmall==NULL)
        return DPERR_NOMEM;
    pFirstFreeLarge = AllocQueue(nLarge,LargeSize,STAT_LONG);
    if (pFirstFreeSmall==NULL)
        return DPERR_NOMEM;
    HeadQueue.pNext = NULL;
    pTailQueue = &HeadQueue;
    return 0;
}


/**************************************************************
** NAME:        AllocQueue
** SYNOPSIS:    DPBUF *AllocQueue(n,size,mode)
**              int n,size,mode;
** DESCRIPTION: Allocate the queues for the dispatcher.
**              allocing large buffers is better than
**              many small buffers. (faster and less memory).
** SEE ALSO:    FreeQueue
** RETURNS:     ptr to first DPBUF, no error
**              NULL, otherwise
**************************************************************/
DPBUF *AllocQueue(n,size,mode)
int n,size,mode;
{
    DPBUF *bufptr, *FirstBuf;
    int i, maxparts;
    BYTE far *ptr;

    if (n>65000/sizeof(DPBUF))
        return NULL;
    if ((FirstBuf=bufptr=IPBufGet(n*sizeof(DPBUF)))==NULL)
        return NULL;
    maxparts=(int)(65000/size);
    for (i=0;i<n;i++)
        bufptr[i].Status = mode;
    while (n>0)
    {
        if (n<maxparts) maxparts=n;
        if ((ptr=IPBufGet(maxparts*size))==NULL)
        {
            IPBufFree(FirstBuf);
            return NULL;
        }
        bufptr->Status |= STAT_MALLOC;  /* This one points to a allocate block */
        for (i=0;i<maxparts;i++)
        {
            bufptr->pBuf = ptr;         /* Alloc part */
            bufptr->pNext = bufptr+1;   /* Point to next DPBUF */
            bufptr++;
            ptr+=size;          
        }
        n-=maxparts;                    /* Another maxparts done */
    }
    bufptr[-1].pNext = NULL;            /* Mark end of list */
    return FirstBuf;
}


/**************************************************************
** NAME:        FreeQueue
** SYNOPSIS:    void FreeQueue(pB)
**              DPBUF *pB;
** DESCRIPTION: Frees the memory used by the queue.
** SEE ALSO:    AllocQueue
** RETURNS:     void
**************************************************************/
void FreeQueue(pB)
DPBUF *pB;
{
    DPBUF *pBuf;

    while (pB)                 /* free all small buffers */
    {
        pBuf = pB;
        pB = pBuf->pNext;
        if (pBuf->Status & STAT_MALLOC)
            IPBufFree(pBuf->pBuf);
        IPBufFree(pBuf);
    }
}


/*************************************************************************
** NAME:        DpBufEnd
** SYNOPSIS:    void DpBufEnd()
** DESCRIPTION: Releases Small and Large DP buffers.
**                previously allocated by a call to DpBufInit.
** RETURNS:     void
** SEE ALSO:    DpBufInit.
*************************************************************************/
void DpBufEnd()
{
    ERR_DEB(ERR_NPD, 4, "DpBufEnd");
    DpBufReset();                         /* empty Packet Pending list */
    FreeQueue(pFirstFreeSmall);
    FreeQueue(pFirstFreeLarge);
}


/*************************************************************************
** NAME:        DpBufReset
** SYNOPSIS:    void DpBufReset()
** DESCRIPTION: Resets the Packet DP's buffers.
**                by removing any buffers from the Packet
**                Pending list, and adding them to
**                the BufFreeSmall and BufFreeSmall lists.
** RETURNS:     void
*************************************************************************/
void DpBufReset()
{
    DPBUF *pBuf;

    while (pBuf = DpBufGetUsed())
        DpBufFreeUsed(pBuf);
}


/*************************************************************************
** NAME:        DpBufGetUsed
** SYNOPSIS:    DPBUF *DpBufGetUsed()
** DESCRIPTION: Removes pkt buffer from the head of
**              Packet Pending list.
** RETURNS:     A pointer to the tail DP buffer or
**              (DPBUF *)0 if no more buffers are pending.
*************************************************************************/
DPBUF *DpBufGetUsed()
{
    DPBUF *pBuf;

    _disable();
    pBuf = HeadQueue.pNext;
    if (pBuf != NULL)
    {
        HeadQueue.pNext = pBuf->pNext;
        if (HeadQueue.pNext == NULL)             /* if used queue is empty */
            pTailQueue = &HeadQueue;                /* Tail points to head */
    }
    _enable();
    return pBuf;
}


/*************************************************************************
** NAME:        DpBufFreeUsed
** SYNOPSIS:    void DpBufFreeUsed(pBuf)
**              DPBUF *pBuf       Pointer to DP buffer handler
** DESCRIPTION: Frees the packet buffer 'pBuf'.
**              Adding them to the 'BufFreeSmall' or
**              'BufFreeLarge' list, depending on the buffer
**              status 'pBuf->Status'.
** RETURNS:     void
*************************************************************************/
void DpBufFreeUsed(pBuf)
DPBUF *pBuf;
{
    _disable();
    if (pBuf->Status & STAT_LONG)
    {
        pBuf->pNext = pFirstFreeLarge;
        pFirstFreeLarge = pBuf;
    }
    else
    {
        pBuf->pNext = pFirstFreeSmall;
        pFirstFreeSmall = pBuf;
    }
    _enable();
}


