/************************************************************************
** MODULE INFORMATION*
**********************
**     FILE     NAME:       DPINI.C
**     SYSTEM   NAME:       PACKET DISPATCHER
**     ORIGINAL AUTHOR(S):  Ling Thio
**     VERSION  NUMBER:     v1.00
**     CREATION DATE:       1989/11/30
**
** DESCRIPTION: Functions to parse the DP.INI file
**              
*************************************************************************
** CHANGES INFORMATION **
*************************
** REVISION:    $Revision:   1.1  $
** WORKFILE:    $Workfile:   DPUINI.C  $
** LOGINFO:     $Log:   I:/ETSTJAN/C600/BEHOLDER/NPD/DPU/VCS/DPUINI.C_V  $
**              
**                 Rev 1.1   17 Sep 1991 10:51:32   tirza
**              Added functions DpuIniCountDefs and DpuIniGetDefPairs.
**              Converted to new style function declarations. Cleaning-up.
**              
**                 Rev 1.0   25 Oct 1990 15:26:44   etstjan
**              Initial revision.
*************************************************************************/
#if ! defined(PRD)
static char _pvcs_hdr[] =
"$Header:   I:/ETSTJAN/C600/BEHOLDER/NPD/DPU/VCS/DPUINI.C_V   1.0   25 Oct 1990 15:26:44   etstjan  $";
#endif

#include <stdio.h>
#include <stdlib.h>                         /* for _searchenv() */
#include <string.h>                         /* for str*() */
#include <memory.h>                         /* for mem*() */
#include <malloc.h>                         /* for malloc(), free() */
#include <dp.h>                             /* for BYTE */
#include <dpu.h>


#define MAX_FDFIELDS        50
#define MAX_FIELDS          20
#define FIELDNAMELEN        10
#define COMMENT             (-2)


/* Function prototypes */
int DpuIniInit(char *initfile);
int DpuIniFindId(char *fname);
FIELDDEF *DpuIniFindDef(int id, BYTE *key);
int DpuIniCountDefs(int id);
void DpuIniGetDefPairs(int id, FIELDDEFPAIR *array);
void DpuIniGetKeys(int id, KEY *array);
void DpuIniGetDefs(int id, DESC *array);
void DpuIniEnd(void);
static int name_to_number(char **array, char *name);
static void add_field(FILE *fh);
#ifdef FUTURE
static void add_fdfield(FILE *fh);
#endif
static void add_def(FILE *fh, int id);
static void read_key(FILE *fh, int id, FIELDDEF *fielddef);


FIELDINFO fieldinfo[MAX_FIELDS];
FDFIELD fdfield[MAX_FDFIELDS];
static char *fdkey[] = {
    "FIELD",
    NULL,
};
static char *keyname[] = {
    "DEFINE",
    "FIELD",
    NULL,
};
static char *fieldnameptr[MAX_FIELDS] = {
    NULL
};
static char *fdtype[] = {
    "BIT",
    "NUM",
    "STR",
    NULL
};
static char fieldname[MAX_FIELDS][FIELDNAMELEN];
static int initialized = 0;
static int nfields = 0;
static int nfdfields = 0;
static char scratch[256];

/************************************************************************
** NAME:        DpuIniInit
** SYNOPSIS:    int DpuIniInit(char *initfile)
**              initfile = filename of INIT-file
** DESCRIPTION: Reads in the INIT-file 'initfile'
** RETURNS:     0:  Successful call.
**              -1: Can't open INIT-file.
*************************************************************************/
int DpuIniInit(char *initfile)
{
    FILE *fh;
    int id, firstword, c;
    char path[128];

    if (initialized)
        return 0;                           /* already initialized */
    initialized++;
    for (id=0; id<MAX_FIELDS; id++)
        fieldinfo[id].first = (FIELDDEF *)0;
    if (!(fh = fopen(initfile, "r")))
    {
        _searchenv(initfile, "INIT", path);
        fh = fopen(path, "r");
    }
    if (fh)
    {
        while (!feof(fh) && (fscanf(fh, "%s", scratch) == 1))
        {
            switch (firstword = name_to_number(keyname, scratch))
            {
                 case COMMENT:                   /* '#' */
                     while (((c = fgetc(fh)) != EOF) && (c != '\n'))
                         ;                       /* skip rest of line */
                     break;
                 case 0:                    /* New definition of field */
                     add_field(fh);
                     break;
#ifdef FUTURE
                 case 1:                     /* New definition of fdfield */
                     add_fdfield(fh);
                     break;
#endif
                 default:                        /* Field (key,desc) to add */
                     if (((id = name_to_number(fieldnameptr, scratch)) < nfields) && (id >= 0))
                         add_def(fh, id);
                     else
                         pw_warning("DpuIniInit: ID of fieldname %s in init file %s was too large", scratch, initfile);
                     break;
            }
        }
        fclose(fh);
        return 0;
    }
    else
        return -1;                          /* can't open INIT-file */
}

/************************************************************************
** NAME:        DpuIniFindId
** SYNOPSIS:    int DpuIniFindId(char *fname)
**              fname = fieldname
** DESCRIPTION: Rearches the id of a fieldname
** RETURNS:     id:  Successful call.
**              -1: Can't find fieldname.
*************************************************************************/
int DpuIniFindId(char *fname)
{
    return name_to_number(fieldnameptr, fname);
}

/************************************************************************
** NAME:        DpuIniFindDef
** SYNOPSIS:    FIELDDEF *DpuIniFindDef(int id, BYTE *key)
**              id = id of fieldname
**              key = search key
** DESCRIPTION: Rearches for 'key' in fielddef-list of id
** RETURNS:     fielddef:  Successful call.
*************************************************************************/
FIELDDEF *DpuIniFindDef(int id, BYTE *key)
{
    int found = 0;
    FIELDDEF *fielddef = (FIELDDEF *)0;

    if (id < nfields)
        fielddef = fieldinfo[id].first;    
    while ((fielddef) && (!found))
    {
        found = (memcmp(key, fielddef->key, fieldinfo[id].keylen) == 0);
        if (!found)
            fielddef = fielddef->next;
    }
    return fielddef;
}

/************************************************************************
** NAME:        DpuIniCountDefs
** SYNOPSIS:    int DpuIniCountDefs(int id)
**              id = id of fieldname
** DESCRIPTION: Counts the numbers of fielddefs in fielddef-list of id
** RETURNS:     ndefs:  Successful call.
*************************************************************************/
int DpuIniCountDefs(int id)
{
    FIELDDEF *tmp=fieldinfo[id].first;
    int ndefs = 0;

    while (tmp)
    {
        ndefs++;
        tmp=tmp->next;
    }
    return(ndefs);
}

/************************************************************************
** NAME:        DpuIniGetDefPairs
** SYNOPSIS:    void DpuIniGetDefPairs(int id, FIELDDEFPAIR *array);
**              id = id of fieldname
**              array = destination array for key-definition pairs
** DESCRIPTION: Gets key-definition pairs from fielddef-list of id
** RETURNS:     Nothing.
*************************************************************************/
void DpuIniGetDefPairs(int id, FIELDDEFPAIR *array)
{
    FIELDDEF *fielddef=fieldinfo[id].first;
    int i=0;

    while (fielddef)
    {
        memcpy(array[i].key, fielddef->key, fieldinfo[id].keylen);
        strcpy(array[i].desc, fielddef->desc);
        fielddef=fielddef->next;
        i++;
    }   
}

/************************************************************************
** NAME:        DpuIniGetKeys
** SYNOPSIS:    void DpuIniGetKeys(int id, KEY *array);
**              id = id of fieldname
**              array = destination array for keys
** DESCRIPTION: Gets keys from fielddef-list of id
** RETURNS:     Nothing.
*************************************************************************/
void DpuIniGetKeys(int id, KEY *array)
{
    FIELDDEF *fielddef=fieldinfo[id].first;
    int i=0;

    while (fielddef)
    {
        memcpy(array[i], fielddef->key, fieldinfo[id].keylen);
        fielddef=fielddef->next;
        i++;
    }   
}

/************************************************************************
** NAME:        DpuIniGetDefs
** SYNOPSIS:    void DpuIniGetDefs(int id, DESC *array);
**              id = id of fieldname
**              array = destination array for key-definition pairs
** DESCRIPTION: Gets key-definition pairs from fielddef-list of id
** RETURNS:     The destination array.
*************************************************************************/
void DpuIniGetDefs(int id, DESC *array)
{
    FIELDDEF *fielddef=fieldinfo[id].first;
    int i=1;

    while (fielddef)
    {
        strcpy(array[i], fielddef->desc);
        fielddef=fielddef->next;
        i++;
    }
}

/************************************************************************
** NAME:        DpuIniEnd                               
** SYNOPSIS:    void DpuIniEnd()                        
** DESCRIPTION: Deletes the fielddef-lists              
** RETURNS:     Nothing.                           
*************************************************************************/
void DpuIniEnd()                                        
{                                                       
    int id;                                       
    FIELDDEF *tmp;                                      
    FIELDDEF *next;                                     
                                                        
    for (id=0; id<MAX_FIELDS; id++)                     
    {                                                   
        tmp = fieldinfo[id].first;                      
        while (tmp)
        {
            next = tmp->next;
            free(tmp);
            tmp = next;
        }
    }
}

static int name_to_number(char *array[], char *name)
{
    int ret = 0;

    if (*name == '#')
        return COMMENT;
    while ((array[ret]) && (stricmp(name, array[ret])))
        ret++;
    if (array[ret])
        return ret;
    else
        return -1;
}

static void add_field(FILE *fh)
{
    int c;

    if (nfields < MAX_FIELDS-1)
    {
        fscanf(fh, "%s", scratch);
        strncpy(fieldname[nfields], scratch, FIELDNAMELEN);
        fscanf(fh, "%i %i %s", 
            &(fieldinfo[nfields].keylen),
            &(fieldinfo[nfields].desclen),
            scratch);
        if (fieldinfo[nfields].keylen > MAX_FIELDKEYLEN)
        {
            pw_warning("DpuIniInit: keylength of fieldname %s in init file too large", scratch);
            fieldinfo[nfields].keylen = MAX_FIELDKEYLEN;
        }
        if (fieldinfo[nfields].desclen > MAX_FIELDDESCLEN)
        {
            pw_warning("DpuIniInit: descriptionlength of fieldname %s in init file too large", scratch);
            fieldinfo[nfields].keylen = MAX_FIELDDESCLEN;
        }
        fieldinfo[nfields].hex = 0;
        if (stricmp(scratch, "HEX") == 0)
            (fieldinfo[nfields].hex)++;
        fieldnameptr[nfields] = fieldname[nfields];
        fieldnameptr[++nfields] = NULL;
    }
    else
    {
        fscanf(fh, "%s", scratch);
        pw_warning("DpuIniInit: ID of fieldname %s in init file too large", scratch);
        fscanf(fh, "%i %i %s", &c, &c, scratch);
    }
}


#ifdef FUTURE
static void add_fdfield(FILE *fh)
{
    register FDFIELD *field;
    int c;

    if (nfdfields < MAX_FDFIELDS-1)
    {
        fscanf(fh, "%i", &c);
        field = &(fdfield[c]);
        fscanf(fh, "%s", scratch);
        if ((field->type = name_to_number(fdtype, scratch)) < 0)
            pw_warning("DpuIniInit: Invalid field definition type '%s'", scratch);
        fscanf(fh, "%i", &(field->off));
        if (fgetc(fh) == '.')
            fscanf(fh, "%i", &(field->offbit));
        else
            field->offbit = 0;
        fscanf(fh, "%i", &(field->len));
        if (fgetc(fh) == '.')
            fscanf(fh, "%i", &(field->lenbit));
        else
            field->lenbit = 0;
        nfdfields++;
    }
    else
    {
        fscanf(fh, "%i", &c);
        pw_warning("DpuIniInit: ID of field %i in init file %s too large", c, initfile);
        fscanf(fh, "%i %i %s", &c, &c, scratch);
    }
}
#endif

static void add_def(FILE *fh, int id)
{
    int i;
    FIELDDEF *fielddef = (FIELDDEF *)malloc(sizeof(FIELDDEF));

    if (!fielddef)
    {
        pw_warning("DpuIniInit: Not enough memory too add new definition");
        return;
    }
    read_key(fh, id, fielddef);
    fscanf(fh, "%s", scratch);
    for (i=strlen(scratch); i<fieldinfo[id].desclen; i++)
        scratch[i] = ' ';
    scratch[fieldinfo[id].desclen] = '\0';
    strcpy(fielddef->desc, scratch);
    fielddef->next = fieldinfo[id].first;
    fieldinfo[id].first = fielddef;
}

static void read_key(FILE *fh, int id, FIELDDEF *fielddef)
{
    int i;
    int num;

    for (i=0; i<fieldinfo[id].keylen; i++)
    {
        fscanf(fh, "%i", &num);
        fielddef->key[i] = (BYTE)num;
    }
}
