/* [DOC] File ****************************************************
** MODULE INFORMATION*
**********************
**      FILE    NAME:       PWMENU.C
**      SYSTEM  NAME:       POWER
**      VERSION NUMBER      3.00
**
** Descr:       menu, choice and select functions for
**              Programmer Oriented Window Environment Routines.
**              (NoC) July 1989 - Ling Thio Software Productions.
******************************************************************
** ADMINISTRATIVE INFORMATION*
******************************
**      ORIGINAL AUTHOR:    Ling Thio
**      CREATION DATE:      89/07/10
**      CHANGES:            none
*****************************************************************/

#include <string.h>                         /* for str*() */
#include <ctype.h>                          /* for toupper() */
#include <conio.h>                          /* for kbhit() */
#include "pwinc.h"                          /* Main include file */

extern  int _pwv_scrnrow;

#define MENUTYPE    -2
#define SELECTTYPE  -1

typedef struct _PWM {
    int topelem;
    int element;
    int line;
    int nelem;
    int type;
    char **textarray;
    char far * far *fartextarray;
    int (*process)(int element);
} PWM;

/*
fartextarray:
    A far pointer to: an array of
                      far pointers to: Structures
                                       with a char *
                                       at offset OFFS to: A character string
    ----------------- ---------------- ------------------ ------------------
    fartextarray -->  [0]
                      [1]  --------->  Struct bla {                 ^
                      ...                 int a                     | OFFS
                      [n]                 ...                       | bytes
                      NULL                char *Name -->  "Quit"    V
                                          ...
                                       }                       
*/

static PWM *pwm;

static int pw_proc_menu(PWWIN *win, int action, int element);
static void pw_display_menu(PWWIN *win, int (*proc_select)(PWWIN *, int, int));
static void pw_display_menubar(PWWIN *win);
static PWKEY pw_men(int srow, int scol, int nrow, int ncol, char *header,
                    int (*proc_action)(PWWIN *, int, int),
                    int (*process)(int), int *element);


/* [DOC] Function *********************************************
** Name:        pw_cls                                    [API]
** SYNOPSIS:    void pw_cls(win);
**              PWWIN *win          Pointer to window handle
** Descr:       Clears a window.
**              The 'pw_cls' function clears the window 'win'.
**                The attribute used is PWT_NORMAL.
**              The cursor position is set to the top left.
** RETURNS:     void
** SEE ALSO:    pw_type
**************************************************************/
void pw_cls(PWWIN *win)
{
    pw_vscroll(win, 0, 0, 0, 0, 0);
    win->crow = win->ccol = 0;
}

/* [DOC] Function *********************************************
** Name:        pw_menu                                   [API]
** SYNOPSIS:    PWKEY pw_menu(srow, scol, nrow, ncol,
**                           header, text, proc_menu, element)
**              int  srow           Top left row
**              int  scol           Top left column
**              int  nrow           Menu window height
**              int  ncol           Menu window width
**              char *header        Menu header
**              char **text         Array of character strings
**              int  (*proc_menu) Function to process key actions.
**              int  *element       Start element number.
** Descr:       Executes a menu window.
**              The 'pw_menu' function opens a menu window of
**                on the coordinates given by 'scol', 'srow',
**                'ncol' and 'nrow'.  It displays 'header' on
**                the top border and fills the window with
**                the elements of 'text'.
**              The reversed selection bar is then placed on 
**                element '*element', or on element 0 if NULL.
**              The contents of the menu is determined by 'text',
**                which points to a NULL terminated constant
**                array of character strings.
**              For variable arrays, where elements can be added
**                 and deleted, pw_select() must be used.
**              The paremeters 'nrow' and 'ncol' may be set to
**                0; Menu window will be as large as needed or
**                as large as possible.
**              If 'nrow' is smaller than the number of strings
**                in 'text', the menu will scroll when needed.
**              The function 'proc_menu' must be provided by the
**                programmer and is called when the user presses
**                the <Enter> key.
**                  PWKEY proc_menu(element)
**                  int element    The current element of 'text'.
**              After 'proc_menu' processes the menu selection, it
**                can return:
**                >0: the element number where 'pw_menu'
**                    has to continue.
**                PWA_EXITMENU:   exit 'pw_menu' immediately.
**                PWA_UPDATELINE: re-display current menu element.
**                PWA_UPDATEMENU: re-display entire menu.
** RETURNS:     PWK_ENTER:  'proc_menu' has returned -1
**              PWK_ESC:    The menu function has been terminated
**                normally using the <Escape> key.
**              The variable '*element' will contain the selected
**                element number when pw_menu() terminated.
** SEE ALSO:    pw_choice, pw_select
** EXAMPLE:     static char main_menu[] = {
**                  "Add",
**                  "Edit",
**                  "Delete",
**                  "Quit",
**                  NULL
**              };
**                  ....
**              int proc_main_menu(element)
**              int element;
**              {
**                  switch(element) {
**                  case 0:
**                      .. add element ..
**                      .. sort elements ..
**                      element = .. new element's place ..
**                      break;
**                  case 1:
**                      .. edit element ..
**                      break;
**                  case 2:
**                      .. delete element ..
**                      break;
**                  case 3:
**                      return -1       ...close immediately)...
**                  }
**                  return element;
**              }
**                  ....
**              pw_menu(20, 6, PWC_AUTO, PWC_AUTO,
**                  "Action", proc_main_menu);
**                  ....
**************************************************************/
PWKEY pw_menu(int srow, int scol, int nrow, int ncol, char *header,
              char **text, int (*proc_menu)(int), int *element)
{
    int elem = 0;
    int ret;
    PWM *old_pwm = pwm;                     /* make reentrant */
    PWM new_pwm;

    new_pwm.textarray = text;
    new_pwm.type = MENUTYPE;
    pwm = &new_pwm;                         /* make reentrant */
    if (ncol == PWC_AUTO)
    {
        while (text[elem] != NULL) 
        {
            if ((int)(strlen(text[elem])) > ncol)
                ncol = strlen(text[elem]);
            elem++;
        }
        ncol += 2;
    }
    ret = pw_men(srow, scol, nrow, ncol, header, pw_proc_menu, proc_menu, element);
    pwm = old_pwm;                          /* make reentrant */
    return ret;
}

/* [DOC] Function *********************************************
** Name:        pw_select                                 [API]
** SYNOPSIS:    PWKEY pw_select(srow, scol, nrow, ncol,
**                             header, proc_select, element)
**              int  srow           Top left row
**              int  scol           Top left column
**              int  nrow           Menu window height
**              int  ncol           Menu window width
**              char *header        Menu header
**              int  (*proc_select) Programmer function
**              int  *element       Start element number.
** Descr:       Executes a selection window.
**              The function 'pw_select' is a more advanced
**                version of 'pw_menu()'.  The function
**                'proc_select' is not only called when the
**                <Enter> key is hit (PWA_ENTER), but also
**                during keypresses of <Insert> (PWA_INSERT)
**                and <Delete> (PWA_DELETE), during menu
**                initialization (PWA_INIT), to count the
**                menu elements (PWA_COUNT), to display the
**                menu elements (PWA_DISPLAY), and to
**                handle element selection through first
**                letters (PWA_FIRSTCHAR)
**              The paremeters 'nrow' and 'ncol' may be set to
**                0; Menu window will be as large as needed or
**                as large as possible.
**              The programmer must provide the function
**                'proc_select',  which must be able to process
**                several action requests made by 'pw_select'.
**                  proc_select(win, action, element)
**                  PWWIN *win      Pointer to select window
**                  int   action    'pw_select' request type
**                  int   element   Action element
**              The 'proc_select' must perform the following
**                following things during the following actions:
**              Action          Description
**              -----------------------------------------------
**              PWA_DISPLAY     Display data element 'element'.
**              PWA_ENTER       Process <Enter>  on 'element'.
**              PWA_INSERT      Process <Insert> on 'element'.
**              PWA_DELETE      Process <Delete> on 'element'.
**              PWA_COUNT       Return number of elements.
**              PWA_FIRSTCHAR   Return first char of 'element'.
**              PWA_INIT        Return the start element number
**                              where pw_select will start.
** RETURNS:     PWK_ENTER:  'proc_menu' has returned -1
**              PWK_ESC:    The menu function has been terminated
**                normally using the <Escape> key.
**              The variable '*element' will contain the selected
**                element number when pw_menu() terminated.
** SEE ALSO:    pw_choice, pw_menu
**************************************************************/
PWKEY pw_select(int srow, int scol, int nrow, int ncol, char *header,
                int (*proc_select)(PWWIN *, int, int), int *element)
{
    PWM *old_pwm = pwm;                     /* make reentrant */
    PWM new_pwm;
    int ret;

    new_pwm.type = SELECTTYPE;
    pwm = &new_pwm;
    ret = pw_men(srow, scol, nrow, ncol, header, proc_select, NULL, element);
    pwm = old_pwm;                          /* make reentrant */
    return ret;
}

PWKEY pw_selectstruct(int srow, int scol, int nrow, int ncol, char *header,
                      void **structarray, void far * far *farstructarray,
                      int (*proc_select)(PWWIN *, int, int),
                      int offset, int *element)
{
    PWM *old_pwm = pwm;                     /* make reentrant */
    PWM new_pwm;
    int ret;

    new_pwm.textarray = structarray;
    new_pwm.fartextarray = farstructarray;
    new_pwm.type = offset;
    pwm = &new_pwm;
    ret = pw_men(srow, scol, nrow, ncol, header, proc_select, NULL, element);
    pwm = old_pwm;                          /* make reentrant */
    return ret;
}

/* [DOC] Function *********************************************
** Name:        pw_choice                                 [API]
** SYNOPSIS:    PWKEY pw_choice(srow, scol, nrow, ncol,
**                              header, text, element)
**              int  srow           Top left row
**              int  scol           Top left column
**              int  nrow           Menu window height
**              int  ncol           Menu window width
**              char *header        Menu header
**              char **text         Array of character strings
**              int  *element       Start element number
** Descr:       Executes a choice window.
**              The 'pw_choice' function is a simplified version
**                of 'pw_menu'.  It will open a menu, with the
**                reversed bar on element '*element' or on 0 if
**                'element' == NULL.  The user selects the 
**                element of his choosing and then hits the 
**                <Enter> or <Escape> key.
**              <Enter>: the selected element number will be
**                returned in '*element'.
**                The function returns with PWK_ENTER.
**              <Escape>: '*element' will not be changed.
**                The function returns with PWK_ESC.
**              The paremeters 'nrow' and 'ncol' may be set to
**                0; Menu window will be as large as needed or
**                as large as possible.
** RETURNS:     PWK_ENTER: User has made a selection, 
**                element number returned in '*element'.
**              PWK_ESC:   User has canceled,
**                '*element' unchanged.
** EXAMPLE:     static char monitor_menu[] = {
**                  "CGA+color",
**                  "CGA+mono",
**                  "MDA+mono",
**                  NULL
**              };
**              static int mon_type = 1;
**                  ....
**              pw_choice(10, 5, 0, 0, "Monitor", 
**                        monitor_menu, &mon_type);
**                  ....
** SEE ALSO:    pw_menu, pw_select
**************************************************************/
PWKEY pw_choice(int srow, int scol, int nrow, int ncol, char *header,
                char **text, int *element)
{
    int elem = 0;
    PWM *old_pwm = pwm;                     /* make reentrant */
    PWM new_pwm;
    int ret;

    new_pwm.textarray = text;
    new_pwm.type = MENUTYPE;
    pwm = &new_pwm;                         /* make reentrant */
    if (ncol == PWC_AUTO)
    {
        while (text[elem] != NULL) 
        {
            if ((int)(strlen(text[elem])) > ncol)
                ncol = strlen(text[elem]);
            elem++;
        }
        ncol += 2;
    }
    elem = (element) ? *element : 0;
    if ((ret = pw_men(srow, scol, nrow, ncol, header, pw_proc_menu, NULL, &elem)) == PWK_ENTER)
        if (element)
            *element = elem;
    pwm = old_pwm;                          /* make reentrant */
    return ret;
}

static int pw_proc_menu(PWWIN *win, int action, int element)
{
    int ret = element;
    char far *fptr = 0;
    char *ptr;
    int i = 0;

    switch(action)
    {
    case PWA_FIRSTCHAR:
        if (pwm->type == MENUTYPE)
            ptr = (pwm->textarray[element]);
        else
        {
            if (pwm->textarray)
                fptr = (char far *)(pwm->textarray[element])+pwm->type;
            else
            {
                if (pwm->fartextarray)
                    fptr = (pwm->fartextarray[element])+pwm->type;
            }
            ptr = *((char * far *)fptr);
        }
        ret = (unsigned int)*ptr;
        break;
    case PWA_DISPLAY:
        if (pwm->type == MENUTYPE)
            ptr = (pwm->textarray[element]);
        else
        {
            if (pwm->textarray)
                fptr = (char far *)(pwm->textarray[element])+pwm->type;
            else
            {
                if (pwm->fartextarray)
                    fptr = (pwm->fartextarray[element])+pwm->type;
            }
            ptr = *((char * far *)fptr);
        }
        if (ptr)
        {
            while ((i<win->ncol-1) && (*ptr))   /* print text */
            {
                pw_putc(win, *(ptr++));
                i++;
            }
            while (i<win->ncol-1)               /* print trailing spaces */
            {
                pw_putc(win, ' ');
                i++;
            }
        }
        break;
    case PWA_ENTER:
        if (pwm->process)                   /* pw_menu(), pw_select() */
            ret = (*pwm->process)(element);
        else                                /* pw_choice() */
            ret = -1;
        break;
    case PWA_INIT:
        break;
    case PWA_COUNT:
        ret = 0;
        if (pwm->textarray)
        {
            while (pwm->textarray[ret] != NULL)
                ret++;
        }
        else
        {
            if (pwm->fartextarray)
                while (pwm->fartextarray[ret] != NULL)
                    ret++;
        }
        break;
    }
    return ret;
}

static void pw_display_menubar(PWWIN *win)
{
    int maxrow = (pwm->nelem < win->nrow) ? pwm->nelem : win->nrow;

    if (!maxrow)
        return;
    if (!(pwm->topelem))                    /* top menu bar */
        pw_putcat(win, 0, -1, (int)'');
    else
        pw_putcat(win, 0, -1, 30);       /* arrow up */

    if (pwm->topelem == pwm->nelem - maxrow)/* bottom menu bar */
    {
        if (maxrow == 1)
            pw_putcat(win, maxrow-1, -1, (int)'');
        else
            pw_putcat(win, maxrow-1, -1, (int)'');
    }
    else
        pw_putcat(win, maxrow-1, -1, 31); /* arrow down */
    if (maxrow > 2)                         /* in case of add element */
        pw_putcat(win, maxrow-2, -1, '');
}

static void pw_display_menu(PWWIN *win, int (*proc_action)(PWWIN *, int, int))
{
    int i;
    int maxrow = (pwm->nelem < win->nrow) ? pwm->nelem : win->nrow;

    if (maxrow < win->nrow)
        pw_putcat(win, maxrow, -1, _pwcf_border[PWCF_V]);
    pwm->line = pwm->element - pwm->topelem;
    if (maxrow)                             /* # elements > 0 */
    {
        for (i=0; i<maxrow; i++)
        {
            pw_putcat(win, i, -1, '');
            win->ccol = 1;
            win->crow = i;
            (void)(*proc_action)(win, PWA_DISPLAY, pwm->topelem+i);
        }
        pw_display_menubar(win);
        pw_block(win, 0, 1, maxrow, 1, PWT_HIGHLIGHT);
    }
    pw_block(win, pwm->line, 0, 1, 0, PWT_REVERSE);
}

static PWKEY pw_men(int srow, int scol, int nrow, int ncol, char *header,
                    int (*proc_action)(PWWIN *, int, int),
                    int (*process)(int), int *element)
{
    int i;
    int o_topelem;
    int o_nelem;
    int key;
    PWWIN * menu = NULL;
    int ret;
    int (*proc_specialaction)(PWWIN *, int, int) = (pwm->type < 0) ? proc_action : pw_proc_menu;

    pwm->process = process;
    if (nrow == PWC_AUTO)
        nrow = (*proc_specialaction)(menu, PWA_COUNT, 0);
    if (nrow > (_pwv_scrnrow-2) - srow)
        nrow = (_pwv_scrnrow-2) - srow;
    if (nrow == 0)
        nrow++;
    menu = pw_open(srow, scol, nrow, ncol, header, PWM_HIDE, PWW_MENU);
    pw_keyinfo(menu, NULL, "Letter,Cursor=Select  Enter=Accept");
    pwm->nelem = (*proc_specialaction)(menu, PWA_COUNT, 0);
    pwm->line = -1;                          /* first time */
    if (element && ((*element < 0) || (*element >= pwm->nelem)))
        *element = 0;
    pwm->element = (element) ? *element : 0;
    (*proc_action)(menu, PWA_INIT, 0);
    if ((pwm->element > pwm->nelem-1) && (pwm->nelem))
        pwm->element = pwm->nelem-1;
    pwm->topelem = 0;
    if (pwm->element < pwm->topelem)
        pwm->topelem = pwm->element;
    if ((pwm->element > pwm->topelem + nrow - 1) && (pwm->nelem))
        pwm->topelem = pwm->element - nrow + 1;
    pw_display_menu(menu, proc_specialaction);
    pw_show(menu);
    while ((key = pw_getkey()) != PWK_ESC)
    {
        pw_pop(menu);
        o_topelem = pwm->topelem;
        o_nelem = pwm->nelem;
        ret = -99;
        switch(key) 
        {
        case PWK_HOME:
            pwm->element = 0;
            break;
        case PWK_UP:
            pwm->element--;
            break;
        case PWK_PGUP:
            if (pwm->element != pwm->topelem)
                pwm->element = pwm->topelem;
            else
                pwm->element = pwm->element - nrow;
            break;
        case PWK_DOWN:
            pwm->element++;
            break;
        case PWK_END:
            pwm->element = pwm->nelem-1;
            break;
        case PWK_PGDN:
            if (pwm->element != pwm->topelem + nrow - 1)
                pwm->element = pwm->topelem + nrow - 1;
            else
                pwm->element = pwm->element + nrow;
            break;
        case PWK_INSERT:
            ret = (*proc_action)(menu, PWA_INSERT, pwm->element);
            pwm->nelem = (*proc_specialaction)(menu, PWA_COUNT, 0);
            break;
        case PWK_DEL:
            if (pwm->nelem)
            {
                ret = (*proc_action)(menu, PWA_DELETE, pwm->element);
                pwm->nelem = (*proc_specialaction)(menu, PWA_COUNT, 0);
                if ((pwm->nelem) && (pwm->element > pwm->nelem-1))
                    pwm->element = pwm->nelem-1;
                pw_pop(menu);
            }
            break;
        case PWK_ENTER:
            if (!pwm->nelem)
            {
                ret = -1;
                break;
            }
            i = pwm->element;
            pwm->process = process;
            ret = (*proc_action)(menu, PWA_ENTER, pwm->element);
            pwm->nelem = (*proc_specialaction)(menu, PWA_COUNT, 0);
            break;
//        default:
//            ret = (*proc_specialaction)(menu, PWA_KEY, pwm->element, key);
//            break;
        }
        switch (ret)
        {
        case PWA_EXITMENU:   /* exit immediately */
            pw_close(menu);
            if (element)
                *element = pwm->element;
            return PWK_ENTER;
        case PWA_UPDATELINE: /* re-paint menu element */
            menu->ccol = 1;
            menu->crow = pwm->line;
            pw_pop(menu);
            (void)(*proc_specialaction)(menu, PWA_DISPLAY, pwm->element);
            break;
        case PWA_UPDATEMENU: /* re-paint entire menu */
            o_nelem = -1;
            break;
        case -99:
            break;
        default:
            pwm->element = ret;
            break;
        }
        if (pwm->element > pwm->nelem-1)
            pwm->element = pwm->nelem-1;
        if (pwm->element < 0)
            pwm->element = 0;
        if (key < 256)
            key = toupper(key);
        if ((key >= 'A') && (key <= 'Z') && (pwm->nelem))
        {
            i = pwm->element;
            do 
            {
                i++;
                if (i == pwm->nelem)
                    i = 0;
            } while ((i != pwm->element) &&
                     (toupper((*proc_specialaction)(menu, PWA_FIRSTCHAR, i)) != key));
            if (toupper((*proc_specialaction)(menu, PWA_FIRSTCHAR, i)) == key)
                pwm->element = i;
        }
        if (pwm->element < pwm->topelem) 
            pwm->topelem = pwm->element;
        if ((pwm->element > pwm->topelem + nrow - 1) && (pwm->nelem))
            pwm->topelem = pwm->element - nrow + 1;
        if ((pwm->topelem <= o_topelem-nrow) ||
            (pwm->topelem >= o_topelem+nrow) ||
            (pwm->nelem != o_nelem))
        {
            pw_cls(menu);
            pw_display_menu(menu, proc_specialaction);
        }
        else
        {
            if ((i=pwm->topelem-o_topelem) ||
                (pwm->line != pwm->element - pwm->topelem))
            {
                pw_block(menu, pwm->line, 0, 1, 0, PWT_NORMAL);
                pw_block(menu, pwm->line, 1, 1, 1, PWT_HIGHLIGHT);
            }
            if (i < 0)
            {
                for (; i; i++)
                {
                    pwb_scroll_down(menu, 0, 0, menu->nrow, menu->ncol, 1);
                    menu->ccol = 1;
                    menu->crow = 0;
                    (void)(*proc_specialaction)(menu, PWA_DISPLAY, pwm->topelem-i-1);
                    pw_block(menu, 0, 1, 1, 1, PWT_HIGHLIGHT);
                }
            }
            if (i > 0)
            {
                for (; i; i--)
                {
                    pwb_scroll_up(menu, 0, 0, menu->nrow, menu->ncol, 1);
                    menu->ccol = 1;
                    menu->crow = nrow-1;
                    (void)(*proc_specialaction)(menu, PWA_DISPLAY, pwm->topelem+nrow-i);
                    pw_block(menu, nrow-1, 1, 1, 1, PWT_HIGHLIGHT);
                }
            }
            pw_display_menubar(menu);
        }
        pwm->line = pwm->element - pwm->topelem;
        pw_block(menu, pwm->line, 0, 1, 0, PWT_REVERSE);
    }
    pw_close(menu);
    if (element)
        *element = pwm->element;
    return PWK_ESC;
}

