/************************************************************************
** MODULE INFORMATION*
**********************
**     FILE     NAME:       IPINIT.C
**     SYSTEM   NAME:       IP
**     ORIGINAL AUTHOR(S):  Wim van Campen
**     VERSION  NUMBER:     1.0
**     CREATION DATE:       1990/10/10
**
** DESCRIPTION:  Initialization module for the IP stack
**              
*************************************************************************
** CHANGES INFORMATION **
*************************
** REVISION:    $Revision:   1.1  $
** WORKFILE:    $Workfile:   IPINIT.C  $
** LOGINFO:     $Log:   I:/ETSTJAN/CPROG/BEHOLDER/UDPIP/IP/VCS/IPINIT.C_V  $
**              
**                 Rev 1.1   21 Nov 1990 14:23:46   etstjan
**              No explicit note
**              
**                 Rev 1.0   20 Nov 1990 16:14:46   etstjan
**              No explicit note
*************************************************************************/
#if ! defined(PRD)
static char _pvcs_hdr[] =
"$Header:   I:/ETSTJAN/CPROG/BEHOLDER/UDPIP/IP/VCS/IPINIT.C_V   1.1   21 Nov 1990 14:23:46   etstjan  $";
#endif

#include      <stdio.h>                /* for NULL                     */
#include      <stdlib.h>
#include      <string.h>
#include      <stdarg.h>
#include      <power3.h>               /* for POWER screen functions   */
#include      <beholder.h>

#include      "ip.h"                   /* general ip defines           */
#include      "ipcodes.h"              /* and ip return codes          */
#include      "iplib.h"                /* ip library functions         */
#include      "ipif.h"                 /* interface layer              */
#include      "ipevents.h"             /* event processing             */
#include      "ipether.h"              /* ethernet interface           */
#include      "iplayer.h"              /* for initialization IP layer  */
#include      "socklib.h"              /* idem, socket layer           */

/* global variables */
PWWIN *IPWin;                         /* IP messages window pointer        */

/* prototypes of static functions */
static void PrErr(PWWIN *ThisWin, char *Message, ...);


BYTE  SpecTokens[]    = "\x1\x1NETSTAT\x1"       /* \x1 is for preventing  */
                        "\x2\x1NETMODIFY\x1"     /* other characters to    */
                        "\x3\x1HOSTSTAT\x1"      /* concatenate with       */
                        "\x4\x1HOSTMODIFY\x1";   /* strings                */

NETHOST  SpecArray[] = {
           NETSTAT,                    /* dummy entry                      */
           NETSTAT,                    /* index corresponds with the       */
           NETMODIFY,                  /* number specified in the          */
           HOSTSTAT,                   /* SpecTokens array                 */
           HOSTMODIFY
           };


/**************************************************************
** NAME:        IPStackInit
** SYNOPSIS:    int IPStackInit(void);
**
** DESCRIPTION: Initializes all of the IP environment.
**              Reads settings from file 'ConfigFile'.
**              If an error occures, a message is printed
**              in the IP window, if opened. In this case,
**              an error code is returned.
** RETURNS:     0 -->   no error
**              else    error code
**************************************************************/
int IPStackInit(void)
{
  int        LineNr, UsRet, i;
  long       TmpStor;
  char       ConfBuf[255];
  char       *Token, *Posit;
  USHORT     BufSize = 0x8000;
  int        BufNr = 1;
  FILE       *ConFile;
  NETHOST    RoutKind;
  ADDRESS_T  HostAddress[2];

#ifdef DEBUGANY
  if ((IPWin = pw_open(18, 1, 5, 58, "IP Messages",
                       PWM_DEFAULT, PWW_NORMAL)) == NULL) {
    return NOSPACE;
    }
#endif

  if ((ConFile = fopen(ConfigFile, "r")) == NULL) {
    PrErr(IPWin, "\nError: initialization file '%s' not found.", ConfigFile);
    return NOT_FOUND;
    }

  if (FFindSection(ConFile, "IPDOS") < 0) {
    PrErr(IPWin, "\nError: section '[IPDOS]' not found.");
    fclose(ConFile);
    return NOT_FOUND;
    }

  if ((UsRet = InitSockets()) != NO_ERR) {
    PrErr(IPWin, "\nError: initializing Socket Library. Code: %d.", UsRet);
    fclose(ConFile);
    return UsRet;
    }

  if ((UsRet = InitIPLayer()) != NO_ERR) {
    PrErr(IPWin, "\nError: initializing IP Layer. Code: %d.", UsRet);
    fclose(ConFile);
    return UsRet;
    }

  if ((UsRet = InitIfLayer()) != NO_ERR) {
    PrErr(IPWin, "\nError: initializing Interface Layer. Code: %d.", UsRet);
    fclose(ConFile);
    return UsRet;
    }

  /* default routing selection for packets is 'DONT_ROUTE' */
  if ((LineNr = FGetVar(ConFile, "forwarding", ConfBuf, 255, UPC)) > 0) {
    if (strncmp("YES", ConfBuf, 3) == 0) {
      RouteChoice = DO_ROUTE;
      }
    else {
      if (strncmp("NO", ConfBuf, 2) != 0) {
        PrErr(IPWin, "\nError: illegal value for RoutePackets. Taking default.");
        }
      }
    }

  /* default routing selection for source routed packets is 'DO_ROUTE' */
  if ((LineNr = FGetVar(ConFile, "sourceroute", ConfBuf, 255, UPC)) > 0) {
    if (strncmp("NO", ConfBuf, 2) == 0) {
      RouteChoice = DONT_ROUTE;
      }
    else {
      if (strncmp("YES", ConfBuf, 3) != 0) {
        PrErr(IPWin, "\nError: illegal value for SourceRoute. Taking default.");
        }
      }
    }

  if ((LineNr = FGetVar(ConFile, "nd0address", ConfBuf, 255, UPC)) < 0) {
    PrErr(IPWin, "\nError: nd0address not specified.");
    fclose(ConFile);
    return ND0ERROR;
    }

  if (DotConvert(HostAddress, ConfBuf) != 0) {
    PrErr(IPWin, "\nError: nd0address not correct.");
    fclose(ConFile);
    return ND0ERROR;
    }

  if ((LineNr = FGetVar(ConFile, "nd0mask", ConfBuf, 255, UPC)) < 0) {
    PrErr(IPWin, "\nError: nd0mask not specified.");
    fclose(ConFile);
    return ND0ERROR;
    }

  if ((sscanf(ConfBuf, "%lx", &TmpStor) == 0) || (TmpStor == 0l)) {
    PrErr(IPWin, "\nError: nd0mask not correct.");
    fclose(ConFile);
    return ND0ERROR;
    }

  HostAddress[1] = htonl(TmpStor);
  if ((UsRet = IfConfig("ND0", HostAddress[0], HostAddress[1],
                        0xffffffffl)) != NO_ERR) {
    PrErr(IPWin, "\nError: configuring interface nd0. Code: %d.", UsRet);
    fclose(ConFile);
    return UsRet;
    }

  if ((LineNr = FGetVar(ConFile, "lo0address", ConfBuf, 255, UPC)) < 0) {
    PrErr(IPWin, "\nError: lo0address not specified.");
    fclose(ConFile);
    return ND0ERROR;
    }

  if (DotConvert(HostAddress, ConfBuf) != 0) {
    PrErr(IPWin, "\nError: lo0address not correct.");
    fclose(ConFile);
    return ND0ERROR;
    }

  HostAddress[1] = htonl(0xff000000l);
  if ((UsRet = IfConfig("LO0", HostAddress[0], HostAddress[1],
                        0xffffffffl)) != NO_ERR) {
    PrErr(IPWin, "\nError: configuring interface lo0. Code: %d.", UsRet);
    fclose(ConFile);
    return UsRet;
    }

  if (FFindSection(ConFile, "ROUTES") >= 0) {
    while ((LineNr = FGetNext(ConFile, ConfBuf, 255, UPC)) >= 0) {
      if (strlen(ConfBuf) > 0) {
        if ((Token = strtok(ConfBuf, " ")) == NULL) {
          PrErr(IPWin, "\nError: syntax incorrect in line: %d.", LineNr);
          fclose(ConFile);
          return ROUTE_ERROR;
          }
        if ((Posit = strstr(SpecTokens, Token)) == NULL) {
          PrErr(IPWin, "\nError: unknown token '%s'in line: %d.",
                       Token, LineNr);
          fclose(ConFile);
          return VAL_ERROR;
          }
        RoutKind = SpecArray[Posit[-2]];
        for (i = 0; i < 2; i++) {
          if ((Token = strtok(NULL, " ")) == NULL) {
            PrErr(IPWin, "\nError: syntax incorrect in line: %d.", LineNr);
            fclose(ConFile);
            return ROUTE_ERROR;
            }
          if (DotConvert(HostAddress + i, Token) != 0) {
            if ((i == 0) && (strcmp("DEFAULT", Token) == 0)) {
                HostAddress[0] = htonl(0l);
                }
            else {
              PrErr(IPWin, "\nError: address '%s' in line %d.",
                           Token, LineNr);
              fclose(ConFile);
              return ROUTE_ERROR;
              }
            }
          }
        if ((RoutKind == NETSTAT) || (RoutKind == NETMODIFY)) {
          if ((Token = strtok(NULL, " ")) == NULL) {
            PrErr(IPWin, "\nError: Netmask missing in line: %d.", LineNr);
            fclose(ConFile);
            return ROUTE_ERROR;
            }
          if ((sscanf(Token, "%lx", &TmpStor) == 0) || (TmpStor == 0l)) {
            PrErr(IPWin, "\nError: Netmask incorrect in line: %d.", LineNr);
            fclose(ConFile);
            return ROUTE_ERROR;
            }
          }
        if ((UsRet = AddRoute(RoutKind, HostAddress[0],
                              HostAddress[1], htonl(TmpStor))) != NO_ERR) {
          PrErr(IPWin, "\nError: adding route in line %d. Code: %d.",
                       LineNr, UsRet);
          fclose(ConFile);
          return ROUTE_ERROR;
          }
        }
      }
    }
  fclose(ConFile);
  return NO_ERR;
}

/**************************************************************
** NAME:        PrErr
** SYNOPSIS:    static void PrErr(PWWIN *ThisWin,
**                                char *Message, ...);
**           
** DESCRIPTION: Prints message 'Message' in ThisWin if
**              DEBUGANY has been defined. Else, nothing
**              is done. The variable argument list is also
**              printed.
** RETURNS:   
**************************************************************/
static void PrErr(PWWIN *ThisWin, char *Message, ...)
{
#ifdef DEBUGANY
  char     Buffer[255];
  va_list  ArgPointer;

  va_start(ArgPointer, Message);
  vsprintf(Buffer, Message, ArgPointer);
  pw_printf(ThisWin, "%s", Buffer);
  va_end(ArgPointer);
#endif
}
