/************************************************************************
** MODULE INFORMATION*
**********************
**     FILE     NAME:       IPREC.C
**     SYSTEM   NAME:       IP
**     ORIGINAL AUTHOR(S):  Wim van Campen
**     VERSION  NUMBER:     1.0
**     CREATION DATE:       1990/5/21
**
** DESCRIPTION: Contains the receive part of the Internet Protocol.
**              Handles checking the packet, option processing,
**              reassembly, passing to higher layers and sending
**              ICMP error messages, if necessary.   
**              
*************************************************************************
** CHANGES INFORMATION **
*************************
** REVISION:    $Revision:   1.1  $
** WORKFILE:    $Workfile:   IPREC.C  $
** LOGINFO:     $Log:   I:/ETSTJAN/CPROG/BEHOLDER/UDPIP/IP/VCS/IPREC.C_V  $
**              
**                 Rev 1.1   21 Nov 1990 14:24:16   etstjan
**              No explicit note
**              
**                 Rev 1.0   20 Nov 1990 16:14:58   etstjan
**              No explicit note
*************************************************************************/
#if ! defined(PRD)
static char _pvcs_hdr[] =
"$Header:   I:/ETSTJAN/CPROG/BEHOLDER/UDPIP/IP/VCS/IPREC.C_V   1.1   21 Nov 1990 14:24:16   etstjan  $";
#endif

#include      <stdio.h>
#include      <stdlib.h>
#include      <string.h>
#include      <conio.h>
#include      <power3.h>
#include      <beholder.h>

#include      "ip.h"                   /* general ip defines */
#include      "ipcodes.h"              /* and ip return codes */
#include      "iplib.h"                /* include ip library */
#include      "ipif.h"                 /* interface layer */
#include      "ipevents.h"             /* event processing */
#include      "iplayer.h"              /* general ip layer header file */
#include      "socklib.h"

typedef struct {                       /* reassembly hole descriptor */
          USHORT            HoleStart;
          USHORT            HoleEnd;
          int               NextHoleOffset;
          } HOLE_TYPE;

#define   MIN_REASS_SIZE    576 + sizeof(HOLE_TYPE) 
                                       /* minimum reassembly buffer size */
#define   MAX_REASS         3          /* max number of packets in reass. */

/* global variables */ 
IPADDRESS   *HostAddresses;            /* list of hostadresses */

RT_CHOICE   RouteChoice = DONT_ROUTE;  /* to route or not to route */

RT_CHOICE   RouteSrcRoute = DO_ROUTE;  /* route source routed diagrams */

REASSTYPE   ReassSpace[MAX_REASS];     /* reassembly buffers */

USHORT      ReassTimers[MAX_REASS];    /* timeout store */

USHORT      ReassTimeVal = REASSTIME;  /* time out in reassembly */

int         (*Protoc[256])             /* pointers to higher level */
            (PRREQ, CIPHEAD *,         /* protocol routines */
            DATALINK *,
            int, int);

/* prototypes */

int       TestBroadCast(ADDRESS_T ThisAddress, IPADDRESS **AddressStruc,
                        IFDESCRIPTOR *Interf);
void      IPRoute(CIPHEAD *ThisHeader, DATALINK *ThisPacket,
                  RT_CHOICE RouteChoice, DIR_CHOICE Hops,
                  IFDESCRIPTOR *If);
int       ProcessOptions(CIPHEAD *ThisHeader, DATALINK *Packet,
                         IFDESCRIPTOR *If);
int       AddOption(BYTE *OpPoint, BYTE *Value, WORD ValLen);

/**************************************************************
** NAME:        IPPassUp
** SYNOPSIS:    void IPPassUp(CIPHEAD *IPHeader,
**                            DATALINK *IPPacket
**                            int BCasted);
**
** DESCRIPTION: Passes an IP packet to the protocol
**              specified in the IP header.
**              If the protocol is not implemented or a
**              port is not bound, a 'protocol unreachable'
**              or 'port unreachable' message is sent (only
**              when it wasn't a broadcast message).
** RETURNS:     
**************************************************************/
void IPPassUp(CIPHEAD *IPHeader, DATALINK *IPPacket, int BCasted)
{
  BYTE       ICMPCode = 0xff;
  int        RetCode;

  IPStat.NrPackRec++;
  RetCode = (*Protoc[IPHeader->Protocol])
             (RECREQ, IPHeader, IPPacket, BCasted, 0);
  if (RetCode != NO_ERR) {
    if (!BCasted) {
      switch (RetCode) {
        case NO_PROTOC : ICMPCode = PROTUNRCH;
                         break;
        case NO_PORT :   ICMPCode = PORTUNRCH;
                         break;
        }
      if (ICMPCode != 0xff) {
        SendProb(DSTUNRCH, ICMPCode, 0, IPHeader, IPPacket);
        IPBufFree(IPHeader);
        return;
        }
      }
    IPBufFree(IPHeader);
    DiscardAll(IPPacket);
    }
}


/**************************************************************
** NAME:        InitIPLayer
** SYNOPSIS:    int InitIPLayer(void);
**
** DESCRIPTION: Initializes the IP layer
** RETURNS:     NO_ERR -> everything just fine
**
**************************************************************/
int InitIPLayer(void)
{
  int    i;

  for (i = 0; i < MAX_REASS; i++) {    /* initialize reassembly buffers */
    ReassSpace[i].ReassHeader = NULL;
    ReassSpace[i].ReassBuffer = NULL;
    }

  for (i = 0; i < 256; i++) {          /* initialize protocol pointers */
    Protoc[i] = RawProt->IPRec;        /* all unknown to IPRaw */
    }
  Protoc[1] = ICMPRec;                 /* 1 = ICMP */
  Protoc[17] = UDPProt->IPRec;         /* 17 = UDP */

  HostAddresses = NULL;                /* no host adresses yet           */
  HostRoutes = NULL;                   /* host routing table empty       */
  NetRoutes = NULL;                    /* network routing table empty    */
  DefaultRoutes = NULL;                /* default routing table empty    */

  memset(&IPStat, 0, sizeof(IPStat));  /* clear statistics */
  return NO_ERR;
}

/**************************************************************
** NAME:        ReassComp
** SYNOPSIS:    int ReassComp(CIPHEAD *Header1,
**                            CIPHEAD *Header2)
**  
** DESCRIPTION: Compares two IP headers using the
**              following fields:
**              Source Address, Destination Address,
**              Identity, Protocol.
** RETURNS:     0    -->   headers equal
**              1    -->   headers not equal
**************************************************************/
int ReassComp(CIPHEAD *Header1, CIPHEAD *Header2)
{
  return ((Header1->Ident != Header2->Ident) || 
          (Header1->Source != Header2->Source) ||
          (Header1->Protocol != Header2->Protocol) ||
          (Header1->Destin != Header2->Destin)) ? 1 : 0;
}

/**************************************************************
** NAME:        UpdateReassBuf
** SYNOPSIS:    int UpDateReassBuf(REASSTYPE *ThisReass,
**                                 WORD ReassBufSize,
**                                 BYTE *ThisPacket,
**                                 WORD ThisOffset,
**                                 WORD ThisLength,
**                                 int MoreFrag);
**
** DESCRIPTION: Updates reassembly buffer *ThisReass.
**              ReassBufSize contains total length of
**              reassembly buffer.
**              ThisPacket is a pointer to the new packet.
**              ThisOffset, ThisLength determine the
**              packets' position.
**              MoreFrag indicates last fragment or not.
** RETURNS:     NO_ERR      -> no error
**              REASS_COMPL -> reassembly completed
**              REASS_ERR   -> error in reassembly
**************************************************************/
int UpDateReassBuf(REASSTYPE *ThisReass, WORD ReassBufSize,
                   BYTE *ThisPacket, WORD ThisOffset,
                   WORD ThisLength, int MoreFrag)
{
  int        HoleOffset = ThisReass->FirstHole;
  int        *HoleOffPlace = &ThisReass->FirstHole;
  HOLE_TYPE  *HolePoint = (HOLE_TYPE *)(ThisReass->ReassBuffer + HoleOffset);
  USHORT     EndSave;
  USHORT     NextSave;
  USHORT     NextFrag = ThisOffset + ThisLength;

  while (HoleOffset != -1) {
    if ((ThisOffset >= HolePoint->HoleStart) &&
        (NextFrag <= HolePoint->HoleEnd + 1)) {
      break;
      }
    else {
      HoleOffset = HolePoint->NextHoleOffset;
      HoleOffPlace = &HolePoint->NextHoleOffset;
      HolePoint = (HOLE_TYPE *)(ThisReass->ReassBuffer + HoleOffset);
      }
    }
  if (HoleOffset != -1) {
    EndSave = HolePoint->HoleEnd;
    NextSave = HolePoint->NextHoleOffset;
    if (HolePoint->HoleStart == ThisOffset) {
      *HoleOffPlace = NextFrag;
      }        
    else {
      HolePoint->HoleEnd = ThisOffset - 1;
      HoleOffPlace = &HolePoint->NextHoleOffset;
      }
    if (EndSave == NextFrag - 1) {
      *HoleOffPlace = NextSave;
      }
    else {
      if (MoreFrag) {
        HolePoint = (HOLE_TYPE *)(ThisReass->ReassBuffer + NextFrag);
        HolePoint->HoleStart = NextFrag;
        HolePoint->HoleEnd = ReassBufSize;
        HolePoint->NextHoleOffset = NextSave;
        *HoleOffPlace = (int)(NextFrag);
        }
      else {
        *HoleOffPlace = -1;
        }
      }
    memcpy(ThisReass->ReassBuffer + ThisOffset, ThisPacket, ThisLength);
    return (ThisReass->FirstHole == -1) ? REASS_COMPL : NO_ERR;
    }
  else {
    return REASS_ERR;
    }
}

/**************************************************************
** NAME:        ReassTimeOut
** SYNOPSIS:    void ReassTimeOut(EVENTUNION *ThisEvent);
**              
** DESCRIPTION: Timeout server for the reassembly feature.
**              Calls IPReport to update statistics.
**              Allocated buffers are freed.
** RETURNS:     
**              
**************************************************************/
void ReassTimeOut(EVENTUNION *ThisEvent)
{
  REASSTYPE  *ThisReas = ThisEvent->ReassData.ReassStruct;
  CIPHEAD    *IPHeader = ThisReas->ReassHeader;
  CICMPHEAD  ICMPHead = {
               TIMEX,
               REASEXCEED,
               0,
               0L
               };

  /* send ICMP only if first fragment has arrived */
  if (ThisReas->FirstHole != 0) {
    /* send old IP header and first 64 bits of datagram */
    ThisReas->SavePack->Length = IPHeader->HLength + 8;
    IPHeader->OLength = 0;
    ICMPSend(IPHeader, &ICMPHead, ThisReas->SavePack, HOPS_ALLOWED);
    }
  else {
    DiscardAll(ThisReas->SavePack);
    }
  IPStat.ReassTimeOut++;
  IPReport(REASSTIMEOUT, ThisReas->ReassHeader, NULL);
  IPBufFree(ThisReas->ReassBuffer);
  ThisReas->ReassHeader = NULL;
}


/**************************************************************
** NAME:        IPReceive
** SYNOPSIS:    void IPReceive(EVENTUNION *RecMessage);
** 
** DESCRIPTION: Handles reception of all IP packets.
**              Performs validity checks and makes an IP
**              header in C format.
**              Processes options in the IP header and routes
**              packets if desired by options. The state of
**              the options is stored in an Option State.
**              If the packet is fragmented, reassembly is
**              started. When a fragmented packet is complete
**              again, it is processed further.
**              At last, the packet is handled to the
**              desired transport layer and ICMP 'destination
**              unreachable' messages are sent if necessary.
** RETURNS:     
**************************************************************/
void IPReceive(EVENTUNION *RecMessage)
{
  CIPHEAD      *ThisHeader;
  DATALINK     *Packet = RecMessage->RecData.RecPacket;
  int          NotFnd;
  int          RetCode;
  int          i, j;
  USHORT       ReassBufSize= MIN_REASS_SIZE;  /* reassembly buffer size */
  EVENTUNION   ReassTimerData;
  IPADDRESS    *AddressPnt;            /* address structure for this packet */
  int          KeepHeader = 0;
  int          BroadCasted = 0;        /* packet received by broadcast */

  /* first, perform some checks to stay alive */
  if ((ThisHeader = IPBufGet(sizeof(CIPHEAD))) == NULL) {
    IPReport(NOSPACE, NULL, Packet);
    return;
    }
  if (RecMessage->RecData.RecPacket->Length < 20) {
    IPStat.NrTooShort++;
    IPReport(TOOSHORT, ThisHeader, Packet);
    return;
    }
  RetCode = ByteToCip(ThisHeader, Packet->DataStart);
  if (RetCode) {
    switch (RetCode) {
      case VERERR   :  IPStat.NrBadVersion++;
                       break;
      case CHECKERR :  IPStat.NrCheckErr++;
                       break;
      }
    IPReport(RetCode, ThisHeader, Packet);
    return;
    }
  if (ThisHeader->TLength > Packet->Length) {
    IPStat.NrBadLength++;
    IPReport(DATASHORT, ThisHeader, Packet);
    return;
    }
  /* set length of packet to total IP packet length */
  Packet->Length = ThisHeader->TLength;

  /* set start of data area beyond IP header */
  Packet->DataStart += ThisHeader->HLength;
  Packet->Length -= ThisHeader->HLength;

  /* process options, if any */
  if (ThisHeader->OLength > 0) {
    RetCode = ProcessOptions(ThisHeader, Packet, RecMessage->RecData.If); 
    }

  if (RetCode == NO_ERR) {
    if (((AddressPnt = IPAddressSearch(ThisHeader->Destin, UNMASKED,
                                       HostAddresses)) == NULL) &&
        ((BroadCasted = TestBroadCast(ThisHeader->Destin, &AddressPnt, 
                                      RecMessage->RecData.If)) == 0)) {
      /* no host and no broadcast address so route this packet */
      IPRoute(ThisHeader, Packet, RouteChoice, HOPS_ALLOWED,
              RecMessage->RecData.If);
      return;
      }

    if (BroadCasted && !((RecMessage->RecData.LinkBrdCst) ||
                         (RecMessage->RecData.If->Flags & IF_POINTPOINT))) {
      IPStat.IllBCast++;
      IPReport(ILLBCAST, ThisHeader, Packet);
      return;
      }

    /* set specific destination address */
    ThisHeader->ExtraAddress = AddressPnt->HostAddress;

    /* check if reassembly is necessary */
    if ((ThisHeader->Offset != 0) || (ThisHeader->Flag & MORE_FRAGMENT)) {
      /* start reassembly */
      IPStat.NrFragRec++;
      i = 0;
      j = MAX_REASS;
      NotFnd = 1;
      while ((i < MAX_REASS) && NotFnd) {
        if (ReassSpace[i].ReassHeader != NULL) {
          NotFnd = ReassComp(ReassSpace[i].ReassHeader, ThisHeader);
          }
        else {
          j = i;                         /* this buffer free for reassembly */
          }
        if (NotFnd) {
          i++;
          }
        }
      if (RecMessage->RecData.If->Mms_R > ReassBufSize) {
        ReassBufSize = RecMessage->RecData.If->Mms_R + sizeof(HOLE_TYPE);
        }
      if (i == MAX_REASS) {              /* try to allocate reassembly space */
        if ((i = j) == MAX_REASS) {      /* no buffer available */
          IPStat.NoReassSpace++;
          IPReport(NOREASSSPACE, ThisHeader, Packet);
          return;
          }
        else {                           /* try to allocate buffer space */
          if ((ReassSpace[i].ReassBuffer =
              IPBufGet(ReassBufSize)) == NULL) {
            IPReport(NOSPACE, ThisHeader, Packet);
            return;
            }
          else {
            /* set timer on this reassembly buffer */
            ReassTimerData.ReassData.ReassStruct = ReassSpace + i;
            if ((RetCode = SetTimer(LOW, ReassTimeOut, &ReassTimerData,
                            &ReassTimers[i], ReassTimeVal)) != NO_ERR) {
              IPBufFree(ReassSpace[i].ReassBuffer);
              IPReport(RetCode, ThisHeader, Packet);
              return;
              }
            ReassSpace[i].ReassHeader = ThisHeader;
            ReassSpace[i].SavePack = Packet;
            KeepHeader = 1;
            ReassSpace[i].ReassLen = 0;
            ReassSpace[i].FirstHole = 0;
            /* start and end of this hole, no more hole descriptors */
            ((HOLE_TYPE *)(ReassSpace[i].ReassBuffer))->HoleStart = 0;  
            ((HOLE_TYPE *)(ReassSpace[i].ReassBuffer))->HoleEnd = ReassBufSize - 1;
            ((HOLE_TYPE *)(ReassSpace[i].ReassBuffer))->NextHoleOffset = -1;
            }
          }
        }
      else {
        /* continue reassembly */
        if (ThisHeader->Offset == 0) {   /* first fragment header found */
          IPBufFree(ReassSpace[i].ReassHeader);
          DiscardAll(ReassSpace[i].SavePack);
          ReassSpace[i].ReassHeader = ThisHeader;
          ReassSpace[i].SavePack = RecMessage->RecData.RecPacket;
          KeepHeader = 1;
          }
        }
      RetCode = UpDateReassBuf(ReassSpace + i,
                               ReassBufSize,
                               Packet->DataStart,
                               ThisHeader->Offset * 8,
                               Packet->Length,
                               ThisHeader->Flag & MORE_FRAGMENT);

      switch(RetCode) {
        case REASS_COMPL: ResetTimer(&ReassTimers[i]);
                          ReassSpace[i].ReassHeader->TLength =
                            ReassSpace[i].ReassHeader->HLength +
                            (ReassSpace[i].ReassLen += Packet->Length);
                          IPBufFree(Packet->DataArea);
                          DiscardAll(ReassSpace[i].SavePack);
                          if (!KeepHeader) {
                            IPBufFree(ThisHeader);
                            }
                          /* modify DATALINK structure and pass up */
                          Packet->DataArea = Packet->DataStart =
                            ReassSpace[i].ReassBuffer;
                          Packet->Length = ReassSpace[i].ReassLen;

                          IPPassUp(ReassSpace[i].ReassHeader,
                                   Packet, BroadCasted);
                          ReassSpace[i].ReassHeader = NULL;
                          break;
        case REASS_ERR:   IPStat.NrReassError++;
                          if (KeepHeader) {
                            IPReport(REASS_ERR, NULL, NULL);
                            }
                          else {
                            IPReport(REASS_ERR, ThisHeader, Packet);
                            }
                          break;
        default:          ReassSpace[i].ReassLen += Packet->Length;
                          if (!KeepHeader) {
                            IPBufFree(ThisHeader);
                            DiscardAll(Packet);
                          }
                          break;
        }
      }
    else {
      /* no reassembly necessary */
      IPPassUp(ThisHeader, Packet, BroadCasted);
      }
  }
}

/**************************************************************
** NAME:        TestBroadCast
** SYNOPSIS:    int TestBroadCast(ADDRESST_T ThisAddress,
**                                IPADDRESS **AddressStruc,
**                                IFDESCRIPTOR *Interf);
**
** DESCRIPTION: Checks if ThisAddress is an IP broadcast
**              address. 0's and 1,s are accepted as
**              broadcast address. All IP address structures
**              in AddressList are checked. Statistics are
**              updated. The following addresses are checked:
**              a) {-1,-1}
**              b) {<Network-number>,-1}
**              c) {<Network-number>,<Subnet-number>,-1}
**              d) {<Network-number>,-1,-1} (equals b))
**              If a matching broadcast address is found,
**              the address of the corresponding IPADDRESS
**              structure is returned in AddressStruc.
** RETURNS:     0  -->   Not a broadcast address.
**              1  -->   ThisAdress is a broadcast address.
**              -1 -->   ERROR!
**************************************************************/
int TestBroadCast(ADDRESS_T ThisAddress, IPADDRESS **AddressStruc,
                  IFDESCRIPTOR *Interf)
{
  if ((ThisAddress == LIMBRCAST) || (ThisAddress == NONSTBRCAST)) {
    if ((Interf->FirstAdd == NULL) ||
        ((*AddressStruc = IPAddressSearch(Interf->FirstAdd->ThisAddress,
                                          UNMASKED, HostAddresses)) == NULL)) 
      return -1;
    IPStat.LimBroadCast++;
    return 1;
    }
  /* search in all address structures and
     make all the possible broadcast addresses */
  *AddressStruc = HostAddresses;

  while (*AddressStruc != NULL) {
    if ((ThisAddress & (*AddressStruc)->NetWorkMask) ==
        (*AddressStruc)->NetWorkPart) {
      /* first, {<Network-number>,-1} */
      /* and  {<Network-number>,0}    */
      if ((ThisAddress == ((*AddressStruc)->NetWorkPart |
                          (*AddressStruc)->SubHostMask)) ||
          (ThisAddress == (*AddressStruc)->NetWorkPart)) {
        /* correct broadcast address found */
        IPStat.DirecBrdC++;
        return 1;
        }
      else {
        /* try {<Network-number>,<Subnet-number>,-1}  */
        /* and {<Network-number>,<Subnet-number>,0}   */
        if ((ThisAddress == ((*AddressStruc)->SubNetPart |
                            (*AddressStruc)->HostMask)) || 
            (ThisAddress == (*AddressStruc)->SubNetPart)) {
          /* correct broadcast address found */
          IPStat.SubNetDir++;
          return 1;
          }
        }
      }
    *AddressStruc = (*AddressStruc)->NextAddress;
    }
  return 0;
}

/**************************************************************
** NAME:        IPRoute
** SYNOPSIS:    void IPRoute(CIPHEAD *ThisHeader,
**                           DATALINK *ThisPacket,
**                           RT_CHOICE RouteChoice,
**                           DIR_CHOICE Hops,
**                           IFDESCRIPTOR *If);
**
** DESCRIPTION: Performs IP level routing.
**              'RouteChoice' determines whether there should
**              be routing or not, 'DONT_ROUTE' disables
**              routing.
**              If TTL is 0 or 1, a 'time exceeded' message is
**              sent. If it ain't a source routed packet and
**              the interface by which the packet will be
**              sent is the same as the receiving interface
**              or if there ain't a route to the destination,
**              a 'destination unreachable' is sent.
**              In al cases, all buffer space used by the
**              packet is freed by this function.
** RETURNS:  
**************************************************************/
void IPRoute(CIPHEAD *ThisHeader, DATALINK *ThisPacket,
             RT_CHOICE RouteChoice, DIR_CHOICE Hops,
             IFDESCRIPTOR *If)
{
  DATALINK      *SavePacket;
  IFDESCRIPTOR  *NextIf = NULL;
  ADDRESS_T     Dummy;
  int           ConBCast, RetCode;

  if (RouteChoice == DO_ROUTE) {
    if ((ThisHeader->Time == 0) || (--(ThisHeader->Time) == 0)) {
      SendProb(TIMEX, TTLEXCEED, 0, ThisHeader, ThisPacket);
      IPStat.RoutTimExc++;
      IPReport(ROUTTIMEX, ThisHeader, NULL);
      }
    else {
      if (!(ThisHeader->OpState & (STRICT_SRC | LOOSE_SRC)) &&
           ((GetFirstHop(&NextIf, &Dummy, ThisHeader->Destin,
                         ThisHeader->Tos, &ConBCast, Hops) != NO_ERR) ||
            (NextIf == If))) {
        SendProb(DSTUNRCH, HOSTUNRCH, 0, ThisHeader, ThisPacket);
        IPStat.NotForwarded++;
        IPReport(ROUTFAIL, ThisHeader, NULL);
        }
      else {
        /* prepare a datalink to send ICMP message, if needed */
        if ((SavePacket = IPBufGet(sizeof(DATALINK))) == NULL) {
          IPReport(NOSPACE, ThisHeader, ThisPacket);
          return;
          }
        *SavePacket = *ThisPacket;
        ThisPacket->SaveData = 1;      /* this packet's data must be saved */

        RetCode = IPSend(ThisHeader, ThisPacket, Hops);

        if (RetCode != NO_ERR) {
          switch (RetCode) {
            case FRAG_NOT_ALL : SendProb(DSTUNRCH, DFSET, 0,
                                         ThisHeader, SavePacket);
                                break;
            case NOT_ON_NET   :
            case NOT_REACHED  : SendProb(DSTUNRCH, HOSTUNRCH, 0,
                                         ThisHeader, SavePacket);
                                break;
            default           : DiscardAll(SavePacket);
                                break;
            }
          IPStat.NotForwarded++;
          IPReport(RetCode, ThisHeader, NULL);
          }
        else {
          IPStat.Forwarded++;
          DiscardAll(SavePacket);
          IPBufFree(ThisHeader);
          }
        }
      }
    }    
  else {
    IPStat.RouteNotAll++;
    IPReport(ROUTNOTALL, ThisHeader, ThisPacket);
    }
}

/**************************************************************
** NAME:        AddOption
** SYNOPSIS:    int AddOption(BYTE *OpPoint,
**                            BYTE *Value,
**                            WORD ValLen);
**      
** DESCRIPTION: Places a value in an option field.
**              Checks if there are at least OptLen bytes
**              left in the option indicated by OpPoint.
**              If the option is exactly filled, no action
**              is taken. If there is space but not enough,
**              -1 is returned.
**              If there is enough space, ValLen bytes from
**              Value are copied in the option at an offset
**              indicated by the option's pointer. The
**              option's pointer is updated.
** RETURNS:     0  -->   enough space left;
**              1  -->   option exactly filled;
**              -1 -->   error detected.
**************************************************************/
int AddOption(BYTE *OpPoint, BYTE *Value, WORD ValLen)
{
    USHORT  OLen = OpPoint[1];
    USHORT  OPnt = OpPoint[2] - 1;

    if (OPnt + ValLen <= OLen) {
      memcpy(OpPoint + OPnt, Value, ValLen);
      OpPoint[2] += (BYTE)ValLen;
      return 0;
      } 
    else {                     /* error in option or pointer past option */
      return (OLen - OPnt != 0) ? -1 : 1;
      }
}

/**************************************************************
** NAME:        ProcessOptions
** SYNOPSIS:    int ProcessOptions(CIPHEAD *ThisHeader,
**                                 DATALINK *Packet,
**                                 IFDESCRIPTOR *If);
**
** DESCRIPTION: Processes options in an IP packet.
**              Values are added to an option field if
**              desired. The state of the option is stored
**              in the IP header. If there is a 'source
**              route' option with more hops to go, the
**              packet is routed.
**              If the packet is in trouble, it is
**              discarded and an ICMP message is sent.
** RETURNS:     NO_ERR    --> no error, packet should be
**                            processed further.
**              PACK_SEND --> the packet is processed
**                            completely and has been sent
**                            to a next hop or is discareded.
**************************************************************/
int ProcessOptions(CIPHEAD *ThisHeader, DATALINK *Packet,
                   IFDESCRIPTOR *If)
{
  IFDESCRIPTOR  *NextIf;
  CICMPHEAD     ICMPHeader = {
                  DSTUNRCH,
                  SRCFAIL,
                  0,
                  0L
                  };
  BYTE          *OPoint = ThisHeader->Options;
  BYTE          OpData[8];
  USHORT        DataOf;
  USHORT        OLen, OCnt, OPnt;
  BYTE          OverFlag;
  USHORT        ErrField;
  int           Routing = HOPS_ALLOWED;
  int           RetCode;
  int           State = 0;             /* 0 = process packet further */
                                       /* 1 = route this packet      */

  ThisHeader->OpState = 0;
  for (OCnt = 0; (OCnt < ThisHeader->OLength) && (OPoint[OCnt] != ENDOFLIST);
       OCnt += OLen) {
    if (OPoint[OCnt] == NOP) {
      OLen = 1;
      }
    else {
      OLen = (USHORT)(OPoint[OCnt + 1]);
      OPnt = (USHORT)OPoint[OCnt + 2] - 1;
      if ((OLen < 3) || (OCnt + OLen > ThisHeader->OLength)) {
        SendParamProb((BYTE)(OCnt + 22), ThisHeader, Packet);
        IPStat.NrIllOptions++;
        return PACK_SEND;
        }
      *(ADDRESS_T *)OpData = If->FirstAdd->ThisAddress;
      switch(OPoint[OCnt]) {
        /* loose source route option */
        case LOOSE     :
        case STRICT    : /* check if option isn't empty */
                         if (OLen == 3) {
                           RemoveOption(ThisHeader, OCnt);
                           }
                         if (((ThisHeader->OpState &
                              (LOOSE_SRC | STRICT_SRC)) == 0) &&
                             (IPAddressSearch(ThisHeader->Destin, UNMASKED,
                                              HostAddresses) != NULL)) {
                           if (OPoint[OCnt] == LOOSE) {
                             Routing = HOPS_ALLOWED;
                             IPStat.NrLooseSrc++;
                             ThisHeader->OpState |= LOOSE_SRC;
                             }
                           else {
                             Routing = DIRECT_ONLY;
                             IPStat.NrStrictSrc++;
                             ThisHeader->OpState |= STRICT_SRC;
                             }
                           if (OPnt + 4 <= OLen) {
                             ThisHeader->Destin = *(ADDRESS_T *)(OPoint + OCnt + OPnt);
                             AddOption(ThisHeader->Options + OCnt, OpData, 4);
                             State = 1;
                             }
                           else {
                             if (OLen - OPnt != 0) {
                               SendParamProb((BYTE)(OCnt + 22),
                                             ThisHeader, Packet);
                               return PACK_SEND;
                               }
                             }
                          }
                         break;
        case RECORD    : IPStat.NrRecRoute++;
                         if (AddOption(ThisHeader->Options + OCnt,
                                       OpData, 4) >= 0) {
                           ThisHeader->OpState |= REC_ROUTE;
                           }
                         else {
                           SendParamProb((BYTE)(OCnt + 22),
                                         ThisHeader, Packet);
                           return PACK_SEND;
                           }
                         break;
        case TIMESTAMP : IPStat.NrTimeStamp++;
                         OverFlag = OPoint[OCnt + 3];
                         *(ULONG *)(OpData + 4) = htonl(TimeMilSec());
                         ErrField= OCnt + 22;
                         switch (OverFlag & 0xf) {
                           case 0 :
                           case 1 :
                             DataOf = 4 - (OverFlag & 0xf) * 4;
                             RetCode =
                               AddOption(ThisHeader->Options + OCnt,
                                         OpData + DataOf, 8 - DataOf);
                             break;
                           case 3 :
                             if (IPAddressSearch(*(ADDRESS_T *)(OPoint + OCnt + OPnt),
                                       UNMASKED, HostAddresses) == NULL) {
                               RetCode = 0;
                               }
                             else {
                               memcpy(OpData, OPoint + OCnt + OPnt, 4);
                               RetCode = AddOption(ThisHeader->Options + OCnt,
                                                   OpData, 8);
                               }
                             break;
                           default:
                             ErrField = OCnt + 24;
                             RetCode = -1;
                             break;
                           }
                         ThisHeader->OpState |= TIME_ST;
                         if (RetCode == 1) {        /* option space full */
                           if (((OPoint[OCnt + 3] += 16) & 0xf0) == 0) {
                             ErrField = OCnt + 24;
                             RetCode = -1;
                             }
                           }
                         if (RetCode == -1) {       /* error occured */
                           SendParamProb((BYTE)ErrField, ThisHeader, Packet);
                           return PACK_SEND;
                           }
                         break;
        default        : break;
        }
      }
    }

  ThisHeader->OLength = OCnt;
  if (State == 1) {
    if ((GetFirstHop(&NextIf, (ADDRESS_T *)OpData, ThisHeader->Destin,
                     ThisHeader->Tos, &RetCode, Routing) != NO_ERR) ||
        ((RouteSrcRoute == DONT_ROUTE) && (NextIf != If))) {
      ThisHeader->Destin = *(ADDRESS_T *)OpData; /* will be swapped on sending */
      SendProb(DSTUNRCH, SRCFAIL, 0, ThisHeader, Packet);
      IPStat.ErrSrcRoute++;
      IPReport(ERRSRCRT, ThisHeader, NULL);
      }
    else {
      IPRoute(ThisHeader, Packet, DO_ROUTE, Routing, If);
      }
    return PACK_SEND;
    }
  return NO_ERR;
}



