{

********************************************************************************
*                                                                              *
*       Evaluate a truth- functional formula to determine whether it is:       *
*                                                                              *
*                               VALID                                          *
*                               CONTRADICTORY                                  *
*                               NON-CONTRADICTORY                              *
*                                                                              *
*       Quine's method.                                                        *
*                                                                              *
********************************************************************************

********************************************************************************
*                                                                              *  
*                         Gianfranco  Boggio-Togna                             *
*                               C.P. 14021                                     *  
*                          I-20147 Milano (Italy)                              *  
*                                                                              *
*                              gbt@computer.org                                *           *
*                                                                              *
*			The code in this file is placed in the Public Domain			   *
*                                                                              *  
*                                                                              *
********************************************************************************

}



CONST

				not_symbol      = '-' ;
				and_symbol      = '.' ;
				or_symbol       = '|' ;
				impl_symbol     = '>' ;
				equiv_symbol    = '=' ;

				alpha =         225 ;           { 'a' + 128 }
				zeta  =         250 ;           { 'z' + 128 }

		{ Bounds for formula arrays }

				Formula_Low_Bound  = 1 ;
				Formula_High_Bound = 160 ;

TYPE

		Formula_buffer =
						 PACKED ARRAY [Formula_Low_Bound..Formula_High_Bound] OF Char ;



CONST


		{ Maximum number of variables }

				N_Variables = 58 ; 

		{ Maximum number of formulas }
		
				N_Formulas  = 26 ;


TYPE
		Node_Type = (Truth_Value, Operand, Operator) ;
		Logical_Operator = (Op_Not, Op_And, Op_Or, Op_Impl, Op_Equiv) ;
		Operand_Value    = (Value_True, Value_False, Value_Undefined);
		Pointer = ^ Node ;

		Node = RECORD

				 CASE N_Type: Node_Type OF

				   Truth_Value:        (N_Truth_Value: Boolean) ;
				   Operand:            (N_Operand:     Char  ;
										N_Value:       Operand_Value) ;
				   Operator:           (N_Operator:    Logical_Operator ;
										N_Left_Link:   Pointer ;
										N_Right_Link:  Pointer) ;
			   END ;


VAR
		p: Pointer ;


		T_Node , F_Node, Equivalence_Node:  Pointer ;

		Variables       :  ARRAY [1..N_Variables] OF Pointer ;
		Formulas        :  ARRAY [1..N_Formulas] OF Pointer ;
		Dummy_Formula   :  Pointer ;



{       Scratch

		************************************************************************
		*                                                                      *
		*                           SCRATCH A TREE                             *
		*                                                                      *
		************************************************************************


		Remove all operator nodes from (sub)tree rooted at 'p' 

}


PROCEDURE Scratch  (VAR p: Pointer) ;

BEGIN
		IF p <> NIL THEN
		  WITH  p^ DO
			IF N_Type = Operator THEN
			  BEGIN
				Scratch (N_Left_Link) ;
				Scratch (N_Right_Link) ;
				Dispose (p {, Operator}) ;
				p := NIL ;
			  END ;
END ;




{       Copy

		************************************************************************
		*                                                                      *
		*                           COPY A TREE                                *
		*                                                                      *
		************************************************************************

		 Creates a copy of the tree rooted at 'p'.
		 Returns a pointer to the root of the copy.

}


FUNCTION Copy (VAR p: Pointer): Pointer ;

VAR
		q: Pointer ; 

BEGIN
		IF p = NIL THEN
		   Copy := NIL
		ELSE
		   WITH p^ DO
			 CASE N_Type OF
				Operand:
						Copy := p ;
				Operator:
						BEGIN
						  New (q {, Operator}) ;
						  WITH q^ DO
							BEGIN
								N_Type       := Operator ;
								N_Operator   := p^.N_Operator ;
								N_Left_Link  := Copy (p^.N_Left_Link) ;
								N_Right_Link := Copy (p^.N_Right_Link) ;
							END ;
						  Copy := q ;
						END ;
			 END ;
END ;


{
		Get_Tree

		************************************************************************
		*                                                                      *
		*            Get tree corresponding to formula to be tested            *
		*                                                                      *
		************************************************************************
}


FUNCTION  Get_Tree (VAR Formula: Formula_buffer ;
					VAR Error_Position: integer): Pointer ;

LABEL
		1, 2 ;

VAR
		R_Polish        : PACKED ARRAY [Formula_Low_Bound..Formula_High_Bound] OF Char;
		Rp_Ptr          : integer ;
		Stack           : PACKED ARRAY [1..128] OF Char ;

		Formula_Ptr,Top : integer ;
		c, Temp         : Char ;
		Rank            : integer ;
		j               : integer ;


		Previous        : Char ;        { private to Next_Char function }


{               Next_Char

				****************************************************************
				*                                                              *
				*                       Get next character                     *
				*                                                              *
				****************************************************************

				Get the next character from the formula.

				Returns TRUE if character found, FALSE if no more characters.

				Inserts a (virtual) And_Symbol between juxtaposed variable
				names, formula names or parenthesized expressions.

				Inserts a (virtual) close parenthesis ')' after the last
				character of the formula.

}


FUNCTION Next_Char (VAR Character: Char): Boolean ;

LABEL
		1 ;

BEGIN
		IF Formula_Ptr = Formula_Low_Bound THEN
		   Previous := ' ' ;
1:
		IF Formula_Ptr <= Formula_High_Bound THEN
		  BEGIN
				Character := Formula [Formula_Ptr] ;
				Formula_Ptr := Formula_Ptr + 1 ;
				IF Character = ' ' THEN
					 GOTO 1 ;

				IF ( Character IN ['a'..'z','A'..'Z', chr(alpha)..chr(zeta),
								   Not_Symbol, '(','[','{' ] )
				   AND
				   ( Previous  IN ['a'..'z','A'..'Z', chr(alpha)..chr(zeta),
								   ')',']','}' ] ) THEN
					 BEGIN      
							Formula_Ptr := Formula_Ptr - 1 ;
							Character   := And_Symbol ;
					 END ;                      

				Previous := Character ;
				Next_Char := TRUE ;
		  END
		ELSE
		  IF Formula_Ptr =  Formula_High_Bound + 1  THEN
			 BEGIN
				Character := ')' ;
				Formula_Ptr := Formula_Ptr + 1 ;
				Next_Char := TRUE ;
			 END
		  ELSE
			 Next_Char := FALSE ;

END ;


{               Ip

				****************************************************************
				*                                                              *
				*                       Return input priority                  *
				*                                                              *
				****************************************************************
}


FUNCTION Ip (Character: Char): integer ;

BEGIN
		IF Character IN ['a'..'z', 'A'..'Z', chr(alpha)..chr(zeta) ] THEN
		  Ip := 11 
		ELSE            
		  IF Character = Equiv_Symbol THEN
			Ip := 1 
		  ELSE
			IF Character = Impl_Symbol THEN
			  Ip := 3 
			ELSE
			  IF Character = Or_Symbol THEN
				Ip := 5 
			  ELSE      
				IF Character =  And_Symbol THEN
				  Ip := 7 
				ELSE
				  IF Character = Not_Symbol THEN
					Ip := 9 
				  ELSE
					IF Character IN ['(','[','{'] THEN
					  Ip := 13 
					ELSE
					  (* IF Character IN [')',']','}'] THEN *)
						Ip := 0 ;

END ;   



{               Rf

				****************************************************************
				*                                                              *
				*               Return operand/operator rank                   *
				*                                                              *
				****************************************************************
}


FUNCTION Rf (Character: Char): integer ;

BEGIN
		IF  ( Character = Equiv_Symbol ) OR ( Character = Impl_Symbol  ) OR
			( Character = Or_Symbol    ) OR ( Character = And_Symbol   ) THEN
				Rf := -1 
		ELSE
		   IF Character = Not_Symbol   THEN
				Rf := 0
		   ELSE
				Rf := 1 ;

END ;   



{               Sp

				****************************************************************
				*                                                              *
				*                  Return stacking priority                    *
				*                                                              *
				****************************************************************
}


FUNCTION Sp (Character: Char): integer ;

BEGIN
		IF Character IN ['a'..'z', 'A'..'Z', chr(alpha)..chr(zeta) ] THEN
		  Sp := 12 
		ELSE
		  IF Character = Equiv_Symbol THEN
			Sp := 2 
		  ELSE
			IF Character = Impl_Symbol THEN
			  Sp := 4 
			ELSE                
			  IF Character = Or_Symbol THEN
				Sp := 6 
			  ELSE              
				IF Character = And_Symbol THEN
				  Sp := 8 
				ELSE
				  IF Character = Not_Symbol THEN
					Sp := 10 
				  ELSE
					{ IF Character IN ['(','[','{'] THEN }
					  Sp := 0 ;

END ;   



{               Build_Tree

				****************************************************************
				*                                                              *
				*            Build tree from reverse Polish expression         *
				*                                                              *
				****************************************************************
}


FUNCTION Build_Tree : Pointer ;

VAR
		p: Pointer ;
		j: integer ;

BEGIN
		Rp_Ptr := Rp_Ptr - 1 ;

		IF R_Polish [Rp_Ptr] IN ['a'..'z', 'A'..'Z', chr(alpha)..chr(zeta)] THEN
		  BEGIN
				IF R_Polish [Rp_Ptr] IN [chr(alpha)..chr(zeta)] THEN
				   BEGIN                        
						j := ord (R_Polish [Rp_Ptr]) - alpha + 1 ;
						IF Formulas [j] <> NIL THEN
						   Build_Tree := Copy (Formulas [j])
						ELSE
						   Build_Tree := Formulas [j] ; { should never happen }
				   END
				ELSE       
				   BEGIN                
						j := ord (R_Polish [Rp_Ptr]) - ord ('A') + 1 ;
						IF Variables [j] = NIL THEN
						  BEGIN
								New (p {, Operand}) ;
								WITH p^ DO
								  BEGIN
										N_Type       := Operand ;
										N_Value      := Value_Undefined ; 
										N_Operand    := R_Polish [Rp_Ptr] ;
								  END ;
								Variables [j] := p ;
						  END ;
						Build_Tree := Variables [j] ;
				   END ;
		  END

		ELSE

		  BEGIN
				New (p {, Operator}) ;
				WITH p^ DO
				  BEGIN
						N_Type := Operator ;
						IF R_Polish [Rp_Ptr] = Not_Symbol THEN
						   BEGIN
							 N_Operator := Op_Not ;
							 N_Left_Link := Build_Tree ;
							 N_Right_Link := NIL ;
							 Build_Tree := p 
						   END
						ELSE
						   BEGIN
							 IF R_Polish [Rp_Ptr] = And_Symbol THEN
							   N_Operator := Op_And 
							 ELSE               
							   IF R_Polish [Rp_Ptr] = Or_Symbol THEN
								 N_Operator := Op_Or 
							   ELSE
								 IF R_Polish [Rp_Ptr] = Impl_Symbol THEN
								   N_Operator := Op_Impl 
								 ELSE
								   N_Operator := Op_Equiv ;
							 N_Right_Link := Build_Tree ;  { Right  ...       }
							 N_Left_Link  := Build_Tree ;  { ... BEFORE  Left }
							 Build_Tree := p ;
						  END ;
				  END ;

		  END ;

END ;


{       Get_Tree

		************************************************************************
		************************************************************************
		************************************************************************
}

		{ The algorithm used to build the reverse Polish espression (a variant
		  of Dijkstra's "shunting yard" algorithm) is REVPOL from
		  'An Introduction to Data Structures with Applications' by Tremblay
		  and Sorenson, 1st edition, McGraw-Hill, 1976. }


BEGIN
		Formula_Ptr := 1 ;
		Top := 1 ;
		Stack [Top] := '(' ;
		Rank := 0 ;
		Rp_Ptr := 0 ;
		
		WHILE Next_Char (c) DO
		  BEGIN
				IF NOT ( c IN ['a'..'z', 'A'..'Z', chr(alpha)..chr(zeta),
							  Not_Symbol, And_Symbol,
							  Or_Symbol, Impl_Symbol, Equiv_Symbol,
							  '(','[','{',')',']','}' ] ) THEN
				   BEGIN                   
						Rank := 0;      { force syntax error }
						GOTO 2 ;
				   END ;


				{ semantic check: formula names must be defined }

				IF c IN [chr(alpha)..chr(zeta)] THEN
				   BEGIN
						IF Formulas [ord (c) - alpha + 1] = NIL THEN
							BEGIN                  
								Rank := 0;      { force error }
								GOTO 2 ;
							END ; 
				   END ;

				IF  Top < 1 THEN
				  BEGIN
						Rank := 0;      { force error }
						GOTO 2 ;
				  END ;

				WHILE Ip (c) <= Sp (Stack [Top]) DO
				  BEGIN
						Temp := Stack [Top] ; Top := Top - 1 ; 
						IF Ip (c) < Sp (Temp) THEN
						  BEGIN
								Rp_Ptr := Rp_Ptr + 1 ;
								R_Polish [Rp_Ptr] := Temp ;
								Rank := Rank + Rf (Temp) ;
								IF Rank < 1 THEN GOTO 2 ;
						  END
						ELSE
						  GOTO 1 ;
				  END ;
				
				  Top := Top + 1 ;
				  Stack [Top] := c ;

1:       END ;

2:      IF (Top <> 0) OR (Rank <> 1) THEN
		   BEGIN
				Error_Position := Formula_Ptr - 1 ;
				Get_Tree := NIL ;          
		   END 
		ELSE
		   BEGIN
				Error_Position := 0 ;
				Rp_Ptr := Rp_Ptr + 1 ;
				Get_Tree := Build_Tree
		   END ;
		
END ;



{       Evaluate_Tree

		************************************************************************
		*                                                                      *
		*            Evaluate tree corresponding to formula to be tested       *
		*                                                                      *
		************************************************************************
}


PROCEDURE Evaluate_Tree (   Tree: Pointer;
							VAR Valid: Boolean ;
							VAR Non_Contradictory: Boolean) ;   






{               Reduce

				****************************************************************
				*                                                              *
				*                       Reduce formula                         *
				*                                                              *
				****************************************************************
}


FUNCTION  Reduce ( p:           Pointer ;
				   Subs_Var:    Pointer): Pointer ;
				
VAR
		  Left, Right, q: Pointer ;



{                       Reduce_Not

						********************************************************
						*                                                      *
						*                   Reduce negation                    *
						*                                                      *
						********************************************************
}


FUNCTION Reduce_Not: Pointer ;

BEGIN
	  IF Left^.N_Type = Truth_Value THEN        { Right is always NIL }
		 IF Left^.N_Truth_Value THEN 
				Reduce_Not := F_Node
		 ELSE           
				Reduce_Not := T_Node
	  ELSE
		 BEGIN          
		   New (q {, Operator}) ;
		   WITH q^ DO
			 BEGIN
				N_Type       := Operator ;
				N_Operator   := Op_Not ;
				N_Left_Link  := Left ;
				N_Right_Link := NIL ;
			 END ;
		   Reduce_Not := q ;
		 END ;

END ;



{                       Reduce_And

						********************************************************
						*                                                      *
						*                   Reduce conjunction                 *
						*                                                      *
						********************************************************
}


FUNCTION Reduce_And: Pointer ;

BEGIN
	  IF (Left^.N_Type = Truth_Value) AND (Right^.N_Type = Truth_Value) THEN  
		 IF Left^.N_Truth_Value AND Right^.N_Truth_Value THEN
			Reduce_And := T_Node 
		 ELSE
			Reduce_And := F_Node
	  ELSE
		 IF Left^.N_Type = Truth_Value THEN
			 IF Left^.N_Truth_Value THEN 
				Reduce_And := Right 
			 ELSE       
				BEGIN
				  Reduce_And := F_Node ;   Scratch (Right) ;
				END
		 ELSE
			 IF Right^.N_Type = Truth_Value THEN
				IF Right^.N_Truth_Value THEN 
				   Reduce_And := Left
				ELSE    
				   BEGIN
					 Reduce_And := F_Node ; Scratch (Left) ;
				   END
			 ELSE
				BEGIN
						New (q {, Operator}) ;
						WITH q^ DO
						  BEGIN
								N_Type       := Operator ;
								N_Operator   := Op_And ;
								N_Left_Link  := Left ;
								N_Right_Link := Right ;
						  END ;
						Reduce_And := q ;
				END ;

END ;



{                       Reduce_Or

						********************************************************
						*                                                      *
						*                   Reduce disjunction                 *
						*                                                      *
						********************************************************
}


FUNCTION Reduce_Or: Pointer ;

BEGIN
	  IF (Left^.N_Type = Truth_Value) AND (Right^.N_Type = Truth_Value) THEN  
		 IF Left^.N_Truth_Value OR Right^.N_Truth_Value THEN
			Reduce_Or := T_Node 
		 ELSE
			Reduce_Or := F_Node
	  ELSE
		 IF Left^.N_Type = Truth_Value THEN
			 IF Left^.N_Truth_Value THEN 
				BEGIN
				  Reduce_Or := T_Node ;  Scratch (Right) ;
				END
			 ELSE       
				Reduce_Or := Right 
		 ELSE
			 IF Right^.N_Type = Truth_Value THEN
				IF Right^.N_Truth_Value THEN 
				   BEGIN
					 Reduce_Or := T_Node ; Scratch (Left) ;
				   END
				ELSE    
				   Reduce_Or := Left
			 ELSE
				BEGIN
				  New (q {, Operator}) ;
				  WITH q^ DO
					BEGIN
						N_Type       := Operator ;
						N_Operator   := Op_Or ;
						N_Left_Link  := Left ;
						N_Right_Link := Right ;
					END ;
				  Reduce_Or := q ;
				END ;

END ;



{                       Reduce_Impl

						********************************************************
						*                                                      *
						*                   Reduce implication                 *
						*                                                      *
						********************************************************
}


FUNCTION Reduce_Impl: Pointer ;

BEGIN
	  IF (Left^.N_Type = Truth_Value) AND (Right^.N_Type = Truth_Value) THEN  
		 IF Left^.N_Truth_Value AND (NOT Right^.N_Truth_Value) THEN
			Reduce_Impl := F_Node 
		 ELSE
			Reduce_Impl := T_Node
	  ELSE
		 IF Left^.N_Type = Truth_Value THEN
			 IF Left^.N_Truth_Value THEN 
				Reduce_Impl := Right 
			 ELSE       
				BEGIN
				  Reduce_Impl := T_Node ; Scratch (Right)
				END
		 ELSE
			 IF Right^.N_Type = Truth_Value THEN
				IF Right^.N_Truth_Value THEN 
				   BEGIN
						Reduce_Impl := T_Node ; Scratch (Left)
				   END
				ELSE
				   BEGIN        
						New (q {, Operator}) ;
						WITH q^ DO
						  BEGIN
								N_Type       := Operator ;
								N_Operator   := Op_Not ;
								N_Left_Link  := Left ;
								N_Right_Link := NIL ;
						  END ;
						  Reduce_Impl := q ;
				   END
			 ELSE
				BEGIN
				  New (q {, Operator}) ;
				  WITH q^ DO
					BEGIN
						N_Type       := Operator ;
						N_Operator   := Op_Impl ;
						N_Left_Link  := Left ;
						N_Right_Link := Right ;
					END ;
				  Reduce_Impl := q ;
				END ;

END ;



{                       Reduce_Equiv

						********************************************************
						*                                                      *
						*                   Reduce equivalence                 *
						*                                                      *
						********************************************************
}


FUNCTION Reduce_Equiv: Pointer ;

BEGIN
	  IF (Left^.N_Type = Truth_Value) AND (Right^.N_Type = Truth_Value) THEN  
		 IF Left^.N_Truth_Value =  Right^.N_Truth_Value  THEN
			Reduce_Equiv := T_Node 
		 ELSE
			Reduce_Equiv := F_Node
	  ELSE
		 IF Left^.N_Type = Truth_Value THEN
			 IF Left^.N_Truth_Value THEN 
				Reduce_Equiv := Right
			 ELSE       
				BEGIN           
				  New (q {, Operator}) ;
				  WITH q^ DO
					BEGIN
						N_Type       := Operator ;
						N_Operator   := Op_Not ;
						N_Left_Link  := Right ;
						N_Right_Link := NIL ;
					END ;
				  Reduce_Equiv := q ;
				END
		 ELSE
			 IF Right^.N_Type = Truth_Value THEN
				IF Right^.N_Truth_Value THEN 
				   Reduce_Equiv := Left
				ELSE
				  BEGIN         
						New (q {, Operator}) ;
						WITH q^ DO
						  BEGIN
								N_Type       := Operator ;
								N_Operator   := Op_Not ;
								N_Left_Link  := Left ;
								N_Right_Link := NIL ;
						  END ;
						Reduce_Equiv := q ;
				  END
			 ELSE
				BEGIN
						New (q {, Operator}) ;
						WITH q^ DO
						  BEGIN
								N_Type       := Operator ;
								N_Operator   := Op_Equiv ;
								N_Left_Link  := Left ;
								N_Right_Link := Right ;
						  END ;
						Reduce_Equiv := q ;
				END ;

END ;



{               Reduce

				****************************************************************
				****************************************************************
				****************************************************************
}


BEGIN


	IF P = NIL THEN
	  Reduce := NIL
	ELSE
	  WITH p^ DO

		CASE N_Type OF

		  Truth_Value:  IF N_Truth_Value THEN
						   Reduce := T_Node
						ELSE
						   Reduce := F_Node ;

		  Operand:      IF p = Subs_Var THEN
						  IF p^.N_Value = Value_TRUE THEN
							 Reduce := T_Node
						  ELSE
							 Reduce := F_Node 
						ELSE
						  Reduce := p ;

		  Operator:
				BEGIN
						Left  := Reduce (N_Left_Link, Subs_Var)  ;
						Right := Reduce (N_Right_Link, Subs_Var) ;
						CASE N_Operator OF
						  Op_Not  :     Reduce := Reduce_Not   ;
						  Op_And  :     Reduce := Reduce_And   ;
						  Op_Or   :     Reduce := Reduce_Or    ;
						  Op_Impl :     Reduce := Reduce_Impl  ;
						  Op_Equiv:     Reduce := Reduce_Equiv ;
						END ;
				END ;
		END ;

END ;



{               Evaluate

				****************************************************************
				*                                                              *
				*       Evaluate formula for all truth value combinations      *
				*                                                              *
				****************************************************************
}

						
PROCEDURE  Evaluate (p: Pointer) ;

VAR
		Variable, f: Pointer ;

BEGIN
		WITH p^ DO
			IF N_Type = Truth_Value THEN
			  BEGIN
				Valid := Valid AND N_Truth_Value ;
				Non_Contradictory := Non_Contradictory OR N_Truth_Value ;
			  END
			ELSE
			  IF (NOT Non_Contradictory) OR Valid  THEN 
				BEGIN   
						Variable := p ;
						WHILE Variable^.N_Type <> Operand DO
							 Variable := Variable^.N_Left_Link ;

						Variable^.N_Value := Value_TRUE ; 
						f :=  Reduce (p, Variable) ;
						Evaluate (f) ;
						Scratch  (f) ;

						Variable^.N_Value := Value_FALSE ; 
						f :=  Reduce (p, Variable) ;
						Evaluate (f) ;
						Scratch  (f) ;

						Variable^.N_Value := Value_Undefined ; 
				END ;


END ;



{
		Evaluate_Tree

		************************************************************************
		************************************************************************
		************************************************************************
}



BEGIN

		Valid := TRUE ;
		Non_Contradictory := FALSE ;

		Evaluate (Tree) ;

END ;



{ *****************************************************************************
  *****************************************************************************
  *****************************************************************************
  *****************************************************************************
  *****************************************************************************
  *****************************************************************************
  *****************************************************************************
}


{       Init

********************************************************************************
*                                                                              *
*                               INITIALIZE                                     *
*                                                                              *
********************************************************************************

}

PROCEDURE Init ;

VAR
		i: integer ;

BEGIN
		{ Create nodes representing Truth and Falsehood }

		New (T_Node {, Truth_Value}) ;
		WITH T_Node^ DO
		  BEGIN
				N_Type := Truth_Value ;
				N_Truth_Value := TRUE ;
		  END ;

		New (F_Node {, Truth_Value}) ;
		WITH F_Node^ DO
		  BEGIN
				N_Type := Truth_Value ;
				N_Truth_Value := FALSE ;
		  END ;

		{ Create a node for testing equivalence of a formula
		  being	selected with the formula it replaces		 }				

		New (Equivalence_Node {, Operator}) ;
		WITH Equivalence_Node^ DO
		  BEGIN
			N_Type       := Operator ;
			N_Operator   := Op_Equiv ;
			N_Left_Link  := Nil ;
			N_Right_Link := Nil ;
		  END ;

		FOR i := 1 TO N_Formulas DO
				Formulas [i] := NIL ;

		Dummy_Formula := NIL ;

		FOR i := 1 TO N_Variables DO
				Variables [i] := NIL ;

END ;



{       Reinit

********************************************************************************
*                                                                              *
*                           RE-INITIALIZE                                      *
*                                                                              *
********************************************************************************

}

PROCEDURE Reinit ;

VAR
		i: integer ;

BEGIN
		FOR i := 1 TO N_Formulas DO
		  BEGIN
				Scratch (Formulas [i]) ;
				Formulas [i] := NIL ;
		  END ;

		Dummy_Formula := NIL ;

		FOR i := 1 TO N_Variables DO
				IF Variables [i] <> NIL THEN    
				  BEGIN
					 Dispose (Variables [i] {, Operand}) ;
					 Variables [i] := NIL ;
				  END ;
END ;



{       Select

********************************************************************************
*                                                                              *
*                              SELECT A FORMULA                                *
*                                                                              *
********************************************************************************

}

FUNCTION  Select  ( VAR Formula_Id: Char ;         { Formula name }
					VAR Formula: Formula_buffer  ; { Formula buffer }
					VAR Formula_changed: Boolean   { True if formula existed }
												   { and has changed		 }	
				  ) : Integer { 0: no error ; <> 0 error position } ;

VAR
		j, Error: integer ;
		p: 		  Pointer ;
		v, nc:	  Boolean ;		
BEGIN
		Formula_changed := False ;
		IF Formula_Id IN [ chr(alpha)..chr(zeta)] THEN
		  BEGIN
				j := ord (Formula_Id) - alpha + 1 ;
				Select := 0 ;
				p := Get_Tree (Formula, Error) ;
				IF Error <> 0 THEN
				   Select := Error
				ELSE
				  BEGIN
					IF Formulas [j] <> NIL THEN
					  BEGIN
						WITH Equivalence_Node^ DO
						  BEGIN
							N_Left_Link  := p ;
							N_Right_Link := Formulas [j] ;
						  END ;
						 Evaluate_Tree (Equivalence_Node, v, nc) ;
					     Formula_changed := NOT v ;
					     Scratch (Formulas [j]) ; 
					  END ;
				    Formulas [j] := p ;
				  END ;
		  END           
		ELSE     
		  IF Formula_Id = ' ' THEN
			 BEGIN
				IF Dummy_Formula <> NIL THEN
				   Scratch (Dummy_Formula) ; 

				Select := 0 ;
				p := Get_Tree (Formula, Error) ;

				IF Error <> 0 THEN
				   Select := Error
				ELSE
				   Dummy_Formula := p ;
			 END
		  ELSE
			 Select := -1 ;

END ;



{       Remove

********************************************************************************
*                                                                              *
*                              REMOVE A FORMULA                                *
*                                                                              *
********************************************************************************

}

FUNCTION  Remove (VAR Formula_Id: Char { Variable name })  
				 : Boolean  { TRUE : success ; FALSE : error } ;

VAR
		j: integer ;

BEGIN
		Remove := True ;

		IF Formula_Id IN [chr(alpha)..chr(zeta)] THEN
		  BEGIN
				j := ord (Formula_Id) - alpha + 1;
				IF Formulas [j] <> NIL THEN
				  BEGIN
				   Scratch (Formulas [j]) ;
				   Formulas [j] := NIL ;
				  END
				ELSE
				   Remove := False ;
		  END
		ELSE     
		  IF Formula_Id = ' ' THEN
			 BEGIN
				IF Dummy_Formula <> NIL THEN
				   BEGIN
						   Scratch (Dummy_Formula) ; 
						   Dummy_Formula := NIL ;
				   END
				ELSE    
				   Remove := False ;
			 END
		  ELSE
			 Remove := False ;

END ;



{       Eval

********************************************************************************
*                                                                              *
*                       EVALUATE A FORMULA                                     *
*                                                                              *
********************************************************************************

}

FUNCTION Eval (VAR Formula_Id : Char { formula name }) 
			  : Char
				{ V valid; C contradictory; N non-contradictory; ? error} ; 

VAR
		j: integer ;
		p: Pointer ;
		Valid, Non_Contradictory: Boolean ;

BEGIN
		IF Formula_Id IN [ chr(alpha)..chr(zeta), ' '] THEN
		  BEGIN
				IF Formula_Id = ' ' THEN
						p := Dummy_Formula 
				ELSE
						p := Formulas [ ord(Formula_Id) - alpha + 1] ;
				IF p <> NIL THEN
				  BEGIN
					 Evaluate_Tree (p, Valid, Non_Contradictory) ;
					 IF Valid THEN
						Eval := 'V' 
					 ELSE
						IF Non_Contradictory THEN
						   Eval := 'N'
						ELSE
						   Eval := 'C' ;
				  END
				ELSE
				   Eval := '?' ;
		  END
		ELSE     
		  Eval := '?' ;

END ;



