// DEMO6.C --- Pen-centric Windows application that demonstrates 
// handwritten input using the Recognize() function.
// Copyright (C) 1992 Ray Duncan

#define WIN31
#define dim(x) (sizeof(x) / sizeof(x[0]))   // returns no. of elements
#define BUFMAX 80                           // size of scratch buffers

#include "windows.h"
#include "penwin.h"
#include "demo6.h"

HANDLE hInst;                               // instance handle
HWND hWndFrame = 0;                         // frame window handle
HWND hWndText = 0;                          // child text window handle
HWND hWndWrite = 0;                         // child writing window handle
HANDLE hPenWin = 0;                         // Pen Windows module handle

int CharX = 0;                              // average char width
int CharY = 0;                              // char height
int CurLine = 0;                            // current line number
int MaxLine = 0;                            // maximum line number
int EnumCount = 0;                          // translation counter

char szFrameClass[] = "DemoFrameClass";     // name of frame window class
char szTextClass[]  = "DemoTextClass";      // child text window class
char szWriteClass[] = "DemoWriteClass";     // child writing window class

struct {                                    // result codes returned
    int Code;                               // by Recognize() function
    char * Name;
    } recogResults[] = {
    REC_OK, "REC_OK",
    REC_ABORT, "REC_ABORT",
    REC_BADHPENDATA, "REC_BADHPENDATA",
    REC_BUFFERTOOSMALL, "REC_BUFFERTOOSMALL",
    REC_BUSY, "REC_BUSY",
    REC_DONE, "REC_DONE",
    REC_NOINPUT, "REC_NOINPUT",
    REC_NOTABLET, "REC_NOTABLET",
    REC_OOM, "REC_OOM",
    REC_OVERFLOW, "REC_OVERFLOW",
    REC_TERMBOUND, "REC_TERMBOUND",
    REC_TERMEX, "REC_TERMEX",
    REC_TERMOEM, "REC_TERMOEM",
    REC_TERMPENUP, "REC_TERMPENUP",
    REC_TERMRANGE, "REC_TERMRANGE",
    REC_TERMTIMEOUT, "REC_TERMTIMEOUT", } ;

struct {                                    // result types returned
    int Code;                               // in RCRESULT structure
    char * Name;
    } rctypeResults[] = {
    RCRT_UNIDENTIFIED, "RCRT_UNIDENTIFIED",
    RCRT_GESTURE, "RCRT_GESTURE",   
    RCRT_NOSYMBOLMATCH, "RCRT_NOSYMBOLMATCH",    
    RCRT_PRIVATE, "RCRT_PRIVATE",
    RCRT_NORECOG, "RCRT_NORECOG",
    RCRT_ALREADYPROCESSED, "RCRT_ALREADYPROCESSED",
    RCRT_GESTURETRANSLATED, "RCRT_GESTURETRANSLATED",
    RCRT_GESTURETOKEYS, "RCRT_GESTURETOKEYS", } ;

//
// WinMain --- entry point from Windows.  Registers window classes,
// creates windows, registers pen app, and processes messages until 
// WM_QUIT received.
//
int PASCAL WinMain(HANDLE hInstance, 
    HANDLE hPrevInstance, LPSTR lpCmdLine, int nCmdShow)
{
    MSG msg;                     

    hInst = hInstance;                      // save this instance handle

    if(!hPrevInstance)                      // if first instance,
        if(!InitApplication(hInstance))     // register window class
            return(FALSE);                  // exit if couldn't register

    if(!InitInstance(hInstance, nCmdShow))  // create this instance's window
    {
        MessageBox(0, "Initialization failed!", "DEMO6", MB_OK|MB_ICONSTOP);
        return(FALSE);
    }

    RegisterPenApp(RPA_DEFAULT, TRUE);      // enable HEDIT/BEDIT controls

    while(GetMessage(&msg, NULL, 0, 0))     // while message != WM_QUIT
    {
        TranslateMessage(&msg);             // translate virtual key codes
        DispatchMessage(&msg);              // dispatch message to window
    }
    return(msg.wParam);                     // return code = WM_QUIT value
}

//
// InitApplication --- registers window classes for this application
//
BOOL InitApplication(HANDLE hInstance)
{
    WNDCLASS  wc;
    BOOL bParent;
    BOOL bText;
    BOOL bWrite;

    // set parameters for frame window class
    wc.style = CS_HREDRAW | CS_VREDRAW;     // class style
    wc.lpfnWndProc = FrameWndProc;           // class callback function
    wc.cbClsExtra = 0;                      // extra per-class data
    wc.cbWndExtra = 0;                      // extra per-window data
    wc.hInstance = hInstance;               // handle of class owner
    wc.hIcon = LoadIcon(NULL, IDI_APPLICATION);     // default icon
    wc.hCursor = LoadCursor(NULL, IDC_ARROW);       // default cursor
    wc.hbrBackground = GetStockObject(WHITE_BRUSH); // background color 
    wc.lpszMenuName =  "DemoMenu";          // name of menu resource
    wc.lpszClassName = szFrameClass;        // name of window class

    bParent = RegisterClass(&wc);           // register frame window class

    // set parameters for text output child window class
    wc.lpfnWndProc = TextWndProc;           // class callback function
    wc.hIcon = NULL;                        // default icon
    wc.lpszMenuName =  NULL;                // name of menu resource
    wc.lpszClassName = szTextClass;         // name of window class

    bText = RegisterClass(&wc);             // register text child class

    // set parameters for writing input child window class
    wc.lpfnWndProc = WriteWndProc;          // class callback function
    wc.lpszClassName = szWriteClass;        // name of window class

    bWrite = RegisterClass(&wc);            // register writing child class

    return(bParent && bText && bWrite);     // return success/failure flag
}

//
// InitInstance --- creates frame window for this application instance
//
BOOL InitInstance(HANDLE hInstance, int nCmdShow)
{
    hWndFrame = CreateWindow(               // create frame window
        szFrameClass,                       // window class name
        "Pen Windows Demo #6",              // text for title bar
        WS_OVERLAPPEDWINDOW,                // window style
        CW_USEDEFAULT, CW_USEDEFAULT,       // default position
        CW_USEDEFAULT, CW_USEDEFAULT,       // default size
        NULL,                               // no parent window
        NULL,                               // use class default menu
        hInstance,                          // window owner
        NULL                                // unused pointer
    );

    if(!hWndFrame) return(FALSE);           // error, can't create window

    ShowWindow(hWndFrame, nCmdShow);        // make frame window visible
    UpdateWindow(hWndFrame);                // force WM_PAINT message
    return(TRUE);                           // return success flag
}

//
// FrameWndProc --- callback function for application frame window
//
long FAR PASCAL FrameWndProc(HWND hWnd, WORD wMsg, WORD wParam, LONG lParam)
{
    switch (wMsg) 
    {
        case WM_CREATE:                     // creating frame window
            hWndText = CreateWindow(        // create child text window
                szTextClass,                // window class name
                NULL,                       // text for title bar
                WS_CHILDWINDOW | WS_VISIBLE | WS_BORDER,    // style
                0, 0, 0, 0,                 // position and size
                hWnd,                       // frame window is parent 
                0,                          // child window identifier
                hInst,                      // window owner
                NULL);                      // unused pointer
            hWndWrite = CreateWindow(       // create child writing window
                szWriteClass,               // window class name
                NULL,                       // text for title bar
                WS_CHILDWINDOW | WS_VISIBLE | WS_BORDER,    // style
                0, 0, 0, 0,                 // position and size
                hWnd,                       // frame window is parent 
                0,                          // child window identifier
                hInst,                      // window owner
                NULL);                      // unused pointer
            break;

        case WM_SIZE:                       // resize & position children:
            MoveWindow(hWndWrite,           // writing input window 
                0, 0, LOWORD(lParam)/2, HIWORD(lParam), TRUE);
            MoveWindow(hWndText,            // text output window
                LOWORD(lParam)/2, 0, LOWORD(lParam), HIWORD(lParam), TRUE);
            return(0);

        case WM_COMMAND:                    // menu command received,
            DoCommand(hWnd, wParam);        // decode it
            break;

        case WM_DESTROY:                    // window being destroyed
            PostQuitMessage(0);             // force WM_QUIT message
            break;

        default:                            // let Windows handle it
            return(DefWindowProc(hWnd, wMsg, wParam, lParam));
    }

    return(0);
}

//
// TextWndProc --- callback function for child text window
//
long FAR PASCAL TextWndProc(HWND hWnd, WORD wMsg, WORD wParam, LONG lParam)
{
    TEXTMETRIC tm;
    HDC hdc;
    LPRCRESULT rcresult;                    // recognition results
    FARPROC lpProc;                         // far pointer for callback 
    char buff1[BUFMAX];                     // scratch buffer
    char buff2[BUFMAX];                     // scratch buffer
    int i;                                  // scratch variable

    switch(wMsg)
    {
        case WM_CREATE:                     // creating text child window
            hdc = GetDC(hWnd);              // get device context
            GetTextMetrics(hdc, &tm);       // get text metrics
            CharX = tm.tmAveCharWidth;      // save character size
            CharY = tm.tmHeight + tm.tmExternalLeading;
            ReleaseDC(hWnd, hdc);           // release device context
            return(0);

        case WM_SIZE:                       // resize/position in progress
            CurLine = 0;                    // reset current line number
            MaxLine = HIWORD(lParam)/CharY; // no. of lines in window
            return(0);

        case WM_RCRESULT:                   // display recognition results
            rcresult = (LPRCRESULT) lParam; // get pointer to results
            DisplayLine("Message WM_RCRESULT received.");
            for(i = 0; i < dim(rctypeResults); i++)
            {                               // decode result type
                if(rctypeResults[i].Code & rcresult->wResultsType)
                {
                    wsprintf(buff1, "Result type code: %s.", 
                        (LPSTR) rctypeResults[i].Name);
                    DisplayLine(buff1);     // display result type
                }
            }
            SymbolToCharacter(rcresult->lpsyv, BUFMAX, buff2, NULL);
            wsprintf(buff1, "Preferred translation is: %s.", (LPSTR) buff2);
            DisplayLine(buff1);             // display best bet string
            wsprintf(buff1, "Symbol graph has %d possible translations.",
                GetSymbolCount(&(rcresult->syg)));   
            DisplayLine(buff1);             // display no. of candidates
            EnumCount = 0;                  // display each candidate
            lpProc = MakeProcInstance(EnumFunc, hInst);
            EnumSymbols(&(rcresult->syg), 1000, lpProc, NULL);
            FreeProcInstance(lpProc);
            return(0);

        default:
            return(DefWindowProc(hWnd, wMsg, wParam, lParam));
    }
}

//
// WriteWndProc --- callback function for child writing window
//
long FAR PASCAL WriteWndProc(HWND hWnd, WORD wMsg, WORD wParam, LONG lParam)
{
    RC rc;                                  // recognition context
    HDC hdc;                                // device context handle
    PAINTSTRUCT ps;                         // painting info structure
    RECT rect;                              // receives client rectangle
    REC result;                             // status from Recognize()
    char buff1[BUFMAX];                     // scratch buffer
    int i;                                  // scratch variable

    switch(wMsg)
    {
        case WM_PAINT:                      // window needs repainting
            hdc = BeginPaint(hWnd, &ps);    // get device context
            GetClientRect(hWnd, &rect);     // get client rectangle
            DrawText(hdc,                   // put legend in writing area
                "Writing Area", -1, &rect, 
                DT_CENTER | DT_TOP | DT_SINGLELINE);
            EndPaint(hWnd, &ps);            // release device context
            break;

        case WM_LBUTTONDOWN:                // mouseclick starts pen input
            if(IsPenEvent(wMsg, GetMessageExtraInfo()))
            {
                InvalidateRect(hWnd, NULL, TRUE);   // erase both windows
                UpdateWindow(hWnd);         
                InvalidateRect(hWndText, NULL, TRUE);   
                UpdateWindow(hWndText);         
                CurLine = 0;                // reset current line number
                DisplayLine("Initializing RC structure.");
                InitRC(hWndWrite, &rc);     // set default RC values
                rc.hwnd = hWndText;         // text window gets results
                rc.rglpdf[0] = NULL;        // assume no dictionary
                DisplayLine("Calling Recognize function.");
                result = Recognize(&rc);    // request pen input
                strcpy(buff1, "Recognize result code: unknown.");
                for(i = 0; i < dim(recogResults); i++)
                {                           // decode Recognize() status
                    if(recogResults[i].Code == result)
                        wsprintf(buff1, "Recognize result code: %s.", 
                            (LPSTR) recogResults[i].Name);
                }
                DisplayLine(buff1);         // display Recognize() status   
            };
            return(0);

        default:
            return(DefWindowProc(hWnd, wMsg, wParam, lParam));
    }
}

// 
// DisplayLine --- displays a string in the text output child window
//
void DisplayLine(char * szOut)
{
    RECT rect;
    HDC hdc;

    hdc = GetDC(hWndText);                  // get device context
    GetClientRect(hWndText, &rect);         // get client rectangle
    if(CurLine == MaxLine)                  // at bottom of window?
    {
        ScrollWindow(hWndText, 0, -CharY,   // yes, scroll window up
            NULL, NULL);    
        UpdateWindow(hWndText);
        CurLine--;
    }
    TextOut(hdc, 0, CurLine * CharY,        // display text in window
        szOut, strlen(szOut));
    CurLine++;                              // move to next line
    ReleaseDC(hWndText, hdc);               // release device context
    return;
}

//
// DoCommand --- handles menu command messages for the frame window
//
void DoCommand(HWND hWnd, WORD wParam)
{
    FARPROC lpProc;                         // far pointer to callback 

    switch(wParam)                          // decode it
    {
        case IDM_EXIT:                      // user picked File-Quit
            SendMessage (hWnd, WM_CLOSE, 0, 0L);
            break;

        default:                            // unknown command, ignore it
            break;
    }
}

//
// EnumFunc --- the callback function for EnumSymbols().  During
// evaluation of a symbol graph, this function is entered 
// repetitively with a pointer to each candidate symbol array.
// The symbol array is translated to a character string and displayed.
//
int FAR PASCAL EnumFunc(LPSYV lpSyv, int cSyv, VOID FAR * lpData)
{
    char buff1[BUFMAX];                     // scratch buffer
    char buff2[BUFMAX];                     // scratch buffer

    SymbolToCharacter(lpSyv, BUFMAX, buff2, NULL);
    wsprintf(buff1, "Translation #%d is: %s.", ++EnumCount, (LPSTR) buff2);
    DisplayLine(buff1);
    return(TRUE);                           // TRUE to continue enumeration
}

