// SpaceFrame.cpp : implementation file
//

#include "stdafx.h"
#include "SpaceFrame.h"
#include "Error.h"
#include "animation.h"

#include <mmsystem.h>

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

double DirX[32] =
{
  0.000000,
  0.195090,
  0.382683,
  0.555570,
  0.707107,
  0.831470,
  0.923880,
  0.980785,
  1.000000,
  0.980785,
  0.923880,
  0.831470,
  0.707107,
  0.555570,
  0.382683,
  0.195090,
  0.000000,
  -0.195090,
  -0.382683,
  -0.555570,
  -0.707107,
  -0.831470,
  -0.923880,
  -0.980785,
  -1.000000,
  -0.980785,
  -0.923880,
  -0.831470,
  -0.707107,
  -0.555570,
  -0.382683,
  -0.195090
};

double DirY[32] =
{
  -1.0,
  -0.980785,
  -0.92388,
  -0.83147,
  -0.707107,
  -0.55557,
  -0.382683,
  -0.19509,
  0.0,
  0.19509,
  0.382683,
  0.55557,
  0.707107,
  0.83147,
  0.92388,
  0.980785,
  1.0,
  0.980785,
  0.92388,
  0.83147,
  0.707107,
  0.55557,
  0.382683,
  0.19509,
  0.0,
  -0.19509,
  -0.382683,
  -0.55557,
  -0.707107,
  -0.83147,
  -0.92388,
  -0.980785
};

/////////////////////////////////////////////////////////////////////////////
// CSpaceFrame

//-----------------------------------------------------------------------------
// default constructor
//-----------------------------------------------------------------------------
CSpaceFrame::CSpaceFrame()
{
  m_pFrameRateSurface = NULL;

	m_dwStartTime = 0;
	m_nFrameCount = 0;
  m_nFrameRate= 0;

  m_ddbltfx.dwSize = sizeof(DDBLTFX);
  m_ddsd.dwSize = sizeof(DDSURFACEDESC);

  m_nLives = 3;
  m_nTurnDelay = 0;
  m_nLevel = 0;
}

CSpaceFrame::~CSpaceFrame()
{
}

//-----------------------------------------------------------------------------
// GetWindowTitle
//-----------------------------------------------------------------------------
CString CSpaceFrame::GetWindowTitle()
{
  return "Space Demo";
}

//-----------------------------------------------------------------------------
// UpdateGame
//-----------------------------------------------------------------------------
BOOL CSpaceFrame::UpdateGame()
{
  // If its ESC, we quit the app
	if (::GetAsyncKeyState(VK_ESCAPE) < 0)
	{
		return FALSE;
	}

  MoveSprites();
  CollideSprites();
  
  // out of lives, game over man
  if (m_nLives <= 0)
  {
    DisplayMessage("Game Over");
    return FALSE;
  }

  // Erase the background
  m_ddbltfx.dwFillColor = 0;
  while (TRUE)
  {
    HRESULT ddrval;
    ddrval = m_pBackBuffer->Blt(NULL, NULL, NULL, DDBLT_COLORFILL, &m_ddbltfx);
    if (ddrval == DD_OK)
      break;

    if (ddrval == DDERR_SURFACELOST)
      if (!RestoreSurfaces())
          return FALSE;

    if (ddrval != DDERR_WASSTILLDRAWING)
      break;
  }

  DrawSprites();

  DrawText();

  // Everyhting is ready in the back buffer, so flip
  m_pFrontBuffer->Flip(NULL, DDFLIP_WAIT);

  return TRUE;
}

//-----------------------------------------------------------------------------
// MoveSprites
//-----------------------------------------------------------------------------
void CSpaceFrame::MoveSprites()
{
  SPRITE *pHold, *pCurr;
  BOOL bCollision;
  int nNumEnemies = 0;

  pCurr = &m_head;
  // move the sprites and detect edge of screen hits
  do
  {
    switch (pCurr->eType) 
    {
      case typeShip:
        pCurr->nDelay++;
        if (pCurr->nDelay >= ANIM_DELAY)
        {
          pCurr->nDelay = 0;

          if (::GetAsyncKeyState(VK_CONTROL) < 0)
          {
            AddSprite(typeBullet, pCurr->dPosX, pCurr->dPosY);
          }
        }

        if (::GetAsyncKeyState(VK_RIGHT) < 0)
        {
          if (m_nTurnDelay > TURN_DELAY)
          {
            m_nTurnDelay = 0;

            ++pCurr->nDirection;
            if (pCurr->nDirection >= NUM_DIRECTIONS)
              pCurr->nDirection = 0;
          }
          else
            m_nTurnDelay++;
        }

        if (::GetAsyncKeyState(VK_LEFT) < 0)
        { 
          if (m_nTurnDelay > TURN_DELAY)
          {
            m_nTurnDelay = 0;
            --pCurr->nDirection;
            if (pCurr->nDirection < 0)
              pCurr->nDirection = NUM_DIRECTIONS - 1;
          }
          else
            m_nTurnDelay++;
        }

        if (::GetAsyncKeyState(VK_UP) < 0)
        {
          pCurr->dVelX += DirX[pCurr->nDirection] / 10;
          pCurr->dVelY += DirY[pCurr->nDirection] / 10;
        }

        if (::GetAsyncKeyState(VK_DOWN) < 0)
        {
          pCurr->dVelX = 0.0;
          pCurr->dVelY = 0.0;
        }


        pCurr->dPosX += pCurr->dVelX;
        pCurr->dPosY += pCurr->dVelY;

        pCurr->nFrame = pCurr->nDirection;
        
        break;

      case typeEnemy:
        nNumEnemies++;
      case typeBullet:
        pCurr->dPosX += pCurr->dVelX;
        pCurr->dPosY += pCurr->dVelY;

        pCurr->nDelay++;
        if (pCurr->nDelay >= ANIM_DELAY)
        {
          pCurr->nDelay = 0;
          pCurr->nFrame++;
          if (pCurr->nFrame >= pCurr->pAnim->GetNumFrames())
            pCurr->nFrame = 0;
        }

        break;
    } 

    // reverse direction when hitting edge of world
    bCollision = FALSE;
    if (int(pCurr->dPosX) < WORLD_MIN_X)
    {
      pCurr->dVelX *= -1;
      pCurr->dPosX = WORLD_MIN_X;
      bCollision = TRUE;
    }

    if (int(pCurr->dPosY) < WORLD_MIN_Y)
    {
      pCurr->dVelY *= -1;
      pCurr->dPosY = WORLD_MIN_Y;
      bCollision = TRUE;
    }

    if ((int(pCurr->dPosX) + pCurr->pAnim->GetWidth()) > WORLD_MAX_X)
    {
      pCurr->dVelX *= -1;
      pCurr->dPosX = WORLD_MAX_X - pCurr->pAnim->GetWidth();
      bCollision = TRUE;
    }

    if ((int(pCurr->dPosY) + pCurr->pAnim->GetHeight()) > WORLD_MAX_Y)
    {
      pCurr->dVelY *= -1;
      pCurr->dPosY = WORLD_MAX_Y - pCurr->pAnim->GetHeight();
      bCollision = TRUE;
    } 
    
    // remove bullets if they go off the edge of the world 
    if (typeBullet == pCurr->eType && bCollision)
    {
      pHold = pCurr->pNext;
      RemoveSprite(pCurr);
      pCurr = pHold;
    }
    else
      pCurr = pCurr->pNext;

  } while (pCurr != &m_head);

  if (nNumEnemies == 0)
  {
    KillLevel();
    InitLevel();
  }
}

//-----------------------------------------------------------------------------
// CollideSprites
//-----------------------------------------------------------------------------
void CSpaceFrame::CollideSprites()
{
  SPRITE *pHold, *pTarget, *pCurr;
  BOOL bCollision;

  pCurr = &m_head;
  do
  {
    bCollision = FALSE;
    if (pCurr->eType != typeBullet && pCurr->eType != typeShip)
    {
      pCurr = pCurr->pNext;
      continue;
    }

    pTarget = m_head.pNext;
    do
    {
      if (pTarget->eType == typeBullet || pTarget == pCurr)
      {
        pTarget = pTarget->pNext;
        continue;
      }
      
      int nX = int(pCurr->dPosX) + (pCurr->pAnim->GetWidth() / 2);
      int nY = int(pCurr->dPosY) + (pCurr->pAnim->GetHeight() / 2);

      if (nX > pTarget->dPosX &&
          nY > pTarget->dPosY &&
          nX < (pTarget->dPosX + pTarget->pAnim->GetWidth()) &&
          nY < (pTarget->dPosY + pTarget->pAnim->GetHeight()))
      {
        bCollision = TRUE;

        pHold = pTarget->pNext;
        RemoveSprite(pTarget);
        pTarget = pHold;
      }  
      else
        pTarget = pTarget->pNext;
    } while (pTarget != &m_head);

    if (bCollision)
    {
      if (pCurr->eType == typeShip)
      {
        m_nLives--;

        m_head.dPosX = 300.0;
        m_head.dPosY = 200.0;
        m_head.dVelX = 0.0;
        m_head.dVelY = 0.0;
        m_head.nDirection = m_head.nFrame = m_head.nDelay = 0;
      }
      else
      {
        pHold = pCurr->pNext;
        RemoveSprite(pCurr);
        pCurr = pHold;
      }
    }
    else
      pCurr = pCurr->pNext;
  } while (pCurr != &m_head);  
}

//-----------------------------------------------------------------------------
// DrawSprites
//-----------------------------------------------------------------------------
void CSpaceFrame::DrawSprites()
{
  HRESULT hr;
  SPRITE* pCurr = m_head.pNext;
  do
  {
    hr = pCurr->pAnim->Render(int(pCurr->dPosX),
                              int(pCurr->dPosY),
                              pCurr->nFrame,
                              m_pBackBuffer);

    if (DDERR_SURFACELOST == hr)
      RestoreSurfaces();

    pCurr = pCurr->pNext;
  } while (pCurr != m_head.pNext);
}

//-----------------------------------------------------------------------------
// AddSprite
//-----------------------------------------------------------------------------
SPRITE* CSpaceFrame::AddSprite(SpriteType eType, 
                               double dX, // = 0.0
                               double dY) // = 0.0
{
  SPRITE* pSprite = (SPRITE*)LocalAlloc(LPTR, sizeof(SPRITE));

  // setup depending on type
  if (pSprite)
  {
    switch (eType) 
    {
      case typeEnemy:
        pSprite->pAnim = m_pEnemyAnim;
        pSprite->dPosX = RandVal(0.0, (640.0 - m_pEnemyAnim->GetWidth()));
        pSprite->dPosY = RandVal(0.0, (480.0 - m_pEnemyAnim->GetHeight()));
        pSprite->nDirection = RandVal(0, 32);
        pSprite->dVelX = DirX[pSprite->nDirection] * RandVal(0.25, 0.5);
        pSprite->dVelY = DirY[pSprite->nDirection] * RandVal(0.25, 0.5);
        pSprite->nFrame = RandVal(0, m_pEnemyAnim->GetNumFrames());
        break;

      case typeBullet:
        pSprite->dPosX = 
          (dX + (m_pShipAnim->GetWidth() / 2)) - (m_pBulletAnim->GetWidth() / 2);
        pSprite->dPosY = 
          (dY + (m_pShipAnim->GetHeight() / 2)) - (m_pBulletAnim->GetHeight() / 2);
        pSprite->dVelX = DirX[m_head.nDirection] * 6.0;
        pSprite->dVelY = DirY[m_head.nDirection] * 6.0;
        pSprite->nFrame = 0;
        pSprite->pAnim = m_pBulletAnim;
        break;
    } 

    // generic stuff
    pSprite->eType = eType;
    pSprite->nDelay = 0;

    // link the new sprite        
    pSprite->pNext = m_head.pNext;
    pSprite->pPrev = &m_head;
    m_head.pNext->pPrev = pSprite;
    m_head.pNext = pSprite;      

    return pSprite;
  }
  else
    return NULL;
}

//-----------------------------------------------------------------------------
// RemoveSprite
//-----------------------------------------------------------------------------
void CSpaceFrame::RemoveSprite(SPRITE* pSprite)
{
  pSprite->pNext->pPrev = pSprite->pPrev;
  pSprite->pPrev->pNext = pSprite->pNext;
  LocalFree(pSprite);
}

//-----------------------------------------------------------------------------
// InitLevel
//-----------------------------------------------------------------------------
BOOL CSpaceFrame::InitLevel(int nDifficulty /* =0 */)
{
  DisplayLevel();

  m_head.pNext = m_head.pPrev = &m_head;
  m_head.eType = typeShip;
  m_head.dPosX = 300.0;
  m_head.dPosY = 200.0;
  m_head.dVelX = 0.0;
  m_head.dVelY = 0.0;
  m_head.nDirection = m_head.nFrame = m_head.nDelay = 0;
  m_head.pAnim = m_pShipAnim;

  for (int nIndex = 0; nIndex < (m_nLevel * 2) + 3; nIndex++)
  {
    AddSprite(typeEnemy);
  }

  if (nDifficulty)
    m_nLevel = nDifficulty;
  else
    m_nLevel++;

  return TRUE;
}

//-----------------------------------------------------------------------------
// DisplayLevel
//-----------------------------------------------------------------------------
void CSpaceFrame::DisplayLevel()
{
  char str[9];

  wsprintf(str, "Level %3d", m_nLevel+1);
  DisplayMessage(str);
}

//-----------------------------------------------------------------------------
// DisplayMessage
//-----------------------------------------------------------------------------
void CSpaceFrame::DisplayMessage(char* pStr)
{
  HDC hDC;

  if (m_pTextSurface->GetDC(&hDC) == DD_OK)
  {
    SetTextColor(hDC, RGB(255,255,255));
    SetBkColor(hDC, RGB(0,0,0));
    HGDIOBJ hOldFont = SelectObject(hDC, m_font);
    TextOut(hDC, 0,0, pStr, 9);
    SelectObject(hDC, hOldFont);

    m_pTextSurface->ReleaseDC(hDC);
  }

  CRect srcRect(0, 0, 180, 40);
  for(int nDelay = 0; nDelay < 200; nDelay++)
  {
	  if (::GetAsyncKeyState(VK_ESCAPE) < 0)
		  break;

    while (TRUE)
    {
      HRESULT ddrval;
      ddrval = m_pBackBuffer->Blt(NULL, NULL, NULL, DDBLT_COLORFILL, &m_ddbltfx);
      if (ddrval == DD_OK)
        break;

      if (ddrval == DDERR_SURFACELOST)
        if (!RestoreSurfaces())
            return;
    }

    m_pBackBuffer->BltFast(240, 215, m_pTextSurface, &srcRect,
                           DDBLTFAST_NOCOLORKEY);

    m_pFrontBuffer->Flip(NULL, DDFLIP_WAIT);
  }
}

//-----------------------------------------------------------------------------
// KillLevel
//-----------------------------------------------------------------------------
void CSpaceFrame::KillLevel()
{
  while (m_head.pNext != &m_head)
    RemoveSprite(m_head.pNext);
}

//-----------------------------------------------------------------------------
// InitGraphics
//-----------------------------------------------------------------------------
BOOL CSpaceFrame::InitGraphics()
{
  if (!CGameFrame::InitGraphics())
    return FALSE;

  DDSURFACEDESC ddsd;
  ddsd.dwSize = sizeof(DDSURFACEDESC);
  ddsd.dwFlags = DDSD_CAPS | DDSD_HEIGHT | DDSD_WIDTH;
  ddsd.ddsCaps.dwCaps = DDSCAPS_OFFSCREENPLAIN;
  ddsd.dwHeight = 40;
  ddsd.dwWidth = 50;
  HRESULT ddrval = m_pDD->CreateSurface(&ddsd, &m_pFrameRateSurface, NULL);
  if (ddrval != DD_OK)
  {
    DDrawError("CreateSurface failed:", ddrval);
    return FALSE;
  }

  ddsd.dwSize = sizeof(DDSURFACEDESC);
  ddsd.dwHeight = 40;
  ddsd.dwWidth = 180;
  ddrval = m_pDD->CreateSurface(&ddsd, &m_pTextSurface, NULL);
  if (ddrval != DD_OK)
  {
    DDrawError("CreateSurface failed:", ddrval);
    return FALSE;
  }

  DDCOLORKEY ddck;
  ddck.dwColorSpaceLowValue = RGB(0,0,0);
  ddck.dwColorSpaceHighValue = RGB(0,0,0);
  m_pFrameRateSurface->SetColorKey(DDCKEY_SRCBLT, &ddck);
  m_pTextSurface->SetColorKey(DDCKEY_SRCBLT, &ddck);

  BOOL bSuccess = m_font.CreateFont(48,          // height
                                    0,           // width
                                    0,           // escapement
                                    0,           // orientation
                                    FW_NORMAL,   // weight
                                    FALSE,       // italic
                                    FALSE,       // underline
                                    FALSE,       // strike out
                                    ANSI_CHARSET,  
                                    OUT_DEFAULT_PRECIS,
                                    CLIP_DEFAULT_PRECIS,
                                    NONANTIALIASED_QUALITY,
                                    VARIABLE_PITCH,
                                    "Comic Sans MS");
  if (!bSuccess)
    return FALSE;

  m_dwStartTime = timeGetTime();

  m_pShipAnim = new CAnim(m_pDD, "ship", "media/", 32);
  m_pEnemyAnim = new CAnim(m_pDD, "enem", "media/", 32);
  m_pBulletAnim = new CAnim(m_pDD, "bull", "media/", 8);

  m_pShipAnim->Load();
  m_pEnemyAnim->Load();
  m_pBulletAnim->Load();

  InitLevel();

  return TRUE;
}

//-----------------------------------------------------------------------------
// KillGraphics
//-----------------------------------------------------------------------------
void CSpaceFrame::KillGraphics()
{
  KillLevel();

  delete m_pShipAnim;
  delete m_pEnemyAnim;
  delete m_pBulletAnim;

  if (m_pFrameRateSurface != NULL)
    m_pFrameRateSurface->Release();

  if (m_pTextSurface != NULL)
    m_pTextSurface->Release();

  // The ancestral call must occur last, 
  // since it destroys the Direct Draw object
  CGameFrame::KillGraphics();
}

//-----------------------------------------------------------------------------
// InitPalette
//-----------------------------------------------------------------------------
BOOL CSpaceFrame::InitPalette()
{
  CString             fileName = "media/ship0000.bmp";
  int                 nNumColors;
  int                 nFileHandle;
  PALETTEENTRY        pal[256];

  if ((nFileHandle = _lopen(fileName, OF_READ)) != -1)
  {
    BITMAPFILEHEADER bf;
    BITMAPINFOHEADER bi;

    _lread(nFileHandle, &bf, sizeof(bf));
    _lread(nFileHandle, &bi, sizeof(bi));
    _lread(nFileHandle, pal, sizeof(pal));
    _lclose(nFileHandle);

    if (bi.biSize != sizeof(BITMAPINFOHEADER))
      nNumColors = 0;
    else if (bi.biBitCount > 8)
      nNumColors = 0;
    else if (bi.biClrUsed == 0)
      nNumColors = 1 << bi.biBitCount;
    else
      nNumColors = bi.biClrUsed;

    for(int nIndex = 0; nIndex < nNumColors; nIndex++ )
    {
      BYTE red = pal[nIndex].peRed;
      pal[nIndex].peRed  = pal[nIndex].peBlue;
      pal[nIndex].peBlue = red;
    }
  }

  HRESULT hr = m_pDD->CreatePalette(DDPCAPS_8BIT | DDPCAPS_ALLOW256, pal, &m_pGamePalette, NULL);
  if (DD_OK == hr)
  {
    m_pFrontBuffer->SetPalette(m_pGamePalette);
    return TRUE;
  }
  else
    return FALSE;
}

//-----------------------------------------------------------------------------
// RestoreSurfaces
//-----------------------------------------------------------------------------
BOOL CSpaceFrame::RestoreSurfaces()
{
  // NOTE: You MUST call the parent class's RestoreSurfaces function
  if (!CGameFrame::RestoreSurfaces())
    return FALSE;

  m_pFrameRateSurface->Restore();
  m_pTextSurface->Restore();


  m_pShipAnim->Restore();
  m_pEnemyAnim->Restore();
  m_pBulletAnim->Restore();

  return TRUE;
}

//-----------------------------------------------------------------------------
// DrawText
//-----------------------------------------------------------------------------
void CSpaceFrame::DrawText()
{
  HDC hDC;
  char str[9];

  // Calculate the frame rate, if necessary
  m_nFrameCount++;
  DWORD dwTimeDiff = timeGetTime() - m_dwStartTime;

  if (dwTimeDiff > 1000)
  {
    m_nFrameRate = m_nFrameCount * 1000 / dwTimeDiff;
    m_dwStartTime = timeGetTime();
    m_nFrameCount = 0;

    if (m_nFrameRate > 99)
      m_nFrameRate = 99;

    str[0] = char('0' + (m_nFrameRate / 10));
    str[1] = char('0' + (m_nFrameRate % 10));
    str[2] = ' ';

    if (m_pFrameRateSurface->GetDC(&hDC) == DD_OK)
    {
      SetTextColor(hDC, RGB(255,255,255));
      SetBkColor(hDC, RGB(0,0,0));
      HGDIOBJ hOldFont = SelectObject(hDC, m_font);
      TextOut(hDC, 0,0, str, 3);
      SelectObject(hDC, hOldFont);

      m_pFrameRateSurface->ReleaseDC(hDC);
    }
  }

  CRect srcRect(0, 0, 50, 40);
  m_pBackBuffer->BltFast(270, 0, m_pFrameRateSurface, &srcRect, 
                         DDBLTFAST_SRCCOLORKEY);

  wsprintf(str, "Lives: %2d", m_nLives);
  if (m_pTextSurface->GetDC(&hDC) == DD_OK)
  {
    SetTextColor(hDC, RGB(255,255,255));
    SetBkColor(hDC, RGB(0,0,0));
    HGDIOBJ hOldFont = SelectObject(hDC, m_font);
    TextOut(hDC, 0,0, str, 9);
    SelectObject(hDC, hOldFont);

    m_pTextSurface->ReleaseDC(hDC);
  }

  srcRect.SetRect(0, 0, 180, 40);
  m_pBackBuffer->BltFast(0, 439, m_pTextSurface, &srcRect,
                         DDBLTFAST_SRCCOLORKEY);
}

//-----------------------------------------------------------------------------
// RandVal
//-----------------------------------------------------------------------------
int CSpaceFrame::RandVal( int low, int high )
{
    int range = high - low;
    int num = rand() % range;
    return( num + low );
}

//-----------------------------------------------------------------------------
// RandVal
//-----------------------------------------------------------------------------
double CSpaceFrame::RandVal( double low, double high )
{
    double range = high - low;
    double num = range * (double)rand()/(double)RAND_MAX;
    return( num + low );
}


BEGIN_MESSAGE_MAP(CSpaceFrame, CGameFrame)
	//{{AFX_MSG_MAP(CSpaceFrame)
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()
