/***** TASK32.H *****/
/*
 * Version 1.00
 * Header file for Tasker32 programs.
 * All user documentation is contained herein.
 */

/*
 * COPYRIGHT AND DISCLAIMER:
 * This file and associated files are Copyright (c) 1996 by Guy Thornley.
 * Permission is granted for the end user to distribute all end programs that
 * contain these routines and code as long as the copyright notice is not
 * removed or changed in any way.
 * This product is supplied without any express or implied warranty of any
 * kind, that is, it is supplied "AS-IS". It is the end user's
 * responsibilty to cover all support resulting from use or lask of use of
 * this product.
 */

/* #defines for readability and changeability */
/* These come from the .ASM source. */
#define TASK_NUM        10              /* Number of tasks          */
#define TASK_MAX        (TASK_NUM-1)    /* Max task number          */
#define TASK_UNUSED     0               /* Task is unused           */
#define TASK_ACTIVE     1               /* Task is used and active  */
#define TASK_CALLED     2               /* Task has been called     */
#define TASK_PAUSED     4               /* Task is paused           */
#define TASK_FAR        8               /* Task exits with far ret  */
#define TASK_OK         0               /* Task is OK               */
#define TASK_FAILED     1               /* Task is invalid          */
#define TASK_NONE_FREE  65535           /* No free tasks            */

/* Typedefs for useful types */
/* These should be used when interfacing to the tasker. */
typedef unsigned short  ushort;
typedef unsigned long   ulong;
typedef unsigned int    uint;

/* The task entries */
/* For accessing task records in memory. */
#pragma -a1
struct task_struct {
  ulong   poff;
  ushort  psel;
  ulong   left;
  ulong   delay;
  ulong   freq;
  ushort  status;
  ushort  dssel;
};
#pragma -a.

/* External task information. */
extern const struct task_struct tasks[TASK_NUM];

/* Prototypes */
/* Declare the procs that the use can use. */

ushort task_init_tasker(void);
  /* 
   * Initializes the tasker. MUST be called before anything else.
   * Paramaters:
   *    None.
   * Returns:
   *    TASK_OK       For success. Tasker was initialized.
   *    TASK_FAILED   Failure. Tasker cannot be used. Program should
   *                  terminate with an error code. Possible causes of this
   *                  are listed in notes.
   * Notes:
   *    - If it fails a possible cause is an incompatible DOS extender
   *      running before the program starts up and Borland's DOS extender
   *      in DPMI32VM.OVL is not used. In this case, the user should be told
   *      to not use the offending memory manager or make a boot disk.
   *    - This is compatible with Microsoft Windows 3.1x.
   *    - If it fails the interrupt will not be hooked and all other Tasker32
   *      routines may succeed but do nothing.
   */

void task_set_timer_delay(ulong count);
  /* 
   * Sets the delay in the internal hardware timer channel 0 (master
   * clock generator) to count.
   * Paramaters:
   *    count         This is the internal count to set the timer to. Only
   *                  the low 16 bits are used. All others are ignored.
   * Returns:
   *    Nothing.
   * Notes:
   *    - The user should not call this directly but let Tasker32 control
   *      the timer.
   *    - Should the user update the timer speed make sure that the delay
   *      is small enough so fast tasks will still function correctly.
   */

ushort task_add(ushort f, void* p, ushort cs, ushort ds, ushort flg);
  /* 
   * Adds a task to the task queue to run at even intervals.
   * Paramaters:
   *    f             The frequency, in times per second, to call the
   *                  passed routine.
   *    p             The near offset to the function to call.
   *    cs            The code selector of the function to allows far calls.
   *    ds            The preferred data selector of the function.
   *    flg           Aadditional flags or'd together about proc:
   *                  TASK_PAUSED   Set to automatically pause proc.
   *                  TASK_FAR      Set for a far call.
   * Returns:
   *    The task identification number. This should be saved so the caller
   *    can update and control the task.
   *    If TASK_NONE_FREE is returned there was no space left in the task
   *    record structure to hold the task.
   * Notes:
   *    - For normal operation, on calling this the cs paramater should be
   *      set to _CS, ds should be set to _DS and and flg should be set
   *      to 0. Only change this if absolutly necessary and you are aware of
   *      the results.
   *    - TASK_NONE_FREE can be overcome by raising the TASK_NUM number in
   *      the TASK32.ASM source file and reassembling and compiling.
   */

ulong task_getdelay(ushort freq);
  /*
   * Computes the delay for a frequency that is appropriate for the system
   * timer.
   * Paramaters:
   *    freq          The frequency, in timer per second, (Hz), to compute
   *                  the delay for.
   * Returns:
   *    The appropriate delay in system timer ticks to achieve the given
   *    frequency.
   * Notes:
   *    - Only the low 16 bits of the returned delay are used. However all
   *      the others are set to 0 and should stay that way.
   */

ushort task_delete(ushort taskid);
  /*
   * Deletes a task from the task queue. The task will no longer be called
   * and the record is freed up for successive tasks to be added.
   * Paramaters:
   *    taskid        This is the ID number of the task that was returned
   *                  from task_add().
   * Returns:
   *    TASK_OK       For success, task deleted OK.
   *    TASK_FAILED   For failure, task was not deleted.
   * Notes:
   *    - Failure will mean that taskid was invalid.
   */

ushort task_setstatus(ushort taskid, ushort status);
  /*
   * Sets given status bits of a task to control the tasking of the task.
   * Paramaters:
   *    taskid        The task ID number of the task to adjust.
   *    status        The status bits the alter. These can be:
   *                  TASK_PAUSED   To pause the task. The task will not be
   *                                run until this bit is cleared. If the
   *                                task currently has control the task will
   *                                have to exit before this takes effect.
   * Returns:
   *    TASK_OK       For success, task status updated OK.
   *    TASK_FAILED   For failure, task was not updated.
   * Notes:
   *    - Failure will mean that taskid was invalid.
   */

ushort task_clearstatus(ushort taskid, ushort status);
  /*
   * Sets given status bits of a task to control the tasking of the task.
   * Paramaters:
   *    taskid        The task ID number of the task to adjust.
   *    status        The status bits the alter. These can be:
   *                  TASK_PAUSED   To unpause the task. The task will now be
   *                                run if it was previously paused.
   * Returns:
   *    TASK_OK       For success, task status updated OK.
   *    TASK_FAILED   For failure, task was not updated.
   * Notes:
   *    - Failure will mean that taskid was invalid.
   */

void task_end_tasker(void);
  /*
   * Stops the tasker from running, restores interrupt vectors and resets
   * the system timer to normal operation.
   * Paramaters:
   *    None.
   * Returns:
   *    Nothing.
   * Notes:
   *    - This *MUST* be called before ending the program or a system crash
   *      will occur.
   *    - It is suggested that this is installed with atexit() at startup
   *      incase of premature program termination.
   */

/*
 * Syntax for task handlers:
 *    void procname(const ushort tid);
 * Where:
 *    tid can either be ommitted or used. If used, it contains the ID number
 *    of the task that caused it to be called.
 *    No other paramaters are allowed.
 */

