/*
 * uplabel.c  John Dudeck  SIM International  19-Dec-90.
 * simintl@rock.concert.net
 * EasyLink 62013975+
 *
 * Released into the Public Domain.
 *
 * Text filter to generate multiple-up labels from 1-up label text.
 *
 * Command syntax:
 * uplabel [-ln][-sn][-un][-wn] {infile | - } [outfile]
 * -l number of lines per label, default -l6, max -l255
 * -s number of spaces between labels, default -s1, max -s255
 * -u number of columns up, default -u3, max -u255
 * -w width of columns, default -w26, except -w32 for 4-up, max -w255
 * if infile is '-', input is from stdin
 * default outfile is stdout
 *
 * 'uplabel' alone on the command line causes a
 * command summary to be printed to stdout.
 *
 * Processing of input text:
 * The number of lines per label determines how many input lines
 * are to be read in for each label.  Input lines longer than the column
 * width will be truncated.  Input lines shorter than the column width
 * will be padded with spaces.
 *
 * Lines are terminated by \n.  All other characters are passed through.
 * This means that escape codes are passed unmodified.  However, this 
 * may cause the current column to be incorrect, due to non-printing 
 * characters.  
 *
 * Form feeds are passed through, and cause the
 * counting of input lines-per-label to be reset.
 */

#define YES 1
#define NO 0
#include <stdio.h>
#ifdef BSD
#include <strings.h>
#else
#include <string.h>
#define	index	strchr
#endif

void usage(void);

main(argc, argv)
int argc;
char *argv[];
{
   extern char *optarg;
   char *buff;
   int c, linlen, bufsiz;
   int in_fnd = NO, out_fnd = NO;
   int ln_per_lab = 6, cols_up = 3, col_wid = 26, spacs = 1;
   int dflt_wid = YES, eoinput = NO;
   FILE *in_fp, *out_fp = stdout;
   
   if (argc == 1) usage();
   initarg(argc - 1, argv + 1);
   while ((c = getarg("l:s:u:w:")) != 0) {
      switch (c) {
         case 'l':
            ln_per_lab = atoi(optarg);
            if (ln_per_lab < 1 || ln_per_lab > 255) usage();
            break;
         case 's':
            spacs = atoi(optarg);
            if (spacs < 0 || spacs > 255) usage();
            break;
         case 'u':
            cols_up = atoi(optarg);
            if (cols_up < 1 || cols_up > 255) usage();
            break;
         case 'w':
            col_wid = atoi(optarg);
            if (col_wid < 1 || col_wid > 255) usage();
            dflt_wid = NO;
            break;
         case -1:
            if (!in_fnd) {
               if ((in_fp = fopen(optarg, "r")) == NULL) {
                  fprintf(stderr, "Can't open input file %s.\n", optarg);
                  exit(1);
               }
               in_fnd = YES;
            } else if (!out_fnd) {
               if ((out_fp = fopen(optarg, "w")) == NULL) {
                  fprintf(stderr, "Can't open output file %s.\n", optarg);
                  exit(1);
               }
               out_fnd = YES;
            } else {
               usage();
            }
            break;
         case '-':
            in_fp = stdin;
            in_fnd = YES;
            break;
         case '?':
            usage();
      }
   }
   /* Check if default column width needs to be used. */
   if (dflt_wid) col_wid = (cols_up == 4) ? 32 : 26;

   /* Allocate buffer. */
   linlen = cols_up * col_wid + (cols_up - 1) * spacs + 2;
   bufsiz = ln_per_lab * linlen + 1;
   buff = (char *) malloc(bufsiz);
   
   while (!eoinput) {
      int curcol, curlin, curlen, curcnt;
      int ch;
      char *cur;
      
      (void) memset(buff, ' ', bufsiz);
      for (curcol = 0; curcol < cols_up && !eoinput; curcol++) {
         for (curlin = 0; curlin < ln_per_lab && !eoinput; curlin++) {
            cur = buff + (curlin * linlen) + (curcol * (col_wid + spacs));
            curcnt = 0;
            while ((ch = getc(in_fp)) != EOF) {
               if (ch == '\n') break;
               if (curcnt++ < col_wid) {
                  *cur++ = ch;
               }
            }
            if (ch == EOF) eoinput = YES;
         }
      }
      for (curlin = 1; curlin <= ln_per_lab; curlin++) {
         *(buff + curlin * linlen - 1) = '\n';
      }
      *(buff + bufsiz - 1) = '\0';
      fputs(buff, out_fp);
   }
}


/*
 * void usage(void);
 */
void usage(void)
{
   fprintf(stderr, "\n\
Text filter to generate multiple-up labels from 1-up label text.\n\
\n\
Command syntax:\n\
 uplabel [-ln][-sn][-un][-wn] {infile | - } [outfile]\n\
 -l number of lines per label, default -l6, max -l255\n\
 -s number of spaces between labels, default -s1, max -s255\n\
 -u number of columns up, default -u3, max -u255\n\
 -w width of columns, default -w26, except -w32 for 4-up, max -w255\n\
 if infile is '-', input is from stdin\n\
 default outfile is stdout\n");
 
   fprintf(stderr, "\n\
 The number of lines per label determines how many input lines\n\
 are to be read in for each label.  Input lines longer than the column\n\
 width will be truncated.\n\
\n\
 All characters in input are passed through to the output.\n\
 Non-printable characters may cause unexpected results.\n\
\n\
 By John Dudeck, SIM International, 19-Dec-90.  Public Domain.\n\
");
   exit(0);
}
