// semaphor.hxx
//
// defines the OS/2 semaphore class for IPC
//
// author: vaughn vernon
// (c) Copyright 1988 Aspen Scientific
// All Rights Reserved.

#ifndef __SEMCLASS__
# define __SEMCLASS__
# ifndef APIENTRY
#	include <os2def.h>
# endif
# ifndef INCL_DOS
#	define	INCL_DOS
#	include <bsedos.h>
# endif

// the semaphore class definition

class Semaphore {

	union	{
		ULONG	RAM;		// a RAM Semaphore
		HSYSSEM	SYS;		// a System Semaphore
	} Sem;

	HSEM	SemH;			// this actual handle
	USHORT	returnCode;		// a good create/open?

	// a private error routine
	VOID	ErrorOp( CHAR *s )	{
		cout << "\nclass: Semaphore: bad operator: " << s << "\n";
	}

	// the class methods
public:

	// constructor and destructor

	// system sem. constructor
	Semaphore( PSZ sysSemName, USHORT owner, BOOL create ) {

		if ( create )
			returnCode = DosCreateSem(
					(owner ? CSEM_PRIVATE:CSEM_PUBLIC),
					(PHSYSSEM) &Sem.SYS,
					sysSemName );
		else
			returnCode = DosOpenSem( (PHSYSSEM) &Sem.SYS,
					sysSemName );

		SemH = Sem.SYS;		// use system handle
	}

	// RAM sem. constructor
	Semaphore( )		{

		Sem.RAM = 0L;		// initialize
		SemH = &Sem.RAM;	// handle is address; use RAM handle
		returnCode = 0;	
	}

	// destructor may have to close the system resource
	~Semaphore( )		{

		if ( Sem.SYS != HSYSSEM(0) )
			DosCloseSem( Sem.SYS );
	}

	// control methods

	// was the open/create valid?
	BOOL	GetStatus()	{ return (returnCode ? FALSE:TRUE); }

	// SetNow is a forced set, usually ment for thread signaling
	VOID	SetNow()	{ DosSemSet( SemH ); }

	// SetRequest is used to wait on a possibly owned semaphore
	BOOL	SetRequest( LONG timeOut ) {
		return (DosSemRequest( SemH, timeOut ) ? FALSE:TRUE);
	}

	// Wait is used to wait on the clearing of a semaphore.
	// this is typically used to wait on a semaphore signal.
	BOOL	Wait( LONG timeOut )	{
		return DosSemWait( SemH, timeOut );
	}

	// Fire is used to send a signal to a waiting thread
	// that self-set this semaphore
	VOID	Fire()		{ Clear(); }

	// Clear is used to clear ownership of this semaphore
	VOID	Clear()		{ DosSemClear( SemH ); }

	// C++ operator overloading:  implements a signaling
	// environment. "++" sets, and "--" fires.
	VOID	operator++()	{ SetNow(); }
	VOID	operator--()	{ Fire(); }

	// invalid C++ operators for this class: although this
	// is not really necessary, BS recommends it for
	// users of the class other than the inventor.
	// for instance, I sure don't want someone trying to
	// add semaphores!:  sem3 = sem1 + sem2;

	VOID	operator+()	{ ErrorOp( "'+'" ); }
	VOID	operator-()	{ ErrorOp( "'-'" ); }
	VOID	operator+=()	{ ErrorOp( "'+='" ); }
	VOID	operator-=()	{ ErrorOp( "'-='" ); }
	VOID	operator()()	{ ErrorOp( "'()'" ); }
	VOID	operator[]()	{ ErrorOp( "'[]'" ); }
	VOID	operator*()	{ ErrorOp( "'*'" ); }
	// etc....
};

#endif
