/* resource.c	support routines for string resources
 *
 *	author: vaughn vernon
 *
 *	(c) 1990 Aspen Scientific
 *	All Rights Reserved
 */

#include <stdio.h>
#include <string.h>
#ifdef __TURBOC__
#include <alloc.h>
#else
#include <malloc.h>
#endif

#include "resource.h"

static int		lineEatSpace(unsigned char *fbuf);
static StringResource *	makeResource(unsigned char *fbuf);
static void		freeResource(StringResource *res);

/* LoadResources() loads a resource file into memory.
 *
 * returns a resource list pointer, or NULL if an error is found.
 */
StringResource *
LoadResources(char *fileName)
{
	FILE		*fp = fopen(fileName, "r");
	unsigned char	fbuf[BUFSIZ];
	StringResource	*res = NULL, *cur, *tmp;

	if (fp == NULL)
		return (NULL);

	while (fgets(fbuf, sizeof (fbuf), fp)) {

		if (lineEatSpace(fbuf) == 0)
			continue;

		tmp = makeResource(fbuf);

		if (tmp == NULL) {
			FreeResources(res);
			return (NULL);
		}

		if (res == NULL)
			res = cur = tmp;
		else {
			cur->next = tmp;
			cur = tmp;
		}
	}

	fclose(fp);
	return (res);
}

/* makeResource() creates a resource link on the free store.
 *
 * returns a pointer to the new link.
 */
StringResource *
makeResource(unsigned char *fbuf)
{
	StringResource	*res;
	unsigned char	*idName, *idValue, *string, *end;

	res = (StringResource *) malloc(sizeof (StringResource));

	if (res == NULL)
		return (NULL);

	res->idName = res->string = NULL;

	/* parse the line... */

	idName = fbuf;
	idValue = strchr(fbuf, '=');

	if (idName == NULL || idValue == NULL) {
		freeResource(res);
		return (NULL);
	}

	*idValue++ = '\0';

	string = strchr(idValue, ',');

	if (string == NULL) {
		freeResource(res);
		return (NULL);
	}

	*string++ = '\0';
	if (*string != '\"') {
		freeResource(res);
		return (NULL);
	}
	++string;
	end = strrchr(string, '\"');

	if (end == NULL) {
		freeResource(res);
		return (NULL);
	}

	*end = '\0';

	/* set up the link ... */
	res->idName = (unsigned char *) malloc(strlen(idName) + 1);
	res->string = (unsigned char *) malloc(strlen(string) + 1);

	if (res->idName == NULL || res->string == NULL) {
		freeResource(res);
		return (NULL);
	}

	strcpy(res->idName, idName);
	strcpy(res->string, string);

	res->value = (unsigned) atoi(idValue);
	res->magic = _ResourceMagic;
	res->next  = NULL;

	return (res);
}

/* lineEatSpace() is your basic text line parser.
 *
 * returns number of "useable" characters; 0 if all white or comments.
 */
static int
lineEatSpace(unsigned char *fbuf)
{
	register int	quote=0;
	unsigned char	tmp[BUFSIZ+1];
	unsigned char	*p1 = fbuf, *p2 = tmp;

	memset(tmp, 0, sizeof (tmp));

	while (*p1 && *p1 != '\n') {

		switch (*p1)
		{
		case	'#':
			*p1 = '\0';
			break;

		case	'\t':
			++p1;
			continue;

		case	' ':
			if (!quote) {
				++p1;
				continue;
			}

			break;

		case	'\\':
			++p1;
			break;

		case	'\'':
			quote = (quote == 1 ? 0: quote == 2 ? 2:1);
			break;

		case	'\"':
			quote = (quote == 2 ? 0: quote == 1 ? 1:2);
			break;

		default:
			break;
		}

		if (*p1 == '\0')
			break;

		*p2++ = *p1++;
	}

	memcpy(fbuf, tmp, BUFSIZ);

	return (strlen(fbuf));
}

/* FreeResources() frees an entire resource list
 */
void
FreeResources(StringResource *res)
{
	StringResource	*tmp;

	while (res) {
		tmp = res->next;
		freeResource(res);
		res = tmp;
	}
}

/* freeResource() frees an individual resource
 */
void
freeResource(StringResource *res)
{
	if (res == NULL)
		return;

	if (res->idName)
		free( res->idName );
	if (res->string)
		free( res->string );

	res->magic = 0;	/* no reuse */

	free( res );
}

/* LookUpResource() finds a resource matching the given ID.
 *
 * returns the resource pointer, or NULL if not found.
 */
StringResource *
LookUpResource(StringResource *res, unsigned id)
{
	while (res && res->magic == _ResourceMagic) {
		if (res->value == id)
			break;
		res = res->next;
	}

	return (res && res->magic == _ResourceMagic ? res:NULL);
}
