/***************************************************************************/
/* nPlay.c  -- Routines to read and playback sound files                   */
/* Copyright (c) 1996 John A. Ball                                         */
/*                                                                         */
/* This source is available for your own personal use only! You may make   */
/* any changes you wish but please do not distribute modified source code. */
/* Please notify me of any errors or improvements.                         */
/*                                                                         */
/* by John A. Ball   June 29, 1996                                         */
/***************************************************************************/

#define BUFFSIZE 16*1024
#define BIGBUFFER 63*1024
#define getrandom( min, max ) ((rand() % (int)(((max)+1) - (min))) + (min))
#define TRUE 1
#define NOCOMPRESS 100

#include <stdio.h>
#include <dos.h>
#include <errno.h>
#include <stdlib.h>
#include <fcntl.h>
#include <sys\types.h>
#include <sys\stat.h>
#include <io.h>
#include <malloc.h>
#include <math.h>
#include <string.h>
#include <sound.h>

extern void show_card_info(void);
extern void delay(char tc, int period);
extern void show_dac_error(int error);

	long fileinfo(struct find_t *find);


	struct names
	{
	char name[13];
	};

	struct CARD_INFO card_info;
	int verbose=0;
	char *buffer;
	int volume=MIDVOLUME;
	int compression=0;
	int magnitude=0;
	int default_bits=8;
	int halve_buffer=0;
	int stop=0;
	int debug=0;
	int repeat=1;
	int error=0;
	int multi_error=0;
	int random=0;
	unsigned p=0;
	int pp=0;
	int conversion_table=1;
	unsigned int play_rate=11000;
	unsigned int pitch=1;
	char _far **fbuffer;
	char *workbuf;
	unsigned int dma_size=0;
	unsigned long mem_total=600000;
	char source_file_name[80];
	char msg[]="Error! ";
	union REGS inregs, outregs;
	struct SREGS segregs;
	long fileinfo(struct find_t *find);
	long append=0;
	char a_file_name[13];
	struct SOUND info;

char *get_file_name(int argc, char *argv[]);
void help(void);
int play_sound(char *file_name);
char get_file_type(char *file_name);
int read_tandy(struct SOUND *hp, int source_handle);
int read_voc(struct SOUND *hp, int source_handle);
int read_wav(struct SOUND *hp, int source_handle);
int read_iff(struct SOUND *hp, int source_handle);
int read_sam(struct SOUND *hp, int source_handle);
int read_mod(struct SOUND *hp, int source_handle);
int read_steve(struct SOUND *hp, int source_handle);
int read_unknown(struct SOUND *hp, int source_handle);
int read_mac(struct SOUND *hp, int source_handle);
int read_au(struct SOUND *hp, int source_handle);
int play_block(struct SOUND *hp, int source_handle);
int print_file_info(struct SOUND *hp);
unsigned get_dma_size(void);
unsigned long get_mem_size(void);
long mm(long l);
unsigned int mmi(unsigned int i);
void mmf(double long *i);
void to_unsigned(char _far *hp,unsigned int k);
void to_signed(char _far *hp,unsigned int k);
void to_8bit(char _far *hp,unsigned int k,int byte_format);
void down_sample(char _far *hp,unsigned int k);
unsigned int change_frequency(char _far *hp,unsigned int k,unsigned int freq,unsigned int nfreq);
int m_gain(unsigned char _far *hp, unsigned int k);
int p_gain(unsigned char _far *hp, unsigned int k);
void get_speed(char *pdest);
void get_volume(char *pdest);
int get_card_type(void);
int uncompress_ulaw(char _far *hp, unsigned int number_samples);
int uncompress_alaw(char _far *hp, unsigned int number_samples);
void display_menu(void);
int showfiles(char *fnp,int num);
char *get_files(char *fnp);
int find_files(void);
int show_cursor(char *fnp, int cp);
int show_page(char *fnp,int num);
int play_it(char *fnp);
int get_names(char *rnp, char *file_name);
int binexp(int i);
void p_cerror(char *msg, char *name);

int main(int argc, char *argv[])
{
	struct find_t find;
	char file_path[80];
	char file_name[13];
	char file[93];
	char dot_star[]=".*";
	char *name;
	char *path;
	char *pfile;
	char *rnp;
	char *rn;
	int sh='\\';
	int colon=':';
	int dot='.';
	int found;
	int length=0;
	int i=0,k=0;
	int key;
	int num_files=0;

	name=get_file_name(argc, argv);
	if(name != NULL){
	  path=strrchr(name,sh);
	  if(path==NULL)path=strrchr(name,colon);
	  length=(int)(path-name);
	  if(length > 80)length=80;
	  for(i=0;i<length+1;i++)
	   {
	   file_path[i]=source_file_name[i];
	   }
	  file_path[i]='\0';
	  pfile=strcpy(file,file_path);
	  pfile=strcpy(file_name,&source_file_name[i]);
	  name=strrchr(file_name,dot);
	  if(name==NULL)strcat(source_file_name,dot_star);
	  found=_dos_findfirst(source_file_name,_A_NORMAL,&find);
	  if(!found){
	   name=&find.name[0];
	   length=strlen(name);
	   if(length > 12)length=12;
	   k=0;
	   for(i=0;i<length+1;i++)
	    {
	    file_name[i]=find.name[i];
	    }
	   file_name[i]='\0';
	   k++;
	   if(!random){
	    pfile=strcat(file,file_name);
	    play_sound(file);
	    }
	   while(!_dos_findnext(&find))
	   {
	   if(stop)exit(1);
	   name=&find.name[0];
	   length=strlen(name);
	   if(length > 12)length=12;
	   for(i=0;i<length+1;i++)
	    {
	    file_name[i]=find.name[i];
	    }
	   file_name[i]='\0';
	   k++;
	   if(!random){
	    pfile=strcpy(file,file_path);
	    pfile=strcat(file,file_name);
	    play_sound(file);
	    }
	   }
	   num_files=k;
	   if(random){
	    rnp=(char *)malloc(14*num_files);
	    if(rnp){
	       rn=rnp;
	       get_names(rn, source_file_name);
	       srand((unsigned)time( NULL ));
	       k=getrandom(0,num_files-1);
	       rn+=(k*14);
	       pfile=strcpy(file,file_path);
	       pfile=strcat(file,rn);
	       play_sound(file);
	       free(rnp);
	       }
	    else{
	      printf("Failure to allocate memory for file names!\n");
	      multi_error=1;
	    }
	   }
	   }
	   if(found){
	     printf("File %s not found!\n",source_file_name);
	     multi_error=1;
	   }
	  }
	else
	return(99);
}
/* Get Sound file name or give program information if error*/

char *get_file_name(int argc, char *argv[])

{
	char *pdest;
	char options[80];
	int i=0;

	if(argc == 1){
	   return(NULL);
	}

	strcpy(source_file_name, argv[1]);
	strupr(source_file_name);
	if(argc > 2){
	   i=2;
	   do{
	   strcpy(options, argv[i]);
	   strupr(options);
	   if(strstr(options,"V"))verbose=1;
	   if(strstr(options,"!"))magnitude=1;
	   if(strstr(options,"&"))magnitude=-1;
	   if(strstr(options,"#"))default_bits=-8;
	   if(strstr(options,"X"))debug=1;
	   if(strstr(options,"RAN"))random=1;
	   if(pdest=strstr(options,"R:"))get_speed(pdest);
	   if(pdest=strstr(options,"L:"))get_volume(pdest);
	   i++;
	   }
	   while(i < argc);

	   return(source_file_name);
	   }
	if(strstr(source_file_name,"/?")){
	   help();
	   return (NULL);
	   }

	return(source_file_name);
}

/* Print out Program information */

void help(void)
{
	printf("\n" PROGRAM PURPOSE"\n");
	printf("\nPlays VOC, WAV, SND, IFF, AIF, SAM & MOD sound files on ");
	printf(CARD "\n");
	printf("\n"PROGRAM" [drive:][path][filename] [/V][/!][/&][/#][/X][/RAN]");
	printf("[R:xxxxx][L:xxx]\n");
	printf("\n/V Show File Information \n");
	printf("/! Amplify \n");
	printf("/& Silence \n");
	printf("/# Data is 8 Bit Signed\n");
	printf("/X Show Debugging Information\n");
	printf("/RAN Select file at random to play\n");
	printf("R:xxxxx Set Default Playback Rate\n");
	printf("L:xxx Set Default Volume (%i - %i)\n",MINVOLUME,MAXVOLUME);
	printf("\n" VERSION "\n");
	printf("\n" COPYRIGHT "\n");
}

/* Play sound file on PC speaker */

int play_sound(char *file_name)
{
	char type;
	char name[80];
	char *pdest;
	int source_handle;
	unsigned long file_length;
	unsigned long i;
	int j;
	int k;
	unsigned long l;

/* Find Sound Card Available */

	card_info.type=PCSPEAKER;
	card_info.type=get_card_type();

/* Initalize default values */

	info.length=0;
	info.number_samples=0;
	info.frequency=0;
	info.bits=8;
	info.channels=1;
	info.compression=0;             /* Type of hardware compression */


	error=0;
	multi_error=0;
	compression=0;
	stop=0;
	repeat=1;
	halve_buffer=0;
	card_info.mode=0;          /* Compression capabilities of card */
	conversion_table=1;

	strcpy(name,file_name);
	if(verbose)printf("\nSound File %s Information\n",name);

	/* Find out total memory available */

	mem_total=get_mem_size();

	/* Find out size of DMA buffer in protected mode */

	if(dma_size == 0){
	 dma_size=32000;
	 k=get_dma_size();
	 if(k < 32 && k > 0)dma_size=k*1024;
	 else dma_size=32000;
	 if(debug)printf("The DMA buffer size is %u and available memory is %lu bytes\n"
	    ,dma_size,mem_total);
	}

	card_info.dma_size=dma_size;

	/* Allocate memory for work buffer */

	if((workbuf =(char *)malloc(BIGBUFFER)) == NULL){
	  printf("Failed to allocate memory for work buffer!\n");
	  multi_error=1;
	  exit(100);
	}

	/* Allocate memory for buffer pointers */

	k=((mem_total/dma_size) +1);
	if(k < 50)k=50;
	if((fbuffer=(char _far **)malloc((size_t)sizeof(char _far *) * k)) == NULL){
	  printf("Failed to allocate memory for buffer pointers!\n");
	  multi_error=1;
	  exit(100);
	}

	/* Allocate memory for file header buffer*/

	if((buffer =(char *)malloc(BUFFSIZE)) == NULL){
	   perror(msg);
	   printf("Malloc could not allocate.\n");
	   multi_error=1;
	   exit(1);
	}
	source_handle = open(name, O_RDONLY | O_BINARY);
	if(source_handle == -1){
	   perror(msg);
	   printf("Couldn't open sound file %s\n",name);
	   multi_error=1;
	   return(99);
	   }

	if(opensnd())printf("Error opening sound device\n");

	type=get_file_type(name);
	switch( type )
	{
	   case 'S':
	      read_tandy(&info, source_handle);
	      break;
	   case 'V':
	      read_voc(&info, source_handle);
	      break;
	   case 'W':
	      read_wav(&info, source_handle);
	      break;
	   case 'I':
	      read_iff(&info, source_handle);
	      break;
	   case 'A':
	      read_sam(&info, source_handle);
	      break;
	   case 'M':
	      read_mod(&info, source_handle);
	      break;
	   case '?':
	      read_unknown(&info, source_handle);
	      break;
	}

	if(closesnd())printf("Error closing sound device\n");

	close(source_handle);
	free(buffer);
	free(fbuffer);
	free(workbuf);
	if(verbose)show_card_info();
	return(error);
}

	/* Determine type of Sound File by looking at file extension*/

char get_file_type(char *file_name)
{
	char *pdest;
	char name[80];

	strcpy(name,file_name);

	if(strstr(name, ".SND")){
	  return('S');
	  }
	if(strstr(name, ".WAV")){
	  return('W');
	  }
	if(strstr(name, ".VOC")){
	  return('V');
	  }
	if(strstr(name, ".IFF")){
	  return('I');
	  }
	if(strstr(name, ".AIF")){
	  return('I');
	  }
	if(strstr(name, ".SAM")){
	  return('A');
	  }
	if(strstr(name, ".MOD")){
	  return('M');
	  }
	return('?');
}

	/* Read Sound File Information */
	/* Tandy SND, VOC, WAV, & IFF Files Supported */

/* Tandy SND files */

int read_tandy(struct SOUND *hp, int source_handle)
{
	struct TANDY header;
	struct TANDY_R n_sample[16];
	struct OLDTANDY oldheader;
	struct OLDTANDY_R o_sample[16];
	long position=0;
	int samples=0;
	int i=0;

	hp->byte_format=INTEL;

	read(source_handle,&header, sizeof(struct TANDY));
	if(header.id == 32794) {
	   hp->frequency=header.frequency;
	   samples=header.number_clips;
	   if(samples>16)samples=16;
	   if(verbose)printf("Tandy SND File %s",&header.name);
	   if(verbose)printf("\tcontains %i sound sample(s)\n",samples);
	   if(header.compression != 0)compression=TCOMP;
	   hp->bits=8;

	   for(i=0; i<samples;i++){
	     read(source_handle,&n_sample[i],sizeof(struct TANDY_R));
	   }

	   for(i=0; i<samples;i++){
	     hp->length=n_sample[i].length;
	     hp->number_samples=n_sample[i].number_samples;
	     position=n_sample[i].start;
	     lseek(source_handle,position,SEEK_SET);
	     play_block(hp,source_handle);
	     if(error)return(error);
	     }
	   }
	else if(header.name[0] == 0x1a) {
	   lseek(source_handle,position,SEEK_SET);
	   read(source_handle,&oldheader, sizeof(struct OLDTANDY));
	   if(oldheader.id == 0x1A) {
	      hp->frequency=oldheader.frequency;
	      samples=oldheader.number_clips;
	      if(samples>16)samples=16;
	      if(verbose)printf("Old Tandy SND File %s",oldheader.name);
	      if(verbose)printf("\tcontains %i sound sample(s)\n",samples);
	      if(oldheader.compression != 0)compression=TCOMP;
	      hp->bits=8;

	   for(i=0; i<samples;i++){
	     read(source_handle,&o_sample[i],sizeof(struct OLDTANDY_R));
	   }

	      for(i=0;i<samples;i++){
		hp->length=o_sample[i].length;
		hp->number_samples=o_sample[i].number_samples;
		position=o_sample[i].start;
		lseek(source_handle,position,SEEK_SET);
		play_block(hp,source_handle);
		if(error)return(error);
		}
	      }
	   }
	   else read_unknown(hp, source_handle);
}

/* Creative Labs VOC files */

int read_voc(struct SOUND *hp, int source_handle)
{
	struct LABS header;
	struct LABSV110 data;
	struct LABSV120 ndata;

	char *pdest;
	unsigned int temp=0;
	unsigned long int temp1=0;
	int terminate_not_found=1;
	unsigned char type=0;
	int bytes_read=0;
	unsigned int period=0;
	unsigned char tc=0;
	unsigned int marker=0;
	
	hp->byte_format=INTEL;

	read(source_handle,&header, sizeof(struct LABS));
	if(strstr(header.id , "Creative Voice File")){

	   if(verbose)printf("Creative Labs ver %i.%i VOC file\n"
	   ,header.major_version,header.minor_version);

	   hp->bits=8;

	   do{
	      bytes_read=read(source_handle,&type,1);
	      if(bytes_read < 1)type=0;
	   switch(type)
	   {
	   case TERMINATE:
	      if(debug) printf("Terminate Block\n");
	      terminate_not_found=0;
	      break;
	   case VOICE:
	      if(debug) printf("Voice Block\n");
	      read(source_handle,&data, sizeof(struct LABSV110));
	      if(hp->channels == 1)
	      hp->frequency=(unsigned) (1000000 / (256-data.frequency));
	      else
	      hp->frequency=(unsigned) (500000 / (256-data.frequency));
	      temp=data.number_samples1;
	      temp=temp + data.number_samples2 * 256;
	      temp1=(unsigned long)(temp+ data.number_samples3 * 65536);
	      hp->length=temp1-2;
	      hp->number_samples=temp1-2;
	      switch (data.compression)
	      {
	      case 1:
		compression=ADPCM4;
		info.bits=4;
		break;
	      case 2:
		compression=ADPCM26;
		info.bits=3;
		break;
	      case 3:
		compression=ADPCM2;
		info.bits=2;
		break;
	      }
	      error=play_block(hp,source_handle);
	      if(error)return(error);
	      break;
	   case CONTINUE:
	      if(debug) printf("Voice Continuation Block\n");
	      read(source_handle,buffer, 3);
	      temp=buffer[0];
	      temp=temp + buffer[1] * 256;
	      temp1=(unsigned long)(temp+ buffer[2] * 65536);
	      hp->length=(unsigned int)temp1;
	      error=play_block(hp,source_handle);
	      if(error)return(error);
	      break;
	   case SILENCE:
	      read(source_handle,buffer, 3);
	      read(source_handle,&period, sizeof(period));
	      read(source_handle,&tc, sizeof(tc));
	      if(debug) printf("Silence Block %x %x \n",tc,period);
	      delay(tc,period);
	      break;
	   case MARKER:
	      read(source_handle,buffer, 3);
	      read(source_handle,&marker, sizeof(marker));
	      if(debug) printf("Marker Block %x\n",marker);
	      break;
	   case ASCII:
	      if(debug) printf("ASCII Block\n");
	      read(source_handle,buffer, 3);
	      temp=buffer[0];
	      temp=temp + buffer[1] * 256;
	      temp1=(unsigned long)(temp+ buffer[2] * 65536);
	      temp=(unsigned int)temp1;
	      temp=read(source_handle,buffer, temp);
	      buffer[temp]=0;
	      if(verbose)printf("%s\n",buffer);
	      break;
	   case REPEAT:
	      read(source_handle,buffer, 3);
	      temp=buffer[0];
	      temp=temp + buffer[1] * 256;
	      temp1=(unsigned long)(temp+ buffer[2] * 65536);
	      temp=(unsigned int)temp1;
	      temp=read(source_handle,buffer, temp);
	      repeat=repeat+temp;
	      if(debug) printf("Repeat Block %i\n",repeat);
	      break;
	   case VOCEND:
	      if(debug) printf("End Repeat Block\n");
	      read(source_handle,buffer, 3);
	      break;
	   case EXTEND:
	      if(debug) printf("Extend Block\n");
	      read(source_handle,buffer, 3);
	      temp=buffer[0];
	      temp=temp + buffer[1] * 256;
	      temp1=(unsigned long)(temp+ buffer[2] * 65536);
	      temp=(unsigned int)temp1;
	      temp=read(source_handle,buffer, temp);
	      hp->channels=buffer[3]+1;
	      temp=buffer[1]*256  + buffer[0];
	      /* future enhancements go here */
	      switch (buffer[2])
	      {
	      case 1:
		compression=ADPCM4;
		info.bits=4;
		break;
	      case 2:
		compression=ADPCM26;
		info.bits=3;
		break;
	      case 3:
		compression=ADPCM2;
		info.bits=2;
		break;
	      }
	      break;
	   case NEWVOC:
	      if(debug) printf("New Voice Block\n");
	      read(source_handle,&ndata, sizeof(struct LABSV120));
	      temp=(unsigned int)ndata.frequency1;
	      hp->frequency=(unsigned int)(ndata.frequency2 * 256 + temp);
	      temp=ndata.number_samples1;
	      temp=temp + ndata.number_samples2 * 256;
	      temp1=(unsigned long)(temp+ ndata.number_samples3 * 65536);
	      hp->length=temp1 - 12;
	      hp->bits=ndata.bits_per_sample;
	      hp->channels=ndata.channels;

	      switch(ndata.compression)
	      {
	      case 0x00:
		 error=play_block(hp,source_handle);
		 if(error)return(error);
		 break;
	      case 0x01:
		 compression=ADPCM4;
		 info.bits=4;
		 error=play_block(hp,source_handle);
		 if(error)return(error);
		 break;
	      case 0x02:
		 compression=ADPCM26;
		 info.bits=3;
		 error=play_block(hp,source_handle);
		 if(error)return(error);
		 break;
	      case 0x03:
		 compression=ADPCM2;
		 info.bits=2;
		 error=play_block(hp,source_handle);
		 if(error)return(error);
		 break;
	      case 0x04:
		 hp->bits=-16;
		 error=play_block(hp,source_handle);
		 if(error)return(error);
		 break;
	      case 0x05:
		 error=play_block(hp,source_handle);
		 if(error)return(error);
		 break;
	      default:
		 if(debug)printf("Unknown Compression Type");
		 break;
	     }
	   default:
	      /* skip unwanted or unknown block data */
	      read(source_handle,buffer, 3);
	      temp=buffer[0];
	      temp=temp + buffer[1] * 256;
	      temp1=(unsigned long)(temp+ buffer[2] * 65536);
	      temp=(unsigned int)temp1;
	      temp=read(source_handle,buffer, temp);
	      if(debug)printf("Skipping Unknown Voc chunk of %u bytes\n",temp);
	      break;
	   }
	   }while(terminate_not_found && !error);
	  }
	else read_unknown(hp, source_handle);
}


/* Microsoft Wave Files */

int read_wav(struct SOUND *hp, int source_handle)
{
	struct RIFFWAVE riff;
	struct SUBCHUNK subchunk;
	struct WAVEfmt  * wavefmt;
	struct RIFFLIST rifflist;
	struct FACTinfo fact;

	int data_not_found=1;
	unsigned long temp=0;

	hp->byte_format=INTEL;
	temp=0;
	read(source_handle,&riff,sizeof(struct RIFFWAVE));
	if((riff.id==RIFF) && (riff.form_type==WAVE)){

	if(verbose)printf("Microsoft WAVE file\n");

	     do{
	      read(source_handle,&subchunk,sizeof(struct SUBCHUNK));

	      switch(subchunk.id)
	      {
	       case FMT:
		 wavefmt=(struct WAVEfmt *)malloc((int)subchunk.subchunk_size);
		 read(source_handle,wavefmt,(int)subchunk.subchunk_size);
		 switch(wavefmt->format_tag)
		 {
		  case  2:
		    compression=MSADPCM;
		    break;
		  case  6:
		    compression=ALAW;
		    info.bits=14;
		    break;
		  case  7:
		    compression=ULAW;
		    info.bits=14;
		    break;
		  case 17:
		    compression=ADPCM;
		    break;
		  case 512:
		    compression=ADPCM4;
		    break;
		  default:
		    break;
		 }
		 hp->bits=wavefmt->bits_per_sample;
		 if(wavefmt->bits_per_sample == 16)hp->bits= -16;
		 temp=(wavefmt->bytes_per_second*8/(wavefmt->channels*abs(hp->bits)));
		 hp->frequency=(unsigned)temp;
		 hp->channels=wavefmt->channels;
		 free(wavefmt);
		 break;
	       case WDATA:
		 hp->length=subchunk.subchunk_size;
		 error=play_block(hp,source_handle);
		 if(error)return(error);
		 data_not_found=0;
		 break;
	       case WLIST:
		 /* do nothing with data and read past to next chunk */
		 read(source_handle,buffer,(int)subchunk.subchunk_size);
		 break;
	       case FACT:
		 read(source_handle,&fact,(int)subchunk.subchunk_size);
		 hp->number_samples=fact.number_samples;
		 break;
	       default:
		 /* Skip unknown subchunk information */
		 read(source_handle,buffer,(int)subchunk.subchunk_size);
		 if(debug)printf("Skipping Unknown WAV Subchunk!\n");
		 break;

	       }
	      } while(data_not_found);
	   }
	   else read_unknown(hp, source_handle);
}

/* Apple & Electronic Arts IFF Sound Files */

int read_iff(struct SOUND *hp, int source_handle)
{
	struct SOUNDIFF iff;
	struct SUBCHUNK subchunk;
	struct IFFvhdr vhdr;
	struct AIFFCOMM aiff;

	int data_not_found=1;
	int temp=0;
	unsigned int exponent=0;
	unsigned int mantissa=0;
	unsigned long l=0;

	hp->byte_format=MM;
	read(source_handle,&iff,sizeof(struct SOUNDIFF));

	if(iff.id==FORM){

	if((verbose) && (iff.form_type==SVX))printf("Electronic Arts IFF file\n");
	if((verbose) && (iff.form_type==AIFF))printf("Apple AIFF file\n");

	     do{
	      read(source_handle,&subchunk,sizeof(struct SUBCHUNK));

	      switch(subchunk.id)
	      {
	       case VHDR:
		 read(source_handle,&vhdr,(int)mm(subchunk.subchunk_size));
		 temp=vhdr.samples_per_second;
		 hp->frequency=(((temp & 0xff00L) >> 8) +
		  ((temp & 0x00ffL) << 8));
		 if(vhdr.compression != 0)compression=UNKNOWN;
		 break;
	       case COMM:
		 read(source_handle,&aiff,(int)mm(subchunk.subchunk_size));
		 hp->channels=mmi(aiff.channels);
		 hp->number_samples=mm(aiff.number_samples);
		 hp->bits=-mmi(aiff.bits_per_sample);
		 mmf(&aiff.frequency);
		 hp->frequency=(unsigned)aiff.frequency;
		 break;
	       case BODY:
		 hp->length=mm(subchunk.subchunk_size);
		 hp->bits= -8;
		 error=play_block(hp,source_handle);
		 if(error)return(error);
		 data_not_found=0;
		 break;
	       case SSND:
		 hp->length=mm(subchunk.subchunk_size);
		 error=play_block(hp,source_handle);
		 if(error)return(error);
		 data_not_found=0;
		 break;
	       case INST:
		 read(source_handle,buffer,(int)mm(subchunk.subchunk_size));
		 break;
	       case ANNO:
		 temp=(int)mm(subchunk.subchunk_size);
		 read(source_handle,buffer,temp);
		 buffer[temp]=0;
		 if(verbose)printf("%s\n",buffer);
		 break;
	       case AUTH:
		 temp=(int)mm(subchunk.subchunk_size);
		 read(source_handle,buffer,temp);
		 buffer[temp]=0;
		 if(verbose)printf("The author is %s\n",buffer);
		 break;
	       case NAME:
		 temp=(int)mm(subchunk.subchunk_size);
		 read(source_handle,buffer,temp);
		 buffer[temp]=0;
		 if(verbose)printf("Name is %s\n",buffer);
		 break;
	       default:
		 /* Skip unknown subchunk information */
		 read(source_handle,buffer,(int)mm(subchunk.subchunk_size));
		 if(debug)printf("Skipping Unknown IFF Subchunk!\n");
		 break;

	       }
	      } while(data_not_found);
	   }
	   else read_unknown(hp, source_handle);

}

/* MOD SAMple file   */

int read_sam(struct SOUND *hp, int source_handle)
{
	long position=0;

	hp->byte_format=MM;
	if(verbose) printf("Mod SAMple file.\n");
	lseek(source_handle,position,SEEK_SET);
	hp->bits= -8;
	if(play_rate==11000)play_rate=8000;
	hp->frequency=play_rate;
	hp->length=filelength(source_handle);
	error=play_block(hp,source_handle);
	if(error)return(error);
}

/* Amiga MOD file */

int read_mod(struct SOUND *hp, int source_handle)
{
	struct MODhdr header;
	struct MODsample sample[31];
	struct MODraw raw[31];
	struct MODsong song;
	int i=0;
	int j=0;
	int k=0;
	int patch=0;                    /* Sample Number */
	unsigned int note=0;            /* Note */
	int effect=0;                   /* Effect Command */
	int e_data=0;                   /* Effect Data */
	int max=0;                      /* Number of 1024byte patterns */
	int number_samples=0;
	int max_sam=0;
	int newmod=0;
	long int n=0;
	char *ppattern[128];
	long position=1080;              /* check for type of Mod */

	hp->byte_format=MM;
	lseek(source_handle,position,SEEK_SET);
	read(source_handle,&n,sizeof(n));

	switch(n)
	{
	case M_k_:
	case FLT4:
	   max_sam=31;
	   newmod=1;
	break;
	default:
	   max_sam=15;
	break;
	}
	position=0;                        /* restore file pointer */

	lseek(source_handle,position,SEEK_SET);

	read(source_handle,&header,sizeof(struct MODhdr));
	for(i=0;i < max_sam;i++){
	  read(source_handle,&sample[i],sizeof(struct MODsample));
	  n=0;
	  n=(long)mmi(sample[i].number_samples)*2;
	  raw[i].number_samples=n;
	  if(debug && sample[i].name[0] != 0)printf("Sample %i\t%lu\t%.22s\n",
	  i+1,n,sample[i].name);
	}
	j=0;
	read(source_handle,&song,sizeof(struct MODsong));

	 j=(int)song.s_length;
	 if(debug)printf("Song length is %i\t",j);
	 if(newmod)read(source_handle,&n,sizeof(n));
	 if(debug && newmod)printf("File id is %.4s\n",&n);
	 for(i=0; i < j;i++){
	   if((int)song.sequence[i] > max)max=(int)song.sequence[i];
	   if(debug)printf("%i ",(int)song.sequence[i]);
	 }
	 max=max+1;
	 if(debug)printf("\nNumber of Patterns is %i\n",max);
	 for(i=0; i < max;i++){
	   ppattern[i]=(char *)malloc(1024);
	   if(ppattern[i] == NULL){
	      printf("Not enough memory for patterns!\n");
	      multi_error=1;
	      exit(1);
	   }
	   k=read(source_handle,ppattern[i],1024);
	   if(k != 1024){
	     printf("Error reading pattern %u\n",k);
	     multi_error=1;
	   }
	 }

	for(i=0; i<31;i++){
	 raw[i].frequency=0;
	}

	 for(i=0; i < max;i++){
	   if(debug)printf("\nSample\tNote\tEffect\tData\tPattern %i\n\n",i+1);
	   for(j=0; j < 1024;j=j+4){
	    patch=(((*(ppattern[i]+j)&240) | ((*(ppattern[i]+j+2)&240) >> 4))&31);
	    note=((((*(ppattern[i]+j)&15) << 8) | (*(ppattern[i]+j+1)&255)&1023));
	    effect=(*(ppattern[i]+j+2)&15);
	    e_data=(*(ppattern[i]+j+3)&255);
	    if(debug && (note != 0))printf("%u\t%u\t%X\t%X\n",patch,note,
	    effect,e_data);
	    if(note != 0)raw[patch-1].frequency=note;
	   }
	 }
	 for(i=0; i < max_sam;i++){
	 hp->bits=-8;
	 hp->number_samples=raw[i].number_samples;
	 hp->length=raw[i].number_samples;
	  if(raw[i].number_samples > 0){
	   if(verbose)printf("Sample %i\t%.22s\n",i+1,sample[i].name);
	   j=(sample[i].finetune & 15);
	   if ((j&8) == 8)j=j-16;
	   if(debug)printf("Fine Tune =%i\tVolume=%i\n",
	   j,(int)sample[i].volume);
	   if(debug)printf("Repeat Start=%u\tRepeat Length=%u\n",
	   mmi(sample[i].r_start)*2,mmi(sample[i].r_length)*2);
	   if(raw[i].frequency == 0)raw[i].frequency=428;
	   note=((55930/raw[i].frequency) <<  6);
	   raw[i].frequency=note;
	   hp->frequency=raw[i].frequency;
	   error=play_block(hp,source_handle);
	   if(error)break;
	   if(error)break;
	  }
	 }
	 for(i=0;i<max;i++)free(ppattern[i]);
	 return(error);
}

int     read_steve(struct SOUND *hp, int source_handle)
{
	struct STEVE steve;
	long position=0;
	long int length=0;

	hp->byte_format=INTEL;
	lseek(source_handle,position,SEEK_SET);

	read(source_handle,&steve,sizeof(struct STEVE));

	if(verbose)printf("STEVE Sound file\n");
	conversion_table=0;
	hp->length=steve.number_samples;
	if(steve.version==2)hp->frequency=steve.frequency;
	else hp->frequency=8900;
	error=play_block(hp,source_handle);
	return(error);
}

/* Check for MAC, AU, WAV, STEVE files by looking for Header */

int read_unknown(struct SOUND *hp, int source_handle)
{
	union unknown_header
	{
	struct MACBINARY mac_header;
	struct SOUNDIFF iff;
	struct SUN au_header;
	struct RIFFWAVE riff;
	struct LABS labs;
	struct STEVE steve;
	} snd;

	long position=0;

	lseek(source_handle,position,SEEK_SET);

	read(source_handle,&snd.mac_header,MAX_HEADER);

	if(snd.mac_header.type==MACTYPE)read_mac(hp,source_handle);

	else
	  if(snd.au_header.id==AU_SND)read_au(hp,source_handle);

	else
	  if((snd.riff.id==RIFF) && (snd.riff.form_type==WAVE)){
	    lseek(source_handle,position,SEEK_SET);
	    read_wav(hp,source_handle);
	  }

	else
	  if(strstr(snd.labs.id , "Creative Voice File")){
	    lseek(source_handle,position,SEEK_SET);
	    read_voc(hp,source_handle);
	  }

	else
	  if(strstr(snd.steve.id , "STEVE")){
	    lseek(source_handle,position,SEEK_SET);
	    read_steve(hp,source_handle);
	  }

	else
	  if(snd.iff.id==FORM){
	    lseek(source_handle,position,SEEK_SET);
	    read_iff(hp,source_handle);
	  }

	else{
	  if(verbose) printf("Unknown file format using defaults for playback.\n");
	  hp->byte_format=INTEL;
	  lseek(source_handle,position,SEEK_SET);
	  error=play_block(hp,source_handle);
	  if(error)return(error);
	  }
}

int read_mac(struct SOUND *hp, int source_handle)
{
	struct MACBINARY mac_header;
	long position=0;
	long int length=0;
	long int mac_comp=0;

	hp->byte_format=MM;
	lseek(source_handle,position,SEEK_SET);

	read(source_handle,&mac_header,sizeof(struct MACBINARY));

	if(verbose)printf("MAC Sound file %s\n",&mac_header.name);
	length=mm(mac_header.datafork_size);
	hp->length=length-4;
	hp->number_samples=length;
	hp->bits=8;
	read(source_handle,&mac_comp,sizeof(mac_comp));
	if(mac_comp==HCOM)compression=HCOM;
	error=play_block(hp,source_handle);
	return(error);
}

int read_au(struct SOUND *hp, int source_handle)
{
	struct SUN au_header;
	long position=0;
	long int length=0;
	long au_type=0;

	hp->byte_format=MM;
	lseek(source_handle,position,SEEK_SET);

	read(source_handle,&au_header,sizeof(struct SUN));

	if(verbose)printf("SUN au file\n");
	hp->length=mm(au_header.data_size);
	hp->number_samples=mm(au_header.data_size);
	hp->frequency=mm(au_header.frequency);
	hp->channels=mm(au_header.channels);
	au_type=mm(au_header.compression);
	switch(au_type)
	{
	case 1:
	  info.bits= 14;
	  compression=ULAW;
	  break;
	case 2:
	  hp->bits=-8;
	  break;
	case 3:
	  hp->bits= -16;
	  break;
	default:
	  if(debug)printf("Unknown or not supported .au file\n");
	  break;
	}
	position=mm(au_header.chunk_size);

	lseek(source_handle,position,SEEK_SET);
	error=play_block(hp,source_handle);
	if(error)return(error);
}

int play_block(struct SOUND *hp, int source_handle)
{
	long freq=0;
	long i=0;
	int type_comp=0;
	unsigned k=0;
	unsigned block_size=0;
	unsigned int m=0;
	unsigned frequency=0;
	int thru=0;
	long num_bytes=0;
	int file_error=0;
	int bit_sign=0;

	if(hp->length ==0)hp->length=hp->number_samples;
	if(hp->length ==0)hp->length=filelength(source_handle);
	if(hp->bits ==0)hp->bits=default_bits;
	if(hp->number_samples ==0){
	  i=hp->length*8/(abs(hp->bits));
	  hp->number_samples=i;
	}

	/* Check for 0 Playback Rate */

	if(hp->frequency == 0)hp->frequency=play_rate;

	num_bytes=hp->length;
	type_comp=compression;

	if(hp->channels==0){
	  hp->channels=1;
	  printf("Defaulting to single channel\n");
	}

	if(!thru)print_file_info(hp);

	/* Allocate memory for sound data */

	k=block_size=dma_size/2;

	   if((fbuffer[0] =(char *)_fmalloc((size_t)k)) == NULL){
	     printf("Error allocating memory for playback buffer\n");
	   }

	/* Play all samples */

	do{

	compression=type_comp;
	i=0;

	/* Read Sound File Data into Memory */

	m=0;
	k=block_size=dma_size/2;

	  if(num_bytes < (long)k) k=num_bytes;
	   m = read(source_handle,fbuffer[0],k);
	  if(m!=k){
	    if(m<k)num_bytes=m;
	    if(m>k)goto end_play_block;
	  }

/* Check for compressed files */

	if(compression){

	  switch(compression)
	  {
	  case ULAW:
	   hp->bits=-8;
	   card_info.mode=(card_info.mode | BIT8_M);
	   if(num_bytes < (long)k)k=num_bytes;
	      uncompress_ulaw(fbuffer[0],k);
	   if(debug)printf("Converted %u ULAW samples to 8 bit data\n",k);
	   break;
	  case ALAW:
	   hp->bits=-8;
	   card_info.mode=(card_info.mode | BIT8_M);
	   if(num_bytes < (long)k)k=num_bytes;
	      uncompress_alaw(fbuffer[0],k);
	   if(debug)printf("Converted %u ALAW samples to 8 bit data\n",k);
	   break;
	  }
	}
/* Set card mode for 8 and 16 bit data */

	if((hp->bits == 8) && hp->channels==1)
	  card_info.mode=(card_info.mode | BIT8_M);
	if((hp->bits == -8) && hp->channels==1)
	  card_info.mode=(card_info.mode | BIT_MINUS | BIT8_M);

/* Convert 16 bit to 8 bit data for non SB16 cards */

       if(card_info.type != BLASTER16){
	if((hp->bits > 8) || (hp->bits <  -8) && !compression){
	  info.bits=-16;
	    if(num_bytes < (long)k)k=num_bytes;
	      to_8bit(fbuffer[0],k,hp->byte_format);
	  if(debug)printf("Converted %u samples to 8 bit data\n",k);
	  hp->length=hp->length/2;
	  block_size=block_size/2;
	  k=k/2;
	}
       }
       else if((hp->bits == 16) || (hp->bits == -16)){
	 if(hp->channels==2)card_info.mode=(card_info.mode | BIT16_S);
	 else card_info.mode=(card_info.mode | BIT16_M);
	 }

/* Convert Data to Unsigned 8 bit if required for non SB16 cards */

       if(card_info.type != BLASTER16){
	if(hp->bits < 0){
	    if(num_bytes < (long)k)k=num_bytes;
	      to_unsigned(fbuffer[0],k);
	if(debug)printf("Converted %u samples to unsigned data\n",k);
	}
       }

/* Decrease Amplitude if desired */

	if(magnitude < 0){
	    if(num_bytes < (long)k)k=num_bytes;
	      m_gain(fbuffer[0],k);
	if(debug)printf("%u samples silenced\n",k);
	}

/* Increase Amplitude if desired */

	if(magnitude > 0){
	    if(num_bytes < (long)k)k=num_bytes;
	      p_gain(fbuffer[0],k);
	  if(debug)printf("%u samples amplified\n",k);
	}

	frequency=hp->frequency;

/* Double Playback Frequency for stereo files on non SB16 cards*/

	freq=hp->frequency;

	if(hp->channels==2){
	  if(card_info.type != BLASTER16){
	    freq=hp->frequency*2;
	    frequency=hp->frequency *2;
	    if(card_info.type==BLASTERPRO)
	      card_info.mode=card_info.mode=(card_info.mode | BIT8_S);
	  }
	  else{
	    if((hp->bits > 8) || (hp->bits <  -8))
	      card_info.mode=card_info.mode=(card_info.mode | BIT16_S);
	    else card_info.mode=card_info.mode=(card_info.mode | BIT8_S);
	  }
	}


/* Halve Playback Frequency if greater than 44100 Hertz */

	while ((freq * hp->channels) > 44100 )
	{
	if(num_bytes < (long)k)k=num_bytes;
	down_sample(fbuffer[0],k);
	if(debug)printf("Down sampling %u samples @%lu hertz\n",k,freq);
	block_size=block_size/2;
	freq=freq/2;
	hp->length=hp->length/2;
	k=k/2;
	frequency=freq;
	}

/* Play Back Sound Samples */

	compression=(compression & 0xffcf);     /* ULAW & ALAW ok */

	if(hp->bits < 0)bit_sign=BIT_MINUS;
	else bit_sign=BIT_PLUS;

/* display error message for compressed files */

	if(compression)error=compression;
	card_info.mode=(card_info.mode | compression | bit_sign);

	if(verbose && (card_info.type!=PCSPEAKER) && !compression){
	  if(!thru)printf("\nUse the <Esc> key to stop playback!\n\n");
	}

	    if(num_bytes < (long)k)k=num_bytes;
	    if(k>0){
	       error=dacplay((char _far *)fbuffer[0],k,frequency,card_info.mode,volume);
	      }
	     else error=NOCOMPRESS;

	num_bytes-=m;

	if(error > 0 && error!=NOCOMPRESS)show_dac_error(error);

	if(error < 0){
	 if(verbose)printf("Sound output ended!\n");
	 stop=TRUE;
	}
	thru=1;
	} while((num_bytes > 0) && !error);

end_play_block:

	if(fbuffer[0]!=NULL){
	  _ffree(fbuffer[0]);
	}

	if(error==NOCOMPRESS && (card_info.type==PCSPEAKER)){
	    printf("Type of compression not supported by PC speaker!\n");
	}
	hp->number_samples=0;
	multi_error=file_error;
	append=0;
	return(error);
}

/* convert Little Endian to Big Endian (Motorolla to Intel) */

long mm(long l)
{

	return(((l & 0xff000000L) >> 24) +
	       ((l & 0x00ff0000L) >> 8) +
	       ((l & 0x0000ff00L) << 8) +
	       ((l & 0x000000ffL) << 24));
}

/* convert little Endian INT to Big Endian (MM to intel) */

unsigned int mmi(unsigned int n)
{
	return(((n & 0xff00) >> 8) + ((n & 0x00ff) << 8));
}

/* convert Little Endian to Big Endian (Motorolla to Intel) */

void mmf(double long *i)
{
	unsigned j=0;
	unsigned k=9;
	char *number;
	char temp[12];

	number=(char *)i;

	for(j=0;j<10;j++){
	   temp[j]=number[k];
	   k--;
	}
	for(j=0;j<10;j++){
	*((char *)i+j)=temp[j];
	}
}

/* convert to unsigned int */

void to_unsigned(char _far *hp,unsigned int k)
{
	unsigned int i=0;

	if(&hp==NULL){
	  printf("Convert to unsigned pointer error\n");
	  multi_error=1;
	  return;
	}
	if(k<1){
	  printf("Convert to unsigned int error\n");
	  multi_error=1;
	  return;
	}
	do
	{
	hp[i]=hp[i]+128;
	i++;
	} while(i < k);
}

/* convert to signed int */

void to_signed(char _far *hp,unsigned int k)
{
	unsigned int i=0;

	if(&hp==NULL){
	  printf("Convert to signed pointer error\n");
	  multi_error=1;
	  return;
	}
	if(k<1){
	  printf("Convert to signed int error\n");
	  multi_error=1;
	  return;
	}
	do
	{
	hp[i]=hp[i]-128;
	i++;
	} while(i < k);
}

/* convert 16 bit to 8 bit data */

void to_8bit(char _far *hp, unsigned int k,signed int byte_format)
{
	unsigned int i=0;
	unsigned int j=0;
	int n=1;

	if(&hp==NULL){
	  printf("Convert to 8 bit pointer error\n");
	  multi_error=1;
	  return;
	}
	if(k<1){
	  printf("Convert to 8 bit int error\n");
	  multi_error=1;
	  return;
	}
	if(byte_format==INTEL)n=1;
	if(byte_format==MM)n=0;
	do
	{
	hp[i]=hp[j+n];
	i++;
	j++;
	j++;
	} while(j < k);
}

void down_sample(char _far *hp, unsigned int k)
{
	unsigned int i=0;
	unsigned int j=0;

	if(&hp==NULL){
	  printf("Down sample pointer error\n");
	  multi_error=1;
	  return;
	}
	if(k<1){
	  printf("Down sample int error\n");
	  multi_error=1;
	  return;
	}
	do
	{
	hp[i]=hp[j];
	i++;
	j++;
	hp[i]=hp[j];
	i++;
	j+=3;
	} while(j < k);
}

int m_gain(unsigned char _far *hp, unsigned int k)
{
	unsigned int i=0;

	if(&hp==NULL){
	  printf("m_gain pointer error\n");
	  multi_error=1;
	  return(-1);
	}
	if(k<1){
	  printf("m_gain int error\n");
	  multi_error=1;
	  return(-1);
	}
	do
	{
	hp[i]=((hp[i]-128) >> 1)+128;
	i++;
	} while(i < k);
}

int p_gain(unsigned char _far *hp, unsigned int k)
{
	unsigned int i=0;

	if(&hp==NULL){
	  printf("p_gain pointer error\n");
	  multi_error=1;
	  return(-1);
	}
	if(k<1){
	  printf("p_gain int error\n");
	  multi_error=1;
	  return(-1);
	}
	do
	{
	hp[i]=((hp[i]-128) << 1)+128;
	i++;
	} while(i < k);
}

/* Change Default Playback Rate for unknown files */

void get_speed(char *pdest)
{
	char command;

	sscanf(pdest,"%c%c%u",&command,&command,&play_rate);
	if(debug)printf("Default Playback Rate changed to %u hz\n",play_rate);
}

unsigned get_dma_size(void)
{
	unsigned int dma_buffer_size=0;

	_asm{
	push di
	push si
	mov ax,354bh            ; Check for INT 4b vector
	int 21h                 ; Call DOS to get it
	mov dx,es
	cmp dx,0
	je no_irq               ; Skip if vector is zero
	mov ax,8102h            ; Call DMA services to find
	mov dx,0                ; buffer size
	int 4bh
	jnc dpm
no_irq: mov ax,64000            ; If Virtual DMA Function not available
	mov dx,0                ; use 64k buffer
	jmp gdexit
dpm:
	mov ax,si
	mov cx,7
	shl ax,cl
	mov cx,10
	mov bx,di
	shr bx,cl
	add ax,bx
gdexit:        
	mov dma_buffer_size,ax
	pop si
	pop di
	}
	return(dma_buffer_size);
}

/* Print File Information if wanted */

int print_file_info(struct SOUND *hp)
{
	int i=0;

	i=abs(info.bits);
	if(verbose){
	   printf("%i Bit",i);
	   if(hp->channels==1)printf(" Mono");
	   if(hp->channels==2)printf(" Stereo");
	   if(hp->bits < 0)printf(" Signed Data  ");
	   if(hp->bits >= 0)printf(" Unsigned Data");
	   printf("\tFrequency: %u hz\n",hp->frequency);
	   printf("Number of samples: %lu  \t",hp->number_samples/hp->channels);
	   if(compression){
	    switch(compression)
	    {
	      case ADPCM4:
		printf("4-Bit ADPCM Compression");
		break;
	      case ADPCM26:
		printf("2.6-Bit ADPCM Compression");
		break;
	      case ADPCM2:
		printf("2-Bit ADPCM Compression");
		break;
	      case ALAW:
		printf("ALAW Compression");
		break;
	      case ADPCM:
		printf("ADPCM Compression");
		break;
	      case MSADPCM:
		printf("MSADPCM Compression");
		break;
	      case TCOMP:
		printf("Tandy Compression");
		break;
	      case ULAW:
		printf("ULAW Compression");
		break;
	      default:
		printf("Unknown Compression");
		break;
	    }}
	   printf("\n");
       }
}

/* Change frequency of samples to freq */

unsigned int change_frequency(char _far *hp,unsigned int k,unsigned int freq,unsigned int nfreq)
{
	unsigned int i=0;
	unsigned int new_k;
	long int temp=0;
	temp=(long)nfreq*k;
	new_k=temp/freq;
	i=0;

	do{
	  workbuf[i]=*(hp+((long)freq*i/nfreq));
	  i++;
	}while(i < new_k);
	return(i);
}

/* Uncompress ULAW to 8 bit data */

int uncompress_ulaw(char _far *hp, unsigned int number_samples)
{
	int ulaw_table[256]={
	-124,-120,-116,-112,-108,-104,-100,-96,
	-92,-88,-84,-80,-76,-72,-68,-64,
	-62,-60,-58,-56,-54,-52,-50,-48,
	-46,-44,-42,-40,-38,-36,-34,-32,
	-31,-30,-29,-28,-27,-26,-25,-24,
	-23,-22,-21,-20,-19,-18,-17,-16,
	-15,-15,-14,-14,-13,-12,-12,-12,
	-11,-10,-10,-10,-9,-8,-8,-8,
	-7,-7,-7,-7,-6,-6,-6,-6,
	-5,-5,-5,-5,-4,-4,-4,-4,
	-3,-3,-3,-3,-3,-3,-3,-3,
	-2,-2,-2,-2,-2,-2,-2,-2,
	-1,-1,-1,-1,-1,-1,-1,-1,
	-1,-1,-1,-1,-1,-1,-1,-1,
	0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,
	124,120,116,112,108,104,100,96,
	92,88,84,80,76,72,68,64,
	62,60,58,56,54,52,50,48,
	46,44,42,40,38,36,34,32,
	31,30,29,28,27,26,25,24,
	23,22,21,20,19,18,17,16,
	15,15,14,14,13,12,12,12,
	11,10,10,10,9,8,8,8,
	7,7,7,7,6,6,6,6,
	5,5,5,5,4,4,4,4,
	3,3,3,3,3,3,3,3,
	2,2,2,2,2,2,2,2,
	1,1,1,1,1,1,1,1,
	1,1,1,1,1,1,1,1,
	0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0};

	unsigned int i=0;

	do
	{
	  hp[i]=ulaw_table[(hp[i] & 0xff)];
	i++;
	} while(i < number_samples);
}

/* Uncompress ALAW to 8 bit data */

int uncompress_alaw(char _far *hp, unsigned int number_samples)
{
	int alaw_table[256]={
	-22,-21,-24,-23,-18,-17,-20,-19,
	-30,-29,-32,-31,-26,-25,-28,-27,
	-11,-10,-12,-11,-9,-8,-10,-9,
	-15,-14,-16,-15,-13,-12,-14,-13,
	-86,-82,-94,-90,-70,-66,-78,-74,
	-118,-114,-126,-122,-102,-98,-110,-106,
	-43,-41,-47,-45,-35,-33,-39,-37,
	-59,-57,-63,-61,-51,-49,-55,-53,
	-1,-1,-1,-1,-1,-1,-1,-1,
	-2,-2,-2,-2,-2,-2,-2,-2,
	0,0,0,0,0,0,0,0,
	-1,-1,-1,-1,-1,-1,-1,-1,
	-5,-5,-6,-6,-4,-4,-5,-5,
	-7,-7,-8,-7,-6,-6,-7,-7,
	-3,-3,-3,-3,-2,-2,-2,-2,
	-4,-4,-4,-4,-3,-3,-3,-3,
	22,21,24,23,18,17,20,19,
	30,29,32,31,26,25,28,27,
	11,10,12,11,9,8,10,9,
	15,14,16,15,13,12,14,13,
	86,82,94,90,70,66,78,74,
	118,114,126,122,102,98,110,106,
	43,41,47,45,35,33,39,37,
	59,57,63,61,51,49,55,53,
	1,1,1,1,1,1,1,1,
	2,2,2,2,2,2,2,2,
	0,0,0,0,0,0,0,0,
	1,1,1,1,1,1,1,1,
	5,5,6,6,4,4,5,5,
	7,7,8,7,6,6,7,7,
	3,3,3,3,2,2,2,2,
	4,4,4,4,3,3,3,3};

	unsigned int i=0;

	do
	{
	  hp[i]=alaw_table[(hp[i] & 0xff)];
	i++;
	} while(i < number_samples);
}


int get_names(char *rnp, char *file_name)
{
	struct find_t find;
	char *rn;

	rn=rnp;
	if(!_dos_findfirst(file_name,_A_NORMAL, &find)){
	   strcpy(rn,find.name);
	   rn+=14;
	   while(!_dos_findnext(&find)){
	     strcpy(rn,find.name);
	     rn+=14;
	   }
	}
	return(0);
}

/* Change Default Volume */

void get_volume(char *pdest)
{
	char command;

	sscanf(pdest,"%c%c%ui",&command,&command,&volume);
	volume=(volume & MAXVOLUME);
	if(debug)printf("Volume changed to %u\n",volume);
}

int binexp(int exponent)
{
	unsigned int i=0;
	unsigned int k=2;

	for(i=0;i<exponent;i++){
	k=k*2;
	}
	return(k);
}

/* get memory available in pages */

unsigned long get_mem_size(void)
{
	unsigned int mem_size=0;

	_asm{
	mov bx,0ffffh
	mov ah,048h
	int 21h
	jnc mem_error
	cmp ax,08h
	jne mem_error
	mov mem_size,bx
mem_error:
	}
	return((long)mem_size *16);
}

