/*
 *     Program: FREQ.C
 *      Author: Philip VanBaren
 *        Date: 15 August 1993
 *
 * Description: This program samples data from a sound card, performs an FFT,
 *              and displays the result.
 *              Can handle up to 2048 points (actually any size is possible
 *              with a little fiddling of buffers to get around 64k limits).
 *              (This restriction is given freq.h, and may be changed.)
 *              On a 486/33 this code can perform and plot 1024-point and
 *              below in nearly real-time at 44100kHz.  (1024 FFT=31ms)
 *
 *  The DOS portion of this code was written for Borland C, but should work 
 *  with other compilers if the graphics and console-io base functions are
 *  changed appropriately.  The UNIX specific stuff has been compiled using
 *  GCC on Linux and on Sun Workstations.
 *
 *  The source for specific graphics environments and sound cards may require
 *  other packages.  Refer to the specific files for more details.
 *
 *  Most changes are required only in the sc_*.c and gr_*.c files.
 *
 *  Copyright (C) 1995  Philip VanBaren
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "freq.h"
#include "fft.h"
#include "extern.h"
#include "display.h"

/*
 *  Table for approximating the logarithm.
 *  These values are round(log2(index/16)*16384) for index=0:31
 */
/*
long ln[]={ -262144L,-65536L,-49152L,-39568L,-32768L,-27494,-23184,-19540,
	    -16384,-13600,-11110,-8857,-6800,-4908,-3156,-1526,0,1433,2784,
	    4062,5274,6428,7527,8578,9584,10549,11476,12368,13228,14057,
	    14858,15634 };
*/
 /*  These values are round(log2(index/16)*8192) for index=0:31 */
long ln[]={ -131072L,-32768L,-24576L,-19784L,-16384L,-13747L,-11592L,
            -9770L,-8192L,-6800L,-5555L,-4428L,-3400L,-2454L,-1578L,
            -763L,0L,716L,1392L,2031L,2637L,3214L,3764L,4289L,4792L,
            5274L,5738L,6184L,6614L,7029L,7429L,7817L };

volatile int flag[BUFFERS]; /* Array of flags indicating fullness of buffers */
volatile int record_buffer; /* Pointer to next buffer to be filled */
int queue_buffer;           /* Pointer to next buffer to be queued */
int process_buffer;         /* Pointer to next buffer to be FFTed */

short *fftdata;           /* Array for FFT data */
short *wind;              /* Array storing windowing function */

int x[WINDOW_RIGHT-WINDOW_LEFT+1];     /* Array of bin #'s displayed */
int x2[WINDOW_RIGHT-WINDOW_LEFT+1];    /* Array of terminal bin #'s */
int lasty[WINDOW_RIGHT-WINDOW_LEFT+1]; /* Last y position for FFT bins */
unsigned int yscale[WINDOW_RIGHT-WINDOW_LEFT+1]; /* Scaling factors */
long *ybase;            /* Scaling offset for log calculations */

float *lastval_f;       /* Last value buffer for exponential decay mode */
long *lastval;          /* Last value buffer for uniform averaging */
long *displayval;
long FAR **databuf;     /* Memory for averaging mode */
int shift=0;            /* Number of bits for gain shift */
double shiftscale=1;    /* Multiplication factor that does this shift */
float log_scalefactor;  /* Scaling factor for log values */
float disp_scalefactor; /* Display scalefactor for log values */

void FAR *buffer[BUFFERS]; /* Buffers for gathering data */

short *p1,*p2;        /* Various indexing pointers */
int *bri;
long *pDisplayval;
long FAR *pDatabuf;
long *pLastval;
float *pLastval_f;
int *pLasty;
long *pYbase;
unsigned int *pYscale;
int *pX,*pX2;

unsigned char FAR *sample8;
short FAR *sample16;

long a2,root,mask;      /* Variables for computing Sqrt/Log of Amplitude^2 */
long peak_amp;          /* Peak amplitude found */
int peak_index;         /* Bin number of the peak amplitude */
float decay_factor=0.95; /* Geometric decay factor */
int decay_count=1;      /* Temporal averaging parameter */
int decay_count_actual=0; /* Total number of bins averaged so far */
int max_decay_count;    /* Maximum value for the above parameter */
int decay_ptr=0;        /* Index for current averaging buffer location */
long back1,back2;       /* Variables for differencing */
char ini_file[100];     /* Filename for the ini file */
int barmode=1;          /* Flag for bar display or dot display */
int done=0;             /* Flag indicating program should exit*/

int main(int argc,char *argv[],char *environ[])
{
   int i;
   long y;
   int first;
   int key=0;

   draw_init();

   DOUT("Getting the command line arguments");

   /*
    *  Check if the first parameter is an ini file name
    */
   if(argc>1 && argv[1][0]!='-' && argv[1][0]!='/')
   {
      strncpy(ini_file,argv[1],sizeof(ini_file));
      first=2;
   }
   else
   {
      strncpy(ini_file,"freq.ini",sizeof(ini_file));
      first=1;
   }

   /*
    *  Parse the ini file and command line
    */
   DOUT("Parsing the ini file");
   parse_ini_file();

   DOUT("Parsing the command line");
   parse_command((argc-first),&argv[first],environ);

   /*
    *  Initialize the buffer info for the maximum size we will encounter
    */
   DOUT("Allocating the buffer space");
   setup_buffers(MAX_LEN);

   DOUT("Computing the window functions");
   compute_window_function();

   /*
    *  Set up the required arrays in the FFT code.
    */
   DOUT("Initializing the FFT code");
   InitializeFFT(fftlen);

   /*
    *  Initialize the graphics to 640x480 VGA mode
    */
   DOUT("Setting the graphics mode");
   setup_graphics();

   setnormalpalette();
   draw_fontcolor(TEXT_COLOR);
   draw_rectangle(WINDOW_LEFT-2,WINDOW_TOP-2,
		  WINDOW_RIGHT+2,WINDOW_BOTTOM+2,BORDER_COLOR);

   DOUT("Resetting the sound card");
   reset_soundcard();

   /*
    *  Initalize the graph scales
    */
   setup_xscale();
   amplitude_scale();

   DOUT("Drawing the header information");
   update_header();

   /*
    *  Keep getting data and plotting it.
    *  A space will pause, a second space will continue.
    *  Any other key will quit.
    */
   DOUT("Entering data loop");
   while(!done)
   {
      /* Wait for current buffer to fill up, and check for an input */
      int input;
      key=draw_getkey();
      while((!flag[process_buffer] || freeze) && !key)
	 key=draw_getkey();
      input=key;
      while(input)
      {
	 /* Grab and count repeated keystrokes */
	 int repetitions=1;

	 key=input;
	 input=draw_getkey();
	 while(input==key)
	 {
	    repetitions++;
	    input=draw_getkey();
	 }
	 done |= process_input(key,repetitions);
      }

      if(!key && !freeze)
      {
	 int clip=0;
	 /*
	  *  Perform windowing on the data
	  */
	 p1=fftdata;
	 p2=wind;

	 if(deriv==0)
	 {
	    if(sample_size==8)
	    {
	       sample8=(unsigned char FAR *)buffer[process_buffer];
	       for(i=0;i<fftlen;i++)
	       {
		  if((*sample8==0) || (*sample8==255))
		     clip=1;
		  *p1=(short)(( ((long)(*sample8)-128L)  * (long)(*p2)) >> 7);
		  sample8++;
		  p1++;
		  p2++;
	       }
	    }
	    else
	    {
	       sample16=(short FAR *)buffer[process_buffer];

	       for(i=0;i<fftlen;i++)
	       {
		  if((*sample16==32767) || (*sample16==-32768L))
		     clip=1;
		  *p1=(short)(((long)(*sample16) * (long)(*p2)) >> 15);
		  sample16++;
		  p1++;
		  p2++;
	       }
	    }
	 }
	 else if(deriv==1)
	 {
	    if(sample_size==8)
	    {
	       sample8=(unsigned char FAR *)buffer[process_buffer];
	       back1=*sample8;
	       for(i=0;i<fftlen;i++)
	       {
		  if((*sample8==0) || (*sample8==255))
		     clip=1;
		  *p1=(short)((((long)(*sample8)-back1) * (long)(*p2)) >> 7);
		  back1=*sample8;
		  sample8++;
		  p1++;
		  p2++;
	       }
	    }
	    else
	    {
	       sample16=(short FAR *)buffer[process_buffer];
	       back1=*sample16;
	       for(i=0;i<fftlen;i++)
	       {
		  if((*sample16==32767) || (*sample16==-32768L))
		     clip=1;
		  *p1=(short)((((long)(*sample16)-back1) * (long)(*p2)) >> 15);
		  back1=*sample16;
		  sample16++;
		  p1++;
		  p2++;
	       }
	    }
	 }
	 else   /* deriv==2 */
	 {
	    if(sample_size==8)
	    {
	       sample8=(unsigned char FAR *)buffer[process_buffer];
	       back1=back2=*sample8;

	       for(i=0;i<fftlen;i++)
	       {
		  if((*sample8==0) || (*sample8==255))
		     clip=1;
		  *p1=(short)((((long)(*sample8)-2*back1+back2) * (long)(*p2)) >> 7);
		  back2=back1;
		  back1=*sample8;
		  sample8++;
		  p1++;
		  p2++;
	       }
	    }
	    else
	    {
	       sample16=(short FAR *)buffer[process_buffer];
	       back1=back2=*sample16;
	       for(i=0;i<fftlen;i++)
	       {
		  if((*sample16==32767) || (*sample16==-32768L))
		     clip=1;
		  *p1=(short)((((long)(*sample16)-2*back1+back2) * (long)(*p2)) >> 15);
		  back2=back1;
		  back1=*sample16;
		  sample16++;
		  p1++;
		  p2++;
	       }
	    }
	 }

	 if(clip)
	    draw_setpalette(0,warn.red,warn.green,warn.blue);
	 else
	    draw_setpalette(0,background.red,background.green,background.blue);

	 /* Pass the Waveform data to a processing routine, if defined */
         #ifdef PASS_WAVEDATA
	    PASS_WAVEDATA(buffer[process_buffer],fftlen,sample_size);
         #endif

	 /* Free up the buffer we just processed. */
	 flag[process_buffer]=0;
	 if(++process_buffer>=BUFFERS)
	    process_buffer=0;

	 /* Now that we have processed the buffer, queue it up again. */
	 recordblock(buffer[queue_buffer]);
	 if(++queue_buffer>=BUFFERS)
	    queue_buffer=0;

	 /* The real meat of the code lies elsewhere! */
	 RealFFT(fftdata);

	 /* Pass the FFT data to a processing routine, if defined */
         #ifdef PASS_FFTDATA
	    PASS_FFTDATA(fftdata,fftlen);
         #endif
      }

      /* Use #define NODISPLAY to skip the display part of the code */
      /* This is useful if you just want to log the data */
      #ifndef NODISPLAY

      if(freeze)   /* Unhighlight the bin in freeze mode */
	 highlight(0);

      /*
       *  First, go through the data set and convert it to magnitude form
       */
      if(!freeze)
      {
	 if(++decay_ptr>=decay_count) decay_ptr=0;
         decay_count_actual++;
         /* If using infinite averaging, reset at 30000 */
         if(!decay_count && (decay_count_actual>30000))
         {
            /* Reset the decay buffers */
            reset_decay_buffers();
            decay_count_actual=1;
         }
         if(decay_count && (decay_count_actual>decay_count))
            decay_count_actual=decay_count;
         else if(!help_mode && (decay_mode==3))
         {
            char buffer[3];
            int x=DCX+6.5*_font_width;
            if(decay_count)
            {
               sprintf(buffer,"%2d",decay_count_actual);
               draw_bar(x,DCY,x+2*_font_width,DCY+_font_height-1,0);
               draw_text_left(x,DCY,buffer);
            }
            else
            {
               sprintf(buffer,"%5d",decay_count_actual);
               draw_bar(x,DCY,x+5*_font_width,DCY+_font_height-1,0);
               draw_text_left(x,DCY,buffer);
            }
         }

	 /* Use sqrt(a2) in averaging mode and linear-amplitude mode */
	 if(decay_mode || !logamp || logfreq==2)
	 {
	    /* Use pointers for indexing to speed things up a bit. */
	    bri=BitReversed;
	    pDisplayval=displayval;
	    pDatabuf=databuf[decay_ptr];
	    pLastval=lastval;
	    pLastval_f=lastval_f;

	    for(i=0;i<fftlen/2;i++)
	    {
	       /* Compute the magnitude */
	       register long re=fftdata[*bri];
	       register long im=fftdata[(*bri)+1];
	       register long root;
	       if((a2=re*re+im*im)<0) a2=0;  /* Watch for possible overflow */
               #ifdef USE_INTEGER_CODE
	          /* Use higher resolution only for small values */
	          if(a2>4194304L)
		  {
		     root=32;
		     do
		     {
			mask=a2/root;
			root=(root+mask)>>1;
		     } while(labs(root-mask)>1);
		     root*=16;
		  }
		  else
		  {
		     root=512;
		     a2*=256;
		     do
		     {
			mask=a2/root;
			root=(root+mask)>>1;
		     } while(labs(root-mask)>1);
		  }
               #else
                  root=sqrt(a2)*16;
               #endif
	       /* In decay mode, need to average this value */
	       switch(decay_mode)
	       {
		case 1:
		  *pLastval_f*=decay_factor;
		  if(root<*pLastval_f)
		     root=*pLastval_f;
		  else
		     *pLastval_f=root;
    	          pLastval_f++;
		  break;
		case 2:
		  *pLastval_f=decay_factor * *pLastval_f + (1-decay_factor) * root;
		  root=floor(*pLastval_f);
    	          pLastval_f++;
		  break;
		case 3:
                  if(decay_count)
                  {
                     *pLastval+=root-*pDatabuf;
                     *pDatabuf=root;
                  }
                  else
                     *pLastval+=root;
		  root=*pLastval/decay_count_actual;
    	          pLastval++;
		  break;
	       }
	       *pDisplayval=root;
	       bri++;
	       pDisplayval++;
	       pDatabuf++;
	    }
	 }
         else /* No averaging, log-amplitude mode */
	 {
	    /* Use pointers for indexing to speed things up a bit. */
	    bri=BitReversed;
	    pDisplayval=displayval;
	    pYbase=ybase;

	    for(i=0;i<fftlen/2;i++)
	    {
	       /* Compute the log(squared magnitude) */
	       register long re=fftdata[*bri];
	       register long im=fftdata[(*bri)+1];
	       register long root=32768L;
	       if((a2=re*re+im*im)<0) a2=0;  /* Watch for possible overflow */
               #ifdef USE_INTEGER_CODE
	          while(a2>=32)
	          {
		     root+=8192;
		     a2>>=1;
	          }
	          root+=ln[(int)a2]-*pYbase;
               #else
	          if(a2>0)
		     root=log(a2)*log_scalefactor-*pYbase;
	          else
		     root=0;
               #endif
	       if(root<0) root=0;
	       *pDisplayval=root;
	       bri++;
	       pDisplayval++;
	       pYbase++;
	    }
	 }
      }

      /*
       *  Next, put this data up on the display
       */
      #ifdef DOS
	 setup_vga();   /* Prepare VGA for video update */
      #endif

      pLasty=lasty;
      pX=x;
      pX2=x2;
      pYscale=yscale;
      peak_amp=0;
      peak_index=0;
      y=WINDOW_BOTTOM;
      if(logfreq==2)
      {
	 /* Equalizer mode.  Display bands as RMS value of the set of bins */
	 int index;
	 for(i=WINDOW_LEFT;i<=WINDOW_RIGHT;i++)
	 {
	    index=*pX;
	    if(index>=0)
	    {
	       register double dv=0;
	       for(;index<*pX2;index++)
		  dv+=(double)displayval[index]*(double)displayval[index];
	       dv=sqrt(dv/(index-*pX));
	       if(logamp)
	       {
		  if(dv>0)
		     dv=(log(dv/16.0)*2*log_scalefactor - ybase[index])
			* disp_scalefactor;
		  if(dv<0) dv=0;
	       }
	       else
	       {
		  dv*=*pYscale*shiftscale;
	       }
	       y=(WINDOW_BOTTOM) - floor(dv+0.5);
	       if(y<WINDOW_TOP) y=WINDOW_TOP;
	       if(y>WINDOW_BOTTOM) y=WINDOW_BOTTOM;
	       if(dv>peak_amp)
	       {
		  peak_amp=dv;
		  peak_index=(*pX+*pX2)/2;
	       }
	    }
            if(barmode)
            {
               if(y>*pLasty)
               {
                  /* Draw a black line */
                  #ifdef DOS
                     unsigned char bit=~(0x80 >> (i&0x07));
                     unsigned int endbase=(unsigned int)(y*80);
                     unsigned int base=(unsigned int)(*pLasty*80+(i>>3));
                     while(base<endbase)
                     {
                        screen(base)&=bit;
                        base+=80;
                     }
                  #else
                     draw_line(i,y-1,i,*pLasty,0);
                  #endif /* DOS */
               }
               else
               {
                  /* Draw a blue line. */
                  #ifdef DOS
                     unsigned char bit=0x80 >> (i&0x07);
                     unsigned int endbase=(unsigned int)((*pLasty+1)*80);
                     unsigned int base=(unsigned int)(y*80+(i>>3));

                     while(base<endbase)
                     {
                        screen(base)|=bit;
                        base+=80;
                     }
                  #else
                     draw_line(i,*pLasty,i,y,GRAPH_COLOR);
                  #endif /* DOS */
               }
	    }
            else /* Draw and erase dots */
            {
               #ifdef DOS
                  unsigned char bit=0x80 >> (i&0x07);
                  screen((*pLasty*80+(i>>3)))&=~bit;
                  screen((y*80+(i>>3)))|=bit;
               #else
                  /* Erase the old dot */
                  draw_pixel(i,*pLasty,0);
                  /* Draw the new dot */
                  draw_pixel(i,y,GRAPH_COLOR);
               #endif
            }

	    *pLasty=(unsigned int)y;
	    pX++;
	    pX2++;
	    pLasty++;
	    pYscale++;
	 }
      }
      else if(decay_mode && logamp)
      {
	 /* In log-amp decay mode, need to do some special things */
	 int xval;
	 int index;
	 for(i=WINDOW_LEFT;i<=WINDOW_RIGHT;i++)
	 {
	    /*
	     *  If this line is the same as the previous one,
	     *  just use the previous y value.
	     *  Else go ahead and compute the value.
	     */
	    index=*pX;
	    if(index!=-1)
	    {
	       /* Convert the amplitude values to log scale */
	       register double val;
	       register long dv=displayval[index];
	       if(*pX2) /* Take the max of a set of bins */
	       {
		  for(xval=index;xval<*pX2;xval++)
		  {
		     if(displayval[xval]>dv)
		     {
			dv=displayval[xval];
			index=xval;
		     }
		  }
	       }
	       if(dv)
		  val=(log(dv/16.0)*2*log_scalefactor - ybase[index])
                      * disp_scalefactor;
	       else
		  val=0;
	       if(val<0) val=0;
	       y=(WINDOW_BOTTOM) - val;
	       if(y<WINDOW_TOP) y=WINDOW_TOP;
	       if(y>WINDOW_BOTTOM) y=WINDOW_BOTTOM;
	       if(dv>peak_amp)
	       {
		  peak_amp=dv;
		  peak_index=*pX;
	       }
	    }
            if(barmode)
            {
               if(y>*pLasty)
               {
                  /* Draw a black line */
                  #ifdef DOS
                     unsigned char bit=~(0x80 >> (i&0x07));
                     unsigned int endbase=(unsigned int)(y*80);
                     unsigned int base=(unsigned int)(*pLasty*80+(i>>3));
                     while(base<endbase)
                     {
                        screen(base)&=bit;
                        base+=80;
                     }
                  #else
                     draw_line(i,y-1,i,*pLasty,0);
                  #endif /* DOS */
               }
               else
               {
                  /* Draw a blue line. */
                  #ifdef DOS
                     unsigned char bit=0x80 >> (i&0x07);
                     unsigned int endbase=(unsigned int)((*pLasty+1)*80);
                     unsigned base=(unsigned int)(y*80+(i>>3));

                     while(base<endbase)
                     {
                        screen(base)|=bit;
                        base+=80;
                     }
                  #else
                     draw_line(i,*pLasty,i,y,GRAPH_COLOR);
                  #endif /* DOS */
               }
	    }
            else /* Draw and erase dots */
            {
               #ifdef DOS
                  unsigned char bit=0x80 >> (i&0x07);
                  screen((*pLasty*80+(i>>3)))&=~bit;
                  screen((y*80+(i>>3)))|=bit;
               #else
                  /* Erase the old dot */
                  draw_pixel(i,*pLasty,0);
                  /* Draw the new dot */
                  draw_pixel(i,y,GRAPH_COLOR);
               #endif
            }

	    *pLasty=(unsigned int)y;
	    pX++;
	    pX2++;
	    pLasty++;
	    pYscale++;
	 }
      }
      else /* For linear amplitude mode and log amp without decay */
      {
	 int index,xval;
	 for(i=WINDOW_LEFT;i<WINDOW_RIGHT+1;i++)
	 {
	    /*
	     *  If this line is the same as the previous one,
	     *  just use the previous y value.
	     *  Else go ahead and compute the value.
	     */
	    index=*pX;
	    if(index!=-1)
	    {
	       register long dv=displayval[index];
	       if(*pX2)  /* Take the maximum of a set of bins */
	       {
		  for(xval=index;xval<*pX2;xval++)
		  {
		     if(displayval[xval]>dv)
		     {
			dv=displayval[xval];
			index=xval;
		     }
		  }
	       }
	       y=(WINDOW_BOTTOM) - ((dv * *pYscale) >> shift) ;
	       if(y<WINDOW_TOP) y=WINDOW_TOP;
	       if(dv>peak_amp)
	       {
		  peak_amp=dv;
		  peak_index=*pX;
	       }
	    }
            if(barmode)  /* Draw bars */
            {
               if(y>*pLasty)
               {
                  /* Draw a black line */
                  #ifdef DOS
                     unsigned char bit=~(0x80 >> (i&0x07));
                     unsigned int endbase=(unsigned int)(y*80);
                     unsigned int base=(unsigned int)(*pLasty*80+(i>>3));
                     while(base<endbase)
                     {
                        screen(base)&=bit;
                        base+=80;
                     }
                  #else
                     draw_line(i,y-1,i,*pLasty,0);
                  #endif /* DOS */
               }
               else
               {
                  /* Draw a blue line. */
                  #ifdef DOS
                     unsigned char bit=0x80 >> (i&0x07);
                     unsigned int endbase=(unsigned int)((*pLasty+1)*80);
                     unsigned int base=(unsigned int)(y*80+(i>>3));

                     while(base<endbase)
                     {
                        screen(base)|=bit;
                        base+=80;
                     }
                  #else
                     draw_line(i,*pLasty,i,y,GRAPH_COLOR);
                  #endif /* DOS */
               }
            }
            else /* Draw and erase dots */
            {
               #ifdef DOS
                  unsigned char bit=0x80 >> (i&0x07);
                  screen((*pLasty*80+(i>>3)))&=~bit;
                  screen((y*80+(i>>3)))|=bit;
               #else
                  /* Erase the old dot */
                  draw_pixel(i,*pLasty,0);
                  /* Draw the new dot */
                  draw_pixel(i,y,GRAPH_COLOR);
               #endif
            }

	    *pLasty=(unsigned int)y;
	    pDisplayval++;
	    pX++;
	    pX2++;
	    pLasty++;
	    pYscale++;
	 }
      }
      #ifdef DOS
	 cleanup_vga();  /* Reset VGA for normal functions */
      #endif

      if(display_peak)
      {
	 char ach[20];
	 sprintf(ach,"%7.1f",(double)SampleRate*peak_index/fftlen);
	 draw_bar(PKX,PKY-1,PKX+63,PKY+_font_height,0);
	 draw_text_left(PKX,PKY,ach);
      }
      if(freeze)     /* Highlight the bin in freeze mode */
	 highlight(1);
      #endif /* NODISPLAY */
   }
   /*
    *  Shut down the DMA system.
    */
   cleanup_soundcard();

   cleanup_graphics();

   printf("You have been using Freq " VERSION);
#ifdef SC_PAS16
   if(Soundcard==SC_PAS16) printf(" in ProAudio Spectrum 16-bit mode.");
#endif
#ifdef SC_SB8
   if(Soundcard==SC_SB8) printf(" in Soundblaster 8-bit mode.");
#endif
#ifdef SC_SB16
   if(Soundcard==SC_SB16) printf(" in Soundblaster 16-bit mode.");
#endif
#ifdef SC_VESA
   if(Soundcard==SC_VESA) printf(" in VESA audio driver %d-bit mode.",sample_size);
#endif
#ifdef SC_LINUX
   if(Soundcard==SC_LINUX) printf(" in Linux audio driver %d-bit mode.",sample_size);
#endif
#ifdef SC_MULAW
   if(Soundcard==SC_MULAW) printf(" in 8-bit mu-law audio driver mode.");
#endif
   puts("\nCopyright (C) " COPYRIGHT " Philip VanBaren");
   return(0);
}


