;-----------------------------------------------------------------------------
 title 'SBVol - A resident utility to easily access the SB''s mixer settings.'
;-----------------------------------------------------------------------------
; Written by Jannie Hanekom.
;
; Contact address:  Jannie Hanekom
;                   P.O. Box 40
;                   Prince Alfred's Hamlet
;                   6840
;                   South Africa
;
; Or,      E-mail:  janman@ilink.nis.za / jannie.hanekom@chameleon.alt.za
;         Fidonet:  5:7102/129
;
; Any input on this program greatly appreciated.
;
;-----------------------------------------------------------------------------
; Copyright notice and disclaimer.
;-----------------------------------------------------------------------------
; Only one thing about this version of the program's copyright:  I am not
; claiming copyright for it.  This version (0.92) is in the public domain,
; and can thus be copied/modified in any way you see fit.
;
; All I ask is that you do not use it for malevolent purposes, and if you
; changed it, a message telling the user so when the program is run would
; be appreciated.
;
; Also:  I accept no responsibility for any damage the direct or indirect
;        use of this program can have on you or anything related to you.
;        This is not because I'm a bad programmer, but because I know some
;        people have a habit of changing programs to do bad things, and I
;        don't want to take responsibility for them.  If you do have a
;        problem with the way the program operates in its original form,
;        though, contact me and I'll see what I can do to improve it.
;-----------------------------------------------------------------------------
; Operation.
;-----------------------------------------------------------------------------
; The current version of the program is fairly straightforward.  If it
; detects a Sound Blaster Pro or higher when run, it installs itself and
; hooks the keyboard interrupt.  When it detects a Ctrl-Alt-U or Ctrl-Alt-D
; sequence, it does its stuff.
;
; Please see the documentation for further details on operation.
;-----------------------------------------------------------------------------
; The source.
;-----------------------------------------------------------------------------
; Here it is.  Have fun!
;*****************************************************************************
 Major_ver      equ     0              ; Major version number to report
 Minor_ver      equ     92             ; Minor version number to report
 Lastdate       equ     '95/10/23'     ; Last date modified
 NrTimesToScan  equ     100            ; Nr of times to check for SB presence
 StartScan      equ     210h           ; Starting address to scan for SB
 EndScan        equ     280h           ; Ending address of SB scan
 Bell           equ     7              ; ASCII value of Bell character
 UpKey          equ     16h            ; Keyboard scancode for 'U'
 DownKey        equ     20h            ; Keyboard scancode for 'D'
 LeftInc        equ     10h            ; Value to increase left channel with
 RightInc       equ     01h            ; Value with which to increase right ch.
 LMaxVol        equ     0F0h           ; Maximum volume of left channel
 LMinVol        equ     0Fh            ; Minimum volume of left channel
 RMaxVol        equ     0Fh            ; Maximum volume of right channel
 RMinVol        equ     00h            ; Minimum volume of right channel
 KeyBoardInt    equ     09h            ; Interrupt number of keyboard

 DSP_Reset_Port equ     06h            ; Port to write to when resetting DSP
 DSP_Read_Data  equ     0Ah            ; Port from which to read data
 DSP_Write_Data equ     0Ch            ; Write data to this place
 DSP_Data_Avail equ     0Eh            ; Is data avail?  Look here to see.
                        
 Mixer_Reg_Sel  equ     04h            ; Write to select a mixer register
 Mixer_Data     equ     05h            ; Write data to that selected register

 Master_Volume  equ     022h           ; Mix. reg. nr. of master volume
 DSP_Version    equ     0E1h           ; DSP command to get DSP version

 Delay  macro                          ; Macro to delay between bus i/o
        jmp  $+2
        jmp  $+2
        jmp  $+2
        jmp  $+2
        jmp  $+2
 endm

.model Tiny
.data
 ; Various messages used throughout the program.

 Welcome        db     'SBVol v',Major_Ver+'0','.', Minor_Ver/10+'0',Minor_Ver mod 10+'0',' - Public domain by Jannie Hanekom, ',LastDate,'$'
 ScanMsg        db     '  Scanning for Sound Blaster...$'
 NotFoundMsg    db     'not found.$'
 FoundMsg       db     'found at $'
 UsageMsg       db     '  Press Ctrl-Alt-U for volume increase and Ctrl-Alt-D for volume decrease.$'
 SBRequired     db     '  Error:',Bell,'  This program requires an SBPro or higher.$'
 LineFeed       db     10, 13, '$' ; Move cursor to following line

.code
 org    0h
 Dev_Start     label   word
        dd      -1              ; No further device drivers in this file
        dw      8000h           ; Character device
Strat:  dw      Strategy        ; Address of "strategy" routine
Entry:  dw      Dev_entry       ; Entry address of device driver
        db      '&SBVOL', Major_Ver+'0', Minor_Ver/10+'0'
                                ; Character device's name

 EXE_Start      label   word

;-----------------------------------------------------------------------------
; Procedure to check if hot-keys have been pressed.

 KeyHandler     proc    far
        push    ds              ; Save some important registers
        push    ax
        push    dx

        sub     ax, ax          ; Get to segment 0
        mov     ds, ax
        mov     al, byte ptr ds:[417h] ; Get keyboard shift status
        push    cs              ; Get our data segment into DS
        pop     ds
        and     al, 00001100b
        cmp     al, 00001100b   ; Is both <control> and <alt> pressed?
        jne     @Call_OldHandler
        in      al, 60h         ; If so, get the current key from the KB
        cmp     al, DownKey     ; Is it equal to the "Down" key?
        je      Dec_Volume      ; Go decrease the volume.

 @CheckUpKey:
        cmp     al, UpKey       ; Is it equal to the "Up" key?
        je      Inc_Volume      ; Go increase the volume.
 
 @Call_OldHandler:
        pushf
        call    Old_KeyHandler  ; Call the original handler
        jmp     short @Skip_IRQ_Ack ; We don't need to acknowledge the IRQ

 @Exit_KeyHandler:
        mov     al, 20h         ; Send Ack signal to IRQ controller
        out     20h, al
 @Skip_IRQ_Ack:
        pop     dx              ; Restore those registers.
        pop     ax
        pop     ds
        iret
 KeyHandler     endp


;-----------------------------------------------------------------------------
; "Increase volume" handler.  Uses a universal routine for SBPro+

 Inc_Volume     proc near
        mov     al, Master_Volume  ; Get current Master volume
        call    Read_Mixer
        cmp     al, LMaxVol        ; Are we at max output yet?
        jae     @CheckRightVol_Inc
        add     al, LeftInc        ; No, so go towards it!
 @CheckRightVol_Inc:
        mov     dl, al             ; Another working copy
        and     dl, 0fh            ; Get to right channel info
        cmp     dl, RMaxVol        ; Full Blast?
        je      @Inc_Volume_Exit
        add     al, RightInc       ; No, make it closer to it.

 @Inc_Volume_Exit:
        mov     ah, Master_Volume  ; Write value to Mixer register
        call    Write_Mixer
        jmp     short @Exit_KeyHandler
 Inc_Volume     endp


;-----------------------------------------------------------------------------
; "Decrease volume" handler.  Uses a universal routine for SBPro+

 Dec_Volume     proc near
        mov     al, Master_Volume  ; Get current Master volume
        call    Read_Mixer
        test    al, 0F0h
        jz      @CheckRightVol_Dec
        sub     al, LeftInc        ; No, make it go softer.
 @CheckRightVol_Dec:
        mov     dl, al             ; Another working copy
        and     dl, 0fh            ; Get to right channel info
        jz      @Dec_Volume_Exit
        sub     al, RightInc       ; No, so make it shut up more.

 @Dec_Volume_Exit:
        mov     ah, Master_Volume  ; Write value to Mixer register.
        call    Write_Mixer
        jmp     short @Exit_KeyHandler
 Dec_Volume     endp

;-----------------------------------------------------------------------------
; Writes the value in AL to the mixer register specified in AH.

 Write_Mixer    proc near
        mov     dx, [Base_Addr]    ; Point to SB's address
        add     dx, Mixer_Reg_Sel  ; Point to Mixer register selector
        xchg    al, ah             ; Get register number into AL
        out     dx, al             ; Select register
        mov     al, ah             ; Get value to output into AL
        inc     dx                 ; Point to register data port
        out     dx, al             ; Write value to register
        ret
 Write_Mixer    endp


;-----------------------------------------------------------------------------
; Reads a value from the mixer register specified in AL and places it in AL

 Read_Mixer     proc near
        mov     dx, [Base_Addr]    ; Point to SB's address
        add     dx, Mixer_Reg_Sel  ; Point to Mixer register selector
        out     dx, al             ; Select register specified in AL
        inc     dx                 ; Point to mixer register data port
        in      al, dx             ; Read the current value of the register
        ret
 Read_Mixer     endp


 Base_Addr      dw    ?         ; Base address of SB
 Old_KeyHandler label dword     ; Old Keyboard interrupt handler
 O_KeyHandler_o dw    ?         ;  - offset
 O_KeyHandler_s dw    ?         ;  - segment
 EXE_End        label word
 req_hdr        label dword     ; Memory location to save request header at
 req_hdr_o      dw    ?         ;  - offset
 req_hdr_s      dw    ?         ;  - segment

;-----------------------------------------------------------------------------
; All future references to device driver will return 'unknown command'.

 Error_Proc     proc    far
        mov     es:[bx+3], 8103h ; Set 'Unknown Command' error condition
        ret
 Error_Proc     endp

 Device_End     label word

;-----------------------------------------------------------------------------
; Strategy routine of device driver

 Strategy       proc  far
        mov     cs:[req_hdr_o], bx      ; Save address of request header
        mov     cs:[req_hdr_s], es
        ret
 Strategy       endp


;-----------------------------------------------------------------------------
; Entry point of device driver routines.

 Dev_entry      proc far
        push    ax                 ; Save crucial registers. (all of them)
        push    bx
        push    cx
        push    dx
        push    si
        push    di
        push    es
        push    ds
        pushf
        les     bx, cs:[req_hdr]   ; Get device req header location into ES:BX
        cmp     byte ptr es:[bx+2], 0 ; Is it an "init driver" command?
        jne     @Unknown_command   ; If not, we don't know nuthin'bout it.
        
        call    SBVol              ; Run main program.
        test    ax, ax             ; Did we get back an error condition?
        jnz     @Unknown_command
 
        mov     ah, 25h            ; Set vector to our keyboard handler
        mov     al, KeyBoardInt
        mov     dx, cs             ; Get the address of KB handler into DS:DX
        mov     ds, dx
        mov     dx, offset KeyHandler
        int     21h

        les     bx, cs:[req_hdr]   ; Get device req header location into ES:BX
        mov     es:[bx+3], 100h    ; Set "success" exit code
        mov     word ptr es:[bx+0Eh], offset Device_End ; Leave only the
        mov     word ptr es:[bx+10h], cs ; neccessary part in memory
        mov     word ptr cs:[Strat], offset Error_Proc
        mov     word ptr cs:[Entry], offset Error_Proc
                                   ; All future calls return 'unknown command'
        jmp     short @Dev_Exit

 @Unknown_command:
        mov     ax, 8103h          ; Error exit (unknown command)
        mov     es:[bx+3], ax      ; Set exit code
 @Dev_Exit:       
        popf                       ; Restore those registers.
        pop     ds
        pop     es
        pop     di
        pop     si
        pop     dx
        pop     cx
        pop     bx
        pop     ax
        ret
 Dev_entry      endp


;-----------------------------------------------------------------------------
; Writes the value in AL to the SB's DSP.

 Write_DSP      proc    near
        mov     bl, al             ; Make a working copy
        mov     dx, cs:[Base_Addr] ; Point to SB's address
        add     dx, DSP_Write_Data ; Point to DSP's output port
 @Buffer_Not_Clear:
        in      al, dx             ; Go until we get a clear to send signal
        test    al, 80h
        jnz     @Buffer_Not_Clear
        mov     al, bl             ; Get value to output into AL...
        out     dx, al             ; and send it.
        ret
 Write_DSP      endp


;-----------------------------------------------------------------------------
; Reads a value from the DSP and places it in AL

 Read_DSP      proc near
        mov    dx, cs:[Base_Addr]  ; Point to SB's address
        add    dx, DSP_Data_Avail  ; Get ready to check if any data available
 @No_Data_Avail:
        in     al, dx              ; Go until we get a data ready signal
        test   al, 80h
        jz     @No_Data_Avail
        mov    dx, cs:[Base_Addr]  ; Point to SB's address
        add    dx, DSP_Read_Data   ; Point to read data port
        in     al, dx              ; Read the data
        ret
 Read_DSP      endp


;-----------------------------------------------------------------------------
; Checks at the address in DX to see if a Sound Blaster is present.
; If Sound Blaster was found, returns the address in AX.  Else, -1 in AX.

 Detect_SB      proc    near
        mov     bx, dx             ; Working copy
        add     dx, DSP_Reset_Port ; Point to probable SB reset port
        mov     al, 1              ; Reset it (or so we hope)
        out     dx, al
        delay                      ; Wait a while... ... ... ... ...
        sub     al, al             ; Re-enable DSP
        out     dx, al
        mov     dx, bx             ; Point to base address again
        add     dx, DSP_Data_Avail ; Wait for Data available signal
        mov     cx, NrTimesToScan  ; Nr of times to check for signal
 @CheckForSB:
        in      al, dx
        dec     cx
        jz      @SBNotFound        ; Timeout?
        test    al, 80h
        jz      @CheckForSB        ; Data there yet?

        mov     dx, bx             ; Point to base address again
        add     dx, DSP_Read_Data  ; Get read-data address
        in      al, dx
        cmp     al, 0AAh           ; Do we get a valid response?
        jne     @SBNotFound
        mov     ax, bx             ; If so, return valid address
        ret

 @SBNotFound:
        mov     ax, -1             ; Return error code.
        ret
 Detect_SB      endp


;-----------------------------------------------------------------------------
; Scans for the SB at the start and end addresses specified in equ section.
; Returns the address it was found at in AX, or -1 if not found.

 Scan_SB        proc    near
        mov     dx, StartScan      ; Value to start checking for SB
 @Scan_Loop:
        push    dx                 ; Save a copy
        call    Detect_SB          ; Check for SB at this location
        pop     dx                 ; Get back our previous base address
        test    ax, ax             ; Did we get back a valid reply?
        jns     @Scan_SB_Exit
        add     dx, 10h            ; No, search at next address
        cmp     dx, EndScan        ; Have we reached maximum probable address?
        jbe     @Scan_Loop

        mov     ax, -1             ; Return error code

 @Scan_SB_Exit:
        ret
 Scan_SB        endp


;-----------------------------------------------------------------------------
; Displays the hexadecimal value passed in AX.  Leading zeroes not displayed.

 Display_Hex    proc    near
                mov     bx, ax     ; Working copy
                mov     cx, 16     ; Total number of bits to check
 @ScanZeroes:
                test    ah, 0f0h   ; Is the leftmost nibble zero?
                jnz     @Display_Chars ; Yes, so we've removed all zero chars
                add     ax, ax     ; Else remove the leftmost char
                add     ax, ax     ; (equivalent of shl ax, 4)
                add     ax, ax
                add     ax, ax
                sub     cx, 4      ; and update the number of bits to process
                jnz     @ScanZeroes; If CX != 0 we've not processed everything
 @Display_Chars:
                jcxz    @Display_Hex_Done; If CX=0 the entire thing was zero.
                sub     cx, 4      ; Decrease number of chars to process
                mov     dx, bx     ; Make a working copy
                shr     dx, cl     ; get the char we want to display into the
                                   ;  lower nibble of dx
                and     dl, 0fh    ; single out that number
                add     dl, '0'    ; make it into an ASCII code
                mov     ah, 02h    ; BIOS function to display char in DX
                int     21h
                jmp     short @Display_Chars
 @Display_Hex_Done:
                mov     dl, 'h'    ; Display the 'h' after the number
                mov     ah, 02h    ; BIOS function to display char in DX
                int     21h
                ret
 Display_Hex    endp


;-----------------------------------------------------------------------------
; Universal procedure to display a string to the screen.  Requires address to
;  string in ds:dx.  Set the carry flag to display a linefeed afterwards.

 Display_String   proc  near
        pushf                      ; Make certain we retain the CF status
        mov     ah, 09h            ; BIOS funtion to display a string
        int     21h                ;  Note: DX already contains offset
        popf                       ; Restore CF status
        jnc     @Display_String_Done ; If CF = 0 don't display line feed
        mov     dx, offset LineFeed ; Else do so
        mov     ah, 09h
        int     21h
 @Display_String_Done:
        ret
 Display_String   endp


;=============================================================================
; Main program.

 SBVol  proc    near
        mov     ax, seg @Data      ; Get our data segment into ds
        mov     ds, ax
        mov     dx, offset Welcome ; Display welcome message with linefeed
        stc
        call    Display_String
        mov     dx, offset ScanMsg ; Tell user we're scanning for SB (No LF)
        clc
        call    Display_String
        call    Scan_SB            ; Scan for the SB
        test    ax, ax             ; Check for error code return
        js      @SB_Error          ; If it was negative, leave with error
        mov     cs:[Base_Addr], ax ; Else, save the base address we got
        mov     dx, offset FoundMsg ; Display the fact that we found the SB
        clc
        call    Display_String
        mov     ax, cs:[Base_Addr] ; Next, display where we found it
        call    Display_Hex
        mov     dx, offset LineFeed ; and print a linefeed.
        clc
        call    Display_String

   ; Next, we have to check for a DSP revision of 3 or higher, since only
   ; Sound Blaster Pro's and later contained built-in mixer chips.  (The
   ; DSP version of the SBPro was 3).

        mov     al, DSP_Version     ; Request Version from DSP chip
        call    Write_DSP
        call    Read_DSP            ; Read major version number into AL
        mov     ah, al              ; Put it into AH
        call    Read_DSP            ; Read minor version number into AL
        cmp     ah, 3               ; Is the major version number above 3?
        jb      @MixerNotFound

        mov     dx, offset UsageMsg ; If mixer found, display usage of program
        stc                         ; including linefeed
        call    Display_String

        mov     ah, 35h             ; Get the current keyboard interrupt vect.
        mov     al, KeyBoardInt
        int     21h
        mov     O_KeyHandler_o, bx  ; Save it for when we call it later on
        mov     O_KeyHandler_s, es

        sub     ax, ax              ; Set "no-error" condition

        jmp     short @Exit

 @SB_Error:
        mov     dx, offset NotFoundMsg ; SB not found.  Display as such.
        stc                         
        call    Display_String      
 @MixerNotFound:
        mov     dx, offset SBRequired ; If we get to here, either the SB and/
        stc                           ; or the mixer was not found, so tell  
        call    Display_String        ; the user to get one with a mixer.
        mov     ax, -1                ; Set "error" condition.

 @Exit:
        ret
 SBVol  endp


 SBVolEXE:
        call    SBVol                 ; Check for SB and initialize vectors
        test    ax, ax                ; Error flag set?
        jnz     @ErrExit              

 ; Following corrects offsets for EXE use:  we're going to relocate the code
 ; to a memory area over the previous environment, so we have to modify the
 ; appropriate memory referencing addresses.  Not very good practice, but it
 ; does save quite a number of bytes.

        mov     word ptr cs:[@Call_OldHandler+3], offset Old_KeyHandler-offset EXE_Start
        mov     word ptr cs:[Read_Mixer+2], offset Base_Addr-offset EXE_Start
        mov     word ptr cs:[Write_Mixer+2], offset Base_Addr-offset EXE_Start
        mov     ah, 62h               ; Get program's PSP
        int     21h
        push    bx                    ; Put a copy on the stack,
        mov     es, bx                ;  and another in ES
        mov     es, es:[02Ch]         ; Get address of environment
        mov     ah, 049h              ; And free that memory block
        int     21h

        mov     si, offset EXE_Start  ; Address of where to move code from
        mov     di, 60h               ; Target address - over previous env.
        push    cs
        pop     ds                    ; Source segment = CS
        pop     es                    ; Set destination segment
        mov     cx, offset EXE_End - offset EXE_Start + 1 ; Nr bytes to move
        rep     movsb                 ; GO!
        
        mov     ah, 25h             ; Set vector to our keyboard handler
        mov     al, KeyBoardInt
        mov     dx, es              ; Get the address of KB handler into DS:DX
        add     dx, 6               ; We moved it 6*16 bytes above PSP start
        mov     ds, dx
        mov     dx, offset KeyHandler - offset EXE_Start
        int     21h

        mov     ax, 3100h             ; DOS function - go resident
                                      ; Calculate the resident size.
        mov     dx, (offset EXE_End - offset EXE_Start)/16+7
        int     21h                   ; End program.

 @ErrExit:
        mov     ax, 4CFFh             ; An error occured, so use normal DOS
        int     21h                   ; program termination.  Return an
                                      ; Errorlevel value of 255
ends
end SBVolEXE
