/*
 *  Routines to handle keyboard input for the spectrogram program
 *
 *  Copyright (C) 1995  Philip VanBaren
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <math.h>
#include <time.h>

#include "specgram.h"
#include "fft.h"
#include "display.h"
#include "gifsave.h"
#include "extern.h"

int help_mode=0;        /* Flag indicating help mode 0 (off), 1, or 2 */
int mic_level=0;        /* Storage for mic input mixer level (0-100) */
int ext_level=0;        /* Storage for ext input mixer level (0-100) */
int int_level=0;        /* Storage for int input mixer level (0-100) */


double starttime=0;     /* Clock tick when sampling began (for rate calc) */
long basetime=0;        /* Time value when the program began */
struct timeb tb;        /* Structure for grabbing the time */

void input_text(char *text,int maxlen,int x,int y)
{
   int c=0;
   int count=0;
   char ach[3];
   ach[1]='_';
   ach[2]=0;

   draw_text_left(x,y,text);
   count=strlen(text);
   x+=count*8;
   draw_text_left(x,y,"_");

   while((c!=0x0d) && (c!=0x0a))
   {
      c=draw_getch();
      if(((c==0x7f)||(c==0x08)) && count)
      {
	 count--;
	 x-=8;
	 draw_bar(x,y,x+2*_font_width-1,y+_font_height,0);
	 draw_text_left(x,y,"_");
      }
      else if(count<(maxlen-1))
      {
	 if( ((c>='0') && (c<='9')) || ((c>='A') && (c<='Z')) || ((c>='a') && (c<='z')) || (c=='.') || (c=='/') || (c=='\\') || (c==':') )
	 {
	    draw_bar(x,y,x+_font_width-1,y+_font_height,0);
	    ach[0]=c;
	    text[count]=c;
	    draw_text_left(x,y,ach);
	    count++;
	    x+=8;
	 }
      }
      if(c==0x1b)
      {
	 count=0;
	 break;
      }
   }
   text[count]=0;
}

void SaveImage(void)
{
   clock_t clk;

  #ifdef SRGP
   draw_text_left(GIX+24,GIY,"GIF file save not supported");
   clk=clock();
   while(!draw_getkey() && ((clock()-clk)<CLOCKS_PER_SEC*3));
   draw_bar(GIX,GIY,MAXX,GIY+_font_height,0);
  #else

   unsigned char p[256*3];
   char filename[20];
   char ach[50];
   int i;

   filename[0]=0;
   draw_text_left(GIX+24,GIY,"Filename? ");
   input_text(filename,sizeof(filename),GIX+104,GIY);

   draw_bar(GIX,GIY,MAXX,GIY+_font_height,0);

   if(filename[0]!=0)
   {
      if(GIF_Create(filename,MAXX+1,MAXY+1,256,8)!=GIF_OK)
	 draw_text_left(GIX+40,GIY,"Error creating file");
      else
      {
         int failure=0;
	 /* Set up the palette */
	 fillpalette(p,palette);
	 for(i=0;i<_draw_colors;i++)
	    GIF_SetColor(i,p[i*3],p[i*3+1],p[i*3+2]);

	 i=GIF_CompressImage(0,0,-1,-1,draw_getpixel);
         failure+=i;
         if(i==GIF_ERRWRITE)
            draw_text_left(GIX+40,GIY,"Error writing file");
         if(i==GIF_OUTMEM)
            draw_text_left(GIX,GIY,"GIF memory allocation failure");
         #ifdef DEBUG_OUTPUT
            printf("GIF_CompressImage returned %d\n",i);
         #endif

	 i=GIF_Close();
         failure+=i;
         if(i==GIF_ERRWRITE)
            draw_text_left(GIX+40,GIY,"Error writing file");
         #ifdef DEBUG_OUTPUT
            printf("GIF_Close returned %d\n",i);
         #endif

         if(!failure)
         {
	    sprintf(ach,"GIF image saved in %s",filename);
	    draw_text_left(GIX,GIY,ach);
         }
      }
      clk=clock();
      while(!draw_getkey() && ((clock()-clk)<CLOCKS_PER_SEC*3));
      draw_bar(GIX,GIY,MAXX,GIY+_font_height,0);
   }
  #endif /* SRGP */
   if(!help_mode)
      draw_bar(AFX+80,AFY,AFX+135,AFY+_font_height,0);
   displayed=0;
   ftime(&tb);
   starttime=(double)(tb.time-basetime)+(double)tb.millitm/1000.0;
}

int process_input(int c)
{
   char ach[40];

   if(c==' ')
   {
      freeze=!freeze;
      if(!freeze)
      {
	 if(!help_mode)
	    draw_bar(AFX+80,AFY,AFX+135,AFY+_font_height,0);
	 displayed=0;
	 ftime(&tb);
         starttime=(double)(tb.time-basetime)+(double)tb.millitm/1000.0;
      }
   }
   else if(c=='O' || c=='o')
   {
      if(help_mode)
      {
	 help_mode=0;
	 update_header();
      }
      SaveImage();
   }
   else if(c=='A' || c=='a')
   {
      if(c=='a')
      {
	 accel_factor++;
	 if(displayed>50)
	 {
	    /* Wrap around if we exceed the maximum rate */
	    double rate;
  	    ftime(&tb);
            rate=(double)(tb.time-basetime)+(double)tb.millitm/1000.0-starttime;
            if(rate>0)
               rate=(double)displayed*fftlen/(rate*SampleRate);
            else
               rate=0;
	    if((rate+1.5)<accel_factor)
	       accel_factor=1;
	 }
      }
      else
      {
	 accel_factor--;
	 if(accel_factor==0) accel_factor=8;
      }
      buffer_step=(int)ceil((double)fftlen/accel_factor);
      displayed=0;
      ftime(&tb);
      starttime=(double)(tb.time-basetime)+(double)tb.millitm/1000.0;
      if(!help_mode)
      {
	 sprintf(ach,"Accel:%2d",accel_factor);
         draw_bar(AFX,AFY,AFX+127,AFY+_font_height,0);
	 draw_text_left(AFX,AFY,ach);
      }
   }
   else if(c=='F' || c=='f')
   {
      EndFFT();
      if(c=='f')
      {
	 fftlen=fftlen*2;
	 if(fftlen>(1024/(stereo+1)))
	 {
	    fftlen=64;
	    repcount=16/(stereo+1);
	 }
      }
      else
      {
	 fftlen=fftlen/2;
	 repcount*=2;
	 if(fftlen<64)
	 {
	    fftlen=1024/(stereo+1);
	    repcount=1;
	 }
      }
      InitializeFFT(fftlen);
      compute_window_function();
      draw_axis();

      buffer_step=(int)ceil((double)fftlen/accel_factor);
      displayed=0;
      ftime(&tb);
      starttime=(double)(tb.time-basetime)+(double)tb.millitm/1000.0;

      /* Re-initialize the display */
      if(!help_mode)
         update_header();
   }
   else if(c=='R' || c=='r')
   {
      char in[6];
      if(help_mode)
      {
         help_mode=0;
         update_header();
      }
      halt_soundcard();
      draw_text_left(GIX+20,GIY,"Sampling rate: ");
      in[0]=0;
      input_text(in,sizeof(in),GIX+140,GIY);
      if(in[0]!=0)
      {
         SampleRate=atol(in);
         if(SampleRate<5000L) SampleRate=5000L;
         if(SampleRate>(88200L/(stereo+1))) SampleRate=88200L/(stereo+1);
            reset_soundcard();

         /* Re-initialize the display */
         draw_axis();
      }
      displayed=0;
      ftime(&tb);
      starttime=(double)(tb.time-basetime)+(double)tb.millitm/1000.0;

      update_header();
   }
   else if(c=='S' || c=='s')
   {
      if(help_mode)
      {
         help_mode=0;
         update_header();
      }
      draw_text_left(GIX,GIY,"Save the current state to: ");
      strncpy(ach,ini_file,20);
      input_text(ach,20,GIX+216,GIY);
      if(ach[0]!=0)
      {
         FILE *fp;
         strcpy(ini_file,ach);
         fp=fopen(ini_file,"w");
         if(fp)
         {
            fprintf(fp,"Soundcard: %d\n",Soundcard);
            fprintf(fp,"Sample rate: %ld\n",SampleRate);
            fprintf(fp,"FFT length: %d\n",fftlen);
            fprintf(fp,"Accel factor: %d\n",accel_factor);
            fprintf(fp,"Pixel height: %d\n",repcount);
            fprintf(fp,"Window function: %d\n",windfunc);
            fprintf(fp,"Log mode: %d\n",logamp);
            fprintf(fp,"Log scale base: %d\n",log_base);
            fprintf(fp,"Log scale factor: %g\n",log_scale);
            fprintf(fp,"Linear scale factor: %g\n",amp_scale);
            fprintf(fp,"DB/Octave gain: %d\n",gain);
            fprintf(fp,"Embossed mode: %d\n",diff_mode);
            fprintf(fp,"Palette: %d\n",palette);
            fprintf(fp,"Stereo: %d\n",stereo);
            fprintf(fp,"Stereo L-R: %d\n",stereo_diff);

            fclose(fp);
         }
         else
         {
            clock_t clk=clock();
            sprintf(ach,"Unable to open %s",ini_file);
            draw_bar(GIX,GIY,MAXX,GIY+_font_height,0);
            draw_text_left(GIX,GIY,ach);
            while(!draw_getkey() && ((clock()-clk)<3*CLOCKS_PER_SEC));
         }
      }
      displayed=0;
      ftime(&tb);
      starttime=(double)(tb.time-basetime)+(double)tb.millitm/1000.0;
      if(!help_mode)
         draw_bar(AFX+80,AFY,AFX+135,AFY+_font_height,0);
      draw_bar(GIX,GIY,MAXX,GIY+_font_height,0);
   }
   else if((c==LEFT_ARROW) || (c=='4'))
   {
      if(logamp)
         log_base--;
   }
   else if((c==RIGHT_ARROW) || (c=='6'))
   {
      if(logamp)
         log_base++;
   }
   else if((c==UP_ARROW) || (c=='8'))
   {
      if(logamp)
      {
         if(log_scale<1000)
            log_scale*=1.25;
      }
      else
      {
         if(amp_scale<1000)
            amp_scale*=1.25;
      }
   }
   else if((c==DOWN_ARROW) || (c=='2'))
   {
      if(logamp)
      {
         if(log_scale>0.000001)
         log_scale*=0.80;
      }
      else
      {
	 if(amp_scale>0.000001)
	    amp_scale*=0.80;
      }
   }
   else if(c=='L' || c=='l')
   {
      logamp=!logamp;
      if(!help_mode)
      {
	 draw_bar(AMX,AMY,AMX+199,AMY+_font_height,0);
	 if(logamp)
	    draw_text_left(AMX,AMY,"Log amplitude scale");
	 else
	    draw_text_left(AMX,AMY,"Linear amplitude scale");
      }
   }
   else if(c=='E' || c=='e' || c=='M' || c=='m')
   {
      diff_mode=!diff_mode;
      if(!help_mode)
      {
	 draw_bar(MDX,MDY,MDX+119,MDY+_font_height,0);
	 if(diff_mode)
	    draw_text_left(MDX,MDY,"Embossed mode");
	 else
	    draw_text_left(MDX,MDY,"Amplitude mode");
      }
   }
   else if(c=='P' || c=='p')
   {
      if(c=='p')
      {
	 palette++;
	 if(palette>=PALETTES) palette=0;
      }
      else
      {
	 palette--;
	 if(palette<0) palette=PALETTES-1;
      }
      set_palette(palette);

      if(!help_mode)
      {
	 sprintf(ach,"%s palette",palette_name[palette]);
	 draw_bar(PLX,PLY,PLX+144,PLY+_font_height,0);
	 draw_text_left(PLX,PLY,ach);
      }
   }
   else if(c=='W' || c=='w')
   {
      /*
       *  Change the windowing function
       */
      if(c=='w')
      {
	 windfunc++;
	 if(windfunc>6) windfunc=0;
      }
      else
      {
	 windfunc--;
	 if(windfunc<0) windfunc=6;
      }
      compute_window_function();

      if(!help_mode)
      {
	 sprintf(ach,"%s window function",window_name[windfunc]);
	 draw_bar(WFX,WFY,WFX+215,WFY+_font_height,0);
	 draw_text_left(WFX,WFY,ach);
      }
   }
   else if((c=='G') || (c=='g'))
   {
      /*  Change the gain level */
      gain+=6;
      if(gain>12) gain=0;
      deriv=gain/6;

      if(!help_mode)
      {
	 char ach[30];
	 sprintf(ach,"Gain of %d dB per octave",gain);
	 draw_bar(DGX,DGY,DGX+191,DGY+_font_height,0);
	 draw_text_left(DGX,DGY,ach);
      }
   }
   else if(c=='H' || c=='h' || c=='?' || c=='/')
   {
      help_mode=!help_mode;
      if(help_mode)
	 show_help();
      else
	 update_header();
   }

   if(stereo && (c=='D' || c=='d'))
   {
      stereo_diff=!stereo_diff;
      draw_bar(0,MAXY-_font_height,23,MAXY,0);
      draw_bar(0,MAXY-5-_font_height-repcount*fftlen/2,23,MAXY-repcount*fftlen/2-5,0);
      if(stereo_diff)
      {
	 draw_text_left(0,MAXY-_font_height,"L+R");
	 draw_text_left(0,MAXY-5-_font_height-repcount*fftlen/2,"L-R");
      }
      else
      {
	 draw_text_left(0,MAXY-_font_height,"L");
	 draw_text_left(0,MAXY-5-_font_height-repcount*fftlen/2,"R");
      }
   }

   if(mixers)
   {
      if(c=='[') /* External jack down */
      {
         ext_level-=2;
         if(ext_level<0) ext_level=0;
         if(!help_mode)
         {
      	    sprintf(ach,"%3d",ext_level);
	    draw_bar(LVX+104,LVY,LVX+127,LVY+_font_height,0);
	    draw_text_left(LVX+104,LVY,ach);
         }
         set_mixer(MIXER_EXT,ext_level);
      }
      else if(c==']') /* External jack up */
      {
         ext_level+=2;
         if(ext_level>100) ext_level=100;
         if(!help_mode)
         {
            sprintf(ach,"%3d",ext_level);
            draw_bar(LVX+104,LVY,LVX+127,LVY+_font_height,0);
            draw_text_left(LVX+104,LVY,ach);
         }
         set_mixer(MIXER_EXT,ext_level);
      }
      else if(c=='{') /* CD input down */
      {
         int_level-=2;
         if(int_level<0) int_level=0;
         if(!help_mode)
         {
            sprintf(ach,"%3d",int_level);
	    draw_bar(LVX+168,LVY,LVX+191,LVY+_font_height,0);
            draw_text_left(LVX+168,LVY,ach);
         }
         set_mixer(MIXER_INT,int_level);
      }
      else if(c=='}') /* CD input up */
      {
         int_level+=2;
         if(int_level>100) int_level=100;
         if(!help_mode)
         {
            sprintf(ach,"%3d",int_level);
	    draw_bar(LVX+168,LVY,LVX+191,LVY+_font_height,0);
            draw_text_left(LVX+168,LVY,ach);
         }
         set_mixer(MIXER_INT,int_level);
      }
      else if(c==',' || c=='<' || c=='(') /* Mic input down */
      {
         mic_level-=2;
         if(mic_level<0) mic_level=0;
         if(!help_mode)
         {
            sprintf(ach,"%3d",mic_level);
	    draw_bar(LVX+32,LVY,LVX+55,LVY+_font_height,0);
	    draw_text_left(LVX+32,LVY,ach);
         }
         set_mixer(MIXER_MIC,mic_level);
      }
      else if(c=='.' || c=='>' || c==')') /* Mic input up */
      {
         mic_level+=2;
         if(mic_level>100) mic_level=100;
         if(!help_mode)
         {
            sprintf(ach,"%3d",mic_level);
	    draw_bar(LVX+32,LVY,LVX+55,LVY+_font_height,0);
            draw_text_left(LVX+32,LVY,ach);
         }
         set_mixer(MIXER_MIC,mic_level);
      }
   }

   return(c=='Q' || c=='q' || c=='X' || c=='x' || c==0x03);
}
