program Setup; {$V-}

(*
 *                      SETUP
 *
 *  An Interactive IBM PC/AT CMOS RAM Setup Program
 *
 *          by Tyler Ivanco and Bob Kamins
 *
 *)

{ Version 2.1 by Kenneth Herron.  This version is placed in the
  Public Domain.  Based on code which was originally placed in
  the Public Domain by Tyler Ivanco and Bob Kamins
}

{ Changes for version 2.1:
     1) Extended drive types are handled *properly* <ahem>.
        (code courtesy Keith Ericson of Tektronix)
     2) Bugs in screen-mode save are fixed.
     3) Added warning screen for status byte D flags.
     4) If you $DEFINE NOWRITES then no writes to CMOS ram will be
        made.  Instead, a file called CMOS.OUT will be written, showing
        what addresses and values would have been written (all in Hex).
        Useful for debugging.
}

uses crt, dos, cmos;

type
     Boxtype    = (PCAT, Date, Time, DiskA, DiskB, 
                   DiskC, DiskD, Disp, CoPr, BMem, EMem);
     CornerType = (Left, Top, Right, Bottom);
     Fieldtype  = (DW, S100,              { Stuff you can't set }
                  Mon, Day, Yr,           { Date }
                  Hr, Min, Sec,           { Time }
                  DA, DB, DC, DD,         { Disks }
                  Di, CP, BK, EK);        { Equipment/Memory }
     ParamList = array[FieldType] of word;

     FieldEnt = record
          PosX,
          PosY: byte;         { cursor x/y position }
          LowEnd,
          HiEnd: word;        { min, max values }
          Adjust: byte        { How much to adjust at a time }
     end;
     FieldEntList = array[FieldType] of FieldEnt;

     HDDefine = record             { Format of a BIOS HD type block }
          cylinders: word;
          Heads:     byte;
          G1:        word;         { Gx denotes unused parts of the block }
          PreComp:   word;
          G2:        byte;
          Control:   byte;
          G3:        byte;
          G4:        word;
          LandZn:    word;
          Sectors:   byte;
          G5:        byte
     end;

const
  BaseSize  = $12;       { INT to get base memory size }
  ExtdSize  = $15;       { INT to get expanded memory size }

  ESC       = #27;
  UP        = #72;
  DN        = #80;
  LT        = #75;
  RT        = #77;

     FieldRange: FieldEntList = 
    ((PosX: 9; PosY: 8; LowEnd: 0; HiEnd: 6; Adjust: 0),    { day of week }
     (PosX: 38; PosY: 8; lowend: 0; hiend: 99; adjust: 0),  { sec/100 }
     (PosX: 12; PosY: 8; LowEnd: 1; HiEnd: 12; Adjust: 1),  { month }
     (PosX: 15; PosY: 8; LowEnd: 1; HiEnd: 31; Adjust: 1),  { day }
     (PosX: 18; PosY: 8; lowend: 0; Hiend: 99; adjust: 1),  { year }
     (PosX: 29; PosY: 8; lowend: 0; Hiend: 23; adjust: 1),  { hour }
     (PosX: 32; PosY: 8; lowend: 0; hiend: 59; adjust: 1),  { minute }
     (PosX: 35; PosY: 8; lowend: 0; hiend: 99; adjust: 1),  { second }
     (PosX: 15; PosY: 11; lowend: 0; Hiend: 15; adjust: 1), { floppy 1 }
     (PosX: 33; PosY: 11; lowend: 0; hiend: 15; adjust: 1), { floppy 2 }
     (posX: 7; PosY: 14; lowend: 0; hiend: 255; adjust: 1), { hard 1 }
     (PosX: 7; PosY: 17; lowend: 0; hiend: 255; adjust: 1), { hard 2 }
     (PosX: 12; PosY: 20; lowend: 0; hiend: 3; adjust: 1),  { display }
     (PosX: 36; PosY: 20; lowend: 0; hiend: 1; adjust: 1),  { coprocessor }
     (PosX: 15; PosY: 23; lowend: 0; hiend: 640; adjust: 128), { main memory }
     (PosX: 32; PosY: 23; lowend: 0; hiend: 15360; adjust: 128));{ ext. memory }

     Box:array[boxtype, cornertype] of byte
       = ((16, 1,  25, 3),       {PCAT}
          (1,  7,  20, 9),       {Date}
          (21, 7,  40, 9),       {Time}
          (3,  10, 20, 12),      {Floppy A}
          (21, 10, 38, 12),      {Floppy B}
          (1,  13, 40, 15),      {Hard Disk C}
          (1,  16, 40, 18),      {Hard Disk D}
          (1,  19, 20, 21),      {Display}
          (21, 19, 40, 21),      {CoProcessor}
          (3,  22, 19, 24),      {Base Memory}
          (20, 22, 38, 24));     {Extended Memory}

     DayNames: array[0..6, 1..2] of char = 
     ('SU', 'MO', 'TU', 'WE', 'TH', 'FR', 'SA');

     FloppyNames: array[0..15, 1..5] of char = 
     ('none ','360K ','1.2M ','720K ','1.44M','# 5  ','# 6  ','# 7  ',
      '# 8  ','# 9  ','# 10 ','# 11 ','# 12 ','# 13 ','# 14 ','# 15 ');

     DispNames: array[0..3, 1..8] of char = 
     ('EGA/VGA ','Color 40','Color 80','Mono    ');

     CopNames: array[0..1, 1..3] of char =
     ('no ', 'yes');

     MonthLen: array[1..12] of byte = 
     (31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31);

     DefParams: ParamList =
     (5,        {Day of week} { 1/1/88 was a friday }
      0,        {Sec/100}
      1,        {Mon}
      1,        {Day}
     88,        {Yr}
      0,        {Hr}
      0,        {Min}
      0,        {Sec}
      2,        {DA}
      0,        {DB}
      0,        {DC}
      0,        {DD}
      2,        {Di}
      0,        {CP}
    256,        {BK}
      0);       {EK}

     PDNotice1:string[28] = 'SETUP 2.1 by Kenneth Herron.';
     PDNotice2:string[28] = 'Placed in the public domain.';
var
     SysClock: longint absolute $0:$46C;
     HDData: array[1..255] of HDDefine absolute $F000:$E401;
     Params: ParamList;
     Reg: registers;
     OldAttr: byte;
     OldTextMode: word;
{$ifdef NoWrites}
     CmosFile: text;
{$endif}

procedure SaveVideo(var A: byte; var M: word);
begin
     A := TextAttr;
     M := LastMode;
end; {SaveVideo}

procedure RestoreVideo(A: byte);
begin
     TextAttr := A;
     writeln;
     ClrScr
end; {RestoreVideo}


{$ifdef NoWrites}
{ This replaces the WRITECMOS in the CMOS unit }
procedure WriteCmos(Address, data: byte);

const HexDigit:array[0..15] of char = '0123456789ABCDEF';

begin
     writeln(CmosFile, 'Addr: ', HexDigit[Address div 16], 
             HexDigit[Address mod 16], ' Data: ',
             HexDigit[Data div 16], HexDigit[Data mod 16]);
end;
{$endif}

function Bcd(I: byte):byte;
{ Binary to BCD }
  begin
    Bcd :=(I div 10) * 16 + (I mod 10)
  end; {Bcd}

function Bin(I: byte): byte;
{ BCD to binary }
  begin
    Bin := (I div 16) * 10 + (I mod 16)
  end; {Bin}

procedure At(Line, Col: byte; Msg: String);
begin
     GotoXY(Col, Line);
     Write(Msg)
end; {At}

procedure AtField(F: FieldType; Msg: String);
begin
     with FieldRange[F] do
          Gotoxy(PosX, PosY);
     write(Msg)
end;

function SaveChanges: boolean;
var
     Ch: char;

begin
     clrscr;
     at(12,8, 'Save new configuration?');
     repeat
          Ch := upcase(Readkey)
     until (ch = 'Y') or (ch = 'N');
     SaveChanges := (Ch = 'Y')
end;

function Str2(I: integer): String;
var
     S: string[3];
begin
     Str(I + 100: 3, S);
     Str2 := S[2] + S[3];
end; {Str2}

function Str3(I: integer): String;
var
     S: string[3];
begin
     Str(I: 3, S);
     Str3 := S
end; {Str3}

function Str5(I: integer): String;
var
     S: string[5];
begin
     Str(I: 5, S);
     Str5 := S
end; {Str5}

procedure CheckTime;

{ Updates time and date as the program runs }

var
     Hour,     Minute,
     Second,   Sec100,
     Year,     Month,
     DayVal,   DOW: word;

  procedure SetParam(P: Fieldtype; V: byte);
    var
      OldX, OldY: byte;
    begin
      if (Params[P] <> V) then
        begin
          OldX := WhereX;
          OldY := WhereY;
          Params[P] := V;
          if P = DW then
               Atfield(P, DayNames[V])
          else
               Atfield(P, Str2(V));
          GotoXY(OldX, OldY)
        end  {if}
    end; {SetParam}

begin {CheckTime}
     GetTime(Hour, Minute, Second, Sec100);
     SetParam(S100, Sec100);
     if Second <> Params[Sec] then
     begin
          SetParam(Sec, Second);
          SetParam(Min, Minute);
          SetParam(Hr, Hour);
     end;
     GetDate(Year, Month, DayVal, DOW);
     if DOW <> Params[DW] then
     begin
          SetParam(DW, DOW);
          SetParam(Day, DayVal);
          SetParam(Mon, Month);
          SetParam(Yr, Year - 1900)
     end;
end; {CheckTime}

function DriveDescrip(DN: byte): string;
{ Given a drive number, returns a string describing the drive }

var
     Tstr: string[33];
     Tmp: string[5];

begin
     str(DN:3, Tstr);
     if DN = 0 then
          Tstr := Tstr + ' no hard drive                   '
     else
     with HDData[DN] do
     begin
          str(cylinders:4, Tmp);
          Tstr := Tstr + ' cyl:' + tmp;
          str(Heads:2, tmp);
          Tstr := Tstr + ' hd:' + tmp;
          if precomp = $FFFF then
               Tmp := 'no '
          else
          if precomp = 0 then
               Tmp := 'all'
          else
               str(Precomp:3, tmp);
          Tstr := Tstr + ' pc:' + tmp;
          str(LandZn:4, Tmp);
          Tstr := Tstr + ' lz:' + tmp;
     end;
     DriveDescrip := Tstr
end;

procedure ReportDiagStatus;

var
     Dstat: byte;
     Pstat: byte;

begin
     Dstat := ReadCmos(DiagStat) and $FC; { last two bits are reserved }
     Pstat := ReadCmos(StatRegD) and $80; { High bit is only valid one }
     if (Dstat <> 0) or (Pstat = 0) then
     begin
          clrscr;
          at(9, 4, 'Warning: CMOS RAM Status shows:');
          if (Dstat and $80) <> 0 then
               at(succ(WhereY), 5, 'CMOS has lost power');
          if (Dstat and $40) <> 0 then
               at(succ(WhereY), 5, 'CMOS Checksum is incorrect');
          if (Dstat and $20) <> 0 then
               at(succ(WhereY), 5, 'Configuration is incorrect');
          if (Dstat and $10) <> 0 then
               at(succ(WhereY), 5, 'Stored memory size is incorrect');
          if (Dstat and $8) <> 0 then
               at(Succ(WhereY), 5, 'A fixed disk failed to initialize');
          if (Dstat and $4) <> 0 then
               at(succ(WhereY), 5, 'Time of Day is incorrect');
          if Pstat = 0 then
               at(succ(WhereY), 5, 'Your battery is dead');
          at(25, 5, 'Press SPACE to continue');
          repeat
               Dstat := ord(ReadKey);
          until Dstat = ord(' ');
     end
end;

procedure Initialize;

  function RamValid: boolean;
     { Note:  We only test here for power failures.  Any other problems
       will generate a warning, but won't cause a CMOS reset }
    var
      I: integer;
      Key: char;
    begin
     RamValid := ((ReadCmos(StatRegD) and $80) <> 0) and { battery good }
                 ((ReadCmos(DiagStat) and $80) = 0); { hasn't lost power }
    end; {RamValid}

procedure GetParams(var P: ParamList);
var
     Key: char;
begin {assumes BCD and 24-hour formats...}
     GetDate(P[Yr], P[Mon], P[Day], P[DW]);
     P[Yr] := P[Yr] mod 100;
     GetTime(P[Hr], P[Min], P[Sec], P[S100]);
     P[DA] := (ReadCmos(Diskettes) and $F0) shr 4;
     P[DB] := ReadCmos(Diskettes) and $0F;
     P[DC] := (ReadCmos(HardDisk) and $F0) shr 4;
     if P[DC] = $F then 
          P[DC] := ReadCmos(HdExt1);
     P[DD] := ReadCmos(HardDisk) and $0F;
     if P[DD] = $F then
          P[DD] := ReadCmos(HdExt2);
     P[Di] := (ReadCmos(Equipment) and $30) shr 4;
     P[CP] := (ReadCmos(Equipment) and $02) shr 1;
     P[BK] := ReadCmos(BaseLo) + (256 * ReadCmos(BaseHi));
     P[EK] := ReadCmos(ExpdLo) + (256 * ReadCmos(ExpdHi))
end; {GetParams}

procedure SetDefaults(var P: ParamList);
begin
     P := DefParams;
     Intr(BaseSize, Reg);
     P[BK] := Reg.AX;
     Reg.AH := $88;
     Intr(ExtdSize, Reg);
     P[EK] := Reg.AX
end; {SetDefaults}

  procedure MakeBox(B: Boxtype; Msg: String);

     procedure HLine(X1, X2, Y: integer);
     var
          I: byte;
     begin
          for I := X1 to X2 do 
          begin
               gotoxy(I, Y);
               write(#205)    { Horizontal double line }
          end;
     end; {HLine}

     procedure VLine(X, Y1, Y2: integer);
     var
          I: byte;
     begin
          Gotoxy(X, Y1);
          for I := Y1 to Y2 do write(#179)  { vertical single line }
     end; {VLine}

    begin {MakeBox}
      TextColor(LightGray);
      At(Box[B, Top], Box[B, Left], #213); {upper left corner}
      At(Box[B, Top], Box[B, Right], #184); {upper right corner}
      At(Box[B, Bottom], Box[B, Left], #212); {lower left corner}
      At(Box[B, Bottom], Box[B, Right], #190); {lower right corner}
      HLine(Box[B, Left] + 1, Box[B, Right] - 1, Box[B, Top]);
      VLine(Box[B, Left], Box[B, Top] + 1, Box[B, Top] + 1);
      HLine(Box[B, Left] + 1, Box[B, Right] - 1, Box[B, Bottom]);
      VLine(Box[B, Right], Box[B, Top] + 1, Box[B, Top] + 1);
      TextColor(White);
      At(Box[B, Top] + 1, Box[B, Left] + 1, Msg)
    end; {MakeBox}

begin {Initialize}
    if RamValid then 
         GetParams(Params)
    else 
         SetDefaults(Params);
    TextMode(BW40);
    ClrScr;
    MakeBox(PCAT, 'AT SETUP');
    TextColor(LightGray);
    At(4, 2, 'Cursor keys choose field-ESC to end');
    At(5, 8, '"+" and "-" change value');
    At(6, 11, '0 syncs the clock');
    TextColor(White);
    MakeBox(Date, 'DATE =      /  /');
    Atfield(DW, Daynames[Params[dw]]);
    AtField(Mon, Str2(Params[Mon]));
    AtField(Day, Str2(Params[Day]));
    Atfield(Yr, Str2(Params[Yr]));
    MakeBox(Time, 'TIME =   :  :  :');
    Atfield(Hr, Str2(Params[Hr]));
    Atfield(Min, Str2(Params[Min]));
    Atfield(Sec, Str2(Params[Sec]));
    MakeBox(DiskA, 'FLOPPY A =');
    Atfield(DA, FloppyNames[Params[DA]]);
    MakeBox(DiskB, 'FLOPPY B =');
    Atfield(DB, FloppyNames[Params[DB]]);
    MakeBox(DiskC, 'HDC =');
    Atfield(DC, DriveDescrip(Params[DC]));
    MakeBox(DiskD, 'HDD =');
    Atfield(DD, DriveDescrip(Params[DD]));
    MakeBox(Disp, 'DISPLAY =');
    Atfield(DI, DispNames[Params[Di]]);
    MakeBox(CoPr, 'COPROCESSOR =');
    AtField(CP, CopNames[Params[Cp]]);
    MakeBox(BMem, 'BASE MEM =    k');
    AtField(BK, Str3(Params[BK]));
    MakeBox(EMem, 'EXTD MEM =      k');
    AtField(EK, Str5(Params[EK]));
    TextColor(LightGray);
    { On this last line, watch the length--if you print to col. 40
      you'll make the screen scroll on CGA/EGA?/VGA? }
    At(25, 2, 'Version 2.1 by Kenneth Herron 3-Jun-88');
    TextColor(White)
end; {Initialize}

procedure Process;
var
    Key:      char;
    FieldPtr: Fieldtype;

procedure SyncClock;
{ rounds the clock down to the last whole second }
var
     Tclock: longint;
begin
     Tclock := (Sysclock * 4) div 73; { 73/4 = 18.25 ~= 18.2065 }
     SysClock := (Tclock * 73) div 4;
end;

procedure GetData(var Key: char);

     procedure ReadKeyBoard(var Key: char);
     begin
          repeat
               Checktime
          until KeyPressed;
          Key := ReadKey;
          if Key = #0 then Key := ReadKey;
     end; {ReadKeyBoard}

     function MaxDay: byte;
     begin
          if (Params[Mon] = 2) and ((Params[Yr] mod 4) = 0) then
               MaxDay := 29
          else
               MaxDay := MonthLen[Params[Mon]]
     end;

     procedure ChangeParam(F: Fieldtype; Dir: shortInt);

     begin
          FieldRange[Day].HiEnd := MaxDay;
          with FieldRange[F] do
               if Dir > 0 then
                    if Params[F] >= HiEnd then
                         Params[F] := LowEnd
                    else
                         Params[F] := Params[F] + Adjust
               else
                    if Params[F] <= LowEnd then
                         Params[F] := HiEnd
                    else
                         Params[F] := Params[F] - Adjust;
          { $0F is an illegal HD value--here's the el kludge-O fix }
          if ((F = DD) or (F = DC)) and (Params[F] = $F) then
               if Dir > 0 then
                    inc(Params[F])
               else
                    dec(Params[F]);
          case F of
          S100,
          DW: {do nothing };
          Hr,
          Min,
          Sec: begin
                    SetTime(Params[Hr],Params[Min],Params[Sec],Params[S100]);
                    AtField(F, str2(Params[F]))
               end;
          Mon,
          Day,
           Yr: begin
                    if Params[Day] > MaxDay then 
                    begin
                         Params[Day] := MaxDay;
                         Atfield(Day, Str2(Params[Day]));
                    end;
                    SetDate(Params[Yr] + 1900, Params[Mon], Params[Day]);
                    AtField(F, Str2(params[F]));
               end;
           BK: AtField(F, Str3(Params[F]));
           EK: AtField(F, Str5(Params[F]));
           DC,
           DD: atfield(F, DriveDescrip(Params[F]));
           DI: atfield(F, DispNames[Params[F]]);
           DA,
           DB: atfield(F, FloppyNames[Params[F]]);
           CP: atfield(F, CopNames[Params[F]]);
          else;
        end  {case}
      end;

     procedure LastField(var FL: FieldType);
     begin
          if (FL = Mon) then
               FL := EK
          else
               dec(FL)
     end; {LastField}

     procedure NextField(var FL: FieldType);
     begin
          if (FL = EK) then
               FL := Mon
          else
               inc(FL)
     end; {NextField}

     procedure Beep;
     begin
          Sound(440);
          Delay(100);
          NoSound
     end; {Beep}

begin {GetData}
     with FieldRange[FieldPtr] do Gotoxy(PosX, PosY);
     ReadKeyBoard(Key);
     case Key of
        '0': SyncClock;
        '-',
        '_': ChangeParam(FieldPtr, -1);      { down }
        '+',
        '=': ChangeParam(FieldPtr, 1);       { up }
         UP,
         LT: LastField(FieldPtr);            { backward }
         DN,
         RT: NextField(FieldPtr);            { foreward }
        ESC:
         else Beep
     end  {case}
end; {GetData}

begin {Process}
     FieldPtr := Mon;
     repeat GetData(Key) until Key = ESC
end; {Process}

procedure Terminate;

  procedure SetDate;

  var
     Hour,     Minute,
     Second,   Sec100,
     OldSecond: word;

    begin
      WriteCmos(StatRegA, $26);    { Standard value to set clock speed }
      WriteCmos(StatRegB, $02);    { Set defaults }
      WriteCmos(StatRegC, $00);    { Most of this reg. is read-only }
      WriteCmos(StatRegD, $80);    { High bit is power-good, others = 0 }
      { set date in real-time clock }
      { Some manuals don't mention--this int requires its args in BCD }
      Reg.AH := 5;
      Reg.DL := bcd(Params[Day]);
      Reg.DH := bcd(Params[Mon]);
      Reg.CL := bcd(Params[Yr]);
      Reg.CH := bcd(19);      { have to fix this for > yr 2000 }
      Intr($1A, Reg);         { Set the date }
      { Set time in real-time clock }
      { Some manuals don't mention--this int takes args in BCD }
      { some neat stuff is done here to keep the clock as accurate
        as possible, considering the RT clock doesn't accept secs/100.}
      clrscr;
      at(12, 8, 'Stand by--setting clock');
      Reg.AH := 3;
      Reg.DL := 0; { Daylight savings flag }
      GetTime(Hour, Minute, OldSecond, Sec100);
      repeat
          GetTime(Hour, Minute, Second, Sec100)
      until Second <> OldSecond;
      Reg.CH := bcd(Hour);
      Reg.CL := bcd(Minute);
      Reg.DH := bcd(Second);
      Intr($1A, Reg);
      WriteCmos(DiagStat, $00);
      WriteCmos(ShutDown, $00)
    end; {SetDate}

procedure SetFloppies;
begin
     WriteCmos(Diskettes, Params[DA] shl 4 + Params[DB]);
end; {SetFloppies}

procedure SetHardDisks;
var
     HDval,
     HD1Extval,
     HD2Extval: byte;

begin
     if Params[DC] >= $F then
     begin
          HdVal := $F0;
          Hd1ExtVal := Params[DC]
     end
     else
     begin
          HdVal := Params[DC] shl 4;
          Hd1ExtVal := 0
     end;
     if Params[DD] >= $F then
     begin
          HdVal := HdVal + $F;
          Hd2ExtVal := Params[DD];
     end
     else
     begin
          HdVal := HdVal + Params[DD];
          Hd2ExtVal := 0;
     end;
     WriteCmos(HardDisk, HdVal);
     WriteCmos(HdExt1, Hd1ExtVal);
     WriteCmos(HdExt2, Hd2ExtVal);
end;

  procedure SetEquipment;
    var
      NumDisks, DiskPresent: byte;
    begin
      if ((Params[DA] > 0) or (Params[DB] > 0)) then 
          DiskPresent := 1
      else
          DiskPresent := 0;
      if (Params[DA] > 0) and (Params[DB] > 0) then     { two disks }
          NumDisks := 1
      else
          NumDisks := 0;
      WriteCmos(Equipment, (NumDisks shl 6) + ((Params[Di]) shl 4)
                 + 12 + (Params[Cp] shl 1) + DiskPresent)
    end; {SetEquipment}

  procedure SetMemory;
    begin
      WriteCmos(BaseLo, Lo(Params[BK]));
      WriteCmos(BaseHi, Hi(Params[BK]));
      WriteCmos(ExpdLo, Lo(Params[EK]));
      WriteCmos(ExpdHi, Hi(Params[EK]));
    end; {SetMemory}

  begin {Terminate}
    SetDate;
    SetFloppies;
    SetHardDisks;
    SetEquipment;
    SetMemory;
    SetCMOSCheckSum;
    case Params[Di] of
      0: TextMode(CO80); { Use CO80+Font8x8 for 43/50 line mode on EGA/VGA }
      1: TextMode(CO40);
      2: TextMode(CO80);
      3: TextMode(mono);
    end; {case}
  end; {Terminate}

begin {Main}
{$ifdef NoWrites}
     assign(CmosFile, 'CMOS.OUT');
     rewrite(CmosFile);
{$endif}
     SaveVideo(OldAttr, OldTextMode);
     ReportDiagStatus;
     Initialize;
     Process;
     if SaveChanges then
          Terminate
     else
          TextMode(OldTextMode);
     RestoreVideo(OldAttr);
{$ifdef NoWrites}
     Close(CmosFile);
{$endif}
end. {Main}

