unit Bitmap;

{ define bmp_usegraph if you don't have VgaMem
}

{ Unit Bitmap, Version 1.30, Copr. 1994,96 Matthias Kppe

  Translate a device-independent bitmap, loaded by LoadBitmap or
  LoadBitmapFile, into an image, which can be drawn by PutImage.

  Supported video modes		Supported bitmaps
  * 16-color modes		* monochrome bitmaps
  * 256-color modes		* 16-color bitmaps
				* 256-color bitmaps

  Call BitmapToImage, and a nearest-color mapping table will be used.
  This is created by analyzing the bitmap's and the device's palette.

  Call BitmapToImageWithUserPalette if you want to use a user-defined
  mapping table. Set UserPalette nil for using an original map with
  color bitmaps, or the Color/BkColor variables with monochrome bitmaps.

  Instead of Graph's PutImage procedure, we recommend using the
  one implemented in our VgaMem unit: It performs correct clipping.
}

interface

{$ifdef Windows}
uses WinTypes;
{$else}
uses WinRes;
{$endif}

var
  Color, BkColor: Word;

const
  ColorWeights: TRGBTriple = (rgbtBlue: 1; rgbtGreen: 1; rgbtRed: 1);

{ Bitmap functions
}

function BitmapToImage(ABitmap: PBitmap): pointer;
function BitmapToImageWithUserPalette(ABitmap: PBitmap;
  UserPalette: pointer): pointer;
procedure AddDevicePalette(ABitmap: PBitmap);

{ Palette functions
}

type
  TColorRef = LongInt;

function GetNearestPaletteIndex(Color: TColorRef): Word;
function RGB(Red, Green, Blue: Byte): TColorRef;

{$ifndef Windows}
var
  DevicePalette: array[0..255] of TRGBTriple;
  sDevicePalette: Word;

{
}

procedure SetTable(Table: Word);
procedure CalcDevicePalette;

{$ifdef bmp_usegraph}
procedure FreeImage(Image: pointer);
{$endif}
{$endif}

implementation {  }

{$ifdef Windows}
{$i bmpw.pas}
{$else}

{$ifdef bmp_usegraph}
uses Graph, Objects, Memory;	{ for ImageSize/Stream }

procedure FreeImage(Image: pointer);
begin
  with PImage(Image)^ do
    FreeMem(Image, ImageSize(0, 0, imSizeXm1, imSizeYm1))
end;

{$else}
uses Gr, VgaMem, Objects, Memory;	{ for Notification/ImageSize/Stream }

var
  NextNotify: TNotifyProc;

function NotifyBitmap(Notice: Word; Info: LongInt): LongInt; far;
Begin
  case Notice of
    gnpInitGraphics:
      begin
	SetTable(grFlags);
	CalcDevicePalette
      end;
    gnpCloseGraphics:
      SetTable(0);
    gnpPalette:
      CalcDevicePalette;
  end;
  NotifyBitmap := DefaultNotify(Notice, Info, NotifyBitmap, NextNotify, 20);
End;

{$endif}

{ Low-level bitmap functions 
}
var
  Bmp2ImgRowProc: Word;

procedure Bmp2ImgRow_16; near; external;
procedure Bmp2ImgRow_256; near; external;

procedure DoBitmapToImage(ABitmap: PBitmap; Image: pointer;
  Palette: pointer); near; external;

{$L bmp.obj (bmp.asm) }

{ Low-level palette functions 
}
var
  GetDevPalProc: Word;

procedure GetDevPal_16; near; external;
procedure GetDevPal_256; near; external;

function GetDevicePalette(var Palette): Integer; near; external;
procedure DoCalcPalette(var BmpPal; Count, EntrySize: Word;
  var PalBuf); near; external;

{$L palette.obj (palette.asm) }

procedure TripleToQuad(var Triple, Quad; Count: Word); near; external;

{$L wrespal.obj (wrespal.asm) }

{ 
}

procedure CalcDevicePalette;
begin
  sDevicePalette := GetDevicePalette(DevicePalette)
end;

function PrepareImage(SrcBmp: PBitmap): pointer;
var
  Image: PImage;
  size: LongInt;
begin
  Image := nil;
  with SrcBmp^ do
  begin
    size := ImageSize(0, 0, bmWidth-1, abs(bmHeight)-1);
    If size <> 0
    then begin		{ small image }
      Image := MemAlloc(size);
      If Image <> nil then
      with Image^ do
      begin
	imSizeXm1 := bmWidth - 1;
	imSizeYm1 := abs(bmHeight) - 1;
      end
    end
  end;
  PrepareImage := Image
end;

function BitmapToImageWithUserPalette(ABitmap: PBitmap;
  UserPalette: pointer): pointer;
var
  Image: pointer;
  BitmapChunk: pointer;
  i: Integer;
Begin
  If ABitmap = nil
  then begin
    BitmapToImageWithUserPalette := nil;
    Exit
  end;
  with ABitmap^ do
  begin
    If bmChunks = 0
    then begin
      Image := PrepareImage(ABitmap);
      If Image <> nil then
	DoBitmapToImage(ABitmap, Image, UserPalette)
    end
    else begin
      GetMem(Image, SizeOf(TImage) + (bmChunks - 1) * SizeOf(pointer));
      with PImage(Image)^ do
      begin
	imSizeXm1 := -1;
	imSizeYm1 := -1;
	imCount := bmChunks;
	BitmapChunk := bmBits;
	For i := bmChunks - 1 downto 0 do
	begin
	  imPtrs[i] :=
	    BitmapToImageWithUserPalette(PBitmap(BitmapChunk^), UserPalette);
	  Inc(PtrRec(BitmapChunk).Ofs, SizeOf(pointer))
	end
      end;
    end
  End;
  BitmapToImageWithUserPalette := Image
End;

function BitmapToImage(ABitmap: PBitmap): pointer;
var
  PalBuf: array[0..255] of Byte;
  Pal: pointer;
Begin
  If ABitmap = nil
  then begin
    BitmapToImage := nil;
    Exit
  end;
  Pal := nil;
  with ABitmap^ do
    If bmPalette <> nil then Begin
      DoCalcPalette(bmPalette^, 1 shl (bmBitsPixel * bmPlanes),
	SizeOf(TRGBQuad), PalBuf);
      Pal := @PalBuf
    End;
  BitmapToImage := BitmapToImageWithUserPalette(ABitmap, Pal)
End;

procedure AddDevicePalette(ABitmap: PBitmap);
var
  count: Word;
  p: pointer;
Begin
  If ABitmap = nil then Exit;
  with ABitmap^ do
    If bmPalette = nil then Begin
      count := 1 shl (bmBitsPixel * bmPlanes);
      GetMem(bmPalette, SizeOf(TRGBQuad) * count);
      If Count = 2 then Begin
	FillChar(bmPalette^, SizeOf(TRGBQuad) * 2, 0);
	Move(DevicePalette[BkColor], bmPalette^, SizeOf(TRGBTriple));
	p := bmPalette;
	Inc(PtrRec(p).Ofs, SizeOf(TRGBQuad));
	Move(DevicePalette[Color], p^,
	  SizeOf(TRGBTriple));
      End
      else TripleToQuad(DevicePalette, bmPalette^, Count)
    End
End;

function GetNearestPaletteIndex(Color: TColorRef): Word;
var
  Res: TColorRef;
Begin
  DoCalcPalette(Color, 1, 4, Res);
  GetNearestPaletteIndex := Res
End;

function RGB(Red, Green, Blue: Byte): TColorRef; assembler;
asm
	mov	al, Red
	mov	ah, Green
	mov	dl, Blue
	xor	dh, dh
end;

procedure SetTable(Table: Word);
Begin
  if Table and 1 = 0
  then Begin
    Bmp2ImgRowProc := Ofs(Bmp2ImgRow_16);
    GetDevPalProc := Ofs(GetDevPal_16);
    VirtualBitmapChunkSize := 32000;
  End
  else Begin
    Bmp2ImgRowProc := Ofs(Bmp2ImgRow_256);
    GetDevPalProc := Ofs(GetDevPal_256);
    VirtualBitmapChunkSize := 64000;
  End
End;

var
  oldBitmapLoadProc: TBitmapLoadProc;

function BmpBitmapLoad(var BitmapInfoHeader: TBitmapInfoHeader;
  var S: TStream; Size: LongInt; Palette: pointer;
  CreateImage: Boolean): pointer; far;
var
  Bitmap: PBitmap;
  Image: pointer;
Begin
  Bitmap := oldBitmapLoadProc(BitmapInfoHeader, S, Size, Palette, false);
  If (Bitmap <> nil) and CreateImage then Begin
    Image := BitmapToImage(Bitmap);
    DeleteBitmap(Bitmap);
    BmpBitmapLoad := Image
  End
  else
    BmpBitmapLoad := Bitmap
End;

begin
  { Install WinRes' bitmap load procs
  }
  oldBitmapLoadProc := BitmapLoadProc;
  BitmapLoadProc := BmpBitmapLoad;
  { Install Gr's notification proc
  }
{$ifndef bmp_usegraph}
  InstallNotifyProc(GrNotifyProc, NotifyBitmap);
{$endif}
  { init
  }
  SetTable(0)
{$endif Windows}
end.
