{ RTSGEN.PAS : Pascal Run-Time Statistics trace generator

  title    : RTSGEN
  version  : 1.1
  date     : jan 09,1994
  author   : J R Ferguson
  language : Turbo Pascal v4.0/5.0/5.5/7.0
  usage    : refer procedure Help
}

{$V-}
{$R+}

{$UNDEF OUTBUFHEAP}   { UNDEF to work around a BP 7.0 bug resulting in
                        erroneous file output }

program RTSGEN;
uses DefLib, ArgLib, CvtLib, StpLib, StfLib, ChrLib, RtsLib;


const
  DFLINPT   = '.PAS';          { input program source }
  DFLOUTT   = '.RTS';          { generated program source }
  DFLLSTT   = '.RTI';          { index of generated trace numbers }
  RTSOUT    = 'RTSSTATS.OUT';  { trace report of generated program }
(*
  { Option defaults: }
*)

  MAXFNAME  = 79; { max filename length (including drive and path) }

  INPBUFSIZ = 4096;
  OUTBUFSIZ = 4096;

  { Error codes and messages: }
  ERROK     = 0;
  ERRARG    = 1;
  ERRINP    = 2;
  ERROUT    = 3;
  ERRLST    = 4;
  ERRSYN    = 5;
  ERREOF    = 6;
  ERROVF    = 7;

  ERRMSG    : array[ERRARG..ERROVF] of StpTyp =
 ('',
  'File not found : ',
  'Can''t open output : ',
  'Can''t open listfile : ',
  'Syntax error : ',
  'Unexpected end of file : ',
  'Too many trace points'
 );

type
  InpBufTyp = array[1..INPBUFSIZ] of char; InpBufPtr = ^InpBufTyp;
  OutBufTyp = array[1..OUTBUFSIZ] of char; OutBufPtr = ^OutBufTyp;

var
  InpFname,
  InpFnameShort,
  LstFname,
  OutFname  : StpTyp;
  InpFvar,
  OutFvar,
  LstFvar   : Text;
  InpBuf    : InpBufPtr;
{$IFDEF OUTBUFHEAP}
  OutBuf    : OutBufPtr;
{$ELSE}
  OutBuf    : OutBufTyp;
{$ENDIF}
  InpOpen,
  OutOpen,
  LstOpen   : boolean;
  ErrCod    : integer;
  NestLevel : integer;    { procedure/function declaration nesting level }
  TraceCnt  : RtsIndTyp;  { current statistics trace point number }
  TraceCStp : StpTyp;     { string version of TranceCnt }
  CurLine   : StpTyp;     { current input line }
  CurNum    : integer;    { current input line number }
  CurChr    : char;       { current input character }
  CurIdn    : StpTyp;     { current identifier or keyword }
  CmpIdn    : StpTyp;     { ' ' + ToUpper(CurIdn) + ' ' }
  SavChr    : boolean;    { CurChr still to be processed }
  SavIdn    : boolean;    { CurIdn still to be processed }
  EofInp    : boolean;


{
--- General routines ---
}


procedure Help;
begin
  WriteLn('Reads Pascal program sourcefile. Produces a new Pascal source');
  WriteLn('with trace calls in all global procedures and functions, plus');
  WriteLn('a list of all trace call numbers with the corresponding names.');
  WriteLn('');
  WriteLn('usage   : RTSGEN inpfile [outfile] [lstfile]'
(*
	+ '[/option[...] [...]]'
*)
	 );
  WriteLn('defaults: inpfile type = ',DFLINPT);
  WriteLn('          outfile name = inpfile name');
  WriteLn('                  type = ',DFLOUTT);
  WriteLn('          lstfile name = inpfile name');
  WriteLn('                  type = ',DFLLSTT);
(*
  WriteLn('options :');
*)
  WriteLn('');
  WriteLn('remarks : - Generated program produces trace output on ',RTSOUT);
  WriteLn('          - Include commands {$I filename} are not recognized');
end;


function AllOk: boolean;
begin AllOk:= (ErrCod = ERROK) end;

{
--- Command Line parsing routines ---
}


procedure ReadOpt(arg: StpTyp);
begin
  StpDel(arg,1,1); if StpEmpty(arg) then ErrCod:= ERRARG;
  while AllOk and not StpEmpty(arg) do
(*
  case StpcGet(arg) of

    else
*)
    ErrCod:= ERRARG;
(*
  end;
*)
end;


procedure ReadArgs;
var i   : ArgInd;
    arg : StpTyp;
    p   : StpInd;
begin
  StpCreate(InpFname); StpCreate(OutFname); StpCreate(LstFname);
  GetArgs;
  i:= 0;
  while (i < ArgC) and AllOk do begin
    Inc(i); StpCpy(arg,ArgV[i]); StpUpp(arg);
    if      StpcRet(arg,1) = '/' then ReadOpt(arg)
    else if StpEmpty(InpFname)   then StpNCpy(InpFname,arg,MAXFNAME)
    else if StpEmpty(OutFname)   then StpNCpy(OutFname,arg,MAXFNAME)
    else if StpEmpty(LstFname)   then StpNCpy(LstFname,arg,MAXFNAME)
    else ErrCod:= ERRARG;
  end;
  if StpEmpty(InpFname) then ErrCod:= ERRARG
  else begin
    if StpcPos(InpFname,'.')=0 then StpCat(InpFname,DFLINPT);
    StpCpy(InpFnameShort,InpFname);
    StpDel(InpFnameShort,1,StpcRPos(InpFnameShort,'\'));

    if StpEmpty(OutFname) then StpBefore(OutFname,InpFnameShort,'.');
    if StpcPos(OutFname,'.')=0 then StpCat(OutFname,DFLOUTT);

    if StpEmpty(LstFname) then StpBefore(LstFname,InpFnameShort,'.');
    if StpcPos(LstFname,'.')=0 then StpCat(LstFname,DFLLSTT);
  end;
end;


{
--- I/O routines ---
}


procedure OpenInp;
begin
  Assign(InpFvar,InpFname); new(InpBuf); SetTextBuf(InpFvar,InpBuf^);
  {$I-} reset(InpFvar) {$I+};
  if IOresult <> 0 then ErrCod:= ERRINP else InpOpen:= true;
end;


procedure CloseInp;
begin if InpOpen then begin
  Close(InpFvar); dispose(InpBuf); InpOpen:= false;
end end;


procedure OpenOut;
begin
  Assign(OutFvar,OutFname);
{$IFDEF OUTBUFHEAP}
  new(OutBuf); SetTextBuf(OutFvar,OutBuf^);
{$ELSE}
  SetTextBuf(OutFvar,OutBuf);
{$ENDIF}
  {$I-} rewrite(OutFvar) {$I+};
  if IOresult <> 0 then ErrCod:= ERROUT else OutOpen:= true;
end;


procedure CloseOut;
begin if OutOpen then begin
  Close(OutFvar);
{$IFDEF OUTBUFHEAP}
  dispose(OutBuf);
{$ENDIF}
  OutOpen:= false;
end end;


procedure OpenLst;
begin
  Assign(LstFvar,LstFname);
  {$I-} rewrite(LstFvar) {$I+};
  if IOresult <> 0 then ErrCod:= ERRLST else LstOpen:= true;
end;


procedure CloseLst;
begin if LstOpen then begin
  Close(LstFvar); LstOpen:= false;
end end;


procedure PutChr(c: char);
begin case ord(c) of
  AsciiNUL : { no outpout } ;
  AsciiCR  : WriteLn(OutFvar);
  else       Write(OutFvar,c);
end end;


procedure PutLn;
begin PutChr(chr(AsciiCR)) end;


procedure PutStp(s: StpTyp);
var i: StpInd;
begin for i:= 1 to StpLen(s) do PutChr(s[i]) end;


procedure PutMsg(msg: StpTyp);
begin
  PutLn; PutStp('*** ERROR *** ' + msg); PutLn;
end;


procedure GetChr; { into global variable CurChr }
begin
  if SavChr then SavChr:= false
  else begin
    CurChr:= StpcGet(CurLine);
    if CurChr = chr(AsciiNUL) then begin
      if eof(InpFvar) then EofInp:= true
      else begin
        CurChr:= chr(AsciiCR);
        ReadLn(InpFvar,CurLine); Inc(CurNum);
      end;
    end;
  end;
end;


procedure GenNewTrace(ident: StpTyp);
begin if NestLevel <= 1 then begin
  if TraceCnt < RtsMaxInd then begin
    Inc(TraceCnt); ItoA(TraceCnt,TraceCStp);
    WriteLn(LstFvar,TraceCnt:5,' ',ident);
  end
  else begin PutMsg('Max trace count exceeded'); ErrCod:=ERROVF end;
end end;


procedure GenRtsStart;
begin if NestLevel <= 1 then begin
  PutLn; PutStp('{RTS*} ;RtsStart(' + TraceCStp + '); {*RTS}'); PutLn;
end end;


procedure GenRtsStop;
begin if NestLevel <= 1 then begin
  PutLn; PutStp('{RTS*} ;RtsStop(' + TraceCStp + '); {*RTS}'); PutLn;
end end;


procedure GenRtsReport;
begin
  PutLn; PutStp('{RTS*} ;RtsReport(''RTSSTATS.OUT''); {*RTS}'); PutLn;
end;


procedure GenRtsUse;
begin
  PutLn; PutStp('{RTS*} ,RtsLib {*RTS}'); PutLn;
end;


procedure GenRtsFullUse;
begin
  begin PutLn; PutStp('{RTS*} Uses RtsLib; {*RTS}'); PutLn; end;
end;


{
--- Low level syntax parsing routines ---
}


procedure SkipQuotedString;
var Skipping: boolean;
begin { CurChr = '''' }
  Skipping:= true;
  while Skipping and not EofInp do begin
    PutChr(CurChr); GetChr;
    if CurChr = '''' then begin
      PutChr(''''); GetChr;
      if CurChr <> '''' then begin
        Skipping:= false;
        SavChr:= not EofInp;
      end;
    end
  end;
end;


procedure SkipBracedComment;
begin { CurChr = '{' }
  repeat PutChr(CurChr); GetChr; until EofInp or (CurChr = '}');
  if not EofInp then PutChr('}');
end;


procedure SkipAsterixComment;
var Skipping: boolean;
begin { CurChr = '*', previous char was '(' }
  Skipping:= true;
  while Skipping and not EofInp do begin
    PutChr(CurChr); GetChr;
    if CurChr = '*' then begin
      PutChr('*'); GetChr;
      if CurChr = ')' then begin
        PutChr(')'); Skipping:= false;
      end
      else SavChr:= not EofInp;
    end;
  end;
end;


procedure GetToken;
var Skipping: boolean;
begin
  GetChr;
  case CurChr of
    '''': SkipQuotedString;
    '{' : SkipBracedComment;
    '(' : begin
            PutChr('('); GetChr;
            if CurChr='*' then SkipAsterixComment
            else SavChr:= not EofInp;
          end;
    '.' : CurIdn:= '.';
    ';' : CurIdn:= ';';
    else  begin
            if IsAlpha(CurChr) or (CurChr = '_' ) then begin { ident }
              repeat
                StpcCat(CurIdn,CurChr);
                GetChr;
              until EofInp or not (IsAlnum(CurChr) or (CurChr = '_'));
              SavChr:= not EofInp;
            end
            else PutChr(CurChr);
          end;
  end;
end;


procedure GetIdent;
begin
  if SavIdn then SavIdn:= false
  else begin
    CurIdn:= '';
    repeat GetToken
    until EofInp or (CurIdn <> '');
    CmpIdn:= ' ' + StfUpp(CurIdn) + ' ';
    if EofInp then begin
      PutMsg('Unexpected end of file'); ErrCod:= ERREOF;
    end;
  end;
end;


{
--- High level syntax parsing routines ---
}


procedure SkipToIdn(IdnList: StpTyp);
begin
  GetIdent;
  while AllOk and (StpPos(IdnList,CmpIdn) = 0) do
  begin
    PutStp(CurIdn);
    GetIdent;
  end;
end;


procedure UsesClause;
begin
  SkipToIdn(' UNIT USES CONST TYPE VAR LABEL PROCEDURE FUNCTION BEGIN ');
  if AllOk then begin
    if      CmpIdn = ' UNIT ' then begin
      PutStp(CurIdn); PutLn; PutMsg('Unit not supported'); PutLn;
      ErrCod:= ERRSYN;
    end
    else if CmpIdn = ' USES ' then begin
      PutStp(CurIdn); SkipToIdn(' RTSLIB ; ');
      if AllOk then begin
        if CurIdn = ';' then GenRtsUse;
        PutStp(CurIdn);
      end;
    end
    else begin
      GenRtsFullUse;
      SavIdn:= true;
    end;
  end;
end;


procedure FindMatchingEnd;
var level: integer;
begin
  level:= 1;
  while (level > 0) and AllOk do begin
    SkipToIdn(' BEGIN CASE END ');
    if CmpIdn = ' END ' then Dec(level) else Inc(level);
    if level > 0 then PutStp(CurIdn);
  end;
end;


procedure ProcOrFunc; forward;


procedure ProcOrFuncBody;
begin { UppIdn = ' BEGIN ' }
  PutStp(CurIdn); GenRtsStart;
  if AllOk then begin
    FindMatchingEnd;
    if AllOk then GenRtsStop;
  end;
end;


procedure Declarations;
begin
  SkipToIdn(' BEGIN FUNCTION PROCEDURE ');
  while AllOk and (CmpIdn <> ' BEGIN ') do begin
    ProcOrFunc;
    if AllOk then begin
      PutStp(CurIdn);
      SkipToIdn(' BEGIN FUNCTION PROCEDURE ');
    end;
  end;
end; { UppIdn = ' BEGIN ' }


procedure ProcOrFunc;
var ident: StpTyp;
begin
  Inc(NestLevel);
  PutStp(CurIdn); { 'procedure' or 'function' }
  GetIdent;
  if AllOk then begin
    PutStp(CurIdn); ident:= CurIdn;
    SkipToIdn(' BEGIN FUNCTION PROCEDURE FORWARD EXTERNAL INLINE ');
    if AllOk and (StpPos(' BEGIN FUNCTION PROCEDURE ',CmpIdn) <> 0)
    then begin
      GenNewTrace(ident); SavIdn:= true;
      if AllOk then Declarations;
      if AllOk then ProcOrFuncBody;
    end;
  end;
  Dec(NestLevel);
end;


procedure ProgramBody;
begin { UppIdn = ' BEGIN ' }
  PutStp(CurIdn);
  GenNewTrace('program');
  if AllOk then begin
    GenRtsStart;
    FindMatchingEnd;
    if AllOk then begin
      GenRtsStop; GenRtsReport;
      PutStp(CurIdn); { 'end' }
      SkipToIdn(' . ');
      if AllOk then begin
        while not EofInp do begin PutChr(CurChr); GetChr; end;
      end;
    end;
  end;
end;


{
--- Main Line ---
}


procedure MainInit;
begin
  ErrCod := ERROK;
  InpOpen:= false; OutOpen:= false; LstOpen:= false;
  EofInp:= false; SavChr:= false; SavIdn:= false; StpCreate(CurLine);
  TraceCnt:= 0; NestLevel:=0; CurNum:= 0;
  ReadArgs;
  if AllOk then OpenInp;
  if AllOk then OpenOut;
  if AllOk then OpenLst;
end;


procedure MainExit;
begin
  if ErrCod <> ERROK then begin
    Write(ERRMSG[ErrCod]);
    case ErrCod of
      ERRARG : begin WriteLn; Help; end;
      ERRINP : WriteLn(InpFname);
      ERROUT : WriteLn(OutFname);
      ERRLST : WriteLn(LstFname);
      ERRSYN : WriteLn('Line ',CurNum);
      ERREOF : WriteLn('Line ',CurNum);
      else     WriteLn;
    end;
  end;
  CloseInp; CloseOut; CloseLst;
end;


begin { Main program }
  WriteLn('RTSGEN v1.1 : Pascal Run-Time Statistics trace generator');
  MainInit;
  if AllOk then begin
    WriteLn(InpFname,' ==> ',OutFname,', trace list in ',LstFname);
    PutStp('{ '+OutFname+', created by RTSGEN from '+InpFname+' }'); PutLn;
    UsesClause;
    Declarations;
    ProgramBody;
  end;
  MainExit;
end.
