{ RTSLIB.PAS : Run Time Statistics routine library

  Title   : RTSLIB
  Version : 4.0
  Date    : Nov 11,1996
  Author  : J R Ferguson
  Language: Borland Turbo Pascal 4.0 through 7.0 (all targets)
  Usage   : Unit
}

UNIT RtsLib;

INTERFACE
uses
{$IFDEF WINDOWS}
  WinDos,
{$ELSE}
  Dos,
{$ENDIF}
DefLib;

const
  RtsMaxInd = 100;

type
  RtsIndTyp = 0..RtsMaxInd;
  RtsCntTyp = longint;
  RtsCpuTyp = real;

procedure RtsStart    (i: RtsIndTyp);
{ Execute this procedure at the starting point of the code block you want
  run-time statistics about. Choose a unique measure point index i.
  Put <RtsStop> with the same index value at the end of the code block.
  Use <RtsReport> to produce a statistical report.
}

procedure RtsStop     (i: RtsIndTyp);
{ Execute this procedure at the end of the code block you want run-time
  statistics about. A matching <RtsStart> with the same index i should
  be at the start of the code block.
  Use <RtsReport> to produce a statistical report.
}

procedure RtsInit     (i: RtsIndTyp);
{ This procedure sets the CPU-time counters for measure point i to zero.
  Since all counters are initialized automatically at the start of the
  program, you need to execute this routine explicitly only if you want
  to re-initialize your counters for some reason.

  See also: <RtsInitAll>
}

procedure RtsInitAll;
{ This procedure sets all CPU-time counters to zero. Since it is
  executed automatically at the start of the program, you need to
  execute this routine explicitly only if you want to re-initialize
  your counters for some reason.

  See also: <RtsInit>
}

procedure RtsReport   (fnm: StpTyp);
{ Execute this procedure after all CPU-times have been counted,
  preferably just before the propgram ends. It will create a report
  on CPU-time statistics for all measured code blocks (those that had
  a surrounding <RtsStart> and <RtsStop> call) an write it to the file
  named in parameter fnm.
  The report will include the number of times each measured code block
  was executed, together with the total, minimum, maximum and average
  CPU-time it took to execute and the standard deviation of the CPU-times
  measured. Keep in mind that the execution of the RtsStart and RtsStop
  procedures add some CPU-time themselves, so the measurement of small
  code-blocks may be statisically biased.
}


IMPLEMENTATION

const
  SecPerDag = 86400.0;
  MaxCpu    = 1.0e37;

type
  RtsRecTyp = record cnt: RtsCntTyp; beg,min,max,sum,ssq: RtsCpuTyp end;
  RtsMemTyp = array[1..RtsMaxInd] of RtsRecTyp;

var
  RtsMem    : RtsMemTyp;

function CpuSec: RtsCpuTyp;
var h,m,s,c: word;
begin
  GetTime(h,m,s,c); CpuSec:= ((h*60.0 + m) * 60.0 + s) + c/100.0
end;

procedure RtsInit(i: RtsIndTyp);
begin with RtsMem[i] do begin
  cnt:= 0; sum:= 0.0; ssq:= 0.0; min:= MaxCpu; max:=0.0;
end end;

procedure RtsInitAll;
var i: RtsIndTyp;
begin for i:= 1 to RtsMaxInd do RtsInit(i) end;

procedure RtsStart(i: RtsIndTyp);
begin RtsMem[i].beg:= CpuSec end;

procedure RtsStop(i: RtsIndTyp);
var t: RtsCpuTyp;
begin with RtsMem[i] do begin
  t:= CpuSec - beg; if t < 0 then t:= t + SecPerDag;
  sum:= sum + t;
  ssq:= ssq + t*t;
  if t < min then min:= t;
  if t > max then max:= t;
  inc(cnt);
end end;

procedure RtsReport(fnm: StpTyp);
var f  : text;
    i  : RtsIndTyp;
    avg: RtsCpuTyp;
begin
  Assign(f,fnm); Rewrite(f);
  writeln(f,
  'NR   COUNT    CPU TOTAL   MINIMUM    MAXIMUM    AVERAGE    STD.DEV.');
  writeln(f,
  '-- ---------- ---------- ---------- ---------- ---------- ----------');
  for i:= 1 to RtsMaxInd do with RtsMem[i] do if cnt>0 then begin
    avg:= sum/cnt;
    writeln(f,i:2,' ',cnt:10,' ',sum:10,' ',min:10,' ',max:10,
              ' ',avg:10,' ',sqrt(ssq/cnt - sqr(avg)):10);
  end;
  Close(f);
end;

BEGIN
  RtsInitAll;
END.
