(*****************************************************************************

  Frames
    Version 1.1

  This unit contains a variety of simple frames drawn on the screen.

  Purpose:
    This unit uses the features of the Draw unit to create frames on the
      screen.  It is simply a easy to use frame drawing unit of no
      significant purpose.

  How it works:
    Each frame is draw by several supporting routines.  The inner workings
    of each routine is different, but several can work together for one
    frame or several frames may use the same routine.

  Features:
    Quick efficient code to speed the drawing process.

  Copyright 1993, all rights reserved.
    Paul R. Renaud

  Compiler:
    Turbo Pascal versions 5.0 to 6.0

  System:
    MS-DOS, MDOS

*****************************************************************************)

Unit Frames;

  Interface

    Uses
      Draw;

(***********************************************************

  Vertical Iterations and Horizontal Iterations.
    Define the number of loops, lines or sections used to
    draw the frame.

***********************************************************)

    Const
      Vertical_Iterations: Word = 10;
      Horizontal_Iterations: Word = 10;

(***********************************************************

  Procedure: Set frame color.
    This procedure sets the drawing color used to draw the
    frame on the screen.

***********************************************************)

    Procedure Set_Frame_Color( Color: Byte );

(***********************************************************

  Procedure: Draw frame 1.

    This procedure draws a single curly frame with the ends
    extended outward. (/)(\)  Left defines the left side,
    top the top, right the right side and bottom the bottom
    of the frame.  X_Size defines the width of the frame and
    Y_Size defines it's height.

***********************************************************)

    Procedure Draw_Frame1( Left, Top, Right, Bottom: Integer; X_Size, Y_Size: Byte );

(***********************************************************

  Procedure: Draw frame 2.

    This procedure draws a single curly frame with the ends
    extended inward. (\)(/)  Left defines the left side, top
    the top, right the right side and bottom the bottom of
    the frame.  X_Size defines the width of the frame and
    Y_Size defines it's height.

***********************************************************)

    Procedure Draw_Frame2( Left, Top, Right, Bottom: Integer; X_Size, Y_Size: Byte );

(***********************************************************

  Procedure: Draw frame 3.

    This procedure draws a double curly frame with the ends
    extended outward. (/)(\)(/)(\)  Left defines the left
    side, top the top, right the right side and bottom the
    bottom of the frame.  X_Size defines the width of the
    frame and Y_Size defines it's height.

***********************************************************)

    Procedure Draw_Frame3( Left, Top, Right, Bottom: Integer; X_Size, Y_Size: Byte );

(***********************************************************

  Procedure: Draw frame 4.

    This procedure draws a double curly frame with the ends
    extended inward. (\)(/)(\)(/)  Left defines the left
    side, top the top, right the right side and bottom the
    bottom of the frame.  X_Size defines the width of the
    frame and Y_Size defines it's height.

***********************************************************)

    Procedure Draw_Frame4( Left, Top, Right, Bottom: Integer; X_Size, Y_Size: Byte );

(***********************************************************

  Procedure: Draw frame 5.

    This procedure draws a frame of curls looping inwards.
    Left defines the left side, top the top, right the right
    side and bottom the bottom of the frame.  X_Space
    determines the width of the frame and Y_Space defines
    the height.

***********************************************************)

    Procedure Draw_Frame5( Left, Top, Right, Bottom: Integer; X_Space, Y_Space: Byte );

(***********************************************************

  Procedure: Draw frame 6.

    This procedure draws a frame of curls looping outwards.
    Left defines the left side, top the top, right the right
    side and bottom the bottom of the frame.  X_Space
    determines the width of the frame and Y_Space defines
    the height.

***********************************************************)

    Procedure Draw_Frame6( Left, Top, Right, Bottom: Integer; X_Space, Y_Space: Byte );

(***********************************************************

  Procedure: Draw frame 7.

    This procedure draws a simple frame with lines rounding
    the corners.  Left defines the left side, top the top,
    right the right side and bottom the bottom of the frame.
    X_Space determines the width of the frame and Y_Space
    defines the height.

***********************************************************)

    Procedure Draw_Frame7( Left, Top, Right, Bottom: Integer; X_Space, Y_Space: Byte );

(***********************************************************

  Procedure: Draw frame 8.

    This procedure draws a simple frame with a point of
    lines tipping the corners.  Left defines the left side,
    top the top, right the right side and bottom the bottom
    of the frame.  X_Space determines the width of the frame
    and Y_Space defines the height.

***********************************************************)

    Procedure Draw_Frame8( Left, Top, Right, Bottom: Integer; X_Space, Y_Space: Byte );

(***********************************************************

  Procedure: Draw frame 9.

    This procedure draws a simple frame with a point of
    lines on the left and right sides.  Left defines the
    left side, top the top, right the right side and bottom
    the bottom of the frame.  X_Space determines the width
    of the frame and Y_Space defines the height.

***********************************************************)

    Procedure Draw_Frame9( Left, Top, Right, Bottom: Integer; X_Space, Y_Space: Byte );

(***********************************************************

  Procedure: Draw frame 10.

    This procedure draws a simple frame with a point of
    lines on the top and bottom sides.  Left defines the
    left side, top the top, right the right side and bottom
    the bottom of the frame.  X_Space determines the width
    of the frame and Y_Space defines the height.

***********************************************************)

    Procedure Draw_Frame10( Left, Top, Right, Bottom: Integer; X_Space, Y_Space: Byte );

(***********************************************************

  Procedure: Draw frame 11.

    This procedure draws a simple frame with two points of
    lines on the top and bottom sides.  Left defines the
    left side, top the top, right the right side and bottom
    the bottom of the frame.  X_Space determines the width
    of the frame and Y_Space defines the height.

***********************************************************)

    Procedure Draw_Frame11( Left, Top, Right, Bottom: Integer; X_Space, Y_Space: Byte );

(***********************************************************

  Procedure: Draw frame 12.

    This procedure draws a double frame with boxed corners.
    Left defines the left side, top the top, right the right
    side and bottom the bottom of the frame.  X_Space
    determines the width of the frame and Y_Space defines
    the height.

***********************************************************)

    Procedure Draw_Frame12( Left, Top, Right, Bottom: Integer; X_Space, Y_Space: Byte );

(***********************************************************

  Procedure: Draw frame 13.

    This procedure draws a double frame with crisscross
    lines between the inner and outer sides.  Left defines
    the left side, top the top, right the right side and
    bottom the bottom of the frame.  X_Space determines the
    width of the frame and Y_Space defines the height.

***********************************************************)

    Procedure Draw_Frame13( Left, Top, Right, Bottom: Integer; X_Space, Y_Space: Byte );

(***********************************************************

  Procedure: Draw frame 14.

    This procedure draws a double frame with crisscross
    curves between the inner and outer sides.  Left defines
    the left side, top the top, right the right side and
    bottom the bottom of the frame. X_Space determines the
    width of the frame and Y_Space defines the height.

***********************************************************)

    Procedure Draw_Frame14( Left, Top, Right, Bottom: Integer; X_Space, Y_Space: Byte );

(***********************************************************

  Procedure: Draw frame 15.

    This procedure draws a frame of overlapping arcs.  Left
    defines the left side, top the top, right the right side
    and bottom the bottom of the frame.  X_Space determines
    the width of the frame and Y_Space defines the height.

***********************************************************)

    Procedure Draw_Frame15( Left, Top, Right, Bottom: Integer; X_Space, Y_Space: Byte );

(***********************************************************

  Procedure: Draw frame 16.

    This procedure draws a frame of inward arcs.  Left
    defines the left side, top the top, right the right side
    and bottom the bottom of the frame.  X_Space determines
    the width of the frame and Y_Space defines the height.

***********************************************************)

    Procedure Draw_Frame16( Left, Top, Right, Bottom: Integer; X_Space, Y_Space: Byte );

(***********************************************************

  Procedure: Draw frame 17.

    This procedure draws a rounded box frame.  Left defines
    the left side, top the top, right the right side and
    bottom the bottom of the frame.  X_Space determines
    the width of the frame and Y_Space defines the height.

***********************************************************)

    Procedure Draw_Frame17( Left, Top, Right, Bottom: Integer; X_Space, Y_Space: Byte );

(***********************************************************

  Procedure: Draw frame 18.

    This procedure draws an inward rounded box frame.  Left
    defines the left side, top the top, right the right side
    and bottom the bottom of the frame.  X_Space determines
    the width of the frame and Y_Space defines the height.

***********************************************************)

    Procedure Draw_Frame18( Left, Top, Right, Bottom: Integer; X_Space, Y_Space: Byte );

(***********************************************************

  Procedure: Draw frame 19.

    This procedure draws an downward scroll frame.  Left
    defines the left side, top the top, right the right side
    and bottom the bottom of the frame.  X_Space determines
    the width of the frame and Y_Space defines the height.

***********************************************************)

    Procedure Draw_Frame19( Left, Top, Right, Bottom: Integer; X_Space, Y_Space: Byte );

(***********************************************************

  Procedure: Draw frame 20.

    This procedure draws an upward scroll frame.  Left
    defines the left side, top the top, right the right side
    and bottom the bottom of the frame.  X_Space determines
    the width of the frame and Y_Space defines the height.

***********************************************************)

    Procedure Draw_Frame20( Left, Top, Right, Bottom: Integer; X_Space, Y_Space: Byte );

{----------------------------------------------------------------------------}

  Implementation

    Var
     { This variable holds the current frame drawing color. }
      Frame_Color: Byte;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Curly horizontal.
    This procedure draws two single horizontal
    curlies for a frame, top and bottom.  Left
    defines the left edge of the curl while Right
    defines the right edge.  Stretch defines the
    curl's side radius  Height defines the curl's
    entire height. and Length defines the entire
    length.  Mid_Y1 defines the center for the top
    curl and Mid_Y2 the center for the bottom.

*************************************************)

    Procedure Curly_Horizontal( Left, Right, Stretch, Mid_Y1, Mid_Y2, Height, Length: Integer );
      Var
        Center_L,
        Center_R,
        Start_R,
        Second_R,
        Third_R,
        Forth_R,
        Fifth_V: Integer;
      Begin
        Start_R := ( Height div 2 );
        Center_L := ( Left + Stretch );
        Center_R := ( Right - Stretch );
        Second_R := Abs( ( Length div 2 ) - Stretch );
        Draw_Disjoint_Ellipse( Center_L, Mid_Y1, Center_R, Mid_Y1, Stretch, Start_R );
        Draw_Disjoint_Ellipse( Center_L, Mid_Y2, Center_R, Mid_Y2, Stretch, Start_R );
        Third_R := ( Start_R div 2 );
        Stretch := ( Stretch div 2 );
        Draw_Disjoint_Ellipse( Center_L, Mid_Y1, Center_R, Mid_Y1, Stretch, Third_R );
        Draw_Disjoint_Ellipse( Center_L, Mid_Y2, Center_R, Mid_Y2, Stretch, Third_R );
        Forth_R := ( ( Third_R * 3 ) div 2 );
        Stretch := ( ( Stretch * 3 ) div 2 );
        Fifth_V := ( Third_R div 2 );
        Draw_Elliptical_Arc( Center_L, ( Mid_Y1 + Fifth_V ), 270, 90, Stretch, Forth_R );
        Draw_Elliptical_Arc( Center_R, ( Mid_Y1 - Fifth_V ), 90, 270, Stretch, Forth_R );
        Draw_Elliptical_Arc( Center_L, ( Mid_Y2 - Fifth_V ), 270, 90, Stretch, Forth_R );
        Draw_Elliptical_Arc( Center_R, ( Mid_Y2 + Fifth_V ), 90, 270, Stretch, Forth_R );
        Draw_Elliptical_Arc( Center_L, Mid_Y1, 0, 90, Second_R, Start_R );
        Draw_Elliptical_Arc( Center_R, Mid_Y1, 180, 270, Second_R, Start_R );
        Draw_Elliptical_Arc( Center_L, Mid_Y2, 270, 360, Second_R, Start_R );
        Draw_Elliptical_Arc( Center_R, Mid_Y2, 90, 180, Second_R, Start_R );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Curly vertical.
    This procedure draws two single vertical
    curlies for a frame, Left and Right.  Top
    defines the top edge of the curl while Bottom
    defines the bottom edge.  Stretch defines the
    curl's side radius  Height defines the curl's
    entire height. and Length defines the entire
    length.  Mid_X1 defines the center for the
    left curl and Mid_X2 the center for the right.

*************************************************)

    Procedure Curly_Vertical( Top, Bottom, Stretch, Mid_X1, Mid_X2, Height, Length: Integer );
      Var
        Center_T,
        Center_B,
        Start_R,
        Second_R,
        Third_R,
        Forth_R,
        Fifth_V: Integer;
      Begin
        Start_R := ( Length div 2 );
        Center_T := ( Top + Stretch );
        Center_B := ( Bottom - Stretch );
        Second_R := ( ( Height div 2 ) - Stretch );
        Draw_Disjoint_Ellipse( Mid_X1, Center_T, Mid_X1, Center_B, Start_R, Stretch );
        Draw_Disjoint_Ellipse( Mid_X2, Center_T, Mid_X2, Center_B, Start_R, Stretch );
        Third_R := ( Start_R div 2 );
        Stretch := ( Stretch div 2 );
        Draw_Disjoint_Ellipse( Mid_X1, Center_T, Mid_X1, Center_B, Third_R, Stretch );
        Draw_Disjoint_Ellipse( Mid_X2, Center_T, Mid_X2, Center_B, Third_R, Stretch );
        Forth_R := ( ( Third_R * 3 ) div 2 );
        Stretch := ( ( Stretch * 3 ) div 2 );
        Fifth_V := ( Third_R div 2 );
        Draw_Elliptical_Arc( ( Mid_X1 - Fifth_V ), Center_T, 180, 360, Forth_R, Stretch );
        Draw_Elliptical_Arc( ( Mid_X1 + Fifth_V ), Center_B, 0, 180, Forth_R, Stretch );
        Draw_Elliptical_Arc( ( Mid_X2 + Fifth_V ), Center_T, 180, 360, Forth_R, Stretch );
        Draw_Elliptical_Arc( ( Mid_X2 - Fifth_V ), Center_B, 0, 180, Forth_R, Stretch );
        Draw_Elliptical_Arc( Mid_X1, Center_T, 270, 360, Start_R, Second_R );
        Draw_Elliptical_Arc( Mid_X1, Center_B, 90, 180, Start_R, Second_R );
        Draw_Elliptical_Arc( Mid_X2, Center_T, 180, 270, Start_R, Second_R );
        Draw_Elliptical_Arc( Mid_X2, Center_B, 0, 90, Start_R, Second_R );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Horizontal curlys.
    This procedure draws the horizontal sides of
    a curly frame.  The sides are determined by
    a combination of the given variables.
    Left, Top1, Right, Bottom1 and Left, Top2,
    Right, Bottom2.  Stretch determines the radius
    of the curlies.

*************************************************)

    Procedure Horizontal_Curlys( Left, Top1, Right, Bottom1, Top2, Bottom2, Stretch: Integer );
      Var
        Mid_Y1,
        Mid_Y2,
        Mid_X,
        Height,
        Length: Integer;
      Begin
        Mid_Y1 := ( ( Top1 + Bottom1 ) div 2 );
        Mid_Y2 := ( ( Top2 + Bottom2 ) div 2 );
        Mid_X := ( ( Right + Left ) div 2 );
        Height := ( Bottom1 - Top1 );
        Length := ( Right - Mid_X );
        Curly_Horizontal( Left, Mid_X, Stretch, Mid_Y1, Mid_Y2, Height, Length );
        Curly_Horizontal( Mid_X, Right, Stretch, Mid_Y2, Mid_Y1, Height, Length );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Horizontal two curlys.
    This procedure draws the horizontal sides of
    a double curly frame.  The sides are
    determined by a combination of the given
    variables.  Left, Top1, Right, Bottom1 and
    Left, Top2, Right, Bottom2.  Stretch
    determines the radius of the curlies.

*************************************************)

    Procedure Horizontal_2_Curlys( Left, Top1, Right, Bottom1, Top2, Bottom2, Stretch: Integer );
      Var
        Mid_Y1,
        Mid_Y2,
        Mid_X,
        Height,
        Middle,
        Length: Integer;
      Begin
        Mid_Y1 := ( ( Top1 + Bottom1 ) div 2 );
        Mid_Y2 := ( ( Top2 + Bottom2 ) div 2 );
        Mid_X := ( ( Right + Left ) div 2 );
        Height := ( Bottom1 - Top1 );
        Middle := ( ( Mid_X + Left ) div 2 );
        Length := ( Mid_X - Middle );
        Curly_Horizontal( Middle, Mid_X, Stretch, Mid_Y1, Mid_Y2, Height, Length );
        Curly_Horizontal( Left, Middle, Stretch, Mid_Y2, Mid_Y1, Height, Length );
        Middle := ( ( Mid_X + Right ) div 2 );
        Curly_Horizontal( Mid_X, Middle, Stretch, Mid_Y2, Mid_Y1, Height, Length );
        Curly_Horizontal( Middle, Right, Stretch, Mid_Y1, Mid_Y2, Height, Length );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Vertical curlys.
    This procedure draws the vertical sides of
    a curly frame.  The sides are determined by
    a combination of the given variables.
    Left1, Top, Right1, Bottom and Left2, Top,
    Right2, Bottom.  Stretch determines the radius
    of the curlies.

*************************************************)

    Procedure Vertical_Curlys( Left1, Top, Right1, Bottom, Left2, Right2, Stretch: Integer );
      Var
        Mid_Y,
        Mid_X1,
        Mid_X2,
        Height,
        Length: Integer;
      Begin
        Mid_Y := ( ( Top + Bottom ) div 2 );
        Mid_X1 := ( ( Right1 + Left1 ) div 2 );
        Mid_X2 := ( ( Right2 + Left2 ) div 2 );
        Height := ( Bottom - Mid_Y );
        Length := ( Right1 - Left1 );
        Curly_Vertical( Top, Mid_Y, Stretch, Mid_X1, Mid_X2, Height, Length );
        Curly_Vertical( Mid_Y, Bottom, Stretch, Mid_X2, Mid_X1, Height, Length );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Vertical two curlys.
    This procedure draws the vertical sides of
    a double curly frame.  The sides are
    determined by a combination of the given
    variables.  Left1, Top, Right1, Bottom and
    Left2, Top, Right2, Bottom.  Stretch
    determines the radius of the curlies.

*************************************************)

    Procedure Vertical_2_Curlys( Left1, Top, Right1, Bottom, Left2, Right2, Stretch: Integer );
      Var
        Mid_Y,
        Mid_X1,
        Mid_X2,
        Middle,
        Height,
        Length: Integer;
      Begin
        Mid_Y := ( ( Top + Bottom ) div 2 );
        Mid_X1 := ( ( Right1 + Left1 ) div 2 );
        Mid_X2 := ( ( Right2 + Left2 ) div 2 );
        Length := ( Right1 - Left1 );
        Middle := ( ( Mid_Y + Top ) div 2 );
        Height := ( Mid_Y - Middle );
        Curly_Vertical( Top, Middle, Stretch, Mid_X1, Mid_X2, Height, Length );
        Curly_Vertical( Middle, Mid_Y, Stretch, Mid_X2, Mid_X1, Height, Length );
        Middle := ( ( Mid_Y + Bottom ) div 2 );
        Curly_Vertical( Mid_Y, Middle, Stretch, Mid_X1, Mid_X2, Height, Length );
        Curly_Vertical( Middle, Bottom, Stretch, Mid_X2, Mid_X1, Height, Length );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Line corners.
    This procedure draws lines around the corners
    of a box in a rounded fashion.  The box is
    defined by Left, Top, Right, Bottom.  X_Space
    and Y_Space define how far apart the lines
    will be.

*************************************************)

    Procedure Line_Corners( Left, Top, Right, Bottom: Integer; X_Space, Y_Space: Byte );
      Var
        X1,
        X2,
        Y1,
        Y2,
        Size: Integer;
      Begin
        Size := ( ( ( Bottom - Top ) div 2 ) div X_Space );
        X1 := Left + ( Size * X_Space );
        X2 := Right - ( Size * X_Space );
        Y1 := Top;
        Y2 := Bottom;
        Repeat
          Draw_Line( X1, Top, Left, Y1 );
          Draw_Line( X2, Top, Right, Y1 );
          Draw_Line( X1, Bottom, Left, Y2 );
          Draw_Line( X2, Bottom, Right, Y2 );
          Dec( X1, X_Space );
          Inc( X2, X_Space );
          Inc( Y1, Y_Space );
          Dec( Y2, Y_Space );
        Until ( X1 <= Left );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Line corners.
    This procedure draws lines around the corners
    of a box in a pointed fashion.  The box is
    defined by Left, Top, Right, Bottom.  X_Space
    and Y_Space define how far apart the lines
    will be.

*************************************************)

    Procedure Double_Line_Corners( Left, Top, Right, Bottom: Integer; X_Space, Y_Space: Byte );
      Var
        X1,
        X2,
        Y1,
        Y2,
        Y3,
        Y4,
        Top1,
        Left1,
        Right1,
        Bottom1,
        Size: Integer;
      Begin
        Size := ( ( ( Bottom - Top ) div 2 ) div X_Space );
        X1 := Left + ( Size * X_Space );
        X2 := Right - ( Size * X_Space );
        Y1 := Top;
        Y2 := Bottom;
        Y3 := Top;
        Y4 := Bottom;
        Top1 := Top + ( Size * Y_Space );
        Bottom1 := Bottom - ( Size * Y_Space );
        Left1 := Left;
        Right1 := Right;
        Repeat
          Draw_Line( X1, Top, Left1, Y3 );
          Draw_Line( Left1, Y3, Left, Top1 );
          Draw_Line( X2, Top, Right1, Y3 );
          Draw_Line( Right1, Y3, Right, Top1 );
          Draw_Line( X1, Bottom, Left1, Y4 );
          Draw_Line( Left1, Y4, Left, Bottom1 );
          Draw_Line( X2, Bottom, Right1, Y4 );
          Draw_Line( Right1, Y4, Right, Bottom1 );
          Dec( X1, X_Space );
          Inc( X2, X_Space );
          Inc( Y1, Y_Space );
          Dec( Y2, Y_Space );
          Inc( Y3, Y_Space div 2 );
          Dec( Y4, Y_Space div 2 );
          Inc( Left1, X_Space div 2 );
          Dec( Right1, X_Space div 2 );
          Dec( Top1, Y_Space );
          Inc( Bottom1, Y_Space );
        Until ( X1 <= Left );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Vertical line center.
    This procedure draws a point in the center of
    the vertical lines of a frame.  Left, Middle
    defines the left point and Right, Middle
    defines the right point.  X_Space defines how
    far apart the lines are on the X-axis and
    Y_Space defines the vertical spacing.

*************************************************)

    Procedure Vertical_Line_Center( Left, Middle, Right: Integer; X_Space, Y_Space: Byte );
      Var
        X1,
        X2,
        Y1,
        Y2: Integer;
      Begin
        X1 := Left + ( 5 * X_Space );
        X2 := Right - ( 5 * X_Space );
        Y1 := Middle;
        Y2 := Middle;
        Repeat
          Draw_Line( X1, Middle, Left, Y1 );
          Draw_Line( X1, Middle, Left, Y2 );
          Draw_Line( X2, Middle, Right, Y1 );
          Draw_Line( X2, Middle, Right, Y2 );
          Dec( Y1, Y_Space );
          Inc( Y2, Y_Space );
          Dec( X1, X_Space );
          Inc( X2, X_Space );
        Until ( X1 <= Left );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Horizontal line center.
    This procedure draws a point in the center of
    the horizontal lines of a frame.  Middle, Top
    defines the upper point and Middle, Bottom
    defines the lower point.  X_Space defines the
    horizontal spaceing and Y_Space defines how
    far apart the lines are on the Y-axis.

*************************************************)

    Procedure Horizontal_Line_Center( Top, Middle, Bottom: Integer; X_Space, Y_Space: Byte );
      Var
        X1,
        X2,
        Y1,
        Y2: Integer;
      Begin
        X1 := Middle;
        X2 := Middle;
        Y1 := Top + ( 5 * Y_Space );
        Y2 := Bottom - ( 5 * Y_Space );
        Repeat
          Draw_Line( Middle, Y1, X1, Top );
          Draw_Line( Middle, Y1, X2, Top );
          Draw_Line( Middle, Y2, X1, Bottom );
          Draw_Line( Middle, Y2, X2, Bottom );
          Dec( Y1, Y_Space );
          Inc( Y2, Y_Space );
          Dec( X1, X_Space );
          Inc( X2, X_Space );
        Until ( Y1 <= Top );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Horizontal star.
    This procedure draws a horizontal star in the
    given box defined by Left, Top, Right, Bottom.
    X_Lines determines the amount of lines used to
    make up the star.

*************************************************)

    Procedure Horizontal_Star( Left, Top, Right, Bottom: Integer; X_Lines: Byte );
      Var
        X1,
        X2,
        Space: Integer;
      Begin
        Space := ( ( Right - Left ) div X_Lines );
        X1 := Left;
        X2 := Right;
        Repeat
          Draw_Line( X1, Top, X2, Bottom );
          Dec( X2, Space );
          Inc( X1, Space );
        Until ( X1 > Right );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Horizontal star boxes.
    This procedure draws a series of stars across
    the top and bottom sides of the frame.  The
    boxes are defined by Left, Top1, Right,
    Bottom1 and Left, Top2, Right, Bottom2.  The
    number of stars is defined by Spaces minus
    one.

*************************************************)

    Procedure Horizontal_Star_Boxes( Left, Top1, Right, Bottom1, Top2, Bottom2: Integer; Spaces: Byte );
      Var
        X,
        Space: Integer;
      Begin
        If ( Spaces < 4 )
          Then
            Exit;
        Space := Succ( ( Right - Left ) div Spaces );
        X := Left;
        Repeat
          Horizontal_Star( X, Top1, X + ( 4 * Space ), Bottom1, 5 );
          Horizontal_Star( X, Top2, X + ( 4 * Space ), Bottom2, 5 );
          Inc( X, Space );
        Until ( X >= ( Right - ( 3 * Space ) ) );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Vertical star.
    This procedure draws a vertical star in the
    given box defined by Left, Top, Right, Bottom.
    Y_Lines determines the amount of lines used to
    make up the star.

*************************************************)

    Procedure Vertical_Star( Left, Top, Right, Bottom: Integer; Y_Lines: Byte );
      Var
        Y1,
        Y2,
        Space: Integer;
      Begin
        Space := ( ( Bottom - Top ) div Y_Lines );
        Y1 := Top;
        Y2 := Bottom;
        Repeat
          Draw_Line( Left, Y1, Right, Y2 );
          Inc( Y1, Space );
          Dec( Y2, Space );
        Until ( Y1 > Bottom );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Vertical star boxes.
    This procedure draws a series of stars down
    the sides defined by Left1, Top, Right1,
    Bottom and Left2, Top, Right2, Bottom.  The
    number of stars is defined by Spaces minus
    one.

*************************************************)

    Procedure Vertical_Star_Boxes( Left1, Top, Right1, Bottom, Left2, Right2: Integer; Spaces: Byte );
      Var
        Y,
        Space: Integer;
      Begin
        If ( Spaces < 4 )
          Then
            Exit;
        Space := ( ( Bottom - Top ) div Spaces );
        Y := Top;
        Repeat
          Vertical_Star( Left1, Y, Right1, Y + ( 4 * Space ), 5 );
          Vertical_Star( Left2, Y, Right2, Y + ( 4 * Space ), 5 );
          Inc( Y, Space );
        Until ( Y >= ( Bottom - ( 3 * Space ) ) );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Horizontal arcs boxes.
    This procedure draws a series of arcs in two
    horizontal boxes define by Left, Top1, Right,
    Bottom1 and Left, Top2, Right, Bottom2.  The
    amount of arcs is determined from Spaces.

*************************************************)

    Procedure Horizontal_Arcs_Boxes( Left, Top1, Right, Bottom1, Top2, Bottom2: Integer; Spaces: Byte );
      Var
        X,
        X_Radius,
        Y_Radius,
        Count,
        Space: Integer;
      Begin
        If ( Spaces < 2 )
          Then
            Exit;
        Y_Radius := ( Bottom1 - Top1 );
        Space := Succ( ( Right - Left ) div Spaces );
        X_Radius := 2 * Space;
        X := Left + X_Radius;
        For Count := 0 to ( Spaces - 4 ) do
          Begin
            Draw_Disjoint_Ellipse( X, Bottom1, X, Top1, X_Radius, Y_Radius );
            Draw_Disjoint_Ellipse( X, Bottom2, X, Top2, X_Radius, Y_Radius );
            Inc( X, Space );
          End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Vertical arcs boxes.
    This procedure draws a series of arcs in two
    vertical boxes define by Left1, Top, Right1,
    Bottom and Left2, Top, Right2, Bottom.  The
    amount of arcs is determined from Spaces.

*************************************************)

    Procedure Vertical_Arcs_Boxes( Left1, Top, Right1, Bottom, Left2, Right2: Integer; Spaces: Byte );
      Var
        Y,
        X_Radius,
        Y_Radius,
        Count,
        Space: Integer;
      Begin
        If ( Spaces < 2 )
          Then
            Exit;
        X_Radius := ( Right1 - Left1 );
        Space := Succ( ( Bottom - Top ) div Spaces );
        Y_Radius := 2 * Space;
        Y := Top + Y_Radius;
        For Count := 0 to ( Spaces - 4 ) do
          Begin
            Draw_Disjoint_Ellipse( Right1, Y, Left1, Y, X_Radius, Y_Radius );
            Draw_Disjoint_Ellipse( Right2, Y, Left2, Y, X_Radius, Y_Radius );
            Inc( Y, Space );
          End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Vertical arcs.
    This procedure draws a series of arcs to
    define the vertical sides of the frame.  The
    arcs are draw in the boxes defined by Left1,
    Top, Right1, Bottom and Left2, Top, Right2,
    Bottom.  The number of arcs is determined from
    Space.

*************************************************)

    Procedure Vertical_Arcs( Left1, Top, Right1, Bottom, Left2, Right2: Integer; Spaces: Byte );
      Var
        Y,
        X_Radius,
        Count,
        Space: Integer;
      Begin
        If ( Spaces < 2 )
          Then
            Exit;
        X_Radius := ( Right1 - Left1 );
        Space := Succ( ( Bottom - Top ) div Spaces );
        Y := Top + Space;
        For Count := 0 to ( Spaces - 2 ) do
          Begin
            Draw_Disjoint_Ellipse( Right2, Y, Left1, Y, X_Radius, Space );
            Inc( Y, Space );
          End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Horizontal arcs.
    This procedure draws a series of arcs to
    define the horizontal sides of the frame.  The
    arcs are draw in the boxes defined by Left,
    Top1, Right, Bottom1 and Left, Top2, Right,
    Bottom2.  The number of arcs is determined
    from Space.

*************************************************)

    Procedure Horizontal_Arcs( Left, Top1, Right, Bottom1, Top2, Bottom2: Integer; Spaces: Byte );
      Var
        X,
        Y_Radius,
        Count,
        Space: Integer;
      Begin
        If ( Spaces < 2 )
          Then
            Exit;
        Y_Radius := ( Bottom1 - Top1 );
        Space := Succ( ( Right - Left ) div Spaces );
        X := Left + Space;
        For Count := 0 to ( Spaces - 2 ) do
          Begin
            Draw_Disjoint_Ellipse( X, Bottom2, X, Top1, Space, Y_Radius );
            Inc( X, Space );
          End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Horizontal loop boxes.
    This procedure draws looped lines horizontally
    across the boxes defined by Left, Top1, Right,
    Bottom1 and Left, Top2, Right, Bottom2.
    Spaces define the amount of loops in the
    drawing.

*************************************************)

    Procedure Horizontal_Loop_Boxes( Left, Top1, Right, Bottom1, Top2, Bottom2: Integer; Spaces: Byte );
      Var
        X,
        Y1,
        Y2,
        Count,
        Hold_X,
        X_Space,
        Y_Space,
        Big_X_Radius,
        Big_Y_Radius,
        Small_X_Radius,
        Small_Y_Radius: Integer;
      Begin
        If ( Spaces < 2 )
          Then
            Exit;
        X_Space := Succ( ( Right - Left ) div Spaces );
        Y_Space := ( Bottom1 - Top1 );
        X := Left;
        Small_Y_Radius := ( Y_Space div 3 );
        Y1 := ( Bottom1 - Small_Y_Radius );
        Y2 := ( Top2 + Small_Y_Radius );
        Small_X_Radius := ( X_Space div 6 );
        Big_X_Radius := ( ( X_Space div 2 ) + Small_X_Radius );
        Big_Y_Radius := ( Y_Space - Small_Y_Radius );
        Draw_Disjoint_Ellipse( Right, Y1, Left, Y2, Big_X_Radius, Big_Y_Radius );
        For Count := 0 to Pred( Pred( Spaces ) ) do
          Begin
            Hold_X := X + ( X_Space div 2 );
            Draw_Disjoint_Ellipse( Hold_X, Y2, Hold_X, Y1, Small_X_Radius, Small_Y_Radius );
            Inc( X, X_Space );
            Draw_Disjoint_Ellipse( X, Y1, X, Y2, Big_X_Radius, Big_Y_Radius );
          End;
        Hold_X := X + ( X_Space div 2 );
        Draw_Disjoint_Ellipse( Hold_X, Y2, Hold_X, Y1, Small_X_Radius, Small_Y_Radius );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Vertical loop boxes.
    This procedure draws looped lines vertically
    down the boxes defined by Left1, Top, Right1,
    Bottom and Left2, Top, Right2, Bottom.
    Spaces define the amount of loops in the
    drawing.

*************************************************)

    Procedure Vertical_Loop_Boxes( Left1, Top, Right1, Bottom, Left2, Right2: Integer; Spaces: Byte );
      Var
        Y,
        X1,
        X2,
        Count,
        Hold_Y,
        X_Space,
        Y_Space,
        Big_X_Radius,
        Big_Y_Radius,
        Small_X_Radius,
        Small_Y_Radius: Integer;
      Begin
        If ( Spaces < 2 )
          Then
            Exit;
        Y_Space := Succ( ( Bottom - Top ) div Spaces );
        X_Space := ( Right1 - Left1 );
        Y := Top;
        Small_X_Radius := ( X_Space div 3 );
        X1 := ( Left2 + Small_X_Radius );
        X2 := ( Right1 - Small_X_Radius );
        Small_Y_Radius := ( Y_Space div 6 );
        Big_Y_Radius := ( ( Y_Space div 2 ) + Small_Y_Radius );
        Big_X_Radius := ( X_Space - Small_X_Radius );
        Draw_Disjoint_Ellipse( X2, Bottom, X1, Top, Big_X_Radius, Big_Y_Radius );
        For Count := 0 to Pred( Pred( Spaces ) ) do
          Begin
            Hold_Y := Y + ( Y_Space div 2 );
            Draw_Disjoint_Ellipse( X1, Hold_Y, X2, Hold_Y, Small_X_Radius, Small_Y_Radius );
            Inc( Y, Y_Space );
            Draw_Disjoint_Ellipse( X2, Y, X1, Y, Big_X_Radius, Big_Y_Radius );
          End;
        Hold_Y := Y + ( Y_Space div 2 );
        Draw_Disjoint_Ellipse( X1, Hold_Y, X2, Hold_Y, Small_X_Radius, Small_Y_Radius );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Frame box.
    This procedure draws a frame consisting of
    three boxes.  Left, Top, Right Bottom defines
    the outward box while the two inward boxes are
    spaced apart by X_Space and Y_Space.

*************************************************)

    Procedure Frame_Box( Left, Top, Right, Bottom: Integer; X_Space, Y_Space: Byte );
      Begin
        Draw_Rectangle( Left, Top + Y_Space, Right, Bottom - Y_Space );
        Draw_Rectangle( Left + X_Space, Top, Right - X_Space, Bottom );
        Draw_Rectangle( Left, Top, Right, Bottom );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Scroll corners up.
    This procedure draws a scroll corner on the
    screen with the ends up.

*************************************************)

    Procedure Scroll_Corners_Up( Left, Top, Right, Bottom, X_Space, Y_Space: Integer );
      Var
        X,
        Y,
        X2,
        Y2,
        Left1,
        Right1: Integer;
      Begin
        Y := ( Y_Space div 4 );
        Y2 := ( Y_Space div 2 );
        X := ( X_Space div 4 );
        X2 := ( X_Space div 2 );
        Left1 := ( Left + X + X2 );
        Right1 := ( Right - X - X2 );
        Draw_Disjoint_Ellipse( Right1, Top + Y, Left1, Top + Y, X, Y );
        Draw_Disjoint_Ellipse( Left1, Top + Y + Y2, Right1, Top + Y + Y2, X, Y );
        Draw_Elliptical_Arc( Left1, Bottom - Y2, 180, 270, X, Y );
        Draw_Elliptical_Arc( Right1, Bottom - Y2, 270, 360, X, Y );
        Draw_Line( Left, Top, Left1, Top );
        Draw_Line( Right, Top, Right1, Top );
        Draw_Line( Left + X2, Top + Y + Y2, Left + X2, Bottom - Y2 );
        Draw_Line( Right - X2, Top + Y + Y2, Right - X2, Bottom - Y2 );
        Draw_Line( Left, Bottom - Y_Space, Left + X2, Bottom - Y_Space );
        Draw_Line( Right, Bottom - Y_Space, Right - X2, Bottom - Y_Space );
        Draw_Line( Left1, Top + Y_Space, Right1, Top + Y_Space );
        Draw_Line( Left1, Bottom - Y, Right1, Bottom - Y );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Scroll corners down.
    This procedure draws a scroll corner on the
    screen with the ends down.

*************************************************)

    Procedure Scroll_Corners_Down( Left, Top, Right, Bottom, X_Space, Y_Space: Integer );
      Var
        X,
        Y,
        X2,
        Y2,
        Left1,
        Right1: Integer;
      Begin
        Y := ( Y_Space div 4 );
        Y2 := ( Y_Space div 2 );
        X := ( X_Space div 4 );
        X2 := ( X_Space div 2 );
        Left1 := ( Left + X + X2 );
        Right1 := ( Right - X - X2 );
        Draw_Disjoint_Ellipse( Right1, Bottom - Y, Left1, Bottom - Y, X, Y );
        Draw_Disjoint_Ellipse( Left1, Bottom - Y - Y2, Right1, Bottom - Y - Y2, X, Y );
        Draw_Elliptical_Arc( Left1, Top + Y2, 90, 180, X, Y );
        Draw_Elliptical_Arc( Right1, Top + Y2, 0, 90, X, Y );
        Draw_Line( Left, Bottom, Left1, Bottom );
        Draw_Line( Right, Bottom, Right1, Bottom );
        Draw_Line( Left + X2, Bottom - Y - Y2, Left + X2, Top + Y2 );
        Draw_Line( Right - X2, Bottom - Y - Y2, Right - X2, Top + Y2 );
        Draw_Line( Left, Top + Y_Space, Left + X2, Top + Y_Space );
        Draw_Line( Right, Top + Y_Space, Right - X2, Top + Y_Space );
        Draw_Line( Left1, Bottom - Y_Space, Right1, Bottom - Y_Space );
        Draw_Line( Left1, Top + Y, Right1, Top + Y );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure:  Draw frame 1.
    As previously defined.

*************************************************)

    Procedure Draw_Frame1( Left, Top, Right, Bottom: Integer; X_Size, Y_Size: Byte );
      Var
        Half_X,
        Half_Y: Word;
        Save_Draw_Color: Byte;
      Begin
        Save_Draw_Color := Drawing_Color;
        Drawing_Color := Frame_Color;
        Top := ( Top + Y_Size );
        Left := ( Left + X_Size );
        Right := ( Right - X_Size );
        Bottom := ( Bottom - Y_Size );
        Half_X := ( X_Size div 3 );
        Half_Y := ( Y_Size div 3 );
        Horizontal_Curlys( ( Left - Half_X ), Bottom, ( Right + Half_X ), ( Bottom + Y_Size ), ( Top - Y_Size ), Top, X_Size );
        Vertical_Curlys( ( Left - X_Size ), ( Top - Half_Y ), Left, ( Bottom + Half_Y ), Right, ( Right + X_Size ), Y_Size );
        Drawing_Color := Save_Draw_Color;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure:  Draw frame 2.
    As previously defined.

*************************************************)

    Procedure Draw_Frame2( Left, Top, Right, Bottom: Integer; X_Size, Y_Size: Byte );
      Var
        Half_X,
        Half_Y: Word;
        Save_Draw_Color: Byte;
      Begin
        Save_Draw_Color := Drawing_Color;
        Drawing_Color := Frame_Color;
        Top := ( Top + Y_Size );
        Left := ( Left + X_Size );
        Right := ( Right - X_Size );
        Bottom := ( Bottom - Y_Size );
        Half_X := ( X_Size div 3 );
        Half_Y := ( Y_Size div 3 );
        Horizontal_Curlys( ( Left - Half_X ), ( Top - Y_Size ), ( Right + Half_X ), Top, Bottom, ( Bottom + Y_Size ), X_Size );
        Vertical_Curlys( Right, ( Top - Half_Y ), ( Right + X_Size ), ( Bottom + Half_Y ), ( Left - X_Size ), Left, Y_Size );
        Drawing_Color := Save_Draw_Color;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure:  Draw frame 3.
    As previously defined.

*************************************************)

    Procedure Draw_Frame3( Left, Top, Right, Bottom: Integer; X_Size, Y_Size: Byte );
      Var
        Half_X,
        Half_Y: Word;
        Save_Draw_Color: Byte;
      Begin
        Save_Draw_Color := Drawing_Color;
        Drawing_Color := Frame_Color;
        Top := ( Top + Y_Size );
        Left := ( Left + X_Size );
        Right := ( Right - X_Size );
        Bottom := ( Bottom - Y_Size );
        Half_X := ( X_Size div 3 );
        Half_Y := ( Y_Size div 3 );
        Horizontal_2_Curlys( ( Left - Half_X ), ( Top - Y_Size ), ( Right + Half_X ), Top, Bottom, ( Bottom + Y_Size ),
                             X_Size );
        Vertical_2_Curlys( ( Left - X_Size ), ( Top - Half_Y ), Left, ( Bottom + Half_Y ), Right, ( Right + X_Size ), Y_Size );
        Drawing_Color := Save_Draw_Color;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure:  Draw frame 4.
    As previously defined.

*************************************************)

    Procedure Draw_Frame4( Left, Top, Right, Bottom: Integer; X_Size, Y_Size: Byte );
      Var
        Half_X,
        Half_Y: Word;
        Save_Draw_Color: Byte;
      Begin
        Save_Draw_Color := Drawing_Color;
        Drawing_Color := Frame_Color;
        Top := ( Top + Y_Size );
        Left := ( Left + X_Size );
        Right := ( Right - X_Size );
        Bottom := ( Bottom - Y_Size );
        Half_X := ( X_Size div 3 );
        Half_Y := ( Y_Size div 3 );
        Horizontal_2_Curlys( ( Left - Half_X ), Bottom, ( Right + Half_X ), ( Bottom + Y_Size ), ( Top - Y_Size ), Top,
                             X_Size );
        Vertical_2_Curlys( Right, ( Top - Half_Y ), ( Right + X_Size ), ( Bottom + Half_Y ), ( Left - X_Size ), Left, Y_Size );
        Drawing_Color := Save_Draw_Color;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure:  Draw frame 5.
    As previously defined.

*************************************************)

    Procedure Draw_Frame5( Left, Top, Right, Bottom: Integer; X_Space, Y_Space: Byte );
      Var
        Save_Draw_Color: Byte;
      Begin
        Save_Draw_Color := Drawing_Color;
        Drawing_Color := Frame_Color;
        Vertical_Loop_Boxes( Left, Top, Left + X_Space, Bottom, Right - X_Space, Right, Vertical_Iterations );
        Horizontal_Loop_Boxes( Left, Top, Right, Top + Y_Space, Bottom - Y_Space, Bottom, Horizontal_Iterations );
        Drawing_Color := Save_Draw_Color;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure:  Draw frame 6.
    As previously defined.

*************************************************)

    Procedure Draw_Frame6( Left, Top, Right, Bottom: Integer; X_Space, Y_Space: Byte );
      Var
        Top1,
        Left1,
        Right1,
        Bottom1: Integer;
        Save_Draw_Color: Byte;
      Begin
        Save_Draw_Color := Drawing_Color;
        Drawing_Color := Frame_Color;
        Top1 := Top + Y_Space;
        Left1 := Left + X_Space;
        Right1 := Right - X_Space;
        Bottom1 := Bottom - Y_Space;
        Vertical_Loop_Boxes( Right1, Top1, Right, Bottom1, Left, Left1, Vertical_Iterations );
        Horizontal_Loop_Boxes( Left1, Bottom1, Right1, Bottom, Top, Top1, Horizontal_Iterations );
        Drawing_Color := Save_Draw_Color;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure:  Draw frame 7.
    As previously defined.

*************************************************)

    Procedure Draw_Frame7( Left, Top, Right, Bottom: Integer; X_Space, Y_Space: Byte );
      Var
        Save_Draw_Color: Byte;
      Begin
        Save_Draw_Color := Drawing_Color;
        Drawing_Color := Frame_Color;
        Draw_Rectangle( Left, Top, Right, Bottom );
        Line_Corners( Left, Top, Right, Bottom, X_Space, Y_Space );
        Drawing_Color := Save_Draw_Color;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure:  Draw frame 8.
    As previously defined.

*************************************************)

    Procedure Draw_Frame8( Left, Top, Right, Bottom: Integer; X_Space, Y_Space: Byte );
      Var
        Save_Draw_Color: Byte;
      Begin
        Save_Draw_Color := Drawing_Color;
        Drawing_Color := Frame_Color;
        Draw_Rectangle( Left, Top, Right, Bottom );
        Double_Line_Corners( Left, Top, Right, Bottom, X_Space, Y_Space );
        Drawing_Color := Save_Draw_Color;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure:  Draw frame 9.
    As previously defined.

*************************************************)

    Procedure Draw_Frame9( Left, Top, Right, Bottom: Integer; X_Space, Y_Space: Byte );
      Var
        Save_Draw_Color: Byte;
      Begin
        Save_Draw_Color := Drawing_Color;
        Drawing_Color := Frame_Color;
        Draw_Rectangle( Left, Top, Right, Bottom );
        Vertical_Line_Center( Left, ( ( Bottom + Top ) div 2 ), Right, X_Space, Y_Space );
        Drawing_Color := Save_Draw_Color;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure:  Draw frame 10.
    As previously defined.

*************************************************)

    Procedure Draw_Frame10( Left, Top, Right, Bottom: Integer; X_Space, Y_Space: Byte );
      Var
        Save_Draw_Color: Byte;
      Begin
        Save_Draw_Color := Drawing_Color;
        Drawing_Color := Frame_Color;
        Draw_Rectangle( Left, Top, Right, Bottom );
        Horizontal_Line_Center( Top, ( ( Right + Left ) div 2 ), Bottom, X_Space, Y_Space );
        Drawing_Color := Save_Draw_Color;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure:  Draw frame 11.
    As previously defined.

*************************************************)

    Procedure Draw_Frame11( Left, Top, Right, Bottom: Integer; X_Space, Y_Space: Byte );
      Var
        Middle: Integer;
        Save_Draw_Color: Byte;
      Begin
        Save_Draw_Color := Drawing_Color;
        Drawing_Color := Frame_Color;
        Draw_Rectangle( Left, Top, Right, Bottom );
        Middle := ( ( Right - Left ) div 2 );
        Horizontal_Line_Center( Top, ( ( Right + Middle ) div 2 ), Bottom, X_Space, Y_Space );
        Horizontal_Line_Center( Top, ( ( Middle + Left ) div 2 ), Bottom, X_Space, Y_Space );
        Drawing_Color := Save_Draw_Color;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure:  Draw frame 12.
    As previously defined.

*************************************************)

    Procedure Draw_Frame12( Left, Top, Right, Bottom: Integer; X_Space, Y_Space: Byte );
      Var
        Save_Draw_Color: Byte;
      Begin
        Save_Draw_Color := Drawing_Color;
        Drawing_Color := Frame_Color;
        Frame_Box( Left, Top, Right, Bottom, X_Space, Y_Space );
        Drawing_Color := Save_Draw_Color;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure:  Draw frame 13.
    As previously defined.

*************************************************)

    Procedure Draw_Frame13( Left, Top, Right, Bottom: Integer; X_Space, Y_Space: Byte );
      Var
        Top1,
        Left1,
        Right1,
        Bottom1: Integer;
        Save_Draw_Color: Byte;
      Begin
        Save_Draw_Color := Drawing_Color;
        Drawing_Color := Frame_Color;
        Top1 := Top + Y_Space;
        Left1 := Left + X_Space;
        Right1 := Right - X_Space;
        Bottom1 := Bottom - Y_Space;
        Frame_Box( Left, Top, Right, Bottom, X_Space, Y_Space );
        Vertical_Star_Boxes( Left, Top1, Left1, Bottom1, Right1, Right, Vertical_Iterations );
        Horizontal_Star_Boxes( Left1, Top, Right1, Top1, Bottom1, Bottom, Horizontal_Iterations );
        Drawing_Color := Save_Draw_Color;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure:  Draw frame 14.
    As previously defined.

*************************************************)

    Procedure Draw_Frame14( Left, Top, Right, Bottom: Integer; X_Space, Y_Space: Byte );
      Var
        Top1,
        Left1,
        Right1,
        Bottom1: Integer;
        Save_Draw_Color: Byte;
      Begin
        Save_Draw_Color := Drawing_Color;
        Drawing_Color := Frame_Color;
        Top1 := Top + Y_Space;
        Left1 := Left + X_Space;
        Right1 := Right - X_Space;
        Bottom1 := Bottom - Y_Space;
        Frame_Box( Left, Top, Right, Bottom, X_Space, Y_Space );
        Vertical_Arcs_Boxes( Left, Top1, Left1, Bottom1, Right1, Right, Vertical_Iterations );
        Horizontal_Arcs_Boxes( Left1, Top, Right1, Top1, Bottom1, Bottom, Horizontal_Iterations );
        Drawing_Color := Save_Draw_Color;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure:  Draw frame 15.
    As previously defined.

*************************************************)

    Procedure Draw_Frame15( Left, Top, Right, Bottom: Integer; X_Space, Y_Space: Byte );
      Var
        Top1,
        Left1,
        Right1,
        Bottom1: Integer;
        Save_Draw_Color: Byte;
      Begin
        Save_Draw_Color := Drawing_Color;
        Drawing_Color := Frame_Color;
        Top1 := Top + Y_Space;
        Left1 := Left + X_Space;
        Right1 := Right - X_Space;
        Bottom1 := Bottom - Y_Space;
        Vertical_Arcs( Left, Top, Left1, Bottom, Right1, Right, Vertical_Iterations );
        Horizontal_Arcs( Left, Top, Right, Top1, Bottom1, Bottom, Horizontal_Iterations );
        Drawing_Color := Save_Draw_Color;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure:  Draw frame 16.
    As previously defined.

*************************************************)

    Procedure Draw_Frame16( Left, Top, Right, Bottom: Integer; X_Space, Y_Space: Byte );
      Var
        X_Radius,
        Y_Radius,
        X_Center,
        Y_Center: Integer;
        Save_Draw_Color: Byte;
      Begin
        Save_Draw_Color := Drawing_Color;
        Drawing_Color := Frame_Color;
        X_Center := ( ( Left + Right ) div 2 );
        Y_Center := ( ( Top + Bottom ) div 2 );
        X_Radius := ( ( Right - Left ) div 2 );
        Y_Radius := ( ( Bottom - Top ) div 2 );
        Draw_Disjoint_Ellipse( Right, Y_Center, Left, Y_Center, X_Space, Y_Radius );
        Draw_Disjoint_Ellipse( X_Center, Bottom, X_Center, Top, X_Radius, Y_Space );
        Drawing_Color := Save_Draw_Color;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure:  Draw frame 17.
    As previously defined.

*************************************************)

    Procedure Draw_Frame17( Left, Top, Right, Bottom: Integer; X_Space, Y_Space: Byte );
      Var
        Save_Draw_Color: Byte;
      Begin
        Save_Draw_Color := Drawing_Color;
        Drawing_Color := Frame_Color;
        Draw_Ellipse_Box( Left + X_Space, Top + Y_Space, Right - X_Space, Bottom - Y_Space, X_Space, Y_Space );
        Drawing_Color := Save_Draw_Color;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure:  Draw frame 18.
    As previously defined.

*************************************************)

    Procedure Draw_Frame18( Left, Top, Right, Bottom: Integer; X_Space, Y_Space: Byte );
      Var
        Save_Draw_Color: Byte;
      Begin
        Save_Draw_Color := Drawing_Color;
        Drawing_Color := Frame_Color;
        Draw_Line( Left + X_Space, Top, Right - X_Space, Top );
        Draw_Line( Left + X_Space, Bottom, Right - X_Space, Bottom );
        Draw_Line( Left, Top + Y_Space, Left, Bottom - Y_Space );
        Draw_Line( Right, Top + Y_Space, Right, Bottom - Y_Space );
        Draw_Disjoint_Ellipse( Right, Bottom, Left, Top, X_Space, Y_Space );
        Drawing_Color := Save_Draw_Color;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure:  Draw frame 19.
    As previously defined.

*************************************************)

    Procedure Draw_Frame19( Left, Top, Right, Bottom: Integer; X_Space, Y_Space: Byte );
      Var
        Save_Draw_Color: Byte;
      Begin
        Save_Draw_Color := Drawing_Color;
        Drawing_Color := Frame_Color;
        Scroll_Corners_Up( Left, Top, Right, Bottom, X_Space, Y_Space );
        Draw_Line( Left, Top, Left, Bottom - Y_Space );
        Draw_Line( Right, Top, Right, Bottom - Y_Space );
        Drawing_Color := Save_Draw_Color;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure:  Draw frame 20.
    As previously defined.

*************************************************)

    Procedure Draw_Frame20( Left, Top, Right, Bottom: Integer; X_Space, Y_Space: Byte );
      Var
        Save_Draw_Color: Byte;
      Begin
        Save_Draw_Color := Drawing_Color;
        Drawing_Color := Frame_Color;
        Scroll_Corners_Down( Left, Top, Right, Bottom, X_Space, Y_Space );
        Draw_Line( Left, Top + Y_Space, Left, Bottom );
        Draw_Line( Right, Top + Y_Space, Right, Bottom );
        Drawing_Color := Save_Draw_Color;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure:  Set frame color.
    As previously defined.

*************************************************)

    Procedure Set_Frame_Color( Color: Byte );
      Begin
        Frame_Color := Color;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Main initialization section.

*************************************************)

  Begin
    Frame_Color := 1;
  End.

