
(*************************************************

  Procedure: Combine.
    This procedure changes the high bit of the
    given value to match the flag.

*************************************************)

    Procedure Combine( Data: Byte; Flag: Boolean; Var Value: Byte );
      Begin
        If Flag
          then
            Value := ( Data or $80 )
          else
            Value := ( Data and $7F );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Check range.
    This function returns true if the given window
    value is valid.

*************************************************)

    Function Check_Range( Window_Number: Byte ): Boolean;
      Begin
        Check_Range := ( ( Window_Number < 0 ) or ( Window_Number > Max_Windows ) or
                         ( Data^.Windows[ Window_Number ] = Nil ) )
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Put full character.
    This procedure puts the character on the
    screen if the particular location is a part of
    the given window.

*************************************************)

    Procedure Put_Full_Character( Window, Column, Row: Byte; Character: Char; Attribute: Byte );
      Begin
        If ( ( Data^.Screen_Check[ Row, Column ] and $7F ) = Window )
          then
            Put_Character_On_Screen( Column, Row, Character, Attribute );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Draw frame left.
    This procedure draws the left side of the
    window frame on the screen.

*************************************************)

    Procedure Draw_Frame_Left( Var Frame: Frame_Type; Left, Top, Bottom, Window: Byte );
      Var
        Count: Byte;
      Begin
        If ( Left > 0 )
          then
            For Count := Top to Bottom do
              Begin
                Put_Full_Character( Window, Left, Count, Frame.Data[ 4 ], Frame.Attribute );
                Put_Full_Character( Window, Succ( Left ), Count, Frame.Data[ 5 ], Frame.Attribute );
              End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Draw frame right.
    This procedure draws the right side of the
    window frame on the screen.

*************************************************)

    Procedure Draw_Frame_Right( Var Frame: Frame_Type; Top, Right, Bottom, Window: Byte );
      Var
        Count: Byte;
      Begin
        If ( Right <= Screen_Column_Limit )
          then
            For Count := Top to Bottom do
              Begin
                Put_Full_Character( Window, Right, Count, Frame.Data[ 6 ], Frame.Attribute );
                Put_Full_Character( Window, Pred( Right ), Count, Frame.Data[ 5 ], Frame.Attribute );
              End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Draw frame top.
    This procedure draws the top of the window
    frame on the screen.

*************************************************)

    Procedure Draw_Frame_Top( Var Frame: Frame_Type; Left, Top, Right, Window: Byte );
      Var
        Count: Byte;
      Begin
        If ( Top > 0 )
          then
            For Count := Left to Right do
              Begin
                Put_Full_Character( Window, Count, Top, Frame.Data[ 2 ], Frame.Attribute );
                Put_Full_Character( Window, Count, Succ( Top ), Frame.Data[ 5 ], Frame.Attribute );
              End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Draw frame bottom.
    This procedure draws the bottom of the window
    frame on the screen.

*************************************************)

    Procedure Draw_Frame_Bottom( Var Frame: Frame_Type; Left, Right, Bottom, Window: Byte );
      Var
        Count: Byte;
      Begin
        If ( Bottom <= Screen_Row_Limit )
          then
            For Count := Left to Right do
              Begin
                Put_Full_Character( Window, Count, Bottom, Frame.Data[ 8 ], Frame.Attribute );
                Put_Full_Character( Window, Count, Pred( Bottom ), Frame.Data[ 5 ], Frame.Attribute );
              End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Draw frame top left.
    This procedure draws the top left corner of
    the window frame on the screen.

*************************************************)

    Procedure Draw_Frame_Top_Left( Var Frame: Frame_Type; Left, Top, Window: Byte );
      Begin
        If ( ( Left > 0 ) and ( Top > 0 ) )
          then
            Begin
              Put_Full_Character( Window, Left, Top, Frame.Data[ 1 ], Frame.Attribute );
              Put_Full_Character( Window, Succ( Left ), Top, Frame.Data[ 2 ], Frame.Attribute );
              Put_Full_Character( Window, Left, Succ( Top ), Frame.Data[ 4 ], Frame.Attribute );
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Draw frame bottom left.
    This procedure draws the bottom left corner of
    the window frame on the screen.

*************************************************)

    Procedure Draw_Frame_Bottom_Left( Var Frame: Frame_Type; Left, Bottom, Window: Byte );
      Begin
        If ( ( Left > 0 ) and ( Bottom <= Screen_Row_Limit ) )
          then
            Begin
              Put_Full_Character( Window, Left, Bottom, Frame.Data[ 7 ], Frame.Attribute );
              Put_Full_Character( Window, Succ( Left ), Bottom, Frame.Data[ 8 ], Frame.Attribute );
              Put_Full_Character( Window, Left, Pred( Bottom ), Frame.Data[ 4 ], Frame.Attribute );
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Draw frame top right.
    This procedure draws the top right corner of
    the window frame on the screen.

*************************************************)

    Procedure Draw_Frame_Top_Right( Var Frame: Frame_Type; Top, Right, Window: Byte );
      Begin
        If ( ( Right <= Screen_Column_Limit ) and ( Top > 0 ) )
          then
            Begin
              Put_Full_Character( Window, Right, Top, Frame.Data[ 3 ], Frame.Attribute );
              Put_Full_Character( Window, Right, Succ( Top ), Frame.Data[ 6 ], Frame.Attribute );
              Put_Full_Character( Window, Pred( Right ), Top, Frame.Data[ 2 ], Frame.Attribute );
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Draw frame bottom right.
    This procedure draws the bottom right corner
    of the window frame on the screen.

*************************************************)

    Procedure Draw_Frame_Bottom_Right( Var Frame: Frame_Type; Right, Bottom, Window: Byte );
      Begin
        If ( ( Right <= Screen_Column_Limit ) and ( Bottom <= Screen_Row_Limit ) )
          then
            Begin
              Put_Full_Character( Window, Right, Bottom, Frame.Data[ 9 ], Frame.Attribute );
              Put_Full_Character( Window, Right, Pred( Bottom ), Frame.Data[ 6 ], Frame.Attribute );
              Put_Full_Character( Window, Pred( Right ), Bottom, Frame.Data[ 8 ], Frame.Attribute );
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Draw moment frame.
    This procedure draws the frame on the screen
    for a particular moment of the animation
    sequence.

*************************************************)

    Procedure Draw_Moment_Frame( Var Frame: Frame_Type; Left, Top, Right, Bottom, Window: Byte );
      Begin
        If ( ( Top < Bottom ) and ( Left < Right ) )
          then
            Begin
              Draw_Frame_Top( Frame, Left, Pred( Top ), Right, Window );
              Draw_Frame_Bottom( Frame, Left, Right, Succ( Bottom ), Window );
              Draw_Frame_Left( Frame, Pred( Left ), Top, Bottom, Window );
              Draw_Frame_Right( Frame, Top, Succ( Right ), Bottom, Window );
              Draw_Frame_Top_Left( Frame, Pred( Left ), Pred( Top ), Window );
              Draw_Frame_Bottom_Left( Frame, Pred( Left ), Succ( Bottom ), Window );
              Draw_Frame_Top_Right( Frame, Pred( Top ), Succ( Right ), Window );
              Draw_Frame_Bottom_Right( Frame, Succ( Right ), Succ( Bottom ), Window );
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Set up expand.
    This procedure sets things up in the expand
    record to do the opening animation with the
    given parameters.  It also draws the initial
    window and the initial frame.

*************************************************)

    Procedure Set_Up_Expand( Flag: Boolean; Window, Left, Top, Right, Bottom, Current_Left, Current_Top, Current_Right,
                             Current_Bottom: Byte; Var Data: Expand_Data_Type; Var Frame: Frame_Type );
      Var
        Row,
        Column: Byte;
      Begin
        If Flag
          then
            For Row := Current_Top to Current_Bottom do
              For Column := Current_Left to Current_Right do
                Put_Full_Character( Window, Column, Row, Frame.Data[ 5 ], Frame.Attribute );
        If ( Left < Current_Left )
          then
            Dec( Current_Left );
        If ( Right > Current_Right )
          then
            Inc( Current_Right );
        If ( Top < Current_Top )
          then
            Dec( Current_Top );
        If ( Bottom > Current_Bottom )
          then
            Inc( Current_Bottom );
        Data.Expand_Left := ( Left < Current_Left );
        Data.Expand_Right := ( Right > Current_Right );
        Data.Expand_Up := ( Top < Current_Top );
        Data.Expand_Down := ( Bottom > Current_Bottom );
        Data.Current_Top := Current_Top;
        Data.Current_Bottom := Current_Bottom;
        Data.Current_Left := Current_Left;
        Data.Current_Right := Current_Right;
        Data.Top := Top;
        Data.Bottom:= Bottom;
        Data.Left := Left;
        Data.Right := Right;
        Data.Frame := Frame;
        Data.Window := Window;
        Draw_Moment_Frame( Frame, Current_Left, Current_Top, Current_Right, Current_Bottom, Window );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Set up.
    This procedure sets up the expanding record
    with the appropriated data depending on the
    open method animation value.

*************************************************)

    Procedure Set_Up( Var Frame: Frame_Type; Window, Open_Method, Left, Top, Right, Bottom: Byte; Var Data: Expand_Data_Type;
                      Var Continue: Boolean );
      Var
        My_Try: Boolean; {Try isn't allowed as a variable name in Speed Pascal/2. }
        Mid_1,
        Mid_2: Byte;
      Begin
        My_Try := True;
        Mid_1 := ( ( Bottom + Top ) div 2 );
        Mid_2 := ( ( Left + Right ) div 2 );
        Case Open_Method of
          Window_Pop: Set_Up_Expand( False, Window, Left, Top, Right, Bottom, Left, Top, Right, Bottom, Data, Frame );
          Window_Down: Set_Up_Expand( My_Try, Window, Left, Top, Right, Bottom, Left, Top, Right, Top, Data, Frame );
          Window_Up: Set_Up_Expand( My_Try, Window, Left, Top, Right, Bottom, Left, Bottom, Right, Bottom, Data, Frame );
          Window_Right: Set_Up_Expand( My_Try, Window, Left, Top, Right, Bottom, Left, Top, Left, Bottom, Data, Frame );
          Window_Left: Set_Up_Expand( My_Try, Window, Left, Top, Right, Bottom, Right, Top, Right, Bottom, Data, Frame );
          Window_Vertical: Set_Up_Expand( My_Try, Window, Left, Top, Right, Bottom, Left, Mid_1, Right, Mid_1, Data, Frame );
          Window_Horizontal: Set_Up_Expand( My_Try, Window, Left, Top, Right, Bottom, Mid_2, Top, Mid_2, Bottom, Data, Frame );
          Window_Explode: Set_Up_Expand( My_Try, Window, Left, Top, Right, Bottom, Mid_2, Mid_1, Mid_2, Mid_1, Data, Frame );
          Window_Down_Right: Set_Up_Expand( My_Try, Window, Left, Top, Right, Bottom, Left, Top, Left, Top, Data, Frame );
          Window_Down_Left: Set_Up_Expand( My_Try, Window, Left, Top, Right, Bottom, Right, Top, Right, Top, Data, Frame );
          Window_Up_Right: Set_Up_Expand( My_Try, Window, Left, Top, Right, Bottom, Left, Bottom, Left, Bottom, Data, Frame );
          Window_Up_Left: Set_Up_Expand( My_Try, Window, Left, Top, Right, Bottom, Right, Bottom, Right, Bottom, Data, Frame );
          Window_Horizontal_Right: Set_Up_Expand( My_Try, Window, Left, Top, Right, Bottom, Left, Mid_1, Left, Mid_1, Data,
                                                  Frame );
          Window_Horizontal_Left: Set_Up_Expand( My_Try, Window, Left, Top, Right, Bottom, Right, Mid_1, Right, Mid_1, Data,
                                                 Frame );
          Window_Vertical_Up: Set_Up_Expand( My_Try, Window, Left, Top, Right, Bottom, Mid_2, Bottom, Mid_2, Bottom, Data,
                                             Frame );
          Window_Vertical_Down: Set_Up_Expand( My_Try, Window, Left, Top, Right, Bottom, Mid_2, Top, Mid_2, Top, Data, Frame );
        End; { Case }
        Continue := ( Data.Expand_Left or Data.Expand_Right or Data.Expand_Up or Data.Expand_Down );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Draw moment window.
    This procedure draws the window for a single
    moment of the animation sequence.  Then it
    does calculations to determine which sides
    of the window are at their full positions and
    those that aren't are updated for the next
    time this procedure is called.

*************************************************)

    Procedure Draw_Moment_Window( Var Data: Expand_Data_Type; Var Continue: Boolean );
      Begin
        With Data do
          Begin
            If Expand_Left
              then
                Dec( Current_Left );
            If Expand_Right
              then
                Inc( Current_Right );
            If Expand_Up
              then
                Dec( Current_Top );
            If Expand_Down
              then
                Inc( Current_Bottom );
            If Expand_Up
              then
                Draw_Frame_Top( Frame, Current_Left, Pred( Current_Top ), Current_Right, Window );
            If Expand_Down
              then
                Draw_Frame_Bottom( Frame, Current_Left, Current_Right, Succ( Current_Bottom ), Window );
            If Expand_Left
              then
                Draw_Frame_Left( Frame, Pred( Current_Left ), Current_Top, Current_Bottom, Window );
            If Expand_Right
              then
                Draw_Frame_Right( Frame, Current_Top, Succ( Current_Right ), Current_Bottom, Window );
            If Expand_Up or Expand_Left
              then
                Draw_Frame_Top_Left( Frame, Pred( Current_Left ), Pred( Current_Top ), Window );
            If Expand_Down or Expand_Left
              then
                Draw_Frame_Bottom_Left( Frame, Pred( Current_Left ), Succ( Current_Bottom ), Window );
            If Expand_Up or Expand_Right
              then
                Draw_Frame_Top_Right( Frame, Pred( Current_Top ), Succ( Current_Right ), Window );
            If Expand_Down or Expand_Right
              then
                Draw_Frame_Bottom_Right( Frame, Succ( Current_Right ), Succ( Current_Bottom ), Window );
            If Expand_Left
              then
                Expand_Left := ( Left < Current_Left );
            If Expand_Right
              then
                Expand_Right := ( Right > Current_Right );
            If Expand_Up
              then
                Expand_Up := ( Top < Current_Top );
            If Expand_Down
              then
                Expand_Down := ( Bottom > Current_Bottom );
            Continue := ( Expand_Left or Expand_Right or Expand_Up or Expand_Down );
          End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Frame animation.
    This procedure does the animation for opening
    the window.

*************************************************)

    Procedure Frame_Animation( Locator, Method: Byte );
      Var
        My_Data: Expand_Data_Type;
        Continue: Boolean;
      Begin
        With Data^.Windows[ Locator ]^ do
          If ( Frame_Type <> No_Frame )
            then
              Begin
                Set_Up( The_Frame, Locator, Method, Screen_Left, Screen_Top, Screen_Right, Screen_Bottom, My_Data, Continue );
                While Continue do
                  Begin
                    Draw_Moment_Window( My_Data, Continue );
                    Delay( Delay_Amount );
                  End;
              End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Update screen check.
    This procedure updates the screen checking
    structure so that the window's location is
    clearly represented.

*************************************************)

    Procedure Update_Screen_Check( Window, Left, Top, Right, Bottom, Frame: Byte );
      Var
        Row,
        Value,
        The_Unit: Byte;
      Begin
        Combine( Window, True, The_Unit );
        Value := Succ( Right - Left );
        For Row := Top to Bottom do
          FillChar( Data^.Screen_Check[ Row, Left ], Value, Window );
        If ( Frame <> No_Frame )
          then
            Begin
              For Row := Top to Bottom do
                Begin
                  Data^.Screen_Check[ Row, Pred( Left ) ] := The_Unit;
                  Data^.Screen_Check[ Row, Succ( Right ) ] := The_Unit;
                End;
              Inc( Value, 2 );
              FillChar( Data^.Screen_Check[ Pred( Top ), Pred( Left ) ], Value, The_Unit );
              FillChar( Data^.Screen_Check[ Succ( Bottom ), Pred( Left ) ], Value, The_Unit );
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Update screens checking.
    This procedure updates the entire screen
    checking structure for all the windows.

*************************************************)

    Procedure Update_Screens_Checking;
      Var
        Counter: Byte;
      Begin
        For Counter := Max_Windows downto 0 do
          If ( Data^.Windows[ Counter ] <> Nil )
            then
              With Data^.Windows[ Counter ]^ do
                If ( The_View = Visible )
                  then
                    Update_Screen_Check( Counter, Screen_Left, Screen_Top, Screen_Right, Screen_Bottom, Frame_Type );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Set up blank.
    This procedure sets up a blank line in the
    blank line structure.

*************************************************)

    Procedure Set_Up_Blank( Window_Number: Byte );
      Const
        New_Piece: Cell_Type = ( Character: ' ' );
      Var
        My_Data: Word absolute New_Piece;
      Begin
        New_Piece.Attribute := Data^.Windows[ Window_Number ]^.Storage.Attribute;
        Fill_Word( Data^.Blank_Line, Max_Columns, My_Data );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Update Virtual screen row.
    This procedure copies the new window
    information into the screen buffer structure
    so that the screen can be updated quickly.

*************************************************)

    Procedure Update_Vsr( Window_Number, Row, Left, Right: Byte; Screen: Integer );
      Var
        Value,
        Column,
        Screen_Length: Byte;
        Source: Pointer;
      Begin
        With Data^.Windows[ Window_Number ]^ do
          Begin
            Column := Where_Left;
            Value := ( Screen_Left + ( Column - Where_Left ) );
            While ( Value < Left ) do
              Begin
                Inc( Column );
                Inc( Value );
              End;
            If ( ( Column <= Where_Right ) and ( Value <= Right ) )
              then
                Begin
                  Screen_Length := Succ( Screen_Right - Value );
                  If ( Screen_Length > 0 )
                    then
                      Begin
                        If ( Row > Storage.Row_Length )
                          then
                            Source := Addr( Data^.Blank_Line )
                          else
                            Source := Address_Storage( Storage, Row, Column );
                        Move( Source^, Data^.Data[ Screen, Value ], ( Screen_Length * 2 ) );
                        Column := ( Column + Screen_Length );
                      End;
                End;
          End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Put Data.
    This procedure puts the given character and
    attribute into the screen buffer structure.

*************************************************)

    Procedure Put_D( Column, Row: Byte; The_Character: Char; The_Attribute: Byte );
      Begin
        With Data^.Data[ Row, Column ] do
          Begin
            Character := The_Character;
            Attribute := The_Attribute;
          End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Put data long.
    This procedure puts a number of copy of the
    character and attribute into the screen buffer
    structure starting at the given location.

*************************************************)

    Procedure Put_D_Long( Column, Row: Byte; Character: Char; Attribute, Length: Byte );
      Var
        Piece: Cell_Type;
        Info: Word absolute Piece;
      Begin
        Piece.Character := Character;
        Piece.Attribute := Attribute;
        Fill_Word( Data^.Data[ Row, Column ], Length, Info );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Draw frame.
    This procedure draws the frame of the window
    onto the screen buffer structure.

*************************************************)

    Procedure Draw_F( Var Frame: Frame_Type; Left, Top, Right, Bottom, Window: Byte );
      Var
        Count,
        Length,
        New_Top,
        New_Left,
        New_Right,
        New_Bottom: Byte;
      Begin
        If ( Data^.Windows[ Window ]^.Frame_Type <> No_Frame )
          then
            Begin
              New_Right := Succ( Right );
              New_Bottom := Succ( Bottom );
              New_Top := Pred( Top );
              New_Left := Pred( Left );
              If ( ( Top < Bottom ) and ( Left < Right ) )
                then
                  Begin
                    Length := Succ( Right - Left );
                    If ( Top > 1 )
                      then
                        Put_D_Long( Left, New_Top, Frame.Data[ 2 ], Frame.Attribute, Length );
                    If ( Bottom < Screen_Row_Limit )
                      then
                        Put_D_Long( Left, New_Bottom, Frame.Data[ 8 ], Frame.Attribute, Length );
                    If ( Left > 1 )
                      then
                        Begin
                          For Count := Top to Bottom do
                            Put_D( New_Left, Count, Frame.Data[ 4 ], Frame.Attribute );
                          If ( Top > 1 )
                            then
                              Put_D( New_Left, New_Top, Frame.Data[ 1 ], Frame.Attribute );
                          If ( Bottom < Screen_Row_Limit )
                            then
                              Put_D( New_Left, New_Bottom, Frame.Data[ 7 ], Frame.Attribute );
                        End;
                    If ( Right < Screen_Column_Limit )
                      then
                        Begin
                          For Count := Top to Bottom do
                            Put_D( New_Right, Count, Frame.Data[ 6 ], Frame.Attribute );
                          If ( Top > 1 )
                            then
                              Put_D( New_Right, New_Top, Frame.Data[ 3 ], Frame.Attribute );
                          If ( Bottom < Screen_Row_Limit )
                            then
                              Put_D( New_Right, New_Bottom, Frame.Data[ 9 ], Frame.Attribute );
                        End;
                  End;
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Update virtual screen.
    This procedure updates the entire virtual
    window onto the screen buffer structure.

*************************************************)

    Procedure Update_Vs( Window_Number, Left, Top, Right, Bottom: Byte );
      Var
        Row,
        Value: Byte;
      Begin
        While ( Left < 1 ) do
          Inc( Left );
        While ( Top < 1 ) do
          Inc( Top );
        With Data^.Windows[ Window_Number ]^ do
          Begin
            If ( ( Screen_Bottom >= Pred( Top ) ) and ( Screen_Top <= Succ( Bottom ) ) and
                 ( Screen_Right >= Pred( Left ) ) and ( Screen_Left <= Succ( Right ) ) and
                 ( The_View <> Invisible ) )
              then
                Begin
                  Set_Up_Blank( Window_Number );
                  For Row := Screen_Top to Screen_Bottom do
                    Begin
                      Value := ( Where_Top + ( Row - Screen_Top ) );
                      If ( ( Row >= Top ) and ( Row <= Bottom ) )
                        then
                          Update_Vsr( Window_Number, Value, Left, Right, Row );
                    End;
                  If ( Frame_Type <> No_Frame )
                    then
                      Draw_F( The_Frame, Screen_Left, Screen_Top, Screen_Right, Screen_Bottom, Window_Number );
                End;
          End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Update all virtual windows.
    This procedure updates all the virtual windows
    onto the screen buffer structure.  Then it
    copies the screen buffer structure out onto
    the real screen.

*************************************************)

    Procedure Update_All_Virtual_Windows( Left, Top, Right, Bottom: Byte );
      Var
        Counter: Byte;
      Begin
        If ( Top > 1 )
          then
            Dec( Top )
          else
            Top := 1;
        If ( Left > 1 )
          then
            Dec( Left )
          else
            Left := 1;
        If ( Bottom < Screen_Row_Limit )
          then
            Inc( Bottom )
          else
            Bottom := Screen_Row_Limit;
        If ( Right < Screen_Column_Limit )
          then
            Inc( Right )
          else
            Right := Screen_Column_Limit;
        Update_Screens_Checking;
        For Counter := Max_Windows downto 0 do
          If ( Data^.Windows[ Counter ] <> Nil )
            then
              Update_Vs( Counter, Left, Top, Right, Bottom );
        For Counter := Top to Bottom do
         {$IFNDEF OS2}
          Write_Data( Counter, Left, Data^.Data[ Counter, Left ], Succ( Right - Left ) );
         {$ELSE}
          Write_Data( Counter, Left, Addr( Data^.Data[ Counter, Left ] ), Succ( Right - Left ) );
         {$ENDIF}
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Restore row.
    This procedure restores only the given row
    onto the screen.

*************************************************)

    Procedure Restore_Row( Row, Left, Right, Top: Byte );
      Begin
        If ( Row > 0 ) and ( Row <= Screen_Row_Limit )
          then
            Begin
              While ( Left < 1 ) do
                Inc( Left );
             {$IFNDEF OS2}
              Write_Data( Row, Left, Data^.Data[ Row, Left ], Succ( Right - Left ) );
             {$ELSE}
              Write_Data( Row, Left, Addr( Data^.Data[ Row, Left ] ), Succ( Right - Left ) );
             {$ENDIF}
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Restore column.
    This procedure restores only the given column
    onto the screen.

*************************************************)

    Procedure Restore_Column( Column, Top, Bottom, Left: Byte );
      Var
        Row: Byte;
      Begin
        If ( Column > 0 ) and ( Column <= Screen_Column_Limit )
          then
            Begin
              While ( Top < 1 ) do
                Inc( Top );
              For Row := Top to Bottom do
               {$IFNDEF OS2}
                Write_Data( Row, Column, Data^.Data[ Row, Column ], 1 );
               {$ELSE}
                Write_Data( Row, Column, Addr( Data^.Data[ Row, Column ] ), 1 );
               {$ENDIF}
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Quick restore.
    This procedure restores the given window onto
    the screen quickly, without any animation.

*************************************************)

    Procedure Quick_Restore( Left, Top, Right, Bottom: Byte );
      Var
        Row: Byte;
        Count,
        Value: Word;
      Begin
        Value := ( Top + Bottom );
        For Row := Top to Succ( Value div 2 ) do
          Begin
            Count := ( Value - Row );
            Restore_Row( Row, Left, Right, Top );
            Restore_Row( Count, Left, Right, Top );
          End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Draw small corners.
    This procedure draws all the corners of the
    window frame on the screen.

*************************************************)

    Procedure Draw_Small_Corners( Var Frame: Frame_Type; Left, Top, Right, Bottom, Window: Byte );
      Begin
        If ( Left > 1 )
          then
            Begin
              If ( Top > 1 )
                then
                  Put_Full_Character( Window, Left, Top, Frame.Data[ 1 ], Frame.Attribute );
              If ( Bottom < Screen_Row_Limit )
                then
                  Put_Full_Character( Window, Left, Bottom, Frame.Data[ 7 ], Frame.Attribute );
            End;
        If ( Right < Screen_Column_Limit )
          then
            Begin
              If ( Top > 1 )
                then
                  Put_Full_Character( Window, Right, Top, Frame.Data[ 3 ], Frame.Attribute );
              If ( Bottom < Screen_Row_Limit )
                then
                  Put_Full_Character( Window, Right, Bottom, Frame.Data[ 9 ], Frame.Attribute );
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Draw small top.
    This procedure draws the frame top, minus the
    corners, on the screen.

*************************************************)

    Procedure Draw_Small_Top( Var Frame: Frame_Type; Left, Top, Right, Bottom, Window: Byte );
      Var
        Count: Byte;
      Begin
        If ( ( Top > 0 ) and ( Top < Bottom ) )
          then
            For Count := Succ( Left ) to Pred( Right ) do
              Put_Full_Character( Window, Count, Top, Frame.Data[ 2 ], Frame.Attribute );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Draw small bottom.
    This procedure draws the frame bottom, minus
    the corners, on the screen.

*************************************************)

    Procedure Draw_Small_Bottom( Var Frame: Frame_Type; Left, Top, Right, Bottom, Window: Byte );
      Var
        Count: Byte;
      Begin
        If ( ( Bottom <= Screen_Row_Limit ) and ( Top < Bottom ) )
          then
            For Count := Succ( Left ) to Pred( Right ) do
              Put_Full_Character( Window, Count, Bottom, Frame.Data[ 8 ], Frame.Attribute );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Draw small left.
    This procedure draws the left side of the
    frame, minus the corners, on the screen.

*************************************************)

    Procedure Draw_Small_Left( Var Frame: Frame_Type; Left, Top, Right, Bottom, Window: Byte );
      Var
        Count: Byte;
      Begin
        If ( ( Left > 0 ) and ( Left < Right ) )
          then
            For Count := Top to Bottom do
              Put_Full_Character( Window, Left, Count, Frame.Data[ 4 ], Frame.Attribute );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Draw small right.
    This procedure draws the right side of the
    frame, minus the corners, on the screen.

*************************************************)

    Procedure Draw_Small_Right( Var Frame: Frame_Type; Left, Top, Right, Bottom, Window: Byte );
      Var
        Count: Byte;
      Begin
        If ( ( Right <= Screen_Column_Limit ) and ( Left < Right ) )
          then
            For Count := Top to Bottom do
              Put_Full_Character( Window, Right, Count, Frame.Data[ 6 ], Frame.Attribute );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Restore it.
    This procedure restores the given window of
    the screen buffer structure on the screen,
    using the given side parameters.

*************************************************)

    Procedure Restore_it( Var Frame: Frame_Type; Left, Top, Right, Bottom, Window: Byte;
                          Do_Left, Do_Top, Do_Right, Do_Bottom: Boolean );
      Var
        Current_Top,
        Current_Left,
        Current_Right,
        Current_Bottom: Byte;
      Begin
        Current_Left := Left;
        Current_Top := Top;
        Current_Right := Right;
        Current_Bottom := Bottom;
        If ( Current_Left < 1 )
          then
            Current_Left := 1;
        If ( Current_Top < 1 )
          then
            Current_Top := 1;
        If ( Current_Right > Screen_Column_Limit )
          then
            Current_Right := Screen_Column_Limit;
        If ( Current_Bottom > Screen_Row_Limit )
          then
            Current_Bottom := Screen_Row_Limit;
        While ( ( Current_Left <= Current_Right ) and ( Current_Top <= Current_Bottom ) ) do
          Begin
            Draw_Small_Corners( Frame, Current_Left, Current_Top, Current_Right, Current_Bottom, Window );
            Delay( Delay_Amount );
            If Do_Left
              then
                Begin
                  Restore_Column( Current_Left, Top, Current_Bottom, Left );
                  Inc( Current_Left );
                  Draw_Small_Left( Frame, Current_Left, Current_Top, Current_Right, Current_Bottom, Window );
                  Do_Left := ( Current_Left <= Current_Right );
                End;
            If Do_Top
              then
                Begin
                  Restore_Row( Current_Top, Left, Current_Right, Top );
                  Inc( Current_Top );
                  Draw_Small_Top( Frame, Current_Left, Current_Top, Current_Right, Current_Bottom, Window );
                  Do_Top := ( Current_Top <= Current_Bottom );
                End;
            If Do_Right
              then
                Begin
                  Restore_Column( Current_Right, Top, Current_Bottom, Left );
                  Dec( Current_Right );
                  Draw_Small_Right( Frame, Current_Left, Current_Top, Current_Right, Current_Bottom, Window );
                  Do_Right := ( Current_Right >= Current_Left );
                End;
            If Do_Bottom
              then
                Begin
                  Restore_Row( Current_Bottom, Left, Current_Right, Top );
                  Dec( Current_Bottom );
                  Draw_Small_Bottom( Frame, Current_Left, Current_Top, Current_Right, Current_Bottom, Window );
                  Do_Bottom := ( Current_Bottom >= Current_Top );
                End;
          End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Restore window.
    This procedure restores the portion of the
    screen using the given animation method.

*************************************************)

    Procedure Restore_Window( Var Frame: Frame_Type; Method, Left, Top, Right, Bottom, Window: Byte );
      Begin
        Case Method of
          Window_Down: Restore_it( Frame, Left, Top, Right, Bottom, Window, False, False, False, True );
          Window_Up: Restore_it( Frame, Left, Top, Right, Bottom, Window, False, True, False, False );
          Window_Right: Restore_it( Frame, Left, Top, Right, Bottom, Window, False, False, True, False );
          Window_Left: Restore_it( Frame, Left, Top, Right, Bottom, Window, True, False, False, False );
          Window_Vertical: Restore_it( Frame, Left, Top, Right, Bottom, Window, True, False, True, False );
          Window_Horizontal: Restore_it( Frame, Left, Top, Right, Bottom, Window, False, True, False, True );
          Window_Explode: Restore_it( Frame, Left, Top, Right, Bottom, Window, True, True, True, True );
          Window_Horizontal_Right: Restore_it( Frame, Left, Top, Right, Bottom, Window, False, True, True, True );
          Window_Horizontal_Left: Restore_it( Frame, Left, Top, Right, Bottom, Window, True, True, False, True );
          Window_Vertical_Up: Restore_it( Frame, Left, Top, Right, Bottom, Window, True, True, True, False );
          Window_Vertical_Down: Restore_it( Frame, Left, Top, Right, Bottom, Window, True, False, True, True );
          Window_Down_Right: Restore_it( Frame, Left, Top, Right, Bottom, Window, False, False, True, True );
          Window_Down_Left: Restore_it( Frame, Left, Top, Right, Bottom, Window, True, False, False, True );
          Window_Up_Right: Restore_it( Frame, Left, Top, Right, Bottom, Window, False, True, True, False );
          Window_Up_Left: Restore_it( Frame, Left, Top, Right, Bottom, Window, True, True, False, False );
          else Quick_Restore( Left, Top, Right, Bottom );
        End; { Case }
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Close no frame.
    This procedure restores the given screen
    window quickly.

*************************************************)

    Procedure Close_No_Frame( Left, Top, Right, Bottom: Byte );
      Var
        Counter: Byte;
      Begin
        Update_Screens_Checking;
        For Counter := Max_Windows downto 0 do
          If ( Data^.Windows[ Counter ] <> Nil )
            then
              Update_Vs( Counter, Left, Top, Right, Bottom );
        Quick_Restore( Left, Top, Right, Bottom );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Close frame.
    This procedure closes the given window
    location on the screen then updates the
    screen definition location structure.

*************************************************)

    Procedure Close_Frame( Var Frame: Frame_Type; Left, Top, Right, Bottom, Method, Window: Byte );
      Var
        Counter: Byte;
      Begin
        For Counter := Max_Windows downto 0 do
          If ( Data^.Windows[ Counter ] <> Nil )
            then
              Update_Vs( Counter, Left, Top, Right, Bottom );
        Restore_Window( Frame, Method, Left, Top, Right, Bottom, Window );
        Update_Screens_Checking;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Get the length.
    This function calculates the length of the
    valid screen segment so that more data can
    be written to the screen when updating it.

*************************************************)

    Function Get_The_Length( Mark, Row, Column: Byte ): Word;
      Var
        The_Length: Word;
      Begin
        The_Length := 0;
        While ( Column <= Screen_Column_Limit ) and ( Data^.Screen_Check[ Row, Column ] = Mark ) do
          Begin
            Inc( The_Length );
            Inc( Column );
          End;
        Get_The_Length := The_Length;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Update virtual window row.
    This procedure updates only the given section
    of the virtual window on the screen.

*************************************************)

    Procedure Update_Virtual_Window_Row( Window_Number, Row, Screen: Byte );
      Var
        Value,
        Column,
        The_Length: Byte;
        Source: Pointer;
      Begin
        With Data^.Windows[ Window_Number ]^ do
          Begin
            Column := Where_Left;
            Value := ( Screen_Left + ( Column - Where_Left ) );
            While ( Column <= Where_Right ) do
              Begin
                The_Length := Get_The_Length( Window_Number, Screen, Value );
                If ( The_Length > 0 )
                  then
                    Begin
                      Source := Address_Storage( Storage, Row, Column );
                     {$IFNDEF OS2}
                      Write_Data( Screen, Value, Source^, The_Length );
                     {$ELSE}
                      Write_Data( Screen, Value, Source, The_Length );
                     {$ENDIF}
                      Column := ( Column + The_Length );
                      Value := ( Value + The_Length );
                    End
                  else
                    Begin
                      Inc( Column );
                      Inc( Value );
                    End;
              End;
          End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Update virtual window.
    This procedure updates the entire virtual
    window on the screen.

*************************************************)

    Procedure Update_Virtual_Window( Window_Number: Byte );
      Var
        Row,
        Value: Byte;
      Begin
        With Data^.Windows[ Window_Number ]^ do
          If ( The_View = Visible )
            then
              For Row := Where_Top to Where_Bottom do
                Begin
                  Value := ( Screen_Top + ( Row - Where_Top ) );
                  Update_Virtual_Window_Row( Window_Number, Row, Value );
                End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Update virtual character.
    This procedure only updates the part of the
    screen for the given character on the screen.
    { Makes for much faster processing. }

*************************************************)

    Procedure Update_Virtual_Character( Window_Number, Row, Column: Byte; Character: Char; Attribute: Byte );
      Var
        Value1,
        Value2: Byte;
      Begin
        With Data^.Windows[ Window_Number ]^ do
          If ( The_View = Visible )
            then
              Begin
                Value1 := ( Screen_Left + ( Column - Where_Left ) );
                Value2 := ( Screen_Top + ( Row - Where_Top ) );
                If ( Data^.Screen_Check[ Value2, Value1 ] = Window_Number )
                  then
                    Put_Character_On_Screen( Value1, Value2, Character, Attribute );
              End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Write character on virtual window.
    This procedure handles writing a single
    character onto the virtual window and the
    appropriate location on the screen.  If the
    new character is at the end of the line, then
    this procedure takes care of the overhead of
    skipping over to the next line.

*************************************************)

    Procedure Write_Character_On_Virtual_Window( Window_Number: Byte; Character: Char );
      Var
        Attribute: Byte;
      Begin
        With Data^.Windows[ Window_Number ]^ do
          Begin
            If ( ( Storage.cursor_Column > Storage.Column_Length ) or
                 ( Storage.cursor_Row > Storage.Row_Length ) or
                 ( Storage.cursor_Row < 1 ) or ( Storage.cursor_Column < 1 ) )
              then
                Write_Error( 201, 'Write_Character_On_Virtual_Window: Invalid cursor' );
            Attribute := Storage.Attribute;
            Put_To_Storage( Storage, Storage.Cursor_Row, Storage.Cursor_Column, Character, Attribute );
            Update_Virtual_Character( Window_Number, Storage.Cursor_Row, Storage.Cursor_Column, Character, Attribute );
            Inc( Storage.Cursor_Column );
            If ( Storage.Cursor_Column > Storage.Column_Length )
              then
                Begin
                  Storage.Cursor_Column := 1;
                  Inc( Storage.Cursor_Row );
                  If ( Storage.Cursor_Row > Storage.Row_Length )
                    then
                      Begin
                        Scroll_Storage_Up( Storage );
                        Update_Virtual_Window( Window_Number );
                        Storage.Cursor_Row := Storage.Row_Length;
                      End;
                End;
          End;
      End;

{----------------------------------------------------------------------------}

{$IFNDEF OS2}

(*************************************************

  Function: Window output.
    This function is substituted in the text file
    to handle the output to the virtual window.

*************************************************)

    Function Window_Output( Var The_File: TextRec ): Integer;
      Var
        Point: Word;
      Begin
        With The_File, Extra_Record( UserData ) do
          Begin
            If not Check_Range( Current_Window )
              then
                Begin
                  Point := 0;
                  While Point < BufPos do
                    Begin
                      Case BufPtr^[ Point ] of
                        #10:
                          With Data^.Windows[ Current_Window ]^ do
                            Begin
                              Inc( Storage.Cursor_Row );
                              If ( Storage.Cursor_Row > Storage.Row_Length )
                                then
                                  Begin
                                    Scroll_Storage_Up( Storage );
                                    Update_Virtual_Window( Current_Window );
                                    Storage.Cursor_Row := Storage.Row_Length;
                                 End;
                            End;
                        #13:
                          Data^.Windows[ Current_Window ]^.Storage.Cursor_Column := 1;
                        else
                          Write_Character_On_Virtual_Window( Current_Window, BufPtr^[ Point ] );
                      End; { Case }
                      Inc( Point );
                    End;
                  BufPos := 0;
                End;
          End;
        Window_Output := 0;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Window close.
    This function handles closing the virtual
    window output file.

*************************************************)

    Function Window_Close( Var The_File: TextRec ): Integer;
      Begin
        Window_Close := 0;
        The_File.Mode := FmClosed;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Window Open.
    This function handles opening the virtual
    window output file.

*************************************************)

    Function Window_Open( Var The_File: TextRec ): Integer;
      Begin
        Window_Open := 0;
        The_File.Mode := FmOutput;
        The_File.InOutFunc := @Window_Output;
        The_File.FlushFunc := @Window_Output;
        The_File.CloseFunc := @Window_Close;
      End;

{$ENDIF}

{----------------------------------------------------------------------------}

(*************************************************

  Function: Move up.
    This function moves the virtual window up on
    the screen.

*************************************************)

    Function Move_Up( The_Window: Byte ): Boolean;
      Begin
        With Data^.Windows[ The_Window ]^ do
          If ( Screen_Top > 1 )
            then
              Begin
                Dec( Screen_Top );
                Dec( Screen_Bottom );
                Update_All_Virtual_Windows( Screen_Left, Screen_Top, Screen_Right, Succ( Screen_Bottom ) );
                Move_Up := True;
              End
            else
              Move_Up := False;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Move down.
    This function moves the virtual window down
    on the screen.

*************************************************)

    Function Move_Down( The_Window: Byte ): Boolean;
      Begin
        With Data^.Windows[ The_Window ]^ do
          If ( Screen_Bottom < Screen_Row_Limit )
            then
              Begin
                Inc( Screen_Bottom );
                Inc( Screen_Top );
                Update_All_Virtual_Windows( Screen_Left, Pred( Screen_Top ), Screen_Right, Screen_Bottom );
                Move_Down := True;
              End
            else
              Move_Down := False;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Move left.
    This function moves the virtual window left
    on the screen.

*************************************************)

    Function Move_Left( The_Window: Byte ): Boolean;
      Begin
        With Data^.Windows[ The_Window ]^ do
          If ( Screen_Left > 1 )
            then
              Begin
                Dec( Screen_Left );
                Dec( Screen_Right );
                Update_All_Virtual_Windows( Screen_Left, Screen_Top, Succ( Screen_Right ), Screen_Bottom );
                Move_Left := True;
              End
            else
              Move_Left := False;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Move right.
    This function moves the virtual window right
    on the screen.

*************************************************)

    Function Move_Right( The_Window: Byte ): Boolean;
      Begin
        With Data^.Windows[ The_Window ]^ do
          If ( Screen_Right < Screen_Column_Limit )
            then
              Begin
                Inc( Screen_Left );
                Inc( Screen_Right );
                Update_All_Virtual_Windows( Pred( Screen_Left ), Screen_Top, Screen_Right, Screen_Bottom );
                Move_Right := True;
              End
            else
              Move_Right := False;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Expand up.
    This function expands the virtual window up
    on the screen.

*************************************************)

    Function Expand_Up( The_Window: Byte ): Boolean;
      Begin
        With Data^.Windows[ The_Window ]^ do
          If ( Screen_Top > 1 ) and ( Where_Bottom < Storage.Row_Length )
            then
              Begin
                Dec( Screen_Top );
                Inc( Where_Bottom );
                Update_All_Virtual_Windows( Screen_Left, Screen_Top, Screen_Right, Screen_Bottom );
                Expand_Up := True;
              End
            else
              Expand_Up := False;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Expand down.
    This function expands the virtual window down
    on the screen.

*************************************************)

    Function Expand_Down( The_Window: Byte ): Boolean;
      Begin
        With Data^.Windows[ The_Window ]^ do
          If ( Screen_Bottom < Screen_Row_Limit ) and ( Where_Bottom < Storage.Row_Length )
            then
              Begin
                Inc( Screen_Bottom );
                Inc( Where_Bottom );
                Update_All_Virtual_Windows( Screen_Left, Screen_Top, Screen_Right, Screen_Bottom );
                Expand_Down := True;
              End
            else
              Expand_Down := False;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Expand left.
    This function expands the virtual window left
    on the screen.

*************************************************)

    Function Expand_Left( The_Window: Byte ): Boolean;
      Begin
        With Data^.Windows[ The_Window ]^ do
          If ( Screen_Left > 1 ) and ( Where_Right < Storage.Column_Length )
            then
              Begin
                Dec( Screen_Left );
                Inc( Where_Right );
                Update_All_Virtual_Windows( Screen_Left, Screen_Top, Screen_Right, Screen_Bottom );
                Expand_Left := True;
              End
            else
              Expand_Left := False;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Expand right.
    This function expands the virtual window right
    on the screen.

*************************************************)

    Function Expand_Right( The_Window: Byte ): Boolean;
      Begin
        With Data^.Windows[ The_Window ]^ do
          If ( Screen_Right < Screen_Column_Limit ) and ( Where_Right < Storage.Column_Length )
            then
              Begin
                Inc( Screen_Right );
                Inc( Where_Right );
                Update_All_Virtual_Windows( Screen_Left, Screen_Top, Screen_Right, Screen_Bottom );
                Expand_Right := True;
              End
            else
              Expand_Right := False;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Reduce up.
    This function reduces the virtual window up
    on the screen.

*************************************************)

    Function Reduce_Up( The_Window: Byte ): Boolean;
      Begin
        With Data^.Windows[ The_Window ]^ do
          If ( ( Screen_Bottom - Screen_Top ) > 2 )
            then
              Begin
                Dec( Screen_Bottom );
                Dec( Where_Bottom );
                Update_All_Virtual_Windows( Screen_Left, Screen_Top, Screen_Right, Succ( Screen_Bottom ) );
                Reduce_Up := True;
              End
            else
              Reduce_Up := False;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Reduce down.
    This function reduces the virtual window down
    on the screen.

*************************************************)

    Function Reduce_Down( The_Window: Byte ): Boolean;
      Begin
        With Data^.Windows[ The_Window ]^ do
          If ( ( Screen_Bottom - Screen_Top ) > 2 )
            then
              Begin
                Inc( Screen_Top );
                Dec( Where_Bottom );
                Update_All_Virtual_Windows( Screen_Left, Pred( Screen_Top ), Screen_Right, Screen_Bottom );
                Reduce_Down := True;
              End
            else
              Reduce_Down := False;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Reduce left.
    This function reduces the virtual window left
    on the screen.

*************************************************)

    Function Reduce_Left( The_Window: Byte ): Boolean;
      Begin
        With Data^.Windows[ The_Window ]^ do
          If ( ( Screen_Right - Screen_Left ) > 2 )
            then
              Begin
                Dec( Screen_Right );
                Dec( Where_Right );
                Update_All_Virtual_Windows( Screen_Left, Screen_Top, Succ( Screen_Right ), Screen_Bottom );
                Reduce_Left := True;
              End
            else
              Reduce_Left := False;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Reduce right.
    This function reduces the virtual window right
    on the screen.

*************************************************)

    Function Reduce_Right( The_Window: Byte ): Boolean;
      Begin
        With Data^.Windows[ The_Window ]^ do
          If ( ( Screen_Right - Screen_Left ) > 2 )
            then
              Begin
                Inc( Screen_Left );
                Dec( Where_Right );
                Update_All_Virtual_Windows( Pred( Screen_Left ), Screen_Top, Screen_Right, Screen_Bottom );
                Reduce_Right := True;
              End
            else
              Reduce_Right := False;
      End;
