{$B-} { Compiler Directive: Complete boolean equation evaluation: Off }
{$F+} { Compiler Directive: Generate far procedure calls: On }
{$O+} { Compiler Directive: Generate overlay code: On }

(*****************************************************************************

  Page Maker
    version 1.21

  This unit holds several integrated procedures designed to make printing
    pages easier.

  Purpose:
    To make it easy to write variable output to the printer without having
    to concern the program with the particulars about the size of the lines.

  How it works:
    A page in memory is allocated to compose the page in, then all information
    passed to the unit is handled on the page before it is committed to the
    printer.

  Features:
    The unit can be set up with the initialization codes for many printers.

  Limitations:
    This unit is set up for an Epson Fx compatible printer.
    These routines work only with text based printer, not with graphics.
    For this routine to work, it must allocate a large amount of memory to
      hold the working page.
    The PageFile text output system is not currently supported under Speed 
      Pascal/2.

  Versions
    1.2 - Adds the PageFile text output system to ease the handling of output
          to the printer. ( PageFile_Attribute, PageFile )
    1.21 - Added code to allow the unit to compile under Speed Pascal/2.

  Copyright 1989, 1995, All rights reserved.
    Paul R. Renaud

  Compilers:
    Turbo Pascal versions 4.0 to 6.0
    Speed Pascal/2 version 1.5

  Systems:
    MS-DOS, MDOS, OS/2

*****************************************************************************)

{$IFNDEF OS2}
Unit PageMaker;
{$ELSE}
Unit PageMake;
{$ENDIF}

  Interface

    Uses
      DOS,
      Core,
      Printer,
     {$IFNDEF OS2}
      String_Utilities;
     {$ELSE}
      String_U;
     {$ENDIF}

(***********************************************************

  Attributes.
    The attributes are defined here for use in the
    procedures.  They may all be combined to work together
    except that when both Pica and Elite are chosen, Elite
    will dominate.
      Pica allows 10 characters per inch, Elite allows 12.
      Italics slants the characters toward the right.
      Reverse displays characters by drawing the background
        instead.
      Expanded increases the width of the character.
      Underline puts an underscore under the character.
      Subscript causes the character to be shorter than
        normal.
      Heightened causes the character to be extended down-
        ward.
      Compressed decreases the width of the character.
      Emphasized increases line thickness in formation of
        characters.
      Superscript pushes the shortened character upward.
      Doublestrike causes printer to print character again
        over itself.

***********************************************************)

    Const
      Pica         = 0;
      Elite        = 1;
      Italics      = 32;
      Reverse      = 128; { Non standard }
      Expanded     = 16;
      Underline    = 64;
      Subscript    = 512; { Non standard }
      Heightened   = 1024; { Non standard }
      Compressed   = 2;
      Emphasized   = 4;
      Superscript  = 256; { Non standard }
      Doublestrike = 8;

(***********************************************************

  Line styles for setting up the line.
    The line styles determine how the procedure will adjust
    the line when adding it to the page.  Only one can be
    chosen.
      Regular performs no changes to the line.
      Centered pushes the text to the center of the line.
      Justified spreads out the text to fill the entire
        line.
      Left justified moves the text flush to the left
        margin.
      Right justified moves the text flush to the right
        margin.

***********************************************************)

      Regular         = 0;
      Centered        = 1;
      Justified       = 4;
      Left_Justified  = 3;
      Right_Justified = 2;

(***********************************************************

  Small string type
    This is defined as a way to save space both internally
    to the unit and externally when passing printer
    initiation codes to the unit.

***********************************************************)

    Type
      Small_String = String[ 8 ];

(***********************************************************

  PageFile_Attribute
    Defines the attribute style of the PageFile. ( See
    Attribute )

***********************************************************)

    Var
      PageFile_Attribute: Word;

(***********************************************************

  PageFile
    Defines a text file with direct output to the printer
    subsystem.  Note that PageFile does not support
    justification of text.  {Not currently supported using
    Speed Pascal/2}

***********************************************************)

     {$IFNDEF OS2}
      PageFile: Text;
     {$ENDIF}

(***********************************************************

  Function: Tab.

    This simple function returns a string of spaces with a
    length equal to the value given in amount.

***********************************************************)

    Function Tab( Amount: Byte ): String;

(***********************************************************

  Procedure: Set up the printer.

    This procedure is included in the event that the built
    in printer control codes do not work with the printer
    connected to the computer.  In the event they are
    incompatible, the program should change the codes using
    this procedure.
      If the value passed is '' then the new value will be
      disregarded.
      Page size can also be altered with this procedure.

***********************************************************)

    Procedure Set_Up_Printer( Pica,            Elite,
                              Expanded_On,     Expanded_Off,
                              Compressed_On,   Compressed_Off,
                              Emphasized_On,   Emphasized_Off,
                              Doublestrike_On, Doublestrike_Off,
                              Italics_On,      Italics_Off,
                              Underline_On,    Underline_Off,
                              Reverse_On,      Reverse_Off,
                              Superscript_On,  Superscript_Off,
                              Subscript_On,    Subscript_Off,
                              Heightened_On,   Heightened_Off: Small_String;
                              Number_Of_Lines_Per_Page: Byte );

(***********************************************************

  Procedure: Page clear.

    This procedure clears the current page no matter what
    is in it.  All the defaults are then set back to normal.

***********************************************************)

    Procedure Page_Clear;

(***********************************************************

  Procedure: Page margins.

    This procedure allows the page margins to be set.  If
    the value passed is zero, the margin will not be
    changed.

***********************************************************)

    Procedure Page_Margins( Left, Right: Byte );

(***********************************************************

  Procedures: Page headings.

    These procedures allow you to define a heading for the
    page.  The headings will not change until another
    request to do so is performed or the page is cleared.
    Page headings will normally print on the top of all
    pages.
    How takes the parameter of the line styles which were
      previously defined.  The attributes are determined by
      adding up the predefined attributes.
      Example..
        To get both underscore and emphasized print use..
        ( UnderScore + Emphasized + Pica )
    Once an attribute is set, it should not be necessary to
    change it.

***********************************************************)

    Procedure Page_Header1( How: Byte; Attributes: Word; Title: String );
    Procedure Page_Header2( How: Byte; Attributes: Word; Title: String );
    Procedure Page_Header3( How: Byte; Attributes: Word; Title: String );
    Procedure Page_Header4( How: Byte; Attributes: Word; Title: String );
    Procedure Page_Header5( How: Byte; Attributes: Word; Title: String );

(***********************************************************

  Procedures: Page write and page writeLn.

    These procedures works similar to write and writeln but
    must take the data as a string.  How and attributes work
    like previously defined.
    Write will not process a carriage return after the line.
    WriteLn will process a carriage return after the line.
    If the line is too long, a carriage return will occur.
    If the carriage return occurs at the bottom of the page,
    that page will be printed and a new page will be
    started.

***********************************************************)

    Procedure Page_Write( How: Byte; Attributes: Word; Data: String );
    Procedure Page_WriteLn( How: Byte; Attributes: Word; Data: String );

(***********************************************************

  Procedures: Page footers.

    These work similarly to header but will be on the bottom
    of the page.

***********************************************************)

    Procedure Page_Footer1( How: Byte; Attributes: Word; Foot: String );
    Procedure Page_Footer2( How: Byte; Attributes: Word; Foot: String );
    Procedure Page_Footer3( How: Byte; Attributes: Word; Foot: String );
    Procedure Page_Footer4( How: Byte; Attributes: Word; Foot: String );
    Procedure Page_Footer5( How: Byte; Attributes: Word; Foot: String );

(***********************************************************

  Procedure: Page print.

    This procedure causes the page to be printed and then
    reset for more data.  It is intended for when a
    partially filled page needs to be printed.

***********************************************************)

    Procedure Page_Print;

(***********************************************************

  Procedure: Page Reset.
    This procedure works similarly to page clear but will
    not affect the headers or footers.

***********************************************************)

    Procedure Page_Reset;

{----------------------------------------------------------------------------}

  Implementation

    Const
      { These constants are defined for an Epson FX compatible printer. }
      Default_Pica_On: Small_String = #27#80;
      Default_Elite_On: Small_String = #27#77;
      Default_Italics_On: Small_String = #27#52;
      Default_Reverse_On: Small_String = #27#126#50#01;
      Default_Reverse_Off: Small_String = #27#126#50#00;
      Default_Italics_Off: Small_String = #27#53;
      Default_Expanded_On: Small_String = #27#87#01;
      Default_Expanded_Off: Small_String = #27#87#00;
      Default_Underline_On: Small_String = #27#45#01;
      Default_Subscript_On: Small_String = #27#83#01;
      Default_Subscript_Off: Small_String = #27#84;
      Default_Heightened_On: Small_String = #27#126#49#01;
      Default_Underline_Off: Small_String = #27#45#00;
      Default_Compressed_On: Small_String = #15;
      Default_Emphasized_On: Small_String = #27#69;
      Default_Heightened_Off: Small_String = #27#126#49#00;
      Default_Compressed_Off: Small_String = #18;
      Default_Superscript_On: Small_String = #27#83#00;
      Default_Emphasized_Off: Small_String = #27#70;
      Default_Superscript_Off: Small_String = #27#84;
      Default_Doublestrike_On: Small_String = #27#71;
      Default_Doublestrike_Off: Small_String = #27#72;

      { Standard amount of lines per page. }
      Maximum_Lines_Per_Page = 65;
      { Limit to the amount of characters allowed per line. }
      Maximum_Characters_Per_Line = 160;
      { Default amount of characters to allow per line. }
      Default_Characters_Per_Line = 80;

    Type
      { Definition of the page storage structure. }
      Character_Type = Record
                         Character: Char;
                         Attribute: Word;
                       End;
      Line_Array_Type = array[ 1 .. Maximum_Characters_Per_Line ] of Character_Type;
      Line_Type = Record
                    Position: Byte;
                    Line: Line_Array_Type;
                    Length_Free: Real;
                  End;
      Page_Array_Type = array[ 1 .. Maximum_Lines_Per_Page ] of Line_Type;
      Page_Type = Record
                    Positions: Byte;
                    Lines: Page_Array_Type;
                  End;
      Page_Pointer_Type = ^Page_Type;

    Const
      { Used to location a line with a heightened character. }
      High: Boolean = False;
      { Defines the left margin stop. }
      Left_Margin: Byte = 5;
      { Defines the right margin stop. }
      Right_Margin: Byte = 5;
      { Defines the initial character mode. }
      Initialize: Byte = Pica;
      { Defines the initial character attribute. }
      Default_Attribute: Word = Pica;
      { Defines the maximum lines per page. }
      Lines_Per_Page: Byte = Maximum_Lines_Per_Page;

    Var
      { Used to allocate the page buffer on the heap. }
      Page: Page_Pointer_Type;
      { Used to hold the current characters in the line. }
      Characters_In_Line: Byte;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Check expanded.
    This function performs the calculation
    depending on the value of expanded in the
    attribute.

*************************************************)

    Function Check_Expanded( Attribute: Word; Base1, Base2: Byte ): Real;
      Begin
        If ( ( Attribute and Expanded ) <> 0 )
          then
            Check_Expanded := ( Default_Characters_Per_Line / Base1 )
          else
            Check_Expanded := ( Default_Characters_Per_Line / Base2 );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Check compressed.
    This function performs the calculation
    depending on the value of compressed in the
    attribute.

*************************************************)

    Function Check_Compressed( Attribute: Word; Base1, Base2, Base3, Base4: Byte ): Real;
      Begin
        If ( ( Attribute and Compressed ) <> 0 )
          then
            Check_Compressed := Check_Expanded( Attribute, Base1, Base2 )
          else
            Check_Compressed := Check_Expanded( Attribute, Base3, Base4 );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Get amount.
    This function returns a real value depending
    on attribute which determines how much of the
    line the character will take up.  This is
    very useful in determining how long the line
    will be.

*************************************************)

    Function Get_Amount( Attribute: Word ): Real;
      Begin
        If ( ( Attribute and Elite ) <> 0 )
          then
            Get_Amount := Check_Compressed( Attribute, 80, 160, 48, 96 )
          else
            Get_Amount := Check_Compressed( Attribute, 68, 136, 40, 80 );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Characters per line.
    This function calculates the amount of
    characters that can go into the line as a
    real number.

*************************************************)

    Function Characters_Per_Line( Attribute: Word ): Real;
      Begin
        Characters_Per_Line := ( Characters_In_Line / Get_Amount( Attribute ) );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Add character.
    This function returns false if there is no
    more room on the line for the character.
    Otherwise, it adds the character to the line.

*************************************************)

    Function Add_Character( Row: Byte; Letter: Char; Letter_Attribute: Word ): Boolean;
      Var
        Amount_Needed: Real;
      Begin
        If ( Row > Page^.Positions )
          then
            Page^.Positions := Row;
        Amount_Needed := Get_Amount( Letter_Attribute );
        With Page^.Lines[ Row ] Do
          Begin
            If ( ( Amount_Needed <= Length_Free ) and ( Position <= Maximum_Characters_Per_Line ) )
              then
                Begin
                  With Line[ Position ] do
                    Begin
                      Character := Letter;
                      Attribute := Letter_Attribute;
                    End;
                  Length_Free := ( Length_Free - Amount_Needed );
                  Inc( Position );
                  Add_Character := True;
                End
              else
                Add_Character := False;
          End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Line clear.
    This procedure clears the line of characters
    selected by row from the page buffer.

*************************************************)

    Procedure Line_Clear( Row: Byte );
      Var
        Column: Byte;
      Begin
         With Page^.Lines[ Row ] do
           Begin
             Position := Left_Margin;
             Characters_In_Line := ( Default_Characters_Per_Line - ( Left_Margin + Right_Margin ) );
             Length_Free := Characters_In_Line;
             For Column := 1 to Maximum_Characters_Per_Line do
               With Line[ Column ] do
                 Begin
                   Character := ' ';
                   Attribute := 0;
                 End;
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Page clear.
    As previously defined.

*************************************************)

    Procedure Page_Clear;
      Var
        Row,
        Column: Word;
      Begin
        Page^.Positions := 1;
        For Row := 1 to Maximum_Lines_Per_Page do
          With Page^.Lines[ Row ] do
            Begin
              Position := 1;
              For Column := 1 to Maximum_Characters_Per_Line do
                With Line[ Column ] do
                  Begin
                    Character := ' ';
                    Attribute := Initialize;
                  End;
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Page reset.
    As previously defined.

*************************************************)

    Procedure Page_Reset;
      Var
        Row,
        Column: Word;
      Begin
        Page^.Positions := 6;
        For Row := 6 to ( Lines_Per_Page - 5 ) do
          With Page^.Lines[ Row ] do
            Begin
              Position := Left_Margin;
              For Column := 1 to Maximum_Characters_Per_Line do
                With Line[ Column ] do
                  Begin
                    Character := ' ';
                    Attribute := Initialize;
                  End;
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Page margins.
    As previously defined.

*************************************************)

    Procedure Page_Margins( Left, Right: Byte );
      Begin
        If ( Left <> 0 )
          then
            Left_Margin := Left;
        If ( Right <> 0 )
          then
            Right_Margin := Right;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Add justified.
    This procedure adds the given data to the
    page buffer at the given row.  How determines
    how the data will be justified in regards to
    the given row.

*************************************************)

    Procedure Add_Justified( How: Byte; Attributes: Word; Var Title: String; Row: Byte );
      Var
        Okay: Boolean;
        Count: Integer;
        Amounts: Real;
      Begin
        Line_Clear( Row );
        Case How of
          Regular:
            Begin
              For Count := 1 to Length( Title ) do
               Okay := Add_Character( Row, Title[ Count ], Attributes );
            End;
          Centered:
            Begin
              While ( Title[ 1 ] = ' ' ) do
                Delete( Title, 1, 1 );
              While ( Title[ Length( Title ) ] = ' ' ) do
                Delete( Title, Length( Title ), 1 );
              Amounts := ( ( Characters_Per_Line( Attributes ) - Length( Title ) ) / 2 );
              For Count := 1 to Round( Amounts ) do
                Okay := Add_Character( Row, ' ', Attributes );
              For Count := 1 to Length( Title ) do
                Okay := Add_Character( Row, Title[ Count ], Attributes );
            End;
          Left_Justified:
            Begin
              While ( Title[ 1 ] =  ' ' ) do
                Delete( Title, 1, 1 );
              While ( Title[ Length( Title ) ] = ' ' ) do
                Delete( Title, Length( Title ), 1 );
              For Count := 1 to Length( Title ) do
                Okay := Add_Character( Row, Title[ Count ], Attributes );
            End;
          Right_Justified:
            Begin
              While ( Title[ 1 ] =  ' ' ) do
                Delete( Title, 1, 1 );
              While ( Title[ Length( Title ) ] = ' ' ) do
                Delete( Title, Length( Title ), 1 );
              Amounts := ( Characters_Per_Line( Attributes ) - Length( Title ) );
              For Count := 1 to Round( Amounts ) do
                Okay := Add_Character( Row, ' ', Attributes );
              For Count := 1 to Length( Title ) do
                Okay := Add_Character( Row, Title[ Count ], Attributes );
            End;
          Justified:
            Begin
              If ( Length( Title ) < Maximum_Characters_Per_Line )
                then
                  Spread_Out_String( Title, Trunc( Characters_Per_Line( Attributes ) ) );
              For Count := 1 to Length( Title ) do
                Okay := Add_Character( Row, Title[ Count ], Attributes );
            End;
        End; { Case }
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Page header 1.
    As previously defined.

*************************************************)

    Procedure Page_Header1( How: Byte; Attributes: Word; Title: String );
      Begin
        Add_Justified( How, Attributes, Title, 1 );
        Page^.Positions := 6;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Page header 2.
    As previously defined.

*************************************************)

    Procedure Page_Header2( How: Byte; Attributes: Word; Title: String );
      Begin
        Add_Justified( How, Attributes, Title, 2 );
        Page^.Positions := 6;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Page header 3.
    As previously defined.

*************************************************)

    Procedure Page_Header3( How: Byte; Attributes: Word; Title: String );
      Begin
        Add_Justified( How, Attributes, Title, 3 );
        Page^.Positions := 6;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Page header 4.
    As previously defined.

*************************************************)

    Procedure Page_Header4( How: Byte; Attributes: Word; Title: String );
      Begin
        Add_Justified( How, Attributes, Title, 4 );
        Page^.Positions := 6;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Page header 5.
    As previously defined.

*************************************************)

    Procedure Page_Header5( How: Byte; Attributes: Word; Title: String );
      Begin
        Add_Justified( How, Attributes, Title, 5 );
        Page^.Positions := 6;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Page footer 1.
    As previously defined.

*************************************************)

    Procedure Page_Footer1( How: Byte; Attributes: Word; Foot: String );
      Var
        Save: Byte;
      Begin
        Save := Page^.Positions;
        Add_Justified( How, Attributes, Foot, ( Lines_Per_Page - 4 ) );
        Page^.Positions := Save;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Page footer 2.
    As previously defined.

*************************************************)

    Procedure Page_Footer2( How: Byte; Attributes: Word; Foot: String );
      Var
        Save: Byte;
      Begin
        Save := Page^.Positions;
        Add_Justified( How, Attributes, Foot, ( Lines_Per_Page - 3 ) );
        Page^.Positions := Save;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Page footer 3.
    As previously defined.

*************************************************)

    Procedure Page_Footer3( How: Byte; Attributes: Word; Foot: String );
      Var
        Save: Byte;
      Begin
        Save := Page^.Positions;
        Add_Justified( How, Attributes, Foot, ( Lines_Per_Page - 2 ) );
        Page^.Positions := Save;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Page footer 4.
    As previously defined.

*************************************************)

    Procedure Page_Footer4( How: Byte; Attributes: Word; Foot: String );
      Var
        Save: Byte;
      Begin
        Save := Page^.Positions;
        Add_Justified( How, Attributes, Foot, ( Lines_Per_Page - 1 ) );
        Page^.Positions := Save;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Page footer 5.
    As previously defined.

*************************************************)

    Procedure Page_Footer5( How: Byte; Attributes: Word; Foot: String );
      Var
        Save: Byte;
      Begin
        Save := Page^.Positions;
        Add_Justified( How, Attributes, Foot, Lines_Per_Page );
        Page^.Positions := Save;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Tab.
    As previously defined.

*************************************************)

    Function Tab( Amount: Byte ): String;
      Var
        Counter: Byte;
        Temporary_Data: String;
      Begin
        Temporary_Data := '';
        For Counter := 1 to Amount do
          Temporary_Data := Temporary_Data + ' ';
        Tab := Temporary_Data;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Set attributes.
    This function adds the the given attributes
    codes into the given data string.

*************************************************)

    Function Set_Attribute( The_Attribute: Word ): String;
      Var
        Data: String;
      Begin
        Data := '';
        If ( The_Attribute <> Default_Attribute )
          then
            Begin
              If ( Lo( The_Attribute ) <> Lo( Default_Attribute ) )
                then
                  Begin
                    If ( ( The_Attribute and Pica ) = Pica )
                      then
                        Data := Data + Default_Pica_On
                      else
                        If ( ( Default_Attribute and Pica ) = Pica )
                          then
                            Data := Data + Default_Elite_On;
                    If ( ( The_Attribute and Expanded ) = Expanded )
                      then
                        Data := Data + Default_Expanded_On
                      else
                        If ( ( Default_Attribute and Expanded ) = Expanded )
                          then
                            Data := Data + Default_Expanded_Off;
                    If ( ( The_Attribute and Compressed ) = Compressed )
                      then
                        Data := Data + Default_Compressed_On
                      else
                        If ( ( Default_Attribute and Compressed ) = Compressed )
                          then
                            Data := Data + Default_Compressed_Off;
                    If ( ( The_Attribute and Emphasized ) = Emphasized )
                      then
                        Data := Data + Default_Emphasized_On
                      else
                        If ( ( Default_Attribute and Emphasized ) = Emphasized )
                          then
                            Data := Data + Default_Emphasized_Off;
                    If ( ( The_Attribute and Doublestrike ) = Doublestrike )
                      then
                        Data := Data + Default_Doublestrike_On
                      else
                        If ( ( Default_Attribute and Doublestrike ) = Doublestrike )
                          then
                            Data := Data + Default_Doublestrike_Off;
                    If ( ( The_Attribute and Italics ) = Italics )
                      then
                        Data := Data + Default_Italics_On
                      else
                        If ( ( Default_Attribute and Italics ) = Italics )
                          then
                            Data := Data + Default_Italics_Off;
                    If ( ( The_Attribute and Underline ) = Underline )
                      then
                        Data := Data + Default_Underline_On
                      else
                        If ( ( Default_Attribute and Underline ) = Underline )
                          then
                            Data := Data + Default_Underline_Off;
                  End;
              If ( Hi( The_Attribute ) <> Hi( Default_Attribute ) )
                then
                  Begin
                    If ( ( The_Attribute and Superscript ) = Superscript )
                      then
                        Data := Data + Default_Superscript_On
                      else
                        If ( ( The_Attribute and Subscript ) = Subscript )
                          then
                            Data := Data + Default_Subscript_On
                          else
                            Data := Data + Default_Subscript_Off;
                    If ( ( The_Attribute and Reverse ) = Reverse )
                      then
                        Data := Data + Default_Reverse_On
                      else
                        If ( ( Default_Attribute and Reverse ) = Reverse )
                          then
                            Data := Data + Default_Reverse_Off;
                    If ( ( The_Attribute and Heightened ) = Heightened )
                      then
                        Begin
                          Data := Data + Default_Heightened_On;
                          High := True;
                        End
                      else
                        If ( ( Default_Attribute and Heightened ) = Heightened )
                          then
                            Data := Data + Default_Heightened_Off;
                  End;
              Default_Attribute := The_Attribute;
            End;
        Set_Attribute := Data;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Print line.
    This procedure writes the given row of
    information in the page buffer out on the
    output device.

*************************************************)

    Procedure Print_Line( Var Lst: Text; Row: Byte );
      Var
        Column: Byte;
      Begin
        With Page^.Lines[ Row ] do
          Begin
            High := False;
            For Column := 1 to Pred( Position ) do
              Begin
                Write( Lst, Set_Attribute( Line[ Column ].Attribute ) );
                Write( Lst, Line[ Column ].Character );
              End;
            If ( Not High )
              then
                Write( Lst, #13, #10 ) { WriteLn = Carriage_Return + Line_Feed }
              else
                Write( Lst, #13 ); { Carriage_Return }
          End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Page print.
    As previously defined.

*************************************************)

    Procedure Page_Print;
      Var
        Row: Byte;
      Begin
        For Row := 1 to Lines_Per_Page do
          If ( Page^.Lines[ Row ].Position > 1 )
            then
              Print_Line( Lst, Row )
            else
              WriteLn( Lst );
        Write( Lst, #12 );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Add character to line.
    This procedure adds the given character and
    the given attribute to the page buffer, taking
    care to go down to the next line, or the next
    page, if the character can't fit on the
    current buffer line.

*************************************************)

    Procedure Add_Character_To_Line( Character: Char; Attributes: Word );
      Begin
        If ( Page^.Positions < 6 )
          then
            Page^.Positions := 6;
        If ( Not Add_Character( Page^.Positions, Character, Attributes ) )
          then
            Begin
              Repeat;
                Inc( Page^.Positions );
                If ( Page^.Positions > ( Lines_Per_Page - 5 ) )
                  then
                    Begin
                      Page_Print;
                      Page_Reset;
                      If ( Page^.Positions < 6 )
                        then
                          Page^.Positions := 6;
                    End
                  else
                    Line_Clear( Page^.Positions );
              Until Add_Character( Page^.Positions, Character, Attributes );
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Page write.
    As previously defined.

*************************************************)

    Procedure Page_Write( How: Byte; Attributes: Word; Data: String );
      Var
        Counter: Byte;
      Begin
        If ( Page^.Positions > ( Lines_Per_Page - 5 ) )
          then
            Begin
              Page_Print;
              Page_Reset;
            End;
        If ( How = Regular )
          then
            For Counter := 1 to Length( Data ) do
              Add_Character_To_Line( Data[ Counter ], Attributes )
          else
            Add_Justified( How, Attributes, Data, Page^.Positions );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Page write line.
    As previously defined.

*************************************************)

    Procedure Page_WriteLn( How: Byte; Attributes: Word; Data: String );
      Begin
        Page_Write( How, Attributes, Data );
        Inc( Page^.Positions );
        If ( Page^.Positions > ( Lines_Per_Page - 5 ) )
          then
            Begin
              Page_Print;
              Page_Reset;
            End;
        If ( Page^.Positions < 6 )
          then
            Page^.Positions := 6;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Set up printer.
    As previously defined.

*************************************************)

    Procedure Set_Up_Printer( Pica,            Elite,
                              Expanded_On,     Expanded_Off,
                              Compressed_On,   Compressed_Off,
                              Emphasized_On,   Emphasized_Off,
                              Doublestrike_On, Doublestrike_Off,
                              Italics_On,      Italics_Off,
                              Underline_On,    Underline_Off,
                              Reverse_On,      Reverse_Off,
                              Superscript_On,  Superscript_Off,
                              Subscript_On,    Subscript_Off,
                              Heightened_On,   Heightened_Off: Small_String;
                              Number_Of_Lines_Per_Page: Byte );
      Begin
        If ( Pica <> '' )
          then
            Default_Pica_On := Pica;
        If ( Elite <> '' )
          then
            Default_Elite_On := Elite;
        If ( Italics_On <> '' )
          then
            Default_Italics_On := Italics_On;
        If ( Reverse_On <> '' )
          then
            Default_Reverse_On := Reverse_On;
        If ( Reverse_Off <> '' )
          then
            Default_Reverse_Off := Reverse_Off;
        If ( Italics_Off <> '' )
          then
            Default_Italics_Off := Italics_Off;
        If ( Expanded_On <> '' )
          then
            Default_Expanded_On := Expanded_On;
        If ( Expanded_Off <> '' )
          then
            Default_Expanded_Off := Expanded_Off;
        If ( Underline_On <> '' )
          then
            Default_Underline_On := Underline_On;
        If ( Subscript_On <> '' )
          then
            Default_Subscript_On := Subscript_On;
        If ( Subscript_Off <> '' )
          then
            Default_Subscript_Off := Subscript_Off;
        If ( Heightened_On <> '' )
          then
            Default_Heightened_On := Heightened_On;
        If ( Underline_Off <> '' )
          then
            Default_Underline_Off := Underline_Off;
        If ( Compressed_On <> '' )
          then
            Default_Compressed_On := Compressed_On;
        If ( Emphasized_On <> '' )
          then
            Default_Emphasized_On := Emphasized_On;
        If ( Emphasized_Off <> '' )
          then
            Default_Emphasized_Off := Emphasized_Off;
        If ( Compressed_Off <> '' )
          then
            Default_Compressed_Off := Compressed_Off;
        If ( Superscript_On <> '' )
          then
            Default_Superscript_On := Superscript_On;
        If ( Heightened_Off <> '' )
          then
            Default_Heightened_Off := Heightened_Off;
        If ( Superscript_Off <> '' )
          then
            Default_Superscript_Off := Superscript_Off;
        If ( Doublestrike_On <> '' )
          then
            Default_Doublestrike_On := Doublestrike_On;
        If ( Doublestrike_Off <> '' )
          then
            Default_Doublestrike_Off := Doublestrike_Off;
        If ( Number_Of_Lines_Per_Page <> 0 )
          then
            Lines_Per_Page := Number_Of_Lines_Per_Page;
      End;

{----------------------------------------------------------------------------}

{$IFNDEF OS2}

(*************************************************

  Function: Printer output.
    This function is substituted in the text file
    to handle the output to the printer.

*************************************************)

    Function Printer_Output( Var The_File: TextRec ): Integer;
      Var
        Point: Word;
      Begin
        With The_File do
          Begin
            Point := 0;
            While Point < BufPos do
              Begin
                Case BufPtr^[ Point ] of
                  #10: { Line feed }
                    Begin
                      Inc( Page^.Positions );
                      If ( Page^.Positions > ( Lines_Per_Page - 5 ) )
                        then
                          Begin
                            Page_Print;
                            Page_Reset;
                          End;
                      If ( Page^.Positions < 6 )
                        then
                          Page^.Positions := 6;
                      Line_Clear( Page^.Positions );
                    End;
                  #13: { Carrage return }
                    { Ignore this character } ;
                  else
                    Add_Character_To_Line( BufPtr^[ Point ], PageFile_Attribute )
                End; { Case }
                Inc( Point );
              End;
            BufPos := 0;
          End;
        Printer_Output := 0;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function:  Printer close.
    This function handles closing the printer
    output file.

*************************************************)

    Function Printer_Close( Var The_File: TextRec ): Integer;
      Begin
        Printer_Close := 0;
        The_File.Mode := FmClosed;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Printer Open.
    This function handles opening the printer
    output file.

*************************************************)

    Function Printer_Open( Var The_File: TextRec ): Integer;
      Begin
        Printer_Open := 0;
        The_File.Mode := FmOutput;
        The_File.InOutFunc := @Printer_Output;
        The_File.FlushFunc := @Printer_Output;
        The_File.CloseFunc := @Printer_Close;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Assign file to the printer system.
    This procedure ties a file to the printer
    system.

*************************************************)

    Procedure Assign_File_To_Printer_System( Var The_File: Text );
      Begin
        With TextRec( The_File ) do
          Begin
            Mode := FmClosed;
            BufSize := SizeOf( Buffer );
            BufPtr := @Buffer;
            OpenFunc := @Printer_Open;
            Name[ 0 ] := #0;
          End;
      End;

{$ENDIF}

{----------------------------------------------------------------------------}

(*************************************************

  Main initialization section.
    Allocate memory on the heap for the printer
    page buffer.

*************************************************)

    Begin
      New( Page );
      If ( Page = Nil )
        then
         {$IFDEF VER40}
          Halt( 203 );
         {$ELSE}
          RunError( 203 );
         {$ENDIF}
     {$IFNDEF OS2}
      Assign_File_To_Printer_System( PageFile );
      Reset( PageFile );
     {$ENDIF}
      PageFile_Attribute := Regular;
    End.

