{$F+} { Compiler directive: Generate far procedure calls: On } { Do not change! }
{$O+} { Compiler directive: Generate overlay code: On }

(*****************************************************************************

  Structures
    Version 1.2

  This unit holds all the procedures necessary to manage stacks, queues and
    binary trees.  The structure is defined when implemented and created.

  Purpose:
    To allow easy implementation of various programming structures in memory
      for quickest data access.

  How it works!
    First the structure is created in memory.
    Next the data is inserted into the structure with calls to insert it.
    After that, the data can be easily accessed using various methods unique
      to each structure.

  Features:
    User defined structures allow each structure to have it's own code
      record.  This allows the same code to support several different
      structures of the same type.
    Using undefined pointers allow easy defining of record types.  This code
      also supports the use of objects.

  Versions
    1.1 - Added two functions, one for the stack and one for the queue
      structure that returning true if the given structure is empty.
    1.2 - Updated code to allow for larger OS/2 memory blocks.

  CopyRight 1991, 1996, all rights reserved.
    Paul R. Renaud

  Compilers:
    Turbo Pascal versions 4.0 to 6.0
    Speed Pascal/2 version 1.5

  Systems:
    MS-DOS, MDOS, OS/2

*****************************************************************************)

{$IFNDEF OS2}
Unit Structures;
{$ELSE}
Unit Structur;
{$ENDIF}

  Interface

    Uses
      DOS;

    Const
     { Maximum allowed record size.  Shouldn't exceed stack space. }
     {$IFNDEF OS2}
      Maximum_Record_Size = 6144;
     {$ELSE}
      Maximum_Record_Size = 10000;
     {$ENDIF}
     { Enables the best balance of the binary tree. }
      Prime_Factor_Default = 2;
     { May be change to increase speed, but is set for optimal balance. }
      Prime_Factor: Byte = Prime_Factor_Default;

    Type
     {$IFNDEF OS2}
      Word_Type = Word;
     {$ELSE}
      Word_Type = LongWord;
     {$ENDIF}
     { Structure that holds the data. }
      Data_Type = packed Array[ 1 .. Maximum_Record_Size ] of Byte;
      { Structure that holds the stack node and the queue node. }
      Structure_Element_Pointer = ^Structure_Element_Type;
      Structure_Element_Type = Record
                                 Next: Structure_Element_Pointer;
                                 Data: Data_Type;
                               End;
      { Structure that holds the binary tree node. }
      Point_Type = ^Tree_Node;
      Look_Type = Record
                    Height: Integer;
                    Left_Node_Pointer,
                    Right_Node_Pointer,
                    Previous_Node_Pointer: Point_Type;
                  End;
      Tree_Node = Record
                    Point: Look_Type;
                    Data: Data_Type;
                  End;
      { Structure that holds the necessary stack information. }
      Stack_Type = Record
                     Top: Structure_Element_Pointer;
                     Data_Size,
                     Record_Size: Word_Type;
                   End;
      { Structure that holds the necessary queue information. }
      Queue_Type = Record
                     Top,
                     Bottom: Structure_Element_Pointer;
                     Data_Size,
                     Record_Size: Word_Type;
                   End;
      { Structure that holds the function parameter. }
     {$IFNDEF VER40}
      Compare_Function = Function( Var Data1, Data2; Word1, Word2: Word_Type ): Boolean;
     {$ENDIF}
      { Structure that holds the necessary tree information. }
      Tree_Type = Record
                    Start,
                    Finish,
                    Data_Size,
                    Record_Size: Word_Type;
                    Top,
                    Where: Point_Type;
                   {$IFNDEF VER40}
                    Equal,
                    Less_Than: Compare_Function;
                   {$ENDIF}
                  End;

(***********************************************************

  Procedure: Create stack.

    This procedure prepares the stack record for data of the
    given size.

***********************************************************)

    Procedure Create_Stack( Var Stack: Stack_Type; Data_Size: Word_Type );

(***********************************************************

  Procedure: Push on.

    This procedure takes the data and copies it onto the
    stack.

***********************************************************)

    Procedure Push_On( Var Stack: Stack_Type; Var Data );

(***********************************************************

  Procedure: Examine stack.

    This procedure returns the value of the top of the
    stack.

***********************************************************)

    Procedure Examine_Stack( Var Stack: Stack_Type; Var Data );

(***********************************************************

  Procedure: Pop off.

    This procedure removes the top element of the stack and
    returns it.

***********************************************************)

    Procedure Pop_Off( Var Stack: Stack_Type; Var Data );

(***********************************************************

  Procedure: Discard stack.

    This procedure clears all the elements off of the Stack.

***********************************************************)

    Procedure Discard_Stack( Var Stack: Stack_Type );

(***********************************************************

  Function: Is stack empty?

    This procedure returns true if the stack structure is
    empty, otherwise it returns false.

***********************************************************)

    Function Stack_Empty( Var Stack: Stack_Type ): Boolean;

(***********************************************************

  Procedure: Create queue.

    This procedure prepares the queue record for data of the
    given size.

***********************************************************)

    Procedure Create_Queue( Var Queue: Queue_Type; Data_Size: Word_Type );

(***********************************************************

  Procedure: Put in.

    This procedure takes the data and copies it onto the
    queue.

***********************************************************)

    Procedure Put_In( Var Queue: Queue_Type; Var Data );

(***********************************************************

  Procedure: Examine queue.

    This procedure returns the value of the top of the
    queue.

***********************************************************)

    Procedure Examine_Queue( Var Queue: Queue_Type; Var Data );

(***********************************************************

  Function: Is queue empty?

    This function returns true if the queue is empty,
    otherwise it returns false.

***********************************************************)

    Function Queue_Empty( Var Queue: Queue_Type ): Boolean;

(***********************************************************

  Procedure: Get out.

    This procedure removes the top element of the queue and
    returns it.

***********************************************************)

    Procedure Get_Out( Var Queue: Queue_Type; Var Data );

(***********************************************************

  Procedure: Discard queue.

    This procedure clears all the elements off of the queue.

***********************************************************)

    Procedure Discard_Queue( Var Queue: Queue_Type );

(***********************************************************

  Procedure: Create the tree.

    This procedure initializes the tree structure for use.
      the tree must be initialize before it is used.
    The values passed are...
      Tree: This holds all the data needed to manage the
        three structure.
      Total_Data_Size: holds the size of the data to be
        put on the tree.
      Key_Offset: Holds the offset of the data key in the
        data record.
      Key_Length: Holds the length of the data key in the
        data record.

***********************************************************)

    Procedure Create_Tree( Var Tree: Tree_Type; Total_Data_Size, Key_Offset, Key_Length: Word_Type );

(***********************************************************

  Procedure: Destroy the tree.

    Disposes of the entire tree and resets it for
      rebuilding.

***********************************************************)

    Procedure Destroy_Tree( Var Tree: Tree_Type );

(***********************************************************

  Function: Insert in tree.

    This function takes the new data and inserts in into
      the tree in proper order.
    This function will return false if the operation
      couldn't be performed.

***********************************************************)

    Function Insert_In_Tree( Var Tree: Tree_Type; Var New_Data ): Boolean;

(***********************************************************

  Function: Update in tree.

    This function takes a copy of the old data and the new
      data.  The old data is deleted from the tree and the
      new data is inserted.
    This function returns false if the old data can't be
      found, or the new data can't be added.

***********************************************************)

    Function Update_In_Tree( Var Tree: Tree_Type; Var Old_Data, New_Data ): Boolean;

(***********************************************************

  Function: Delete from tree.

    This function deletes the given data from the tree.
    This function returns false if the given data is not
      found in the tree.  The given data must match exactly
      what is found in the tree to be considered valid.

***********************************************************)

    Function Delete_From_Tree( Var Tree: Tree_Type; Var Old_Data ): Boolean;

(***********************************************************

  Function: Find in tree.

    This function tries to find the given key value in the
      data record.  If it finds it, then the whole record
      is returned in data, otherwise it will return false.

***********************************************************)

    Function Find_In_Tree( Var Tree: Tree_Type; Var Data ): Boolean;

(***********************************************************

  Function: Find first in tree.

    This function will look for the first value in the tree
      and return it in data.  It is especially useful for
      creating ordered listing.

***********************************************************)

    Function Find_First_In_Tree( Var Tree: Tree_Type; Var Data ): Boolean;

(***********************************************************

  Function: Find next in tree.

    This function is a complement to the Find_First_In_Tree
      function and will return the next record it finds in
      the tree if it exists.  If not, then it returns false.

***********************************************************)

    Function Find_Next_In_Tree( Var Tree: Tree_Type; Var Data ): Boolean;

(***********************************************************

  Function: Find last in tree.

    This function will look for the last value in the tree
      and return it in data.  It is especially useful for
      creating reverse ordered listing.

***********************************************************)

    Function Find_Last_In_Tree( Var Tree: Tree_Type; Var Data ): Boolean;

(***********************************************************

  Function: Find previous in tree.

    This function is a complement to the Find_Last_In_Tree
      function and will return the previous record it finds
      in the tree if it exists.  If not, then it returns
      false.

***********************************************************)

    Function Find_Previous_In_Tree( Var Tree: Tree_Type; Var Data ): Boolean;

(***********************************************************

  Function: Get absolute address.

    This procedure returns the absolute address of where
      the data is stored in memory.  The data being the
      last accessed piece of information.

    This function is intended for advanced programming only.

***********************************************************)

    Function Get_Absolute_Address( Var Tree: Tree_Type; Var Address: Point_Type ): Boolean;

(***********************************************************

  Function: Read absolute address.

    This procedure returns the data at the absolute address
      as stored in memory.  The data being accessed is with
      the pointer provided from Get_Absolute_Address.

    This function is intended for advanced programming only.

***********************************************************)

    Function Read_Absolute_Address( Var Tree: Tree_Type; Var Address: Point_Type; Var Data ): Boolean;

(***********************************************************

  Procedure: Change key routines.

    This procedure will substitute your own provided
      comparing routines in place of the default routines.

    This function is intended for advanced programming only.

    This function is not supported with Pascal version 4.0.

***********************************************************)

   {$IFNDEF VER40}
    Procedure Change_Key_Routines( Var Tree: Tree_Type; Var Equal, Less_Than: Compare_Function );
   {$ENDIF}

{----------------------------------------------------------------------------}

  Implementation

    Type
      Data_Point = ^Data_Type;   { Pointer used to allocate storage space. }

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Copy data.
    This procedure copies the data in source to
    destination.

*************************************************)

    Procedure Copy_Data( Var Source, Destination; Amount: Word_Type );
      Var
        Source_Array: Data_Type absolute Source;
        Destination_Array: Data_Type absolute Destination;
      Begin
        If ( Amount <> 0 )
          then
            Move( Source_Array, Destination_Array, Amount );
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Write error.
    This procedure checks for an error and writes
    it out if there is one.  Then it halts the
    program.

*************************************************)

    Procedure Write_Error( Result: Word; Sentence: String );
      Begin
        If ( Result <> 0 )
          then
            Begin
              WriteLn( 'Error ', Result, ' in ', Sentence, '.' );
             {$IFNDEF VER40}
              RunError( Result );
             {$else}
              Halt( Result );
             {$ENDIF}
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Allocate storage.
    This procedure allocates the given amount of
    memory on the heap.

*************************************************)

    Procedure Allocate_Storage( Var Location: Pointer; Amount: Word_Type );
      Begin
        If ( MaxAvail < Amount )
          then
            Write_Error( 203, 'Allocate_Storage: Out of memory' );
        GetMem( Location, Amount );
        If ( Location = Nil )
          then
            Write_Error( 203, 'Allocate_Storage: Out of memory' );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Deallocate storage.
    This procedure releases the memory of the
    given pointer back to the heap.

*************************************************)

    Procedure Deallocate_Storage( Var Location: Pointer; Amount: Word_Type );
      Begin
        If ( Location <> Nil )
          then
            Begin
              FreeMem( Location, Amount );
              Location := Nil;
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Create queue.
    As previously defined.

*************************************************)

    Procedure Create_Queue( Var Queue: Queue_Type; Data_Size: Word_Type );
      Begin
        Queue.Top := Nil;
        Queue.Bottom := Nil;
        Queue.Data_Size := Data_Size;
        Queue.Record_Size := ( Data_Size + SizeOf( Structure_Element_Pointer ) );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Put in.
    As previously defined.

*************************************************)

    Procedure Put_In( Var Queue: Queue_Type; Var Data );
      Var
        Holding_Pointer: Structure_Element_Pointer;
      Begin
        If ( ( Queue.Top = Nil ) and ( Queue.Bottom = Nil ) )
          then
            Begin
              Allocate_Storage( Pointer( Queue.Bottom ), Queue.Record_Size );
              Queue.Top := Queue.Bottom;
              Queue.Bottom^.Next := Nil;
              Copy_Data( Data, Queue.Bottom^.Data, Queue.Data_Size );
            End
          else
            Begin
              Holding_Pointer := Queue.Bottom;
              Allocate_Storage( Pointer( Queue.Bottom ), Queue.Record_Size );
              Queue.Bottom^.Next := Nil;
              Holding_Pointer^.Next := Queue.Bottom;
              Copy_Data( Data, Queue.Bottom^.Data, Queue.Data_Size );
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Get out.
    As previously defined.

*************************************************)

    Procedure Get_Out( Var Queue: Queue_Type; Var Data );
      Var
        Holding_Pointer: Structure_Element_Pointer;
      Begin
        If ( Queue.Top <> Nil )
          then
            If ( Queue.Top = Queue.Bottom )
              then
                Begin
                  Copy_Data( Queue.Top^.Data, Data, Queue.Data_Size );
                  Deallocate_Storage( Pointer( Queue.Top ), Queue.Record_Size );
                  Queue.Top := Nil;
                  Queue.Bottom := Nil;
                End
              else
                Begin
                  Copy_Data( Queue.Top^.Data, Data, Queue.Data_Size );
                  Holding_Pointer := Queue.Top^.Next;
                  Deallocate_Storage( Pointer( Queue.Top ), Queue.Record_Size );
                  Queue.Top := Holding_Pointer;
                End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Examine queue.
    As previously defined.

*************************************************)

    Procedure Examine_Queue( Var Queue: Queue_Type; Var Data );
      Begin
        If ( Queue.Top <> Nil )
          then
            Copy_Data( Queue.Top^.Data, Data, Queue.Data_Size );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Is queue empty.
    As previously defined.

*************************************************)

    Function Queue_Empty( Var Queue: Queue_Type ): Boolean;
      Begin
        Queue_Empty := ( Queue.Top = Nil );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Discard queue.
    As previously defined.

*************************************************)

    Procedure Discard_Queue( Var Queue: Queue_Type );
      Var
        Holding_Pointer: Structure_Element_Pointer;
      Begin
        While ( Queue.Top <> Nil ) do
          Begin
            Holding_Pointer := Queue.Top^.Next;
            Deallocate_Storage( Pointer( Queue.Top ), Queue.Record_Size );
            Queue.Top := Holding_Pointer;
          End;
        Queue.Bottom := Nil;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Create stack.
    As previously defined.

*************************************************)

    Procedure Create_Stack( Var Stack: Stack_Type; Data_Size: Word_Type );
      Begin
        If ( Data_Size > Maximum_Record_Size )
          then
            Write_Error( 201, 'Create_Stack: Data size too large for data structure' );
        Stack.Top := Nil;
        Stack.Data_Size := Data_Size;
        Stack.Record_Size := ( Data_Size + Sizeof( Structure_Element_Pointer ) );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Push on.
    As previously defined.

*************************************************)

    Procedure Push_On( Var Stack: Stack_Type; Var Data );
      Var
        Holding_Pointer: Structure_Element_Pointer;
      Begin
        Holding_Pointer := Stack.Top;
        Allocate_Storage( Pointer( Stack.Top ), Stack.Record_Size );
        Copy_Data( Data, Stack.Top^.Data, Stack.Data_Size );
        Stack.Top^.Next := Holding_Pointer;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Pop off.
    As previously defined.

*************************************************)

    Procedure Pop_Off( Var Stack: Stack_Type; Var Data );
      Var
        Holding_Pointer: Structure_Element_Pointer;
      Begin
        If ( Stack.Top <> Nil )
          then
            Begin
              Holding_Pointer := Stack.Top^.Next;
              Copy_Data( Stack.Top^.Data, Data, Stack.Data_Size );
              Deallocate_Storage( Pointer( Stack.Top ), Stack.Record_Size );
              Stack.Top := Holding_Pointer;
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Examine stack.
    As previously defined.

*************************************************)

    Procedure Examine_Stack( Var Stack: Stack_Type; Var Data );
      Begin
        If ( Stack.Top <> Nil )
          then
            Copy_Data( Stack.Top^.Data, Data, Stack.Data_Size );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Discard stack.
    As previously defined.

*************************************************)

    Procedure Discard_Stack( Var Stack: Stack_Type );
      Var
        Holding_Pointer: Structure_Element_Pointer;
      Begin
        While ( Stack.Top <> Nil ) do
          Begin
            Holding_Pointer := Stack.Top^.Next;
            Deallocate_Storage( Pointer( Stack.Top ), Stack.Record_Size );
            Stack.Top := Holding_Pointer;
          End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Is stack empty.
    As previously defined.

*************************************************)

    Function Stack_Empty( Var Stack: Stack_Type ): Boolean;
      Begin
        Stack_Empty := ( Stack.Top = Nil );
      End;

{-----------------------------------------------------------------------------}

(***********************************************************

  Procedure: Get data.
    This procedure gets the data in the node pointed to by
    address and returns the data in the Data variable.

***********************************************************)

    Procedure Get_Data( Var Tree: Tree_Type; Address: Point_Type; Var Data: Data_Type );
      Begin
        If ( Address = Nil )
          then
            Write_Error( 204, 'Get_Data: Invalid address for pointer' )
          else
            Move( Address^.Data, Data, Tree.Data_Size );
      End;

{-----------------------------------------------------------------------------}

(***********************************************************

  Procedure: Put data.
    This procedure replaced the data referred to by address
    with the supplied new data.

***********************************************************)

    Procedure Put_Data( Var Tree: Tree_Type; Address: Point_Type; Data: Data_Type );
      Begin
        If ( Address = Nil )
          then
            Write_Error( 204, 'Put_Data: Invalid address for pointer' )
          else
            Move( Data, Address^.Data, Tree.Data_Size );
      End;

{-----------------------------------------------------------------------------}

(***********************************************************

  Function: Equal default.
    This function returns true if data1's key and data2's
    keys are equal.

***********************************************************)

    Function Equal_Default( Var Data_1, Data_2; Start, Finish: Word_Type ): Boolean;
      Var
        Counter: Word_Type;
        Data1: Data_Type absolute Data_1;
        Data2: Data_Type absolute Data_2;
      Begin
        Counter := Start;
        While ( ( Counter <= Finish ) and ( Data1[ Counter ] = Data2[ Counter ] ) ) do
          Inc( Counter );
        Equal_Default := ( Counter > Finish );
      End;

{-----------------------------------------------------------------------------}

(***********************************************************

  Function: Less than default.
    This function returns true if data1's key is less than
    that of data2's.

***********************************************************)

    Function Less_Than_Default( Var Data_1, Data_2; Start, Finish: Word_Type ): Boolean;
      Var
        Counter: Word_Type;
        Data1: Data_Type absolute Data_1;
        Data2: Data_Type absolute Data_2;
      Begin
        Counter := Start;
        While ( ( Counter <= Finish ) and ( Data1[ Counter ] = Data2[ Counter ] ) ) do
          Inc( Counter );
        Less_Than_Default := ( Counter <= Finish ) and ( Data1[ Counter ] < Data2[ Counter ] );
      End;

{-----------------------------------------------------------------------------}

(***********************************************************

  Procedure: Initialize the tree.
    This procedure initialized the basic tree data
    structure.

***********************************************************)

    Procedure Initialize_Tree( Var Tree: Tree_Type; Total_Record_Size, Total_Data_Size: Word_Type );
      Begin
        With Tree do
          Begin
            Top := Nil;
            Record_Size := Total_Record_Size;
            Data_Size := Total_Data_Size;
            Start := 1;
            Finish := Total_Data_Size;
           {$IFNDEF VER40}
            Equal := Equal_Default;
            Less_Than := Less_Than_Default;
           {$ENDIF}
          End;
      End;

{-----------------------------------------------------------------------------}

(***********************************************************

  Procedure: Create the tree.
    As defined previously.

***********************************************************)

    Procedure Create_Tree( Var Tree: Tree_Type; Total_Data_Size, Key_Offset, Key_Length: Word_Type );
      Var
        Total_Record_Size: Word_Type;
      Begin
        Total_Record_Size := ( SizeOf( Look_Type ) + Total_Data_Size );
        Initialize_Tree( Tree, Total_Record_Size, Total_Data_Size );
        Tree.Start := Key_Offset;
        Tree.Finish := Pred( Key_Length + Key_Offset );
        If ( Tree.Start > Total_Data_Size )
          then
            Write_Error( 201, 'Create_Tree: Key offset beyond record size' )
          else
            If ( Tree.Finish > Total_Data_Size )
              then
                Write_Error( 201, 'Create_Tree: Key size beyond end of record' )
              else
                If ( Tree.Start > Tree.Finish )
                  then
                    Write_Error( 201, 'Create_Tree: Key size invalid' )
                  else
                    If ( Total_Data_Size > Maximum_Record_Size )
                      then
                        Write_Error( 201, 'Create_Tree: Record size too large' );
      End;

{-----------------------------------------------------------------------------}

(***********************************************************

  Procedure: Delete branch.
    Deletes the current branch.

***********************************************************)

    Procedure Delete_Branch( Var Branch: Point_Type; Size: Word_Type );
      Begin
        If ( Branch <> Nil )
          then
            Begin
              With Branch^.Point do
                Begin
                  Delete_Branch( Left_Node_Pointer, Size );
                  Delete_Branch( Right_Node_Pointer, Size );
                End;
              FreeMem( Branch, Size );
              Branch := Nil;
            End;
      End;

{-----------------------------------------------------------------------------}

(***********************************************************

  Procedure: Close the tree.
    As defined previously.

***********************************************************)

    Procedure Destroy_Tree( Var Tree: Tree_Type );
      Begin
        Delete_Branch( Tree.Top, Tree.Record_Size );
        Tree.Top := Nil;
      End;

{-----------------------------------------------------------------------------}

(***********************************************************

  Procedure: Get a new pointer.
    This procedure allocates a new node for the tree.

***********************************************************)

    Procedure New_Pointer( Var Tree: Tree_Type; Var Pointer: Point_Type );
      Var
        Data_Node: Tree_Node;
      Begin
        If ( MaxAvail >= SizeOf( Tree_Node ) )
          then
            Begin
              GetMem( Pointer, Tree.Record_Size );
              Move( Data_Node, Pointer^, Tree.Record_Size )
            End
          else
            Write_Error( 203, 'New_Pointer: Out of memory' )
      End;

{-----------------------------------------------------------------------------}

(***********************************************************

  Procedure: Dispose the pointer.
    This procedure is designed to dispose of an unused node
    pointer;

***********************************************************)

    Procedure Dispose_Pointer( Var Tree: Tree_Type; Var Pointer: Point_Type );
      Begin
        FreeMem( Pointer, Tree.Record_Size );
        Pointer := Nil;
      End;

{-----------------------------------------------------------------------------}

(***********************************************************

  Function: Height.
    This function returns the height of the given branch.

***********************************************************)

    Function Height( Branch: Point_Type ): Integer;
      Var
        The_Height,
        Left_Height,
        Right_Height,
        First_Height: Integer;
      Begin
        If ( Branch = Nil )
          then
            The_Height := 0
          else
            Begin
              First_Height := Branch^.Point.Height;
              If ( First_Height = 0 )
                then
                  Begin
                    Right_Height := Height( Branch^.Point.Right_Node_Pointer );
                    Left_Height := Height( Branch^.Point.Left_Node_Pointer );
                    If ( Right_Height > Left_Height )
                      then
                        The_Height := Succ( Right_Height )
                      else
                        The_Height := Succ( Left_Height );
                    Branch^.Point.Height := The_Height;
                  End
                else
                  The_Height := First_Height;
            End;
        Height := The_Height;
      End;

{-----------------------------------------------------------------------------}

(***********************************************************

  Function: Balanced.
    This function returns a value showing how off balance
    the particular branch is.  Way shows which direction
    the branch is off balance.

***********************************************************)

    Function Balanced( Var Branch: Point_Type; Var Way: Boolean ): Integer;
      Var
        Left_Height,
        Right_Height: Integer;
      Begin
        If ( Branch = Nil )
          then
            Balanced := 0
          else
            Begin
              Right_Height := Height( Branch^.Point.Right_Node_Pointer );
              Left_Height := Height( Branch^.Point.Left_Node_Pointer );
              Balanced := ( Abs( Left_Height - Right_Height ) );
              Way := ( Left_Height > Right_Height );
            End;
      End;

{-----------------------------------------------------------------------------}

(***********************************************************

  Function: Find way.
    This function returns true if the left branch is heavier
    then the right branch.

***********************************************************)

    Function Find_Way( Var Branch: Point_Type ): Boolean;
      Var
        Left_Height,
        Right_Height: Integer;
      Begin
        Right_Height := Height( Branch^.Point.Right_Node_Pointer );
        Left_Height := Height( Branch^.Point.Left_Node_Pointer );
        Find_Way := ( Left_Height > Right_Height );
      End;

{-----------------------------------------------------------------------------}

(***********************************************************

  Procedure: Rearrange 1.
    This procedure rearranges the branch according to case
    1.

***********************************************************)

    Procedure Rearrange_1( Var Branch, Pointer_1, Pointer_2, Pointer_3: Point_Type );
      Var
        Hold: Point_Type;
      Begin
        Hold := Pointer_2^.Point.Right_Node_Pointer;
        Branch := Pointer_2;
        Pointer_2^.Point.Previous_Node_Pointer := Pointer_1^.Point.Previous_Node_Pointer;
        Pointer_2^.Point.Right_Node_Pointer := Pointer_1;
        Pointer_2^.Point.Left_Node_Pointer := Pointer_3;
        Pointer_1^.Point.Previous_Node_Pointer := Pointer_2;
        Pointer_1^.Point.Left_Node_Pointer := Hold;
        Pointer_3^.Point.Previous_Node_Pointer := Pointer_2;
        If ( Hold <> Nil )
          then
            Hold^.Point.Previous_Node_Pointer := Pointer_1;
      End;

{-----------------------------------------------------------------------------}

(***********************************************************

  Procedure: Rearrange 2.
    This procedure rearranges the branch according to case
    2.

***********************************************************)

    Procedure Rearrange_2( Var Branch, Pointer_1, Pointer_2, Pointer_3: Point_Type );
      Var
        Hold_1,
        Hold_2: Point_Type;
      Begin
        Hold_1 := Pointer_3^.Point.Left_Node_Pointer;
        Hold_2 := Pointer_3^.Point.Right_Node_Pointer;
        Branch  := Pointer_3;
        Pointer_3^.Point.Previous_Node_Pointer := Pointer_1^.Point.Previous_Node_Pointer;
        Pointer_3^.Point.Right_Node_Pointer := Pointer_1;
        Pointer_3^.Point.Left_Node_Pointer := Pointer_2;
        Pointer_1^.Point.Previous_Node_Pointer := Pointer_3;
        Pointer_1^.Point.Left_Node_Pointer := Hold_2;
        Pointer_2^.Point.Previous_Node_Pointer := Pointer_3;
        Pointer_2^.Point.Right_Node_Pointer := Hold_1;
        If ( Hold_1 <> Nil )
          then
            Hold_1^.Point.Previous_Node_Pointer := Pointer_2;
        If ( Hold_2 <> Nil )
          then
            Hold_2^.Point.Previous_Node_Pointer := Pointer_1;
      End;

{-----------------------------------------------------------------------------}

(***********************************************************

  Procedure: Rearrange 3.
    This procedure rearranges the branch according to case
    3.

***********************************************************)

    Procedure Rearrange_3( Var Branch, Pointer_1, Pointer_2, Pointer_3: Point_Type );
      Var
        Hold: Point_Type;
      Begin
        Hold := Pointer_2^.Point.Left_Node_Pointer;
        Branch  := Pointer_2;
        Pointer_2^.Point.Previous_Node_Pointer := Pointer_1^.Point.Previous_Node_Pointer;
        Pointer_2^.Point.Left_Node_Pointer := Pointer_1;
        Pointer_2^.Point.Right_Node_Pointer := Pointer_3;
        Pointer_1^.Point.Previous_Node_Pointer := Pointer_2;
        Pointer_1^.Point.Right_Node_Pointer := Hold;
        Pointer_3^.Point.Previous_Node_Pointer := Pointer_2;
        If ( Hold <> Nil )
          then
            Hold^.Point.Previous_Node_Pointer := Pointer_1;
      End;

{-----------------------------------------------------------------------------}

(***********************************************************

  Procedure: Rearrange 4.
    This procedure rearranges the branch according to case
    4.

***********************************************************)

    Procedure Rearrange_4( Var Branch, Pointer_1, Pointer_2, Pointer_3: Point_Type );
      Var
        Hold_1,
        Hold_2: Point_Type;
      Begin
        Hold_1 := Pointer_3^.Point.Left_Node_Pointer;
        Hold_2 := Pointer_3^.Point.Right_Node_Pointer;
        Branch  := Pointer_3;
        Pointer_3^.Point.Previous_Node_Pointer := Pointer_1^.Point.Previous_Node_Pointer;
        Pointer_3^.Point.Left_Node_Pointer := Pointer_1;
        Pointer_3^.Point.Right_Node_Pointer := Pointer_2;
        Pointer_1^.Point.Previous_Node_Pointer := Pointer_3;
        Pointer_1^.Point.Right_Node_Pointer := Hold_1;
        Pointer_2^.Point.Previous_Node_Pointer := Pointer_3;
        Pointer_2^.Point.Left_Node_Pointer := Hold_2;
        If ( Hold_1 <> Nil )
          then
            Hold_1^.Point.Previous_Node_Pointer := Pointer_1;
        If ( Hold_2 <> Nil )
          then
            Hold_2^.Point.Previous_Node_Pointer := Pointer_2;
      End;

{-----------------------------------------------------------------------------}

(***********************************************************

  Procedure: Adjust.
    This procedure is designed to correct any imbalance in
    the particular branch of the tree.

***********************************************************)

    Procedure Adjust( Var Branch: Point_Type );
      Var
        Pointer_1,
        Pointer_2,
        Pointer_3: Point_Type;
        First_Way: Boolean;
      Begin
        If ( ( Prime_Factor >= Prime_Factor_Default ) and ( Balanced( Branch, First_Way ) > Prime_Factor ) )
          then
            Begin
              Pointer_1 := Branch;
              If First_Way
                then
                  Pointer_2 := Pointer_1^.Point.Left_Node_Pointer
                else
                  Pointer_2 := Pointer_1^.Point.Right_Node_Pointer;
              If ( Pointer_2 = Nil )
                then
                  Begin
                    WriteLn( 'Fatal error with Pointer_2!' );
                    WriteLn( Height( Pointer_1 ), ' ', Height( Pointer_2 ) );
                    Halt;
                  End;
              If Find_Way( Pointer_2 )
                then
                  Pointer_3 := Pointer_2^.Point.Left_Node_Pointer
                else
                  Pointer_3 := Pointer_2^.Point.Right_Node_Pointer;
              If ( Pointer_3 = Nil )
                then
                  Begin
                    WriteLn( 'Fatal error With Pointer_3!' );
                    WriteLn( Height( Pointer_1 ), ' ', Height( Pointer_2 ), ' ', Height( Pointer_3 ) );
                    Halt;
                  End;
              Pointer_1^.Point.Height := 0;
              Pointer_2^.Point.Height := 0;
              Pointer_3^.Point.Height := 0;
              If ( Pointer_1^.Point.Left_Node_Pointer = Pointer_2 )
                then
                  If ( Pointer_2^.Point.Left_Node_Pointer = Pointer_3 )
                    then
                      Rearrange_1( Branch, Pointer_1, Pointer_2, Pointer_3 )
                    else
                      Rearrange_2( Branch, Pointer_1, Pointer_2, Pointer_3 )
                else
                  If ( Pointer_2^.Point.Right_Node_Pointer = Pointer_3 )
                    then
                      Rearrange_3( Branch, Pointer_1, Pointer_2, Pointer_3 )
                    else
                      Rearrange_4( Branch, Pointer_1, Pointer_2, Pointer_3 );
            End;
      End;

{-----------------------------------------------------------------------------}

(***********************************************************

  Function: Add to branch.
    This function takes care of adding new data to a
    particular branch of the tree.  It determines which side
    of the tree the new data will go and then calls itself
    recursively to add it to that sub branch.

***********************************************************)

    Function Add_To_Branch( Var Tree: Tree_Type; Var Branch: Point_Type; Previous: Point_Type;
                            Var New_Data: Data_Type ): Boolean;
      Var
        Old_Data: Data_Point;
        Temporary: Point_Type;
       {$IFDEF OS2}
        Hold: Boolean;
       {$ENDIF}
      Begin
        If ( Branch <> Nil )
          then
            Begin
              GetMem( Old_Data, Tree.Data_Size );
              Branch^.Point.Height := 0;
              Get_Data( Tree, Branch, Old_Data^ );
             {$IFDEF OS2}
              Hold := Tree.Less_Than( New_Data, Old_Data^, Tree.Start, Tree.Finish );
              If Hold
             {$ELSE}
             {$IFNDEF VER40}
              If Tree.Less_Than( New_Data, Old_Data^, Tree.Start, Tree.Finish )
             {$ELSE}
              If Less_Than_Default( New_Data, Old_Data^, Tree.Start, Tree.Finish )
             {$ENDIF}
             {$ENDIF}
                then
                  Begin
                    Temporary := Branch^.Point.Left_Node_Pointer;
                    If Add_To_Branch( Tree, Temporary, Branch, New_Data )
                      then
                        Begin
                          Branch^.Point.Left_Node_Pointer := Temporary;
                          Add_To_Branch := True;
                        End
                      else
                        Add_To_Branch := False;
                  End
                else
                  Begin
                    Temporary := Branch^.Point.Right_Node_Pointer;
                    If Add_To_Branch( Tree, Temporary, Branch, New_Data )
                      then
                        Begin
                          Branch^.Point.Right_Node_Pointer := Temporary;
                          Add_To_Branch := True;
                        End
                      else
                        Add_To_Branch := False;
                  End;
              Branch^.Point.Height := Height( Branch );
              Adjust( Branch );
              FreeMem( Old_Data, Tree.Data_Size );
            End
          else
            Begin
              New_Pointer( Tree, Branch );
              Put_Data( Tree, Branch, New_Data );
              Branch^.Point.Right_Node_Pointer := Nil;
              Branch^.Point.Left_Node_Pointer := Nil;
              Branch^.Point.Previous_Node_Pointer := Previous;
              Branch^.Point.Height := 1;
              Add_To_Branch := True;
              Tree.Where := Branch;
            End;
      End;

{-----------------------------------------------------------------------------}

(***********************************************************

  Procedure: Delete from branch.
    This procedure takes care of deleting old data from
    the given branch.  It determines which side of the tree
    the branch is on and may call itself recursively.

***********************************************************)

    Procedure Delete_From_Branch( Var Tree: Tree_Type; Var Branch: Point_Type; Var Data: Data_Type; Var Result: Boolean );
      Var
        Old_Data:  Data_Point;
        Pointer_1,
        Pointer_2: Point_Type;
       {$IFDEF OS2}
        Hold: Boolean;
       {$ENDIF}
      Begin
        If ( Branch <> Nil )
          then
            Begin
              GetMem( Old_Data, Tree.Data_Size );
              Get_Data( Tree, Branch, Old_Data^ );
             {$IFDEF OS2}
              Hold := Tree.Equal( Old_Data^, Data, 1, Tree.Data_Size );
              If Hold
             {$ELSE}
             {$IFNDEF VER40}
              If Tree.Equal( Old_Data^, Data, 1, Tree.Data_Size )
             {$ELSE}
              If Equal_Default( Old_Data^, Data, 1, Tree.Data_Size )
             {$ENDIF}
             {$ENDIF}
                then
                  Begin
                    If ( Branch^.Point.Left_Node_Pointer = Nil )
                      then
                        If ( Branch^.Point.Right_Node_Pointer = Nil )
                          then
                            Begin
                              Dispose_Pointer( Tree, Branch );
                              Branch := Nil;
                              Result := True;
                            End
                          else
                            Begin
                              Pointer_1 := Branch;
                              Branch := Branch^.Point.Right_Node_Pointer;
                              If ( Branch <> Nil )
                                then
                                  Branch^.Point.Previous_Node_Pointer := Pointer_1^.Point.Previous_Node_Pointer;
                              Dispose_Pointer( Tree, Pointer_1 );
                              Result := True;
                            End
                      else
                        If ( Branch^.Point.Right_Node_Pointer = Nil )
                          then
                            Begin
                              Pointer_1 := Branch;
                              Branch := Branch^.Point.Left_Node_Pointer;
                              If ( Branch <> Nil )
                                then
                                  Branch^.Point.Previous_Node_Pointer := Pointer_1^.Point.Previous_Node_Pointer;
                              Dispose_Pointer( Tree, Pointer_1 );
                              Result := True;
                            End
                          else
                            Begin
                              Pointer_2 := Branch;
                              Pointer_1 := Branch^.Point.Right_Node_Pointer;
                              While ( ( Pointer_1 <> Nil ) and ( Pointer_1^.Point.Left_Node_Pointer <> Nil ) ) do
                                Begin
                                  Pointer_2 := Pointer_1;
                                  Pointer_1 := Pointer_1^.Point.Left_Node_Pointer;
                                End;
                              Get_Data( Tree, Pointer_1, Old_Data^ );
                              Delete_From_Branch( Tree, Pointer_1, Old_Data^, Result );
                              If Result
                                then
                                  Begin
                                    Put_Data( Tree, Branch, Old_Data^ );
                                    If ( Pointer_2 = Branch )
                                      then
                                        Pointer_2^.Point.Right_Node_Pointer := Pointer_1
                                      else
                                        Pointer_2^.Point.Left_Node_Pointer := Pointer_1;
                                  End;
                            End;
                    If ( Branch <> Nil )
                      then
                        Branch^.Point.Height := 0;
                  End
                else
                  Begin
                   {$IFDEF OS2}
                    Hold := Tree.Less_Than( Data, Old_Data^, Tree.Start, Tree.Finish );
                    If Hold
                   {$ELSE}
                   {$IFNDEF VER40}
                    If Tree.Less_Than( Data, Old_Data^, Tree.Start, Tree.Finish )
                   {$ELSE}
                    If Less_Than_Default( Data, Old_Data^, Tree.Start, Tree.Finish )
                   {$ENDIF}
                   {$ENDIF}
                      then
                        Begin
                          Pointer_1 := Branch^.Point.Left_Node_Pointer;
                          Delete_From_Branch( Tree, Pointer_1, Data, Result );
                          Branch^.Point.Left_Node_Pointer := Pointer_1;
                          Branch^.Point.Height := 0;
                        End
                      else
                        Begin
                          Pointer_1 := Branch^.Point.Right_Node_Pointer;
                          Delete_From_Branch( Tree, Pointer_1, Data, Result );
                          Branch^.Point.Right_Node_Pointer := Pointer_1;
                          Branch^.Point.Height := 0;
                        End;
                  End;
              FreeMem( Old_Data, Tree.Data_Size );
            End
          else
            Result := False;
        If ( Branch <> Nil )
          then
            Begin
              Branch^.Point.Height := 0;
              Adjust( Branch );
            End;
      End;

{-----------------------------------------------------------------------------}

(***********************************************************

  Procedure: Find in branch.
    This procedure searches the branches for the data
    containing the specified key.

***********************************************************)

    Function Find_In_Branch( Var Tree: Tree_Type; Branch: Point_Type; Var Data: Data_Type; Var Where: Point_Type ): Boolean;
      Var
        Old_Data: Data_Point;
       {$IFDEF OS2}
        Hold,
       {$ENDIF}
        Not_Found: Boolean;
      Begin
        GetMem( Old_Data, Tree.Data_Size );
        Not_Found := True;
        While ( ( Branch <> Nil ) and Not_Found ) do
          Begin
            Where := Branch;
            Get_Data( Tree, Branch, Old_Data^ );
           {$IFDEF OS2}
            Hold := Tree.Equal( Data, Old_Data^, Tree.Start, Tree.Finish );
            If Hold
           {$ELSE}
           {$IFNDEF VER40}
            If Tree.Equal( Data, Old_Data^, Tree.Start, Tree.Finish )
           {$ELSE}
            If Equal_Default( Data, Old_Data^, Tree.Start, Tree.Finish )
           {$ENDIF}
           {$ENDIF}
              then
                Begin
                  Move( Old_Data^, Data, Tree.Data_Size );
                  Not_Found := False;
                End
              else
               {$IFDEF OS2}
                Hold := Tree.Less_Than( Data, Old_Data^, Tree.Start, Tree.Finish );
                If Hold
               {$ELSE}
               {$IFNDEF VER40}
                If Tree.Less_Than( Data, Old_Data^, Tree.Start, Tree.Finish )
               {$ELSE}
                If Less_Than_Default( Data, Old_Data^, Tree.Start, Tree.Finish )
               {$ENDIF}
               {$ENDIF}
                  then
                    Branch := Branch^.Point.Left_Node_Pointer
                  else
                    Branch := Branch^.Point.Right_Node_Pointer;
          End;
        Find_In_Branch := ( not Not_Found );
        FreeMem( Old_Data, Tree.Data_Size );
      End;

{-----------------------------------------------------------------------------}

(***********************************************************

  Function: Find In Tree.
    As defined previously.

***********************************************************)

    Function Find_In_Tree( Var Tree: Tree_Type; Var Data ): Boolean;
      Var
        Data_Storage: Data_Type absolute Data;
      Begin
        Tree.Where := Nil;
        Find_In_Tree := Find_In_Branch( Tree, Tree.Top, Data_Storage, Tree.Where )
      End;

{-----------------------------------------------------------------------------}

(***********************************************************

  Function: Find next in tree.
    As defined previously.

***********************************************************)

    Function Find_Next_In_Tree( Var Tree: Tree_Type; Var Data ): Boolean;
      Var
        Data_Storage: Data_Type absolute Data;
      Begin
        If ( Tree.Where <> Nil )
          then
            Begin
              If ( Tree.Where^.Point.Right_Node_Pointer <> Nil )
                then
                  Begin
                    Tree.Where := Tree.Where^.Point.Right_Node_Pointer;
                    While ( ( Tree.Where <> Nil ) and ( Tree.Where^.Point.Left_Node_Pointer <> Nil ) ) do
                      Tree.Where := Tree.Where^.Point.Left_Node_Pointer;
                    Get_Data( Tree, Tree.Where, Data_Storage );
                    Find_Next_In_Tree := True;
                  End
                else
                  Begin
                    While ( Tree.Where <> Nil ) and ( Tree.Where^.Point.Previous_Node_Pointer <> Nil ) and
                          ( Tree.Where^.Point.Previous_Node_Pointer^.Point.Right_Node_Pointer = Tree.Where ) do
                      Tree.Where := Tree.Where^.Point.Previous_Node_Pointer;
                    If ( Tree.Where <> Nil )
                      then
                        Begin
                          If ( Tree.Where^.Point.Previous_Node_Pointer <> Nil )
                            then
                              Begin
                                Tree.Where := Tree.Where^.Point.Previous_Node_Pointer;
                                If ( Tree.Where <> Nil )
                                  then
                                    Begin
                                      Get_Data( Tree, Tree.Where, Data_Storage );
                                      Find_Next_In_Tree := True;
                                    End
                                  else
                                    Find_Next_In_Tree := False;
                              End
                            else
                              Find_Next_In_Tree := False;
                        End
                      else
                        Find_Next_In_Tree := False;
                  End;
            End;
      End;

{-----------------------------------------------------------------------------}

(***********************************************************

  Function: Find previous in tree.
    As defined previously.

***********************************************************)

    Function Find_Previous_In_Tree( Var Tree: Tree_Type; Var Data ): Boolean;
      Var
        Data_Storage: Data_Type absolute Data;
      Begin
        If ( Tree.Where <> Nil )
          then
            Begin
              If ( Tree.Where^.Point.Left_Node_Pointer <> Nil )
                then
                  Begin
                    Tree.Where := Tree.Where^.Point.Left_Node_Pointer;
                    While ( ( Tree.Where <> Nil ) and ( Tree.Where^.Point.Right_Node_Pointer <> Nil ) ) do
                      Tree.Where := Tree.Where^.Point.Right_Node_Pointer;
                    Get_Data( Tree, Tree.Where, Data_Storage );
                    Find_Previous_In_Tree := True;
                  End
                else
                  Begin
                    While ( ( Tree.Where <> Nil ) and ( Tree.Where^.Point.Previous_Node_Pointer <> Nil ) and
                            ( Tree.Where^.Point.Previous_Node_Pointer^.Point.Left_Node_Pointer = Tree.Where ) ) do
                      Tree.Where := Tree.Where^.Point.Previous_Node_Pointer;
                    If ( Tree.Where <> Nil )
                      then
                        Begin
                          If ( Tree.Where^.Point.Previous_Node_Pointer <> Nil )
                            then
                              Begin
                                Tree.Where := Tree.Where^.Point.Previous_Node_Pointer;
                                If ( Tree.Where <> Nil )
                                  then
                                    Begin
                                      Get_Data( Tree, Tree.Where, Data_Storage );
                                      Find_Previous_In_Tree := True;
                                    End
                                  else
                                    Find_Previous_In_Tree := False;
                              End
                            else
                              Find_Previous_In_Tree := False;
                        End
                      else
                        Find_Previous_In_Tree := False;
                  End;
            End;
      End;

{-----------------------------------------------------------------------------}

(***********************************************************

  Function: Find first in tree.
    As defined previously.

***********************************************************)

    Function Find_First_In_Tree( Var Tree: Tree_Type; Var Data ): Boolean;
      Var
        Data_Storage: Data_Type absolute Data;
      Begin
        Tree.Where := Tree.Top;
        If ( Tree.Where <> Nil )
          then
            Begin
              While ( Tree.Where^.Point.Left_Node_Pointer <> Nil ) do
                Tree.Where := Tree.Where^.Point.Left_Node_Pointer;
              Get_Data( Tree, Tree.Where, Data_Storage );
              Find_First_In_Tree := True;
            End
          else
            Find_First_In_Tree := False;
      End;

{-----------------------------------------------------------------------------}

(***********************************************************

  Function: Find last in tree.
    As defined previously.

***********************************************************)

    Function Find_Last_In_Tree( Var Tree: Tree_Type; Var Data ): Boolean;
      Var
        Data_Storage: Data_Type absolute Data;
      Begin
        Tree.Where := Tree.Top;
        If ( Tree.Where <> Nil )
          then
            Begin
              While ( Tree.Where^.Point.Right_Node_Pointer <> Nil ) do
                Tree.Where := Tree.Where^.Point.Right_Node_Pointer;
              Get_Data( Tree, Tree.Where, Data_Storage );
              Find_Last_In_Tree := True;
            End
          else
            Find_Last_In_Tree := False;
      End;

{-----------------------------------------------------------------------------}

(***********************************************************

  Function: Update in tree.
    As defined previously.

***********************************************************)

    Function Update_In_Tree( Var Tree: Tree_Type; Var Old_Data, New_Data ): Boolean;
      Var
       {$IFDEF OS2}
        Hold,
       {$ENDIF}
        Result: Boolean;
        The_Old_Data: Data_Type absolute Old_Data;
        The_New_Data: Data_Type absolute New_Data;
      Begin
       {$IFDEF OS2}
        Hold := Tree.Equal( The_Old_Data, The_New_Data, Tree.Start, Tree.Finish );
        If Hold
       {$ELSE}
       {$IFNDEF VER40}
        If Tree.Equal( The_Old_Data, The_New_Data, Tree.Start, Tree.Finish )
       {$ELSE}
        If Equal_Default( The_Old_Data, The_New_Data, Tree.Start, Tree.Finish )
       {$ENDIF}
       {$ENDIF}
          then
            Begin
              If Find_In_Branch( Tree, Tree.Top, The_Old_Data, Tree.Where )
                then
                  Begin
                    Put_Data( Tree, Tree.Where, The_New_Data );
                    Update_In_Tree := True;
                  End
                else
                  Update_In_Tree := False;
            End
          else
            Begin
              Delete_From_Branch( Tree, Tree.Top, The_Old_Data, Result );
              If Result
                then
                  Update_In_Tree := Add_To_Branch( Tree, Tree.Top, Nil, The_New_Data )
                else
                  Update_In_Tree := False;
            End;
      End;

{-----------------------------------------------------------------------------}

(***********************************************************

  Function: Delete from tree.
    As defined previously.

***********************************************************)

    Function Delete_From_Tree( Var Tree: Tree_Type; Var Old_Data ): Boolean;
      Var
        Result: Boolean;
        Data_Storage: Data_Type absolute Old_Data;
      Begin
        Delete_From_Branch( Tree, Tree.Top, Data_Storage, Result );
        Delete_From_Tree := Result;
      End;

{-----------------------------------------------------------------------------}

(***********************************************************

  Function: Insert in tree.
    As defined previously.

***********************************************************)

    Function Insert_In_Tree( Var Tree: Tree_Type; Var New_Data ): Boolean;
      Var
        Data_Storage: Data_Type absolute New_Data;
      Begin
        Insert_In_Tree := Add_To_Branch( Tree, Tree.Top, Nil, Data_Storage );
      End;

{-----------------------------------------------------------------------------}

(***********************************************************

  Function Get absolute address.
    As defined previously.

***********************************************************)

    Function Get_Absolute_Address( Var Tree: Tree_Type; Var Address: Point_Type ): Boolean;
      Begin
        If ( ( Tree.Where <> Nil ) and ( Tree.Where^.Point.Height >= 0 ) )
          then
            Begin
              Address := Tree.Where;
              Get_Absolute_Address := True;
            End
          else
            Begin
              Address := Nil;
              Get_Absolute_Address := False;
            End
      End;

{-----------------------------------------------------------------------------}

(***********************************************************

  Function Read absolute address.
    As defined previously.

***********************************************************)

    Function Read_Absolute_Address( Var Tree: Tree_Type; Var Address: Point_Type; Var Data ): Boolean;
      Var
        Data_Storage: Data_Type absolute Data;
      Begin
        If ( Address <> Nil )
          then
            Begin
              If ( Address^.Point.Height < 0 )
                then
                  Begin
                    Tree.Where := Address;
                    Get_Data( Tree, Address, Data_Storage );
                    Read_Absolute_Address := True;
                  End
                else
                  Read_Absolute_Address := False;
            End
          else
            Read_Absolute_Address := False;
      End;

{-----------------------------------------------------------------------------}

(***********************************************************

  Procedure Change key routines.
    As previously defined.

***********************************************************)

   {$IFNDEF VER40}
    Procedure Change_Key_Routines( Var Tree: Tree_Type; Var Equal, Less_Than: Compare_Function );
      Begin
        Tree.Equal := Equal;
        Tree.Less_Than := Less_Than;
      End;
   {$ENDIF}

{----------------------------------------------------------------------------}

  End.

