unit PrintUnt;

{@ Generic Simple printing object }

{

	******************************************************************
	*                                                                *
	*     This is a Generic Form printing object for TP7/BP7.0       *
	*                                                                *
	* The printer settings are configured using printer definition   *
	*	files. These are plain ASCII files which you can easily adapt  *
	*	for your own printer.                                          *
	*	Look at EXAMPLE.FMP for help on the ini file.                  *
	*                                                                *
	*	Written by Fran Moerel & donated to the public domain.         *
	*	If you have any problems or suggestions please contact me      *
	*	on:                                                            *
	*                                                                *
	*		BBS Source Paradise +31-15-621077                            *
	*                                                                *
	*		FidoNet             2:281/912                                *
	*                                                                *
	*		PascalNet           115:3113/0                               *
	*                                                                *
	*		Internet            moerel@nni.nl                            *
	*                                                                *
	*		SnailMail           Fran Moerel                              *
	*												Diepenbrockstraat 210                    *
	*												2625XB DELFT                             *
	*												The Netherlands                          *
	*                                                                *
	******************************************************************

	How to use this unit:


	Initialising & disposing the object
	-----------------------------------

	Init(strIniFileName', bytPrintMode);

			bytPrintMode = BPrintPreview  -> On screen only
			bytPrintMode = BPrintNormal   -> On printer only
			bytPrintMode = BPrintBoth     -> On screen & printer

	Don't forget to dispose the object


	Initialising the printing of an object
	--------------------------------------

	You can set a Header and a Footer on the document using:

		SetPageTitle('This is a test file printed on ');
		SetHeader('Name | Adress       | City',
							'-----+--------------+------------');
		SetFooter('Printed on  ');

	In the Page Title, header & the footer strings you can use several 
	control codes.

	These codes are:

		Ctrl-A () -> Insert page number
		Ctrl-B () -> Insert date string
		Ctrl-C () -> Insert time string

	If you don't give a header or footer, they will not be printed.

	Before you start printing you will now have to Initiliase the printing
	using:

		PrinterInit;

	When you are at the end of your document use:

		PrinterDone;


	Printing text
	-------------

	You can print a text using

		Print('This is a text');
		PrintLn('This is a text with Carrier Return');

	If you need some basic formatting you can use the following functions :

		Left('String to Leftalign', bytLength) : string;
		Center('String to center', bytLength) : string;
		Right('String to Rightalign', bytLength) : string;

		if bytLength = 0 then the line will be aligned using the entire line length

	If you need to print numbers or packed dates you can use:

		NumStr(linToConvert) : string;
		DatePackStr(linToConvert) : string;

		example:

		Print(Left('This is test number ' + NumStr(intNumber) + ' on ' +
							 DatePackStr(linPackedDate) + '.'), 10);


	Styles
	------

	You switch styles using the following commands:

		Normal;      -> Switch to normal
		Bold;        -> Switch to bold
		Italic;      -> Switch to italic
		BoldItalic;  -> Switch to bold & italic


	Misc. commands
	--------------

	The following commands can also be used:

		NewLine ;   -> Goto the next line
		FormFeed;  -> Goto the next page

		Fill(chrToFill, bytLength) : string;
							 -> Returns a string of bytLength characters chrToFill

		LinesLeft : integer;   -> Returns the number of lines left on the page
		CurrentLine : integer; -> Returns the current Line number
		CurrentPage : integer; -> Returns the current Page number
		PageWidth : integer;   -> Returns the width of the page in chars
		PageLength : integer;  -> Returns the length of the page in lines


	IMPORTANT
	---------

	Only use provided commands for NewLine & FormFeed, because otherwise
	the unit will lose track of the line & page numbers.

	You will not have to keep track of page & line numbers, because all
	is handled internally in the object.
}


interface


uses Toolkit, IniUnit;

{ Units
	-----

	Toolkit  : Collection of handy tools - Not all the functions are used
						 in the printer objec, but perhaps there are some handy routines
						 in there for you.
	IniUnit  : An Microsoft(R) style .INI file object
	
}


const
	BPrintPreview = 1;
	BPrintNormal  = 2;
	BPrintBoth    = 3;

	{ The character used on the Footer separation line }
	CLine = '';


var
	strFF : string;
	strLF : string;
	strCR : string;

type
	
	PPrintUnit = ^TPrintUnit;
	TPrintUnit = object

	private

		{ Printer vars }
		strPrinterPort   : string;

		{ Texts to use }
		strPageTitle     : string;
		strHeader1       : string;
		strHeader2       : string;
		strFooter        : string;
		strBannerFile    : string;

		{ Page sizes }
		strMarginLeft    : string;
		intPageLength    : integer;
		intPageWidth     : integer;
		intMarginTop     : integer;
		intMarginBottom  : integer;
		intMarginLeft    : integer;
		intMarginRight   : integer;

		{ Printer codes }
		strNormal        : string;
		strItalic        : string;
		strBold          : string;
		strBoldItalic    : string;
		strPrinterInit   : string;
		strPrinterDone   : string;

		{ Printer file }
		filPrinter       : text;

		{ Positioning vars }
		intCurrLine      : integer;
		intPageEnd       : integer;
		intLineWidth     : integer;
		intCurrPage      : integer;

		{ Misc. }
		bytPrintMode     : byte;
		strLastMode      : string;
		bolNewPage       : boolean;

		{ Internal functions & procedures }
		procedure 	Tokenise(var strToDo : string);
		procedure 	PrintRaw(strToPrint : string; bolToScreen : boolean);
		function 		ControlCodes(strToDo : string) : string;
		procedure 	CR;
		procedure 	PrintHeader;

	public

		{ Object constructor & destructor }
		constructor Init(strPrintIniFile : string; bytNewPrintMode : byte);
		destructor  Done;

		{ Document Initialisation commands }
		procedure   SetPageTitle(strNewPageTitle : string);
		procedure   SetHeader(strNewHeader1, strNewHeader2 : string);
		procedure   SetFooter(strNewFooter : string);
		procedure   SetBannerFile(strNewBannerFile : string);
		procedure   PrinterInit;
		procedure   PrinterDone;

		{ Printing commands }
		procedure 	Print(strToPrint : string);
		procedure 	PrintLn(strToPrint : string);

		{ Style switch commands }
		procedure   Bold;
		procedure   Normal;
		procedure   Italic;
		procedure   BoldItalic;

		{ Positioning commands }
		procedure   FormFeed;
		procedure 	NewLine;

		{ Alignment commands }
		function    Center(strToCenter : string; bytLength : byte) : string;
		function    Right(strToRight : string; bytLength : byte) : string;
		function 		Left(strToLeft : string; bytLength : byte) : string;
		function 		Fill(chrToFill : char; bytLength : byte) : string;

		{ Position information }
		function    LinesLeft : integer;
		function    CurrentLine : integer;
		function    CurrentPage : integer;
		function    PageWidth : integer;
		function    PageLength : integer;

		{ Conversion commands }
		function    NumStr(linToConvert : longint) : string;
		function    DatePackStr(linToConvert : longint) : string;

	end;


implementation


constructor TPrintUnit.Init(strPrintIniFile : string; bytNewPrintMode : byte);
{ Init the printer object & get all info out of Ini file }

var
	Ini           : PIni;
	strTmp        : string;
	intCode       : integer;

begin

	bytPrintMode  := bytNewPrintMode;
	strPageTitle  := '';
	strHeader1    := '';
	strHeader2    := '';
	strFooter     := '';
	strBannerFile := '';

	{ Get Ini file info }
	Ini := New(PIni, Init(strPrintIniFile));

	{ Get printer ini info }
	strPrinterPort := Ini^.GetItem('Printer','Port','LPT1');
	Val(Ini^.GetItem('PrintSize','PageLength','60'), intPageLength, intCode); 
	Val(Ini^.GetItem('PrintSize','PageWidth','94'), intPageWidth, intCode); 
	Val(Ini^.GetItem('PrintSize','MarginLeft','4'), intMarginLeft, intCode); 
	Val(Ini^.GetItem('PrintSize','MarginRight','4'), intMarginRight, intCode); 
	Val(Ini^.GetItem('PrintSize','MarginTop','4'), intMarginTop, intCode); 
	Val(Ini^.GetItem('PrintSize','MarginBottom','4'), intMarginBottom, intCode); 

	{ Printer codes }
	strNormal      := Ini^.GetItem('PrintCode','Normal','ESC,"(s0s0B"');
	strItalic      := Ini^.GetItem('PrintCode','Italic','ESC,"(s1s0B"');
	strBold        := Ini^.GetItem('PrintCode','Bold','ESC,"(s0s3B"');
	strBoldItalic  := Ini^.GetItem('PrintCode','BoldItalic','ESC,"(s1s3B"');
	strFF          := Ini^.GetItem('PrintCode','FF', '#12');
	strLF          := Ini^.GetItem('PrintCode','LF', '#10');
	strCR          := Ini^.GetItem('PrintCode','CR', '#13');
	strPrinterInit := Ini^.GetItem('PrintCode','Init','ESC,"(10U",ESC,"(s0p12.00h10.0v0s0b8T"');
	strPrinterDone := Ini^.GetItem('PrintCode','Done','');

	Dispose(Ini, Done);

	{ Tokenise printing strings }

	Tokenise(strItalic);
	Tokenise(strNormal);
	Tokenise(strBold);
	Tokenise(strBoldItalic);
	Tokenise(strFF);
	Tokenise(strLF);
	Tokenise(strCR);
	Tokenise(strPrinterInit);
	Tokenise(strPrinterDone);

	{ Save last mode for title purposes }
	strLastMode := strNormal;

	{ Set for new page }
	bolNewPage := TRUE;

	{ Set margines, etc. }
	intPageEnd := intPageLength - intMarginTop - intMarginBottom;

	intLineWidth := intPageWidth - intMarginLeft - intMarginRight;
	if intLineWidth <= 0 then
	begin
		WriteLn('Impossible Left/Right margin combination');
		Halt(100);
	end;

	{ Define margin string }
	strMarginLeft := Fill(' ', intMarginLeft);
	strMarginLeft[0] := chr(intMarginLeft);

end;


destructor TPrintUnit.Done;
{ Deallocate & terminate the printer object }

begin
end;


procedure TPrintUnit.SetPageTitle(strNewPageTitle : string);
begin
	strPageTitle := strNewPageTitle;
end;


procedure TPrintUnit.SetHeader(strNewHeader1, strNewHeader2 : string);
begin
	strHeader1 := strNewHeader1;
	strHeader2 := strNewHeader2;
end;


procedure TPrintUnit.SetFooter(strNewFooter : string);
begin
	strFooter := strNewFooter;
end;


procedure TPrintUnit.SetBannerFile(strNewBannerFile : string);
begin
	strBannerFile := strNewBannerFile;
end;


function TPrintUnit.LinesLeft : integer;
{ Return the number of blank lines left }
begin
	LinesLeft := intPageEnd - intCurrLine + 1;
end;


function TPrintUnit.CurrentLine : integer;
{ Return the current line number }
begin
	CurrentLine := intCurrLine;
end;


function TPrintUnit.CurrentPage : integer;
{ Return the current line number }
begin
	CurrentPage := intCurrPage;
end;


function TPrintUnit.PageWidth : integer;
{ Return the current page width }
begin
	PageWidth := intLineWidth;
end;


function TPrintUnit.PageLength : integer;
{ Return the current page length }
begin
	PageLength := intPageEnd;
end;


procedure TPrintUnit.PrintHeader;
{ Print Header(s) when needed }

var
	intToDo : integer;

begin
	{ Skip Margin }
	for intToDo := 1 to intMarginTop do	CR;

	if strPageTitle <> '' then 
	begin
		PrintRaw(strBold, FALSE);
		PrintRaw(Center(ControlCodes(strPageTitle), 0), TRUE);
		PrintRaw(strLastMode, FALSE);
		CR;
		CR;
	end;

	if strHeader1 <> '' then 
	begin
		PrintRaw(strBold, FALSE);
		PrintRaw(ControlCodes(strHeader1), TRUE);
		PrintRaw(strLastMode, FALSE);
		CR;
	end;

	if strHeader2 <> '' then 
	begin
		PrintRaw(strBold, FALSE);
		PrintRaw(ControlCodes(strHeader2), TRUE);
		PrintRaw(strLastMode, FALSE);
		CR;
	end;

	{ Clear new page }
	bolNewPage := FALSE;
end;



procedure TPrintUnit.Tokenise(var strToDo : string);
{ Tokenise a printer string to a printable string }

var
	strLeft     : string;
	strRight    : string;
	intChar     : integer;
	intCode     : integer;

begin
	strRight := strToDo;
	strToDo  := '';

	repeat
		tlkTokenSplit(strRight, ',', strLeft, strRight);
		strLeft := tlkStrip(strLeft, ' ', BStripBoth);

		{ Check for string }
		if strLeft <> '' then 
		begin
			if strLeft[1] = '"' then
			begin

				{ Check if terminating double quote present }
				if strLeft[Length(strLeft)] <> '"' then
				begin
					WriteLn('String ', strLeft,' does not have a terminating "');
					Halt(100);
				end;

				{ strip the double quotes }
				strLeft := tlkStrip(strLeft, '"', BStripBoth);

				{ Add to the init string }
				strToDo := strToDo + strLeft;
			end

			{ Check for esacpe }
			else if tlkUpper(strLeft) = 'ESC' then
			begin
				strToDo := strToDo + #27;
			end

			{ Check if valid keycode }
			else
			begin
				{ Check for decimal code }
				if strLeft[1] = '#' then
				begin
					strLeft := Copy(strLeft, 2, Length(strLeft) - 1);
					if strLeft = '' then
					begin
						WriteLn('Empty # code is not allowed');
						Halt(100);
					end;
					Val(strLeft, intChar, intCode);
					if intCode <> 0 then
					begin
						WriteLn(strLeft, ' is not a number');
						Halt(100);
					end;
					strToDo := strToDo + chr(intChar);
				end

				{ Check for hex code }
				else if strLeft[1] = '$' then
				begin
					strLeft := Copy(strLeft, 2, Length(strLeft) - 1);
					if strLeft = ''  then
					begin
						WriteLn('No hex info after @');
						Halt(100);
					end;
					if Length(strLeft) > 2 then
					begin
						WriteLn(strLeft, ' is to long to be a Hex byte');
						Halt(100);
					end;
					strToDo := strToDo + chr(tlkHexToWord(strLeft));
				end;

			end;
		end;

	until strRight = '';
end;


procedure TPrintUnit.PrinterInit;
{ Prepare the printer for a document }

var
	filBanner     : text;
	strBannerLine : string;

begin
 { Calculate all margins }

	if strPageTitle <> '' then intPageEnd := intPageEnd - 2;
	if strHeader1 <> '' then intPageEnd := intPageEnd - 1;
	if strHeader2 <> '' then intPageEnd := intPageEnd - 1;
	if strFooter <> '' then intPageEnd := intPageEnd - 2;

	if intPageEnd <= 0 then
	begin
		WriteLn('Impossible Top/Bottom margin combination');
		Halt(100);
	end;

	{ Set positioning vars }
	intCurrLine := 1;
	intCurrPage := 0;

	if bytPrintMode <> BPrintPreview then
	begin
		{ Open the printer port }
		Assign(filPrinter, strPrinterPort);
		{$I-}
		ReWrite(filPrinter);
		{$I+}
		tlkDosErr('Opening printer', IOResult);
	end;

	{ Send Init string }
	PrintRaw(strPrinterInit, FALSE);

	{ Print Banner when needed }
	if strBannerFile <> '' then
	begin
		{ Open banner file }
		Assign(filBanner, strBannerFile);
		{$I-}
		Reset(filBanner);
		{$I+}
		tlkDosErr('Opening Print-Banner file', IOResult);


		CR;
		PrintRaw('File printed on ' + tlkDateString + ' ' + tlkTimeString(BTimeLong), TRUE);
		CR;

		repeat
			ReadLn(filBanner, strBannerLine);
			if Length(strBannerLine) > intLineWidth 
				then strBannerLine := Copy(strBannerLine, 1, intLineWidth);
			PrintRaw(Center(strBannerLine, 0), TRUE);
			CR;
		until Eof(filBanner);

		Close(filBanner);

		FormFeed;
	end;

	intCurrPage := 1;
end;


procedure TPrintUnit.PrinterDone;
begin
	{ FormFeed last page }
	FormFeed;
	
	{ Send Closing string }
	PrintRaw(strPrinterDone, FALSE);

	{ Close printer port }
	if bytPrintMode <> BPrintPreview then
	begin
		{ Close the printer port }
		Close(filPrinter);
	end;
end;


procedure TPrintUnit.Bold;
begin
	PrintRaw(strBold, FALSE);
	strLastMode := strBold;
end;


procedure TPrintUnit.Normal;
begin
	PrintRaw(strNormal, FALSE);
	strLastMode := strNormal;
end;


procedure TPrintUnit.Italic;
begin
	PrintRaw(strItalic, FALSE);
	strLastMode := strItalic;
end;


procedure TPrintUnit.BoldItalic;
begin
	PrintRaw(strBoldItalic, FALSE);
	strLastMode := strBoldItalic;
end;


function TPrintUnit.Fill(chrToFill : char; bytLength : byte) : string;
{ Return a filled string }

var
	strToFill     : string;

begin
	FillChar(strToFill, bytLength + 1, Ord(chrToFill));
	strToFill[0] := Chr(bytLength);
	Fill := strToFill;
end;


procedure TPrintUnit.FormFeed;

var
	intSkipLine : integer;

begin
	{ Check if at end of page, FALSE ==> go there }
	if intCurrPage <> 0 then
	begin
		if intCurrLine <= intPageEnd then
		begin
			for intSkipLine := intCurrLine to intPageEnd do
			begin
				CR;
			end;
		end;

		{ Check for Footer }
		if strFooter <> '' then
		begin
			{ Print it }
			PrintRaw(strBold, FALSE);
			PrintRaw(Fill(CLine, intLineWidth), TRUE);
			CR;
			PrintRaw(Center(ControlCodes(strFooter), 0), TRUE);
			PrintRaw(strLastMode, FALSE);
		end;
	end;
			
	{ Goto new page }
	PrintRaw(strFF, FALSE);
	intCurrLine := 1;
	Inc(intCurrPage);

	bolNewPage := TRUE;

end;


procedure TPrintUnit.CR;
begin
	PrintRaw(strCR + strLF + strMarginLeft, TRUE);
end;


procedure TPrintUnit.NewLine;
begin

	{ Goto NextLine }
	CR;

	if bolNewPage then PrintHeader;

	{ Check for End of Page }
	if intCurrLine > intPageEnd then 
	begin
		FormFeed;
	end
		else
	begin
		Inc(intCurrLine);
	end;

end;


function TPrintUnit.Center(strToCenter : string; bytLength : byte) : string;
{ Return a Centered string }
begin
	{ Check for enitire line adjust }
	if bytLength = 0 then bytLength := intLineWidth;

	{ Check if possible align length }
	if bytLength < Length(strToCenter) then
	begin
		WriteLn('Cannot align a ', Length(strToCenter),' long line on ',
						bytLength, ' chars.');
		Halt(100);
	end;

	{ Strip blank chars }
	strToCenter := tlkStrip(strToCenter, ' ', BStripBoth);

	{ Get length to add }
	Center := Fill(' ', (bytLength - Length(strToCenter)) DIV 2) + strToCenter;
end;


function TPrintUnit.Right(strToRight : string; bytLength : byte) : string;
{ Return a Right Justified string }
begin     
	{ Check for enitire line adjust }
	if bytLength = 0 then bytLength := intLineWidth;

	{ Check if possible align length }
	if bytLength < Length(strToRight) then
	begin
		WriteLn('Cannot rightalign a ', Length(strToRight),' long line on ',
						bytLength, ' chars.');
		Halt(100);
	end;

	{ Strip blank chars }
	strToRight := tlkStrip(strToRight, ' ', BStripBoth);

	{ Get length to add }
	Right := Fill(' ', bytLength - Length(strToRight)) + strToRight;
end;


function TPrintUnit.Left(strToLeft : string; bytLength : byte) : string;
{ Return a Left Justified string }
begin     
	{ Check for enitire line adjust }
	if bytLength = 0 then bytLength := intLineWidth;

	{ Check if possible align length }
	if bytLength < Length(strToLeft) then
	begin
		WriteLn('Cannot leftalign a ', Length(strToLeft),' long line on ',
						bytLength, ' chars.');
		Halt(100);
	end;

	{ Strip blank chars }
	strToLeft := tlkStrip(strToLeft, ' ', BStripBoth);

	{ Get length to add }
	Left := strToLeft + Fill(' ', bytLength - Length(strToLeft));
end;


procedure TPrintUnit.PrintRaw(strToPrint : string; bolToScreen : boolean);
{ Raw printer output routine }
begin

	case bytPrintMode of

		BPrintPreview : 
		begin
			if bolToScreen then Write(strToPrint);
		end;

		BPrintNormal  : 
		begin
			Write(filPrinter, strToPrint);
		end;

		BPrintBoth    :
		begin
			if bolToScreen then Write(strToPrint);
			Write(filPrinter, strToPrint);
		end;
	end;

end;


function TPrintUnit.ControlCodes(strToDo : string) : string;
{ Insert a page number in a string when needed }

var
	strToPrint : string;
	bytPos     : byte;
	strPage    : string[5];

begin
	{ Check if page number needed }
	bytPos := Pos('', strToDo);
	if bytPos = 0 then
	begin
		strToPrint := strToDo;
	end
		else
	begin
		if bytPos > 1 then
		begin
			strToPrint := Copy(strToDo, 1, bytPos - 1);
		end;
		Str(intCurrPage, strPage);
		strToPrint := strToPrint + strPage;
		if bytPos < Length(strToDo) then
		begin
			strToPrint := strToPrint + 
												Copy(strToDo, bytPos + 1, Length(strToDo) - bytPos);
		end;
	end;

	strToDo := strToPrint;

	bytPos := Pos('', strToDo);
	if bytPos = 0 then
	begin
		strToPrint := strToDo;
	end
		else
	begin
		if bytPos > 1 then
		begin
			strToPrint := Copy(strToDo, 1, bytPos - 1);
		end;
		strToPrint := strToPrint + tlkDateString;
		if bytPos < Length(strToDo) then
		begin
			strToPrint := strToPrint + 
												Copy(strToDo, bytPos + 1, Length(strToDo) - bytPos);
		end;
	end;

	strToDo := strToPrint;

	bytPos := Pos('', strToDo);
	if bytPos = 0 then
	begin
		strToPrint := strToDo;
	end
		else
	begin
		if bytPos > 1 then
		begin
			strToPrint := Copy(strToDo, 1, bytPos - 1);
		end;
		strToPrint := strToPrint + tlkTimeString(BTimeLong);
		if bytPos < Length(strToDo) then
		begin
			strToPrint := strToPrint + 
												Copy(strToDo, bytPos + 1, Length(strToDo) - bytPos);
		end;
	end;

	ControlCodes := strToPrint;

end;


procedure TPrintUnit.Print(strToPrint : string);
{ Write a string & keep track of positioning }

var
	intToDo        : integer;

begin

	{ Check line width }
	if Length(strToPrint) > intLineWidth then
	begin
		WriteLn('Line too long to Print');
		Halt(100);
	end;

	{ Check for End of Page }
	if intCurrLine > intPageEnd then 
	begin
		FormFeed;
	end;

	if bolNewPage then PrintHeader;

	{ Print the line }
	PrintRaw(strToPrint, TRUE);

end;


procedure TPrintUnit.PrintLn(strToPrint : string);
{ Write a string & keep track of positioning }

begin
	Print(strToPrint);
	NewLine;
end;


function TPrintUnit.NumStr(linToConvert : longint) : string;
{ Convert a number to a string }

var
	strResult : string;

begin
	 Str(linToConvert, strResult);
	 NumStr := strResult;
end;


function TPrintUnit.DatePackStr(linToConvert : longint) : string;
{ Convert a packed date to a string }
begin
	DatePackStr := tlkPackToDate(linToConvert);
end;


end.