/*-

  PE112 Library: definitions

  Author:	Marc Vauclair
  Date:		July 3, 1993
  Revised: 	July 13, 1993
		(The MS-DOS modifications are authored by Marc Stern)

  RCS Information:
  ---------------
  $Author: vauclair $
  $Date: 1993/07/15 08:16:15 $
  $Revision: 1.9 $
  $RCSfile: PE112.h,v $
  $Source: /u/vauclair2/dd/PE112/RCS/PE112.h,v $
  $State: Exp $
  $Locker: vauclair $

  Notes:
  -----
  1) if compiled under Unix, the UNIX symbol must be defined

  2) if compiled under MS-DOS, the MSDOS symbol must be defined

  3) This file is still to be restructured and extensively documented.

-*/

#if !defined(___PE112_h)

#define ___PE112_h

#if ! defined(UNIX) && ! defined(MSDOS)
/* The #error directive is not accepted by the Sun C compiler */
/*#error You must define MSDOS or UNIX */
#endif

/*-
  The traditional includes.
  They are scattered throughout the various .c files to speed up
  the compilation and earn disk space with precompiled header files.
-*/

#if defined(DEBUG)
# include <stdio.h>
#endif



/*-
  What follow is a bit special. To guarantee the portability of the code
  one has to make sure that all the symbols to be manipulated by the linker
  are unique in the first 6 characters. This means that with our convention
  of prefixing all names with PE112_ the 6 characters are already eaten and
  that for a dumb linker they are all alike.

  This is the reason why we define here a six letters equivalent symbol
  for all the EXTERN or static functions and for all the global variables.

  The disadvantage of this method is that if one has to resort to a symbolic
  debugger he won't see the API names but rather the nonsense SC__xx names.
  This is a no problem because the entire code of the library is written
  with a defensive programming approach that would (hopefully ;-) make void
  the need for a symbolic debugger.

  One advantage of this method is that the symbol table is scrambled: someone
  having access to it can't infer the use of the functions from their names.
-*/

#define PE112_sys_open		SC__01
#define PE112_sys_close		SC__02
#define PE112_sys_read		SC__03
#define PE112_sys_write		SC__04

#define PE112_debug		SC__08

#define PE112_reply_names	SC__09

#define PE112_set_lock_duration SC__10
#define PE112_open_session	SC__11
#define PE112_close_session	SC__12
#define PE112_process_command	SC__13

#define PE112_reset		SC__18
#define PE112_swallow		SC__19
#define PE112_eject		SC__20
#define PE112_reader_identification_request	SC__21
#define PE112_reader_status_request		SC__22
#define PE112_get_mask		SC__23
#define PE112_get_config	SC__24
#define PE112_card_power_on	SC__25
#define PE112_card_power_off	SC__26



/* convenient types */
typedef unsigned int UINT;
typedef unsigned char BYTE;

#define PRIVATE	static
#define PUBLIC

#define FPRINTF	(void)fprintf
#define FFLUSH	(void)fflush

/* INFINITE_TIMEOUT is the no timeout value */
#define INFINITE_TIMEOUT -1L

#define FAILURE	-1

#if !defined(TRUE)
# define TRUE 1
/* pedantic: # define TRUE (1==1) */
#endif

#if !defined(FALSE)
#define FALSE 0
/* pedantic: # define FALSE (0==1) */
#endif

/* note the "/" at the end, it is mandatory */
/* this is the name of the directory where the locks are kept,
   it must be readable, writeable and searchable by everybody */
#if !defined(PE112_lock_directory)

#if defined(UNIX)
# define PE112_lock_directory	"/tmp/"
#endif

#if defined(MSDOS)
# define PE112_lock_directory	"c:\\"
#endif

#endif !defined(PE112_lock_directory)

#if defined(UNIX)
# define PE112_library_id	"PE112_"
#endif defined(UNIX)

#if defined(MSDOS)
/* - those poor MSDOS users are still limited to 8 chars for the principal
     part of a file name :-(
   - device name = COMx
 */
# define PE112_library_id	"PE_"
#endif defined(MSDOS)


#define PE112_magic_number	0x15
#define PE112_packet_length	256

#if !defined(PE112_default_lock_duration)
# define PE112_default_lock_duration 120
#endif

#if defined(HAVE_PROTOTYPES)
#define PROTO(name, args)  name args
#else
#define PROTO(name, args)  name()
#endif

#ifndef EXTERN

# if defined(__cplusplus) || defined(__cplusplus__)
#  define EXTERN   extern "C"
# else
#  define EXTERN   extern 
# endif

#endif



/* Debugging macros and declarations */

/*- 
  The tracing is controlled by two things: a #defined symbol DEBUG and
  a global variable PE112_debug. The DEBUG symbol can be turned on
  on the compilation command line in the makefile.

  If DEBUG is not defined, there is not tracing at all and all the tracing
  code is not compiled in.

  If DEBUG is defined, the tracing code is compiled in and a global
  variable PE112_debug controls the selection of the messages to be
  printed out. This approach allows the application programmer to
  details to be traced and also to control dynamically the sections 
  of his code where he wants the tracing turned on or off.

  PE112_debug takes as value an OR combination of the DEBUG_xxx values.
  A special DEBUG_NONE value is given to turn dynamically all the tracing
  off.

  The application program modifies dynamically the tracing details by
  modifying the content of the PE112_debug variable. It is the 
  responsability of the application program to make sure that 
  these operations are only performed if the DEBUG symbol is defined.
-*/

/*-
  The dprint is a special macro in the sense that it is one of the
  possible implementations of a macro with a variable number of
  arguments.

  When used, args MUST be put within (). E.g.:

  	dprint(DEBUG_FLAG, ("Value: %d, Label: %s\n", value, label));
-*/

#define DEBUG_NONE	0		/* no tracing at all */
#define DEBUG_SYS	(1<<0)		/* System layer      */
#define DEBUG_0LAYER	(1<<1)		/* Layer 0 main      */
#define DEBUG_0AUX	(1<<2)		/* Layer 0 auxiliary */
#define DEBUG_1LAYER	(1<<3)		/* Layer 1           */

#if defined(DEBUG)

	EXTERN int PE112_debug;

#	define dprint(flag, args) \
  	    if (PE112_debug & (flag)) { \
  		(void)printf("flag: "); \
  		(void)printf args; \
  		(void)fflush(stdout); \
            }
#	define dprint2(flag, args) \
  	    if (PE112_debug & (flag)) { \
  		(void)printf args; \
  		(void)fflush(stdout); \
            }
#	define dperror(flag, val, mes) \
  	    if (PE112_debug & (flag)) { \
  		if ((val) < 0) { \
		   (void)printf("flag: "); \
		   (void)fflush(stdout); \
		   perror(mes); \
		   (void)fflush(stderr); \
		} \
	    }
#else !defined(DEBUG)

#	define dprint(flag, args)
#	define dprint2(flag, args)
#	define dperror(flag, val, mes)

#endif !defined(DEBUG)



typedef struct {
  UINT length;
  BYTE buf[PE112_packet_length];
} PE112_packet;

/*-
  The PE112_status_code enumerates the possible values returned by
  Layer 0 functions within the PE112_session_info data.
-*/

typedef enum { 
  /* success code */
  PE112_SUCCESS = 0,

  /* generic failure code (used mainly for initialisations) */
  PE112_FAILURE,

  /* Error codes */
  PE112_TTY_OPEN_FAILURE,
  PE112_TTY_OPEN_TIMEOUT,
  PE112_TTY_GET_TERMIOS_FAILURE,
  PE112_TTY_SET_INPUT_SPEED_FAILURE,
  PE112_TTY_SET_OUTPUT_SPEED_FAILURE,
  PE112_TTY_SET_TERMIOS_FAILURE,

  PE112_FASTNET_HEADER_WRITE_ERROR,
  PE112_WRITE_COMMAND_ERROR,
  PE112_FASTNET_HEADER_READ_ERROR,
  PE112_READ_ANSWER_ERROR,

  PE112_READER_RESET_FAILURE,

  PE112_IN_USE,

  PE112_DEVICE_CLOSE_FAILURE,
  PE112_UNLOCK_FAILURE

} PE112_status_code;

typedef struct {
  PE112_status_code status;
  char * devname;
  char * lockname;
  int fd;
} PE112_session_info;

/*-
  The PE112_reply_code enumerates the possible values for the interpretation
  of the Smart Card reader answer packets.

  These values are not to be confused with those defined by PE112_status_code.
-*/

typedef enum {

  RESET_UNKNOWN, 
  RESET_WRONG_LENGTH, 
  RESET_FAIL,
  RESET_OK,

  SWALLOW_UNKNOWN, 
  SWALLOW_WRONG_LENGTH, 
  SWALLOW_FAIL, 
  SWALLOW_OK,
  SWALLOW_SHORT, 
  SWALLOW_JAM, 
  SWALLOW_UNKNOWN_REPLY,
  SWALLOW_WRONG_PREFIX,

  EJECT_UNKNOWN, 
  EJECT_WRONG_LENGTH, 
  EJECT_FAIL, 
  EJECT_OK,
  EJECT_JAM_IN, 
  EJECT_JAM_ENTRY, 
  EJECT_UNKNOWN_REPLY,
  EJECT_WRONG_PREFIX,

  RSR_UNKNOWN, 
  RSR_WRONG_LENGTH, 
  RSR_FAIL, 
  RSR_NOSMART,
  RSR_INSLOT, 
  RSR_TOOSMALL, 
  RSR_INPOSITION, 
  RSR_UNKNOWN_REPLY,
  RSR_WRONG_PREFIX,

  RIR_UNKNOWN, 
  RIR_WRONG_LENGTH, 
  RIR_FAIL, 
  RIR_OK

} PE112_reply_code;

EXTERN char * PE112_reply_names[];

/* Command result structures */
/* For each command, we define a structure that contains the
   result of the analysis of the answer of the SmartCard Reader to 
   the command */

typedef struct {
  PE112_reply_code status;
} PE112_RESET_reply;

typedef struct {
  PE112_reply_code status;
} PE112_SWALLOW_reply;

typedef struct {
  PE112_reply_code status;
} PE112_EJECT_reply;

typedef struct {
  PE112_reply_code status;
} PE112_STATUS_reply;

typedef struct {
  PE112_reply_code status;
  unsigned int hardware_number;
  unsigned int microprogram_number;
  unsigned int version_number;
} PE112_IDENT_reply;



/* System layer */

EXTERN
PUBLIC int PROTO(PE112_sys_open,
		 (char *path, int oflag, long timeout));

EXTERN
PUBLIC int PROTO(PE112_sys_close,
		 (int fd, long timeout));

EXTERN
PUBLIC int PROTO(PE112_sys_read,
		 (int fd, 
		  BYTE *buf, 
		  UINT nbyte, 
		  long timeout));

EXTERN
PUBLIC int PROTO(PE112_sys_write,
		 (int fd, 
		  BYTE *buf, 
		  UINT nbyte, 
		  long timeout));



/* Layer 0 */

EXTERN
PUBLIC PE112_session_info *
PROTO(PE112_open_session, 
      (char *devname, long timeout, int retry_count, int duration));

EXTERN
PUBLIC PE112_status_code
PROTO(PE112_close_session, 
      (PE112_session_info *session_info, long timeout, int retry_count));

EXTERN
PUBLIC PE112_status_code
PROTO(PE112_process_command,
      (PE112_session_info *session_info,
       PE112_packet * command, 
       long timeout, 
       int retry_count, 
       PE112_packet * answer));

EXTERN
PUBLIC void
PROTO(PE112_set_lock_duration,
      (PE112_session_info * session_info,
       int duration));



/* Layer 1 */

EXTERN
PUBLIC PE112_status_code
PROTO(PE112_reset,
      (PE112_session_info *session_info,
       long timeout, 
       int retry_count,
       PE112_packet * out,
       PE112_RESET_reply * reply));

EXTERN
PUBLIC PE112_status_code
PROTO(PE112_swallow,
      (PE112_session_info * session_info,
       long timeout, 
       int retry_count,
       PE112_packet * out,
       PE112_SWALLOW_reply * reply));

EXTERN
PUBLIC PE112_status_code
PROTO(PE112_eject,
      (PE112_session_info * session_info,
       long timeout, 
       int retry_count,
       PE112_packet * out,
       PE112_EJECT_reply * reply));

EXTERN
PUBLIC PE112_status_code
PROTO(PE112_reader_status_request,
      (PE112_session_info * session_info,
       long timeout, 
       int retry_count,
       PE112_packet * out,
       PE112_STATUS_reply * reply));

EXTERN
PUBLIC PE112_status_code
PROTO(PE112_reader_identification_request,
      (PE112_session_info * session_info,
       long timeout, 
       int retry_count,
       PE112_packet * out,
       PE112_IDENT_reply * reply));

EXTERN
PUBLIC PE112_status_code
PROTO(PE112_card_power_on,
      (PE112_session_info * session_info,
       long timeout, 
       int retry_count,
       PE112_packet * out));

EXTERN
PUBLIC PE112_status_code
PROTO(PE112_card_power_off,
      (PE112_session_info * session_info,
       long timeout, 
       int retry_count,
       PE112_packet * out));

EXTERN
PUBLIC PE112_status_code
PROTO(PE112_get_mask,
      (PE112_session_info * session_info,
       long timeout, 
       int retry_count,
       PE112_packet * out));

EXTERN
PUBLIC PE112_status_code
PROTO(PE112_get_config,
      (PE112_session_info * session_info,
       long timeout, 
       int retry_count,
       PE112_packet * out));

#endif !defined(___PE112_h)



