/* logtr.c
 * -------
 *
 * A program to truncate a UUPC logfile by a given number of days.
 *
 * Based on ideas by Jiri Kuchta <kuchta.dcse.fee.vutbr.cz>
 *
 * Author: Mike Lawrie <mlawrie@apies.frd.ac.za>
 * Modified by: Stephen Marquard <scm@picasso.wcape.school.za>
 *
 *  $Log: logtr.c $
 *  Revision 1.3  1995/04/30 20:02:45  mlawrie
 *  Incorporated RCS features into identifying the version of
 *  the program.
 *
 *
 */

#include <stdio.h>
#include <dos.h>
#include <direct.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

/*
 * Program returns:-
 * 0 = all OK
 * 1 = incorrect number of parameters
 * 2 = invalid number of days in the command line
 * 3 = cannot open the old log file
 * 4 = no filename was detected on the command line
 * 5 = invalid format in the existing log file
 * 6 = cannot open the new output file
 * 7 = could not close the old log file (1) (impossible?)
 * 8 = could not delete the old log file (1)
 * 9 = could not write to the new log file (disk full?)
 * 10 = could not write final record to new log file
 * 11 = could not close the old log file (2) (impossible?)
 * 12 = could not close the new log file (impossible?)
 * 13 = could not delete the old log file (2)
 * 14 = could not rename the new output file
 */

/*
 * The year-wrapping is a sod. Assume that the .LOG file is in chronological
 * order, and consider the date in the first record of the file. If this
 * is greater than today's date, then this record is for last year. So,
 * adjust the "today" date to be 365 days ahead, and this affects the
 * "delete" date by 365 days as well. (If leap years worry, you, then
 * adjust the program yourself :-).
 * 
 * When calculating the day of the year for each record, keep track of the
 * day that was found in the record. If the current record is earlier than
 * the previous day, then the .LOG file has gone through an end of year, so
 * add 365 days to every date in the .LOG file from this point onwards.
 * 
 * 
 *     days
 *       |                             +=============================== today
 *       |                             !              D                 +365
 *       |                             !             D
 *       |                             !            D
 *       |                             !  +--------d------------------- delete
 *       |                             !  !       d                     +365
 *       |                             !  !      d
 *  365  |.............................!..!.....d...................... 365
 *       |                             !  !    d  
 *       |   a                         !  !   dd
 *       |  a                          !  !  d d  
 *       | a                           !  ! d  d  
 *       |                             !  !    d  
 * today |=============================+==!====d====================
 *       |            B           C       !    d      D
 *       |           B           C        !    d     D
 *       |          B           C         !    d    D
 * delete|---------------------c----------+----d---d----------------
 *       |                    c                d  d
 *       |                   c                 d d
 *       |                                     dd
 *       +----------------------------------------------------------> records
 *
 *Consider 4 files, viz A, B, C and D.
 *
 *   "aaa" deletes, all records are later than "today" so they are
 *         last year's records (ie very old).
 *
 *   "BBB" passes into the new log file, as they are all this year's
 *         records, and are later than the delete date.
 *
 *   "ccc" deletes, being this year's records but before the delete date.
 *
 *   "CCC" passes into the new log file.
 *
 *   "dddddddd/ddddd" deletes, being last year's and this year's records
 *         until the delete date.
 *
 *   "DDD" passes into the new log file.
 *
 * Clearly, the line "delete+365" can be below the "365" line while the
 * "today+365" line is above the "365" line, and the principles are not
 * altered.
 *
 */

#define EXT ".$$$"
#define LOGREC 200

static char rcsid[] = "$Id: logtr.c 1.3 1995/04/30 20:02:45 mlawrie Exp $";

int md[12] = {31,28,31,30,31,30,31,31,30,31,30,31};
int amd[12];

int dayinstr( char *buf);
void usage( char *progname );

/*
 * int dayinstr( char *buf )
 * {
 *     int mth, day;
 *
 *     mth = (int)(buf[0]-'0')*10 + (int)(buf[1]-'0');
 *     day = (int)(buf[3]-'0')*10 + (int)(buf[4]-'0');
 *     return( amd[mth-1] + day );
 * }
 */


main (int argc, char *argv[])
{
	 int deldays, deldate;
	 int i;

	 int today;
	 struct date datebuf;

	 FILE *oldf, *newf;
	 char newfname[MAXPATH],
			drive[MAXDRIVE], dir[MAXDIR], file[MAXFILE], ext[MAXEXT];
	 int oldflags;
	 char logrec[LOGREC];
	 long int oldfsz = 0;
	 long int newfsz = 0;
	 int  firstdate;

 	 fprintf( stderr, "Version of %s = %s\n", argv[0], rcsid );

	 if (argc >= 3)
		fprintf( stderr, "%s: %s %s\n", argv[0], argv[1], argv[2] );
	 else
		if (argc==2)
			fprintf( stderr, "%s: %s \n", argv[0], argv[1] );
		else
			fprintf( stderr, "%s:\n", argv[0] );

    if( argc != 3 )
    {
		  usage( argv[0] );
        exit( 1 );
    }
/*
 * Calculate accumulated month-days once.
 * amd[0]=0, amd[1]=31, amd[2]=58/59, amd[3] 89/90,....
 * Note that this program is unlikely to be operating in year 2000, so
 * let's not get into an argument as to whether or not that is a leap year.
 */

    getdate( &datebuf );
    if( datebuf.da_year == 1996 )
        md[1]++;

    amd[0] = 0;
	 for( i=1; i<=11; i++ )
        amd[i] = md[i-1] + amd[i-1];

/*
 * Test for a non-zero numeric value of days as first parameter of
 * the command line.
 */

    deldays = atoi( argv[2] );

    if( (deldays <= 0) || (deldays >365)  )
    {
        fprintf( stderr, "Invalid number of days in command line (%s) \n",
					  argv[2] );
        usage( argv[0] );
        exit( 2 );
    }

	 today = datebuf.da_day + amd[datebuf.da_mon-1];

/*
 * Open the old log file as the input file.
 */

	 if( !(oldf=fopen(argv[1], "r")) )
	 {
		  fprintf( stderr, "Cannot open the old log file (%s)\n", argv[1] );
		  exit( 3 );
	 }

	 fseek(oldf, 0, SEEK_END);
	 oldfsz = newfsz = ftell(oldf);
	 fseek(oldf, 0, SEEK_SET);

/*
 * Form the name of the output file, don't open it yet.
 */

    oldflags = fnsplit( argv[1], drive, dir, file, ext );
    if( !(oldflags & FILENAME) )
	 {
	 fprintf( stderr, "A filename must be specified on the command line (%s)\n",
                 argv[1] );
        usage( argv[0] );
        exit( 4 );
	 }
    fnmerge( newfname, drive, dir, file, EXT );


/*
 * Read the old log file, find the first day in that file. Watch out for
 * a wrap of a year.
 */
	if( fgets(logrec, LOGREC, oldf) == NULL )
	 {
		fprintf( stderr, "Invalid log file format (%s)\n", argv[1] );
		  exit( 5 );
	 }

	/* skip all records beginning with ( */

	while (!isdigit(logrec[0]) && (!feof(oldf)))
		fgets(logrec, LOGREC, oldf);

	firstdate = dayinstr( logrec );

	 if( firstdate > today )
		  today += 365;
	 deldate = today - deldays;

/*
 * It may be that there are no old records in the log file....
 */

	 if ( deldate < firstdate )

		  while( !feof(oldf) )
				fgets(logrec, LOGREC, oldf);

	 else {

/*
 * Read the old log file until deletion date
 */

		  if (deldate > dayinstr(logrec))
			do
				do
					fgets(logrec, LOGREC, oldf);
				while ( !isdigit(logrec[0]) && !feof(oldf) );
			while ((deldate > dayinstr(logrec)) && (!feof(oldf)));

/*
 * Maybe EOF co-incides with the cut-off date?
 */

		  if( feof(oldf) )
		  {
				if( fclose(oldf) == EOF )
				{
					 fprintf( stderr, "Could not close the old log file (1)\n" );
					 exit( 7 );
				}

				if( unlink(argv[1]) != 0)
				{
					 fprintf( stderr, "Could not delete the old log file (1)\n" );
					 exit( 8 );
				}

		  } else {

/*
 * Open the new log file, there is something to write.
 */

				if( !(newf=fopen(newfname,"w")) )
				{
					 fprintf( stderr, "Cannot open the new log file (%s)\n", newfname );
					 exit( 6 );
				}

/*
 * Transcribe the wanted records until EOF on the old log file
*/

				while( !feof(oldf) )
				{
					 if( fputs(logrec, newf) == EOF )
					 {
						  fprintf( stderr, "Could not write to new log file\n" );
						  exit( 9 );
					 }
					 fgets( logrec, LOGREC, oldf );
				}

				if( fclose(oldf) == EOF )
				{
					 fprintf( stderr, "Could not close the old log file (2)\n" );
					 exit( 11 );
				}

				newfsz= ftell(newf);

				if( fclose(newf) == EOF )
				{
					 fprintf( stderr, "Could not close the new log file\n" );
					 exit( 12 );
				}

				if( unlink(argv[1]) != 0 )
				{
					 fprintf( stderr, "Could not delete the old log file (2)\n" );
					 exit( 13 );
				}

				if( rename(newfname, argv[1]) != 0 )
				{
					 fprintf( stderr, "Could not rename the output file\n" );
					 exit( 14 );
				}
			}
	 }

	 fprintf( stderr, "%s: Old size=%li, new size=%li\n",
				 argv[1], oldfsz, newfsz );

	 return 0;
}

void usage( char *progname )
{
	 fprintf( stdout, "Usage: %s filename days\n", progname );
}

int dayinstr( char *buf) {

	 static int last;  /* Set to zero initially */
	 int mth, day, sum;

	 mth = (int)(buf[0]-'0')*10 + (int)(buf[1]-'0');
	 day = (int)(buf[3]-'0')*10 + (int)(buf[4]-'0');
	 sum = amd[mth-1] + day;

	 if( sum < last )
		  last = sum + 365;
	 else
		  last = sum;

	 return last;
}

/*
   Changes:
	1.02  SCM  25/3/95
		- modified to handle intermediate () lines with
		  UUPC debug levels > 1
		- relocated some comments to make more sense
		- altered some checking of return codes for library calls
		- used ftell() calls to report file sizes
		- added prototypes & #includes so as to compile cleanly
		- fixed banner so as not to dereference null argv[]
		  pointers
	Compiled with Borland C++ 3.1

*/
