    /*
        Originally written by Ed Cogburn (ecogburn@greene.xtn.net), based on
        info and code gotten from multiple sources on the internet and some
        help from some books on register level programming of video cards.
        Released under the GPL license.  The license can be found at
        http://www.gnu.org/licenses/gpl.html.

        This code was originally created using Turbo C++ 3.0.  It will compile
        on Borland C++ v5.0 too.  Although the code is strict C, several
        non-portable things are used.  Two functions are written in Intel
        syntax asm.  The routines poke() pokeb(), int86() and int86x() are
        used in places as well.
    */



#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <dos.h>

#define byte unsigned char
#define word unsigned int

#define TRUE 1
#define FALSE 0


    /*
        The VESA BIOS interface uses data blocks that it fills with info
        when called.  These are needed even though we will only use a
        few of these variables.
    */
struct VESAInfoStruc {
    byte Signature[4];
    word Version;
    byte far *OEMStringPtr;
    byte Capabilities[4];
    word far *VideoModesPtr;
    word TotalMemory;
    byte far *OemSoftwareRevPtr;
    byte far *OemVendorNamePtr;
    byte far *OemProductNamePtr;
    byte far *OemProductRevPtr;

    byte Dummy1[222];
    byte Dummy2[256];

    /* There is more, see VESA docs. */
};


struct VESAModeInfoStruc {
    word ModeAttributes;
    byte WinAAttributes;
    byte WinBAttributes;
    word WinGranularity;
    word WinSize;
    word WinASegment;
    word WinBSegment;
    word far *WinFunctionPtr;
    word BytesPerScanLine;
    word XResolution;
    word YResolution;
    byte XCharSize;
    byte YCharSize;
    byte NmbrPlanes;
    byte BitsPerPixel;
    byte NmbrBanks;
    byte MemoryModel;


    byte Dummy1[228];

    /* There is more, see VESA docs. */
};

struct VESATextModesStruc {
    word v80x60;
    word v132x25;
    word v132x43;
    word v132x50;
    word v132x60;
};


    /*
        The magic numbers.  They can be found in multiple places on
        the internet, but I have never seen any documentation explaining
        these.  You really need a book on register level programming of
        video cards to begin to see what these are doing.  Such books
        were available in the late '80s, but can't be found today (AFAIK).
        The books were enough help to let me create the 94x25 and 94x50
        modes.
    */

word Data80x25[] = {
    0x0E11,0x5F00,0x4F01,0x5002,0x8203,0x5404,0x8005,0xBF06,
    0x1F07,0x4F09,0x9C10,0x8E11,0x8F12,0x2813,0x9615,0xB916
};

word Data80x28[] = {
    0x0E11,0x5F00,0x4F01,0x5002,0x8203,0x5404,0x8005,0xBF06,
    0x1F07,0x4D09,0x9C10,0x8E11,0x8712,0x2813,0x9615,0xB916
};

word Data80x30[] = {
    0x0C11,0x5F00,0x4F01,0x5002,0x8203,0x5504,0x8105,0x0B06,
    0x3E07,0x4F09,0xEA10,0x8C11,0xDF12,0x2813,0xE715,0x0416
};

word Data80x34[] = {
    0x0C11,0x5F00,0x4F01,0x5002,0x8203,0x5504,0x8105,0x0B06,
    0x3E07,0x4D09,0xEA10,0x8C11,0xDF12,0x2813,0xE715,0x0416
};

word Data80x43[] = {
    0x0511,0x5F00,0x4F01,0x5002,0x8203,0x5404,0x8005,0xBF06,
    0x1F07,0x4709,0x8310,0x8511,0x5712,0x2813,0x6315,0xBA16
};

word Data80x50[] = {
    0x0E11,0x5F00,0x4F01,0x5002,0x8203,0x5404,0x8005,0xBF06,
    0x1F07,0x4709,0x9C10,0x8E11,0x8F12,0x2813,0x9615,0xB916
};

word Data80x60[] = {
    0x0C11,0x5F00,0x4F01,0x5002,0x8203,0x5504,0x8105,0x0B06,
    0x3E07,0x4709,0xEA10,0x8C11,0xDF12,0x2813,0xE715,0x0416
};

word Data90x25[] = {
    0x0C11,0x6B00,0x5901,0x5A02,0x8E03,0x5F04,0x8A05,0xBF06,
    0x1F07,0x4F09,0x9C10,0x8E11,0x8F12,0x2D13,0x9615,0xB916
};

word Data90x28[] = {
    0x0E11,0x6B00,0x5901,0x5A02,0x8E03,0x5F04,0x8A05,0xBF06,
    0x1F07,0x4D09,0x9C10,0x8E11,0x8712,0x2D13,0x9615,0xB916
};

word Data90x30[] = {
    0x0C11,0x6D00,0x5901,0x5A02,0x9003,0x6004,0x8B05,0x0B06,
    0x3E07,0x4F09,0xEA10,0x8C11,0xDF12,0x2D13,0xE715,0x0416
};

word Data90x34[] = {
    0x0C11,0x6D00,0x5901,0x5A02,0x9003,0x6004,0x8B05,0x0B06,
    0x3E07,0x4D09,0xEA10,0x8C11,0xDB12,0x2D13,0xE715,0x0416
};

word Data90x43[] = {
    0x0E11,0x6B00,0x5901,0x5A02,0x8E03,0x5F04,0x8A05,0xBF06,
    0x1F07,0x4709,0x9C10,0x8E11,0x8F12,0x2D13,0x9615,0xB916
};

word Data90x50[] = {
    0x0E11,0x6B00,0x5901,0x5A02,0x8E03,0x5F04,0x8A05,0xBF06,
    0x1F07,0x4709,0x9C10,0x8E11,0x8F12,0x2D13,0x9615,0xB916
};

word Data90x60[] = {
    0x0C11,0x6D00,0x5901,0x5A02,0x9003,0x6004,0x8B05,0x0B06,
    0x3E07,0x4709,0xEA10,0x8C11,0xDF12,0x2D13,0xE715,0x0416
};

word Data94x25[] = {
    0x0C11,0x6C00,0x5D01,0x5E02,0x8F03,0x6204,0x8E05,0xBF06,
    0x1F07,0x4F09,0x9C10,0x8E11,0x8F12,0x2F13,0x9615,0xB916
};

word Data94x28[] = {
    0x0E11,0x6C00,0x5D01,0x5E02,0x8F03,0x6204,0x8E05,0xBF06,
    0x1F07,0x4D09,0x9C10,0x8E11,0x8712,0x2F13,0x9615,0xB916
};

word Data94x30[] = {
    0x0C11,0x6C00,0x5D01,0x5E02,0x8F03,0x6204,0x8E05,0x0B06,
    0x3E07,0x4F09,0xEA10,0x8C11,0xDF12,0x2F13,0xE715,0x0416
};

word Data94x34[] = {
    0x0C11,0x6C00,0x5D01,0x5E02,0x8F03,0x6204,0x8E05,0x0B06,
    0x3E07,0x4D09,0xEA10,0x8C11,0xDB12,0x2F13,0xE715,0x0416
};

word Data94x43[] = {
    0x0511,0x6C00,0x5D01,0x5E02,0x8F03,0x6204,0x8E05,0xBF06,
    0x1F07,0x4709,0x8310,0x8511,0x5712,0x2F13,0x6315,0xBA16
};

word Data94x50[] = {
    0x0E11,0x6C00,0x5D01,0x5E02,0x8F03,0x6204,0x8E05,0xBF06,
    0x1F07,0x4709,0x9C10,0x8E11,0x8F12,0x2F13,0x9615,0xB916
};

word Data94x60[] = {
    0x0C11,0x6C00,0x5D01,0x5E02,0x8F03,0x6204,0x8E05,0x0B06,
    0x3E07,0x4709,0xEA10,0x8C11,0xDF12,0x2F13,0xE715,0x0416
};



/*---------------------------------------------------------------------------
    What follows is some notes on setting various text modes.  I keep this
    near to the data above for convenience.  Unfortunately, this by itself
    won't help you without some book(s) on register level programming of
    the VGA.


    Just set 480 vertical scan lines
0C11 0C  12 00001100  Vertical Retrace End (first write to it)
                      b7=0 don't protect registers 0-7
                      b6=0 3 DRAM cycles for Bandwidth
                      b5=0 enable Vertical Interrupts
                      b4=0 clear Vertical Interrupts
                      b3-0=1100=12 lower 5 bits of Vertical Retrace End
                          ??? 5 bits or 4 ???
0B06 0B 011 00001011  lower 8 bits of Vertical Total plus 9th and 10th
                          bits from the Overflow register results in
                          1000001011=523 Vertical Total
3E07 3E  62 00111110  b7=0 bit 10 of Vertical Retrace Start
                      b6=0 bit 10 of Vertical Display Enable End
                      b5=1 bit 10 of Vertical Total
                      b4=1 bit 9 of Line Compare
                      b3=1 bit 9 of Vertical Blank Start
                      b2=1 bit 9 of Vertical Retrace Start
                      b1=1 bit 9 of Vertical Display Enable End
                      b0=0 bit 9 of Vertical Total
EA10 EA 234 11101010  lower 8 bits of Vertical Retrace Start plus
                      9th and 10th bits from Overflow results in
                      0111101010=490 Vertical Retrace Start
8C11 8C 140 10001100  Vertical Retrace End (second write to it)
                      b7=1 protect registers 0-7
                      b6=0 3 DRAM cycles for Bandwidth
                      b5=0 enable Vertical Interrupts
                      b4=0 clear Vertical Interrupts
                      b3-0=1100=12 lower 5 bits of Vertical Retrace End
                          ??? 5 bits or 4 ???
DF12 DF 223 11011111  lower 8 bits of Vertical Display Enable End plus 9th
                          and 10th bits from Overflow register results in
                          0111011111=479 (16*30=480-1)
E715 E7 231 11100111  lower 8 bits of Vertical Blank Start plus 9th and
                      10th bits from Overflow and Maximum Scan Line
                      registers results in 0111100111=487
0416 04   4 00000100  b7=0 ???
                      b6-0=4 Vertical Blank End



    Just set 350 vertical scan lines
0511 0C  12 00000101  Vertical Retrace End (first write to it)
                      b7=0 don't protect registers 0-7
                      b6=0 3 DRAM cycles for Bandwidth
                      b5=0 enable Vertical Interrupts
                      b4=0 clear Vertical Interrupts
                      b3-0=101=5 lower 5 bits of Vertical Retrace End
                          ??? 5 bits or 4 ???
BF06 BF 191 10111111  lower 8 bits of Vertical Total plus 9th and 10th
                      bits from the Overflow register results in
                      0110111111=447 Vertical Total
1F07 1F  31 00011111  b7=0 bit 10 of Vertical Retrace Start
                      b6=0 bit 10 of Vertical Display Enable End
                      b5=0 bit 10 of Vertical Total
                      b4=1 bit 9 of Line Compare
                      b3=1 bit 9 of Vertical Blank Start
                      b2=1 bit 9 of Vertical Retrace Start
                      b1=1 bit 9 of Vertical Display Enable End
                      b0=1 bit 9 of Vertical Total
4709 47  71 01000111  b7=0 200-to-400-line Conversion (scan lines)
                      b6=1 bit 10 of Line Compare
                      b5=0 bit 10 of Vertical Blank Start
                      b4-0=7+1=8 Maximum Scan Line, # of scan lines
                          used to display a character cell - 1
8310 83 131 10000011  lower 8 bits of Vertical Retrace Start plus
                      9th and 10th bits from Overflow results in
                       0110000011=387 Vertical Retrace Start
8511 85 140 10000101  Vertical Retrace End (second write to it)
                      b7=1 protect registers 0-7
                      b6=0 3 DRAM cycles for Bandwidth
                      b5=0 enable Vertical Interrupts
                      b4=0 clear Vertical Interrupts
                      b3-0=101=5 lower 5 bits of Vertical Retrace End
                          ??? 5 bits or 4 ???
5712 57  87 01010111  lower 8 bits of Vertical Display Enable End plus 9th
                      and 10th bits from Overflow register results in
                      0101010111=343 (8*43=344-1)
6315 63  99 01100011  lower 8 bits of Vertical Blank Start plus 9th and
                      10th bits from Overflow and Maximum Scan Line
                      registers results in 0101100011=487
BA16 BA 186 10111010  b7=1 ???
                      b6-0=58 Vertical Blank End



   80x30
0C11 0C  12 00001100  Vertical Retrace End (first write to it)
                      b7=0 don't protect registers 0-7
                      b6=0 3 DRAM cycles for Bandwidth
                      b5=0 enable Vertical Interrupts
                      b4=0 clear Vertical Interrupts
                      b3-0=1100=12 lower 5 bits of Vertical Retrace End
                          ??? 5 bits or 4 ???
5F00 5F  95 01011111  b7-2=???
                      b1=1 Synchronous Reset
                      b0=1 Synchronous Reset
4F01 4F  79 01001111  79+1=80 Displayed Horizontal Chars
5002 50  80 01010000  80 Start Horizontal Blanking
8203 82 130 10000010  b7=1 enable access to Vertical Retrace Start
                          and Vertical Retrace End (index 10h&11h)
                      b5-6=00 Display Enable Skew (Horizontal Delay)
                      b4-0 lower 5 bits of End Horizontal Blanking
                          plus 6th bit from register 5 below results in
                          100010=34 for End Horizontal Blanking (80+34-100=14)
5F04 5F  95 01011111  5F=95 Start Horizontal Retrace
8105 81 129 10000001  b7=1=6th bit of End Horizontal Blanking
                      b5-6=00 Horizontal Retrace Delay
                      b4-0=00001=1 End Horizontal Retrace
0B06 0B 011 00001011  lower 8 bits of Vertical Total plus 9th and 10th
                          bits from the Overflow register results in
                          1000001011=523 Vertical Total
3E07 3E  62 00111110  b7=0 bit 10 of Vertical Retrace Start
                      b6=0 bit 10 of Vertical Display Enable End
                      b5=1 bit 10 of Vertical Total
                      b4=1 bit 9 of Line Compare
                      b3=1 bit 9 of Vertical Blank Start
                      b2=1 bit 9 of Vertical Retrace Start
                      b1=1 bit 9 of Vertical Display Enable End
                      b0=0 bit 9 of Vertical Total
4F09 4F  79 01001111  b7=0 200-to-400-line Conversion (double scan lines)
                      b6=1 bit 10 of Line Compare
                      b5=0 bit 10 of Vertical Blank Start
                      b4-0=15+1=16 Maximum Scan Line, # of scan lines
                          used to display a character cell - 1
EA10 EA 234 11101010  lower 8 bits of Vertical Retrace Start plus
                      9th and 10th bits from Overflow results in
                      0111101010=490 Vertical Retrace Start
8C11 8C 140 10001100  Vertical Retrace End (second write to it)
                      b7=1 protect registers 0-7
                      b6=0 3 DRAM cycles for Bandwidth
                      b5=0 enable Vertical Interrupts
                      b4=0 clear Vertical Interrupts
                      b3-0=1100=12 lower 5 bits of Vertical Retrace End
                          ??? 5 bits or 4 ???
DF12 DF 223 11011111  lower 8 bits of Vertical Display Enable End plus 9th
                          and 10th bits from Overflow register results in
                          0111011111=479 (16*30=480-1)
2813 28  40 00101000  Offset register (40*2=80 chars per line)
E715 E7 231 11100111  lower 8 bits of Vertical Blank Start plus 9th and
                      10th bits from Overflow and Maximum Scan Line
                      registers results in 0111100111=487
0416 04   4 00000100  b7=0 ???
                      b6-0=4 Vertical Blank End




   90x25
0C11 0C  12 00001100  Vertical Retrace End (first write to it)
                      b7=0 don't protect registers 0-7
                      b6=0 3 DRAM cycles for Bandwidth
                      b5=0 enable Vertical Interrupts
                      b4=0 clear Vertical Interrupts
                      b3-0=1100=12 lower 5 bits of Vertical Retrace End
                          ??? 5 bits or 4 ???
6B00 6B 107 01101100  b7-2=???
                      b1=0 Synchronous Reset
                      b0=0 Synchronous Reset
5901 59  89 01011001  59+1=90 Displayed Horizontal Chars
5A02 5A  90 01011010  5A=90 Start Horizontal Blanking
8E03 8E 142 10001110  b7=1 enable access to Vertical Retrace Start
                          and Vertical Retrace End (index 10h&11h)
                      b5-6=00 Display Enable Skew (Horizontal Delay)
                      b4-0=01110=14 lower 5 bits of End Horizontal Blanking
                          plus 6th bit from register 5 below results in
                          11110=30 for End Horizontal Blanking
5F04 5F  95 01011111  5F=95 Start Horizontal Retrace
8A05 8A 138 10001010  b7=1=6th bit of End Horizontal Blanking
                      b5-6=00 Horizontal Retrace Delay
                      b4-0=01010=10 End Horizontal Retrace
BF06 BF 191 10111111  lower 8 bits of Vertical Total plus 9th and 10th
                          bits from the Overflow register results in
                          0110111111=447 Vertical Total
1F07 1F  31 00011111  b7=0 bit 10 of Vertical Retrace Start
                      b6=0 bit 10 of Vertical Display Enable End
                      b5=0 bit 10 of Vertical Total
                      b4=1 bit 9 of Line Compare
                      b3=1 bit 9 of Vertical Blank Start
                      b2=1 bit 9 of Vertical Retrace Start
                      b1=1 bit 9 of Vertical Display Enable End
                      b0=1 bit 9 of Vertical Total
4F09 4F  79 01001111  b7=0 200-to-400-line Conversion (scan lines)
                      b6=1 bit 10 of Line Compare
                      b5=0 bit 10 of Vertical Blank Start
                      b4-0=15+1=16 Maximum Scan Line, # of scan lines
                          used to display a character cell - 1
9C10 9C 156 10011100  lower 8 bits of Vertical Retrace Start plus
                      9th and 10th bits from Overflow results in
                      0110011100=412 Vertical Retrace Start
8E11 8E 142 10001110  Vertical Retrace End (second write to it)
                      b7=1 protect registers 0-7
                      b6=0 3 DRAM cycles for Bandwidth
                      b5=0 enable Vertical Interrupts
                      b4=0 clear Vertical Interrupts
                      b3-0=1110=14 lower 5 bits of Vertical Retrace End
                          ??? 5 bits or 4 ???
8F12 8F 143 10001111  lower 8 bits of Vertical Display Enable End plus 9th
                          and 10th bits from Overflow register results in
                          0110001111=399 (16*25=400-1)
2D13 2D  45 00101111  Offset register (45*2=90 chars per line)
9615 96 150 10010110  lower 8 bits of Vertical Blank Start plus 9th and
                      10th bits from Overflow and Maximum Scan Line
                      registers results in 0110010110=406
B916 B9 185 10111001  b7=1 ???
                      b6-0=57 Vertical Blank End



   94x25
0C11 0C  12 00001100  Vertical Retrace End (first write to it)
                      b7=0 don't protect registers 0-7
                      b6=0 3 DRAM cycles for Bandwidth
                      b5=0 enable Vertical Interrupts
                      b4=0 clear Vertical Interrupts
                      b3-0=1100=12 lower 5 bits of Vertical Retrace End
                          ??? 5 bits or 4 ???
6B00 6B 107 01101011  b7-2=???
                      b1=1 Synchronous Reset
                      b0=1 Synchronous Reset
5D01 5D  93 01011101  5D+1=94 Displayed Horizontal Chars
5E02 5E  94 01011110  5E=94 Start Horizontal Blanking
8F03 8F 143 10001111  b7=1 enable access to Vertical Retrace Start
                          and Vertical Retrace End (index 10h&11h)
                      b5-6=00 Display Enable Skew (Horizontal Delay)
                      b4-0=01111=15 lower 5 bits of End Horizontal Blanking
                          plus 6th bit from register 5 below results in
                          11111=31 for End Horizontal Blanking
6204 62  98 01100010  62=98 Start Horizontal Retrace
8E05 8E 142 10001110  b7=1=6th bit of End Horizontal Blanking
                      b5-6=00 Horizontal Retrace Delay
                      b4-0=01110=14 End Horizontal Retrace
BF06 BF 191 10111111  lower 8 bits of Vertical Total plus 9th and 10th
                          bits from the Overflow register results in
                          0110111111=447 Vertical Total
1F07 1F  31 00011111  b7=0 bit 10 of Vertical Retrace Start
                      b6=0 bit 10 of Vertical Display Enable End
                      b5=0 bit 10 of Vertical Total
                      b4=1 bit 9 of Line Compare
                      b3=1 bit 9 of Vertical Blank Start
                      b2=1 bit 9 of Vertical Retrace Start
                      b1=1 bit 9 of Vertical Display Enable End
                      b0=1 bit 9 of Vertical Total
4F09 4F  79 01001111  b7=0 200-to-400-line Conversion (scan lines)
                      b6=1 bit 10 of Line Compare
                      b5=0 bit 10 of Vertical Blank Start
                      b4-0=15+1=16 Maximum Scan Line, # of scan lines
                          used to display a character cell - 1
9C10 9C 156 10011100  lower 8 bits of Vertical Retrace Start plus
                      9th and 10th bits from Overflow results in
                      0110011100=412 Vertical Retrace Start
8E11 8E 142 10001110  Vertical Retrace End (second write to it)
                      b7=1 protect registers 0-7
                      b6=0 3 DRAM cycles for Bandwidth
                      b5=0 enable Vertical Interrupts
                      b4=0 clear Vertical Interrupts
                      b3-0=1110=14 lower 5 bits of Vertical Retrace End
                          ??? 5 bits or 4 ???
8F12 8F 143 10001111  lower 8 bits of Vertical Display Enable End plus 9th
                          and 10th bits from Overflow register results in
                          0110001111=399 (16*25=400-1)
2F13 2F  47 00101111  Offset register (47*2=94 chars per line)
9615 96 150 10010110  lower 8 bits of Vertical Blank Start plus 9th and
                      10th bits from Overflow and Maximum Scan Line
                      registers results in 0110010110=406
B916 B9 185 10111001  b7=1 ???
                      b6-0=57 Vertical Blank End



   94x30
0C11 0C  12 00001100  Vertical Retrace End (first write to it)
                      b7=0 don't protect registers 0-7
                      b6=0 3 DRAM cycles for Bandwidth
                      b5=0 enable Vertical Interrupts
                      b4=0 clear Vertical Interrupts
                      b3-0=1100=12 lower 5 bits of Vertical Retrace End
                          ??? 5 bits or 4 ???
6C00 6C 108 01101100  b7-2=???
                      b1=0 no Synchronous Reset
                      b0=0 no Synchronous Reset
5D01 5D  93 01011101  5D+1=94 Displayed Horizontal Chars
5E02 5E  94 01011110  5E=94 Start Horizontal Blanking
8F03 8F 143 10001111  b7=1 enable access to Vertical Retrace Start
                          and Vertical Retrace End (index 10h&11h)
                      b5-6=00 Display Enable Skew (Horizontal Delay)
                      b4-0=01111=15 lower 5 bits of End Horizontal Blanking
                          plus 6th bit from register 5 below results in
                          11111=31 for End Horizontal Blanking
6204 62  98 01100010  62=98 Start Horizontal Retrace
8E05 8E 142 10001110  b7=1=6th bit of End Horizontal Blanking
                      b5-6=00 Horizontal Retrace Delay
                      b4-0=01110=14 End Horizontal Retrace
0B06 0B  11 00001011  lower 8 bits of Vertical Total plus 9th and 10th
                          bits from the Overflow register results in
                          1000001011=523 Vertical Total
3E07 3E  62 00111110  b7=0 bit 10 of Vertical Retrace Start
                      b6=0 bit 10 of Vertical Display Enable End
                      b5=1 bit 10 of Vertical Total
                      b4=1 bit 9 of Line Compare
                      b3=1 bit 9 of Vertical Blank Start
                      b2=1 bit 9 of Vertical Retrace Start
                      b1=1 bit 9 of Vertical Display Enable End
                      b0=0 bit 9 of Vertical Total
4F09 4F  79 01001111  b7=0 200-to-400-line Conversion (scan lines)
                      b6=1 bit 10 of Line Compare
                      b5=0 bit 10 of Vertical Blank Start
                      b4-0=15+1=16 Maximum Scan Line, # of scan lines
                          used to display a character cell - 1
EA10 EA 234 11101010  lower 8 bits of Vertical Retrace Start plus
                      9th and 10th bits from Overflow results in
                      0111101010=490 Vertical Retrace Start
8C11 8C 140 10001100  Vertical Retrace End (second write to it)
                      b7=1 protect registers 0-7
                      b6=0 3 DRAM cycles for Bandwidth
                      b5=0 enable Vertical Interrupts
                      b4=0 clear Vertical Interrupts
                      b3-0=1100=12 lower 5 bits of Vertical Retrace End
                          ??? 5 bits or 4???
DF12 DF 223 11011111  lower 8 bits of Vertical Display Enable End plus 9th
                          and 10th bits from Overflow register results in
                          0111011111=479 (16*30=480-1)
2F13 2F  47 00101111  Offset register (47*2=94 chars per line)
E715 E7 231 11100111  lower 8 bits of Vertical Blank Start plus 9th and
                      10th bits from Overflow and Maximum Scan Line
                      registers results in 0111100111=487
0416 04   4 00000100  b7=0 ???
                      b6-0=4 Vertical Blank End
---------------------------------------------------------------------------*/



    /* Used throughout the code with int86() and int86x(). */
union REGS InRegs, OutRegs;
struct SREGS SegRegs;

    /* VESA stuff. */
struct VESAInfoStruc VInfo;
struct VESAModeInfoStruc VModeInfo;
struct VESATextModesStruc VTextModes;
byte VesaPresent;
word VI_SupportedModes[6];



/*---------------------------------------------------------------------------
                            SetAltPrintScreen()

    Set Alternate Print Screen function.  This causes the BIOS to switch to
    a new print screen function that can handle non-standard sized video
    modes.  Apparently, the first BIOSes had a function that was hardwired
    to only do 80x25, so a new function was required, and a way to set it.
    This may be unnecessary on modern systems.  This is not needed for VESA
    modes.
---------------------------------------------------------------------------*/
void SetAltPrintScreen(void)
{
    asm mov ah,0x12
    asm mov bl,0x20
    asm int 0x10
}



/*---------------------------------------------------------------------------
                            SetBiosValues()


    Set the Columns, Rows, and Buffer Length variables in the VIDEO
    BIOS Data Area which starts at 0040h:0049h.  There is mention in places
    that some old BIOSes do not update these correctly when modes are set
    which are completely or partialy created outside BIOS control, therefore
    most of the code I've seen does it directly as done here.  As an
    opposite example though, my VIDEO BIOS sets these values correctly for
    many of the modes here.  If working solely on modern video systems, this
    routine may be unnecessary.  This is not needed for VESA modes.
---------------------------------------------------------------------------*/
void SetBiosValues(word Cols, byte Rows)
{
        /* Video BIOS word CRT_COLS */
    poke(0x40, 0x4A, Cols);

        /* Video BIOS byte ROWS; In the BIOS, its Rows-1 */
    pokeb(0x40, 0x84, --Rows);

        /*
            Video BIOS word CRT_LEN (length of buffer in bytes)
            length = columns * rows * 2
        */
    poke(0x40, 0x4C, ((Cols * Rows) << 1));
}



/*---------------------------------------------------------------------------
                            SetFontSize()


    Set font to 8 pixels or 14 pixels or 16 pixels in height.  These fonts
    are the standard ones in the ROM.  There aren't any others.  Note that the
    function assumes only 8, 14, or 16 will be sent to it.  These are the only
    values that make sense.  This is not needed for VESA modes.
---------------------------------------------------------------------------*/
void SetFontSize(int FontSize)
{
    InRegs.h.ah = 0x11;
    InRegs.h.bl = 0;

    if (FontSize == 8)
        InRegs.h.al = 0x12;
    if (FontSize == 14)
        InRegs.h.al = 0x11;
    if (FontSize == 16)
        InRegs.h.al = 0x14;

    int86(0x10, &InRegs, &OutRegs);
}



/*---------------------------------------------------------------------------
                            SetSpecialTextMode()


    Major voodoo.  Amalgamated from several sources on the internet, but
    as mentioned above, no documentation.  Some of the descriptions below
    I got from books, some I don't understand.  This is not used for VESA
    modes.
---------------------------------------------------------------------------*/
void SetSpecialTextMode(word *DataPtr)
{
    asm mov ax,3   /* set BIOS 80x25 mode */
    asm int 0x10

        /* wait for vertical retrace */
    asm mov dx,03DAh
        loop1:
    asm in al,dx
    asm and al,00001000b
    asm cmp al,0
    asm jnz loop1

        loop2:
    asm in al,dx
    asm and al,00001000b
    asm cmp al,0
    asm jz loop2


        /*
        Clocking Mode register, index 01 of sequencer registers
        b5=0 turn screen on
        b4=0 load serializers every cycle
        b3=0 set dot clock same as master clock
        b2=0 use high horizontal resolution (640/720)
        b1=0 CRTC controls memory bus 4 out of 5 cycles
        b0=1 set dot clock to 9 (9 dots character width, 720)
        */
    asm mov dx,03C4h
    asm mov ax,0101h
    asm out dx,ax

        /*
        reset data flip-flop for 0x3C0, the Attribute Address register
        to address mode by reading a byte
        */
    asm mov dx,03DAh
    asm in al,dx        /* ignore input */

        /*
        set Attribute Address register:
        b5=1 allow display memory to access palette ram
        b4-0=10h select Mode Control register
        */
    asm mov dx,03C0h
    asm mov al,30h   /*00110000*/
    asm out dx,al
        /*
        set Mode Control register
        b7=0 internal palette size
        b6=0 pixel clock select
        b5=0 pixel panning compatibility
        b4 not used
        b3=0 enable blink/intensity
        b2=1 enable line graphics char codes
        b1=0 display type
        b0=0 graphics/alphanumeric
        */
    asm mov al,4
    asm out dx,al

        /*
        set Attribute Address register:
        b5=1 allow display memory to access palette ram
        b4-0=13h select Horizontal Pixel Panning register
        */
    asm mov al,33h
    asm out dx,al
        /*
        set Horizontal Pixel Panning register
        b3-0=0 horizontal pixel pan
        */
    asm xor al,al
    asm out dx,al

        /* Send the 16 words of data. */
    asm cld
    asm mov si,[DataPtr]
    asm mov cx,16d
    asm mov dx,03D4h
    asm rep outsw
}



/*---------------------------------------------------------------------------
                            GetVESAModes()



    If we have a VESA system, get the mode list, and get from that a list of
    the textmodes.  There are 5 VESA textmodes (there will be no more).  Go
    through those and get the raw mode numbers into VTextModes.  We validate
    every mode because there are cards that do not implement all the
    textmodes.  This isn't an error, VESA does not mandate that all modes be
    supported.

    There are VESA docs available on the internet.
---------------------------------------------------------------------------*/
byte GetVESAModes(void)
{
    word far *VModes;
    int i, VI_SM;

    SegRegs.es = FP_SEG(&VInfo);
    InRegs.x.di = FP_OFF(&VInfo);
    InRegs.x.ax = 0x4F00;
    int86x(0x10, &InRegs, &OutRegs, &SegRegs);
    if (OutRegs.x.ax != 0x004F)
            /* al=4F if VESA present, ah=00 if function successful */
        return FALSE;

        /*
            Get all the supported text modes into VI_SupportModes array.
            Not all VESA video cards will support all the VESA textmodes.
            Note VI_SupportModes has an extra word, so there will always
            be a zero to terminate the list.
        */
    VModes = VInfo.VideoModesPtr;
    for (i = 0; i < 6; i++)
            VI_SupportedModes[i] = 0;
    VI_SM = 0;

        /* The VESA BIOS ends its list with 0xFFFF */
    while (*VModes != 0xFFFF) {
        if (*VModes >= 0x0108 && *VModes <= 0x010C) {
            if (VI_SM > 4)
                break;  /* this should never happen, only 5 text modes */
            VI_SupportedModes[VI_SM++] = *VModes;
        }
        VModes++;
    }

        /*
            Go thru all the modes in VI_SupportModes, for every mode verify
            flags in ModeAttributes, MemoryModel, and (X/Y)Resolution.
            If valid enter in VTextModes array.
        */
    VTextModes.v80x60 = 0;
    VTextModes.v132x25 = 0;
    VTextModes.v132x43 = 0;
    VTextModes.v132x50 = 0;
    VTextModes.v132x60 = 0;
    VI_SM = 0;
    while(VI_SupportedModes[VI_SM] != 0) {
        SegRegs.es = FP_SEG(&VModeInfo);
        InRegs.x.di = FP_OFF(&VModeInfo);
        InRegs.x.ax = 0x4F01;
        InRegs.x.cx = VI_SupportedModes[VI_SM];
        int86x(0x10, &InRegs, &OutRegs, &SegRegs);
            /* al=4F if VESA present, ah=00 if function successful */
        if (OutRegs.x.ax == 0x004F) {
                /* bit0 = 1 = mode supported in hardware */
            if (VModeInfo.ModeAttributes & 0x0001) {
                    /* bit4 = 0 = text mode */
                if (!(VModeInfo.ModeAttributes & 0x0010)) {
                        /* memory model is 0 for text modes */
                    if (VModeInfo.MemoryModel == 0) {
                        if (VModeInfo.XResolution == 80) {
                            if (VModeInfo.YResolution == 60)
                                VTextModes.v80x60 = VI_SupportedModes[VI_SM];
                        }
                        if (VModeInfo.XResolution == 132) {
                            if (VModeInfo.YResolution == 25)
                                VTextModes.v132x25 = VI_SupportedModes[VI_SM];
                            if (VModeInfo.YResolution == 43)
                                VTextModes.v132x43 = VI_SupportedModes[VI_SM];
                            if (VModeInfo.YResolution == 50)
                                VTextModes.v132x50 = VI_SupportedModes[VI_SM];
                            if (VModeInfo.YResolution == 60)
                                VTextModes.v132x60 = VI_SupportedModes[VI_SM];
                        }
                    }
                }
            }
        }
        VI_SM++;
    }

    return TRUE;
}



/*---------------------------------------------------------------------------
                            PrintProgramUsage()


---------------------------------------------------------------------------*/
void PrintProgramUsage(void)
{
    printf("STM.EXE v2.0 Copyright @2002 by Ed Cogburn (ecogburn@xtn.net)\n");
    printf("    Set various text modes on VGA and VESA systems running DOS or\n");
    printf("    in a fullscreen DOS box under Windows.  This program is free\n");
    printf("    software released under the GPL license.  The GPL license can\n");
    printf("    be found at http://www.gnu.org/licenses/gpl.html.\n\n");

    printf("    WARNING:  Use at your own risk, there is no warranty.\n\n");

    printf("    Syntax:  STM [Dimension]\n\n");

    printf("        Dimensions:\n");
    printf("    Dimensions may be the full text, like \"80x25\", or the shortcuts\n");
    printf("    in the parentheses below such as \"25\".  90 and 94 column modes do\n");
    printf("    not have shortcuts, nor does the VESA \"V80x60\" mode.\n\n");

    printf("        VGA Dimensions Available:\n");
    printf("    80x25(25)  80x28(28)  80x30(30)  80x34(34)  80x43(43)  80x50(50)  80x60(60)\n");
    printf("    90x25  90x28  90x30  90x34  90x43  90x50  90x60\n");
    printf("    94x25  94x28  94x30  94x34  94x43  94x50  94x60\n\n");

    if (VesaPresent) {
        printf("    VESA BIOS extension detected.  Dimensions Available:\n        ");
        if (VTextModes.v80x60)
            printf("V80x60  ");
        if (VTextModes.v132x25)
            printf("V132x25(V25)  ");
        if (VTextModes.v132x43)
            printf("V132x43(V43)  ");
        if (VTextModes.v132x50)
            printf("V132x50(V50)  ");
        if (VTextModes.v132x60)
            printf("V132x60(V60)  ");
        printf("\n");
    }else
        printf("    No VESA BIOS extension detected.\n");

    printf("\n");
}



/*---------------------------------------------------------------------------
                                main()


---------------------------------------------------------------------------*/
int main(int argc, char *argv[])
{
    char ArgStr[256];
    int i;


        /*
            VIDEO BIOS ROM function.  Check for display type using
            Alternate Select (12h).  If bl changes, then its EGA/VGA.
        */
    InRegs.h.ah = 0x12;
    InRegs.h.bl = 0x10;
    int86(0x10, &InRegs, &OutRegs);
    if (OutRegs.h.bl == 0x10) {
        PrintProgramUsage();
        printf("\n    STM only works with Color VGA or later video cards.\n");
        return 1;
    }


        /*
            VIDEO BIOS ROM function.  Video Display Combination
            If al=1A its a VGA
            if bl=08 active display is a Color VGA.
        */
    InRegs.x.ax = 0x1A00;
    int86(0x10, &InRegs, &OutRegs);
    if (OutRegs.h.al != 0x1A || OutRegs.h.bl != 0x08) {
        PrintProgramUsage();
        printf("\n    STM only works with Color VGA or later video cards.\n");
        return 1;
    }


        /*
            Ok, its Color VGA.  Now is it VESA SVGA?
        */
    VesaPresent = GetVESAModes();


        /*
            Only 1 argument allowed
        */
    if (argc != 2) {
        PrintProgramUsage();
        return 1;
    }


        /*
            Make ArgStr a lowercase version of our argument
        */
    if (strlen(argv[1]) > 255) {
        PrintProgramUsage();
        printf("\nCommand line too long.\n");
        return 1;
    }
    strcpy(ArgStr,argv[1]);
    for (i = 0; i < strlen(ArgStr); i++)
        ArgStr[i] = tolower(ArgStr[i]);


        /*
            Standard help strings
        */
    if (!strcmp(ArgStr,"/h") || !strcmp(ArgStr,"-h")) {
        PrintProgramUsage();
        return 1;
    }
    if (!strcmp(ArgStr,"/?") || !strcmp(ArgStr,"-?")) {
        PrintProgramUsage();
        return 1;
    }


        /*
            80 column modes
        */
    if (!strcmp(ArgStr,"80x25") || !strcmp(ArgStr,"25")) {
        SetSpecialTextMode(Data80x25);
        return 0;
    }

    if (!strcmp(ArgStr,"80x28") || !strcmp(ArgStr,"28")) {
        SetSpecialTextMode(Data80x28);
        SetFontSize(14);
        SetBiosValues(80, 28);
        SetAltPrintScreen();
        return 0;
    }

    if (!strcmp(ArgStr,"80x30") || !strcmp(ArgStr,"30")) {
        SetSpecialTextMode(Data80x30);
        SetFontSize(16);
        SetBiosValues(80, 30);
        SetAltPrintScreen();
        return 0;
    }

    if (!strcmp(ArgStr,"80x34") || !strcmp(ArgStr,"34")) {
        SetSpecialTextMode(Data80x34);
        SetFontSize(14);
        SetBiosValues(80, 34);
        SetAltPrintScreen();
        return 0;
    }

    if (!strcmp(ArgStr,"80x43") || !strcmp(ArgStr,"43")) {
        SetSpecialTextMode(Data80x43);
        SetFontSize(8);
        SetBiosValues(80, 43);
        SetAltPrintScreen();
        return 0;
    }

    if (!strcmp(ArgStr,"80x50") || !strcmp(ArgStr,"50")) {
        SetSpecialTextMode(Data80x50);
        SetFontSize(8);
        SetBiosValues(80, 50);
        SetAltPrintScreen();
        return 0;
    }

    if (!strcmp(ArgStr,"80x60") || !strcmp(ArgStr,"60")) {
        SetSpecialTextMode(Data80x60);
        SetFontSize(8);
        SetBiosValues(80, 60);
        SetAltPrintScreen();
        return 0;
    }


        /*
            90 column modes
        */
    if (!strcmp(ArgStr,"90x25")) {
        SetSpecialTextMode(Data90x25);
        SetFontSize(16);
        SetBiosValues(90, 25);
        SetAltPrintScreen();
        return 0;
    }

    if (!strcmp(ArgStr,"90x28")) {
        SetSpecialTextMode(Data90x28);
        SetFontSize(14);
        SetBiosValues(90, 28);
        SetAltPrintScreen();
        return 0;
    }

    if (!strcmp(ArgStr,"90x30")) {
        SetSpecialTextMode(Data90x30);
        SetFontSize(16);
        SetBiosValues(90, 30);
        SetAltPrintScreen();
        return 0;
    }

    if (!strcmp(ArgStr,"90x34")) {
        SetSpecialTextMode(Data90x34);
        SetFontSize(14);
        SetBiosValues(90, 34);
        SetAltPrintScreen();
        return 0;
    }

    if (!strcmp(ArgStr,"90x43")) {
        SetSpecialTextMode(Data90x43);
        SetFontSize(8);
        SetBiosValues(90, 43);
        SetAltPrintScreen();
        return 0;
    }

    if (!strcmp(ArgStr,"90x50")) {
        SetSpecialTextMode(Data90x50);
        SetFontSize(8);
        SetBiosValues(90, 50);
        SetAltPrintScreen();
        return 0;
    }

    if (!strcmp(ArgStr,"90x60")) {
        SetSpecialTextMode(Data90x60);
        SetFontSize(8);
        SetBiosValues(90, 60);
        SetAltPrintScreen();
        return 0;
    }


        /*
            94 column modes
        */
    if (!strcmp(ArgStr,"94x25")) {
        SetSpecialTextMode(Data94x25);
        SetFontSize(16);
        SetBiosValues(94, 25);
        SetAltPrintScreen();
        return 0;
    }

    if (!strcmp(ArgStr,"94x28")) {
        SetSpecialTextMode(Data94x28);
        SetFontSize(14);
        SetBiosValues(94, 28);
        SetAltPrintScreen();
        return 0;
    }

    if (!strcmp(ArgStr,"94x30")) {
        SetSpecialTextMode(Data94x30);
        SetFontSize(16);
        SetBiosValues(94, 30);
        SetAltPrintScreen();
        return 0;
    }

    if (!strcmp(ArgStr,"94x34")) {
        SetSpecialTextMode(Data94x34);
        SetFontSize(14);
        SetBiosValues(94, 34);
        SetAltPrintScreen();
        return 0;
    }

    if (!strcmp(ArgStr,"94x43")) {
        SetSpecialTextMode(Data94x43);
        SetFontSize(8);
        SetBiosValues(94, 43);
        SetAltPrintScreen();
        return 0;
    }

    if (!strcmp(ArgStr,"94x50")) {
        SetSpecialTextMode(Data94x50);
        SetFontSize(8);
        SetBiosValues(94, 50);
        SetAltPrintScreen();
        return 0;
    }

    if (!strcmp(ArgStr,"94x60")) {
        SetSpecialTextMode(Data94x60);
        SetFontSize(8);
        SetBiosValues(94, 60);
        SetAltPrintScreen();
        return 0;
    }


        /*
            VESA modes
        */
    if (!strcmp(ArgStr,"v80x60")) {
        if (VesaPresent) {
            if (VTextModes.v80x60) {
                InRegs.x.ax = 0x4F02;
                InRegs.x.bx = VTextModes.v80x60;
                int86(0x10, &InRegs, &OutRegs);
                if (OutRegs.x.ax != 0x004F) {
                    printf("Attempted VESA mode set failed!\n\n");
                    return 1;
                }
            }else{
                printf("VESA BIOS says mode V80x60 not available.\n");
                return 1;
            }
        }else{
            printf("No VESA BIOS present.\n");
            return 1;
        }
        return 0;
    }

    if (!strcmp(ArgStr,"v132x25") || !strcmp(ArgStr,"v25")) {
        if (VesaPresent) {
            if (VTextModes.v132x25) {
                InRegs.x.ax = 0x4F02;
                InRegs.x.bx = VTextModes.v132x25;
                int86(0x10, &InRegs, &OutRegs);
                if (OutRegs.x.ax != 0x004F) {
                    printf("Attempted VESA mode set failed!\n\n");
                    return 1;
                }
            }else{
                printf("VESA BIOS says mode V132x25 not available.\n");
                return 1;
            }
        }else{
            printf("No VESA BIOS present.\n");
            return 1;
        }
        return 0;
    }

    if (!strcmp(ArgStr,"v132x43") || !strcmp(ArgStr,"v43")) {
        if (VesaPresent) {
            if (VTextModes.v132x43) {
                InRegs.x.ax = 0x4F02;
                InRegs.x.bx = VTextModes.v132x43;
                int86(0x10, &InRegs, &OutRegs);
                if (OutRegs.x.ax != 0x004F) {
                    printf("Attempted VESA mode set failed!\n\n");
                    return 1;
                }
            }else{
                printf("VESA BIOS says mode V132x43 not available.\n");
                return 1;
            }
        }else{
            printf("No VESA BIOS present.\n");
            return 1;
        }
        return 0;
    }

    if (!strcmp(ArgStr,"v132x50") || !strcmp(ArgStr,"v50")) {
        if (VesaPresent) {
            if (VTextModes.v132x50) {
                InRegs.x.ax = 0x4F02;
                InRegs.x.bx = VTextModes.v132x50;
                int86(0x10, &InRegs, &OutRegs);
                if (OutRegs.x.ax != 0x004F) {
                    printf("Attempted VESA mode set failed!\n\n");
                    return 1;
                }
            }else{
                printf("VESA BIOS says mode V132x50 not available.\n");
                return 1;
            }
        }else{
            printf("No VESA BIOS present.\n");
            return 1;
        }
        return 0;
    }

    if (!strcmp(ArgStr,"v132x60") || !strcmp(ArgStr,"v60")) {
        if (VesaPresent) {
            if (VTextModes.v132x60) {
                InRegs.x.ax = 0x4F02;
                InRegs.x.bx = VTextModes.v132x60;
                int86(0x10, &InRegs, &OutRegs);
                if (OutRegs.x.ax != 0x004F) {
                    printf("Attempted VESA mode set failed!\n\n");
                    return 1;
                }
            }else{
                printf("VESA BIOS says mode V132x60 not available.\n");
                return 1;
            }
        }else{
            printf("No VESA BIOS present.\n");
            return 1;
        }
        return 0;
    }


    PrintProgramUsage();
    return 1;
}
