#ifndef PLOT3D_H
#define PLOT3D_H

//      This is an abstract class.  Classes derived from this class may be used
// to instantiate objects that will plot z=f(x,y) in three dimensions.  The
// derived classes must supply the functions "aspect ratio", 
// "display_initialize", "pset", "num_x_pixels", "num_y_pixels", and 
// "write_outfile".

//      This class uses the template "varray" (one dimensional virtual array)
// and the class "titillat" (to let the user know the code is running).

#include "varray.h"

#define NUM_COLORS        65
// Color 65 is reserved for parts of the plot to be highlighted.  Colors
// 0 through 64 represent shades of gray from black to white.  So that parts
// of a plot may be omitted, "pset" should not plot colors greater than 65.
 
#define SOLUTION_GRAY     64
//      This is the shade of gray to be displayed for highlighted areas.
// (On color displays, a color may displayed instead of a shade of gray.)

#define LIGHTEST_GRAY     61
//      This is the lightest shade of gray that will actually be displayed
// (for other than highlighted areas).

#define DARKEST_GRAY       8
//      This is the darkest shade of gray that will actually be displayed
// (for parts that can be seen).

typedef struct
          {
            unsigned char base_z;    // '1' - normal, '2' -- highlight, or
                                     // '3' -- don't display; external to plot.
            unsigned char color;     // shade of gray for this quadrilateral.
            float         x;         // coordinates of corner of this
            float         y;         // quadrilateral.
            float         z;
            int           x_division_index;
            int           y_division_index;
          } prime_rec, *prime_rec_ptr;
//     One of the many quadrilaterals composing the plot.

typedef struct
          {
            double x;
            double y;
            double z;
          } vertex_rec;
//     A point or vector.

class plot3d
  {
    private:
      void              adjust_perspective(void);
                          // Make parallel lines running away from the viewer
                          // appear to converge at the horizon.
      unsigned char     color_max;
                          // Maximum shade of gray for a quadrilateral
                          // (before adjusted between DARKEST_GRAY and
                          // LIGHTEST_GRAY).
      unsigned char     color_min;
                          // Minimum shade of gray for a quadrilateral
                          // (before adjusted between DARKEST_GRAY and
                          // LIGHTEST_GRAY).
      void              evaluate_and_transform(double (*f)(double,double),
                         double x_min,double x_max,double y_min,double y_max,
                         int num_x_divisions,int num_y_divisions,
                         double rotation,double tilt,
                         int (*external_to_plot)(double,double),
                         int (*red)(double,double));
                          // Computes the vertices, etc. for each quadrilateral
                          // composing the plot.
      vertex_rec        light;
                          // Vector pointing to the source of light. 
      long              num_primes;
                          // Number of quadrilaterals that compose the plot.
      int               num_x_divisions;
                          // Number of divisions of the x-axis used to define
                          // the quadrilaterals composing the plot.
      int               num_y_divisions;
                          // Number of divisions of the y-axis used to define  
                          // the quadrilaterals composing the plot.
      int               plot_prepared;
                          // TRUE if the last call to "prepare_plot" was
                          // successful.
      varray<prime_rec> *prime_array;
                          // Virtual array of the quadrilaterals composing the
                          // plot.
      int               prime_array_allocated;
                          // TRUE if "prime_array" was allocated.
      void              quicksort(long,long);
                          // Recursive quicksort.
      void              rearrange(long,long,long *);
                          // Called by "quicksort".
      double            rotation;
                          // Degrees the plot is rotated about the z-axis.
      void              shade(void);
                          // Computes the shade of gray for each quadrilateral
                          // composing the plot.
      void              sort_back_to_front();
                          // The painter's algorithm is used; items farther
                          // from the viewer are drawn earlier.
      double            tilt;
                          // Degrees the plot is tilted after it is rotated.
      double            x_prime_max;
      double            y_prime_max;
      double            y_prime_min;
      double            z_prime_max;
      double            z_prime_min;
    protected:
      titillator *titillator_ptr;
                    // Lets the user know the code is running.
    public:
      virtual double aspect_ratio(void) = 0;
//      Derived classes must supply this.  The aspect ratio can be calculated
// as (HEIGHT_OF_DISPLAY/WIDTH_OF_DISPLAY)/(NUM_Y_PIXELS/NUM_X_PIXELS) where
//
//           HEIGHT_OF_DISPLAY is the height of the display in centimeters, 
//           WIDTH_OF_DISPLAY is the width of the display in centimeters,
//           NUM_X_PIXELS is the width of the display in pixels, and
//           NUM_Y_PIXELS is the height of the display in pixels.

      virtual int    display_initialized(void) = 0;
//     Derived classes must supply this; it is called by "prepare_plot".  It
// does whatever is necessary to initialize the display. 

                     plot3d(void);

      virtual        ~plot3d(void);

      virtual void   pset(int x,int y,int color_num) = 0;
//     Derived classes must supply this; it is called by "plot".  It sets the
// pixel at column "x" and row "y" to represent the shade of gray represented by
// "color_num".

      virtual int    num_x_pixels(void) = 0;
//     Derived classes must supply this; it is called by "plot".  It returns the
// number of columns in the display.  That is, it returns the width of the
// display in pixels.

      virtual int    num_y_pixels(void) = 0;
//     Derived classes must supply this; it is called by "plot".  It returns the
// number of rows in the display.  That is, it returns the height of the display
// in pixels.

              int    plot(char *file_name,int show_red,
                      int titillate=-1,double bias=0.5);
//     This function returns TRUE if and only if it is successful in generating
// the 3D plot.  It calls "aspect_ratio", "pset", and "write_outfile".  Its
// parameters are as follow:
//
//           file_name -- the name of the file to which the plot is to be 
//      written.  For plots on a cathode ray tube, this will usually be "".     
//
//           show_red -- highlight quadrilaterals having each vertex flagged
//      to be highlighted.  Since only those quadrilaterals are redrawn,
//      "plot" should not be called with "show_red" set to TRUE until after 
//      it has been called with "show_red" set to FALSE.
//
//           titillate -- TRUE if the user is to be kept amused while the
//      plot is being generated; FALSE otherwise.  In general, "titillate"
//      should be TRUE for printers and FALSE for cathode ray tubes.
//
//           bias -- a positive number used to adjust the contrast.
//
// "prepare_plot" must be called before "plot", after which "plot" may be called
// as many times as desired.

              int    prepare_plot(double (*f)(double,double),double x_min,
                      double x_max,double y_min,double y_max,
                      int (*external_to_plot)(double,double),
                      int (*red)(double,double),int x_division_count,
                      int y_division_count,double rotation_in_degrees,
                      double tilt_in_degrees,double light_x,double light_y,
                      double light_z);
//      This function prepares a plot for generation.  If returns TRUE if
// and only if it is successful.  If it is successful, "plot" may be called
// to actually generate the plot.  Its parameters are as follow:
//
//           f -- z=f(x,y), the function to be plotted.  Before the plot is
//      tilted or rotated, the z-axis runs from the bottom to the top of the
//      display, the y-axis runs from the left to the right of the display,
//      and the x-axis runs out of the display.
//
//           x_min -- the minimum value of x to be plotted.
//
//           x_max -- the maximum value of x to be plotted.
//
//           y_min -- the minimum value of y to be plotted.
//
//           y_max -- the maximum value of y to be plotted.
//
//           external_to_plot -- a function that returns TRUE if and only if a
//      point should be omitted from the plot.
//
//           red -- a function that returns TRUE if and only if a point should
//      be flagged for highlighting.  A point should be so flagged only if it
//      can be seen in the final plot.
//
//           x_division_count -- the number of x divisions to be used in
//      constructing the plot.  At least two must be specified.
//
//           y_division_count -- the number of y divisions to be used in
//      constructing the plot.  At least two must be specified.
//
//           rotation_in_degrees -- rotation (degrees) about an axis parallel to
//      the z-axis and through the center of the surface.
//
//           tilt_in_degrees -- tilt (degrees) about an axis through the center
//      of the surface and parallel to a line from the lower left hand corner of
//      the display to the lower right hand corner of the display.  The plot is
//      tilted after it is rotated.
//
//           (light_x,light_y,light_z) -- a vector pointing to the light source
//      (at infinity).  The light source remains fixed while the plot is rotated
//      or tilted.

      virtual int    write_outfile(char *file_name) = 0;
//     Derived classes must supply this; it is called by "plot".  For printers,
// this function returns TRUE if and only if the 3D plot is successfully written
// to the named file. For cathode ray tube displays, this function need do
// nothing but return TRUE.

  };

#endif
