/*===========================================================================
SOLAR :: ..\common\fossil.c

Contributed by Kevin Sartorelli
This software module has been placed into the public domain.
===========================================================================*/

#include <conio.h>
#include <dos.h>
#include <stdarg.h>
#include <stdio.h>
#include <time.h>
#include "..\common\fossil.h"
#include "..\common\yesno.h"

extern int use_fossil;
extern int port;

void PutChar(int);             /* Output one character locally and remotely */
void RPrintf(const char *, ...);  /* Output formatted string to serial port */
void Rgets(char *str);                                /* Get string routine */
int  Rgetc(int port);                       /* Get a key (-1 exit on error) */

#define KEY_BELL           7                  /* Bell character (Control-G) */
#define KEY_LF            10             /* Line feed character (Control-J) */
#define KEY_CR            13                /* RETURN character (Control-M) */
#define KEY_DEL          127                            /* Delete character */

/****************************************************************************/
/*                                                                          */
/*                    Input string from FOSSIL or local                     */
/*                                                                          */
/****************************************************************************/

void Rgets(char *strg)
{
   int numchar;                           /* Number of characters in string */
   int maxlen = 80;
   int ch;

   numchar = 0;
   do {
      ch = Rgetc(port);                             /* Get key without echo */
      if (ch == -1) {
         exit(99);                          /* Fatal exit - something wrong */
      }
      if (ch == '\b' || ch == KEY_DEL)            /* If backspace or delete */
         if (numchar > 0) {                 /* If a character in the string */
            numchar--;                                /* One less character */
            RPrintf("\b \b");               /* Delete character from screen */
         }
/*      if (ch == CAN)            */            /* Ctrl-X: erase input line */
/*         while (numchar > 0) {  */    /* For each character in the string */
/*            numchar--;          */                  /* One less character */
/*            RPrintf("\b \b");   */        /* Delete character from screen */
/*         }                      */
      if (ch >= ' ' && ch != KEY_DEL && numchar < maxlen) {
         strg[numchar++] = ch;          /* Add character to string of valid */
         PutChar(ch);
      }
   } while (ch != KEY_CR);
   strg[numchar] = '\0';                                /* Terminate string */
   PutChar('\n');                              /* Go to new line on display */
}

/****************************************************************************/
/*                                                                          */
/*                  Input keystroke from FOSSIL or local                    */
/*                                                                          */
/****************************************************************************/

int Rgetc(port)
{
   int i,gotone,fstatus;
   int warned = NO;
   time_t timestart;                             /* Time we started waiting */
   time_t timenow;                                              /* Time now */

   if (use_fossil == NO) {
      i = getch();                                    /* Get local key only */
      if (!i) {                       /* Character is an extended character */
         i = getch();
         if (i == 'K')                                /* Back arrow pressed */
            i = KEY_DEL;
         else
            i = 0;
      }
      return(i);
   }

   i = 0;
   time(&timestart);
   gotone = NO;
   do {
      time(&timenow);
      if (difftime(timenow,timestart) >= 180L && (warned == NO)) {
         RPrintf("%c%c%c",KEY_BELL,KEY_BELL,KEY_BELL);    /* Output warning */
         warned = YES;                          /* Only output warning once */
      }
      if (difftime(timenow,timestart) >= 300L) {
         RPrintf("You must have gone to sleep!\n");
         return(-1);
      }
      if (kbhit()) {                                /* Check for local keys */
         i = getch();
         if (!i) {                    /* Character is an extended character */
            i = getch();
            if (i == 'K')                             /* Back arrow pressed */
               i = KEY_DEL;
            else
               i = 0;
         }
         gotone = YES;
      }
      fstatus = Fossil_Status(port);                /* Get status of FOSSIL */
      if (!(fstatus & CARRIER_DETECT)) {                /* Carrier lost :-( */
         return(-1);
      }
      if ((fstatus & DATA_AVAILABLE)) {
         i = Fossil_Getc(port);
         gotone = YES;
      }
   } while ((gotone == NO) && (i != -1));/* Loop until key pressed or error */
   return(i);                                     /* Return the key pressed */
}

/****************************************************************************/
/*                                                                          */
/*              Formatted output to comm port and local screen              */
/*                                                                          */
/****************************************************************************/

void RPrintf(const char *format, ...)
{
   int i;
   char dlin[0x100];
   va_list ap;

   va_start(ap, format);
   vsprintf(dlin, format, ap);
   va_end(ap);
   for (i = 0; dlin[i] != '\0'; i++)
      PutChar(dlin[i]);
}

/****************************************************************************/
/*                                                                          */
/*                        Output a single character                         */
/*                                                                          */
/****************************************************************************/

void PutChar(int c)
{
   if (c != KEY_LF)
      putc(c,stdout);                           /* Output character locally */
   else {
      putc(KEY_LF,stdout);                             /* Scroll the screen */
      putc(KEY_CR,stdout);
   }

   if (use_fossil == YES) {
      if (c == KEY_LF)
         c = KEY_CR;             /* C uses LF as next line, CR as same line */
      Fossil_Putc(port,c);                       /* Send character to modem */
      if (c == KEY_CR)
         Fossil_Putc(port,KEY_LF);                   /* Send LF if required */
   }
}

/****************************************************************************/
/*                                                                          */
/*            Fossil_Init - initialises the fossil driver.                  */
/*                                                                          */
/****************************************************************************/

#define SIGNATURE   0x1954

int Fossil_Init(int port)
{
   union REGS regs;

   if (port < 0)
      return(0);
   regs.h.ah = 4;                                      /* Initialise Driver */
   regs.x.bx = 0;                               /* Leave ^C interrupt as is */
   regs.x.dx = port;
   int86(FOSSIL, &regs, &regs);

   return (regs.x.ax == SIGNATURE);
}

/****************************************************************************/
/*                                                                          */
/*                Deinitialises the Fossil driver after use                 */
/*                                                                          */
/****************************************************************************/

void Fossil_Deinit(int port)
{
   union REGS regs;

   if (port < 0)
      return;
   regs.h.ah = 5;                                   /* De-initialise driver */
   regs.x.dx = port;                           /* 0 = COM1:, 1 = COM2:, ... */
   int86(FOSSIL, &regs, &regs);
}

/****************************************************************************/
/*                                                                          */
/*               Controls the DTR line via the Fossil driver                */
/*                                                                          */
/****************************************************************************/

void Fossil_Set_DTR(int port, int new_state)
{
   union REGS regs;

   if (port <0)
      return;
   regs.h.ah = 6;
   regs.x.dx = port;                           /* 0 = COM1:, 1 = COM2:, ... */
   regs.h.al = new_state & 1;
   int86(FOSSIL, &regs, &regs);
}

/****************************************************************************/
/*                                                                          */
/*                 Gets status of the specified serial port                 */
/*                                                                          */
/****************************************************************************/

int Fossil_Status(int port)
{
   union REGS regs;

   if (port < 0)
                             /* CD, room available, possibly data available */
      return ( 0x3088 | (kbhit() ? DATA_AVAILABLE:0) );
   else {
      regs.h.ah = 3;                                      /* Request status */
      regs.x.dx = port;                        /* 0 = COM1:, 1 = COM2:, ... */
      int86(FOSSIL, &regs, &regs);
      return ( regs.x.ax );
   }
}

/****************************************************************************/
/*                                                                          */
/*            Sends a character out the serial port via fossil              */
/*                                                                          */
/****************************************************************************/

int Fossil_Putc(int port, int ch)
{
   union REGS regs;

   if (port < 0)
      return (Fossil_Status(port));
   else {
      ch &= 0xff;
      regs.h.ah = 1;                        /* Transmit character with wait */
      regs.x.dx = port;
      regs.h.al = (unsigned char) ch;
      int86(FOSSIL, &regs, &regs);

      if (ch == '\n') {
         regs.h.ah = 1;                     /* Transmit character with wait */
         regs.x.dx = port;
         regs.h.al = (unsigned char) '\r';
         int86(FOSSIL, &regs, &regs);
      }
      return(regs.x.ax);
   }
}

/****************************************************************************/
/*                                                                          */
/*             Sends a string out the serial port via fossil                */
/*                                                                          */
/****************************************************************************/

int Fossil_Puts(int port, char *s)
{
   int result;

   if (port < 0)
      return(0);
   while (*s)
      result = Fossil_Putc(port, (unsigned int) *s++);
   return (result);
}

/****************************************************************************/
/*                                                                          */
/*           Sends a character out the serial port via fossil               */
/*                      ............................._always_               */
/*                                                                          */
/****************************************************************************/

int Fossil_Aputc(int port, int ch)
{
   time_t     now;
   union REGS regs;

   if (port < 0)
      return (Fossil_Status(port));
   else {
      ch &= 0xff;
      now = time(NULL) + 10;
      while (!(Fossil_Status(port) & ALL_AVAILABLE) && (time(NULL) < now))
         ;

      if (!(Fossil_Status(port) & ALL_AVAILABLE))
         Fossil_Purge(port);

      regs.h.ah = 1;                        /* Transmit character with wait */
      regs.x.dx = port;
      regs.h.al = (unsigned char) ch;
      int86(FOSSIL, &regs, &regs);

      return (regs.x.ax);
   }
}

/****************************************************************************/
/*                                                                          */
/*            Gets a character from the serial port via fossil              */
/*                                                                          */
/****************************************************************************/

int Fossil_Getc(int port)
{
   union REGS  regs;

   if (port < 0)
      return(getch());
   else {
      regs.h.ah = 2;                         /* Receive character with wait */
      regs.x.dx = port;
      int86(FOSSIL, &regs, &regs);

      return (regs.h.al & 0xff);
   }
}

/****************************************************************************/
/*                                                                          */
/*                        Fossil 'gets' routine                             */
/*                                                                          */
/****************************************************************************/

char *Fossil_Gets(char *s, int n, int port)
{
   union REGS   regs;
   struct SREGS sregs;

   if (port < 0)
      return(NULL);
   sregs.es  = FP_SEG( (void far *) s );               /* Address of string */
   regs.x.di = FP_OFF( (void far *) s );

   regs.h.ah = 0x18;                                          /* Read block */
   regs.x.cx = n;                                       /* Maximum transfer */
   regs.x.dx = port;                                       /* Port affected */
   int86x( FOSSIL, &regs, &regs, &sregs);

   s[regs.x.ax] = '\0';                    /* Terminate the string returned */
   return(s);
}

/****************************************************************************/
/*                                                                          */
/*               Immediately dumps output buffers to limbo                  */
/*                                                                          */
/****************************************************************************/

void Fossil_Purge(int port)
{
   union REGS regs;

   if (port < 0)
      return;

   regs.h.ah = 9;                                /* Purge output buffers */
   regs.x.dx = port;
   int86(FOSSIL, &regs, &regs);
}

/****************************************************************************/
/*                                                                          */
/*                 Turns the watch CD function on or off                    */
/*                                                                          */
/****************************************************************************/

void Fossil_WatchCD(int port, int status)
{
   union REGS regs;

   if (port < 0)
      return;
   regs.h.ah = 0x14;                             /* Enable/disable watchdog */
   regs.h.al = status & 1;
   regs.x.dx = port;
   int86(FOSSIL, &regs, &regs);
}
